/*
 * SPDX-FileCopyrightText: 2022-2023 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Apache-2.0
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "soc/soc_caps.h"
#include "esp_log.h"
#include "esp_adc/adc_oneshot.h"
#include "esp_adc/adc_cali.h"
#include "esp_adc/adc_cali_scheme.h"
#include "bsp_Adc.h"
// #include "esp_adc/include/esp_adc/adc_cali_schemes.h"

ADC_User_Data_Struct KL30_Data;
ADC_User_Data_Struct KL15_Data;

void ADC_User_Data_Reset(void)
{
    KL30_Data.Raw = 0;
    KL15_Data.Raw = 0;
    KL30_Data.Raw_Cnt = 0;
    KL15_Data.Raw_Cnt = 0;
}

void ADC_User_Data_Filter(ADC_User_Data_Struct *ADC_User, uint16_t Val_In)
{
    if (ADC_User->Raw_Cnt < sizeof(ADC_User->Raw_Buffer) / sizeof(uint16_t))
    {
        ADC_User->Raw_Buffer[ADC_User->Raw_Cnt] = Val_In;
        ADC_User->Raw_Cnt++;
    }
    else
    {
        // App_SortNWord_u16(ADC_User->Raw_Buffer, sizeof(ADC_User->Raw_Buffer)/sizeof(uint16_t));
        ADC_User->Raw = App_Average_u16((uint16_t *)(&ADC_User->Raw_Buffer[3]), (sizeof(ADC_User->Raw_Buffer) / sizeof(uint16_t) - 6));
        ADC_User->Raw_Cnt = 0;
    }
}

const static char *TAG = "EXAMPLE";

/*---------------------------------------------------------------
        ADC General Macros
---------------------------------------------------------------*/
// ADC1 Channels
#if CONFIG_IDF_TARGET_ESP32
#define EXAMPLE_ADC1_CHAN0 ADC_CHANNEL_4
#define EXAMPLE_ADC1_CHAN1 ADC_CHANNEL_5
#else
#define EXAMPLE_ADC1_CHAN0 ADC_CHANNEL_1
#define EXAMPLE_ADC1_CHAN1 ADC_CHANNEL_1
#endif

#if (SOC_ADC_PERIPH_NUM >= 2) && !CONFIG_IDF_TARGET_ESP32C3
/**
 * On ESP32C3, ADC2 is no longer supported, due to its HW limitation.
 * Search for errata on espressif website for more details.
 */
#define EXAMPLE_USE_ADC2 0
#endif

#define EXAMPLE_ADC_ATTEN ADC_ATTEN_DB_11

static int adc_raw[2][10];
static int voltage[2][10];
static bool example_adc_calibration_init(adc_unit_t unit, adc_atten_t atten, adc_cali_handle_t *out_handle);
static void example_adc_calibration_deinit(adc_cali_handle_t handle);

static adc_oneshot_unit_handle_t adc1_handle = NULL;
static adc_cali_handle_t adc1_cali_handle = NULL;
static bool do_calibration1;
uint8_t Ad_Valid = 0;
uint8_t Adc_Sleep_Mode;

void adc_oneshot_read_task(void *args)
{
    while (1)
    {
        ESP_ERROR_CHECK(adc_oneshot_read(adc1_handle, ADC_CHANNEL_1, &adc_raw[0][0]));
        if (do_calibration1)
        {
            ESP_ERROR_CHECK(adc_cali_raw_to_voltage(adc1_cali_handle, adc_raw[0][0], &voltage[0][0]));
        }

        if (Adc_Sleep_Mode)
        {
            KL30_Data.Raw = voltage[0][0]  * 34 / 10;
            vTaskDelay(pdMS_TO_TICKS(100));
        }
        else
        {
            ADC_User_Data_Filter((ADC_User_Data_Struct *)&KL30_Data, (uint16_t)(voltage[0][0] * 34 / 10));

            if (Ad_Valid == 0)
            {
                KL30_Data.Raw = voltage[0][1] * 34 / 10;
            }
            //  ESP_LOGI("ad","KL30 %d",KL30_Data.Raw);
            vTaskDelay(pdMS_TO_TICKS(2));

            Ad_Valid = 1;
        }
    }

    // Tear Down
    ESP_ERROR_CHECK(adc_oneshot_del_unit(adc1_handle));
    if (do_calibration1)
    {
        example_adc_calibration_deinit(adc1_cali_handle);
    }
}
/**
 * @brief 获取KL30电压
 * @return 0~0XFFFF
 */
uint16_t Get_KL30_Raw(void)
{
    return KL30_Data.Raw;
}
uint8_t Get_Ad_Valid(void)
{
    return Ad_Valid;
}

void adc_oneshot_read_init(void)
{
    //-------------ADC1 Init---------------//

    Ad_Valid = 0;
    Adc_Sleep_Mode = 0;
    ADC_User_Data_Reset();

    adc_oneshot_unit_init_cfg_t init_config1 = {
        .unit_id = ADC_UNIT_1,
    };
    ESP_ERROR_CHECK(adc_oneshot_new_unit(&init_config1, &adc1_handle));

    //-------------ADC1 Config---------------//
    adc_oneshot_chan_cfg_t config = {
        .bitwidth = ADC_BITWIDTH_DEFAULT,
        .atten = EXAMPLE_ADC_ATTEN,
    };
    ESP_ERROR_CHECK(adc_oneshot_config_channel(adc1_handle, ADC_CHANNEL_1, &config));

    //-------------ADC1 Calibration Init---------------//

    do_calibration1 = example_adc_calibration_init(ADC_UNIT_1, EXAMPLE_ADC_ATTEN, &adc1_cali_handle);
}

/*---------------------------------------------------------------
        ADC Calibration
---------------------------------------------------------------*/
static bool example_adc_calibration_init(adc_unit_t unit, adc_atten_t atten, adc_cali_handle_t *out_handle)
{
    adc_cali_handle_t handle = NULL;
    esp_err_t ret = ESP_FAIL;
    bool calibrated = false;

#if ADC_CALI_SCHEME_CURVE_FITTING_SUPPORTED
    if (!calibrated)
    {
        // ESP_LOGI(TAG, "calibration scheme version is %s", "Curve Fitting");
        adc_cali_curve_fitting_config_t cali_config = {
            .unit_id = unit,
            .atten = atten,
            .bitwidth = ADC_BITWIDTH_DEFAULT,
        };
        ret = adc_cali_create_scheme_curve_fitting(&cali_config, &handle);
        if (ret == ESP_OK)
        {
            calibrated = true;
        }
    }
#endif

#if ADC_CALI_SCHEME_LINE_FITTING_SUPPORTED
    if (!calibrated)
    {
        // ESP_LOGI(TAG, "calibration scheme version is %s", "Line Fitting");
        adc_cali_line_fitting_config_t cali_config = {
            .unit_id = unit,
            .atten = atten,
            .bitwidth = ADC_BITWIDTH_DEFAULT,
        };
        ret = adc_cali_create_scheme_line_fitting(&cali_config, &handle);
        if (ret == ESP_OK)
        {
            calibrated = true;
        }
    }
#endif

    *out_handle = handle;
    if (ret == ESP_OK)
    {
        ESP_LOGI(TAG, "Calibration Success");
    }
    else if (ret == ESP_ERR_NOT_SUPPORTED || !calibrated)
    {
        ESP_LOGW(TAG, "eFuse not burnt, skip software calibration");
    }
    else
    {
        ESP_LOGE(TAG, "Invalid arg or no memory");
    }

    return calibrated;
}

static void example_adc_calibration_deinit(adc_cali_handle_t handle)
{
#if ADC_CALI_SCHEME_CURVE_FITTING_SUPPORTED
    // ESP_LOGI(TAG, "deregister %s calibration scheme", "Curve Fitting");
    ESP_ERROR_CHECK(adc_cali_delete_scheme_curve_fitting(handle));

#elif ADC_CALI_SCHEME_LINE_FITTING_SUPPORTED
    // ESP_LOGI(TAG, "deregister %s calibration scheme", "Line Fitting");
    ESP_ERROR_CHECK(adc_cali_delete_scheme_line_fitting(handle));
#endif
}
