/*
 * SPDX-FileCopyrightText: 2021 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Unlicense OR CC0-1.0
 */

/****************************************************************************
 *
 * This demo showcases creating a GATT database using a predefined attribute table.
 * It acts as a GATT server and can send adv data, be connected by client.
 * Run the gatt_client demo, the client demo will automatically connect to the gatt_server_service_table demo.
 * Client demo will enable GATT server's notify after connection. The two devices will then exchange
 * data.
 *
 ****************************************************************************/

#include "freertos/FreeRTOS.h"
#include "freertos/task.h"
#include "freertos/event_groups.h"
#include "esp_system.h"
#include "esp_log.h"
#include "nvs_flash.h"
#include "esp_bt.h"

#include "esp_gap_ble_api.h"
#include "esp_gatts_api.h"
#include "esp_bt_main.h"
#include "gatts_table_creat_demo.h"
#include "esp_gatt_common_api.h"
// #include "bsp_UART.h"

#include <string.h>
// #include "CAN_CH0_CAN_Communication_Matrix.h"
// #include "Services_Mileage_User.h"

#define GATTS_TABLE_TAG "GATTS_TABLE_DEMO"

#define PROFILE_NUM 1
#define PROFILE_APP_IDX 0
#define ESP_APP_ID 0x55
#define SAMPLE_DEVICE_NAME u8BLEdevicename
#define SVC_INST_ID 0

/* The max length of characteristic value. When the GATT client performs a write or prepare write operation,
 *  the data length must be less than GATTS_DEMO_CHAR_VAL_LEN_MAX.
 */
#define GATTS_DEMO_CHAR_VAL_LEN_MAX 512
#define PREPARE_BUF_MAX_SIZE 1024
#define CHAR_DECLARATION_SIZE (sizeof(uint8_t))

#define ADV_CONFIG_FLAG (1 << 0)
#define SCAN_RSP_CONFIG_FLAG (1 << 1)

#define BLE_MAC_BLOCK EEPROM_BLOCK_12

static uint8_t adv_config_done = 0;

uint16_t heart_rate_handle_table[HRS_IDX_NB];

typedef struct
{
    uint8_t *prepare_buf;
    int prepare_len;
} prepare_type_env_t;

static prepare_type_env_t prepare_write_env;

static esp_bd_addr_t spp_remote_bda = {0x0};

uint8_t u8BLEdevicename[] =
    {

        'Y', 'D', '4', '5', '1', '5', '0', '0', '0', '8', '\0'};

#define BLE_MAC_LEN 7
uint32_t u32BLEtime = 0;
uint8_t u8BLE_MAC_Addr_Backup[BLE_MAC_LEN] =
    {
        0xC1, 0X80, 0x45, 0x15, 0x00, 0x08, '\0'};
uint8_t u8BLE_MAC_Addr[BLE_MAC_LEN] =
    {
        0xC1, 0X80, 0x45, 0x15, 0x00, 0x08, '\0'};
QueueHandle_t ble_rx_que;
uint32_t u32BleConnect = 0;
uint32_t u32BleAdv = 0;

#define BLE_RX_BUF_LEN 200
#define BLE_RX_DATA_LEN 175

typedef struct
{
    uint8_t BleData[BLE_RX_BUF_LEN - 1];
    uint8_t length;
} BleDataPacket;

#define CONFIG_SET_RAW_ADV_DATA
#ifdef CONFIG_SET_RAW_ADV_DATA
static uint8_t raw_adv_data[] = { // 广播
    /* flags */
    // 0x02, 0x01, 0x06,
    /* tx power */
    //  0x02, 0x0a, 0xeb,
    /* service uuid */
    // 0x03, 0x02, 0x12, 0x18,
    // 0x05, 0x04, 0xD2, 0x0B,0x81, 0xE7,

    // 0x03, 0x19, 0x40, 0x02,

    //  0x0B, 0x09, 'Y', 'D', '4', '5', '1', '5', '0', '0', '0', '1',
    //  len  type
    0x1A, 0XFF,
    // 蓝牙供应商 ID
    0xFF, 0xFF, 0xFF, 0xFF,
    // 蓝牙 Mac 地址
    // 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
    0xC1, 0X80, 0x45, 0x15, 0x00, 0x01,
    // 保留
    0xFF, 0xFF,
    // 次通信协议版本
    0x01,
    // 主通信协议版本
    0x00,
    // 次固件版本号
    0x13,
    // 主固件版本号
    0x01,
    // 保留
    0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF};

static uint8_t raw_scan_rsp_data[] = { // 广播回复时
    /* flags */
    0x02, 0x01, 0x06,
    /* tx power */
    //  0x02, 0x0a, 0xeb,
    /* service uuid */
    0x03, 0x02, 0x12, 0x18,
    0x05, 0x04, 0xD2, 0x0B, 0x81, 0xE7,

    0x03, 0x19, 0x40, 0x02,

    0x0B, 0x09, 'Y', 'D', '4', '5', '1', '5', '0', '0', '0', '8'

};

#else
static uint8_t service_uuid[16] = {
    /* LSB <--------------------------------------------------------------------------------> MSB */
    // first uuid, 16bit, [12],[13] is the value
    0xfb,
    0x34,
    0x9b,
    0x5f,
    0x80,
    0x00,
    0x00,
    0x80,
    0x00,
    0x10,
    0x00,
    0x00,
    0xFF,
    0x00,
    0x00,
    0x00,
};

/* The length of adv data must be less than 31 bytes */
static esp_ble_adv_data_t adv_data = {
    .set_scan_rsp = false,
    .include_name = true,
    .include_txpower = true,
    .min_interval = 0x0006, // slave connection min interval, Time = min_interval * 1.25 msec
    .max_interval = 0x0010, // slave connection max interval, Time = max_interval * 1.25 msec
    .appearance = 0x00,
    .manufacturer_len = 0,       // TEST_MANUFACTURER_DATA_LEN,
    .p_manufacturer_data = NULL, // test_manufacturer,
    .service_data_len = 0,
    .p_service_data = NULL,
    .service_uuid_len = sizeof(service_uuid),
    .p_service_uuid = service_uuid,
    .flag = (ESP_BLE_ADV_FLAG_GEN_DISC | ESP_BLE_ADV_FLAG_BREDR_NOT_SPT),
};

// scan response data
static esp_ble_adv_data_t scan_rsp_data = {
    .set_scan_rsp = true,
    .include_name = true,
    .include_txpower = true,
    .min_interval = 0x0006,
    .max_interval = 0x0010,
    .appearance = 0x00,
    .manufacturer_len = 0,       // TEST_MANUFACTURER_DATA_LEN,
    .p_manufacturer_data = NULL, //&test_manufacturer[0],
    .service_data_len = 0,
    .p_service_data = NULL,
    .service_uuid_len = sizeof(service_uuid),
    .p_service_uuid = service_uuid,
    .flag = (ESP_BLE_ADV_FLAG_GEN_DISC | ESP_BLE_ADV_FLAG_BREDR_NOT_SPT),
};
#endif /* CONFIG_SET_RAW_ADV_DATA */

static esp_ble_adv_params_t adv_params = {
    .adv_int_min = 0x20,
    .adv_int_max = 0x40,
    .adv_type = ADV_TYPE_IND,
    .own_addr_type = BLE_ADDR_TYPE_RANDOM, // BLE_ADDR_TYPE_PUBLIC, // 公共
    .channel_map = ADV_CHNL_ALL,
    .adv_filter_policy = ADV_FILTER_ALLOW_SCAN_ANY_CON_ANY,
};

struct gatts_profile_inst
{
    esp_gatts_cb_t gatts_cb;
    uint16_t gatts_if;
    uint16_t app_id;
    uint16_t conn_id;
    uint16_t service_handle;
    esp_gatt_srvc_id_t service_id;
    uint16_t char_handle;
    esp_bt_uuid_t char_uuid;
    esp_gatt_perm_t perm;
    esp_gatt_char_prop_t property;
    uint16_t descr_handle;
    esp_bt_uuid_t descr_uuid;
};

static void gatts_profile_event_handler(esp_gatts_cb_event_t event,
                                        esp_gatt_if_t gatts_if, esp_ble_gatts_cb_param_t *param);

/* One gatt-based profile one app_id and one gatts_if, this array will store the gatts_if returned by ESP_GATTS_REG_EVT */
static struct gatts_profile_inst heart_rate_profile_tab[PROFILE_NUM] = {
    [PROFILE_APP_IDX] = {
        .gatts_cb = gatts_profile_event_handler,
        .gatts_if = ESP_GATT_IF_NONE, /* Not get the gatt_if, so initial is ESP_GATT_IF_NONE */
    },
};

static const uint8_t ServiceUUID[16] = {0xF2, 0xC3, 0xF0, 0xAE, 0xA9, 0xFA, 0x15, 0x8C, 0x9D, 0x49, 0xAE, 0x73, 0xD2, 0x0B, 0x81, 0xE7};
static const uint8_t CharWriteUUID[16] = {0x9F, 0x9F, 0x00, 0xC1, 0x58, 0xBD, 0x32, 0xB6, 0x9E, 0x4C, 0x21, 0x9C, 0x21, 0xE8, 0xF8, 0xBE};
// static const uint8_t CharNotifyUUID[16] = {0x9F, 0x9F, 0x00, 0xC1, 0x58, 0xBD, 0x32, 0xB6, 0x9E, 0x4C, 0x21, 0x9C, 0x20, 0xE8, 0xF8, 0xBE};

/* Service */
static const uint16_t GATTS_SERVICE_UUID_TEST = 0x00FF;
static const uint16_t GATTS_CHAR_UUID_TEST_A = 0xFF01;
static const uint16_t GATTS_CHAR_UUID_TEST_B = 0xFF02;
static const uint16_t GATTS_CHAR_UUID_TEST_C = 0xFF03;

static const uint16_t primary_service_uuid = ESP_GATT_UUID_PRI_SERVICE;
static const uint16_t character_declaration_uuid = ESP_GATT_UUID_CHAR_DECLARE;
static const uint16_t character_client_config_uuid = ESP_GATT_UUID_CHAR_CLIENT_CONFIG;
static const uint8_t char_prop_read = ESP_GATT_CHAR_PROP_BIT_READ;
static const uint8_t char_prop_write = ESP_GATT_CHAR_PROP_BIT_WRITE;
static const uint8_t char_prop_read_write_notify = ESP_GATT_CHAR_PROP_BIT_WRITE | ESP_GATT_CHAR_PROP_BIT_READ | ESP_GATT_CHAR_PROP_BIT_NOTIFY;
static const uint8_t heart_measurement_ccc[2] = {0x00, 0x00};
static const uint8_t char_value[4] = {0x11, 0x22, 0x33, 0x44};

/* Full Database Description - Used to add attributes into the database */
static const esp_gatts_attr_db_t gatt_db[HRS_IDX_NB] =
    {
        // Service Declaration
        [IDX_SVC] =
            {{ESP_GATT_AUTO_RSP}, {ESP_UUID_LEN_16, (uint8_t *)&primary_service_uuid, ESP_GATT_PERM_READ, sizeof(uint16_t), sizeof(ServiceUUID), (uint8_t *)&ServiceUUID}},

        /* Characteristic Declaration */
        [IDX_CHAR_A] =
            {{ESP_GATT_AUTO_RSP}, {ESP_UUID_LEN_16, (uint8_t *)&character_declaration_uuid, ESP_GATT_PERM_READ, CHAR_DECLARATION_SIZE, CHAR_DECLARATION_SIZE, (uint8_t *)&char_prop_write}},

        /* Characteristic Value */
        [IDX_CHAR_VAL_A] =
            {{ESP_GATT_AUTO_RSP}, {ESP_UUID_LEN_128, (uint8_t *)&CharWriteUUID, ESP_GATT_PERM_WRITE, GATTS_DEMO_CHAR_VAL_LEN_MAX, sizeof(char_value), (uint8_t *)char_value}},

        // /* Client Characteristic Configuration Descriptor */
        // [IDX_CHAR_CFG_A] =
        //     {{ESP_GATT_AUTO_RSP}, {ESP_UUID_LEN_16, (uint8_t *)&character_client_config_uuid, ESP_GATT_PERM_READ | ESP_GATT_PERM_WRITE, sizeof(uint16_t), sizeof(heart_measurement_ccc), (uint8_t *)heart_measurement_ccc}},

        // /* Characteristic Declaration */
        // [IDX_CHAR_B] =
        //     {{ESP_GATT_AUTO_RSP}, {ESP_UUID_LEN_16, (uint8_t *)&character_declaration_uuid, ESP_GATT_PERM_READ, CHAR_DECLARATION_SIZE, CHAR_DECLARATION_SIZE, (uint8_t *)&char_prop_read}},

        // /* Characteristic Value */
        // [IDX_CHAR_VAL_B] =
        //     {{ESP_GATT_AUTO_RSP}, {ESP_UUID_LEN_16, (uint8_t *)&GATTS_CHAR_UUID_TEST_B, ESP_GATT_PERM_READ | ESP_GATT_PERM_WRITE, GATTS_DEMO_CHAR_VAL_LEN_MAX, sizeof(char_value), (uint8_t *)char_value}},

        // /* Characteristic Declaration */
        // [IDX_CHAR_C] =
        //     {{ESP_GATT_AUTO_RSP}, {ESP_UUID_LEN_16, (uint8_t *)&character_declaration_uuid, ESP_GATT_PERM_READ, CHAR_DECLARATION_SIZE, CHAR_DECLARATION_SIZE, (uint8_t *)&char_prop_write}},

        // /* Characteristic Value */
        // [IDX_CHAR_VAL_C] =
        //     {{ESP_GATT_AUTO_RSP}, {ESP_UUID_LEN_16, (uint8_t *)&GATTS_CHAR_UUID_TEST_C, ESP_GATT_PERM_READ | ESP_GATT_PERM_WRITE, GATTS_DEMO_CHAR_VAL_LEN_MAX, sizeof(char_value), (uint8_t *)char_value}},

};

static void gap_event_handler(esp_gap_ble_cb_event_t event, esp_ble_gap_cb_param_t *param)
{
    switch (event)
    {
#ifdef CONFIG_SET_RAW_ADV_DATA
    case ESP_GAP_BLE_ADV_DATA_RAW_SET_COMPLETE_EVT:
        adv_config_done &= (~ADV_CONFIG_FLAG);
        if (adv_config_done == 0)
        {
            esp_ble_gap_start_advertising(&adv_params);
        }
        break;
    case ESP_GAP_BLE_SCAN_RSP_DATA_RAW_SET_COMPLETE_EVT:
        adv_config_done &= (~SCAN_RSP_CONFIG_FLAG);
        if (adv_config_done == 0)
        {
            esp_ble_gap_start_advertising(&adv_params);
        }
        break;
#else
    case ESP_GAP_BLE_ADV_DATA_SET_COMPLETE_EVT:
        adv_config_done &= (~ADV_CONFIG_FLAG);
        if (adv_config_done == 0)
        {
            esp_ble_gap_start_advertising(&adv_params);
        }
        break;
    case ESP_GAP_BLE_SCAN_RSP_DATA_SET_COMPLETE_EVT:
        adv_config_done &= (~SCAN_RSP_CONFIG_FLAG);
        if (adv_config_done == 0)
        {
            esp_ble_gap_start_advertising(&adv_params);
        }
        break;
#endif
    case ESP_GAP_BLE_ADV_START_COMPLETE_EVT:
        /* advertising start complete event to indicate advertising start successfully or failed */
        if (param->adv_start_cmpl.status != ESP_BT_STATUS_SUCCESS)
        {
            ESP_LOGE(GATTS_TABLE_TAG, "advertising start failed");
        }
        else
        {
            ESP_LOGI(GATTS_TABLE_TAG, "advertising start successfully");
            u32BleAdv = 1;
        }
        break;
    case ESP_GAP_BLE_ADV_STOP_COMPLETE_EVT:
        if (param->adv_stop_cmpl.status != ESP_BT_STATUS_SUCCESS)
        {
            ESP_LOGE(GATTS_TABLE_TAG, "Advertising stop failed");
        }
        else
        {
            ESP_LOGI(GATTS_TABLE_TAG, "Stop adv successfully\n");
            u32BleAdv = 0;
        }
        break;
    case ESP_GAP_BLE_UPDATE_CONN_PARAMS_EVT:
        ESP_LOGI(GATTS_TABLE_TAG, "update connection params status = %d, min_int = %d, max_int = %d,conn_int = %d,latency = %d, timeout = %d",
                 param->update_conn_params.status,
                 param->update_conn_params.min_int,
                 param->update_conn_params.max_int,
                 param->update_conn_params.conn_int,
                 param->update_conn_params.latency,
                 param->update_conn_params.timeout);
        break;
    default:
        break;
    }
}

void example_prepare_write_event_env(esp_gatt_if_t gatts_if, prepare_type_env_t *prepare_write_env, esp_ble_gatts_cb_param_t *param)
{
    ESP_LOGI(GATTS_TABLE_TAG, "prepare write, handle = %d, value len = %d", param->write.handle, param->write.len);
    esp_gatt_status_t status = ESP_GATT_OK;
    if (prepare_write_env->prepare_buf == NULL)
    {
        prepare_write_env->prepare_buf = (uint8_t *)malloc(PREPARE_BUF_MAX_SIZE * sizeof(uint8_t));
        prepare_write_env->prepare_len = 0;
        if (prepare_write_env->prepare_buf == NULL)
        {
            ESP_LOGE(GATTS_TABLE_TAG, "%s, Gatt_server prep no mem", __func__);
            status = ESP_GATT_NO_RESOURCES;
        }
    }
    else
    {
        if (param->write.offset > PREPARE_BUF_MAX_SIZE)
        {
            status = ESP_GATT_INVALID_OFFSET;
        }
        else if ((param->write.offset + param->write.len) > PREPARE_BUF_MAX_SIZE)
        {
            status = ESP_GATT_INVALID_ATTR_LEN;
        }
    }
    /*send response when param->write.need_rsp is true */
    if (param->write.need_rsp)
    {
        esp_gatt_rsp_t *gatt_rsp = (esp_gatt_rsp_t *)malloc(sizeof(esp_gatt_rsp_t));
        if (gatt_rsp != NULL)
        {
            gatt_rsp->attr_value.len = param->write.len;
            gatt_rsp->attr_value.handle = param->write.handle;
            gatt_rsp->attr_value.offset = param->write.offset;
            gatt_rsp->attr_value.auth_req = ESP_GATT_AUTH_REQ_NONE;
            memcpy(gatt_rsp->attr_value.value, param->write.value, param->write.len);
            esp_err_t response_err = esp_ble_gatts_send_response(gatts_if, param->write.conn_id, param->write.trans_id, status, gatt_rsp);
            if (response_err != ESP_OK)
            {
                ESP_LOGE(GATTS_TABLE_TAG, "Send response error");
            }
            free(gatt_rsp);
        }
        else
        {
            ESP_LOGE(GATTS_TABLE_TAG, "%s, malloc failed", __func__);
        }
    }
    if (status != ESP_GATT_OK)
    {
        return;
    }
    memcpy(prepare_write_env->prepare_buf + param->write.offset,
           param->write.value,
           param->write.len);
    prepare_write_env->prepare_len += param->write.len;
}

void example_exec_write_event_env(prepare_type_env_t *prepare_write_env, esp_ble_gatts_cb_param_t *param)
{
    if (param->exec_write.exec_write_flag == ESP_GATT_PREP_WRITE_EXEC && prepare_write_env->prepare_buf)
    {
        esp_log_buffer_hex(GATTS_TABLE_TAG, prepare_write_env->prepare_buf, prepare_write_env->prepare_len);
    }
    else
    {
        ESP_LOGI(GATTS_TABLE_TAG, "ESP_GATT_PREP_WRITE_CANCEL");
    }
    if (prepare_write_env->prepare_buf)
    {
        free(prepare_write_env->prepare_buf);
        prepare_write_env->prepare_buf = NULL;
    }
    prepare_write_env->prepare_len = 0;
}

static void gatts_profile_event_handler(esp_gatts_cb_event_t event, esp_gatt_if_t gatts_if, esp_ble_gatts_cb_param_t *param)
{
    switch (event)
    {
    case ESP_GATTS_REG_EVT:
    {
        esp_err_t set_dev_name_ret = esp_ble_gap_set_device_name(SAMPLE_DEVICE_NAME); // u8BLE_MAC_Addr_Backup
        if (set_dev_name_ret)
        {
            ESP_LOGE(GATTS_TABLE_TAG, "set device name failed, error code = %x", set_dev_name_ret);
        }
#ifdef CONFIG_SET_RAW_ADV_DATA
        esp_err_t raw_adv_ret = esp_ble_gap_config_adv_data_raw(raw_adv_data, sizeof(raw_adv_data)); //
        if (raw_adv_ret)
        {
            ESP_LOGE(GATTS_TABLE_TAG, "config raw adv data failed, error code = %x ", raw_adv_ret);
        }
        adv_config_done |= ADV_CONFIG_FLAG;
        esp_err_t raw_scan_ret = esp_ble_gap_config_scan_rsp_data_raw(raw_scan_rsp_data, sizeof(raw_scan_rsp_data));
        if (raw_scan_ret)
        {
            ESP_LOGE(GATTS_TABLE_TAG, "config raw scan rsp data failed, error code = %x", raw_scan_ret);
        }
        adv_config_done |= SCAN_RSP_CONFIG_FLAG;
#else
        // config adv data
        esp_err_t ret = esp_ble_gap_config_adv_data(&adv_data);
        if (ret)
        {
            ESP_LOGE(GATTS_TABLE_TAG, "config adv data failed, error code = %x", ret);
        }
        adv_config_done |= ADV_CONFIG_FLAG;
        // config scan response data
        ret = esp_ble_gap_config_adv_data(&scan_rsp_data);
        if (ret)
        {
            ESP_LOGE(GATTS_TABLE_TAG, "config scan response data failed, error code = %x", ret);
        }
        adv_config_done |= SCAN_RSP_CONFIG_FLAG;
#endif
        esp_err_t create_attr_ret = esp_ble_gatts_create_attr_tab(gatt_db, gatts_if, HRS_IDX_NB, SVC_INST_ID);
        if (create_attr_ret)
        {
            ESP_LOGE(GATTS_TABLE_TAG, "create attr table failed, error code = %x", create_attr_ret);
        }
    }
    break;
    case ESP_GATTS_READ_EVT: // 设备发送出去//主机来读时
        ESP_LOGI(GATTS_TABLE_TAG, "ESP_GATTS_READ_EVT");
        break;
    case ESP_GATTS_WRITE_EVT: // 设备接收//主机发送过来
        if (!param->write.is_prep)
        {
            // the data length of gattc write  must be less than GATTS_DEMO_CHAR_VAL_LEN_MAX.
            ESP_LOGI(GATTS_TABLE_TAG, "GATT_WRITE_EVT, handle = %d, value len = %d, value :", param->write.handle, param->write.len);
            esp_log_buffer_hex(GATTS_TABLE_TAG, param->write.value, param->write.len);
            // /*IDX_CHAR_VAL_A*/ /*通知*/
            // if (heart_rate_handle_table[IDX_CHAR_CFG_A] == param->write.handle && param->write.len == 2)
            // {
            //     uint16_t descr_value = param->write.value[1] << 8 | param->write.value[0];
            //     if (descr_value == 0x0001)
            //     {
            //         ESP_LOGI(GATTS_TABLE_TAG, "notify enable");
            //         uint8_t notify_data[15];
            //         for (int i = 0; i < sizeof(notify_data); ++i)
            //         {
            //             notify_data[i] = i % 0xff;
            //         }
            //         // the size of notify_data[] need less than MTU size
            //         esp_ble_gatts_send_indicate(gatts_if, param->write.conn_id, heart_rate_handle_table[IDX_CHAR_VAL_A],
            //                                     sizeof(notify_data), notify_data, false);
            //     }
            //     else if (descr_value == 0x0002)
            //     {
            //         ESP_LOGI(GATTS_TABLE_TAG, "indicate enable");
            //         uint8_t indicate_data[15];
            //         for (int i = 0; i < sizeof(indicate_data); ++i)
            //         {
            //             indicate_data[i] = i % 0xff;
            //         }
            //         // the size of indicate_data[] need less than MTU size
            //         esp_ble_gatts_send_indicate(gatts_if, param->write.conn_id, heart_rate_handle_table[IDX_CHAR_VAL_A],
            //                                     sizeof(indicate_data), indicate_data, true);
            //     }
            //     else if (descr_value == 0x0000)
            //     {
            //         ESP_LOGI(GATTS_TABLE_TAG, "notify/indicate disable ");
            //     }
            //     else
            //     {
            //         ESP_LOGE(GATTS_TABLE_TAG, "unknown descr value");
            //         esp_log_buffer_hex(GATTS_TABLE_TAG, param->write.value, param->write.len);
            //     }
            // }
            if (heart_rate_handle_table[IDX_CHAR_VAL_A] == param->write.handle)
            {
                // esp_log_buffer_hex(GATTS_TABLE_TAG, param->write.value, param->write.len);
                printf("ble test \n");
                BleDataPacket stBleDataPacket;

                stBleDataPacket.length = param->write.len;

                memcpy(stBleDataPacket.BleData, param->write.value, stBleDataPacket.length);

                xQueueSend(ble_rx_que, &stBleDataPacket, 0);
            }
            /* send response when param->write.need_rsp is true*/
            if (param->write.need_rsp)
            {
                esp_ble_gatts_send_response(gatts_if, param->write.conn_id, param->write.trans_id, ESP_GATT_OK, NULL);
            }
        }
        else
        {
            /* handle prepare write */
            example_prepare_write_event_env(gatts_if, &prepare_write_env, param);
        }
        break;
    case ESP_GATTS_EXEC_WRITE_EVT:
        // the length of gattc prepare write data must be less than GATTS_DEMO_CHAR_VAL_LEN_MAX.
        ESP_LOGI(GATTS_TABLE_TAG, "ESP_GATTS_EXEC_WRITE_EVT");
        example_exec_write_event_env(&prepare_write_env, param);
        break;
    case ESP_GATTS_MTU_EVT: // 吞吐量
        ESP_LOGI(GATTS_TABLE_TAG, "ESP_GATTS_MTU_EVT, MTU %d", param->mtu.mtu);
        break;
    case ESP_GATTS_CONF_EVT: // config
        ESP_LOGI(GATTS_TABLE_TAG, "ESP_GATTS_CONF_EVT, status = %d, attr_handle %d", param->conf.status, param->conf.handle);
        break;
    case ESP_GATTS_START_EVT:
        ESP_LOGI(GATTS_TABLE_TAG, "SERVICE_START_EVT, status %d, service_handle %d", param->start.status, param->start.service_handle);
        break;
    case ESP_GATTS_CONNECT_EVT: // 连接成功
        ESP_LOGI(GATTS_TABLE_TAG, "ESP_GATTS_CONNECT_EVT, conn_id = %d", param->connect.conn_id);
        esp_log_buffer_hex(GATTS_TABLE_TAG, param->connect.remote_bda, 6);
        esp_ble_conn_update_params_t conn_params = {0};
        memcpy(conn_params.bda, param->connect.remote_bda, sizeof(esp_bd_addr_t)); // mac
        /* For the iOS system, please refer to Apple official documents about the BLE connection parameters restrictions. */
        conn_params.latency = 0;
        conn_params.max_int = 0x20; // max_int = 0x20*1.25ms = 40ms
        conn_params.min_int = 0x10; // min_int = 0x10*1.25ms = 20ms
        conn_params.timeout = 400;  // timeout = 400*10ms = 4000ms
        // start sent the update connection parameters to the peer device.
        esp_ble_gap_update_conn_params(&conn_params);
        u32BleConnect = 1;
        u32BleAdv = 0;
        memcpy(spp_remote_bda, param->connect.remote_bda, sizeof(esp_bd_addr_t));
        break;
    case ESP_GATTS_DISCONNECT_EVT:
        memset(spp_remote_bda, 0, sizeof(esp_bd_addr_t));
        ESP_LOGI(GATTS_TABLE_TAG, "ESP_GATTS_DISCONNECT_EVT, reason = 0x%x", param->disconnect.reason);
        if (param->disconnect.reason != ESP_GATT_CONN_TERMINATE_LOCAL_HOST)
        {
            esp_ble_gap_start_advertising(&adv_params);
        }

        u32BleConnect = 0;

        break;
    case ESP_GATTS_CREAT_ATTR_TAB_EVT:
    {
        if (param->add_attr_tab.status != ESP_GATT_OK)
        {
            ESP_LOGE(GATTS_TABLE_TAG, "create attribute table failed, error code=0x%x", param->add_attr_tab.status);
        }
        else if (param->add_attr_tab.num_handle != HRS_IDX_NB)
        {
            ESP_LOGE(GATTS_TABLE_TAG, "create attribute table abnormally, num_handle (%d) \
                        doesn't equal to HRS_IDX_NB(%d)",
                     param->add_attr_tab.num_handle, HRS_IDX_NB);
        }
        else
        {
            ESP_LOGI(GATTS_TABLE_TAG, "create attribute table successfully, the number handle = %d\n", param->add_attr_tab.num_handle);
            memcpy(heart_rate_handle_table, param->add_attr_tab.handles, sizeof(heart_rate_handle_table));
            esp_ble_gatts_start_service(heart_rate_handle_table[IDX_SVC]);
        }
        break;
    }
    case ESP_GATTS_STOP_EVT:
    case ESP_GATTS_OPEN_EVT:
    case ESP_GATTS_CANCEL_OPEN_EVT:
    case ESP_GATTS_CLOSE_EVT:
    case ESP_GATTS_LISTEN_EVT:
    case ESP_GATTS_CONGEST_EVT:
    case ESP_GATTS_UNREG_EVT:
    case ESP_GATTS_DELETE_EVT:
    default:
        break;
    }
}

static void gatts_event_handler(esp_gatts_cb_event_t event, esp_gatt_if_t gatts_if, esp_ble_gatts_cb_param_t *param)
{

    /* If event is register event, store the gatts_if for each profile */
    if (event == ESP_GATTS_REG_EVT) // 注册
    {
        if (param->reg.status == ESP_GATT_OK)
        {
            heart_rate_profile_tab[PROFILE_APP_IDX].gatts_if = gatts_if; // 3
        }
        else
        {
            ESP_LOGE(GATTS_TABLE_TAG, "reg app failed, app_id %04x, status %d",
                     param->reg.app_id,
                     param->reg.status);
            return;
        }
    }
    do
    {
        int idx;
        for (idx = 0; idx < PROFILE_NUM; idx++)
        {
            /* ESP_GATT_IF_NONE, not specify a certain gatt_if, need to call every profile cb function */
            if (gatts_if == ESP_GATT_IF_NONE || gatts_if == heart_rate_profile_tab[idx].gatts_if)
            {
                if (heart_rate_profile_tab[idx].gatts_cb)
                {
                    heart_rate_profile_tab[idx].gatts_cb(event, gatts_if, param);
                }
            }
        }
    } while (0);
}

void BLE_Variate_Init(void)
{
    // uint8_t i = 0;

    // u32BleConnect = 0;
    // u32BleAdv = 0;

    // // for (i = 0; i < BLE_MAC_LEN; i++)
    // // {
    // //     u8BLE_MAC_Addr_Backup[i] = u8BLE_MAC_Addr[i];
    // // }

    ble_rx_que = xQueueCreate(5, BLE_RX_BUF_LEN);

    // eeprom_ReadRecord(BLE_MAC_BLOCK, u8BLE_MAC_Addr_Backup, 6);

    // if ((u8BLE_MAC_Addr_Backup[0] == 0xFF) && (u8BLE_MAC_Addr_Backup[1] == 0xFF) &&
    //     (u8BLE_MAC_Addr_Backup[2] == 0xFF) && (u8BLE_MAC_Addr_Backup[3] == 0xFF) &&
    //     (u8BLE_MAC_Addr_Backup[4] == 0xFF) && (u8BLE_MAC_Addr_Backup[5] == 0xFF))
    // {
    //     u8BLE_MAC_Addr_Backup[0] = 0xC1;
    //     u8BLE_MAC_Addr_Backup[1] = 0x80;
    //     u8BLE_MAC_Addr_Backup[2] = 0x45;
    //     u8BLE_MAC_Addr_Backup[3] = 0x15;
    //     u8BLE_MAC_Addr_Backup[4] = 0x00;
    //     u8BLE_MAC_Addr_Backup[5] = 0x01;

    //     eeprom_WriteRecord(BLE_MAC_BLOCK, u8BLE_MAC_Addr_Backup, 6);
    // }
    // for (i = 0; i < BLE_MAC_LEN; i++)
    // {
    //     u8BLE_MAC_Addr[i] = u8BLE_MAC_Addr_Backup[i];
    // }

    // raw_adv_data[6] = u8BLE_MAC_Addr_Backup[0];
    // raw_adv_data[7] = u8BLE_MAC_Addr_Backup[1];
    // raw_adv_data[8] = u8BLE_MAC_Addr_Backup[2];
    // raw_adv_data[9] = u8BLE_MAC_Addr_Backup[3];
    // raw_adv_data[10] = u8BLE_MAC_Addr_Backup[4];
    // raw_adv_data[11] = u8BLE_MAC_Addr_Backup[5];

    // u8BLEdevicename[2] = u8BLE_MAC_Addr_Backup[2] / 0x10 + 0X30;
    // u8BLEdevicename[3] = u8BLE_MAC_Addr_Backup[2] % 0x10 + 0X30;
    // u8BLEdevicename[4] = u8BLE_MAC_Addr_Backup[3] / 0x10 + 0X30;
    // u8BLEdevicename[5] = u8BLE_MAC_Addr_Backup[3] % 0x10 + 0X30;
    // u8BLEdevicename[6] = u8BLE_MAC_Addr_Backup[4] / 0x10 + 0X30;
    // u8BLEdevicename[7] = u8BLE_MAC_Addr_Backup[4] % 0x10 + 0X30;
    // u8BLEdevicename[8] = u8BLE_MAC_Addr_Backup[5] / 0x10 + 0X30;
    // u8BLEdevicename[9] = u8BLE_MAC_Addr_Backup[5] % 0x10 + 0X30;

    // raw_scan_rsp_data[21] = u8BLE_MAC_Addr_Backup[2] / 0x10 + 0X30;
    // raw_scan_rsp_data[22] = u8BLE_MAC_Addr_Backup[2] % 0x10 + 0X30;
    // raw_scan_rsp_data[23] = u8BLE_MAC_Addr_Backup[3] / 0x10 + 0X30;
    // raw_scan_rsp_data[24] = u8BLE_MAC_Addr_Backup[3] % 0x10 + 0X30;
    // raw_scan_rsp_data[25] = u8BLE_MAC_Addr_Backup[4] / 0x10 + 0X30;
    // raw_scan_rsp_data[26] = u8BLE_MAC_Addr_Backup[4] % 0x10 + 0X30;
    // raw_scan_rsp_data[27] = u8BLE_MAC_Addr_Backup[5] / 0x10 + 0X30;
    // raw_scan_rsp_data[28] = u8BLE_MAC_Addr_Backup[5] % 0x10 + 0X30;

    if (!ble_rx_que)
    {
        return;
    }
}
void BLE_Init(void)
{
    esp_err_t ret;

    BLE_Variate_Init();

    /************************************************************/

    /* Initialize NVS. */
    // ret = nvs_flash_init();
    // if (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND)
    // {
    //     ESP_ERROR_CHECK(nvs_flash_erase());
    //     ret = nvs_flash_init();
    // }
    // ESP_ERROR_CHECK(ret);

    // ESP_ERROR_CHECK(esp_bt_controller_mem_release(ESP_BT_MODE_CLASSIC_BT));

    // esp_bt_controller_config_t bt_cfg = BT_CONTROLLER_INIT_CONFIG_DEFAULT();
    // ret = esp_bt_controller_init(&bt_cfg);
    // if (ret)
    // {
    //     ESP_LOGE(GATTS_TABLE_TAG, "%s enable controller failed: %s", __func__, esp_err_to_name(ret));
    //     return;
    // }

    // ret = esp_bt_controller_enable(ESP_BT_MODE_BTDM);
    // if (ret)
    // {
    //     ESP_LOGE(GATTS_TABLE_TAG, "%s enable controller failed: %s", __func__, esp_err_to_name(ret));
    //     return;
    // }

    // ret = esp_bluedroid_init();
    // if (ret)
    // {
    //     ESP_LOGE(GATTS_TABLE_TAG, "%s init bluetooth failed: %s", __func__, esp_err_to_name(ret));
    //     return;
    // }

    // ret = esp_bluedroid_enable();
    // if (ret)
    // {
    //     ESP_LOGE(GATTS_TABLE_TAG, "%s enable bluetooth failed: %s", __func__, esp_err_to_name(ret));
    //     return;
    // }

    esp_ble_gap_set_rand_addr(u8BLE_MAC_Addr_Backup);

    ret = esp_ble_gatts_register_callback(gatts_event_handler); // 连接之后通信
    if (ret)
    {
        ESP_LOGE(GATTS_TABLE_TAG, "gatts register error, error code = %x", ret);
        return;
    }

    ret = esp_ble_gap_register_callback(gap_event_handler); // 广播通信
    if (ret)
    {
        ESP_LOGE(GATTS_TABLE_TAG, "gap register error, error code = %x", ret);
        return;
    }

    ret = esp_ble_gatts_app_register(ESP_APP_ID); // 触发GATT EVENT
    if (ret)
    {
        ESP_LOGE(GATTS_TABLE_TAG, "gatts app register error, error code = %x", ret);
        return;
    }

    esp_err_t local_mtu_ret = esp_ble_gatt_set_local_mtu(GATTS_DEMO_CHAR_VAL_LEN_MAX); // 吞吐量//一帧最长字节
    if (local_mtu_ret)
    {
        ESP_LOGE(GATTS_TABLE_TAG, "set local  MTU failed, error code = %x", local_mtu_ret);
    }

    ESP_LOGE(GATTS_TABLE_TAG, "BLE INIT");
}

BleDataPacket stBleDataRx;

uint8_t BleValidData[BLE_RX_DATA_LEN];

uint32_t u32BleRxDataBegin = 0;
uint32_t u32BleRxDataRe = 0;
uint32_t u32BleRxDataValidLenTem = 0;
uint32_t u32BleRxDataValidLen = 0;
uint32_t u32BleDataBufLen = 0;

uint8_t BleDataBuf[BLE_RX_BUF_LEN];

void ble_rx_analysis_tasks(void *arg)
{
    
    // BLE_Init();
    while (1)
    {
        if (xQueueReceive(ble_rx_que, &stBleDataRx, portMAX_DELAY))
        {

            //      //   crc = ble_rx_cal_crc(BleDataBuf, BLE_RX_BUF_LEN);

              ESP_LOGI(GATTS_TABLE_TAG, "ble_rx_analysis_tasks \r\n");

            // ESP_LOGI(GATTS_TABLE_TAG, "stBleDataRx: %x ,%d \r\n", stBleDataRx.BleData[0], stBleDataRx.length);
            //  //  esp_log_buffer_hex(GATTS_TABLE_TAG, BleDataBuf, BLE_RX_BUF_LEN);
            //   ESP_LOGI(GATTS_TABLE_TAG, "ble_rx_analysis_tasks \r\n");

            if ((u32BleDataBufLen + stBleDataRx.length) >= BLE_RX_BUF_LEN)
            {
                u32BleDataBufLen = 0;
                BleDataBuf[0] = 0;
            }
            else
            {

                memcpy(&BleDataBuf[u32BleDataBufLen], stBleDataRx.BleData, stBleDataRx.length);

                u32BleDataBufLen += stBleDataRx.length;
            }

            // for (int i = 0; i < u32BleDataBufLen; i++)
            // {
            //     ESP_LOGE("GATTS_TABLE_TAG", "BleDataBuf[%x] \n", BleDataBuf[i]);
            // }

            /*******************协议解析***************************/

            if (u32BleDataBufLen > 4)
            {
                u32BleRxDataBegin = 0;
                while (u32BleRxDataBegin < (u32BleDataBufLen - 4))
                {

                    if ((BleDataBuf[0 + u32BleRxDataBegin] != 0x59) ||
                        (BleDataBuf[1 + u32BleRxDataBegin] != 0x44) ||
                        (BleDataBuf[2 + u32BleRxDataBegin] != 0x3D))
                    {
                        u32BleRxDataBegin++;
                        continue;
                    }

                    u32BleRxDataValidLenTem = BleDataBuf[3 + u32BleRxDataBegin];
                    if (u32BleRxDataValidLenTem > 175)
                    {
                        u32BleRxDataBegin++;
                        continue;
                    }

                    u32BleRxDataRe = u32BleDataBufLen - u32BleRxDataBegin;
                    if (u32BleRxDataRe >= (u32BleRxDataValidLenTem + 5))
                    {
                        // DATA CHECK
                        uint8_t u8YadiDataSum = 0;
                        u8YadiDataSum = Yadi_CAL_Data_Sum(&BleDataBuf[2 + u32BleRxDataBegin], (u32BleRxDataValidLenTem + 2));
                         printf("u8YadiDataSum %x \n", u8YadiDataSum);
                        // if (u8YadiDataSum != BleDataBuf[u32BleRxDataValidLenTem + 4 + u32BleRxDataBegin])
                        // {
                        //     u32BleRxDataBegin++;
                        //     continue;
                        // }
                    }
                    else
                    {
                        break;
                    }

                    u32BleRxDataRe = u32BleDataBufLen - u32BleRxDataBegin;

                    if (u32BleRxDataRe >= (u32BleRxDataValidLenTem + 7))
                    {
                        if ((BleDataBuf[u32BleRxDataValidLenTem + 5 + u32BleRxDataBegin] != 0x4B) ||
                            (BleDataBuf[u32BleRxDataValidLenTem + 6 + u32BleRxDataBegin] != 0x4A))
                        {
                            u32BleRxDataBegin++;
                            continue;
                        }
                        else
                        {

                            // for (i = 0; i < u32BleRxDataValidLenTem; i++)
                            // {
                            //     BleValidData[i] = BleDataBuf[4 + u32BleRxDataBegin];
                            // }
                            u32BleRxDataValidLen = u32BleRxDataValidLenTem;
                            memcpy(BleValidData, &BleDataBuf[4 + u32BleRxDataBegin], u32BleRxDataValidLenTem);
                            u32BleRxDataBegin += (u32BleRxDataValidLenTem + 7);

                            // ESP_LOGE(GATTS_TABLE_TAG, "BleValidData, value len = %d, value :", u32BleRxDataValidLenTem);

                            // for (int i = 0; i < u32BleRxDataValidLenTem; i++)
                            // {
                            //     ESP_LOGE(GATTS_TABLE_TAG,"BleValidData[%x]",BleValidData[i]);
                            // }

                            // esp_log_buffer_hex(GATTS_TABLE_TAG, BleValidData, u32BleRxDataValidLenTem);

                            continue;
                        }
                    }
                    else
                    {
                        break;
                    }
                }

                if (u32BleRxDataBegin > 0)
                {
                    // for (j = 0; j < (u32BleDataBufLen - u32BleRxDataBegin); j++)
                    // {
                    //     BleDataBuf[j] = BleDataBuf[j + u32BleRxDataBegin];
                    // }
                    memcpy(BleDataBuf, &BleDataBuf[u32BleRxDataBegin], (u32BleDataBufLen - u32BleRxDataBegin));
                    u32BleDataBufLen -= u32BleRxDataBegin;
                }
            }
        }
    }
}

void ble_mac_handle(void)
{
    // uint32_t i = 0;
    // uint32_t u32blemacch = 0;

    // if (CAN_MSG_Status(&CAN_CH0_CanMsgOp, CAN_CH0_ID_CAN_0x0C75FFD3_Msg_Count) != CAN_SIG_LOST)
    // {

    //     u8BLE_MAC_Addr[5] = Get_CAN_CH0_ID_0C75FFD3_Sig_MAC0();
    //     u8BLE_MAC_Addr[4] = Get_CAN_CH0_ID_0C75FFD3_Sig_MAC1();
    //     u8BLE_MAC_Addr[3] = Get_CAN_CH0_ID_0C75FFD3_Sig_MAC2();
    //     u8BLE_MAC_Addr[2] = Get_CAN_CH0_ID_0C75FFD3_Sig_MAC3();
    //     u8BLE_MAC_Addr[1] = Get_CAN_CH0_ID_0C75FFD3_Sig_MAC4() + 0x80;
    //     u8BLE_MAC_Addr[0] = Get_CAN_CH0_ID_0C75FFD3_Sig_MAC5();

    //     for (i = 0; i < 6; i++)
    //     {
    //         if (u8BLE_MAC_Addr[i] != u8BLE_MAC_Addr_Backup[i])
    //         {
    //             u32blemacch = 1;
    //             break;
    //         }
    //     }

    //     if (u32blemacch)
    //     {
    //         //   ESP_LOGI(GATTS_TABLE_TAG, "u32BLEtime = %d", u32BLEtime);
    //         if (u32BLEtime >= 60)
    //         {

    //             /********************/
    //             if (u32BleConnect)
    //             {
    //                 if (esp_ble_gap_disconnect(spp_remote_bda) == 0)
    //                 {
    //                 }
    //                 return;
    //             }

    //             if (u32BleAdv)
    //             {
    //                 if (esp_ble_gap_stop_advertising() == 0)
    //                 {
    //                 }
    //                 return;
    //             }
    //             u32BLEtime = 0;
    //             memcpy(u8BLE_MAC_Addr_Backup, u8BLE_MAC_Addr, 6);
    //             eeprom_WriteRecord(BLE_MAC_BLOCK, u8BLE_MAC_Addr_Backup, 6);

    //             raw_adv_data[6] = u8BLE_MAC_Addr_Backup[0];
    //             raw_adv_data[7] = u8BLE_MAC_Addr_Backup[1];
    //             raw_adv_data[8] = u8BLE_MAC_Addr_Backup[2];
    //             raw_adv_data[9] = u8BLE_MAC_Addr_Backup[3];
    //             raw_adv_data[10] = u8BLE_MAC_Addr_Backup[4];
    //             raw_adv_data[11] = u8BLE_MAC_Addr_Backup[5];

    //             u8BLEdevicename[2] = u8BLE_MAC_Addr_Backup[2] / 0x10 + 0X30;
    //             u8BLEdevicename[3] = u8BLE_MAC_Addr_Backup[2] % 0x10 + 0X30;
    //             u8BLEdevicename[4] = u8BLE_MAC_Addr_Backup[3] / 0x10 + 0X30;
    //             u8BLEdevicename[5] = u8BLE_MAC_Addr_Backup[3] % 0x10 + 0X30;
    //             u8BLEdevicename[6] = u8BLE_MAC_Addr_Backup[4] / 0x10 + 0X30;
    //             u8BLEdevicename[7] = u8BLE_MAC_Addr_Backup[4] % 0x10 + 0X30;
    //             u8BLEdevicename[8] = u8BLE_MAC_Addr_Backup[5] / 0x10 + 0X30;
    //             u8BLEdevicename[9] = u8BLE_MAC_Addr_Backup[5] % 0x10 + 0X30;

    //             raw_scan_rsp_data[21] = u8BLE_MAC_Addr_Backup[2] / 0x10 + 0X30;
    //             raw_scan_rsp_data[22] = u8BLE_MAC_Addr_Backup[2] % 0x10 + 0X30;
    //             raw_scan_rsp_data[23] = u8BLE_MAC_Addr_Backup[3] / 0x10 + 0X30;
    //             raw_scan_rsp_data[24] = u8BLE_MAC_Addr_Backup[3] % 0x10 + 0X30;
    //             raw_scan_rsp_data[25] = u8BLE_MAC_Addr_Backup[4] / 0x10 + 0X30;
    //             raw_scan_rsp_data[26] = u8BLE_MAC_Addr_Backup[4] % 0x10 + 0X30;
    //             raw_scan_rsp_data[27] = u8BLE_MAC_Addr_Backup[5] / 0x10 + 0X30;
    //             raw_scan_rsp_data[28] = u8BLE_MAC_Addr_Backup[5] % 0x10 + 0X30;

    //             // ESP_LOGI(GATTS_TABLE_TAG, "u8BLEdevicename[2] = %X", u8BLEdevicename[2]);
    //             // ESP_LOGI(GATTS_TABLE_TAG, "u8BLEdevicename[3] = %X", u8BLEdevicename[3]);
    //             // ESP_LOGI(GATTS_TABLE_TAG, "u8BLEdevicename[4] = %X", u8BLEdevicename[4]);
    //             // ESP_LOGI(GATTS_TABLE_TAG, "u8BLEdevicename[5] = %X", u8BLEdevicename[5]);
    //             // ESP_LOGI(GATTS_TABLE_TAG, "u8BLEdevicename[6] = %X", u8BLEdevicename[6]);
    //             // ESP_LOGI(GATTS_TABLE_TAG, "u8BLEdevicename[7] = %X", u8BLEdevicename[7]);
    //             // ESP_LOGI(GATTS_TABLE_TAG, "u8BLEdevicename[8] = %X", u8BLEdevicename[8]);
    //             // ESP_LOGI(GATTS_TABLE_TAG, "u8BLEdevicename[9] = %X", u8BLEdevicename[9]);

    //             esp_ble_gap_set_rand_addr(u8BLE_MAC_Addr_Backup);

    //             esp_err_t set_dev_name_ret = esp_ble_gap_set_device_name(SAMPLE_DEVICE_NAME);
    //             if (set_dev_name_ret)
    //             {
    //                 ESP_LOGE(GATTS_TABLE_TAG, "set device name failed, error code = %x", set_dev_name_ret);
    //             }

    //             esp_err_t raw_adv_ret = esp_ble_gap_config_adv_data_raw(raw_adv_data, sizeof(raw_adv_data)); //
    //             if (raw_adv_ret)
    //             {
    //                 ESP_LOGE(GATTS_TABLE_TAG, "config raw adv data failed, error code = %x ", raw_adv_ret);
    //             }
    //             adv_config_done |= ADV_CONFIG_FLAG;
    //             esp_err_t raw_scan_ret = esp_ble_gap_config_scan_rsp_data_raw(raw_scan_rsp_data, sizeof(raw_scan_rsp_data));
    //             if (raw_scan_ret)
    //             {
    //                 ESP_LOGE(GATTS_TABLE_TAG, "config raw scan rsp data failed, error code = %x", raw_scan_ret);
    //             }
    //             adv_config_done |= SCAN_RSP_CONFIG_FLAG;
    //         }
    //         else
    //         {
    //             u32BLEtime++;
    //         }
    //     }
    //     else
    //     {
    //         u32BLEtime = 0;
    //     }
    // }
    // else
    // {
    //     u32BLEtime = 0;
    // }
} /**
   * @brief 获取导航状态
   * @return 1 开启导航 0 关闭导航
   */
uint8_t Nav_Status(void)
{
    uint8_t u8NavResult = 0;

    if ((u32BleConnect == 1) && ((BleValidData[30] & 0x03) == 0x01))
    {
        u8NavResult = 1;
    }
    // printf("BleValidData %d\n",BleValidData[30]);

    return u8NavResult;
}

/**
 * @brief 获取导航转向数据
 * @return 无效0X88
 */
uint8_t Get_Nav_Turn_Data(void)
{
    uint8_t Data = 0X88;
    if (Nav_Status())
    {
        Data = BleValidData[0] & 0X0F;
        if (Data >= 0XC)
        {
            Data = 0X88;
        }
    }
    // ESP_LOGW(GATTS_TABLE_TAG, "Get_Nav_Turn_Data[%x]", Data);
    return Data;
}

/**
 * @brief 获取导航环岛信号
 * @return 0X88 无效值
 */
uint8_t Get_Nav_Circumferential_Data(void)
{
    uint8_t Data = 0X88;
    if (Nav_Status())
    {
        Data = BleValidData[0] & 0XF0;
        Data >>= 4;
        if (Data >= 0XC)
        {
            Data = 0X88;
        }
    }
    // ESP_LOGW(GATTS_TABLE_TAG, "Get_Nav_Circumferential_Data[%x]", Data);
    return Data;
}

/**
 * @brief 获取导航途径区域数据
 * @return 0X88 无效值
 */
uint8_t Get_Nav_Path_Region_Data(void)
{
    uint8_t Data = 0X88;
    if (Nav_Status())
    {
        Data = BleValidData[1];
        if (Data > 0X1F)
        {
            Data = 0X88;
        }
    }
    // ESP_LOGW(GATTS_TABLE_TAG, "Get_Nav_Path_Region_Data[%x]", Data);
    return Data;
}
/**
 * @brief 获取导航转向距离
 * @return 0 ~ 1000m   无效值 0X1234；
 */
uint16_t Get_Nav_Turn_Distance(void)
{
    uint16_t Data = 0X1234;
    if (Nav_Status())
    {
        Data = BleValidData[7] << 8 | BleValidData[8];
        if (Data < 0XFFFF)
        {
            if (Data > 0X3E7)
            {
                Data = 0X3E7;
            }
        }
        else
        {
            Data = 0X1234;
        }
    }
    return Data;
}
/**
 * @brief 获取导航剩余里程距离
 * @return 0 ~ 1000KM   无效值 0X1234
 */
uint16_t Get_Nav_Remain_Distance(void)
{
    uint16_t Data = 0X1234;
    if (Nav_Status())
    {
        Data = BleValidData[9] << 8 | BleValidData[10];
        if (Data < 0XFFFF)
        {
            Data /= 10;
            if (Data > 0X3E8)
            {
                Data = 0X3E8;
            }
        }
        else
        {
            Data = 0X1234;
        }
    }
    return Data;
}

/**
 * @brief 获取导航预计到达时间-小时
 * @return 0X88 无效值
 */
uint8_t Get_Nav_Time_Hour(void)
{
    uint8_t Hour = 0X88;
    if (Nav_Status())
    {
        Hour = BleValidData[17];
        if (Hour > 23)
        {
            Hour = 0X88;
        }
    }
    return Hour;
}
/**
 * @brief 获取当行预计到达时间-分钟
 * @return 0X88 无效值
 */
uint8_t Get_Nav_Time_Minute(void)
{
    uint8_t Minute = 0X88;
    if (Nav_Status())
    {
        Minute = BleValidData[18];
        if (Minute > 59)
        {
            Minute = 0X88;
        }
    }
    return Minute;
}
/**
 * @brief 获取导航名称数据长度
 * @return 0 ~ 15
 */
uint8_t Get_Nav_name_len(void)
{
    uint8_t len = 0;
    if (Nav_Status())
    {
        len = BleValidData[31];
        if (len > 15)
        {
            len = 15;
        }
    }
    return len;
}
#define NameArrayMax 27 /* 27个字节 9个中文 */
uint8_t NavNamelen = 0;
/**
 * @brief 获取导航字符串
 * @return 导航数据数组指针
 */
uint8_t *Get_Nav_name_Data(void)
{
    uint8_t copyLength = 0;

    if (u32BleRxDataValidLen > 32)
    {
        copyLength = u32BleRxDataValidLen - 32 + 1;
    }

    // if (copyLength > NameArrayMax)
    //{
    //     copyLength = NameArrayMax;
    //     NavNamelen = 1;
    // }
    // else
    //{
    //     NavNamelen = 0;
    // }

    if (copyLength > 0)
    {
        uint8_t *Array = (uint8_t *)malloc(copyLength * sizeof(uint8_t));
        if (Array == NULL)
        {
            // 处理内存分配失败的情况
            return NULL;
        }

        for (uint8_t i = 0; i < copyLength; i++)
        {
            Array[i] = BleValidData[i + 32];
            // ESP_LOGE(GATTS_TABLE_TAG, "Array[%x] len [%d]", Array[i], copyLength);
        }
        Array[copyLength] = '\0';

        // char *str = (char *)malloc((copyLength + 1) * sizeof(char));
        // if (str == NULL)
        // {

        //     free(Array);
        //     return NULL;
        // }

        // for (uint8_t i = 0; i < copyLength; i++)
        // {
        //     str[i] = (char)Array[i];
        // }

        // str[copyLength] = '\0'; // 添加字符串终止符

        // free(Array);

        return Array;
    }
    else
    {
        return NULL;
    }
}

uint8_t Get_Name_len(void)
{
    return NavNamelen;
}


/**
 * @brief 获取ble连接状态
 * @return 1 连接 0 未连接
 */
uint32_t Get_Ble_Connect_Status(void)
{
    return u32BleConnect;
}
