/**
 * @file lv_jpg.c
 *
 */

/*********************
 *      INCLUDES
 *********************/

#include "esp_err.h"
#include "esp_log.h"

#include "lvgl.h"
#include "src/misc/lv_fs.h"

#if 1

#include "esp_jpeg_dec.h"
#include "lv_esp_jpg.h"
#include "audio_mem.h"

static char *TAG = "JPG";

/*********************
 *      DEFINES
 *********************/

#define TJPGD_WORKBUFF_SIZE             4096    //Recommended by TJPGD libray

/**********************
 *      TYPEDEFS
 **********************/

typedef struct {
    // enum io_source_type type;
    lv_fs_file_t lv_file;
} io_source_t;

typedef struct {
    uint8_t * sjpeg_data;
    uint32_t sjpeg_data_size;
    int sjpeg_x_res;
    int sjpeg_y_res;
    uint8_t * frame_cache;
    io_source_t io;
} JPEG;

/**********************
 *  STATIC PROTOTYPES
 **********************/
static lv_res_t decoder_info(lv_img_decoder_t * decoder, const void * src, lv_img_header_t * header);
static lv_res_t decoder_open(lv_img_decoder_t * decoder, lv_img_decoder_dsc_t * dsc);
// static lv_res_t decoder_read_line(lv_img_decoder_t * decoder, lv_img_decoder_dsc_t * dsc, lv_coord_t x, lv_coord_t y,
//                                   lv_coord_t len, uint8_t * buf);
static void decoder_close(lv_img_decoder_t * decoder, lv_img_decoder_dsc_t * dsc);
static int is_jpg(const uint8_t * raw_data, size_t len);
static void lv_esp_jpg_cleanup(JPEG * jpg);
static void lv_esp_jpg_free(JPEG * jpg);

/**********************
 *  STATIC VARIABLES
 **********************/

/**********************
 *      MACROS
 **********************/

/**********************
 *   GLOBAL FUNCTIONS
 **********************/
void lv_esp_jpg_init(void)
{
    lv_img_decoder_t * dec = lv_img_decoder_create();
    lv_img_decoder_set_info_cb(dec, decoder_info);
    lv_img_decoder_set_open_cb(dec, decoder_open);
    lv_img_decoder_set_close_cb(dec, decoder_close);
    // lv_img_decoder_set_read_line_cb(dec, decoder_read_line);
}

static lv_fs_res_t read_file(const char *filename, uint8_t **buffer, uint32_t *size)
{
    lv_fs_file_t file;
    lv_fs_res_t res = lv_fs_open(&file, filename, LV_FS_MODE_RD);
    if (res != LV_FS_RES_OK) {
        ESP_LOGE(TAG, "Failed to open file %s", filename);
        return res;
    }

    // Get file size
    lv_fs_seek(&file, 0, LV_FS_SEEK_END);
    lv_fs_tell(&file, size);
    lv_fs_seek(&file, 0, LV_FS_SEEK_SET);

    *buffer = lv_mem_alloc(*size);
    if (!*buffer) {
        ESP_LOGE(TAG, "Failed to allocate memory for file %s", filename);
        lv_fs_close(&file);
        return LV_FS_RES_OUT_OF_MEM;
    }

    uint32_t rn = 0;
    res = lv_fs_read(&file, *buffer, *size, &rn);
    lv_fs_close(&file);

    if (res != LV_FS_RES_OK || rn != *size) {
        lv_mem_free(*buffer);
        *buffer = NULL;
        ESP_LOGE(TAG, "Failed to read file %s", filename);
        return LV_FS_RES_UNKNOWN;
    }
    return LV_FS_RES_OK;
}

static lv_res_t decode_jpeg(const uint8_t *input_buffer, uint32_t input_size, lv_img_header_t *header, uint8_t **output_buffer)
{
    jpeg_error_t ret;
    jpeg_dec_config_t config = {
#if  LV_COLOR_DEPTH == 32
        .output_type = JPEG_PIXEL_FORMAT_RGB888,
#elif  LV_COLOR_DEPTH == 16
#if  LV_BIG_ENDIAN_SYSTEM == 1 || LV_COLOR_16_SWAP == 1
        .output_type = JPEG_PIXEL_FORMAT_RGB565_BE,
#else
        .output_type = JPEG_PIXEL_FORMAT_RGB565_LE,
#endif
#else
#error Unsupported LV_COLOR_DEPTH
#endif
        .rotate = JPEG_ROTATE_0D,
    };

    jpeg_dec_handle_t jpeg_dec;
    jpeg_dec_open(&config, &jpeg_dec);
    if (!jpeg_dec) {
        ESP_LOGE(TAG, "Failed to open jpeg decoder");
        return LV_RES_INV;
    }

    jpeg_dec_io_t *jpeg_io = lv_mem_alloc(sizeof(jpeg_dec_io_t));
    jpeg_dec_header_info_t *out_info = lv_mem_alloc(sizeof(jpeg_dec_header_info_t));
    if (!jpeg_io || !out_info) {
        lv_mem_free(jpeg_io);
        lv_mem_free(out_info);
        jpeg_dec_close(jpeg_dec);
        ESP_LOGE(TAG, "Failed to allocate memory for jpeg decoder");
        return LV_RES_INV;
    }

    jpeg_io->inbuf = input_buffer;
    jpeg_io->inbuf_len = input_size;

    ret = jpeg_dec_parse_header(jpeg_dec, jpeg_io, out_info);
    if (ret == JPEG_ERR_OK) {

        header->w = out_info->width;
        header->h = out_info->height;
        // ESP_LOGI("JPEG", "%d,%d", out_info->width, out_info->height);
        if (output_buffer) {
            *output_buffer = lv_mem_alloc(out_info->height * out_info->width * 2);
            if (!*output_buffer) {
                lv_mem_free(jpeg_io);
                lv_mem_free(out_info);
                jpeg_dec_close(jpeg_dec);
                ESP_LOGE(TAG, "Failed to allocate memory for output buffer");
                return LV_RES_INV;
            }

            jpeg_io->outbuf = *output_buffer;
            ret = jpeg_dec_process(jpeg_dec, jpeg_io);
            if (ret != JPEG_ERR_OK) {
                lv_mem_free(*output_buffer);
                *output_buffer = NULL;
                lv_mem_free(jpeg_io);
                lv_mem_free(out_info);
                jpeg_dec_close(jpeg_dec);
                ESP_LOGE(TAG, "Failed to decode jpeg");
                return LV_RES_INV;
            }
        }
    } else {
        lv_mem_free(jpeg_io);
        lv_mem_free(out_info);
        jpeg_dec_close(jpeg_dec);
        ESP_LOGE(TAG, "Failed to parse jpeg header");
        return LV_RES_INV;
    }

    lv_mem_free(jpeg_io);
    lv_mem_free(out_info);
    jpeg_dec_close(jpeg_dec);

    return LV_RES_OK;
}

static lv_res_t decoder_info(lv_img_decoder_t *decoder, const void *src, lv_img_header_t *header)
{
    LV_UNUSED(decoder);
    lv_img_src_t src_type = lv_img_src_get_type(src);
    lv_res_t lv_ret = LV_RES_OK;

    if (src_type == LV_IMG_SRC_FILE) {
        const char *fn = src;
        if (strcmp(lv_fs_get_ext(fn), "jpg") == 0) {
            uint8_t *workb_temp = NULL;
            uint32_t file_size = 0;

            if (read_file(fn, &workb_temp, &file_size) != LV_FS_RES_OK) {
                return LV_RES_INV;
            }

            lv_ret = decode_jpeg(workb_temp, file_size, header, NULL);
            lv_mem_free(workb_temp);
            return lv_ret;
        }
    } else if (src_type == LV_IMG_SRC_VARIABLE) {
        const lv_img_dsc_t *img_dsc = src;
        if (is_jpg(img_dsc->data, img_dsc->data_size) == true) {
            lv_ret = decode_jpeg(img_dsc->data, img_dsc->data_size, header, NULL);
        }
        return lv_ret;
    }

    return LV_RES_INV;
}

static lv_res_t decoder_open(lv_img_decoder_t *decoder, lv_img_decoder_dsc_t *dsc)
{
    LV_UNUSED(decoder);
    lv_res_t lv_ret = LV_RES_OK;

    if (dsc->src_type == LV_IMG_SRC_FILE) {
        const char *fn = dsc->src;
        if (strcmp(lv_fs_get_ext(fn), "jpg") == 0) {
            uint8_t *workb_temp = NULL;
            uint32_t file_size = 0;

            if (read_file(fn, &workb_temp, &file_size) != LV_FS_RES_OK) {
                return LV_RES_INV;
            }

            JPEG * jpg = (JPEG *) dsc->user_data;
            if (jpg == NULL) {
                jpg = lv_mem_alloc(sizeof(JPEG));
                if (! jpg) {
                    ESP_LOGE(TAG, "Failed to allocate memory for jpg");
                    return LV_RES_INV;
                }

                memset(jpg, 0, sizeof(jpg));
                dsc->user_data = jpg;
            }

            uint8_t *output_buffer = NULL;
            lv_img_header_t header;
            lv_ret = decode_jpeg(workb_temp, file_size, &header, &output_buffer);
            lv_mem_free(workb_temp);

            if (lv_ret == LV_RES_OK) {
                dsc->img_data = output_buffer;
                jpg->frame_cache = output_buffer;
            }
            return lv_ret;
        }
    } else if (dsc->src_type == LV_IMG_SRC_VARIABLE) {
        const lv_img_dsc_t *img_dsc = dsc->src;
        if (is_jpg(img_dsc->data, img_dsc->data_size) == true) {
            uint8_t *output_buffer = NULL;
            lv_img_header_t header;
            lv_ret = decode_jpeg(img_dsc->data, img_dsc->data_size, &header, &output_buffer);

            if (lv_ret == LV_RES_OK) {
                dsc->img_data = output_buffer;
            }
            return lv_ret;
        }
    }

    return LV_RES_INV;
}

/**
 * Free the allocated resources
 * @param decoder pointer to the decoder where this function belongs
 * @param dsc pointer to a descriptor which describes this decoding session
 */
static void decoder_close(lv_img_decoder_t * decoder, lv_img_decoder_dsc_t * dsc)
{
    LV_UNUSED(decoder);
    /*Free all allocated data*/
    JPEG * jpg = (JPEG *) dsc->user_data;
    if (!jpg) {
        return;
    }

    switch (dsc->src_type) {
    case LV_IMG_SRC_FILE:
        // ESP_LOGI("JPEG", "Closing:%p", jpg);
        // if (jpg->io.lv_file.file_d) {
        //     lv_fs_close(&(jpg->io.lv_file));
        // }
        lv_esp_jpg_cleanup(jpg);
        break;

    case LV_IMG_SRC_VARIABLE:
        lv_esp_jpg_cleanup(jpg);
        break;

    default:
        ;
    }
}

static int is_jpg(const uint8_t * raw_data, size_t len)
{
    const uint8_t jpg_signature_JFIF[] = {0xFF, 0xD8, 0xFF,  0xE0,  0x00,  0x10, 0x4A,  0x46, 0x49, 0x46};
    const uint8_t jpg_signature_Adobe[] = {0xFF, 0xD8, 0xFF,  0xEE,  0x00, 0x0E,  0x41, 0x64, 0x6F, 0x62};
    if (len < sizeof(jpg_signature_JFIF)) {
        return false;
    }
    return ((memcmp(jpg_signature_JFIF, raw_data, sizeof(jpg_signature_JFIF)) == 0) | (memcmp(jpg_signature_Adobe, raw_data, sizeof(jpg_signature_Adobe)) == 0));
}

static void lv_esp_jpg_free(JPEG * jpg)
{
    if (jpg->frame_cache) {
        lv_mem_free(jpg->frame_cache);
    }
}

static void lv_esp_jpg_cleanup(JPEG * jpg)
{
    if (! jpg) {
        return;
    }

    lv_esp_jpg_free(jpg);
    lv_mem_free(jpg);
}

#endif /*LV_USE_JPG_ESP*/
