/*============================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                         */
/* Module       = Eth_ETNE_LLDriver.c                                         */
/* SW-VERSION   = 1.5.4                                                       */
/*============================================================================*/
/*                                  COPYRIGHT                                 */
/*============================================================================*/
/* (c) 2021,2022 Renesas Electronics Corporation. All rights reserved.        */
/*============================================================================*/
/* Purpose:                                                                   */
/* This file contains ETNDE specific operations of Eth Driver Component.      */
/*                                                                            */
/*============================================================================*/
/*                                                                            */
/* Unless otherwise agreed upon in writing between your company and           */
/* Renesas Electronics Corporation the following shall apply!                 */
/*                                                                            */
/* Warranty Disclaimer                                                        */
/*                                                                            */
/* There is no warranty of any kind whatsoever granted by Renesas. Any        */
/* warranty is expressly disclaimed and excluded by Renesas, either expressed */
/* or implied, including but not limited to those for non-infringement of     */
/* intellectual property, merchantability and/or fitness for the particular   */
/* purpose.                                                                   */
/*                                                                            */
/* Renesas shall not have any obligation to maintain, service or provide bug  */
/* fixes for the supplied Product(s) and/or the Application.                  */
/*                                                                            */
/* Each User is solely responsible for determining the appropriateness of     */
/* using the Product(s) and assumes all risks associated with its exercise    */
/* of rights under this Agreement, including, but not limited to the risks    */
/* and costs of program errors, compliance with applicable laws, damage to    */
/* or loss of data, programs or equipment, and unavailability or              */
/* interruption of operations.                                                */
/*                                                                            */
/* Limitation of Liability                                                    */
/*                                                                            */
/* In no event shall Renesas be liable to the User for any incidental,        */
/* consequential, indirect, or punitive damage (including but not limited     */
/* to lost profits) regardless of whether such liability is based on breach   */
/* of contract, tort, strict liability, breach of warranties, failure of      */
/* essential purpose or otherwise and even if advised of the possibility of   */
/* such damages. Renesas shall not be liable for any services or products     */
/* provided by third party vendors, developers or consultants identified or   */
/* referred to the User by Renesas in connection with the Product(s) and/or   */
/* the Application.                                                           */
/*                                                                            */
/*============================================================================*/
/* Environment:                                                               */
/*              Devices:        X2x                                           */
/*============================================================================*/

/*******************************************************************************
**                      Revision Control History                              **
*******************************************************************************/
/*
 * 1.5.0: 07/10/2021    : Initial Version
 */
/******************************************************************************/

/*******************************************************************************
**                      Include Section                                       **
*******************************************************************************/
#include "Eth.h"
#include "Eth_ETNDE_Ram.h"
#include "Eth_ETNDE_LLDriver.h"
#include "Eth_ETNE_LLDriver.h"

#if (ETH_CRITICAL_SECTION_PROTECTION == STD_ON)
/* Included for the declaration of the critical section protection functions */
#include "SchM_Eth.h"
#endif

/*******************************************************************************
**                      Version Information                                   **
*******************************************************************************/
/* AUTOSAR release version information */
#define ETH_ETNE_C_AR_RELEASE_MAJOR_VERSION ETH_AR_RELEASE_MAJOR_VERSION_VALUE
#define ETH_ETNE_C_AR_RELEASE_MINOR_VERSION ETH_AR_RELEASE_MINOR_VERSION_VALUE
#define ETH_ETNE_C_AR_RELEASE_REVISION_VERSION ETH_AR_RELEASE_REVISION_VERSION_VALUE

/* File version information */
#define ETH_ETNE_C_SW_MAJOR_VERSION    ETH_SW_MAJOR_VERSION_VALUE
#define ETH_ETNE_C_SW_MINOR_VERSION    ETH_SW_MINOR_VERSION_VALUE

/*******************************************************************************
**                      Version Check                                         **
*******************************************************************************/
#if (ETH_ETNE_AR_RELEASE_MAJOR_VERSION != ETH_ETNE_C_AR_RELEASE_MAJOR_VERSION)
#error "Eth_ETNE_LLDriver.c : Mismatch in Release Major Version"
#endif
#if (ETH_ETNE_AR_RELEASE_MINOR_VERSION != ETH_ETNE_C_AR_RELEASE_MINOR_VERSION)
#error "Eth_ETNE_LLDriver.c : Mismatch in Release Minor Version"
#endif
#if (ETH_ETNE_AR_RELEASE_REVISION_VERSION != ETH_ETNE_C_AR_RELEASE_REVISION_VERSION)
#error "Eth_ETNE_LLDriver.c : Mismatch in Release Revision Version"
#endif

#if (ETH_ETNE_SW_MAJOR_VERSION != ETH_ETNE_C_SW_MAJOR_VERSION)
#error "Eth_ETNE_LLDriver.c : Mismatch in Software Major Version"
#endif
#if (ETH_ETNE_SW_MINOR_VERSION != ETH_ETNE_C_SW_MINOR_VERSION)
#error "Eth_ETNE_LLDriver.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (1:1532)    : The function '%1s' is only referenced in one translation unit - but not the one in which it  */
/*                       is defined.                                                                                  */
/* Rule                : CERTCCM DCL19, MISRA C:2012 Rule-8.7                                                         */
/* JV-01 Justification : This is accepted, due to the specific coding rule, function of each hardware unit is         */
/*                       implemented in separated files for this hardware unit. Could not be static function.         */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0316)    : Cast from a pointer to void to a pointer to object type.                                     */
/* Rule                : MISRA C:2012 Rule-11.5                                                                       */
/* JV-01 Justification : Typecasting from void* is necessary to hide internal types from the header files which are   */
/*                       exposed to user.                                                                             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3383)    : Cannot identify wraparound guard for unsigned arithmetic expression.                         */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : Wraparound can't be occur in this unsigned integer arithmetic so there is no need to add a   */
/*                       wraparound guard here.                                                                       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (6:2994)    : The value of this 'while' or 'for' loop controlling expression is always 'false'.            */
/*                       The loop will not be entered.                                                                */
/* Rule                : MISRA C:2012 Rule-14.3                                                                       */
/* JV-01 Justification : This will be improved in the future ECODE update                                             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (6:2996)    : The result of this logical operation is always 'false'.                                      */
/* Rule                : MISRA C:2012 Rule-2.2                                                                        */
/* JV-01 Justification : This will be improved in the future ECODE update                                             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (6:2880)    : This code is unreachable.                                                                    */
/* Rule                : CERTCCM MSC07, MISRA C:2012 Rule-2.1                                                         */
/* JV-01 Justification : This will be improved in the future ECODE update.                                            */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2824)    : Possible: Arithmetic operation on NULL pointer.                                              */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3469)    : This usage of a function-like macro looks like it could be replaced by an equivalent         */
/*                       function call.                                                                               */
/* Rule                : MISRA C:2012 Dir-4.9                                                                         */
/* JV-01 Justification : To prevent overhead and speed up the processing, there is no need to call a function here    */
/*                       for such a small operation.                                                                  */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0315)    : Implicit conversion from a pointer to object type to a pointer to void.                      */
/* Rule                : MISRA C:2012 Dir-1.1                                                                         */
/* JV-01 Justification : Typecasting is done as per the register size, to access hardware registers.                  */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2982)    : This assignment is redundant. The value of this object is never used before being modified.  */
/* Rule                : CERTCCM MSC07, MSC13, MISRA C:2012 Rule-2.2                                                  */
/* JV-01 Justification : The variable needs to be initialized before using it.                                        */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : Use of #include directive after code fragment.                                               */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is done as per Memory Requirement, (MEMMAP003 - Specification of Memory Mapping).       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:1800)    : The %1s operand (essential type: '%2s') will be implicitly converted to a                    */
/*                       floating type, '%3s', in this arithmetic operation.                                          */
/* Rule                : MISRA C:2012 Rule-10.4                                                                       */
/* JV-01 Justification : This is necessary to set caculated gPTP timer increment value to the GPTP timer.             */
/*                       There is no problem with this implementation.                                                */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:1840)    : The %1s operand is constant, 'essentially signed' (%2s) and non-negative but                 */
/*                       will be implicitly converted to an unsigned type (%3s) in this arithmetic                    */
/*                       operation.                                                                                   */
/* Rule                : MISRA C:2012 Rule-10.4                                                                       */
/* JV-01 Justification : This is necessary to set caculated gPTP timer increment value to the GPTP timer.             */
/*                       There is no problem with this implementation.                                                */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1263)    : Floating constant causes implicit conversion of other (integral) operand.                    */
/* Rule                : CERTCCM INT02, FLP36                                                                         */
/* JV-01 Justification : This is necessary to set caculated gPTP timer increment value to the GPTP timer.             */
/*                       There is no problem with this implementation.                                                */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                    Global Data                                                     **
***********************************************************************************************************************/

/***********************************************************************************************************************
**                                                Function Definitions                                                **
***********************************************************************************************************************/
#define ETH_START_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"

STATIC FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_GWCA_ModeSetting(CONST(uint32, AUTOMATIC) LulMode);

STATIC FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_EthAgentInit(CONST(uint32, AUTOMATIC) LulCtrlIdx);

/***********************************************************************************************************************
** Function Name         : Eth_HwCommonInit (ETNE)
**
** Service ID            : N/A
**
** Description           : Initialize the common parts of ETNE HWUnit
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : E_OK : Success
**                         E_NOT_OK : Failure
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpETNE_MFWDRegs, Eth_GpCtrlConfigPtr
**
** Function(s) invoked   : Eth_GWCA_ModeSetting, GetCounterValue,
**                         Eth_GetTimeOutValue, Eth_TsDescConfig
**
** Registers Used        : FWMFV, FWQMFMC, FWQMMC, FWQRFDC, FWQYFDC, FWQVTCMC,
**                         FWQMCBSC, FWQMCIRC, FWQMEBSC, FWQMEIRC, FWQMCFC,
**                         FWQIUEC, FWQGMC, FWQMEC, FWMDANFV, FWMDANFCSD,
**                         FWSPBFE, FWSPBFCx, FWCEPC, FWSTPFC, FWAC,
**                         FWSOVPC4, FWGGTS, FWGTL, FWGJC, FWRR, FWQTLx, FWQTLR,
**                         FWMTLx, FWMTLR, RR, GWRR, AXIWC, AXIRC, GWRGC,
**                         GTIVt, TME, GTOVt0, GTOVt1, GTOVt2
**
** Reference ID          : ETH_DUD_ACT_307
** Reference ID          : ETH_DUD_ACT_307_REG001, ETH_DUD_ACT_307_REG002
** Reference ID          : ETH_DUD_ACT_307_REG003, ETH_DUD_ACT_307_REG004
** Reference ID          : ETH_DUD_ACT_307_REG005, ETH_DUD_ACT_307_REG006
** Reference ID          : ETH_DUD_ACT_307_REG007, ETH_DUD_ACT_307_REG008
** Reference ID          : ETH_DUD_ACT_307_REG009, ETH_DUD_ACT_307_REG010
** Reference ID          : ETH_DUD_ACT_307_REG011, ETH_DUD_ACT_307_REG012
** Reference ID          : ETH_DUD_ACT_307_REG013, ETH_DUD_ACT_307_REG014
** Reference ID          : ETH_DUD_ACT_307_REG015, ETH_DUD_ACT_307_REG016
** Reference ID          : ETH_DUD_ACT_307_REG017, ETH_DUD_ACT_307_REG018
** Reference ID          : ETH_DUD_ACT_307_REG019, ETH_DUD_ACT_307_REG020
** Reference ID          : ETH_DUD_ACT_307_REG021, ETH_DUD_ACT_307_REG022
** Reference ID          : ETH_DUD_ACT_307_REG023, ETH_DUD_ACT_307_REG024
** Reference ID          : ETH_DUD_ACT_307_REG025, ETH_DUD_ACT_307_REG026
** Reference ID          : ETH_DUD_ACT_307_REG027, ETH_DUD_ACT_307_REG028
** Reference ID          : ETH_DUD_ACT_307_REG029, ETH_DUD_ACT_307_REG030
** Reference ID          : ETH_DUD_ACT_307_REG031, ETH_DUD_ACT_307_REG032
** Reference ID          : ETH_DUD_ACT_307_REG033, ETH_DUD_ACT_307_REG034
** Reference ID          : ETH_DUD_ACT_307_REG035, ETH_DUD_ACT_307_REG036
** Reference ID          : ETH_DUD_ACT_307_REG037, ETH_DUD_ACT_307_REG038
** Reference ID          : ETH_DUD_ACT_307_REG039, ETH_DUD_ACT_307_REG040
** Reference ID          : ETH_DUD_ACT_307_REG041, ETH_DUD_ACT_307_REG042
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_HwCommonInit(void)                                                           /* PRQA S 1532 # JV-01 */
{
  P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;
  uint32 LulRxQueueMax;
  TickType LulTickStart;
  TickType LulTickElap;
  uint32 LulCnt;
  uint32 LulQueIdx;
  Std_ReturnType LucResult;

  LucResult = E_OK;

  /*-------------------------------------------------------------------------*/
  /* mFwd initialization for TSNSWA                                          */
  /*-------------------------------------------------------------------------*/
  /* Forwarding configuration1 */
  for (LulCnt = 0UL; LulCnt < ETH_ETNE_MAX_MFN_QGN; LulCnt++)
  {
    /* Set Qci MSDU(MAC protocol data unit) */
    Eth_GstETNE_Regs.pMFWD->ulFWMFV[LulCnt] = ETH_ETNE_FWMFV_CONFIG;                                                    /* PRQA S 2814 # JV-01 */
  }

  for (LulCnt = 0UL; LulCnt < ETH_ETNE_MAX_QCI_GRP; LulCnt++)
  {
    /* Set Qci MSDU filter mode (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWQMFMC[LulCnt] = 0UL;

    /* Set Qci flow meter mode (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWQMMC[LulCnt] = 0UL;

    /* Set Qci red frame drop (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWQRFDC[LulCnt] = 0UL;

    /* Set Qci yellow frame drop (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWQYFDC[LulCnt] = 0UL;
  }

  for (LulCnt = 0UL; LulCnt < ETH_ETNE_MAX_QCI_VLAN; LulCnt++)
  {
    /* Set Qci vlan tag color mapping (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWQVTCMC[LulCnt] = 0UL;
  }

  for (LulCnt = 0UL; LulCnt < ETH_ETNE_MAX_MFN_QGN; LulCnt++)
  {
    /* Set Qci meter CBS (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWQMCBSC[LulCnt] = 0UL;
    /* Set Qci meter CIR (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWQMCIRC[LulCnt] = 0UL;
    /* Set Qci meter EBS (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWQMEBSC[LulCnt] = 0UL;
    /* Set Qci meter EIR (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWQMEIRC[LulCnt] = 0UL;
    /* Set Qci meter coupling (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWQMCFC[LulCnt] = 0UL;
  }

  for (LulCnt = 0UL; LulCnt < ETH_ETNE_MAX_QCI_GRP; LulCnt++)
  {
    /* Set Qci IPV update enable (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWQIUEC[LulCnt] = 0UL;
    /* Set Qci Stream Gate mode (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWQGMC[LulCnt] = 0UL;
    /* Set Qci meter enable (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWQMEC[LulCnt] = 0UL;
  }

  /* Forwarding configuration2 */
  for (LulCnt = 0UL; LulCnt < ETH_ETNE_MAX_PN; LulCnt++)
  {
    /* Broadcast vector setting (Not yet supported on alpha version) */
    Eth_GstETNE_Regs.pMFWD->ulFWMDANFV[LulCnt] = 0UL;
    Eth_GstETNE_Regs.pMFWD->ulFWMDANFCSD[LulCnt] = 0UL;
  }
  /* Source port based forwarding enable */
  Eth_GstETNE_Regs.pMFWD->ulFWSPBFE = ETH_ETNE_FWSPBFE_CONFIG;
  /* CPU mirroring enable (Not yet supported on alpha version) */
  Eth_GstETNE_Regs.pMFWD->ulFWSPBFC0 = 0UL;
  /* Ethernet mirroring enable (Not yet supported on alpha version) */
  Eth_GstETNE_Regs.pMFWD->ulFWSPBFC1 = 0UL;

  for (LulCnt = 0UL; LulCnt < (ETH_ETNE_MAX_PN - 1UL); LulCnt++)
  {
    LpHwUnitConfig =
      (P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA))Eth_GpCtrlConfigPtr[LulCnt].pHwUnitConfig;                /* PRQA S 0316 # JV-01 */
    LulRxQueueMax = (uint32)LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue;

    /* Destination vector for source port */
    Eth_GstETNE_Regs.pMFWD->ulFWSPBFC2[LulCnt] = LpHwUnitConfig->ulFwdVectorId;                                         /* PRQA S 2814 # JV-01 */
    /* CPU sub destination */
    Eth_GstETNE_Regs.pMFWD->ulFWSPBFC3[LulCnt] = LulCnt + (LulCnt * (LulRxQueueMax - 1UL));                             /* PRQA S 3383 # JV-01 */
  }

  /* MAC authentification forwarding setting is not supported */

  /* MAC destination forwarding setting is not supported */

  /* BPDU forwarding setting is not supported */

  /* Private VLAN setting is not supported */

  /* Double tagging type setting is not supported */

  /* Agent Filtering CSD enable setting is not supported */

  /* Water mark control setting is not supported */

  /* Forwarding configuration3 */
  /* CPU exceptional path setting */
  Eth_GstETNE_Regs.pMFWD->ulFWCEPC = ETH_ETNE_FWCEPC_CONFIG;

  /* CPU lerning path setting is not supported */

  /* Mirroring path setting is not support */

  /* Source port lock function setting */
  Eth_GstETNE_Regs.pMFWD->ulFWSTPFC = ETH_PORT_AVAILABLE_VECTOR;

  /* Migration setting is not supported */

  /* Authentification setting */
  Eth_GstETNE_Regs.pMFWD->ulFWAC = ETH_PORT_AVAILABLE_VECTOR;

  /* Private VLAN Promiscuous/Isolated/community setting is not supported */

  /* Search, vector and priority control setting */
  Eth_GstETNE_Regs.pMFWD->ulFWSOVPC4 = ETH_ETNE_SOVPC4_CONFIG;

  /* Gate filter table configuration */
  for (LulCnt = 0UL; LulCnt < ETH_ETNE_MAX_QCI_GRP; LulCnt++)
  {
    LpHwUnitConfig =
      (P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA))Eth_GpCtrlConfigPtr[LulCnt].pHwUnitConfig;                /* PRQA S 0316 # JV-01 */

    /* gPTP timer select */
    Eth_GstETNE_Regs.pMFWD->ulFWGGTS[LulCnt] = ETH_ETNE_FWGGTS_CONFIG;
    /* Transmission latency setting */
    Eth_GstETNE_Regs.pMFWD->ulFWGTL[LulCnt] = LpHwUnitConfig->ulGateLatency;                                            /* PRQA S 2814 # JV-01 */
    /* Jitter clock setting */
    Eth_GstETNE_Regs.pMFWD->ulFWGJC[LulCnt] = LpHwUnitConfig->ulGateJitter;
  }

  /* QCi table configuration */
  LulTickStart = 0UL;
  (void)GetCounterValue(ETH_OS_COUNTER_ID, &LulTickStart);
  do
  {
    LulTickElap = Eth_GetTimeOutValue(LulTickStart);
  } while ((0UL == (Eth_GstETNE_Regs.pMFWD->ulFWRR & ETH_ETNE_FWRR_QTRR)) &&
           (LulTickElap <= ETH_TIMEOUT_COUNT));

  if (ETH_TIMEOUT_COUNT < LulTickElap)
  {
    LucResult = E_NOT_OK;
  }
  else
  {
    #if 0 /* Required for future features and will be considered when implementing the Qci filtering feature */
    for (LulCnt = 0UL; LulCnt < ETH_ETNE_QCI_TBL_TOTAL; LulCnt++)                                                       /* PRQA S 2994, 2996 # JV-01, JV-01 */
    {
      /* Qci MAC address & DEI/PCP/VLAN setting (Not yet supported on alpha version) */
      Eth_GstETNE_Regs.pMFWD->ulFWQTL0 = 0UL;                                                                           /* PRQA S 2880 # JV-01 */
      Eth_GstETNE_Regs.pMFWD->ulFWQTL1 = 0UL;
      /* Qci source port number setting (Not yet supported on alpha version) */
      Eth_GstETNE_Regs.pMFWD->ulFWQTL2 = 0UL;
      /* Qci information setting (Not yet supported on alpha version) */
      Eth_GstETNE_Regs.pMFWD->ulFWQTL3 = 0UL;

      /* Start learning (Not yet supported on alpha version) */
      Eth_GstETNE_Regs.pMFWD->ulFWQTLR = ETH_ETNE_FWQTLR_QTL;
      LulTickStart = 0UL;
      (void)GetCounterValue(ETH_OS_COUNTER_ID, &LulTickStart);
      do
      {
        LulTickElap = Eth_GetTimeOutValue(LulTickStart);
      } while ((0UL != (Eth_GstETNE_Regs.pMFWD->ulFWQTLR & ETH_ETNE_FWQTLR_QTL)) &&
               (LulTickElap <= ETH_TIMEOUT_COUNT));

      if (ETH_TIMEOUT_COUNT < LulTickElap)
      {
        LucResult = E_NOT_OK;
        break;
      } /* else: No action required */
    }
    #endif
  }

  /* SRP table configuration is not supported */

  /* MAC table configuration */
  LulTickStart = 0UL;
  (void)GetCounterValue(ETH_OS_COUNTER_ID, &LulTickStart);
  do
  {
    LulTickElap = Eth_GetTimeOutValue(LulTickStart);
  } while ((0UL == (Eth_GstETNE_Regs.pMFWD->ulFWRR & ETH_ETNE_FWRR_MTRR)) &&
           (LulTickElap <= ETH_TIMEOUT_COUNT));

  if (ETH_TIMEOUT_COUNT < LulTickElap)
  {
    LucResult = E_NOT_OK;
  }
  else
  {
    for (LulCnt = 0UL; LulCnt < ETH_TOTAL_CTRL_CONFIG; LulCnt++)
    {
      LpHwUnitConfig =
          (P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA))Eth_GpCtrlConfigPtr[LulCnt].pHwUnitConfig;            /* PRQA S 0316 # JV-01 */
      LulRxQueueMax = (uint32)LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue;

      for (LulQueIdx = 1UL; LulQueIdx < LulRxQueueMax; LulQueIdx++)
      {
        /* MAC address to add to the MAC table setting */
        Eth_GstETNE_Regs.pMFWD->ulFWMTL0 =
          ((uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulQueIdx].aaMacFilterPattern[0] << 8) |                /* PRQA S 2824 # JV-01 */
          (uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulQueIdx].aaMacFilterPattern[1];
        Eth_GstETNE_Regs.pMFWD->ulFWMTL1 =
          ((uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulQueIdx].aaMacFilterPattern[2] << 24) |
          ((uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulQueIdx].aaMacFilterPattern[3] << 16) |
          ((uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulQueIdx].aaMacFilterPattern[4] << 8) |
          (uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulQueIdx].aaMacFilterPattern[5];
        /* Routing vector setting */
        Eth_GstETNE_Regs.pMFWD->ulFWMTL2 = LpHwUnitConfig->ulFwdVectorId;
        /* Routing information setting */
        Eth_GstETNE_Regs.pMFWD->ulFWMTL3 =
          LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulQueIdx].ulQueueId + (LulCnt * LulRxQueueMax);                 /* PRQA S 3383 # JV-01 */

        /* Start lerning */
        Eth_GstETNE_Regs.pMFWD->ulFWMTLR = ETH_ETNE_FWMTLR_MTL;
        LulTickStart = 0UL;
        (void)GetCounterValue(ETH_OS_COUNTER_ID, &LulTickStart);
        do
        {
          LulTickElap = Eth_GetTimeOutValue(LulTickStart);
        } while ((0UL != (Eth_GstETNE_Regs.pMFWD->ulFWMTLR & ETH_ETNE_FWMTLR_MTL)) &&
                 (LulTickElap <= ETH_TIMEOUT_COUNT));

        if (ETH_TIMEOUT_COUNT < LulTickElap)
        {
          LucResult = E_NOT_OK;
          break;
        } /* else: No action required */
      }
    }
  }

  /* VLAN table configuration is not supported */

  /* Interrupt configuration */
  Eth_GstETNE_Regs.pMFWD->ulFWEIE1 = ETH_ETNE_FWEIE1_CONFIG;

  /*-------------------------------------------------------------------------*/
  /* GWCA initialization for TSNSWA                                          */
  /*-------------------------------------------------------------------------*/
  LucResult |= Eth_GWCA_ModeSetting(ETH_ETNE_GWMC_DISABLE_MODE);
  LucResult |= Eth_GWCA_ModeSetting(ETH_ETNE_GWMC_CONFIG_MODE);
  if (E_OK == LucResult)
  {
    /* Clear RAM Reset Register Bits */
    Eth_GstETNE_Regs.pAXIBMI->ulRR = ETH_ETNDE_RATRR_RESET | ETH_ETNDE_TATRR_RESET;

    /* RAM Reset Waiting */
    LulTickStart = 0UL;
    (void)GetCounterValue(ETH_OS_COUNTER_ID, &LulTickStart);
    do
    {
      LulTickElap = Eth_GetTimeOutValue(LulTickStart);
    } while ((((ETH_ETNDE_RATRR_RESET | ETH_ETNDE_TATRR_RESET) != Eth_GstETNE_Regs.pAXIBMI->ulRR) ||
             (ETH_ETNDE_RMSRR_RESET != Eth_GstETNE_Regs.pGWCA->ulGWRR)) &&
             (LulTickElap <= ETH_TIMEOUT_COUNT));

    if (ETH_TIMEOUT_COUNT < LulTickElap)
    {
      LucResult = E_NOT_OK;
    } /* else: No action required */
  } /* else: No action required */

  if (E_OK == LucResult)
  {
    /* Set AXI Write and Read Outstanding Number */
    Eth_GstETNE_Regs.pAXIBMI->ulAXIWC = ETH_ETNDE_AXIWC_CONFIG;
    Eth_GstETNE_Regs.pAXIBMI->ulAXIRC = ETH_ETNDE_AXIRC_CONFIG;

    /* Rx generate configuration register setting */
    Eth_GstETNE_Regs.pGWCA->ulGWRGC = ETH_ETNE_GWRGC_FCS_DISABLE;

    #if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
    /* Ts descriptor chain setting */
    Eth_TsDescConfig();

    /*-----------------------------------------------------------------------*/
    /* gPTP timer initialization for TSNSWA                                  */
    /*-----------------------------------------------------------------------*/
    /* Assigning the configured timer increment value to the GPTP timer GTIVt (t = 0 to PTPTN-1) */
    Eth_GstETNE_Regs.pGPTMA->stGptpCfg[ETH_GPTP_TIMER_DOMAIN].ulGTIVt = (uint32)ETH_GPTP_SET_GTIV(ETH_PERI_CLOCK_HZ);   /* PRQA S 1263, 1800, 1840, 3469 # JV-01, JV-01, JV-01, JV-01 */

    /* GPTP Timer Enable */
    Eth_GstETNE_Regs.pGPTMA->ulTME = ETH_GPTP_TIMER_DOMAIN_ENABLE;

    /* Set the nanosecond part of the GPTP timer GTOVt0 (t = 0 to PTPTN-1) */
    Eth_GstETNE_Regs.pGPTMA->stGptpCfg[ETH_GPTP_TIMER_DOMAIN].ulGTOVt0 = 0UL;

    /* Set the lower 32 bit of second part of the GPTP timer GTOVt1 (t = 0 to PTPTN-1) */
    Eth_GstETNE_Regs.pGPTMA->stGptpCfg[ETH_GPTP_TIMER_DOMAIN].ulGTOVt1 = 0UL;

    /* Set the upper 16 bit of second part of the GPTP timer GTOVt1 (t = 0 to PTPTN-1) */
    Eth_GstETNE_Regs.pGPTMA->stGptpCfg[ETH_GPTP_TIMER_DOMAIN].ulGTOVt2 = 0UL;
    #endif
  } /* else: No action required */

  return LucResult;
}

/***********************************************************************************************************************
** Function Name         : Eth_HwInit (ETNE)
**
** Service ID            : N/A
**
** Description           : Initialize a ETNE HWUnit
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx     : Index of a controller
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : E_OK : Success
**                         E_NOT_OK : Failure
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpCtrlConfigPtr, Eth_GaaCtrlStat
**                         Eth_GpETNE_MFWDRegs, Eth_GaaHeap,
**                         Eth_GaaMemPoolBufferTable, Eth_GaaRamSize
**
** Function(s) invoked   : Eth_GWCA_ModeSetting, Eth_Util_RamInit
**                         Eth_TxRxDescConfig, Eth_Util_RamSetCircularAddr,
**                         Eth_EthAgentInit
**
** Registers Used        : TDPC, TFT
**
** Reference ID          : ETH_DUD_ACT_308
** Reference ID          : ETH_DUD_ACT_308_GBL001, ETH_DUD_ACT_308_GBL002
** Reference ID          : ETH_DUD_ACT_308_GBL004
** Reference ID          : ETH_DUD_ACT_308_GBL005, ETH_DUD_ACT_308_GBL006
** Reference ID          : ETH_DUD_ACT_308_GBL007, ETH_DUD_ACT_308_GBL008
** Reference ID          : ETH_DUD_ACT_308_GBL009, ETH_DUD_ACT_308_GBL010
** Reference ID          : ETH_DUD_ACT_308_GBL011, ETH_DUD_ACT_308_GBL012
** Reference ID          : ETH_DUD_ACT_308_GBL013, ETH_DUD_ACT_308_GBL014
** Reference ID          : ETH_DUD_ACT_308_GBL015, ETH_DUD_ACT_308_GBL016
** Reference ID          : ETH_DUD_ACT_308_GBL017, ETH_DUD_ACT_308_GBL018
** Reference ID          : ETH_DUD_ACT_308_GBL019, ETH_DUD_ACT_308_GBL020
** Reference ID          : ETH_DUD_ACT_308_GBL021, ETH_DUD_ACT_308_GBL022
** Reference ID          : ETH_DUD_ACT_308_REG001, ETH_DUD_ACT_308_REG002
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_HwInit(CONST(uint32, AUTOMATIC) LulCtrlIdx)                                  /* PRQA S 1532 # JV-01 */
{
  P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;
  uint32 LulQPriorityVal;
  uint32 LulFrameTypeVal;
  uint32 LulTxQueueMax;
  uint32 LulRxQueueMax;
  uint32 LulCnt;
  Std_ReturnType LucResult;

  LpHwUnitConfig =
    (P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA))Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;              /* PRQA S 0316 # JV-01 */
  LulTxQueueMax = (uint32)LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue;
  LulRxQueueMax = (uint32)LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue;

  #if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
  Eth_GulTsDescTail = 0UL;
  #endif

  /* Initialize Statistics */
  #if (ETH_GET_COUNTER_VALUES_API == STD_ON || ETH_GET_TX_STATS_API == STD_ON || ETH_GET_RX_STATS_API == STD_ON)
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulStatsBroadcastPkts = 0UL;                                                    /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts = 0UL;                                                           /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsOctets = 0UL;                                                         /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsCrcAlignErrors = 0UL;                                                 /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsUndersizePkts = 0UL;                                                  /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsOversizePkts = 0UL;                                                   /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts64Octets = 0UL;                                                   /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts65to127Octets = 0UL;                                              /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts128to255Octets = 0UL;                                             /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts256to511Octets = 0UL;                                             /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts512to1023Octets = 0UL;                                            /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts1024to1518Octets = 0UL;                                           /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxUnicastFrames = 0UL;                                                       /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulTxNumberOfOctets = 0UL;                                                      /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulTxNUcastPkts = 0UL;                                                          /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulTxUniCastPkts = 0UL;                                                         /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulDropPktCrc = 0UL;                                                            /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulUndersizePkt = 0UL;                                                          /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulOversizePkt = 0UL;                                                           /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulAlgnmtErr = 0UL;                                                             /* PRQA S 2844 # JV-01 */
  #endif

  /*-------------------------------------------------------------------------*/
  /* GWCA initialization for TSNSWA                                          */
  /*-------------------------------------------------------------------------*/
  LucResult = Eth_GWCA_ModeSetting(ETH_ETNE_GWMC_DISABLE_MODE);
  LucResult |= Eth_GWCA_ModeSetting(ETH_ETNE_GWMC_CONFIG_MODE);
  if (E_OK == LucResult)
  {
    /* Set Tx Queue priority and Frame type */
    LulQPriorityVal = 0UL;
    LulFrameTypeVal = 0UL;
    for (LulCnt = 0UL; LulCnt < LulTxQueueMax; LulCnt++)                                                                /* PRQA S 2814 # JV-01 */
    {
      LulQPriorityVal |= LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulCnt].ulQueueId << (4UL * LulCnt);              /* PRQA S 2824, 3383 # JV-01, JV-01 */
      LulFrameTypeVal |= (uint32)LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulCnt].enFrameType << LulCnt;
    }
    /* The supported Tx Queue is 0-15(Port0 = 0..7, Port1 = 8..15) */
    Eth_GstETNE_Regs.pAXIBMI->ulTDPC[LulCtrlIdx] = LulQPriorityVal;                                                     /* PRQA S 2814, 2844 # JV-01, JV-01 */
    /* AXIBMI execute TFTP0-2 with p-frame statements and TFTP3-7 with e-frame statements */
    /* Since each statement operates as an independent core, it operates faster by enabling all statements */
    Eth_GstETNE_Regs.pAXIBMI->ulTFT = LulFrameTypeVal;

    Eth_GaaHeap[LulCtrlIdx] = Eth_Util_RamInit(Eth_GaaMemPoolBufferTable[LulCtrlIdx], Eth_GaaRamSize[LulCtrlIdx]);      /* PRQA S 2844, 0315 # JV-01, JV-01 */
    if (NULL_PTR != Eth_GaaHeap[LulCtrlIdx])
    {
      /* Tx/Rx descriptor chain setting */
      LucResult = Eth_TxRxDescConfig(LulCtrlIdx);
      if (E_OK == LucResult)
      {
        /* Set circular address for Tx bufffers */
        Eth_Util_RamSetCircularAddr(Eth_GaaHeap[LulCtrlIdx]);                                                           /* PRQA S 2844 # JV-01 */

        /* Rx Multicast(Mirroring) setting table is not supported */

        /* GWERDQD, GWPRDQD, GWEWMC, GWPWMC and GWRTC are default values */
      } /* else: No action required */
    }
    else
    {
      LucResult = E_NOT_OK;
    }

    /* Set AXIBMI Data Interrupt */
    /* TX Descriptor Data Interrupt */
    if (ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enTxInterruptMode)
    {

      Eth_GstETNE_Regs.pAXIBMI->stTXDI[0].ulTDIEi |= LpHwUnitConfig->stQueueConfig.ulTxIntConfig;
    } /* else: No action required */

    /* RX Descriptor Data Interrupt */
    if (ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enRxInterruptMode)
    {
      Eth_GstETNE_Regs.pAXIBMI->stRXDI[0].ulRDIEi |= LpHwUnitConfig->stQueueConfig.ulRxIntConfig;
    } /* else: No action required */

    #if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
    /* TS Descriptor Data Interrupt */
    Eth_GstETNE_Regs.pAXIBMI->ulTSDIE = ETH_ETNDE_TSDIE_CONFIG;
    #endif

    /* General Interrupt Enable */
    RH850_SV_MODE_ICR_AND(8, &Eth_GpETNE_EICRegs->usINTETNE07, (uint8)(~ETH_EIC_EIMK_MASK));                            /* PRQA S 0404, 0751, 3464 # JV-01, JV-01, JV-01 */
    RH850_SV_MODE_ICR_AND(8, &Eth_GpETNE_EICRegs->usINTETNE08, (uint8)(~ETH_EIC_EIMK_MASK));                            /* PRQA S 0404, 0751, 3464 # JV-01, JV-01, JV-01 */

    /* Tx/Rx Descriptor Data Interrupt Enable */
    if (0UL != (((LpHwUnitConfig->stQueueConfig.ulTxIntConfig >> (LulCtrlIdx * LulTxQueueMax)) |
                 (LpHwUnitConfig->stQueueConfig.ulRxIntConfig >> (LulCtrlIdx * LulRxQueueMax)))
                & ETH_QUEUE_ID_0_4_MASK))
    {
      RH850_SV_MODE_ICR_AND(8, &Eth_GpETNE_EICRegs->usINTETNE00, (uint8)(~ETH_EIC_EIMK_MASK));                          /* PRQA S 0404, 0751, 3464 # JV-01, JV-01, JV-01 */
    } /* else: No action required */

    if (0UL != (((LpHwUnitConfig->stQueueConfig.ulTxIntConfig >> (LulCtrlIdx * LulTxQueueMax)) |
                 (LpHwUnitConfig->stQueueConfig.ulRxIntConfig >> (LulCtrlIdx * LulRxQueueMax)))
                & ETH_QUEUE_ID_1_5_MASK))
    {
      RH850_SV_MODE_ICR_AND(8, &Eth_GpETNE_EICRegs->usINTETNE01, (uint8)(~ETH_EIC_EIMK_MASK));                          /* PRQA S 0751, 0404, 3464 # JV-01, JV-01, JV-01 */
    } /* else: No action required */

    if (0UL != (((LpHwUnitConfig->stQueueConfig.ulTxIntConfig >> (LulCtrlIdx * LulTxQueueMax)) |
                 (LpHwUnitConfig->stQueueConfig.ulRxIntConfig >> (LulCtrlIdx * LulRxQueueMax)))
                & ETH_QUEUE_ID_2_6_MASK))
    {
      RH850_SV_MODE_ICR_AND(8, &Eth_GpETNE_EICRegs->usINTETNE02, (uint8)(~ETH_EIC_EIMK_MASK));                          /* PRQA S 0751, 0404, 3464 # JV-01, JV-01, JV-01 */
    } /* else: No action required */

    if (0UL != (((LpHwUnitConfig->stQueueConfig.ulTxIntConfig >> (LulCtrlIdx * LulTxQueueMax)) |
                 (LpHwUnitConfig->stQueueConfig.ulRxIntConfig >> (LulCtrlIdx * LulRxQueueMax)))
                & ETH_QUEUE_ID_3_7_MASK))
    {
      RH850_SV_MODE_ICR_AND(8, &Eth_GpETNE_EICRegs->usINTETNE03, (uint8)(~ETH_EIC_EIMK_MASK));                          /* PRQA S 0751, 0404, 3464 # JV-01, JV-01, JV-01 */
    } /* else: No action required */

    if (ETH_SGMII == Eth_GpEthConfigPtr[LulCtrlIdx].enEthPHYInterface)
    {
      LucResult = Eth_SGMIIInit(LulCtrlIdx, LpHwUnitConfig->enBypassMode);
    } /* else: No action required */
  } /* else: No action required */

  if (E_OK == LucResult)
  {
    /*-----------------------------------------------------------------------*/
    /* Ethernet-Agent initialization for TSNSWA                              */
    /*-----------------------------------------------------------------------*/
    LucResult = Eth_EthAgentInit(LulCtrlIdx);

    /* Set GWMC mode to DISABLE */
    LucResult |= Eth_GWCA_ModeSetting(ETH_ETNE_GWMC_DISABLE_MODE);

    if (E_OK == LucResult)
    {
      /* Set GWMC mode to OPERATION */
      LucResult = Eth_GWCA_ModeSetting(ETH_ETNE_GWMC_OPERATION_MODE);
    } /* else: No action required */
  } /* else: No action required */

  return LucResult;
}

/***********************************************************************************************************************
** Function Name         : Eth_GWCA_ModeSetting
**
** Service ID            : NA
**
** Description           : Set the mode of GWCA
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulMode : GWCA Mode
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : E_OK - Succeeded to change GWCA mode.
**                         E_NOT_OK - Failed to change GWCA mode.
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpETNE_MFWDRegs
**
** Function(s) invoked   : GetCounterValue, Eth_GetTimeOutValue
**
** Registers Used        : GWMC, GWMS
**
** Reference ID          : ETH_DUD_ACT_335
** Reference ID          : ETH_DUD_ACT_335_REG001
***********************************************************************************************************************/
STATIC FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_GWCA_ModeSetting(CONST(uint32, AUTOMATIC) LulMode)
{
  TickType LulTickStart;
  TickType LulTickElap;
  Std_ReturnType LucResult;

  /* Switch to CONFIG mode (GWMC.OPC bit) */
  Eth_GstETNE_Regs.pGWCA->ulGWMC = LulMode;                                                                             /* PRQA S 2814 # JV-01 */

  /* Wait while CONFIG mode is achieved */
  LulTickStart = 0UL;
  (void)GetCounterValue(ETH_OS_COUNTER_ID, &LulTickStart);
  do
  {
    LulTickElap = Eth_GetTimeOutValue(LulTickStart);
  } while ((Eth_GstETNE_Regs.pGWCA->ulGWMS != LulMode) && (LulTickElap <= ETH_TIMEOUT_COUNT));

  if (ETH_TIMEOUT_COUNT < LulTickElap)
  {
    LucResult = E_NOT_OK;
  }
  else
  {
    LucResult = E_OK;
  }

  return LucResult;
}

/***********************************************************************************************************************
** Function Name         : Eth_ETHA_ModeSetting
**
** Service ID            : NA
**
** Description           : Set the mode of EthAgent
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx : Index of a controller
**                         LulMode : EthAgent Mode
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : E_OK - Succeeded to change ETHA mode.
**                         E_NOT_OK - Failed to change ETHA mode.
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpETNE_MFWDRegs
**
** Function(s) invoked   : GetCounterValue, Eth_GetTimeOutValue
**
** Registers Used        : EAMC, EAMS
**
** Reference ID          : ETH_DUD_ACT_336
** Reference ID          : ETH_DUD_ACT_336_REG001, ETH_DUD_ACT_336_REG002
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_ETHA_ModeSetting(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulMode)
{
  TickType LulTickStart;
  TickType LulTickElap;
  Std_ReturnType LucResult;

  /* Switch to CONFIG mode (EAMC.OPC bit) */
  Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulEAMC = LulMode;                                                                   /* PRQA S 2844, 2814 # JV-01, JV-01 */

  /* Wait while CONFIG mode is achieved */
  LulTickStart = 0UL;
  (void)GetCounterValue(ETH_OS_COUNTER_ID, &LulTickStart);
  do
  {
    LulTickElap = Eth_GetTimeOutValue(LulTickStart);
  } while (((Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulEAMS & ETH_ETNE_EAMS_OPS_MASK) != LulMode) &&
           (LulTickElap <= ETH_TIMEOUT_COUNT));

  if (ETH_TIMEOUT_COUNT < LulTickElap)
  {
    LucResult = E_NOT_OK;
  }
  else
  {
    LucResult = E_OK;
  }

  return LucResult;
}

/***********************************************************************************************************************
** Function Name         : Eth_EthAgentInit
**
** Service ID            : NA
**
** Description           : Create descriptor chain & lerning descriptor.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx : Index of a controller
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : E_OK - Success initialization of Ethernet Agent.
**                         E_NOT_OK - Faile initialization of Ethernet Agent.
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpCtrlConfigPtr, Eth_GpEthConfigPtr,
**                         Eth_GaaETNE_ETHARegs, Eth_GaaCtrlStat
**
** Function(s) invoked   : Eth_ETHA_ModeSetting
**
** Registers Used        : EATMS, EATDQD, EATWMC, EATWMC, EATMTC, EATAFS,
**                         EAPRC, TPTPC, TTML, TTJ, CACC, CAIV, CAUL, CCS,
**                         MRMAC0, MRMAC1, MPIC, MTFFC, MTPFC, MTATC,
**                         MRGC, MRFSCE, MRFSCP, MTRC, MRSCE, MRSCP, MRAFC
**
** Reference ID          : ETH_DUD_ACT_340
** Reference ID          : ETH_DUD_ACT_340_REG001, ETH_DUD_ACT_340_REG002
** Reference ID          : ETH_DUD_ACT_340_REG003, ETH_DUD_ACT_340_REG004
** Reference ID          : ETH_DUD_ACT_340_REG005, ETH_DUD_ACT_340_REG004
** Reference ID          : ETH_DUD_ACT_340_REG007, ETH_DUD_ACT_340_REG008
** Reference ID          : ETH_DUD_ACT_340_REG009, ETH_DUD_ACT_340_REG010
** Reference ID          : ETH_DUD_ACT_340_REG011, ETH_DUD_ACT_340_REG012
** Reference ID          : ETH_DUD_ACT_340_REG013, ETH_DUD_ACT_340_REG014
** Reference ID          : ETH_DUD_ACT_340_REG015, ETH_DUD_ACT_340_REG016
** Reference ID          : ETH_DUD_ACT_340_REG017, ETH_DUD_ACT_340_REG018
** Reference ID          : ETH_DUD_ACT_340_REG019, ETH_DUD_ACT_340_REG020
** Reference ID          : ETH_DUD_ACT_340_REG021, ETH_DUD_ACT_340_REG022
***********************************************************************************************************************/
STATIC FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_EthAgentInit(CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;
  uint32 LulCnt;
  uint32 LulFrameTypeVal;
  Std_ReturnType LucResult;

  LucResult = E_OK;                                                                                                     /* PRQA S 2982 # JV-01 */

  LpHwUnitConfig =
    (P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA))Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;              /* PRQA S 0316 # JV-01 */

  /* 1.Transit to CONFIG mode */
  /* Set EAMC mode to DISABLE */
  LucResult = Eth_ETHA_ModeSetting(LulCtrlIdx, ETH_ETNE_ETHA_DISABLE_MODE);
  if (E_OK == LucResult)
  {
    /* Set EAMC mode to CONFIG */
    LucResult = Eth_ETHA_ModeSetting(LulCtrlIdx, ETH_ETNE_ETHA_CONFIG_MODE);
    if (E_OK == LucResult)
    {
      /* 2.Transmission configuration */
      LulFrameTypeVal = 0UL;
      for (LulCnt = 0UL; LulCnt < (uint32)LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue; LulCnt++)                    /* PRQA S 2814 # JV-01 */
      {
        LulFrameTypeVal |= (uint32)((uint32)LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulCnt].enFrameType << LulCnt);/* PRQA S 2824 # JV-01 */
        Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulEATMS[LulCnt] =                                                             /* PRQA S 2844, 2814 # JV-01, JV-01 */
          LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulCnt].ulMaxFrameSize;
        Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulEATDQD[LulCnt] = ETH_ETNE_EATDQD_CONFIG;
        Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulEATWMC[LulCnt] = 0UL;
      }
      Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulEATMTC = LulFrameTypeVal;
      Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulEATAFS = (uint32)LpHwUnitConfig->enTxFragment;
      Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulEAPRC  = ETH_ETNE_EAPRC_CONFIG;
      /* 3.TAS configuration */
      Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulTPTPC = ETH_GPTP_TIMER_DOMAIN;
      Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulTTML  = LpHwUnitConfig->stTASConfig.ulTxMiniLatency;
      Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulTTJ   = LpHwUnitConfig->stTASConfig.ulTxJitter;
      /* 4.CBS configuration */
      for (LulCnt = 0UL; LulCnt < (uint32)LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue; LulCnt++)
      {
        Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulCACC |=
          (uint32)LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulCnt].stCBSConfig.enCE << LulCnt;
        Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulCAIV[LulCnt] =
          LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulCnt].stCBSConfig.ulCIV;
        Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulCAUL[LulCnt] =
          LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulCnt].stCBSConfig.ulCUL;
        Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulCCS |=
          (uint32)LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulCnt].stCBSConfig.enCE << LulCnt;
      }
      /* Ethernet agent error interrupt enable */
      Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulEAEIE0 = ETH_ETNE_EAEIE0_CONFIG;
    } /* else: No action required */
  } /* else: No action required */

  /* 5.RMAC System configuration */
  if (E_OK == LucResult)
  {
    /* Set MAC address */
    /* When MAC address is 01-23-45-67-89-AB, set 0x0123 */
    Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMRMAC0 =
      Eth_GaaCtrlStat[LulCtrlIdx].stMacAddr.ulH32 >> 16UL;                                                              /* PRQA S 2844 # JV-01 */
    /* When MAC address is 01-23-45-67-89-AB, set 0x456789AB */
    Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMRMAC1 =
      (Eth_GaaCtrlStat[LulCtrlIdx].stMacAddr.ulH32 << 16UL) | Eth_GaaCtrlStat[LulCtrlIdx].stMacAddr.ulL16;

    /* PHY Interface configuration */
    Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMPIC =
      (uint32)(((uint32)Eth_GpEthConfigPtr[LulCtrlIdx].enEthSpeed << 2UL) |
               (uint32)Eth_GpEthConfigPtr[LulCtrlIdx].enEthPHYInterface);

    /* Config MAC Loopback mode */
    if (ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enInternalLoopBackMode)
    {
      Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMLBC = ETH_ETNDE_MLBC_CONFIG;
    } /* else: No action required */

    /* Set Tx function */
    Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMTFFC = 0UL;
    if (ETH_ENABLE == LpHwUnitConfig->stMACConfig.enPauseFrame)
    {
      Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMTPFC =
        (uint32)(((uint32)LpHwUnitConfig->stMACConfig.enPauseTimeZero << 24UL) |
                 (LpHwUnitConfig->stMACConfig.ulRetransmissionTime << 16UL) |
                 LpHwUnitConfig->stMACConfig.ulPauseTime);
    } /* else: No action required */
    /* Do not use automatic timestamp */
    Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMTATC[ETH_GPTP_TIMER_DOMAIN] = ETH_ETNDE_MTATC_CONFIG;

    /* Set Rx function */
    Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMRGC =
      (uint32)(((uint32)LpHwUnitConfig->stMACConfig.enPauseTimeZero << 2UL) |
               ((uint32)LpHwUnitConfig->stMACConfig.enPauseFrame << 1UL));
    Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMRFSCE = LpHwUnitConfig->stRxConfig.ulMaxFrameSize;
    Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMRFSCP = LpHwUnitConfig->stRxConfig.ulMaxFrameSize;

    /* Timestamp capture settings for Rx descriptor */
    Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMTRC = ETH_ETNDE_MTRC_CONFIG;

    /* Set Address Filtering function */
    Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMRSCE =
      (LpHwUnitConfig->stRxConfig.ulBcastThreshold << 16UL) | LpHwUnitConfig->stRxConfig.ulMcastThreshold;
    Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMRSCP =
      (LpHwUnitConfig->stRxConfig.ulBcastThreshold << 16UL) | LpHwUnitConfig->stRxConfig.ulMcastThreshold;
    Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC.ulMRAFC =
      ETH_ETNDE_MRAFC_CONFIG | (uint32)(((uint32)LpHwUnitConfig->stRxConfig.enBcastStormFilter << 20UL) |
                                        ((uint32)LpHwUnitConfig->stRxConfig.enMcastStormFilter << 19UL) |
                                        ((uint32)LpHwUnitConfig->stRxConfig.enBcastStormFilter << 4UL) |
                                        ((uint32)LpHwUnitConfig->stRxConfig.enMcastStormFilter << 3UL));

    /* PTP Filtering function is not supported */
  } /* else: No action required */

  return LucResult;
}

#define ETH_STOP_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */


/***********************************************************************************************************************
**                                                    End of File                                                     **
***********************************************************************************************************************/
