/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Icu_PIN_LLDriver.c                                                                                  */
/* SW-VERSION   = 1.5.0                                                                                               */
/*====================================================================================================================*/
/*                                                     COPYRIGHT                                                      */
/*====================================================================================================================*/
/* (c) 2020-2022 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* This file contains Low Level function implementations of ICU Driver Component                                      */
/*                                                                                                                    */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s) and/or */
/* the Application.                                                                                                   */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs of  */
/* program errors, compliance with applicable laws, damage to or loss of data, programs or equipment, and             */
/* unavailability or interruption of operations.                                                                      */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*              Devices:        X2x                                                                                   */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                              Revision Control History                                              **
***********************************************************************************************************************/
/*
 * 1.4.4:  17/11/2022  : As per ticket ARDAACA-1266 and ARDAACA-1302, update "Input Parameters", "Global Variables", 
 *                       "Functions invoked" for all functions.
 *                       Add IDs ICU_DUD_ACT_057_CRT001, ICU_DUD_ACT_057_CRT002 for Icu_HW_Pin_ExternalInterruptIsr
 *         16/06/2022  : Update QAC message and QAC header
 *         09/06/2022  : Update access register FCLAnCTLm from 16 to 8 bit
 * 1.4.3:  23/05/2022  : Fix QAC header and message.
 *         09/05/2022  : Remove "else" statement when no action required; add QAC message 2004.
 *         17/04/2022  : Merge QAC message from U2Ax, E2x and U2Bx
 * 1.4.2:  02/03/2022  : Fix QAC header and message
 * 1.4.1:  18/10/2021  : Add QAC message 9.5.0
 *         16/09/2021  : Correct preprocessor directives
 *                       Correct name, data type of used variables
 *                       Correct indents, comments
 * 1.3.2:  30/08/2021  : Change from #include "Icu_MemMap.h" to #include "Icu_Mapping.h".
 *                       Update pre-compile parameter ICU_E_INT_INCONSISTENT_CONFIGURED for ICU_DEM_REPORT_ERROR
 * 1.3.1:  02/07/2021  : Add QAC message 9.5.0
 *                       Format source code to 120 characters
 *                       Improve Violation tag (remove START/END)
 * 1.2.0:  24/08/2020  : Release
 * 1.1.1:  14/08/2020  : Corrected the resetting of EXINTCTL in
 *                       Icu_HW_Pin_ChannelDeInit().
 * 1.1.0:  19/06/2020  : Release
 * 1.0.0:  25/03/2020  : Initial Version
 */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (2:0317)    : Implicit conversion from a pointer to void to a pointer to object type.                      */
/* Rule                : MISRA C:2012 Rule-11.5                                                                       */
/* JV-01 Justification : This is done as per implementation requirement.                                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact               */
/**********************************************************************************************************************/
/* Message (2:0751)    : Casting to char pointer type.                                                                */
/* Rule                : CERTCCM EXP11, EXP39                                                                         */
/* JV-01 Justification : Using void due to specific requirement of input parameter. So, this can be skipped           */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (7:0791)    : Macro identifier does not differ from other macro identifier(s) (e.g. '%s') within the       */
/*                       specified number of significant characters.                                                  */
/* Rule                : CERTCCM DCL23, MISRA C:2012 Rule-5.4                                                         */
/* JV-01 Justification : This macro identifier is following AUTOSAR standard rule (Symbolic Name or Published         */
/*                       Macro's name), so this is accepted.                                                          */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0857)    : Number of macro definitions exceeds 1024 - program does not conform strictly to ISO:C90.     */
/* Rule                : MISRA C:2012 Dir-1.1                                                                         */
/* JV-01 Justification : The number of macro depend on module code size. There is no issue when number of macro is    */
/*                       over 1024                                                                                    */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1281)    : Integer literal constant is of an unsigned type but does not include a "U" suffix.           */
/* Rule                : MISRA C:2012 Rule-7.2                                                                        */
/* JV-01 Justification : Integer literal constant is of an unsigned type but does not include a "U" suffix.           */
/*       Verification  : Lack of using "U" is accepted since it cause no problem in software behavior.                */
/**********************************************************************************************************************/
/* Message (1:1532)    : The function '%1s' is only referenced in one translation unit - but not the one in which it  */
/*                       is defined.                                                                                  */
/* Rule                : CERTCCM DCL19, MISRA C:2012 Rule-8.7                                                         */
/* JV-01 Justification : This macro identifier is following AUTOSAR standard rule  (Symbolic Name or Published        */
/*                       Macro's name), so this is accepted                                                           */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:2004)    : No concluding 'else' exists in this 'if'-'else'-'if' statement.                              */
/* Rule                : CERTCCM MSC01, MISRA C:2012 Rule-15.7                                                        */
/* JV-01 Justification : The "else" statement with empty content is removed to improve readability.                   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : This is accepted, due to the implementation is following hardware specification or it was    */
/*                       checked or do not dereference to NULL pointer.                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : It is specific for device register accessing and confirmed has no issue in software behavior.*/
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3383)    : Cannot identify wraparound guard for unsigned arithmetic expression.                         */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : It can still result in values that are out of range for the intended use, as intuitive       */
/*                       "invariants" may not hold                                                                    */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3678)    : The object referenced by '%1s' is not modified through it, so '%1s' could be declared with   */
/*                       type '%2s'.                                                                                  */
/* Rule                : MISRA C:2012 Rule-8.13                                                                       */
/* JV-01 Justification : This is accepted. It just an advise for improve safety by reducing the possibility that the  */
/*                       referenced data is unintentionally modified through an unexpected alias and improves         */
/*                       clarity by indicating that the referenced data is not intended to be modified through this   */
/*                       alias or those depending on it                                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:4342)    : An expression of 'essentially unsigned' type (%1s) is being cast to enum type '%2s'.         */
/* Rule                : MISRA C:2012 Rule-10.5                                                                       */
/* JV-01 Justification : It is assigned to a variable with no conflict in the data.                                   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : Use of #include directive after code fragment.                                               */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is done as per Memory Requirement, (MEMMAP003 - Specification of Memory Mapping).       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                  Include Section                                                   **
***********************************************************************************************************************/

/* Included for Header file inclusion */
#include "Icu_PIN_LLDriver.h"                                                                                           /* PRQA S 0857 # JV-01 */
/* Included for RAM variable declarations */
#include "Icu_Ram.h"

#if (ICU_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
/* Include for interrupt consistency check */
#include "Dem.h"
#endif
/***********************************************************************************************************************
**                                                Version Information                                                 **
***********************************************************************************************************************/
/* AUTOSAR release version information */
#define ICU_PIN_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION    ICU_AR_RELEASE_MAJOR_VERSION_VALUE                               /* PRQA S 0857 # JV-01 */
#define ICU_PIN_LLDRIVER_C_AR_RELEASE_MINOR_VERSION    ICU_AR_RELEASE_MINOR_VERSION_VALUE                               /* PRQA S 0791 # JV-01 */
#define ICU_PIN_LLDRIVER_C_AR_RELEASE_REVISION_VERSION ICU_AR_RELEASE_REVISION_VERSION_VALUE

/* File version information */
#define ICU_PIN_LLDRIVER_C_SW_MAJOR_VERSION            ICU_SW_MAJOR_VERSION_VALUE
#define ICU_PIN_LLDRIVER_C_SW_MINOR_VERSION            ICU_SW_MINOR_VERSION_VALUE                                       /* PRQA S 0857 # JV-01 */

/***********************************************************************************************************************
**                                                   Version Check                                                    **
***********************************************************************************************************************/
#if (ICU_PIN_LLDRIVER_AR_RELEASE_MAJOR_VERSION != ICU_PIN_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION)
  #error "Icu_PIN_LLDriver.c : Mismatch in Release Major Version"
#endif

#if (ICU_PIN_LLDRIVER_AR_RELEASE_MINOR_VERSION != ICU_PIN_LLDRIVER_C_AR_RELEASE_MINOR_VERSION)
  #error "Icu_PIN_LLDriver.c : Mismatch in Release Minor Version"
#endif

#if (ICU_PIN_LLDRIVER_AR_RELEASE_REVISION_VERSION != ICU_PIN_LLDRIVER_C_AR_RELEASE_REVISION_VERSION)
  #error "Icu_PIN_LLDriver.c : Mismatch in Release Revision Version"
#endif

#if (ICU_PIN_LLDRIVER_SW_MAJOR_VERSION != ICU_PIN_LLDRIVER_C_SW_MAJOR_VERSION)
  #error "Icu_PIN_LLDriver.c : Mismatch in Software Major Version"
#endif

#if (ICU_PIN_LLDRIVER_SW_MINOR_VERSION != ICU_PIN_LLDRIVER_C_SW_MINOR_VERSION)
  #error "Icu_PIN_LLDriver.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                                    Global Data                                                     **
***********************************************************************************************************************/

/***********************************************************************************************************************
**                                                Function Definitions                                                **
***********************************************************************************************************************/

/***********************************************************************************************************************
** Function Name        : Icu_HW_Pin_ChannelInit
**
** Service ID           : None
**
** Description          : This service initializes the hardware for all the
**                        configured channels based on the measurement mode.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Non Reentrant
**
** Input Parameters     : LucChannelIndex
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelConfig, Icu_GpChannelRamData, Icu_GpChannelUserRegConfig.
**
** Functions invoked    : Icu_HW_Pin_SetActivation
**
** Registers Used       : EICn
**
** Reference ID         : ICU_DUD_ACT_058, ICU_DUD_ACT_058_GBL001
** Reference ID         : ICU_DUD_ACT_058_GBL004, ICU_DUD_ACT_058_GBL002, ICU_DUD_ACT_058_GBL005
** Reference ID         : ICU_DUD_ACT_058_GBL003, ICU_DUD_ACT_058_REG002, ICU_DUD_ACT_058_REG003
** Reference ID         : ICU_DUD_ACT_058_REG004, ICU_DUD_ACT_058_REG005, ICU_DUD_ACT_058_REG001
***********************************************************************************************************************/
#if ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT))
#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Pin_ChannelInit(const Icu_ChannelType LucChannelIndex)                              /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;
  /* Defining a pointer to the Channel Ram Data */
  P2VAR(volatile Icu_ChannelRamDataType, AUTOMATIC, ICU_CONFIG_DATA) LpRamData;
  /* INTP */
  #if ((ICU_INTP_EDGE_DETECT == STD_ON) && (ICU_EDGE_DETECT_API == STD_ON))
  /* Local variable to store the measurement mode of a channel */
  Icu_MeasurementModeType LenMeasurementMode;
  #endif /* End of (ICU_INTP_EDGE_DETECT == STD_ON) && (ICU_EDGE_DETECT_API == STD_ON)) */
  /* IRQ */
  #if (ICU_IRQ_EDGE_DETECT == STD_ON)
  /* Defining a pointer to point to the External Interrupt registers */
  P2VAR(volatile uint16, AUTOMATIC, ICU_CONFIG_DATA) LpExtIntpcntRegs;
  /* Defining a pointer to point to the External Interrupt registers */
  P2VAR(volatile uint8, AUTOMATIC, ICU_CONFIG_DATA) LpEXINTSTCRegs;
  /* Defining a pointer to the timer channel configuration parameters */
  P2CONST(Icu_PINChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpPINChannelUserRegConfig;
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_PINChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpPinChannelConfig;
  #endif

  /* Update the channel configuration pointer to point to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LucChannelIndex];

  /* Update the global pointer with the first channel's ram address */
  LpRamData = &Icu_GpChannelRamData[LucChannelIndex];
  /* Disable notification for each channel */
  LpRamData->ucNotificationEnable = (uint8)ICU_FALSE;                                                                   /* PRQA S 2814, 2844 # JV-01, JV-01 */
  /* Initialize each channel status as idle */
  LpRamData->ucChannelStatus = (uint8)ICU_IDLE;

  /* IRQ */
  #if (ICU_IRQ_EDGE_DETECT == STD_ON)

  LpPinChannelConfig = Icu_GpChannelConfig[LucChannelIndex].pHWIPChannelConfig;                                         /* PRQA S 0317 # JV-01 */
  /* Update the PIN channel configuration pointer to point to the current channel */
  LpPINChannelUserRegConfig = Icu_GpChannelUserRegConfig[LucChannelIndex].pHWChannelUserRegConfig;                      /* PRQA S 0317 # JV-01 */
  /* Initialize all the configured Icu Channels */
  if (ICU_HW_EXT_IRQ == LpChannelConfig->ucIcuChannelType)                                                              /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    LpExtIntpcntRegs = LpChannelConfig->pCntlRegs;                                                                      /* PRQA S 0317 # JV-01 */
    LpEXINTSTCRegs = LpPinChannelConfig->pEXINSTCRegsAddress;                                                           /* PRQA S 0317, 2814 # JV-01, JV-01 */

    if ((uint8)ICU_RISING_EDGE == LpChannelConfig->ucIcuDefaultStartEdge)
    {
      *(LpExtIntpcntRegs) = (uint16)(*(LpExtIntpcntRegs) & (LpPINChannelUserRegConfig->ddChannelModeUserRegSettings)) | /* PRQA S 2814 # JV-01 */
                         (uint16)((uint16)ICU_IRQ_RISING_EDGE_MASK << (LpChannelConfig->usChannelMaskValue * ICU_TWO)); /* PRQA S 3383 # JV-01 */
    }
    else if ((uint8)ICU_FALLING_EDGE == LpChannelConfig->ucIcuDefaultStartEdge)                                         /* PRQA S 2004 # JV-01 */
    {
      *(LpExtIntpcntRegs) = (uint16)(*(LpExtIntpcntRegs) & (LpPINChannelUserRegConfig->ddChannelModeUserRegSettings)) |
                      ((uint16)((uint16)ICU_IRQ_FALLING_EDGE_MASK << (LpChannelConfig->usChannelMaskValue * ICU_TWO))); /* PRQA S 3383 # JV-01 */
    } /* else No action required */

    /* Clear the IRQ status flag */
    *(LpEXINTSTCRegs) = (uint8)(ICU_ONE << (uint8)LpChannelConfig->usChannelMaskValue);                                 /* PRQA S 2814 # JV-01 */
  } /* else No action required */

  if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)
  {
    /* Disable interrupts */
    RH850_SV_SET_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_DISABLE_EIMK_MASK);                              /* PRQA S 0751, 2814 # JV-01, JV-01 */
  } /* else No action required */
  #endif /* End of ICU_IRQ_EDGE_DETECT == STD_ON */
  /* INTP */
  #if (ICU_INTP_EDGE_DETECT == STD_ON)
  /* Initialize all the configured Icu Channels */
  #if ((ICU_DISABLE_WAKEUP_API == STD_ON) || (ICU_ENABLE_WAKEUP_API == STD_ON) || \
                                                                               (ICU_WAKEUP_FUNCTIONALITY_API == STD_ON))
  /* Disable each channel wakeup from sleep mode */
  LpRamData->ucWakeupEnable = (uint8)ICU_FALSE;
  /* Initialize wakeup occurrence for each channel */
  LpRamData->ucWakeupOccurrence = (uint8)ICU_FALSE;
  #endif
  #if ((STD_ON == ICU_SET_MODE_API) || (STD_ON == ICU_BUSY_OPERATION_DET_CHECK) || (ICU_EDGE_DETECT_API == STD_ON))
  /* Initialize running status for each channel */
  LpRamData->ucChannelRunning = (uint8)ICU_FALSE;
  #endif

  #if (ICU_EDGE_DETECT_API == STD_ON)
  /* Read the channel's measurement mode */
  LenMeasurementMode = (Icu_MeasurementModeType)(LpChannelConfig->ucIcuMeasurementMode);                                /* PRQA S 2814, 2844, 4342 # JV-01, JV-01, JV-01 */

  #endif /* End of (ICU_EDGE_DETECT_API == STD_ON) */

  if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)
  {
    /* Disable the Interrupt processing of the current channel */
    RH850_SV_SET_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_DISABLE_EIMK_MASK);                              /* PRQA S 0751, 2814 # JV-01, JV-01 */
  } /* else No action required */

  #if (ICU_EDGE_DETECT_API == STD_ON)
  if (ICU_MODE_SIGNAL_EDGE_DETECT == LenMeasurementMode)
  {
    /* Configure external interrupt for the active edge */
    Icu_HW_Pin_SetActivation((Icu_ChannelType)LucChannelIndex,
                                                           (Icu_ActivationType)LpChannelConfig->ucIcuDefaultStartEdge); /* PRQA S 4342 # JV-01 */
  } /* else No action required */
  /* End of ICU_MODE_SIGNAL_EDGE_DETECT == LenMeasurementMode */
  #endif /* End of (ICU_EDGE_DETECT_API == STD_ON) */

  #endif /* ICU_INTP_EDGE_DETECT == STD_ON */
} /* End of Internal function Icu_HW_Pin_ChannelInit */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT)) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Pin_ChannelDeInit
**
** Service ID           : None
**
** Description          : This service De-Initializes the hardware for all the
**                        configured channels based on the measurement mode.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Non Reentrant
**
** Input Parameters     : LucChannelIndex
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelConfig, Icu_GpChannelRamData, Icu_GpChannelUserRegConfig.
**
** Functions invoked    : RH850_SV_MODE_REG_READ_ONLY, RH850_SV_CLEAR_ICR_SYNCP, EXECUTE_SYNCP.
**
** Registers Used       : EICn, FCLAnCTLm.
**
** Reference ID         : ICU_DUD_ACT_059, ICU_DUD_ACT_059_REG001, ICU_DUD_ACT_059_REG003
** Reference ID         : ICU_DUD_ACT_059_GBL002, ICU_DUD_ACT_059_GBL001, ICU_DUD_ACT_059_GBL004
** Reference ID         : ICU_DUD_ACT_059_GBL003, ICU_DUD_ACT_059_GBL006, ICU_DUD_ACT_059_REG002
***********************************************************************************************************************/
#if ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT))
#if (ICU_DE_INIT_API == STD_ON)

#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Pin_ChannelDeInit(const Icu_ChannelType LucChannelIndex)                            /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;

  /* Defining a pointer to the Timer registers */
  P2VAR(volatile Icu_ChannelRamDataType, AUTOMATIC, ICU_CONFIG_DATA) LpRamData;

  /* INTP */
  #if ((ICU_EDGE_DETECT_API == STD_ON) && (ICU_INTP_EDGE_DETECT == STD_ON))
  /* Defining a pointer to point to the External Interrupt registers */
  P2VAR(volatile uint8, AUTOMATIC, ICU_CONFIG_DATA) LpFCLAnCTLAddress;
  #endif /* End of (ICU_EDGE_DETECT_API == STD_ON) */

  #if (ICU_EDGE_DETECT_API == STD_ON)
  #if (ICU_IRQ_EDGE_DETECT == STD_ON)
  /* Defining a pointer to the timer channel configuration parameters */
  P2CONST(Icu_PINChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpPINChannelUserRegConfig;
  /* Defining a pointer to point to the External Interrupt registers */
  P2VAR(volatile uint16, AUTOMATIC, ICU_CONFIG_DATA) LpExtIntpcntRegs;                                                  /* PRQA S 3678 # JV-01 */
  #endif
  /* Local variable to store the measurement mode of a channel */
  Icu_MeasurementModeType LenMeasurementMode;
  #endif

  /* Load the channel data pointer */
  LpRamData = &Icu_GpChannelRamData[LucChannelIndex];
  /* Disable notification for each channel */
  LpRamData->ucNotificationEnable = (uint8)ICU_FALSE;                                                                   /* PRQA S 2814, 2844 # JV-01, JV-01 */

  /* Initialize each channel status as idle */
  LpRamData->ucChannelStatus = (uint8)ICU_IDLE;

  /* Update the channel configuration pointer to point to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LucChannelIndex];
  if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)                                                                    /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    /* Disabling the Interrupt processing */
    RH850_SV_SET_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_DISABLE_EIMK_MASK);                              /* PRQA S 0751, 2814 # JV-01, JV-01 */
  } /* else No action required */
  /* INTP */
  #if (ICU_INTP_EDGE_DETECT == STD_ON)

  #if ((ICU_DISABLE_WAKEUP_API == STD_ON) || (ICU_ENABLE_WAKEUP_API == STD_ON) || \
                                                                               (ICU_WAKEUP_FUNCTIONALITY_API == STD_ON))
  /* Disable each channel wakeup from sleep mode */
  LpRamData->ucWakeupEnable = (uint8)ICU_FALSE;

  /* Initialize wakeup occurrence for each channel */
  LpRamData->ucWakeupOccurrence = (uint8)ICU_FALSE;
  #endif

  #if ((STD_ON == ICU_SET_MODE_API) || (STD_ON == ICU_BUSY_OPERATION_DET_CHECK) || (ICU_EDGE_DETECT_API == STD_ON))
  /* Initialize running status for each channel */
  LpRamData->ucChannelRunning = (uint8)ICU_FALSE;
  #endif

  #endif /* End of (ICU_INTP_EDGE_DETECT == STD_ON) */

  #if (ICU_EDGE_DETECT_API == STD_ON)
  /* Read the channel's measurement mode */
  LenMeasurementMode = (Icu_MeasurementModeType)(LpChannelConfig->ucIcuMeasurementMode);                                /* PRQA S 4342 # JV-01 */
  /* Edge Detection Mode */
  if (ICU_MODE_SIGNAL_EDGE_DETECT == LenMeasurementMode)
  {
    /* E2x - IRQ */
    #if (ICU_IRQ_EDGE_DETECT == STD_ON)
    if (ICU_HW_EXT_IRQ == LpChannelConfig->ucIcuChannelType)
    {
      /* Update the PIN channel configuration pointer to point to the current channel */
      LpPINChannelUserRegConfig = Icu_GpChannelUserRegConfig[LucChannelIndex].pHWChannelUserRegConfig;                  /* PRQA S 0317 # JV-01 */
      /* Read the base configuration interrupt address */
      LpExtIntpcntRegs = LpChannelConfig->pCntlRegs;                                                                    /* PRQA S 0317 # JV-01 */
      /* Set EXINTCTL register to the reset value */
      RH850_SV_CLEAR_ICR_SYNCP(16, LpExtIntpcntRegs, (LpPINChannelUserRegConfig->ddChannelModeUserRegSettings));        /* PRQA S 2814 # JV-01 */
    } /* else No action required */
    #endif /* End of ICU_IRQ_EDGE_DETECT == STD_ON */
    /* INTP */
    #if (ICU_INTP_EDGE_DETECT == STD_ON)
    /* Read the base configuration interrupt address */
    LpFCLAnCTLAddress = LpChannelConfig->pCntlRegs;                                                                     /* PRQA S 0317 # JV-01 */
    /* Set FCLAnCTLm register to the reset value */
    *(LpFCLAnCTLAddress) = ICU_FCLA_CTL_RESET_VAL;                                                                      /* PRQA S 2814 # JV-01 */
    /* DummyRead & SYNCP */
    RH850_SV_MODE_REG_READ_ONLY(8, LpFCLAnCTLAddress);
    EXECUTE_SYNCP();
    #endif /* ENd of (ICU_INTP_EDGE_DETECT == STD_ON) */
  } /* else No action required */
  /* End of ICU_MODE_SIGNAL_EDGE_DETECT == LenMeasurementMode */
  #endif /* End of (ICU_EDGE_DETECT_API == STD_ON) */

} /* End of Internal function Icu_HW_Pin_ChannelDeInit */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* End of (ICU_DE_INIT_API == STD_ON) */
#endif /* End of ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT)) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Pin_SetActivation
**
** Service ID           : None
**
** Description          : This service configures the hardware to the active edges of the requested channel.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Reentrant
**
** Input Parameters     : LddChannel, LenActiveEdge
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : NA
**
** Global Variables     : Icu_GpChannelConfig, Icu_GpChannelRamData, Icu_GpChannelUserRegConfig.
**
** Functions invoked    : RH850_SV_CLEAR_ICR_SYNCP, RH850_SV_MODE_REG_READ_ONLY, EXECUTE_SYNCP.
**
** Registers Used       : FCLAnCTLm
**
** Reference ID         : ICU_DUD_ACT_060, ICU_DUD_ACT_060_CRT001
** Reference ID         : ICU_DUD_ACT_060_CRT002, ICU_DUD_ACT_060_REG001
** Reference ID         : ICU_DUD_ACT_060_REG003, ICU_DUD_ACT_060_REG004, ICU_DUD_ACT_060_GBL001
** Reference ID         : ICU_DUD_ACT_060_REG005, ICU_DUD_ACT_060_REG006, ICU_DUD_ACT_060_REG009
** Reference ID         : ICU_DUD_ACT_060_REG007, ICU_DUD_ACT_060_REG008, ICU_DUD_ACT_060_REG002
***********************************************************************************************************************/
#if ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT))
#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE)
                     Icu_HW_Pin_SetActivation(const Icu_ChannelType LddChannel, const Icu_ActivationType LenActiveEdge) /* PRQA S 1532 # JV-01 */
{
  #if (ICU_EDGE_DETECT_API == STD_ON)
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;

  /* INTP */
  #if (STD_ON == ICU_INTP_EDGE_DETECT)
  /* Defining a pointer to point to the External Interrupt registers */
  P2VAR(volatile uint8, AUTOMATIC, ICU_CONFIG_DATA) LpFCLAnCTLAddress;
  #endif /* End of U2Ax - INTP */

  #endif /* End of (ICU_EDGE_DETECT_API == STD_ON) */

  #if (ICU_EDGE_DETECT_API == STD_ON)
  /* To store Channel Measurement Mode */
  Icu_MeasurementModeType LenMeasurementMode;
  #endif

  /* E2x - IRQ */
  #if ((STD_ON == ICU_IRQ_EDGE_DETECT) && (ICU_EDGE_DETECT_API == STD_ON))
  /* Defining a pointer to the PIN timer channel configuration parameters */
  P2CONST(Icu_PINChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpPINChannelUserRegConfig;
  /* Defining a pointer to point to the External Interrupt registers */
  P2VAR(volatile uint16, AUTOMATIC, ICU_CONFIG_DATA) LpExtIntpcntRegs;

  #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON)
  /* Defining a pointer point to the External Interrupt Status clear Register */
  P2VAR(volatile uint8, AUTOMATIC, ICU_CONFIG_DATA) LpEXINTSTCRegs;
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_PINChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpPinChannelConfig;

  LpPinChannelConfig = Icu_GpChannelConfig[LddChannel].pHWIPChannelConfig;                                              /* PRQA S 0317 # JV-01 */
  #endif /* End of (ICU_CLEAR_PENDING_INTERRUPT == STD_ON) */
  #endif /* End of ((STD_ON == ICU_IRQ_EDGE_DETECT) && (ICU_EDGE_DETECT_API == STD_ON)) */

  #if (ICU_EDGE_DETECT_API == STD_ON)
  /* Update the channel configuration pointer to point to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LddChannel];

  /* Read measurement mode */
  LenMeasurementMode = (Icu_MeasurementModeType)(LpChannelConfig->ucIcuMeasurementMode);                                /* PRQA S 2814, 2844, 4342 # JV-01, JV-01, JV-01 */
  #endif

  #if (ICU_EDGE_DETECT_API == STD_ON)

  if (ICU_MODE_SIGNAL_EDGE_DETECT == LenMeasurementMode)
  {
    /* IRQ */
    #if (STD_ON == ICU_IRQ_EDGE_DETECT)
    /* Update the PIN Timer channel configuration pointer to point to the current channel */
    LpPINChannelUserRegConfig = Icu_GpChannelUserRegConfig[LddChannel].pHWChannelUserRegConfig;                         /* PRQA S 0317 # JV-01 */

    if (ICU_HW_EXT_IRQ == LpChannelConfig->ucIcuChannelType)
    {
      /* Read the address of EXINTCTL control register */
      LpExtIntpcntRegs = LpChannelConfig->pCntlRegs;                                                                    /* PRQA S 0317 # JV-01 */

      if (ICU_FALLING_EDGE == LenActiveEdge)
      {
        /* Set the edge detection bits in the EXINTCTL control register as per mask */
        *(LpExtIntpcntRegs) = (uint16)(*(LpExtIntpcntRegs) & (LpPINChannelUserRegConfig->ddChannelModeUserRegSettings)) /* PRQA S 2814 # JV-01 */
                    | ((uint16)((uint16)ICU_IRQ_FALLING_EDGE_MASK << (LpChannelConfig->usChannelMaskValue * ICU_TWO))); /* PRQA S 3383 # JV-01 */
      } /* End of (LenActiveEdge == ICU_FALLING_EDGE) */
      else if (ICU_RISING_EDGE == LenActiveEdge)                                                                        /* PRQA S 2004 # JV-01 */
      {
        /* Set the edge detection bits in the EXINTCTL control register as per mask */
        *(LpExtIntpcntRegs) = (uint16)(*(LpExtIntpcntRegs) & (LpPINChannelUserRegConfig->ddChannelModeUserRegSettings))
                     | ((uint16)((uint16)ICU_IRQ_RISING_EDGE_MASK << (LpChannelConfig->usChannelMaskValue * ICU_TWO))); /* PRQA S 3383 # JV-01 */
      } /* else No action required */
      /* End of (LenActiveEdge == ICU_RISING_EDGE) */
      #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON)
      /* Clear the IRQ status flag */
      LpEXINTSTCRegs = LpPinChannelConfig->pEXINSTCRegsAddress;                                                         /* PRQA S 0317, 2814 # JV-01, JV-01 */
      *(LpEXINTSTCRegs) = (uint8)(ICU_ONE << (uint8)LpChannelConfig->usChannelMaskValue);                               /* PRQA S 2814 # JV-01 */

      #endif /* End of (ICU_CLEAR_PENDING_INTERRUPT == STD_ON) */
    } /* else No action required */
    /* End of (ICU_HW_EXT_IRQ == LpChannelConfig->ucIcuChannelType) */

    #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON)
    /* Check if the Interrupt register is present for particular channel */
    if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)
    {
      /* Clear the pending interrupts */
      RH850_SV_CLEAR_ICR_SYNCP(16, LpChannelConfig->pIntrCntlAddress, (uint16)ICU_CLEAR_PENDING_INTR_MASK);             /* PRQA S 1281, 2814 # JV-01, JV-01 */
    } /* else No action required */
    /* End of LpChannelConfig->pIntrCntlAddress != NULL_PTR */
    #endif /* End of #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON) */
    #endif /* End of STD_ON == ICU_IRQ_EDGE_DETECT */

    /* INTP */
    #if (STD_ON == ICU_INTP_EDGE_DETECT)
    /* Updating the channel config parameter to the current channel */
    /* Read the address of filter control register */
    LpFCLAnCTLAddress = LpChannelConfig->pCntlRegs;                                                                     /* PRQA S 0317 # JV-01 */
    /* Mask the bypass bits of filter control register */
    *(LpFCLAnCTLAddress) = (*(LpFCLAnCTLAddress) & ICU_BYPASS_MASK);                                                    /* PRQA S 2814 # JV-01 */

    /* DummyRead & SYNCP */
    RH850_SV_MODE_REG_READ_ONLY(8, LpFCLAnCTLAddress);
    EXECUTE_SYNCP();

    if (ICU_BOTH_EDGES == LenActiveEdge)
    {
      /* Set the edge detection bits in the filter control register as per mask */
      *(LpFCLAnCTLAddress) = (*(LpFCLAnCTLAddress) | ICU_BOTH_EDGES_MASK);
    } /* End of ICU_BOTH_EDGES == LenActiveEdge */

    else if (ICU_FALLING_EDGE == LenActiveEdge)
    {
      /* Set the edge detection bits in the filter control register as per mask */
      *(LpFCLAnCTLAddress) = (*(LpFCLAnCTLAddress) | ICU_FALLING_EDGE_MASK);
    } /* End of ICU_FALLING_EDGE == LenActiveEdge */

    else /* (ICU_RISING_EDGE == LenActiveEdge) */
    {
      /* Set the edge detection bits in the filter control register as per mask */
      *(LpFCLAnCTLAddress) = (*(LpFCLAnCTLAddress) | ICU_RISING_EDGE_MASK);
    } /* End of ICU_RISING_EDGE == LenActiveEdge */

    /* DummyRead & SYNCP */
    RH850_SV_MODE_REG_READ_ONLY(8, LpFCLAnCTLAddress);
    EXECUTE_SYNCP();

    #endif /* End of STD_ON == ICU_INTP_EDGE_DETECT */

  } /* else No action required */ /* End of ICU_MODE_SIGNAL_EDGE_DETECT == LenMeasurementMode */
  #endif /* End of (ICU_EDGE_DETECT_API == STD_ON) */

  /* Initialize channel status as idle */
  Icu_GpChannelRamData[LddChannel].ucChannelStatus = (uint8)ICU_IDLE;

  #if (ICU_EDGE_DETECT_API == STD_ON)
  #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON)

  /* Check if the Interrupt register is present for particular channel */
  if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)
  {
    /* Clear the pending interrupts */
    RH850_SV_CLEAR_ICR_SYNCP(16, LpChannelConfig->pIntrCntlAddress, ICU_CLEAR_PENDING_INTR_MASK);                       /* PRQA S 1281, 2814 # JV-01, JV-01 */
  } /* else No action required */
  /* End of  NULL_PTR != LpChannelConfig->pIntrCntlAddress */
  #endif /* End of ICU_CLEAR_PENDING_INTERRUPT == STD_ON */
  #endif /* End of (ICU_EDGE_DETECT_API == STD_ON) */
} /* End of Internal function Icu_HW_Pin_SetActivation */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT)) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Pin_SetMode
**
** Service ID           : None
**
** Description          : This service sets the operating mode of the ICU Driver.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Non Reentrant
**
** Input Parameters     : LucChannelIndex, LenMode.
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelConfig, Icu_GpChannelRamData.
**
** Functions invoked    : ICU_ENTER_CRITICAL_SECTION, ICU_EXIT_CRITICAL_SECTION,
**                        RH850_SV_SET_ICR_SYNCP, RH850_SV_CLEAR_ICR_SYNCP.
**
** Registers Used       : EICn.
**
** Reference ID         : ICU_DUD_ACT_065, ICU_DUD_ACT_065_CRT001, ICU_DUD_ACT_065_CRT004
** Reference ID         : ICU_DUD_ACT_065_REG001, ICU_DUD_ACT_065_REG002, ICU_DUD_ACT_065_REG003
***********************************************************************************************************************/
#if ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT))
#if (ICU_SET_MODE_API == STD_ON)

#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Pin_SetMode(const Icu_ChannelType LucChannelIndex, const Icu_ModeType LenMode)      /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;
  /* Local variable to store wakeup status */
  uint8 LucWakeupStatus;

  /* Update the channel configuration pointer to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LucChannelIndex];
  /* Read the current channel wakeup status */
  LucWakeupStatus = (uint8)Icu_GpChannelRamData[LucChannelIndex].ucWakeupEnable;

  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Enter Critical Section */
  ICU_ENTER_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

  /* Check for the current wakeup status of the channel */
  if ((ICU_FALSE == LucWakeupStatus) && (NULL_PTR != LpChannelConfig->pIntrCntlAddress))                                /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    if (ICU_MODE_SLEEP == LenMode)
    {
      /* Disable Interrupt */
      RH850_SV_SET_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress),  ICU_DISABLE_EIMK_MASK);                           /* PRQA S 0751, 2814 # JV-01, JV-01 */
    } /* End of LenMode = ICU_MODE_SLEEP */

    else /* LenMode = ICU_MODE_NORMAL */
    {
      #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON)
      /* Check if the Interrupt register is present for particular channel */
      /* Clear the pending interrupts */
      RH850_SV_CLEAR_ICR_SYNCP(16, LpChannelConfig->pIntrCntlAddress, ICU_CLEAR_PENDING_INTR_MASK);                     /* PRQA S 1281 # JV-01 */
      #endif

      /* Enable Interrupt for on-going channels */
      if ((uint8)ICU_TRUE == Icu_GpChannelRamData[LucChannelIndex].ucChannelRunning)
      {
        RH850_SV_CLEAR_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_ENABLE_EIMK_MASK);                         /* PRQA S 0751 # JV-01 */
      } /* else No action required */
    } /* End of LenMode = ICU_MODE_NORMAL */
  } /* End of ICU_FALSE == LucWakeupStatus && NULL_PTR != LpChannelConfig->pIntrCntlAddress */
  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Exit Critical Section */
  ICU_EXIT_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

} /* End of Internal function Icu_HW_Pin_SetMode */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* End of (ICU_SET_MODE_API == STD_ON) */
#endif /* End of ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT)) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Pin_EnableEdgeDetection
**
** Service ID           : None
**
** Description          : This service routine starts the count measurement for starting edge counting or
**                        signal measurement or timestamp measurement.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Reentrant
**
** Input Parameters     : LddChannel
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelConfig, Icu_GpChannelRamData, Icu_GenModuleMode.
**
** Functions invoked    : ICU_ENTER_CRITICAL_SECTION, ICU_EXIT_CRITICAL_SECTION, RH850_SV_CLEAR_ICR_SYNCP.
**
** Registers Used       : EXINTSTC
**
** Reference ID         : ICU_DUD_ACT_063, ICU_DUD_ACT_063_CRT001,
** Reference ID         : ICU_DUD_ACT_063_CRT002, ICU_DUD_ACT_063_GBL001, ICU_DUD_ACT_063_REG001
** Reference ID         : ICU_DUD_ACT_063_REG002, ICU_DUD_ACT_063_REG003, ICU_DUD_ACT_063_REG004
***********************************************************************************************************************/
#if ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT))
#if (ICU_EDGE_DETECT_API == STD_ON)

#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Pin_EnableEdgeDetection(const Icu_ChannelType LddChannel)                           /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;
  /* IRQ */
  #if (ICU_IRQ_EDGE_DETECT == STD_ON)
  #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON)
  /* Defining a pointer to point to the External */
  /* Interrupt status clear registers */
  P2VAR(volatile uint8, AUTOMATIC, ICU_CONFIG_DATA) LpEXINTSTCRegs;
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_PINChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpPinChannelConfig;
  #endif /* End of #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON) */
  #endif /* End of #if (ICU_IRQ_EDGE_DETECT == STD_ON) */

  /* Update the channel configuration pointer to point to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LddChannel];

  /* IRQ */
  #if (ICU_IRQ_EDGE_DETECT == STD_ON)
  if (ICU_HW_EXT_IRQ == LpChannelConfig->ucIcuChannelType)                                                              /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON)
    LpPinChannelConfig = Icu_GpChannelConfig[LddChannel].pHWIPChannelConfig;                                            /* PRQA S 0317 # JV-01 */
    #endif

    #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
    /* Enter Critical Section */
    ICU_ENTER_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
    #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

    #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON)
    /* Clear the IRQ status flag */
    LpEXINTSTCRegs = LpPinChannelConfig->pEXINSTCRegsAddress;                                                           /* PRQA S 0317, 2814 # JV-01, JV-01 */
    *(LpEXINTSTCRegs) = (uint8)(ICU_ONE << (uint8)LpChannelConfig->usChannelMaskValue);                                 /* PRQA S 2814 # JV-01 */
    #endif /* End of #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON) */
    /* Check if the Interrupt register is present for particular channel */
    if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)
    {
      /* Enable interrupts */
      RH850_SV_CLEAR_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_ENABLE_EIMK_MASK);                           /* PRQA S 0751, 2814 # JV-01, JV-01 */
    } /* else No action required */

    #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
    /* Exit Critical Section */
    ICU_EXIT_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
    #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */
  } /* else No action required */
  #endif /* End of ICU_HW_EXT_IRQ == STD_ON */
  /* INTP */
  #if (ICU_INTP_EDGE_DETECT == STD_ON)
  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Enter Critical Section */
  ICU_ENTER_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

  /* Set the bit status running of channel is ICU_TRUE*/
  Icu_GpChannelRamData[LddChannel].ucChannelRunning = (uint8)ICU_TRUE;
  /* Check if the Interrupt register is present for particular channel */
  if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)                                                                    /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON)
    /* Clear the pending interrupts */
    RH850_SV_CLEAR_ICR_SYNCP(16, LpChannelConfig->pIntrCntlAddress, ICU_CLEAR_PENDING_INTR_MASK);                       /* PRQA S 1281, 2814 # JV-01, JV-01 */
    #endif /* End of ICU_CLEAR_PENDING_INTERRUPT == STD_ON */

    /* Enabling the Interrupt processing */
    #if ((ICU_ENABLE_WAKEUP_API == STD_ON) || (ICU_NOTIFY_WAKEUP_INTERRUPT == STD_ON))
    if ((ICU_MODE_SLEEP == Icu_GenModuleMode) && ((uint8)ICU_FALSE == Icu_GpChannelRamData[LddChannel].ucWakeupEnable))
    {
      /* No action required */
    }
    else
    #endif
    {
      /* Enable the Interrupt processing */
      RH850_SV_CLEAR_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_ENABLE_EIMK_MASK);                           /* PRQA S 0751, 2814 # JV-01, JV-01 */
    }
  } /* else No action required */
  /* End of  NULL_PTR != LpChannelConfig->pIntrCntlAddress */

  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Exit Critical Section */
  ICU_EXIT_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

  #endif /* End of ICU_INTP_EDGE_DETECT == STD_ON */
} /* End of Internal function Icu_HW_Pin_EnableEdgeDetection */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* End of (ICU_EDGE_DETECT_API == STD_ON) */
#endif /* End of ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT)) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Pin_DisableEdgeDetection
**
** Service ID           : None
**
** Description          : This service routine stops the count measurement for starting edge counting or signal 
**                        measurement or timestamp measurement.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Reentrant
**
** Input Parameters     : LddChannel
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelConfig, Icu_GpChannelRamData.
**
** Functions invoked    : ICU_ENTER_CRITICAL_SECTION, ICU_EXIT_CRITICAL_SECTION, RH850_SV_SET_ICR_SYNCP.
**
** Registers Used       : EICn.
**
** Reference ID         : ICU_DUD_ACT_064, ICU_DUD_ACT_064_CRT001, ICU_DUD_ACT_064_CRT004
** Reference ID         : ICU_DUD_ACT_064_GBL001, ICU_DUD_ACT_064_REG001, ICU_DUD_ACT_064_REG002
***********************************************************************************************************************/
#if ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT))
#if (ICU_EDGE_DETECT_API == STD_ON)

#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Pin_DisableEdgeDetection(const Icu_ChannelType LddChannel)                          /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;

  /* Update the channel configuration pointer to point to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LddChannel];

  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Enter Critical Section */
  ICU_ENTER_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

  /* INTP */
  #if (ICU_INTP_EDGE_DETECT == STD_ON)
  /* Set bit flag channel running is ICU_FALSE */
  Icu_GpChannelRamData[LddChannel].ucChannelRunning = (uint8)ICU_FALSE;
  #endif

  #if (ICU_IRQ_EDGE_DETECT == STD_ON)
  if (ICU_HW_EXT_IRQ == LpChannelConfig->ucIcuChannelType)                                                              /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    /* Check if the Interrupt register is present for particular channel */
    if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)
    {
      /* Disable interrupts */
      RH850_SV_SET_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_DISABLE_EIMK_MASK);                            /* PRQA S 0751, 2814 # JV-01, JV-01 */
    } /* else No action required */
  } /* else No action required */
  #endif /* End of ICU_IRQ_EDGE_DETECT == STD_ON */
  /* INTP */
  #if (ICU_INTP_EDGE_DETECT == STD_ON)
  if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)                                                                    /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    /* Disable the Interrupt processing */
    RH850_SV_SET_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_DISABLE_EIMK_MASK);                              /* PRQA S 0751, 2814 # JV-01, JV-01 */
  } /* else No action required */
  #endif /* End of ICU_INTP_EDGE_DETECT == STD_ON */
  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Exit Critical Section */
  ICU_EXIT_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

} /* End of Internal function Icu_HW_Pin_DisableEdgeDetection */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* End of (ICU_EDGE_DETECT_API == STD_ON) */
#endif /* End of ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT)) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Pin_ExternalInterruptIsr
**
** Service ID           : None
**
** Description          : This service routine is invoked from all the external
**                        interrupts which takes care of calling the notification functions.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Non-Reentrant
**
** Input Parameters     : LddChannel
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelConfig, Icu_GenModuleMode, Icu_GpChannelRamData.
**
** Functions invoked    : ICU_ENTER_CRITICAL_SECTION, ICU_EXIT_CRITICAL_SECTION, EcuM_CheckWakeup,
**                        pIcuNotificationPointer, ICU_DEM_REPORT_ERROR.
**
** Registers Used       : EXINTSTC
**
** Reference ID         : ICU_DUD_ACT_057, ICU_DUD_ACT_057_GBL001
** Reference ID         : ICU_DUD_ACT_057_GBL002, ICU_DUD_ACT_057_REG001, ICU_DUD_ACT_057_ERR001
** Reference ID         : ICU_DUD_ACT_057_CRT001, ICU_DUD_ACT_057_CRT002
***********************************************************************************************************************/
#if ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT))
#if (ICU_EDGE_DETECT_API == STD_ON)

#define ICU_START_SEC_CODE_FAST
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_CODE_FAST) Icu_HW_Pin_ExternalInterruptIsr(const Icu_ChannelType LddChannel)                             /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to point to the channel configuration parameters */
  #if ((ICU_REPORT_WAKEUP_SOURCE == STD_ON) || (ICU_NOTIFICATION_CONFIG == STD_ON) || \
                                         (ICU_INTERRUPT_CONSISTENCY_CHECK == STD_ON) || (ICU_IRQ_EDGE_DETECT == STD_ON))

  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;
  #endif /* End of (ICU_REPORT_WAKEUP_SOURCE == STD_ON) || (ICU_NOTIFICATION_CONFIG == STD_ON) */

  /* IRQ */
  #if (ICU_IRQ_EDGE_DETECT == STD_ON)
  /* Defining a pointer to point to the External Interrupt */
  /* Status clear registers */
  P2VAR(volatile uint8, AUTOMATIC, ICU_CONFIG_DATA) LpEXINTSTCRegs;
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_PINChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpPinChannelConfig;

  LpPinChannelConfig = Icu_GpChannelConfig[LddChannel].pHWIPChannelConfig;                                              /* PRQA S 0317 # JV-01 */
  #endif

  #if ((ICU_REPORT_WAKEUP_SOURCE == STD_ON) || (ICU_NOTIFICATION_CONFIG == STD_ON) || \
                                         (ICU_INTERRUPT_CONSISTENCY_CHECK == STD_ON) || (ICU_IRQ_EDGE_DETECT == STD_ON))
  /* Update the channel pointer to point to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LddChannel];
  #endif /* End of (ICU_REPORT_WAKEUP_SOURCE == STD_ON) || (ICU_NOTIFICATION_CONFIG == STD_ON) */        

  #if (ICU_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if ((uint16)ICU_ZERO != (uint16)(*LpChannelConfig->pIntrCntlAddress & ICU_EIC_EIMK_MASK))                             /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    #if (ICU_E_INT_INCONSISTENT_CONFIGURED == STD_ON)
    ICU_DEM_REPORT_ERROR(ICU_E_INT_INCONSISTENT, DEM_EVENT_STATUS_FAILED);
    #endif
  }
  else
  #endif
  {
    /* IRQ */
    #if (ICU_IRQ_EDGE_DETECT == STD_ON)

    #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
    /* Enter Critical Section */
    ICU_ENTER_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
    #endif /*(ICU_CRITICAL_SECTION_PROTECTION == STD_ON)*/

    /* Clear the IRQ status flag */
    LpEXINTSTCRegs = LpPinChannelConfig->pEXINSTCRegsAddress;                                                           /* PRQA S 0317, 2814 # JV-01, JV-01 */

    *(LpEXINTSTCRegs) = (uint8)(ICU_ONE << (uint8)LpChannelConfig->usChannelMaskValue);                                 /* PRQA S 2814, 2844 # JV-01, JV-01 */

    #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
    /* Exit Protected area */
    ICU_EXIT_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
    #endif
    #endif /* End of ICU_IRQ_EDGE_DETECT == STD_ON */
    /* INTP */
    #if (ICU_INTP_EDGE_DETECT == STD_ON)
    #if (((ICU_ENABLE_WAKEUP_API == STD_ON) || (ICU_NOTIFY_WAKEUP_INTERRUPT == STD_ON)) &&\
                                                                                   (ICU_REPORT_WAKEUP_SOURCE == STD_ON))

    /* If Module was in SLEEP mode and reporting wakeup is enabled */
    if ((ICU_MODE_SLEEP == Icu_GenModuleMode) && (ICU_TRUE == (boolean)Icu_GpChannelRamData[LddChannel].ucWakeupEnable))
    {
      if (((uint8)ICU_FALSE == LpChannelConfig->ucIcuDisableEcumWakeupNotification) &&
                                                             (ICU_NOWAKEUP != LpChannelConfig->ucEcuMChannelWakeupInfo))
      {
        /* Set the wakeup status to true */
        Icu_GpChannelRamData[LddChannel].ucWakeupOccurrence = (uint8)ICU_TRUE;
        /* Report Wakeup Event to EcuM */
        EcuM_CheckWakeup((EcuM_WakeupSourceType)(ICU_DOUBLE_ONE << (LpChannelConfig->ucEcuMChannelWakeupInfo)));
      } /* else No action required */
    } /* else No action required */
    #endif /* End of (((ICU_ENABLE_WAKEUP_API == STD_ON) || (ICU_NOTIFY_WAKEUP_INTERRUPT == STD_ON))&&
                                                                                (ICU_REPORT_WAKEUP_SOURCE == STD_ON) */
    #endif /* End of ICU_INTP_EDGE_DETECT == STD_ON */

    /* Set channel input status as active */
    Icu_GpChannelRamData[LddChannel].ucChannelStatus = (uint8)ICU_ACTIVE;

    #if (ICU_NOTIFICATION_CONFIG == STD_ON)
    /* Check if notification is enabled */
    if ((uint8)ICU_TRUE == (Icu_GpChannelRamData[LddChannel].ucNotificationEnable))
    {
      /* Check if the function pointer address is present for particular channel */
      if (NULL_PTR != LpChannelConfig->pIcuNotificationPointer)
      {
        /* callback notification */
        LpChannelConfig->pIcuNotificationPointer();
      } /* else No action required */
    } /* else No action required */
    #endif /* End of ICU_NOTIFICATION_CONFIG */
  }
} /* End of Internal function Icu_HW_Pin_ExternalInterruptIsr */

#define ICU_STOP_SEC_CODE_FAST
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* #if (ICU_EDGE_DETECT_API == STD_ON) */
#endif /* End of ((STD_ON == ICU_IRQ_EDGE_DETECT) || (STD_ON == ICU_INTP_EDGE_DETECT)) */


/***********************************************************************************************************************
**                                                    End of File                                                     **
***********************************************************************************************************************/
