/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Adc_ADCH_LLDriver.c                                                                                 */
/* SW-VERSION   = 1.5.0                                                                                               */
/*====================================================================================================================*/
/*                                                     COPYRIGHT                                                      */
/*====================================================================================================================*/
/* (c) 2020-2022 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* Internal functions implementation of ADC Driver Component                                                          */
/*                                                                                                                    */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s) and/or */
/* the Application.                                                                                                   */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs of  */
/* program errors, compliance with applicable laws, damage to or loss of data, programs or equipment, and             */
/* unavailability or interruption of operations.                                                                      */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*              Devices:        X2x                                                                                   */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                              Revision Control History                                              **
***********************************************************************************************************************/
/*
 * 1.4.5: 14/09/2022 : Updated QAC messages according to QAC analysis results.
 *        12/09/2022 : Replaced the empty else blocks, not from else if, with comments.
 * 1.4.4: 24/08/2022 : Updated information in function header comments.
 *        21/07/2022 : Updated Re-entrancy, Global Variables and Registers Used in function header comments.
 * 1.4.3: 25/05/2022 : Removed unnecessary coding rule messages in header comment.
 *        18/05/2022 : Fixed the typos in comments.
 * 1.4.1: 24/11/2021 : Removed the inclusion of unnecessary header.
 * 1.3.2: 09/09/2021 : Added synchronization after handling each bus group.
 *        06/09/2021 : Changed to Adc_Mapping.h from Adc_MemMap.h
 *        26/08/2021 : Changed the macro used for interrupt mask to ADC_EIC_EIMK_MASK.
 * 1.3.1: 08/07/2021 : Changed the name of the global variable Adc_GpWaitTimeConfig to conform to Camel.
 *        07/07/2021 : Added QAC message 9.5.0
 *                     Changed comment format to 120 columns
 *                     Changed violation tag from START/END
 *        16/06/2021 : Updated by nesting ADC register structures.
 *        01/06/2021 : Changed to the appropriate memory classes and pointer classes.
 *        24/05/2021 : Removed unnecessary include headers.
 *        07/04/2021 : Changed to comply with coding rules.
 * 1.2.0: 14/07/2020 : Release
 *        27/07/2020 : Add QAC 9.3.1 comment.
 * 1.1.0: 19/06/2020 : Release
 * 1.0.1: 09/04/2020 : Add const qualifier for arguments of internal functions
 *                     that are never changed
 * 1.0.0: 09/01/2020 : Initial Version
 */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                  Include Section                                                   **
***********************************************************************************************************************/
/* Included for version check macro definitions and type definitions */
#include "Adc.h"
/* Included for RAM variable declarations */
#include "Adc_Ram.h"
/* Included for the declaration of internal function prototypes */
#include "Adc_LLDriver.h"
#include "Adc_ADCH_LLDriver.h"
/* Included for the macro declaration of supervisor mode(SV) write enabled Registers IMR & ICxxx */
#include "rh850_Types.h"

/***********************************************************************************************************************
**                                                Version Information                                                 **
***********************************************************************************************************************/
/* AUTOSAR Release version information */
#define ADC_ADCH_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION    ADC_AR_RELEASE_MAJOR_VERSION_VALUE
#define ADC_ADCH_LLDRIVER_C_AR_RELEASE_MINOR_VERSION    ADC_AR_RELEASE_MINOR_VERSION_VALUE                              /* PRQA S 0791 # JV-01 */
#define ADC_ADCH_LLDRIVER_C_AR_RELEASE_REVISION_VERSION ADC_AR_RELEASE_REVISION_VERSION_VALUE                           /* PRQA S 0791 # JV-01 */

/* File version information */
#define ADC_ADCH_LLDRIVER_C_SW_MAJOR_VERSION            ADC_SW_MAJOR_VERSION_VALUE
#define ADC_ADCH_LLDRIVER_C_SW_MINOR_VERSION            ADC_SW_MINOR_VERSION_VALUE

/***********************************************************************************************************************
**                                                   Version Check                                                    **
***********************************************************************************************************************/
#if (ADC_ADCH_LLDRIVER_AR_RELEASE_MAJOR_VERSION != ADC_ADCH_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION)
  #error "Adc_ADCH_LLDriver.c : Mismatch in Release Major Version"
#endif

#if (ADC_ADCH_LLDRIVER_AR_RELEASE_MINOR_VERSION != ADC_ADCH_LLDRIVER_C_AR_RELEASE_MINOR_VERSION)
  #error "Adc_ADCH_LLDriver.c : Mismatch in Release Minor Version"
#endif

#if (ADC_ADCH_LLDRIVER_AR_RELEASE_REVISION_VERSION != ADC_ADCH_LLDRIVER_C_AR_RELEASE_REVISION_VERSION)
  #error "Adc_ADCH_LLDriver.c : Mismatch in Release Patch Version"
#endif

#if (ADC_ADCH_LLDRIVER_SW_MAJOR_VERSION != ADC_ADCH_LLDRIVER_C_SW_MAJOR_VERSION)
  #error "Adc_ADCH_LLDriver.c : Mismatch in Software Major Version"
#endif

#if (ADC_ADCH_LLDRIVER_SW_MINOR_VERSION != ADC_ADCH_LLDRIVER_C_SW_MINOR_VERSION)
  #error "Adc_ADCH_LLDriver.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (7:0791)    : [U] Macro identifier does not differ from other macro identifier(s) (e.g. '%s') within       */
/*                       the specified number of significant characters.                                              */
/* Rule                : CERTCCM DCL23, MISRA C:2012 Rule-5.4                                                         */
/* JV-01 Justification : This macro identifier is following AUTOSAR standard rule (Symbolic Name or Published         */
/*                       Macro's name), so this is accepted.                                                          */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1532)    : The function '%1s' is only referenced in one translation unit - but not the one in which     */
/*                       it is defined.                                                                               */
/* Rule                : CERTCCM DCL19, MISRA C:2012 Rule-8.7                                                         */
/* JV-01 Justification : This macro identifier is following AUTOSAR standard rule (Symbolic Name or Published         */
/*                       Macro's name), so this is accepted                                                           */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0303)    : [I] Cast between a pointer to volatile object and an integral type.                          */
/* Rule                : CERTCCM INT36, MISRA C:2012 Rule-11.4                                                        */
/*                       REFERENCE - ISO-6.3.4 Semantics                                                              */
/* JV-01 Justification : Typecasting is done as per the register size, to access hardware registers.                  */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : Use of #include directive after code fragment.                                               */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is done as per Memory Requirement, (MEMMAP003 - Specification of Memory Mapping).       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : This is accepted, due to the implementation is following hardware specification.             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2824)    : Possible: Arithmetic operation on NULL pointer.                                              */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : This is accepted, due to the implementation is following hardware specification.             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                    Global Data                                                     **
***********************************************************************************************************************/

/***********************************************************************************************************************
**                                                Function Definitions                                                **
***********************************************************************************************************************/

/***********************************************************************************************************************
** Function Name        : Adc_EcmErrorInterrupt
**
** Service ID           : NA
**
** Description          : This internal function enables or disables the ECM parity error interrupt.
**
** Sync/Async           : Synchronous
**
** Re-entrancy          : Non-Reentrant
**
** Input Parameters     : LblEnableInterrupt
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return Parameter     : None
**
** Preconditions        : None
**
** Global Variables     : None
**
** Functions invoked    : None
**
** Registers Used       : ECMKCPROT, ECMMICFGn, EIC
**
** Reference ID         : ADC_DUD_ACT_055, ADC_DUD_ACT_055_REG001, ADC_DUD_ACT_055_REG002,
** Reference ID         : ADC_DUD_ACT_055_REG003, ADC_DUD_ACT_055_REG004, ADC_DUD_ACT_055_REG005
***********************************************************************************************************************/
#if (ADC_ENABLE_PARITY_ERR_INT == STD_ON)

#define ADC_START_SEC_PRIVATE_CODE
#include "Adc_Mapping.h"

FUNC(void, ADC_PRIVATE_CODE) Adc_EcmErrorInterrupt(const boolean LblEnableInterrupt)                                    /* PRQA S 1532 # JV-01 */
{
  uint32 LulRegValue;

  if (ADC_FALSE == LblEnableInterrupt)
  {
    /* Set ADC parity error to enable in ECMMICFG7 register (ECM240) */
    LulRegValue = ADC_ECMMICFG_REG & ~ADC_ECM_CFG_ADC_PARITY_ERR;                                                       /* PRQA S 0303 # JV-01 */

    /* Disable ECM interrupt and perform synchronization for INTC1 */
    RH850_SV_SET_ICR_SYNCP(8, ADC_ECM_EIC8, ADC_EIC_EIMK);                                                              /* PRQA S 0303 # JV-01 */
  }
  else
  {
    /* Set ADC parity error to disable in ECMMICFG7 register (ECM240) */
    LulRegValue = ADC_ECMMICFG_REG | ADC_ECM_CFG_ADC_PARITY_ERR;                                                        /* PRQA S 0303 # JV-01 */
  }

  /* Enable write access of ECMMICFG register */
  ADC_ECMKCPROT_REG = ADC_ECM_KEYCODE_ENABLE;                                                                           /* PRQA S 0303 # JV-01 */

  /* Set ADC parity error bit of ECMMICFG register */
  ADC_ECMMICFG_REG = LulRegValue;                                                                                       /* PRQA S 0303 # JV-01 */

  /* Perform synchronization for ECM */
  (void)ADC_ECMMICFG_REG;                                                                                               /* PRQA S 0303 # JV-01 */
  EXECUTE_SYNCP();

  /* Disable write access of ECMMICFG register */
  ADC_ECMKCPROT_REG = ADC_ECM_KEYCODE_DISABLE;                                                                          /* PRQA S 0303 # JV-01 */

  if (ADC_TRUE == LblEnableInterrupt)
  {
    /* Enable ECM interrupt and perform synchronization for INTC1 */
    RH850_SV_CLEAR_ICR_SYNCP(8, ADC_ECM_EIC8, (uint8)~ADC_EIC_EIMK);                                                    /* PRQA S 0303 # JV-01 */
  } /* else: No action required */
}

#define ADC_STOP_SEC_PRIVATE_CODE
#include "Adc_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* (ADC_ENABLE_PARITY_ERR_INT == STD_ON) */

/***********************************************************************************************************************
** Function Name        : Adc_SetWaitTime
**
** Service ID           : NA
**
** Description          : This internal function set Software wait time to registers.
**
** Sync/Async           : Synchronous
**
** Re-entrancy          : Reentrant
**
** Input Parameters     : LucHwUnitIndex
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return Parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Adc_GpHwUnitConfig, Adc_GpWaitTimeConfig
**
** Functions invoked    : None
**
** Registers Used       : ADCXnSGVCOWRx
**
** Reference ID         : ADC_DUD_ACT_056, ADC_DUD_ACT_056_REG001
***********************************************************************************************************************/
#define ADC_START_SEC_PRIVATE_CODE
#include "Adc_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ADC_PRIVATE_CODE) Adc_SetWaitTime(const uint8 LucHwUnitIndex)                                                /* PRQA S 1532 # JV-01 */
{
  P2CONST(Adc_HwUnitConfigType, AUTOMATIC, ADC_CONFIG_DATA) LpHwUnitConfig;
  P2CONST(Adc_WaitTimeConfigType, AUTOMATIC, ADC_CONFIG_DATA) LpWaitTimeConfig;
  P2VAR(volatile Adc_ConfigRegisters, AUTOMATIC, REGSPACE) LpAdcRegisters;
  /* Local pointer to Scan Group registers */
  uint8 LucSgLoopCount;

  /* Initialize the local variable for HW unit configuration */
  LpHwUnitConfig = &Adc_GpHwUnitConfig[LucHwUnitIndex];                                                                 /* PRQA S 2824 # JV-01 */
  /* Read the user base configuration address of the HW unit */
  LpAdcRegisters = LpHwUnitConfig->pHwUnitBaseAddress;                                                                  /* PRQA S 2814 # JV-01 */
  /* Read the wait time registers of the hardware unit */
  LpWaitTimeConfig = &Adc_GpWaitTimeConfig[LpHwUnitConfig->ucWaitTimeIndex];                                            /* PRQA S 2824 # JV-01 */
  /* Set the wait time registers */
  for (LucSgLoopCount = (uint8)0U; LucSgLoopCount < ADC_NUM_OF_WAITTIME; LucSgLoopCount++)
  {
    LpAdcRegisters->aaADCXnSG[LucSgLoopCount].usADCXnSGVCOWRx =                                                         /* PRQA S 2814 # JV-01 */
      LpWaitTimeConfig->usSGVCOWRx[LucSgLoopCount] & ADC_SGVCOWR_VCOW_BIT;                                              /* PRQA S 2814 # JV-01 */
  }
}
#define ADC_STOP_SEC_PRIVATE_CODE
#include "Adc_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name        : Adc_DeInitWaitTime
**
** Service ID           : NA
**
** Description          : This internal function set Software wait time to registers.
**
** Sync/Async           : Synchronous
**
** Re-entrancy          : Reentrant
**
** Input Parameters     : LucHwUnitIndex
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return Parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Adc_GpHwUnitConfig
**
** Functions invoked    : None
**
** Registers Used       : ADCXnSGVCOWRx
**
** Reference ID         : ADC_DUD_ACT_057, ADC_DUD_ACT_057_REG001
***********************************************************************************************************************/
#if (ADC_DEINIT_API == STD_ON)

#define ADC_START_SEC_PRIVATE_CODE
#include "Adc_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ADC_PRIVATE_CODE) Adc_DeInitWaitTime(const uint8 LucHwUnitIndex)                                             /* PRQA S 1532 # JV-01 */
{
  P2CONST(Adc_HwUnitConfigType, AUTOMATIC, ADC_CONFIG_DATA) LpHwUnitConfig;
  P2VAR(volatile Adc_ConfigRegisters, AUTOMATIC, REGSPACE) LpAdcRegisters;
  /* Local pointer to Scan Group registers */
  uint8 LucSgLoopCount;

  /* Initialize the local variable for HW unit configuration */
  LpHwUnitConfig = &Adc_GpHwUnitConfig[LucHwUnitIndex];                                                                 /* PRQA S 2824 # JV-01 */
  /* Read the user base configuration address of the HW unit */
  LpAdcRegisters = LpHwUnitConfig->pHwUnitBaseAddress;                                                                  /* PRQA S 2814 # JV-01 */
  /* Get the pointer to Scan Group registers */
  /* Set the wait time registers */
  for (LucSgLoopCount = (uint8)0U; LucSgLoopCount < ADC_NUM_OF_WAITTIME; LucSgLoopCount++)
  {
    /* Set to wait time for A/D convert time */
    LpAdcRegisters->aaADCXnSG[LucSgLoopCount].usADCXnSGVCOWRx = (uint16)0x0U;                                           /* PRQA S 2814 # JV-01 */
  }
}

#define ADC_STOP_SEC_PRIVATE_CODE
#include "Adc_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* (ADC_DEINIT_API == STD_ON) */

/***********************************************************************************************************************
**                                                    End of File                                                     **
***********************************************************************************************************************/
