/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Icu_TAU_LLDriver.c                                                                                  */
/* SW-VERSION   = 1.5.0                                                                                               */
/*====================================================================================================================*/
/*                                                     COPYRIGHT                                                      */
/*====================================================================================================================*/
/* (c) 2020-2022 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* Define common Internal Function for TAU Timer Unit.                                                                */
/*                                                                                                                    */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s) and/or */
/* the Application.                                                                                                   */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs of  */
/* program errors, compliance with applicable laws, damage to or loss of data, programs or equipment, and             */
/* unavailability or interruption of operations.                                                                      */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*              Devices:        X2x                                                                                   */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                              Revision Control History                                              **
***********************************************************************************************************************/
/*
 * 1.4.3:  09/05/2022  : Remove "else" statement when no action required.
 *         17/04/2022  : Merge QAC message from U2Ax, E2x and U2Bx
 *         08/04/2022  : Add version information and version check for AUTOSAR release version, Module Software version.
 * 1.4.2:  02/03/2022  : Fix QAC header and message
 * 1.4.1:  20/11/2021  : Add QAC message 9.5.0
 *         16/09/2021  : Correct names, datatypes of used variables
 *                       Correct preprocessor directives
 * 1.3.2:  30/08/2021  : Change from #include "Icu_MemMap.h" to #include "Icu_Mapping.h".
 *         24/08/2021  : Add pre-compile parameters (ICU_EDGE_DETECT_API == STD_ON) for Icu_HW_Tau_StartCountMeasurement
 *                       and Icu_HW_Tau_StopCountMeasurement
 * 1.3.1:  02/07/2021  : Add QAC message 9.5.0
 *                       Format source code to 120 characters
 *                       Improve Violation tag (remove START/END)
 * 1.2.0:  17/07/2020  : Release
 * 1.1.0:  19/06/2020  : Release
 * 1.0.0:  25/03/2020  : Initial Version
 */
/**********************************************************************************************************************/
/***********************************************************************************************************************
**                                             Margin for Control History                                             **
***********************************************************************************************************************/

/***********************************************************************************************************************
**                                                  Include Section                                                   **
***********************************************************************************************************************/
/* Included for macro definitions */
#include "Icu_TAU_LLDriver.h"                                                                                           /* PRQA S 0857 # JV-01 */
/* Included for RAM variable declarations */
#include "Icu_Ram.h"

/***********************************************************************************************************************
**                                                Version Information                                                 **
***********************************************************************************************************************/
/* AUTOSAR release version information */
#define ICU_TAU_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION    ICU_AR_RELEASE_MAJOR_VERSION_VALUE
#define ICU_TAU_LLDRIVER_C_AR_RELEASE_MINOR_VERSION    ICU_AR_RELEASE_MINOR_VERSION_VALUE                               /* PRQA S 0791 # JV-01 */
#define ICU_TAU_LLDRIVER_C_AR_RELEASE_REVISION_VERSION ICU_AR_RELEASE_REVISION_VERSION_VALUE

/* File version information */
#define ICU_TAU_LLDRIVER_C_SW_MAJOR_VERSION            ICU_SW_MAJOR_VERSION_VALUE
#define ICU_TAU_LLDRIVER_C_SW_MINOR_VERSION            ICU_SW_MINOR_VERSION_VALUE

/***********************************************************************************************************************
**                                                   Version Check                                                    **
***********************************************************************************************************************/
#if (ICU_TAU_LLDRIVER_AR_RELEASE_MAJOR_VERSION != ICU_TAU_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION)
#error "Icu_TAU_LLDriver.c : Mismatch in Release Major Version"
#endif

#if (ICU_TAU_LLDRIVER_AR_RELEASE_MINOR_VERSION != ICU_TAU_LLDRIVER_C_AR_RELEASE_MINOR_VERSION)
#error "Icu_TAU_LLDriver.c : Mismatch in Release Minor Version"
#endif

#if (ICU_TAU_LLDRIVER_AR_RELEASE_REVISION_VERSION != ICU_TAU_LLDRIVER_C_AR_RELEASE_REVISION_VERSION)
#error "Icu_TAU_LLDriver.c : Mismatch in Release Revision Version"
#endif

#if (ICU_TAU_LLDRIVER_SW_MAJOR_VERSION != ICU_TAU_LLDRIVER_C_SW_MAJOR_VERSION)
#error "Icu_TAU_LLDriver.c : Mismatch in Software Major Version"
#endif

#if (ICU_TAU_LLDRIVER_SW_MINOR_VERSION != ICU_TAU_LLDRIVER_C_SW_MINOR_VERSION)
#error "Icu_TAU_LLDriver.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (2:0751)    : Casting to char pointer type.                                                                */
/* Rule                : CERTCCM EXP11, EXP39                                                                         */
/* JV-01 Justification : Using void due to specific requirement of input parameter. So, this can be skipped           */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (7:0791)    : Macro identifier does not differ from other macro identifier(s) (e.g. '%s') within the       */
/*                       specified number of significant characters.                                                  */
/* Rule                : CERTCCM DCL23, MISRA C:2012 Rule-5.4                                                         */
/* JV-01 Justification : This macro identifier is following AUTOSAR standard rule (Symbolic Name or Published         */
/*                       Macro's name), so this is accepted.                                                          */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0857)    : Number of macro definitions exceeds 1024 - program does not conform strictly to ISO:C90.     */
/* Rule                : MISRA C:2012 Dir-1.1                                                                         */
/* JV-01 Justification : The number of macro depend on module code size. There is no issue when number of macro is    */
/*                       over 1024                                                                                    */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1281)    : Integer literal constant is of an unsigned type but does not include a "U" suffix.           */
/* Rule                : MISRA C:2012 Rule-7.2                                                                        */
/* JV-01 Justification : Integer literal constant is of an unsigned type but does not include a "U" suffix.           */
/*       Verification  : Lack of using "U" is accepted since it cause no problem in software behavior.                */
/**********************************************************************************************************************/
/* Message (1:1532)    : The function '%1s' is only referenced in one translation unit - but not the one in which it  */
/*                       is defined.                                                                                  */
/* Rule                : CERTCCM DCL19, MISRA C:2012 Rule-8.7                                                         */
/* JV-01 Justification : This macro identifier is following AUTOSAR standard rule  (Symbolic Name or Published        */
/*                       Macro's name), so this is accepted                                                           */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : This is accepted, due to the implementation is following hardware specification or it was    */
/*                       checked or do not dereference to NULL pointer.                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : It is specific for device register accessing and confirmed has no issue in software behavior.*/
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3383)    : Cannot identify wraparound guard for unsigned arithmetic expression.                         */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : It can still result in values that are out of range for the intended use, as intuitive       */
/*                       "invariants" may not hold                                                                    */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:4342)    : An expression of 'essentially unsigned' type (%1s) is being cast to enum type '%2s'.         */
/* Rule                : MISRA C:2012 Rule-10.5                                                                       */
/* JV-01 Justification : It is assigned to a variable with no conflict in the data.                                   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : Use of #include directive after code fragment.                                               */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is done as per Memory Requirement, (MEMMAP003 - Specification of Memory Mapping).       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                    Global Data                                                     **
***********************************************************************************************************************/

/***********************************************************************************************************************
**                                                Function Definitions                                                **
***********************************************************************************************************************/
/***********************************************************************************************************************
** Function Name        : Icu_HW_Tau_StartCountMeasurement
**
** Service ID           : None
**
** Description          : This service routine starts the count measurement
**                        for starting edge counting or signal measurement or timestamp measurement.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Reentrant
**
** Input Parameters     : LddChannel
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelUserRegConfig, Icu_GpChannelConfig, Icu_GenModuleMode, Icu_GpChannelRamData.
**
** Functions invoked    : ICU_ENTER_CRITICAL_SECTION, ICU_EXIT_CRITICAL_SECTION, RH850_SV_SET_ICR_SYNCP.
**
** Registers Used       : EICn
**
** Reference ID         : ICU_DUD_ACT_091, ICU_DUD_ACT_091_CRT001, ICU_DUD_ACT_091_CRT002,
** Reference ID         : ICU_DUD_ACT_091_REG001, ICU_DUD_ACT_091_REG002, ICU_DUD_ACT_091_REG003, ICU_DUD_ACT_091_REG004
***********************************************************************************************************************/
#if (ICU_TAU_UNIT_USED == STD_ON)
#if ((ICU_TIMESTAMP_API == STD_ON) || (ICU_EDGE_COUNT_API == STD_ON) || (ICU_SIGNAL_MEASUREMENT_API == STD_ON) || \
                                                                                        (ICU_EDGE_DETECT_API == STD_ON))
#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Tau_StartCountMeasurement(const Icu_ChannelType LddChannel)                         /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to the timer channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;

  /* Defining a pointer to the timer channel configuration parameters */
  P2CONST(Icu_ChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelUserRegConfig;

  /* Local variable to store the measurement mode of a channel */
  Icu_MeasurementModeType LenMeasurementMode;

  /* Read channel configuration pointer */
  LpChannelConfig = &Icu_GpChannelConfig[LddChannel];

  /* Read timer channel configuration pointer */
  LpChannelUserRegConfig = &Icu_GpChannelUserRegConfig[LddChannel];

  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Enter Critical Section */
  ICU_ENTER_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

  /* Check if the Interrupt register is present for particular channel */
  if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)                                                                    /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON)
    /* Clear the pending interrupts */
    RH850_SV_CLEAR_ICR_SYNCP(16, LpChannelConfig->pIntrCntlAddress, ICU_CLEAR_PENDING_INTR_MASK);                       /* PRQA S 1281, 2814 # JV-01, JV-01 */
    #endif
    /* Enabling the Interrupt processing */
    #if ((ICU_ENABLE_WAKEUP_API == STD_ON) || (ICU_NOTIFY_WAKEUP_INTERRUPT == STD_ON))
    if ((ICU_MODE_SLEEP == Icu_GenModuleMode) && ((uint8)ICU_FALSE == Icu_GpChannelRamData[LddChannel].ucWakeupEnable))
    {
      /* No action required */
    }
    else
    #endif
    {
      RH850_SV_CLEAR_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_ENABLE_EIMK_MASK);                           /* PRQA S 0751 # JV-01 */
    }
  } /* else No action required */
  /* End of  NULL_PTR != LpChannelConfig->pIntrCntlAddress */
  /* Read the channel's measurement mode */
  LenMeasurementMode = (Icu_MeasurementModeType)(LpChannelConfig->ucIcuMeasurementMode);                                /* PRQA S 4342 # JV-01 */

  if (ICU_MODE_SIGNAL_MEASUREMENT == LenMeasurementMode)
  {
    /* If the channel is duty cycle channel, then enable interrupt for the next channel also */
    if (ICU_DUTY_PERIOD_ENABLED_MASK == (uint8)(LpChannelUserRegConfig->ucChannelProperties &                           /* PRQA S 2814, 2844 # JV-01, JV-01 */
                                                                                          ICU_DUTY_PERIOD_ENABLED_MASK))
    {
      /* Read channel configuration pointer */
      LpChannelConfig = &Icu_GpChannelConfig[LddChannel + (uint32)ICU_ONE];                                             /* PRQA S 3383 # JV-01 */

      /* Check if the Interrupt register is present for particular channel */
      if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)                                                                /* PRQA S 2814, 2844 # JV-01, JV-01 */
      {
        #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON)
        /* Clear the pending interrupts */
        RH850_SV_CLEAR_ICR_SYNCP(16, LpChannelConfig->pIntrCntlAddress, ICU_CLEAR_PENDING_INTR_MASK);                   /* PRQA S 1281, 2814 # JV-01, JV-01 */
        #endif

        /* Enabling the Interrupt processing */
        #if ((ICU_ENABLE_WAKEUP_API == STD_ON) || (ICU_NOTIFY_WAKEUP_INTERRUPT == STD_ON))
        if ((ICU_MODE_SLEEP == Icu_GenModuleMode) &&
                                                  ((uint8)ICU_FALSE == Icu_GpChannelRamData[LddChannel].ucWakeupEnable))
        {
          /* No action required */
        }
        else
        #endif
        {
          RH850_SV_CLEAR_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_ENABLE_EIMK_MASK);                       /* PRQA S 0751 # JV-01 */
        }
      } /* else No action required */
      /* End of  NULL_PTR != LpChannelConfig->pIntrCntlAddress */
    } /* else No action required */
  } /* else No action required */
  /* End of (ICU_MODE_SIGNAL_MEASUREMENT == LenMeasurementMode) */
  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Exit Critical Section */
  ICU_EXIT_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

} /* End of Internal function Icu_HW_Tau_StartCountMeasurement */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of #if ((ICU_TIMESTAMP_API == STD_ON) || (ICU_EDGE_COUNT_API == STD_ON) \
                                        || (ICU_SIGNAL_MEASUREMENT_API == STD_ON) || (ICU_EDGE_DETECT_API == STD_ON)) */
#endif /* End of #if (ICU_TAU_UNIT_USED == STD_ON) */
/***********************************************************************************************************************
** Function Name        : Icu_HW_Tau_StopCountMeasurement
**
** Service ID           : None
**
** Description          : This service routine stops the count measurement
**                        for starting edge counting or signal measurement or timestamp measurement.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Reentrant
**
** Input Parameters     : LddChannel
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelUserRegConfig, Icu_GpChannelConfig.
**
** Functions invoked    : ICU_ENTER_CRITICAL_SECTION, ICU_EXIT_CRITICAL_SECTION, RH850_SV_SET_ICR_SYNCP.
**
** Registers Used       : EICn
**
** Reference ID         : ICU_DUD_ACT_090, ICU_DUD_ACT_090_CRT001
** Reference ID         : ICU_DUD_ACT_090_REG001, ICU_DUD_ACT_090_REG002, ICU_DUD_ACT_090_CRT002
***********************************************************************************************************************/
#if (ICU_TAU_UNIT_USED == STD_ON)
#if ((ICU_TIMESTAMP_API == STD_ON) || (ICU_EDGE_COUNT_API == STD_ON) || (ICU_SIGNAL_MEASUREMENT_API == STD_ON) || \
                                                                                        (ICU_EDGE_DETECT_API == STD_ON))
#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Tau_StopCountMeasurement(const Icu_ChannelType LddChannel)                          /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to the timer channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;

  /* Defining a pointer to the timer channel configuration parameters */
  P2CONST(Icu_ChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelUserRegConfig;
  /* Local variable to store the measurement mode of a channel */
  Icu_MeasurementModeType LenMeasurementMode;

  /* Read channel configuration pointer */
  LpChannelConfig = &Icu_GpChannelConfig[LddChannel];

  /* Read timer channel configuration pointer */
  LpChannelUserRegConfig = &Icu_GpChannelUserRegConfig[LddChannel];

  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Enter Critical Section */
  ICU_ENTER_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

  if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)                                                                    /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    /* Disabling the Interrupt processing */
    RH850_SV_SET_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress),  ICU_DISABLE_EIMK_MASK);                             /* PRQA S 0751, 2814 # JV-01, JV-01 */
  } /* else No action required */

  /* Read the channel's measurement mode */
  LenMeasurementMode = (Icu_MeasurementModeType)(LpChannelConfig->ucIcuMeasurementMode);                                /* PRQA S 4342 # JV-01 */
  if (ICU_MODE_SIGNAL_MEASUREMENT == LenMeasurementMode)
  {
    /* If the channel is duty cycle channel, then disable interrupt for the next channel also */
    if (ICU_DUTY_PERIOD_ENABLED_MASK == (uint8)(LpChannelUserRegConfig->ucChannelProperties &                           /* PRQA S 2814, 2844 # JV-01, JV-01 */
                                                                                          ICU_DUTY_PERIOD_ENABLED_MASK))
    {
      /* Read channel configuration pointer */
      LpChannelConfig = &Icu_GpChannelConfig[LddChannel + (uint32)ICU_ONE];                                             /* PRQA S 3383 # JV-01 */
      if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)                                                                /* PRQA S 2814, 2844 # JV-01, JV-01 */
      {
        /* Disabling the Interrupt processing */
        RH850_SV_SET_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_DISABLE_EIMK_MASK);                          /* PRQA S 0751, 2814 # JV-01, JV-01 */
      } /* else No action required */
    } /* else No action required */
  } /* else No action required */
  /* End of (ICU_MODE_SIGNAL_MEASUREMENT == LenMeasurementMode) */
  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Exit Critical Section */
  ICU_EXIT_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

} /* End of Internal function Icu_HW_Pin_StopCountMeasurement */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of ((ICU_TIMESTAMP_API == STD_ON) || (ICU_EDGE_COUNT_API == STD_ON) \
                                       || (ICU_SIGNAL_MEASUREMENT_API == STD_ON) || (ICU_EDGE_DETECT_API == STD_ON))  */
#endif /* Edn of #if (ICU_TAU_UNIT_USED == STD_ON)*/
/***********************************************************************************************************************
**                                                    End of File                                                     **
***********************************************************************************************************************/
