/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Lin_RLIN3_LLDriver.c                                                                                */
/* SW-VERSION   = 1.5.0                                                                                               */
/*====================================================================================================================*/
/*                                                     COPYRIGHT                                                      */
/*====================================================================================================================*/
/* (c) 2020-2022 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* ISR functions of the LIN Driver Component.                                                                         */
/*                                                                                                                    */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s) and/or */
/* the Application.                                                                                                   */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs of  */
/* program errors, compliance with applicable laws, damage to or loss of data, programs or equipment, and             */
/* unavailability or interruption of operations.                                                                      */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*              Devices:        X2x                                                                                   */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                              Revision Control History                                              **
***********************************************************************************************************************/
/*
 *
 * 1.4.4:  24/11/2022   : As per ARDAACA-1302:
 *                        Change Dem_SetEventStatus/Dem_ReportErrorStatus to LIN_DEM_REPORT_ERROR
 *                        in Function(s) invoked field for following functions: Lin_TxIsr, Lin_RxIsr, Lin_ErrIsr,
 *                        Lin_HwErrorProcessing and Lin_ErrorReport.
 *                        API Lin_HwProcessData: Added LIN_ENTER_CRITICAL_SECTION, LIN_EXIT_CRITICAL_SECTION to
 *                        Functions invoked field in function description.
 *                        As per ARDAACA-1266:
 *                        API Lin_ErrorReport: Removed LulIndex from function description
 *                        API Lin_HwProcessData: Removed LIN_ENTER_CRITICAL_SECTION, LIN_EXIT_CRITICAL_SECTION
 *                        from Global Variables field in function description
 *                        As per ARDAACA-1265: Function Lin_HwSendHeaderInternal,
 *                        change the Return parameter from None to Std_ReturnType in function description.
 *         24/06/2022   : Add QAC message 9.5.0.
 *         22/06/2022   : Modify the access size passed in RH850_SV_MODE_ICR_AND, when only MKn bits are updated. 
 * 1.4.3:  17/05/2022   : Update Lin_HwInit, add setting value for Interrupt Select register.
 *         10/05/2022   : Remove "else" statement don't use when no action required.
 *         20/04/2022   : Change from ucLin3IntMuxNumer to ucLin3IntMuxNumber
 *         14/04/2022   : Update SW-VERSION to 1.4.3
 * 1.4.2:  04/03/2022   : Fix QAC findings for QAC message 4399.
 * 1.4.1:  30/12/2021   : Update Lin_HwInit, add setting value for Interrupt Select register.
 *                        Add QAC message 9.5.0 (4:0312).
 * 1.4.0:  18/11/2021   : Update Critical section nesting for Lin_HwSendHeaderInternal and Lin_HwHandleWakeupRequest
 * 1.3.2:  25/08/2021   : Add condition to check interrupt disable or enable for Lin_HwSetMasterMode
 *                        Add clear error flag after changing mode for Lin_HwSetMasterMode 
 * 1.3.1:  02/07/2021   : Add QAC message 9.5.0.
 *                        Format source code to 120 characters.
 *                        Improve Violation tag (remove START/END).
 *         20/05/2021   : Fix QAC findings.
 * 1.2.0:  26/08/2020   : Release.
 * 1.1.0:  19/06/2020   : Release.
 * 1.0.2:  08/04/2020   : Update Lin_HWInit, add macros to set wakeup baud rate specifications.
 * 1.0.1:  25/03/2020   : Update Format dd/mm/yyyy and revision format.
 * 1.0.0:  30/03/2020   : Initial Version.
 *
 */
/**********************************************************************************************************************/

/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                  Include Section                                                   **
***********************************************************************************************************************/
/* Lin APIs header file */
#include "Lin.h"

/* Included for extern declarations of global variables */
#include "Lin_Ram.h"

/* Lin Transmit receive header for the target */
#include "Lin_RLIN3_LLDriver.h"
#if (LIN_DEV_ERROR_DETECT == STD_ON)
/* Default Error header */
#include "Det.h"
#endif
/* Included for declaration of the function Dem_SetEventStatus() and Dem_ReportErrorStatus() */
#include "Dem.h"
#if (LIN_WAKEUP_SUPPORT == STD_ON)
/* Included for the declaration of EcuM_CheckWakeup() */
/* EcuM Callback header file */
#include "EcuM_Cbk.h"
#endif
/* Schedule Manager header */
#if (LIN_CRITICAL_SECTION_PROTECTION == STD_ON)
#include "SchM_Lin.h"
#endif
/* Provision for platform dependent types */
#include "rh850_Types.h"

/***********************************************************************************************************************
**                                                Version Information                                                 **
***********************************************************************************************************************/
/* AUTOSAR Release version information */
#define LIN_RLIN3_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION    LIN_AR_RELEASE_MAJOR_VERSION_VALUE
#define LIN_RLIN3_LLDRIVER_C_AR_RELEASE_MINOR_VERSION    LIN_AR_RELEASE_MINOR_VERSION_VALUE                             /* PRQA S 0791 # JV-01 */
#define LIN_RLIN3_LLDRIVER_C_AR_RELEASE_REVISION_VERSION LIN_AR_RELEASE_REVISION_VERSION_VALUE                          /* PRQA S 0791 # JV-01 */

/* File version information */
#define LIN_RLIN3_LLDRIVER_C_SW_MAJOR_VERSION            LIN_SW_MAJOR_VERSION_VALUE
#define LIN_RLIN3_LLDRIVER_C_SW_MINOR_VERSION            LIN_SW_MINOR_VERSION_VALUE

/***********************************************************************************************************************
**                                                   Version Check                                                    **
***********************************************************************************************************************/
#if (LIN_RLIN3_LLDRIVER_AR_RELEASE_MAJOR_VERSION != LIN_RLIN3_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION)
#error "Lin_RLIN3_LLDriver.c : Mismatch in Release Major Version"
#endif

#if (LIN_RLIN3_LLDRIVER_AR_RELEASE_MINOR_VERSION != LIN_RLIN3_LLDRIVER_C_AR_RELEASE_MINOR_VERSION)
#error "Lin_RLIN3_LLDriver.c : Mismatch in Release Minor Version"
#endif

#if (LIN_RLIN3_LLDRIVER_AR_RELEASE_REVISION_VERSION != LIN_RLIN3_LLDRIVER_C_AR_RELEASE_REVISION_VERSION)
#error "Lin_RLIN3_LLDriver.c : Mismatch in Release Revision Version"
#endif

#if (LIN_RLIN3_LLDRIVER_SW_MAJOR_VERSION != LIN_RLIN3_LLDRIVER_C_SW_MAJOR_VERSION)
#error "Lin_RLIN3_LLDriver.c : Mismatch in Software Major Version"
#endif

#if (LIN_RLIN3_LLDRIVER_SW_MINOR_VERSION != LIN_RLIN3_LLDRIVER_C_SW_MINOR_VERSION)
#error "Lin_RLIN3_LLDriver.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (7:0791)    : Macro identifier does not differ from other macro identifier(s) (e.g. '%s') within the       */
/*                       specified number of significant characters.                                                  */
/* Rule                : CERTCCM DCL23, MISRA C:2012 Rule-5.4                                                         */
/* JV-01 Justification : This macro identifier is following AUTOSAR standard rule (Symbolic Name or Published         */
/*                       Macro's name), so this is accepted.                                                          */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : Use of #include directive after code fragment.                                               */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is done as per Memory Requirement, (MEMMAP003 - Specification of Memory Mapping).       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1532)    : The function '%1s' is only referenced in one translation unit - but not the one in which it  */
/*                       is defined.                                                                                  */
/* Rule                : CERTCCM DCL19, MISRA C:2012 Rule-8.7                                                         */
/* JV-01 Justification : This macro identifier is following AUTOSAR standard rule  (Symbolic Name or Published        */
/*                       Macro's name), so this is accepted.                                                          */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3432)    : Simple macro argument expression is not parenthesized.                                       */
/* Rule                : MISRA C:2012 Rule-20.7                                                                       */
/* JV-01 Justification : Compiler keyword (macro) is defined and used followed AUTOSAR standard rule. It is accepted. */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2982)    : This assignment is redundant. The value of this object is never used before being modified.  */
/* Rule                : CERTCCM MSC07, MSC13, MISRA C:2012 Rule-2.2                                                  */
/* JV-01 Justification : The variable needs to be initialized before using it.                                        */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (7:0404)    : More than one read access to volatile objects between sequence points.                       */
/* Rule                : CERTCCM EXP10, EXP30, MISRA C:2012 Rule-1.3, Rule-13.2                                       */
/* JV-01 Justification : This is to get element in array of struct, volatile of counter variable of 'for' loop is     */
/*                       used to ensure no optimization. It is accepted.                                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : It is specific for device register accessing and confirmed has no issue in software behavior.*/
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : This is accepted, due to the implementation is following hardware specification.             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/* JV-02 Justification : This is accepted, the pointer can not be NULL and ensured by gentool.                        */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:3387)    : A full expression containing an increment (++) or decrement (--) operator should have no     */
/*                       potential side effects other than that caused by the increment or decrement operator.        */
/* Rule                : MISRA C:2012 Rule-13.3                                                                       */
/* JV-01 Justification : An increment/decrement is created a side affect. In this case it's accessing a volatile      */
/*                       object. This can be accepted.                                                                */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3678)    : The object referenced by '%1s' is not modified through it, so '%1s' could be declared with   */
/*                       type '%2s'.                                                                                  */
/* Rule                : MISRA C:2012 Rule-8.13                                                                       */
/* JV-01 Justification : This is accepted. It just an advise for improve safety by reducing the possibility that      */
/*                       the referenced data is unintentionally modified through an unexpected alias and improves     */
/*                       clarity by indicating that the referenced data is not intended to be modified through this   */
/*                       alias or those depending on it.                                                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:3206)    : The parameter '%s' is not used in this function.                                             */
/* Rule                : CERTCCM MSC07, MSC13, MISRA C:2012 Rule-2.7                                                  */
/* JV-01 Justification : This is done as per implementation requirement.                                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1281)    : Integer literal constant is of an unsigned type but does not include a "U" suffix.           */
/* Rule                : MISRA C:2012 Rule-7.2                                                                        */
/* JV-01 Justification : Integer literal constant is of an unsigned type but does not include a "U" suffix.           */
/*       Verification  : Lack of using "U" is accepted since it cause no problem in software behavior.                */
/**********************************************************************************************************************/
/* Message (3:3416)    : Logical operation performed on expression with persistent side effects.                      */
/* Rule                : CERTCCM EXP45                                                                                */
/* JV-01 Justification : Logical operation accesses volatile object which is a register access. All register          */
/*                       addresses are generated with volatile qualifier. There is no impact on the functionality     */
/*                       due to this conditional check for mode change.                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:3415)    : Right hand operand of '&&' or '||' is an expression with persistent side effects.            */
/* Rule                : CERTCCM EXP02, MISRA C:2012 Rule-13.5                                                        */
/* JV-01 Justification : Although it is a volatile object, it is not a direct access to the HW register, and there    */
/*                       is no side effect.                                                                           */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3384)    : Cannot identify wraparound guard for dependent unsigned arithmetic expression.               */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : LIN_ZERO is the lower constraint for the parameter.                                          */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0751)    : Casting to char pointer type.                                                                */
/* Rule                : CERTCCM EXP11, EXP39                                                                         */
/* JV-01 Justification : Using void due to specific requirement of input parameter. So, this can be skipped.          */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3464)    : Argument to macro '%s' contains a side effect that will be evaluated more than once.         */
/* Rule                : CERTCCM PRE31                                                                                */
/* JV-01 Justification : This message is only emitted for expressions expanded from argument tokens written out at    */
/*                       the top level, that did not themselves originate from a macro expansion.                     */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:4391)    : A composite expression of 'essentially unsigned' type (unsigned char) is being cast to a     */
/*                       wider unsigned type 'unsigned short'.                                                        */
/* Rule                : MISRA C:2012 Rule-10.8                                                                       */
/* JV-01 Justification : This is necessary to support configuration. The actuall array can't be decided statically.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (7:4399)    : An expression which is the result of a ~ or << has been cast to a wider type.                */
/* Rule                : MISRA-C:2012 Rule 10.8                                                                       */
/* JV-01 Justification : Since the -strictrank option is not used, int and long types are guaranteed to be equivalent.*/
/*                       Therefore, it does not affect the result of the arithmetic processing of the corresponding   */
/*                       part.                                                                                        */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:0312)    : Dangerous pointer cast results in loss of volatile qualification.                            */
/* Rule                : CERTCCM EXP32, MISRA C:2012 Rule-11.8                                                        */
/* JV-01 Justification : According to the coding guide, all global variables must have volatile attribute, but        */
/*                       volatile must be removed when return these variables from APIs according to the AUTOSAR      */
/*                       specification.                                                                               */
/*       Verification  : Since these are normal memory objects, any problem doesn't occurs.                           */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                    Global Data                                                     **
***********************************************************************************************************************/
/***********************************************************************************************************************
**                                             Local Function Prototypes                                              **
***********************************************************************************************************************/
#define LIN_START_SEC_CODE_FAST
#include "Lin_MemMap.h"

STATIC FUNC(void, LIN_CODE_FAST) Lin_HwProcessData(const uint8 LucChannel);

STATIC FUNC(void, LIN_CODE_FAST) Lin_HwErrorProcessing(const uint8 LucChannel);

#if (LIN_WAKEUP_SUPPORT == STD_ON)
STATIC FUNC(void, LIN_CODE_FAST) Lin_HwWakeUpFromBus(const uint8 LucChannel);
#endif

STATIC FUNC(void, LIN_CODE_FAST) Lin_HwWakeupOnGoing(const uint8 LucChannel);

#if defined(LIN_E_TIMEOUT) || (LIN_DEV_ERROR_DETECT == STD_ON)
STATIC FUNC(Std_ReturnType, LIN_CODE_FAST) Lin_ErrorReport(const uint8 LucApiId);
#endif

#define LIN_STOP_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

#define LIN_START_SEC_PRIVATE_CODE
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

STATIC FUNC(void, LIN_PRIVATE_CODE) Lin_HwLoadTxBuffer(const uint8 LucChannel);

#define LIN_STOP_SEC_PRIVATE_CODE
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
**                                                Function Definitions                                                **
***********************************************************************************************************************/

/***********************************************************************************************************************
** Function Name         : Lin_TxIsr
**
** Service ID            : None
**
** Description           : Interrupt Service Handler for transmission interrupt.
**
** Sync/Async            : Sync
**
** Reentrancy            : Non-Reentrant for same channel
**
** Input Parameters      : LucChannel : Number of LIN channel
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Lin_GpChannelRamData, Lin_GpRLIN3Properties
**
** Function(s) invoked   : LIN_DEM_REPORT_ERROR,
**                         Lin_HwWakeupOnGoing(when calling Lin_Wakeup),
**                         Lin_HwProcessData(when callin Lin_SendFrame with
**                         LIN_MASTER_RESPONSE or Lin_GoToSleep)
**
** Registers Used        : None
**
** Reference ID          : LIN_DUD_ACT_013,
** Reference ID          : LIN_DUD_ACT_013_ERR001,
** Reference ID          : LIN_DUD_ACT_013_ERR002
***********************************************************************************************************************/
#define LIN_START_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
FUNC(void, LIN_CODE_FAST) Lin_TxIsr(const uint8 LucChannel)                                                             /* PRQA S 1532 # JV-01 */
{
#if (LIN_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  /* Check if the EIMK bit of the EIC register is set */
  if (LIN_EIC_EIMK_MASK ==
      (uint16)((*(Lin_GpRLIN3Properties[LucChannel].pLin3IntTxEicReg)) & (uint16)LIN_EIC_EIMK_MASK))
  {
    #ifdef LIN_E_INT_INCONSISTENT
    /* Reporting to DEM that interrupt from unknown source */
    LIN_DEM_REPORT_ERROR(LIN_E_INT_INCONSISTENT, DEM_EVENT_STATUS_FAILED);
    #endif /* #ifdef LIN_E_INT_INCONSISTENT */
  }
  else
#endif /* (LIN_INTERRUPT_CONSISTENCY_CHECK == STD_ON) */
  {

    if (LIN_TRUE == Lin_GpChannelRamData[LucChannel].blWakeupCalled)
    {
      Lin_HwWakeupOnGoing(LucChannel);
    }
    else
    {
      /* Invoke Lin_HwProcessData to deal with the Tx Interrupt */
      Lin_HwProcessData(LucChannel);
    }
  }
}
#define LIN_STOP_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name         : Lin_RxIsr
**
** Service ID            : None
**
** Description           : Interrupt Service Handler for reception interrupt.
**
** Sync/Async            : Sync
**
** Reentrancy            : Non-Reentrant for same channel
**
** Input Parameters      : LucChannel : Number of LIN channel
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Lin_GpChannelRamData, Lin_GpRLIN3Properties
**
** Function(s) invoked   : LIN_DEM_REPORT_ERROR,
**                         Lin_HwWakeUpFromBus(when detecting a wake-up pulse),
**                         Lin_HwProcessData(when calling Lin_SendFrame with
**                         LIN_SLAVE_RESPONSE)
**
** Registers Used        : None
**
** Reference ID          : LIN_DUD_ACT_014,
** Reference ID          : LIN_DUD_ACT_014_ERR001,
** Reference ID          : LIN_DUD_ACT_014_ERR002
***********************************************************************************************************************/
#define LIN_START_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
FUNC(void, LIN_CODE_FAST) Lin_RxIsr(const uint8 LucChannel)                                                             /* PRQA S 1532 # JV-01 */
{
#if (LIN_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  /* Check if the EIMK bit of the EIC register is set */
  if (LIN_EIC_EIMK_MASK ==
      (uint16)((*(Lin_GpRLIN3Properties[LucChannel].pLin3IntRxEicReg)) & (uint16)LIN_EIC_EIMK_MASK))
  {
    #ifdef LIN_E_INT_INCONSISTENT
    /* Reporting to DEM that interrupt from unknown source */
    LIN_DEM_REPORT_ERROR(LIN_E_INT_INCONSISTENT, DEM_EVENT_STATUS_FAILED);
    #endif /* #ifdef LIN_E_INT_INCONSISTENT */
  }
  else
#endif /* (LIN_INTERRUPT_CONSISTENCY_CHECK == STD_ON) */
  {
  /* Check if global wakeup support is enabled */
  #if (LIN_WAKEUP_SUPPORT == STD_ON)
    /* Check if channel status is sleep */
    if ((LIN_CH_SLEEP == Lin_GpChannelRamData[LucChannel].enChannelStatus) ||
        (LIN_TRUE == Lin_GpChannelRamData[LucChannel].blSleepPending))
    {
      /* Invoke Lin_HwWakeUpFromBus() */
      Lin_HwWakeUpFromBus(LucChannel);
    }
    else
  #endif
    {
      /* Invoke Lin_HwProcessData to deal with the Rx Interrupt */
      Lin_HwProcessData(LucChannel);
    }
  }
}
#define LIN_STOP_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name         : Lin_ErrIsr
**
** Service ID            : None
**
** Description           : Interrupt Service Handler for error interrupt.
**
** Sync/Async            : Sync
**
** Reentrancy            : Non-Reentrant for same channel
**
** Input Parameters      : LucChannel : Number of LIN channel
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Lin_GpRLIN3Properties
**
** Function(s) invoked   : LIN_DEM_REPORT_ERROR,
**                         Lin_HwErrorProcessing
**
** Registers Used        : None
**
** Reference ID          : LIN_DUD_ACT_015,
** Reference ID          : LIN_DUD_ACT_015_ERR001,
** Reference ID          : LIN_DUD_ACT_015_ERR002
***********************************************************************************************************************/
#define LIN_START_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
FUNC(void, LIN_CODE_FAST) Lin_ErrIsr(const uint8 LucChannel)                                                            /* PRQA S 1532 # JV-01 */
{
#if (LIN_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  /* Check if the EIMK bit of the EIC register is set */
  if (LIN_EIC_EIMK_MASK ==
      (uint16)((*(Lin_GpRLIN3Properties[LucChannel].pLin3IntStEicReg)) & (uint16)LIN_EIC_EIMK_MASK))
  {
    #ifdef LIN_E_INT_INCONSISTENT
    /* Reporting to DEM that interrupt from unknown source */
    LIN_DEM_REPORT_ERROR(LIN_E_INT_INCONSISTENT, DEM_EVENT_STATUS_FAILED);
    #endif /* #ifdef LIN_E_INT_INCONSISTENT */
  }
  else
#endif /* (LIN_INTERRUPT_CONSISTENCY_CHECK == STD_ON) */
  {

    /* Calling error processing API for error check */
    Lin_HwErrorProcessing(LucChannel);
  }
}
#define LIN_STOP_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name      : Lin_HwSendHeaderInternal
**
** Service ID         : NA
**
** Description        : Internal Function to start frame transmission
**
** Sync/Async         : Synchronous
**
** Re-entrancy        : Non Re-entrant
**
** Input Parameters   : LucChannel, LpPduInfoPtr, LucApiId
**
** Output Parameters  : None
**
** Return parameter   : Std_ReturnType
**
** Preconditions      : None
**
** Global Variables   : Lin_GpChannelConfig, Lin_GpChannelRamData,
**                      Lin_GpRLIN3Properties
**
** Functions invoked  : Lin_HwLoadTxBuffer, Lin_HwSetMasterMode,
**                      Lin_HwScheduleInterrupt
**                      LIN_ENTER_CRITICAL_SECTION, LIN_EXIT_CRITICAL_SECTION
**
** Registers Used     : ucRLN3nLIDB,
**                      ucRLN3nLDFC,
**                      ucRLN3nLTRC
**
** Reference ID       : LIN_DUD_ACT_016, LIN_DUD_ACT_016_GBL001, LIN_DUD_ACT_016_CRT003, LIN_DUD_ACT_016_CRT004,
** Reference ID       : LIN_DUD_ACT_016_GBL002, LIN_DUD_ACT_016_GBL003,
** Reference ID       : LIN_DUD_ACT_016_GBL004, LIN_DUD_ACT_016_GBL005,
** Reference ID       : LIN_DUD_ACT_016_GBL006, LIN_DUD_ACT_016_GBL007,
** Reference ID       : LIN_DUD_ACT_016_GBL008, LIN_DUD_ACT_016_REG001,
** Reference ID       : LIN_DUD_ACT_016_REG002, LIN_DUD_ACT_016_REG003,
** Reference ID       : LIN_DUD_ACT_016_REG004, LIN_DUD_ACT_016_REG005,
** Reference ID       : LIN_DUD_ACT_016_REG006, LIN_DUD_ACT_016_REG007,
** Reference ID       : LIN_DUD_ACT_016_CRT001, LIN_DUD_ACT_016_CRT002,
** Reference ID       : LIN_DUD_ACT_016_REG008
***********************************************************************************************************************/
#define LIN_START_SEC_PRIVATE_CODE
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
FUNC(Std_ReturnType, LIN_PRIVATE_CODE)
Lin_HwSendHeaderInternal(const uint8 LucChannel, CONSTP2CONST(Lin_PduType, AUTOMATIC, LIN_PRIVATE_CONST) LpPduInfoPtr,  /* PRQA S 1532 # JV-01 */
                         const uint8 LucApiId)
{
  volatile P2CONST(Lin_ChannelInfo, AUTOMATIC, LIN_PRIVATE_CONST) LpLinChannelPtr;
  volatile P2CONST(Lin3_ChannelConfigType, AUTOMATIC, LIN_PRIVATE_CONST) LpLin3ChannelPtr;
  volatile P2VAR(RLin3_UartRegs, AUTOMATIC, REGSPACE) LpLN3ChannelRegs;                                                 /* PRQA S 3432 # JV-01 */
  volatile P2VAR(Lin_RamData, AUTOMATIC, LIN_VAR_FAST_NO_INIT) LpRamVars;                                               /* PRQA S 3432 # JV-01 */
  Std_ReturnType LucReturnValue;

  Lin_FrameResponseType LenFrameType;

  /* Initialize the return value */
  LucReturnValue = E_NOT_OK;                                                                                            /* PRQA S 2982 # JV-01 */

  /* Get the pointer to channel RAM data */
  LpRamVars = &Lin_GpChannelRamData[LucChannel];                                                                        /* PRQA S 0404 # JV-01 */

  /* Copy the frame Id to frame data structure */
  LpRamVars->ucFrameId = LpPduInfoPtr->Pid;                                                                             /* PRQA S 2844, 2814 # JV-01, JV-02 */

  /* Copy the checksum model to frame data structure */
  LpRamVars->enCheckSumModel = LpPduInfoPtr->Cs;

  /* Copy the data length to frame data structure */
  LpRamVars->ucFrameLength = LpPduInfoPtr->Dl;

  /* Copy the frame type to frame data structure */
  LpRamVars->enFrameType = LpPduInfoPtr->Drc;

  /* Get the address of LIN Channel structure */
  LpLinChannelPtr = &Lin_GpChannelConfig[LucChannel];                                                                   /* PRQA S 0404 # JV-01 */

  LpLin3ChannelPtr = &Lin_GpRLIN3Properties[LpLinChannelPtr->ucPropertiesIndex];                                        /* PRQA S 0404, 2844, 2814 # JV-01, JV-01, JV-02 */

  /* Get the RLIN3 Channel Base Address */
  LpLN3ChannelRegs = LpLin3ChannelPtr->pLn3ChanlBaseAddress;                                                            /* PRQA S 2844, 2814 # JV-01, JV-02 */

  /* Get frame type */
  LenFrameType = LpRamVars->enFrameType;

  if (LIN_TRUE == LpRamVars->blWakeupCalled)
  {
    /* Reset the LinIf wake-up flag */
    LpRamVars->blWakeupCalled = LIN_FALSE;
  } /* else No action required */

  /* Enter the protection area */
  LIN_ENTER_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);

  /* Set RLIN3 to Reset Mode */
  LucReturnValue = Lin_HwSetMasterMode(LucChannel, RLIN3_RESET_MODE, LucApiId);

  if (E_OK == LucReturnValue)
  {
    /* Set RLIN3 to Operation Mode */
    LucReturnValue = Lin_HwSetMasterMode(LucChannel, RLIN3_OPERATION_MODE, LucApiId);
  } /* else No action required */

  /* Exit the protection area */
  LIN_EXIT_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);

  if (E_OK == LucReturnValue)
  {
    /* Set the frame Id */
    LpLN3ChannelRegs->ucRLN3nLIDB = LpRamVars->ucFrameId;                                                               /* PRQA S 2814, 2844 # JV-02, JV-01 */

    /* Reset of LIN data Field Configuration Register */
    LpLN3ChannelRegs->ucRLN3nLDFC = LIN_RESET_VALUE;

    /* If checksum model is of type CLASSIC */
    if (LIN_CLASSIC_CS == LpRamVars->enCheckSumModel)
    {
      /* Set the checksum model to CLASSIC */
      LpLN3ChannelRegs->ucRLN3nLDFC = (LpLN3ChannelRegs->ucRLN3nLDFC & LIN_SET_CLS_CHKSUM);
    }
    /* If checksum model is of type ENHANCED */
    else
    {
      /* Set the checksum model to ENHANCED */
      LpLN3ChannelRegs->ucRLN3nLDFC = (LpLN3ChannelRegs->ucRLN3nLDFC | LIN_SET_ENH_CHKSUM);
    }

    /* Set length of total bytes in a RLIN3 Frame(Response data + Checksum) */
    LpLN3ChannelRegs->ucRLN3nLDFC = (LpLN3ChannelRegs->ucRLN3nLDFC | LpPduInfoPtr->Dl);

    if (LIN_MASTER_RESPONSE == LenFrameType)
    {
      /* To Set the Transmit Data */
      /*Call the internal function to load Tx Buffer and start the frame
        transmission */
      Lin_HwLoadTxBuffer(LucChannel);

      /* Set Response direction for Transmit */
      LpLN3ChannelRegs->ucRLN3nLDFC = (LpLN3ChannelRegs->ucRLN3nLDFC | LIN_MASTER_DIRECTION);
    }
    else
    {
      /* Set Response direction for Receive */
      LpLN3ChannelRegs->ucRLN3nLDFC = (LpLN3ChannelRegs->ucRLN3nLDFC & LIN_SLAVE_DIRECTION);
    }

    /* Enter the protection area */
    LIN_ENTER_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);

    /* Enable RLIN3 Interrupt */
    Lin_HwScheduleInterrupt(LucChannel, LIN_INTERRUPT_EN);

    /* Exit the protection area */
    LIN_EXIT_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);
    switch (LenFrameType)
    {
    case LIN_MASTER_RESPONSE:
      /* Check if the transmitted frame is for go-to-sleep command */
      if (LIN_SET_SLEEP_REQUEST == LpRamVars->ucSlpRqst_RespRdy)
      {
        /* No action required, keep LpRamVars->blSleepPending as LIN_TRUE */
      }
      else
      {
        /* Set the Channel Status */
        LpRamVars->enChannelStatus = LIN_TX_BUSY;
      }
      break;
    case LIN_SLAVE_TO_SLAVE:
      /* Set the Channel Status */
      LpRamVars->enChannelStatus = LIN_TX_BUSY;
      break;
    case LIN_SLAVE_RESPONSE:
      /* Set the Channel Status */
      LpRamVars->enChannelStatus = LIN_RX_BUSY;
      break;
    default:
      /* No action required */
      break;
    }

    /* Initiate Transmission */
    LpLN3ChannelRegs->ucRLN3nLTRC = (LpLN3ChannelRegs->ucRLN3nLTRC | LIN_START_TRANSMIT);
  } /* End of if (E_OK == LucReturnValue) */ /* else No action required */   
  return (LucReturnValue);
}
#define LIN_STOP_SEC_PRIVATE_CODE
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name      : Lin_HwWakeUpFromBus
**
** Service ID         : NA
**
** Description        : This service does wake-up (from the bus) processing
**
** Sync/Async         : Synchronous
**
** Re-entrancy        : Non Re-entrant for the same channel
**
** Input Parameters   : LucChannel
**
** Output Parameters  : None
**
** Return parameter   : None
**
** Preconditions      : None
**
** Global Variables   : Lin_GpChannelConfig, Lin_GpChannelRamData,
**                      Lin_GpRLIN3Properties
**
** Functions invoked  : EcuM_CheckWakeup
**
** Registers Used     : ucRLN3nLST
**
** Reference ID       : LIN_DUD_ACT_017, LIN_DUD_ACT_017_GBL001,
** Reference ID       : LIN_DUD_ACT_017_REG001
***********************************************************************************************************************/
#if (LIN_WAKEUP_SUPPORT == STD_ON)
#define LIN_START_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
STATIC FUNC(void, LIN_CODE_FAST) Lin_HwWakeUpFromBus(const uint8 LucChannel)
{
  volatile P2CONST(Lin_ChannelInfo, AUTOMATIC, LIN_CONST) LpLinChannelPtr;
  volatile P2CONST(Lin3_ChannelConfigType, AUTOMATIC, LIN_CONST) LpLin3ChannelPtr;
  volatile P2VAR(RLin3_UartRegs, AUTOMATIC, REGSPACE) LpLN3ChannelRegs;                                                 /* PRQA S 3432 # JV-01 */
  volatile P2VAR(Lin_RamData, AUTOMATIC, LIN_VAR_FAST_NO_INIT) LpRamVars;                                               /* PRQA S 3432 # JV-01 */

  /* Get the address of LIN Channel structure */
  LpLinChannelPtr = &Lin_GpChannelConfig[LucChannel];                                                                   /* PRQA S 0404 # JV-01 */

  /* Get the pointer to channel RAM data */
  LpRamVars = &Lin_GpChannelRamData[LucChannel];                                                                        /* PRQA S 0404 # JV-01 */

  /* Check if channel specific wake-up is enabled */
  if (LIN_TRUE == LpLinChannelPtr->blWakeupSupport)                                                                     /* PRQA S 2844, 2814 # JV-01, JV-02 */
  {
    LpLin3ChannelPtr = &Lin_GpRLIN3Properties[LpLinChannelPtr->ucPropertiesIndex];                                      /* PRQA S 0404 # JV-01 */

    /* Get the RLIN3 Channel Base Address */
    LpLN3ChannelRegs = LpLin3ChannelPtr->pLn3ChanlBaseAddress;                                                          /* PRQA S 2844, 2814 # JV-01, JV-02 */

    if (LIN_RESPONSE_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_RESPONSE_COMPLETE))                         /* PRQA S 2844, 2814 # JV-01, JV-01 */
    {
      /* Clear sleep requested bit */
      LpRamVars->ucSlpRqst_RespRdy = LIN_WAKEUP;                                                                        /* PRQA S 2814, 2844 # JV-02, JV-01 */

      /* Clear the reception interrupt */
      LpLN3ChannelRegs->ucRLN3nLST = (LpLN3ChannelRegs->ucRLN3nLST & LIN_RECEPTION_DONE);
    } /* else No action required */

    /* Invoke the upper layer wakeup notification */
    EcuM_CheckWakeup((EcuM_WakeupSourceType)LIN_ONE << (LpLinChannelPtr->ucWakeupSourceId));

  } /* Channel specific wakeup is enabled */ /* else No action required */
}
#define LIN_STOP_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
#endif /* #if (LIN_WAKEUP_SUPPORT == STD_ON) */

/***********************************************************************************************************************
** Function Name      : Lin_HwProcessData
**
** Service ID         : NA
**
** Description        : This service processes the transmit and receive
**                      interrupt requests
**
** Sync/Async         : Synchronous
**
** Re-entrancy        : Non Re-entrant for the same channel
**
** Input Parameters   : LucChannel
**
** Output Parameters  : None
**
** Return parameter   : None
**
** Preconditions      : None
**
** Global Variables   : Lin_GpChannelConfig, Lin_GpChannelRamData,
**                      Lin_GpRLIN3Properties
**
** Functions invoked  : Lin_HwHandleWakeupRequest, Lin_HwScheduleInterrupt,
**                      LIN_ENTER_CRITICAL_SECTION, LIN_EXIT_CRITICAL_SECTION
**
** Registers Used     : ucRLN3nLST,  ucRLN3nLDBR
**
** Reference ID       : LIN_DUD_ACT_018, LIN_DUD_ACT_018_GBL001,
** Reference ID       : LIN_DUD_ACT_018_GBL002, LIN_DUD_ACT_018_GBL003,
** Reference ID       : LIN_DUD_ACT_018_GBL004, LIN_DUD_ACT_018_GBL005,
** Reference ID       : LIN_DUD_ACT_018_GBL006, LIN_DUD_ACT_018_REG001,
** Reference ID       : LIN_DUD_ACT_018_REG002, LIN_DUD_ACT_018_REG003,
** Reference ID       : LIN_DUD_ACT_018_REG004, LIN_DUD_ACT_018_REG005,
** Reference ID       : LIN_DUD_ACT_018_REG006,
** Reference ID       : LIN_DUD_ACT_018_CRT001, LIN_DUD_ACT_018_CRT002,
** Reference ID       : LIN_DUD_ACT_018_CRT003, LIN_DUD_ACT_018_CRT004
***********************************************************************************************************************/
#define LIN_START_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
STATIC FUNC(void, LIN_CODE_FAST) Lin_HwProcessData(const uint8 LucChannel)
{
  volatile P2CONST(Lin_ChannelInfo, AUTOMATIC, LIN_CONST) LpLinChannelPtr;
  volatile P2VAR(Lin_RamData, AUTOMATIC, LIN_VAR_FAST_NO_INIT) LpRamVars;                                               /* PRQA S 3432 # JV-01 */
  volatile P2VAR(uint8, AUTOMATIC, LIN_VAR_FAST_NO_INIT) volatile LpStartDataBuffer;                                    /* PRQA S 3432 # JV-01 */
  Lin_FrameResponseType LenFrameType;
  uint8 LucIndex;
  uint8 LucDataLength;

  volatile P2CONST(Lin3_ChannelConfigType, AUTOMATIC, LIN_CONST) LpLin3ChannelPtr;
  volatile P2VAR(RLin3_UartRegs, AUTOMATIC, REGSPACE) LpLN3ChannelRegs;                                                 /* PRQA S 3432 # JV-01 */

  /* Get the pointer to requested channel configuration */
  LpLinChannelPtr = &Lin_GpChannelConfig[LucChannel];                                                                   /* PRQA S 0404 # JV-01 */

  /* Get the pointer to channel RAM data */
  LpRamVars = &Lin_GpChannelRamData[LucChannel];                                                                        /* PRQA S 0404 # JV-01 */

  LpLin3ChannelPtr = &Lin_GpRLIN3Properties[LpLinChannelPtr->ucPropertiesIndex];                                        /* PRQA S 0404, 2814, 2844 # JV-01, JV-02, JV-01 */

  /* Get the RLIN3 Channel Base Address */
  LpLN3ChannelRegs = LpLin3ChannelPtr->pLn3ChanlBaseAddress;                                                            /* PRQA S 2844, 2814 # JV-01, JV-02 */

  /* Copy the data length to local variable */
  LucDataLength = LpRamVars->ucFrameLength;                                                                             /* PRQA S 2844, 2814 # JV-01, JV-02 */

  /* Check if the transmitted frame is for go-to-sleep command */
  if (LIN_SET_SLEEP_REQUEST == LpRamVars->ucSlpRqst_RespRdy)
  {

    /* Clear the sleep requested bit */
    LpRamVars->ucSlpRqst_RespRdy = LIN_CLR_SLEEP_REQUEST;

    /* Check for Frame Transmit Successfully */
    if (LIN_FRAME_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_FRAME_COMPLETE))                               /* PRQA S 2844, 2814 # JV-01, JV-01 */
    {
      /* Clear the transmit interrupt */
      LpLN3ChannelRegs->ucRLN3nLST = (LpLN3ChannelRegs->ucRLN3nLST & LIN_TRANSMIT_DONE);
    } /* else No action required */

    #if (LIN_WAKEUP_SUPPORT == STD_ON)
    /* Check if channel specific wake-up support is enabled */
    if (LIN_TRUE == LpLinChannelPtr->blWakeupSupport)
    {
      /* Set RLIN3 to Slave Wake-up */
      (void)Lin_HwHandleWakeupRequest(LucChannel, LIN_SLAVE_ISSUE, LIN_GO_TO_SLEEP_SID);
    }
    else
    #endif /* #if (LIN_WAKEUP_SUPPORT == STD_ON) */
    {
      /* Enter the protection area */
      LIN_ENTER_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);

      /* Disable RLIN3 Interrupt */
      Lin_HwScheduleInterrupt(LucChannel, LIN_INTERRUPT_DIS);

      /* Exit the protection area */
      LIN_EXIT_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);
    }
  }
  /* If the transmitted frame is not a sleep frame */
  else
  {
    /* Get the frame type */
    LenFrameType = LpRamVars->enFrameType;

    switch (LenFrameType)
    {
    /* Frame is of type LIN_SLAVE_RESPONSE */
    case LIN_SLAVE_RESPONSE:
      /* If LATIT is generated for Header part in Rx Mode */
      if (LIN_RX_BUSY == LpRamVars->enChannelStatus)
      {
        if (LIN_HEADER_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_HEADER_COMPLETE))
        {
          /* Clear the header interrupt */
          LpLN3ChannelRegs->ucRLN3nLST = (LpLN3ChannelRegs->ucRLN3nLST & LIN_HEADER_DONE);
          /* Update channel status to LIN_RX_NO_RESPONSE */
          LpRamVars->enChannelStatus = LIN_RX_NO_RESPONSE;
        } /* else No action required */
      } /* else No action required */
      /* If LATIR is generated for response part in Rx Mode */
      if (LIN_RX_NO_RESPONSE == LpRamVars->enChannelStatus)
      {
        if (LIN_RESPONSE_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_RESPONSE_COMPLETE))
        {
          /* Clear the reception interrupt */
          LpLN3ChannelRegs->ucRLN3nLST = (LpLN3ChannelRegs->ucRLN3nLST & LIN_RECEPTION_DONE);

          LpStartDataBuffer = &LpRamVars->aaFrameData[(uint32)LIN_ZERO];

          /* Copy data to Tx Buffer */
          for (LucIndex = LIN_ZERO; LucDataLength > LucIndex; LucIndex++)
          {
            *LpStartDataBuffer = LpLN3ChannelRegs->ucRLN3nLDBR[LucIndex];                                               /* PRQA S 0404, 2844 # JV-01, JV-01 */

            LpStartDataBuffer++;                                                                                        /* PRQA S 3387 # JV-01 */
          }
          /* Update channel status to LIN_RX_OK */
          LpRamVars->enChannelStatus = LIN_RX_OK;
        } /* else No action required */
      } /* else No action required */
      break;
    /* Frame is of type LIN_MASTER_RESPONSE */
    case LIN_MASTER_RESPONSE:
      /* Check for Frame Transmit Successfully */
      if (LIN_FRAME_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_FRAME_COMPLETE))
      {
        /* Clear the transmit interrupt */
        LpLN3ChannelRegs->ucRLN3nLST = (LpLN3ChannelRegs->ucRLN3nLST & LIN_TRANSMIT_DONE);
        /* Update channel status to LIN_TX_OK */
        LpRamVars->enChannelStatus = LIN_TX_OK;
      } /* else No action required */
      break;
    /* Frame is of type LIN_SLAVE_TO_SLAVE */
    case LIN_SLAVE_TO_SLAVE:
      if (LIN_HEADER_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_HEADER_COMPLETE))
      {
        /* Clear the header interrupt */
        LpLN3ChannelRegs->ucRLN3nLST = (LpLN3ChannelRegs->ucRLN3nLST & LIN_HEADER_DONE);
        /* Set the channel status to LIN_TX_OK */
        LpRamVars->enChannelStatus = LIN_TX_OK;
      } /* else No action required */
      if (LIN_RESPONSE_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_RESPONSE_COMPLETE))
      {
        /* Clear the reception interrupt */
        LpLN3ChannelRegs->ucRLN3nLST = (LpLN3ChannelRegs->ucRLN3nLST & LIN_RECEPTION_DONE);
      } /* else No action required */
      break;
    default:
      /* No action required */
      break;
    }

    /* Enter the protection area */
    LIN_ENTER_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);

    /* Disable RLIN3 Interrupt */
    Lin_HwScheduleInterrupt(LucChannel, LIN_INTERRUPT_DIS);

    /* Exit the protection area */
    LIN_EXIT_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);
  }
}
#define LIN_STOP_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name      : Lin_HwErrorProcessing
**
** Service ID         : NA
**
** Description        : This service processes the status interrupt requests
**
** Sync/Async         : Synchronous
**
** Re-entrancy        : Non Re-entrant for the same channel
**
** Input Parameters   : uint8 LucChannel
**
** Output Parameters  : None
**
** Return parameter   : None
**
** Preconditions      : None
**
** Global Variables   : Lin_GpChannelConfig, Lin_GpChannelRamData,
**                      Lin_GpRLIN3Properties
**
** Functions invoked  : LIN_DEM_REPORT_ERROR,
**                      Lin_HwHandleWakeupRequest,
**                      Lin_HwScheduleInterrupt
**                      LIN_ENTER_CRITICAL_SECTION, LIN_EXIT_CRITICAL_SECTION
**
** Registers Used     : ucRLN3nLEST, ucRLN3nLST
**
** Reference ID       : LIN_DUD_ACT_019, LIN_DUD_ACT_019_ERR001,
** Reference ID       : LIN_DUD_ACT_019_ERR002, LIN_DUD_ACT_019_GBL001,
** Reference ID       : LIN_DUD_ACT_019_ERR003, LIN_DUD_ACT_019_ERR004,
** Reference ID       : LIN_DUD_ACT_019_GBL002, LIN_DUD_ACT_019_GBL003,
** Reference ID       : LIN_DUD_ACT_019_GBL004, LIN_DUD_ACT_019_GBL005,
** Reference ID       : LIN_DUD_ACT_019_GBL006, LIN_DUD_ACT_019_GBL007,
** Reference ID       : LIN_DUD_ACT_019_GBL008, LIN_DUD_ACT_019_GBL009,
** Reference ID       : LIN_DUD_ACT_019_GBL010, LIN_DUD_ACT_019_GBL011,
** Reference ID       : LIN_DUD_ACT_019_GBL012, LIN_DUD_ACT_019_GBL013,
** Reference ID       : LIN_DUD_ACT_019_GBL014, LIN_DUD_ACT_019_GBL015,
** Reference ID       : LIN_DUD_ACT_019_GBL016, LIN_DUD_ACT_019_GBL017,
** Reference ID       : LIN_DUD_ACT_019_REG001, LIN_DUD_ACT_019_REG002,
** Reference ID       : LIN_DUD_ACT_019_REG003, LIN_DUD_ACT_019_REG004,
** Reference ID       : LIN_DUD_ACT_019_REG005, LIN_DUD_ACT_019_REG006,
** Reference ID       : LIN_DUD_ACT_019_REG007, LIN_DUD_ACT_019_REG008,
** Reference ID       : LIN_DUD_ACT_019_REG009, LIN_DUD_ACT_019_REG010,
** Reference ID       : LIN_DUD_ACT_019_REG011, LIN_DUD_ACT_019_REG012,
** Reference ID       : LIN_DUD_ACT_019_REG013, LIN_DUD_ACT_019_REG014,
** Reference ID       : LIN_DUD_ACT_019_REG015,
** Reference ID       : LIN_DUD_ACT_019_CRT001, LIN_DUD_ACT_019_CRT002,
** Reference ID       : LIN_DUD_ACT_019_CRT003, LIN_DUD_ACT_019_CRT004,
** Reference ID       : LIN_DUD_ACT_019_CRT005, LIN_DUD_ACT_019_CRT006,
** Reference ID       : LIN_DUD_ACT_019_CRT007, LIN_DUD_ACT_019_CRT008
***********************************************************************************************************************/
#define LIN_START_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
STATIC FUNC(void, LIN_CODE_FAST) Lin_HwErrorProcessing(const uint8 LucChannel)
{
  volatile P2CONST(Lin_ChannelInfo, AUTOMATIC, LIN_CONST) LpLinChannelPtr;
  volatile P2CONST(Lin3_ChannelConfigType, AUTOMATIC, LIN_CONST) LpLin3ChannelPtr;
  volatile P2VAR(RLin3_UartRegs, AUTOMATIC, REGSPACE) LpLN3ChannelRegs;                                                 /* PRQA S 3432 # JV-01 */
  volatile P2VAR(Lin_RamData, AUTOMATIC, LIN_VAR_FAST_NO_INIT) LpRamVars;                                               /* PRQA S 3432 # JV-01 */

  Lin_FrameResponseType LenFrameType;

  /* Get the pointer to requested channel configuration */
  LpLinChannelPtr = &Lin_GpChannelConfig[LucChannel];                                                                   /* PRQA S 0404 # JV-01 */

  /* Get the pointer to channel RAM data */
  LpRamVars = &Lin_GpChannelRamData[LucChannel];                                                                        /* PRQA S 0404 # JV-01 */

  /* Get the channel index */
  LpLin3ChannelPtr = &Lin_GpRLIN3Properties[LpLinChannelPtr->ucPropertiesIndex];                                        /* PRQA S 0404, 2844, 2814 # JV-01, JV-01, JV-02 */

  /* Get the RLIN3 Channel Base Address */
  LpLN3ChannelRegs = LpLin3ChannelPtr->pLn3ChanlBaseAddress;                                                            /* PRQA S 2814, 2844 # JV-02, JV-01 */

  /* Get frame type */
  LenFrameType = LpRamVars->enFrameType;                                                                                /* PRQA S 2844, 2814 # JV-01, JV-02 */

  /* Check for any error occurred */
  if (LIN_ERROR_OCCURED == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_ERROR_OCCURED))                                   /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    /* Check if the transmitted frame is for go-to-sleep command */
    if (LIN_SET_SLEEP_REQUEST == LpRamVars->ucSlpRqst_RespRdy)
    {
      /* LIN_CH_SLEEP:                                                       */
      /* Lin_GetStatus() has already been called after calling Lin_GoToSleep */
      /* or                                                                  */
      /* LIN_CH_SLEEP_PENDING:                                               */
      /* Lin_GetStatus() has not been called yet after calling Lin_GoToSleep */
      /* The status of channel must not be changed while issuing the         */
      /* go-to-sleep command even if some errors have occured.               */
      if ((LIN_CH_SLEEP == LpRamVars->enChannelStatus) || (LIN_TRUE == LpRamVars->blSleepPending))
      {
        /* Clear the sleep requested bit */
        LpRamVars->ucSlpRqst_RespRdy = LIN_CLR_SLEEP_REQUEST;

        if (LIN_HEADER_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_HEADER_COMPLETE))
        {
          /* Clear the header interrupt */
          LpLN3ChannelRegs->ucRLN3nLST = (LpLN3ChannelRegs->ucRLN3nLST & LIN_HEADER_DONE);
        } /* else No action required */

        /* Clear all error flags and the ERR bit of the RLN3nLST register is automatically cleared */
        LpLN3ChannelRegs->ucRLN3nLEST = LIN_RESET_VALUE;

        #if (LIN_WAKEUP_SUPPORT == STD_ON)
        /* Check if channel specific wake-up support is enabled */
        if (LIN_TRUE == LpLinChannelPtr->blWakeupSupport)
        {
          /* Set RLIN3 to Slave Wake-up */
          (void)Lin_HwHandleWakeupRequest(LucChannel, LIN_SLAVE_ISSUE, LIN_GO_TO_SLEEP_SID);
        }
        else
        #endif /* #if (LIN_WAKEUP_SUPPORT == STD_ON) */
        {

          /* Enter the protection area */
          LIN_ENTER_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);

          /* Disable RLIN3 Interrupt */
          Lin_HwScheduleInterrupt(LucChannel, LIN_INTERRUPT_DIS);

          /* Exit the protection area */
          LIN_EXIT_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);
        }
      } /* else No action required */
    }
    /* Check if an error was occurred in Lin_Wakeup() */
    else if (LIN_TRUE == LpRamVars->blWakeupCalled)
    {
      /* Reset the LinIf wake-up flag */
      LpRamVars->blWakeupCalled = LIN_FALSE;

      /* Check for Bit error */
      if (LIN_BIT_ERROR == (uint8)(LpLN3ChannelRegs->ucRLN3nLEST & LIN_BIT_ERROR))
      {
        /* Update channel status to LIN_TX_HEADER_ERROR */
        LpRamVars->enChannelStatus = LIN_TX_HEADER_ERROR;

        /* Clear BER bit */
        LpLN3ChannelRegs->ucRLN3nLEST = (LpLN3ChannelRegs->ucRLN3nLEST & LIN_BIT_CLEAR);
      } /* else No action required */

      /* Check for Header Transmission error */
      if (LIN_PBUS_ERROR == (uint8)(LpLN3ChannelRegs->ucRLN3nLEST & LIN_PBUS_ERROR))
      {
        /* Update channel status to LIN_TX_HEADER_ERROR */
        LpRamVars->enChannelStatus = LIN_TX_HEADER_ERROR;

        /* Clear PBER bit */
        LpLN3ChannelRegs->ucRLN3nLEST = (LpLN3ChannelRegs->ucRLN3nLEST & LIN_PBUS_CLEAR);
      } /* else No action required */
      /* Enter the protection area */
      LIN_ENTER_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);

      /* Disable RLIN3 Interrupt */
      Lin_HwScheduleInterrupt(LucChannel, LIN_INTERRUPT_DIS);

      /* Exit the protection area */
      LIN_EXIT_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);
    }
    /* Other than go-to-sleep command and Lin_Wakeup(), Lin_SendFrame() with LIN_SLAVE_TO_SLAVE */
    else if (LIN_SLAVE_TO_SLAVE == LenFrameType)
    {
      /* An error was occurred in response part */
      if (LIN_HEADER_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_HEADER_COMPLETE))
      {
        /* Clear the header interrupt */
        LpLN3ChannelRegs->ucRLN3nLST = (LpLN3ChannelRegs->ucRLN3nLST & LIN_HEADER_DONE);

        /* Set the channel status to LIN_TX_OK */
        LpRamVars->enChannelStatus = LIN_TX_OK;

        /* Clear all error flags and the ERR bit of the RLN3nLST register is automatically cleared */
        LpLN3ChannelRegs->ucRLN3nLEST = LIN_RESET_VALUE;
      }
      /* an error was occurred in header part */
      else
      {
        /* Check for Bit error */
        if (LIN_BIT_ERROR == (uint8)(LpLN3ChannelRegs->ucRLN3nLEST & LIN_BIT_ERROR))
        {
          /* Update channel status to LIN_TX_HEADER_ERROR */
          LpRamVars->enChannelStatus = LIN_TX_HEADER_ERROR;

          /* Clear BER bit */
          LpLN3ChannelRegs->ucRLN3nLEST = (LpLN3ChannelRegs->ucRLN3nLEST & LIN_BIT_CLEAR);
        } /* else No action required */

        /* Check for Header Transmission error */
        if (LIN_PBUS_ERROR == (uint8)(LpLN3ChannelRegs->ucRLN3nLEST & LIN_PBUS_ERROR))
        {
          /* Update channel status to LIN_TX_HEADER_ERROR */
          LpRamVars->enChannelStatus = LIN_TX_HEADER_ERROR;

          /* Clear PBER bit */
          LpLN3ChannelRegs->ucRLN3nLEST = (LpLN3ChannelRegs->ucRLN3nLEST & LIN_PBUS_CLEAR);
        } /* else No action required */

        /* Check for any Time out error */
        if (LIN_TIMEOUT_ERROR == (uint8)(LpLN3ChannelRegs->ucRLN3nLEST & LIN_TIMEOUT_ERROR))
        {
          /* Update channel status to LIN_TX_HEADER_ERROR */
          LpRamVars->enChannelStatus = LIN_TX_HEADER_ERROR;

          #ifdef LIN_E_TIMEOUT_FAILURE
          /* Report Error to DEM */
          LIN_DEM_REPORT_ERROR(LIN_E_TIMEOUT_FAILURE, DEM_EVENT_STATUS_FAILED);
          #endif /* #ifdef LIN_E_TIMEOUT_FAILURE */

          /* Clear FTER bit */
          LpLN3ChannelRegs->ucRLN3nLEST = (LpLN3ChannelRegs->ucRLN3nLEST & LIN_TIMEOUT_CLEAR);
        } /* else No action required */
      }

      /* Enter the protection area */
      LIN_ENTER_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);

      /* Disable RLIN3 Interrupt */
      Lin_HwScheduleInterrupt(LucChannel, LIN_INTERRUPT_DIS);

      /* Exit the protection area */
      LIN_EXIT_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);
    }
    /* Lin_SendFrame() with LIN_MASTER_RESPONSE and LIN_SLAVE_RESPONSE */
    else
    {
      /* Check for any checksum field error */
      if (LIN_CHECKSUM_ERROR == (uint8)(LpLN3ChannelRegs->ucRLN3nLEST & LIN_CHECKSUM_ERROR))
      {
        /* Update channel status to LIN_RX_ERROR */
        LpRamVars->enChannelStatus = LIN_RX_ERROR;

        /* Clear CKE bit */
        LpLN3ChannelRegs->ucRLN3nLEST = (LpLN3ChannelRegs->ucRLN3nLEST & LIN_CHECKSUM_CLEAR);
      } /* else No action required */

      /* Check for any Time out error */
      if (LIN_TIMEOUT_ERROR == (uint8)(LpLN3ChannelRegs->ucRLN3nLEST & LIN_TIMEOUT_ERROR))
      {
        /* Check if header frame transmit successfully */
        if (LIN_HEADER_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_HEADER_COMPLETE))
        {
          if (LIN_MASTER_RESPONSE == LenFrameType)
          {
            /* An error was occurred in response transmission part */
            /* Update channel status to LIN_TX_ERROR */
            LpRamVars->enChannelStatus = LIN_TX_ERROR;
          }
          /* Frame type is of LIN_SLAVE_RESPONSE */
          else
          {
            /* An error was occurred in response reception part */
            /* Check if at least 1 response byte has been received */
            if (LIN_DATA1RX_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_DATA1RX_COMPLETE))
            {
              /* Update channel status to LIN_RX_ERROR */
              LpRamVars->enChannelStatus = LIN_RX_ERROR;
            }
            /* If no response byte has been received */
            else
            {
              /* Update channel status to LIN_RX_NO_RESPONSE */
              LpRamVars->enChannelStatus = LIN_RX_NO_RESPONSE;
            }
          }
        }
        /* An error was occurred in header part */
        else
        {
          /* Update channel status to LIN_TX_HEADER_ERROR */
          LpRamVars->enChannelStatus = LIN_TX_HEADER_ERROR;
        }
        #ifdef LIN_E_TIMEOUT_FAILURE
        /* Report Error to DEM */
        LIN_DEM_REPORT_ERROR(LIN_E_TIMEOUT_FAILURE, DEM_EVENT_STATUS_FAILED);
        #endif /* #ifdef LIN_E_TIMEOUT_FAILURE */

        /* Clear FTER bit */
        LpLN3ChannelRegs->ucRLN3nLEST = (LpLN3ChannelRegs->ucRLN3nLEST & LIN_TIMEOUT_CLEAR);
      } /* else No action required */

      /* Check for Bit error */
      if (LIN_BIT_ERROR == (uint8)(LpLN3ChannelRegs->ucRLN3nLEST & LIN_BIT_ERROR))
      {
        /* Check if header frame transmit successfully */
        if (LIN_HEADER_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_HEADER_COMPLETE))
        {
          /* Update channel status to LIN_TX_ERROR */
          LpRamVars->enChannelStatus = LIN_TX_ERROR;
        }
        /* Check if header frame transmit not successful */
        else
        {
          /* Update channel status to LIN_TX_HEADER_ERROR */
          LpRamVars->enChannelStatus = LIN_TX_HEADER_ERROR;
        }

        /* Clear BER bit */
        LpLN3ChannelRegs->ucRLN3nLEST = (LpLN3ChannelRegs->ucRLN3nLEST & LIN_BIT_CLEAR);
      } /* else No action required */

      /* Check for Header Transmission error */
      if (LIN_PBUS_ERROR == (uint8)(LpLN3ChannelRegs->ucRLN3nLEST & LIN_PBUS_ERROR))
      {
        /* Update channel status to LIN_TX_HEADER_ERROR */
        LpRamVars->enChannelStatus = LIN_TX_HEADER_ERROR;

        /* Clear PBER bit */
        LpLN3ChannelRegs->ucRLN3nLEST = (LpLN3ChannelRegs->ucRLN3nLEST & LIN_PBUS_CLEAR);
      } /* else No action required */

      /* Check for any framing error */
      if (LIN_FRAMING_ERROR == (uint8)(LpLN3ChannelRegs->ucRLN3nLEST & LIN_FRAMING_ERROR))
      {
        /* Update channel status to LIN_RX_ERROR */
        LpRamVars->enChannelStatus = LIN_RX_ERROR;

        /* Clear CKE bit */
        LpLN3ChannelRegs->ucRLN3nLEST = (LpLN3ChannelRegs->ucRLN3nLEST & LIN_FRAMING_CLEAR);
      } /* else No action required */

      if (LIN_HEADER_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_HEADER_COMPLETE))
      {
        /* Clear the header interrupt */
        LpLN3ChannelRegs->ucRLN3nLST = (LpLN3ChannelRegs->ucRLN3nLST & LIN_HEADER_DONE);
      } /* else No action required */

      /* Enter the protection area */
      LIN_ENTER_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);

      /* Disable RLIN3 Interrupt */
      Lin_HwScheduleInterrupt(LucChannel, LIN_INTERRUPT_DIS);

      /* Exit the protection area */
      LIN_EXIT_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);
    }
  } /* else No action required */
}
#define LIN_STOP_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name      : Lin_HwLoadTxBuffer
**
** Service ID         : NA
**
** Description        : This service copies data to Tx buffer
**
** Sync/Async         : Synchronous
**
** Re-entrancy        : Non Re-entrant for the same channel
**
** Input Parameters   : LucChannel
**
** Output Parameters  : None
**
** Return parameter   : None
**
** Preconditions      : None
**
** Global Variables   : Lin_GpChannelConfig, Lin_GpChannelRamData,
**                      Lin_GpRLIN3Properties
**
** Functions invoked  : None
**
** Registers Used     : ucRLN3nLDBR
**
** Reference ID       : LIN_DUD_ACT_020, LIN_DUD_ACT_020_REG001
***********************************************************************************************************************/
#define LIN_START_SEC_PRIVATE_CODE
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
STATIC FUNC(void, LIN_PRIVATE_CODE) Lin_HwLoadTxBuffer(const uint8 LucChannel)
{
  volatile P2CONST(Lin_ChannelInfo, AUTOMATIC, LIN_PRIVATE_CONST) LpLinChannelPtr;
  volatile P2CONST(Lin3_ChannelConfigType, AUTOMATIC, LIN_PRIVATE_CONST) LpLin3ChannelPtr;
  volatile P2VAR(RLin3_UartRegs, AUTOMATIC, REGSPACE) LpLN3ChannelRegs;                                                 /* PRQA S 3432 # JV-01 */
  volatile P2VAR(Lin_RamData, AUTOMATIC, LIN_VAR_FAST_NO_INIT) LpRamVars;                                               /* PRQA S 3432, 3678 # JV-01, JV-01 */
  volatile P2VAR(uint8, AUTOMATIC, LIN_VAR_FAST_NO_INIT) volatile LpStartDataBuffer;                                    /* PRQA S 3432, 3678 # JV-01, JV-01 */
  uint8 LucIncrement;
  uint8 LucDataLength;

  /* Get the pointer to frame data */
  LpRamVars = &Lin_GpChannelRamData[LucChannel];                                                                        /* PRQA S 0404 # JV-01 */

  /* Copy the data length to local variable */
  LucDataLength = LpRamVars->ucFrameLength;                                                                             /* PRQA S 2814, 2844 # JV-02, JV-01 */

  LpStartDataBuffer = &LpRamVars->aaFrameData[(uint32)LIN_ZERO];

  /* Get the address of LIN Channel structure */
  LpLinChannelPtr = &Lin_GpChannelConfig[LucChannel];                                                                   /* PRQA S 0404 # JV-01 */

  LpLin3ChannelPtr = &Lin_GpRLIN3Properties[LpLinChannelPtr->ucPropertiesIndex];                                        /* PRQA S 0404, 2814, 2844 # JV-01, JV-02, JV-01 */

  /* Get the RLIN3 Channel Base Address */
  LpLN3ChannelRegs = LpLin3ChannelPtr->pLn3ChanlBaseAddress;                                                            /* PRQA S 2844, 2814 # JV-01, JV-02 */

  /* Copy data to Tx Buffer */
  for (LucIncrement = LIN_ZERO; LucDataLength > LucIncrement; LucIncrement++)
  {
    LpLN3ChannelRegs->ucRLN3nLDBR[LucIncrement] = *LpStartDataBuffer;                                                   /* PRQA S 0404, 2814, 2844 # JV-01, JV-01, JV-01 */

    LpStartDataBuffer++;                                                                                                /* PRQA S 3387 # JV-01 */
  }
}
#define LIN_STOP_SEC_PRIVATE_CODE
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name      : Lin_HwWakeupOnGoing
**
** Service ID         : NA
**
** Description        : This service does wake-up processing
**
** Sync/Async         : Synchronous
**
** Re-entrancy        : Non Re-entrant for the same channel
**
** Input Parameters   : LucChannel
**
** Output Parameters  : None
**
** Return parameter   : None
**
** Preconditions      : None
**
** Global Variables   : Lin_GpChannelConfig, Lin_GpChannelRamData,
**                      Lin_GpRLIN3Properties
**
** Functions invoked  : Lin_HwScheduleInterrupt
**                      LIN_ENTER_CRITICAL_SECTION, LIN_EXIT_CRITICAL_SECTION
**
** Registers Used     : ucRLN3nLST
**
** Reference ID       : LIN_DUD_ACT_021, LIN_DUD_ACT_021_GBL001,
** Reference ID       : LIN_DUD_ACT_021_GBL002, LIN_DUD_ACT_021_REG001,
** Reference ID       : LIN_DUD_ACT_021_CRT001, LIN_DUD_ACT_021_CRT002
***********************************************************************************************************************/
#define LIN_START_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
STATIC FUNC(void, LIN_CODE_FAST) Lin_HwWakeupOnGoing(const uint8 LucChannel)
{
  volatile P2CONST(Lin_ChannelInfo, AUTOMATIC, LIN_CONST) LpLinChannelPtr;
  volatile P2CONST(Lin3_ChannelConfigType, AUTOMATIC, LIN_CONST) LpLin3ChannelPtr;
  volatile P2VAR(RLin3_UartRegs, AUTOMATIC, REGSPACE) LpLN3ChannelRegs;                                                 /* PRQA S 3432 # JV-01 */
  volatile P2VAR(Lin_RamData, AUTOMATIC, LIN_VAR_FAST_NO_INIT) LpRamVars;                                               /* PRQA S 3432 # JV-01 */

  /* Get the address of LIN Channel structure */
  LpLinChannelPtr = &Lin_GpChannelConfig[LucChannel];                                                                   /* PRQA S 0404 # JV-01 */

  /* Get the pointer to channel RAM data */
  LpRamVars = &Lin_GpChannelRamData[LucChannel];                                                                        /* PRQA S 0404 # JV-01 */

  LpLin3ChannelPtr = &Lin_GpRLIN3Properties[LpLinChannelPtr->ucPropertiesIndex];                                        /* PRQA S 0404, 2844, 2814 # JV-01, JV-01, JV-02 */

  /* Get the RLIN3 Channel Base Address */
  LpLN3ChannelRegs = LpLin3ChannelPtr->pLn3ChanlBaseAddress;                                                            /* PRQA S 2844, 2814 # JV-01, JV-02 */

  /* Check for Wakeup Transmit Successfully */
  if (LIN_FRAME_COMPLETE == (uint8)(LpLN3ChannelRegs->ucRLN3nLST & LIN_FRAME_COMPLETE))                                 /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    /* Clear the transmit interrupt */
    LpLN3ChannelRegs->ucRLN3nLST = (LpLN3ChannelRegs->ucRLN3nLST & LIN_TRANSMIT_DONE);

    /* Set the channel status */
    LpRamVars->enChannelStatus = LIN_OPERATIONAL;                                                                       /* PRQA S 2844, 2814 # JV-01, JV-02 */
  }
  else
  {
    /* Reset the LinIf wake-up flag */
    LpRamVars->blWakeupCalled = LIN_FALSE;
  }

  /* Enter the protection area */
  LIN_ENTER_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);

  /* Disable RLIN3 Interrupt */
  Lin_HwScheduleInterrupt(LucChannel, LIN_INTERRUPT_DIS);

  /* Exit the protection area */
  LIN_EXIT_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);
}
#define LIN_STOP_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name      : Lin_HwInit
**
** Service ID         : None
**
** Description        : This service initializes all configured RLIN3 channels.
**
** Sync/Async         : Synchronous
**
** Re-entrancy        : Non-Reentrant
**
** Input Parameters   : LucChannel
**
** InOut Parameters   : None
**
** Output Parameters  : None
**
** Return Parameter   : Std_ReturnType
**
** Preconditions      : None
**
** Global Variables   : Lin_GpChannelConfig,
**                      Lin_GpRLIN3Properties
**
** Functions invoked  : Lin_HwSetMasterMode, Lin_HwScheduleInterrupt
**
** Registers Used     : ucRLN3nLWBR,
**                      ucRLN3nLBRP0,
**                      ucRLN3nLBRP1,
**                      ucRLN3nLMD,
**                      ucRLN3nLBFC,
**                      ucRLN3nLSC,
**                      ucRLN3nLIE,
**                      ucRLN3nLEDE,
**                      ucRLN3nLDFC
**
** Reference ID       : LIN_DUD_ACT_022, LIN_DUD_ACT_022_REG001,
** Reference ID       : LIN_DUD_ACT_022_REG002, LIN_DUD_ACT_022_REG003,
** Reference ID       : LIN_DUD_ACT_022_REG004, LIN_DUD_ACT_022_REG005,
** Reference ID       : LIN_DUD_ACT_022_REG006, LIN_DUD_ACT_022_REG007,
** Reference ID       : LIN_DUD_ACT_022_REG008, LIN_DUD_ACT_022_REG009,
** Reference ID       : LIN_DUD_ACT_022_REG010, LIN_DUD_ACT_022_REG011,
** Reference ID       : LIN_DUD_ACT_022_REG012
***********************************************************************************************************************/
#define LIN_START_SEC_PRIVATE_CODE
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
FUNC(Std_ReturnType, LIN_PRIVATE_CODE) Lin_HwInit(const uint8 LucChannel)                                               /* PRQA S 1532 # JV-01 */
{
  volatile P2CONST(Lin_ChannelInfo, AUTOMATIC, LIN_PRIVATE_CONST) LpLinChannelPtr;
  volatile P2CONST(Lin3_ChannelConfigType, AUTOMATIC, LIN_PRIVATE_CONST) LpLin3ChannelPtr;
  volatile P2VAR(RLin3_UartRegs, AUTOMATIC, REGSPACE) LpLN3ChannelRegs;                                                 /* PRQA S 3432 # JV-01 */
  #if (LIN_INTERRUPT_MUX_SUPPORT == STD_ON)
  volatile P2VAR(uint16, AUTOMATIC, LIN_VAR_FAST_NO_INIT) LpIntMuxAddress;                                              /* PRQA S 3432 # JV-01 */
  #endif

  uint8 LucLWBR;
  #if (LIN_INTERRUPT_MUX_SUPPORT == STD_ON)
  uint8 LucIntMuxShift;
  uint16 LusLn3IntMuxMask;
  #endif

  Std_ReturnType LucReturnValue;

  /* Initialize the return value */
  LucReturnValue = E_NOT_OK;                                                                                            /* PRQA S 2982 # JV-01 */

  /* Get the address of Channel info structure */
  LpLinChannelPtr = &Lin_GpChannelConfig[LucChannel];                                                                   /* PRQA S 0404 # JV-01 */

  /* Get the address of RLIN3 Channel structure */
  LpLin3ChannelPtr = &Lin_GpRLIN3Properties[LpLinChannelPtr->ucPropertiesIndex];                                        /* PRQA S 0404, 2814, 2844 # JV-01, JV-02, JV-01 */

  /* Get the pointer to RLIN3 Channel Base Address */
  LpLN3ChannelRegs = LpLin3ChannelPtr->pLn3ChanlBaseAddress;                                                            /* PRQA S 2844, 2814 # JV-01, JV-02 */

  /* Set RLIN3 to Reset Mode */
  LucReturnValue = Lin_HwSetMasterMode(LucChannel, RLIN3_RESET_MODE, LIN_INIT_SID);

  /* Set Prescaler Clock Select */
  LucLWBR = (uint8)(LpLin3ChannelPtr->ucPrescalerClk_Select << LIN_ONE);

  /* Set Value for number of samples in 1 Bit time period */
  LucLWBR |= (uint8)(LpLin3ChannelPtr->ucBitSamples << LIN_FOUR);

  if (E_OK == LucReturnValue)
  {
    /* Check for LIN protocol specification 1.3 */
    if (LIN_FALSE == LpLin3ChannelPtr->blLinSpec_Select)
    {
      /* Set Baud Rate Selector register for LIN specification 1.3 */
      LpLN3ChannelRegs->ucRLN3nLWBR = LucLWBR & LIN_SET_WAKEUP_BAUDRATE_SPEC_13;                                        /* PRQA S 2814, 2844 # JV-01, JV-01 */
    }
    else
    {
      /* Set Baud Rate Selector register for LIN specification 2.x */
      LpLN3ChannelRegs->ucRLN3nLWBR = LucLWBR | LIN_SET_WAKEUP_BAUDRATE_SPEC_21;
    }
    #if (LIN_INTERRUPT_MUX_SUPPORT == STD_ON)
    if (NULL_PTR != LpLin3ChannelPtr->pLin3IntMuxAddress)
    {
      /* Get the address of Interrupt Select register */
      LpIntMuxAddress = LpLin3ChannelPtr->pLin3IntMuxAddress;

      /* Calculate position of MUXn in Interrupt Select register */
      LucIntMuxShift = LpLin3ChannelPtr->ucLin3IntMuxNumber * LIN_FIVE;

      /* Set the mask before writing value to Interrupt Select register */
      LusLn3IntMuxMask = (uint16)(LIN_INTSEL_MUX_MASK << LucIntMuxShift);                                               /* PRQA S 4391, 4399 # JV-01, JV-01 */

      /* Set Interrupt Select register for LIN channel*/
      *(LpIntMuxAddress) = (uint16)(*(LpIntMuxAddress) & (~LusLn3IntMuxMask)) |                                         /* PRQA S 2844, 2814, 0404 # JV-01, JV-02, JV-01 */
        (uint16)(LpLin3ChannelPtr->ucLin3IntMuxValue << LucIntMuxShift);                                                /* PRQA S 4391, 4399, 0404 # JV-01, JV-01, JV-01 */
    } /* else No action required */
    #endif

    /* Set Baud Rate Prescaler0 register */
    LpLN3ChannelRegs->ucRLN3nLBRP0 = LpLin3ChannelPtr->ucBaudRate;

    /* Set Baud Rate Prescaler1 register */
    LpLN3ChannelRegs->ucRLN3nLBRP1 = LIN_RESET_VALUE;

    /* Enable LIN & Disable UART Interrupt Enable Register */
    /* Rx/Tx/Status Interrupt are used */
    LpLN3ChannelRegs->ucRLN3nLMD = RLIN3_INTERRUPT_REG;

    /* Set System Clock */
    LpLN3ChannelRegs->ucRLN3nLMD = (LpLN3ChannelRegs->ucRLN3nLMD | LpLinChannelPtr->ucModReg);                          /* PRQA S 0404 # JV-01 */
    /* Break Field Configure */
    LpLN3ChannelRegs->ucRLN3nLBFC = LpLinChannelPtr->ucRLINBreakfieldwidth;

    /* Set Inter-byte Header & Response Space width */
    LpLN3ChannelRegs->ucRLN3nLSC = LpLinChannelPtr->ucRLINInterbytespace;

    /* Set RLIN3 Module to LIN Master mode */
    LpLN3ChannelRegs->ucRLN3nLMD = (LpLN3ChannelRegs->ucRLN3nLMD & RLIN3_MASTER_MODE);

    /* Set Interrupt Enable Register */
    LpLN3ChannelRegs->ucRLN3nLIE = LIN_ENABLE_INTERRUPT_REG;

    /* Set Error Detection Enable Register */
    LpLN3ChannelRegs->ucRLN3nLEDE = LIN_ERROR_INTERRUPT;

    /* Set LIN for Frame Mode */
    LpLN3ChannelRegs->ucRLN3nLDFC = (LpLN3ChannelRegs->ucRLN3nLDFC & LIN_FRAME_MODE);

    #if (LIN_AR_VERSION == LIN_AR_422_VERSION)
    /* Set RLIN3 to Operational Mode */
    LucReturnValue = Lin_HwSetMasterMode(LucChannel, RLIN3_OPERATION_MODE, LIN_INIT_SID);
    #elif (LIN_AR_VERSION == LIN_AR_431_VERSION)
    LucReturnValue = Lin_HwSetMasterMode(LucChannel, RLIN3_WAKEUP_MODE, LIN_INIT_SID);
    #endif /* #if (LIN_AR_VERSION == LIN_AR_422_VERSION) */
  } /* else No action required */

  /* Disable RLIN3 Interrupt */
  Lin_HwScheduleInterrupt(LucChannel, LIN_INTERRUPT_DIS);

  return (LucReturnValue);
}
#define LIN_STOP_SEC_PRIVATE_CODE
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name      : Lin_HwSetMasterMode
**
** Service ID         : NA
**
** Description        : This service sets RLIN3 to the corresponding
**                      Master Mode.
**                      It initiates a transition to the requested Master Mode.
**
** Sync/Async         : Synchronous
**
** Re-entrancy        : Non Re-entrant for the same channel
**
** Input Parameters   : LucChannel, LenModeType, LucApiId
**
** Output Parameters  : None
**
** Return parameter   : Std_ReturnType
**
** Preconditions      : None
**
** Global Variables   : Lin_GpChannelConfig, Lin_GpRLIN3Properties
**
** Functions invoked  : Lin_ErrorReport, Lin_HwScheduleInterrupt
**
** Registers Used     : ucRLN3nLCUC, ucRLN3nLEST
**                      ucRLN3nLMST, pLin3IntTxEicReg,
**                      pLin3IntRxEicReg,
**                      pLin3IntStEicReg
**
** Reference ID       : LIN_DUD_ACT_023, LIN_DUD_ACT_023_ERR001,
** Reference ID       : LIN_DUD_ACT_023_REG001, LIN_DUD_ACT_023_REG002,
** Reference ID       : LIN_DUD_ACT_023_REG003, LIN_DUD_ACT_023_REG004,
** Reference ID       : LIN_DUD_ACT_023_REG005, LIN_DUD_ACT_023_REG006,
** Reference ID       : LIN_DUD_ACT_023_REG007
***********************************************************************************************************************/
#define LIN_START_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
FUNC(Std_ReturnType, LIN_CODE_FAST)
Lin_HwSetMasterMode(const uint8 LucChannel, const Lin_ModeType LenModeType, const uint8 LucApiId)                       /* PRQA S 3206 # JV-01 */
{
  volatile P2CONST(Lin_ChannelInfo, AUTOMATIC, LIN_CONST) LpLinChannelPtr;
  volatile P2CONST(Lin3_ChannelConfigType, AUTOMATIC, LIN_CONST) LpLin3ChannelPtr;
  volatile P2VAR(RLin3_UartRegs, AUTOMATIC, REGSPACE) LpLN3ChannelRegs;                                                 /* PRQA S 3432 # JV-01 */

  Std_ReturnType LucReturnValue;
  boolean LblIntEnabling;
  volatile uint32 LulTimeoutCounter;

  /* Initialize the return value */
  LucReturnValue = E_NOT_OK;                                                                                            /* PRQA S 2982 # JV-01 */

  /* Get the address of Channel info structure */
  LpLinChannelPtr = &Lin_GpChannelConfig[LucChannel];                                                                   /* PRQA S 0404 # JV-01 */

  /* Get the address of RLIN3 Channel structure */
  LpLin3ChannelPtr = &Lin_GpRLIN3Properties[LpLinChannelPtr->ucPropertiesIndex];                                        /* PRQA S 0404, 2814, 2844 # JV-01, JV-02, JV-01 */

  /* Get the pointer to RLIN3 Channel Base Address */
  LpLN3ChannelRegs = LpLin3ChannelPtr->pLn3ChanlBaseAddress;                                                            /* PRQA S 2844, 2814 # JV-01, JV-02 */

  /* Set max value of loop counter */
  LulTimeoutCounter = (uint32)LIN_TIMEOUT_DURATION;                                                                     /* PRQA S 1281 # JV-01 */
  /*  Check if Interrupt processing is disabled */
  if (LIN_EIC_EIMK_MASK == ((*(uint8 *)LpLin3ChannelPtr->pLin3IntTxEicReg) & LIN_EIC_EIMK_MASK))                        /* PRQA S 0751, 0312 # JV-01, JV-01 */
  {
    /* Flag to know that interrupt is not enabled before */
    LblIntEnabling = LIN_FALSE;
  }
  else
  {
    /* Temporary disable interrupt */
    Lin_HwScheduleInterrupt(LucChannel, LIN_INTERRUPT_DIS);
    /* Flag to know that interrupt is enabled before */
    LblIntEnabling = LIN_TRUE;
  }
  /* Change RLIN3 to the specified LucMode mode */
  switch (LenModeType)
  {
  case RLIN3_RESET_MODE:
    /* Set RLIN3 to Reset Mode */
    LpLN3ChannelRegs->ucRLN3nLCUC = LIN_RESET_VALUE;                                                                    /* PRQA S 2844, 2814 # JV-01, JV-02 */

    if ((uint32)LIN_ZERO == LulTimeoutCounter)                                                                          /* PRQA S 3416 # JV-01 */
    {
      if (LIN_RESET_VALUE == (uint8)(LpLN3ChannelRegs->ucRLN3nLMST & LIN_SET_VALUE))
      {
        LulTimeoutCounter = (uint32)LIN_ONE;
      } /* else No action required */
    }
    else
    {
      /* After writing to bit[0], further write access to this register is not allowed until bit LMST[0] is 0 */
      while ((LIN_SET_VALUE ==                                                                                          /* PRQA S 3416, 3415 # JV-01, JV-01 */
              (uint8)(LpLN3ChannelRegs->ucRLN3nLMST & LIN_SET_VALUE)) && ((uint32)LIN_ZERO < LulTimeoutCounter))
      {
        LulTimeoutCounter--;                                                                                            /* PRQA S 3384, 3387 # JV-01, JV-01 */
      }
    }
    break;
  case RLIN3_OPERATION_MODE:
    /* Set RLIN3 to Normal Mode*/
    LpLN3ChannelRegs->ucRLN3nLCUC = LIN_SET_OPERATION_MODE;

    if ((uint32)LIN_ZERO == LulTimeoutCounter)                                                                          /* PRQA S 3416 # JV-01 */
    {
      if (LIN_SET_OPERATION_MODE == (uint8)(LpLN3ChannelRegs->ucRLN3nLMST & LIN_SET_OPERATION_MODE))
      {
        LulTimeoutCounter = (uint32)LIN_ONE;
      } /* else No action required */
    }
    else
    {
      /* After writing to bit[0], further write access to this register is not allowed until bit LMST[0] is 0 */
      while ((LIN_SET_OPERATION_MODE !=                                                                                 /* PRQA S 3416, 3415 # JV-01, JV-01 */
              (uint8)(LpLN3ChannelRegs->ucRLN3nLMST & LIN_SET_OPERATION_MODE)) &&
             ((uint32)LIN_ZERO < LulTimeoutCounter))
      {
        LulTimeoutCounter--;                                                                                            /* PRQA S 3384, 3387 # JV-01, JV-01 */
      }
    }
    break;
  case RLIN3_WAKEUP_MODE:
    /* Set RLIN3 to Wake-up Mode*/
    LpLN3ChannelRegs->ucRLN3nLCUC = LIN_SET_VALUE;

    if ((uint32)LIN_ZERO == LulTimeoutCounter)                                                                          /* PRQA S 3416 # JV-01 */
    {
      if (LIN_SET_VALUE == (uint8)(LpLN3ChannelRegs->ucRLN3nLMST & LIN_SET_OPERATION_MODE))
      {
        LulTimeoutCounter = (uint32)LIN_ONE;
      } /* else No action required */
    }
    else
    {
      /* After writing 1 to bit[0], further write access to this register
         is not allowed until bit LMST[0] is 1 and bit LMST[1] is 0*/
      while ((LIN_SET_VALUE !=                                                                                          /* PRQA S 3415, 3416 # JV-01, JV-01 */
              (uint8)(LpLN3ChannelRegs->ucRLN3nLMST & LIN_SET_OPERATION_MODE)) &&
             ((uint32)LIN_ZERO < LulTimeoutCounter))
      {
        LulTimeoutCounter--;                                                                                            /* PRQA S 3384, 3387 # JV-01, JV-01 */
      }
    }
    break;
  default:
    /* No action required */
    break;
  }

  #if defined(LIN_E_TIMEOUT) || (LIN_DEV_ERROR_DETECT == STD_ON)
  if ((uint32)LIN_ZERO == LulTimeoutCounter)                                                                            /* PRQA S 3416 # JV-01 */
  {
    /* Report Error to DEM or DET */
    LucReturnValue = Lin_ErrorReport(LucApiId);
  }
  else
  #endif /* LIN_E_TIMEOUT || (LIN_DEV_ERROR_DETECT == STD_ON) */
  {
    LucReturnValue = E_OK;
  }

  /*  Clear Timeout error flag */
  LpLN3ChannelRegs->ucRLN3nLEST = (LpLN3ChannelRegs->ucRLN3nLEST & LIN_TIMEOUT_CLEAR);

  /*  Clear pending interrupt */
  RH850_SV_MODE_ICR_AND(16, LpLin3ChannelPtr->pLin3IntTxEicReg, LIN_EIC_PENDING_CLEAR);                                 /* PRQA S 0404, 3464 # JV-01, JV-01 */
  RH850_SV_MODE_ICR_AND(16, LpLin3ChannelPtr->pLin3IntRxEicReg, LIN_EIC_PENDING_CLEAR);                                 /* PRQA S 0404, 3464 # JV-01, JV-01 */
  RH850_SV_MODE_ICR_AND(16, LpLin3ChannelPtr->pLin3IntStEicReg, LIN_EIC_PENDING_CLEAR);                                 /* PRQA S 0404, 3464 # JV-01, JV-01 */
  /* Dummy read */
  RH850_SV_MODE_REG_READ_ONLY(16, LpLin3ChannelPtr->pLin3IntStEicReg);
  /* SYNCP execution */
  EXECUTE_SYNCP();

  /*  Check if Interrupt processing is disabled  */
  if (LIN_TRUE == LblIntEnabling)
  {
    /* Re-enable interrupt processing */
    Lin_HwScheduleInterrupt(LucChannel, LIN_INTERRUPT_EN);
  } /* else No action required */

  return (LucReturnValue);
}
#define LIN_STOP_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name      : Lin_HwScheduleInterrupt
**
** Service ID         : None
**
** Description        : This service enables or disables the interrupts
**                      (transmission, reception and error status) of the
**                      requested RLIN3 channel, using the EIMK bit of the
**                      EIC register.
**
** Sync/Async         : Synchronous
**
** Re-entrancy        : Non-Reentrant for the same channel
**
** Input Parameters   : LucChannel, LenInterruptScheduleType
**
** InOut Parameters   : None
**
** Output Parameters  : None
**
** Return Parameter   : None
**
** Preconditions      : None
**
** Global Variables   : Lin_GpChannelConfig, Lin_GpRLIN3Properties
**
** Functions invoked  : None
**
** Registers Used     : pLin3IntTxEicReg,
**                      pLin3IntRxEicReg,
*                       pLin3IntStEicReg
**
** Reference ID       : LIN_DUD_ACT_024, LIN_DUD_ACT_024_REG001,
** Reference ID       : LIN_DUD_ACT_024_REG002, LIN_DUD_ACT_024_REG003,
** Reference ID       : LIN_DUD_ACT_024_REG004, LIN_DUD_ACT_024_REG005,
** Reference ID       : LIN_DUD_ACT_024_REG006
***********************************************************************************************************************/
#define LIN_START_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
FUNC(void, LIN_CODE_FAST)
Lin_HwScheduleInterrupt(const uint8 LucChannel, const Lin_InterruptScheduleType LenInterruptScheduleType)
{
  volatile P2CONST(Lin_ChannelInfo, AUTOMATIC, LIN_CONST) LpLinChannelPtr;
  volatile P2CONST(Lin3_ChannelConfigType, AUTOMATIC, LIN_CONST) LpLin3ChannelPtr;

  /* Get the address of Channel info structure */
  LpLinChannelPtr = &Lin_GpChannelConfig[LucChannel];                                                                   /* PRQA S 0404 # JV-01 */

  /* Get the address of RLIN3 Channel structure */
  LpLin3ChannelPtr = &Lin_GpRLIN3Properties[LpLinChannelPtr->ucPropertiesIndex];                                        /* PRQA S 0404, 2844, 2814 # JV-01, JV-01, JV-02 */

  switch (LenInterruptScheduleType)
  {
  /* Disable RLIN3 Tx/Rx/St Interrupts */
  case LIN_INTERRUPT_DIS:
    /* Disable Interrupts, set the EIMKn bit of the EICn register */
    RH850_SV_MODE_ICR_OR(8, LpLin3ChannelPtr->pLin3IntTxEicReg, LIN_EIC_EIMK_MASK);                                     /* PRQA S 0751, 2814, 2844, 0404, 3464 # JV-01, JV-01, JV-01, JV-01, JV-01 */
    RH850_SV_MODE_ICR_OR(8, LpLin3ChannelPtr->pLin3IntRxEicReg, LIN_EIC_EIMK_MASK);                                     /* PRQA S 0404, 0751, 3464 # JV-01, JV-01, JV-01 */
    RH850_SV_MODE_ICR_OR(8, LpLin3ChannelPtr->pLin3IntStEicReg, LIN_EIC_EIMK_MASK);                                     /* PRQA S 0751, 0404, 3464 # JV-01, JV-01, JV-01 */
    /* Dummy read */
    RH850_SV_MODE_REG_READ_ONLY(8, LpLin3ChannelPtr->pLin3IntStEicReg);                                                 /* PRQA S 0751 # JV-01 */
    /* SYNCP execution */
    EXECUTE_SYNCP();
    break;
  /* Enable RLIN3 Tx/Rx/St Interrupts */
  case LIN_INTERRUPT_EN:
    /* Enable Interrupts, clear the EIMKn bit of the EICn register and clear pending interrupts */
    RH850_SV_MODE_ICR_AND(8, LpLin3ChannelPtr->pLin3IntTxEicReg, LIN_EIC_EIMK_MASK_CLEAR);                              /* PRQA S 0404, 3464, 0751 # JV-01, JV-01, JV-01 */
    RH850_SV_MODE_ICR_AND(8, LpLin3ChannelPtr->pLin3IntRxEicReg, LIN_EIC_EIMK_MASK_CLEAR);                              /* PRQA S 0404, 3464, 0751 # JV-01, JV-01, JV-01 */
    RH850_SV_MODE_ICR_AND(8, LpLin3ChannelPtr->pLin3IntStEicReg, LIN_EIC_EIMK_MASK_CLEAR);                              /* PRQA S 0404, 3464, 0751 # JV-01, JV-01, JV-01 */
    /* Dummy read */
    RH850_SV_MODE_REG_READ_ONLY(8, LpLin3ChannelPtr->pLin3IntStEicReg);                                                 /* PRQA S 0751 # JV-01 */
    /* SYNCP execution */
    EXECUTE_SYNCP();
    break;
  default:
    /* No action required */
    break;
  }
}
#define LIN_STOP_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name      : Lin_HwHandleWakeupRequest
**
** Service ID         : NA
**
** Sync/Async         : Synchronous
**
** Description        : This service enables a wake-up detection from the
**                      LIN bus or issues a wake-up command to the LIN bus
**                      according to the wake-up requested.
**
** Re-entrancy        : Non Re-entrant for the same channel
**
** Input Parameters   : LucChannel, LenWakeupType, LucApiId
**
** Output Parameters  : None
**
** Return parameter   : Std_ReturnType
**
** Preconditions      : None
**
** Global Variables   : Lin_GpChannelConfig, Lin_GpRLIN3Properties
**
** Functions invoked  : Lin_HwSetMasterMode, Lin_HwScheduleInterrupt
**                      LIN_ENTER_CRITICAL_SECTION, LIN_EXIT_CRITICAL_SECTION
**
** Registers Used     : ucRLN3nLWUP,
**                      ucRLN3nLDFC,
**                      ucRLN3nLTRC
**
** Reference ID       : LIN_DUD_ACT_025, LIN_DUD_ACT_025_REG001,
** Reference ID       : LIN_DUD_ACT_025_REG002, LIN_DUD_ACT_025_REG003,
** Reference ID       : LIN_DUD_ACT_025_REG004,
** Reference ID       : LIN_DUD_ACT_025_CRT001, LIN_DUD_ACT_025_CRT002,
** Reference ID       : LIN_DUD_ACT_025_CRT003, LIN_DUD_ACT_025_CRT004
***********************************************************************************************************************/
#define LIN_START_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
FUNC(Std_ReturnType, LIN_CODE_FAST)
Lin_HwHandleWakeupRequest(const uint8 LucChannel, const Lin_WakeupType LenWakeupType, const uint8 LucApiId)             /* PRQA S 1532 # JV-01 */
{
  volatile P2CONST(Lin_ChannelInfo, AUTOMATIC, LIN_CONST) LpLinChannelPtr;
  volatile P2CONST(Lin3_ChannelConfigType, AUTOMATIC, LIN_CONST) LpLin3ChannelPtr;
  volatile P2VAR(RLin3_UartRegs, AUTOMATIC, REGSPACE) LpLN3ChannelRegs;                                                 /* PRQA S 3432 # JV-01 */
  Std_ReturnType LucReturnValue;

  /* Initialize the return value */
  LucReturnValue = E_NOT_OK;                                                                                            /* PRQA S 2982 # JV-01 */

  /* Get the address of Channel info structure */
  LpLinChannelPtr = &Lin_GpChannelConfig[LucChannel];                                                                   /* PRQA S 0404 # JV-01 */

  /* Get the address of RLIN3 Channel structure */
  LpLin3ChannelPtr = &Lin_GpRLIN3Properties[LpLinChannelPtr->ucPropertiesIndex];                                        /* PRQA S 0404, 2844, 2814 # JV-01, JV-01, JV-02 */

  /* Get the pointer to RLIN3 Channel Base Address */
  LpLN3ChannelRegs = LpLin3ChannelPtr->pLn3ChanlBaseAddress;                                                            /* PRQA S 2844, 2814 # JV-01, JV-02 */

  /* Enter the protection area */
  LIN_ENTER_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);

  /* Set RLIN3 to Reset Mode */
  LucReturnValue = Lin_HwSetMasterMode(LucChannel, RLIN3_RESET_MODE, LucApiId);

  if (E_OK == LucReturnValue)
  {

    /* Set RLIN3 Wake-up Low width */
    LpLN3ChannelRegs->ucRLN3nLWUP = LIN_LOW_WIDTH;                                                                      /* PRQA S 2844, 2814 # JV-01, JV-02 */

    switch (LenWakeupType)
    {
    case LIN_SLAVE_ISSUE:
      /* Set the Direction of RLIN3 as Reception */
      LpLN3ChannelRegs->ucRLN3nLDFC = (LpLN3ChannelRegs->ucRLN3nLDFC & LIN_SLAVE_DIRECTION);
      break;
    case LIN_MASTER_ISSUE:
      /* Set the Direction of RLIN3 as Transmit */
      LpLN3ChannelRegs->ucRLN3nLDFC = (LpLN3ChannelRegs->ucRLN3nLDFC | LIN_MASTER_DIRECTION);
      break;
    default:
      /* No action required */
      break;
    }

    /* Set RLIN3 to Wake-up Mode */
    LucReturnValue = Lin_HwSetMasterMode(LucChannel, RLIN3_WAKEUP_MODE, LucApiId);
  } /* else No action required */

  /* Exit the protection area */
  LIN_EXIT_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);

  if (E_OK == LucReturnValue)
  {
    /* Enter the protection area */
    LIN_ENTER_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);

    /* Enable RLIN3 Interrupt */
    Lin_HwScheduleInterrupt(LucChannel, LIN_INTERRUPT_EN);

    /* Initiate Communication */
    LpLN3ChannelRegs->ucRLN3nLTRC = (LpLN3ChannelRegs->ucRLN3nLTRC | LIN_START_TRANSMIT);

    /* Exit the protection area */
    LIN_EXIT_CRITICAL_SECTION(LIN_INTERRUPT_CONTROL_PROTECTION);
  } /* else No action required */
  return (LucReturnValue);
}
#define LIN_STOP_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name      : Lin_ErrorReport
**
** Service ID         : NA
**
** Sync/Async         : Synchronous
**
** Description        : This service reports an error to DEM or DET if a
**                      loop-processing-time-out error occurs
**
** Re-entrancy        : Re-entrant
**
** Input Parameters   : LucApiId
**
** Output Parameters  : None
**
** Return parameter   : Std_ReturnType
**
** Preconditions      : None
**
** Global Variables   : None
**
** Functions invoked  : Det_ReportError, LIN_DEM_REPORT_ERROR
**
** Registers Used     : None
**
** Reference ID       : LIN_DUD_ACT_026, LIN_DUD_ACT_026_ERR001,
** Reference ID       : LIN_DUD_ACT_026_ERR002, LIN_DUD_ACT_026_ERR003
***********************************************************************************************************************/
#define LIN_START_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
#if defined(LIN_E_TIMEOUT) || (LIN_DEV_ERROR_DETECT == STD_ON)
STATIC FUNC(Std_ReturnType, LIN_CODE_FAST) Lin_ErrorReport(const uint8 LucApiId)                                        /* PRQA S 3206 # JV-01 */
{
  Std_ReturnType LucReturnValue;

  /* Initialize the return value */
  LucReturnValue = E_OK;                                                                                                /* PRQA S 2982 # JV-01 */

  #ifdef LIN_E_TIMEOUT
    /* Report Error to DEM */
    LIN_DEM_REPORT_ERROR(LIN_E_TIMEOUT, DEM_EVENT_STATUS_FAILED);
    LucReturnValue = E_NOT_OK;
  #else
  #if (LIN_DEV_ERROR_DETECT == STD_ON)
    /* Report to DET */
    (void)Det_ReportError(LIN_MODULE_ID, LIN_INSTANCE_ID, LucApiId, LIN_E_DET_TIMEOUT);
    LucReturnValue = E_NOT_OK;
  #endif /* (LIN_DEV_ERROR_DETECT == STD_ON) */
  #endif /* LIN_E_TIMEOUT */

  return (LucReturnValue);
}
#endif /* LIN_E_TIMEOUT || (LIN_DEV_ERROR_DETECT == STD_ON) */
#define LIN_STOP_SEC_CODE_FAST
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name      : Lin_SetStatus
**
** Service ID         : None
**
** Description        : This function updates the status of Lin_GblDriverState.
**                      The purpose of this function is to prevent the order of
**                      instructions being changed by the compiler.
**
** Sync/Async         : Synchronous
**
** Re-entrancy        : Non-Reentrant
**
** Input Parameters   : LblStatus
**
** InOut Parameters   : None
**
** Output Parameters  : None
**
** Return Parameter   : None
**
** Preconditions      : None
**
** Global Variables   : Lin_GblDriverState
**
** Functions invoked  : None
**
** Registers Used     : None
**
** Reference ID       : LIN_DUD_ACT_027, LIN_DUD_ACT_027_GBL001
***********************************************************************************************************************/
#define LIN_START_SEC_PRIVATE_CODE
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
FUNC(void, LIN_PRIVATE_CODE) Lin_SetStatus(const boolean LblStatus)                                                     /* PRQA S 1532 # JV-01 */
{
  /* Set Lin_GblDriverState to LblStatus */
  Lin_GblDriverState = LblStatus;
}
#define LIN_STOP_SEC_PRIVATE_CODE
#include "Lin_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
**                                                    End of File                                                     **
***********************************************************************************************************************/
