/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Eth_ETNC_Irq.c                                                                                      */
/* SW-VERSION   = 1.6.0                                                                                               */
/*====================================================================================================================*/
/*                                                COPYRIGHT                                                           */
/*====================================================================================================================*/
/* (c) 2020,2021 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* This file contains Interrupt Service Routine for  Eth Driver Component.                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s)        */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs     */
/* of program errors, compliance with applicable laws, damage to or loss of data, programs or equipment,              */
/* and unavailability or interruption of operations.                                                                  */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*                                      Devices:        X2x                                                           */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                          Revision Control History                                                  **
***********************************************************************************************************************/
/*
 * 1.4.2: 24/08/2021    : Modify the format to 120 characters
 * 1.4.1: 07/05/2021    : Remove vendor ID from the file name, API names
 *                        and parameters according to BSW00347 requirement
 *        09/06/2021    : Add QAC 9.5.0 comment.
 *        06/07/2021    : Updated QA-C 9.5.0 comments.
 * 1.2.0: 13/07/2020    : Release
 *        29/07/2020    : Add QAC 9.3.1 comment.
 * 1.1.0: 19/06/2020    : Release
 * 1.0.1: 04/06/2020    : Update Reentrancy for ISR.
 *                        Changed include file structure.
 * 1.0.0: 25/03/2020    : Initial Version
 */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                        Include Section                                                             **
***********************************************************************************************************************/
/* Eth APIs header file */
#include "Eth.h"
#include "Eth_ETNC_Ram.h"
#include "Eth_ETNC_Irq.h"
#include "EthIf_Cbk.h"
#include "Dem.h"
#if (ETH_CRITICAL_SECTION_PROTECTION == STD_ON)
/* Included for the declaration of the critical section protection functions */
#include "SchM_Eth.h"
#endif
/***********************************************************************************************************************
**                                       Version Information                                                          **
***********************************************************************************************************************/

/* AUTOSAR release version information */
#define ETH_ETNC_IRQ_C_AR_RELEASE_MAJOR_VERSION ETH_AR_RELEASE_MAJOR_VERSION_VALUE
#define ETH_ETNC_IRQ_C_AR_RELEASE_MINOR_VERSION ETH_AR_RELEASE_MINOR_VERSION_VALUE
#define ETH_ETNC_IRQ_C_AR_RELEASE_REVISION_VERSION ETH_AR_RELEASE_REVISION_VERSION_VALUE

/* File version information */
#define ETH_ETNC_IRQ_C_SW_MAJOR_VERSION    ETH_SW_MAJOR_VERSION_VALUE
#define ETH_ETNC_IRQ_C_SW_MINOR_VERSION    ETH_SW_MINOR_VERSION_VALUE

/***********************************************************************************************************************
**                                     Version Check                                                                  **
***********************************************************************************************************************/

/* Functionality related to R4.0 */
#if (ETH_AR_RELEASE_MAJOR_VERSION != ETH_ETNC_IRQ_C_AR_RELEASE_MAJOR_VERSION)
#error "Eth_ETNC_Irq.c : Mismatch in Release Major Version"
#endif
#if (ETH_AR_RELEASE_MINOR_VERSION != ETH_ETNC_IRQ_C_AR_RELEASE_MINOR_VERSION)
#error "Eth_ETNC_Irq.c : Mismatch in Release Minor Version"
#endif
#if (ETH_AR_RELEASE_REVISION_VERSION != ETH_ETNC_IRQ_C_AR_RELEASE_REVISION_VERSION)
#error "Eth_ETNC_Irq.c : Mismatch in Release Revision Version"
#endif

#if (ETH_SW_MAJOR_VERSION != ETH_ETNC_IRQ_C_SW_MAJOR_VERSION)
#error "Eth_ETNC_Irq.c : Mismatch in Software Major Version"
#endif
#if (ETH_SW_MINOR_VERSION != ETH_ETNC_IRQ_C_SW_MINOR_VERSION)
#error "Eth_ETNC_Irq.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (2:3408)    : %s' has external linkage and is being defined without any previous declaration.              */
/* Rule                : CERTCCM DCL07, MISRA C:2012 Rule-8.4                                                         */
/* JV-01 Justification : It is accepted, due to the declaration will be taken care by Os                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1503)    : The function 'name' is defined but is not used within this project.                          */
/* Rule                : CERTCCM MSC07, MISRA C:2012 Rule-2.1                                                         */
/* JV-01 Justification : This is accepted, due to the module's API is exported for user's usage.                      */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : (4:5087) #include statements in a file should only be preceded by other preprocessor         */
/*                       directives or comments.                                                                      */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is accepted, due to implementation for include memmap is following AUTOSAR rule.        */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1505)    : The function 'name' is only referenced in the translation unit where it is defined.          */
/* Rule                : CERTCCM DCL19, MISRA C:2012 Rule-8.7                                                         */
/* JV-01 Justification : This is accepted, due to following coding rule, internal function can be defined in          */
/*                       other C source files                                                                         */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3383)    : Cannot identify wraparound guard for unsigned arithmetic expression.                         */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : Wraparound can't be occur in this unsigned integer arithmetic so there is no need to add     */
/*                       a wraparound guard here.                                                                     */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3469)    : This usage of a function-like macro looks like it could be replaced by an equivalent         */
/*                       function call.                                                                               */
/* Rule                : MISRA C:2012 Dir-4.9                                                                         */
/* JV-01 Justification : To prevent overhead and speed up the processing, there is no need to call a function         */
/*                       here for such a small operation.                                                             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2934)    : Possible: Computing an invalid pointer value.                                                */
/* Rule                : CERTCCM ARR30, ARR37, ARR38, EXP08                                                           */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                             Global Data                                                            **
***********************************************************************************************************************/

/***********************************************************************************************************************
** Function Name         : ETH_ETNC0_ISR
**
** Service ID            : NA
**
** Description           : Interrupt Service Routine for port 0
**
** Sync/Async            : NA
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETNCEICRegs, Eth_GaaETNCRegs
**
** Function(s) invoked   : Eth_DemConfigCheck,
**                         Eth_HwRxIsrHandler, Eth_HwTxIsrHandler
**
** Registers Used        : EIC(INTETNCn), ETNCnEESR
**
** Reference ID          : ETH_DUD_ACT_151, ETH_DUD_ACT_151_ERR001
***********************************************************************************************************************/
#define ETH_START_SEC_CODE_FAST
#include "Eth_MemMap.h"

#if ((ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON) || \
  (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON))

#if defined (Os_ETH_ETNC0_CAT2_ISR) || (ETH_ISR_CATEGORY_2 == STD_ON)
/* Defines the CAT2 interrupt mapping */
ISR(ETH_ETNC0_CAT2_ISR)                                                                                                 /* PRQA S 3408, 1503 # JV-01, JV-01 */
#else
/* Defines the CAT1 interrupt mapping */
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_ETNC0_ISR(void)                                                               /* PRQA S 1503 # JV-01 */
#endif
{
  #if (ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if ((((uint16)ETH_EIC_CONSISTENCY_MASK & (uint16)~ETH_EIC_EIMK_MASK) !=
  (*Eth_GaaETNCEICRegs[0] & (uint16)ETH_EIC_CONSISTENCY_MASK)) ||                                                       /* PRQA S 2814 # JV-01 */
    (0UL == (Eth_GaaETNCRegs[0]->ulEESR & (0UL                                                                          /* PRQA S 2814 # JV-01 */
  #if (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
      | ETH_ETNC_FR
  #endif
  #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON)
      | ETH_ETNC_TWB
  #endif
      ))))
  {
    Eth_DemConfigCheck(Eth_GaaDemEventIntInconsistent[0], DEM_EVENT_STATUS_FAILED);
  }
  else
  #endif
  {
    #if (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
    /* Frame Receive Flag*/
    if (0UL != (Eth_GaaETNCRegs[0]->ulEESR & ETH_ETNC_FR))
    {
      /* RX Handler */
      Eth_HwRxIsrHandler(0U);
    }
    else
    {
      /* No action required */
    }
    #endif

    #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON)
    /* Frame Transfer Complete Flag */
    if (0UL != (Eth_GaaETNCRegs[0]->ulEESR & ETH_ETNC_TWB))
    {
      /* TX Handler */
      Eth_HwTxIsrHandler(0U);
    }
    else
    {
      /* No action required */
    }
    #endif
  }
}

#endif

#define ETH_STOP_SEC_CODE_FAST
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name         : Eth_HwTxIsrHandler (ETNC)
**
** Service ID            : N/A
**
** Description           : Perform Tx ISR handling.
**                         Check completed frame and indicate TxConfirmation if
**                         necessary.
**                         - In interrupt mode
**                           Initiate a transmission of the next one frame if
**                           exists.
**                         - In polling mode
**                           Initiate a transmission of the last frame if not
**                           sent.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LulCtrlIdx     : controller index
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat, Eth_GaaTxDescriptor,
**                         Eth_GaaETNCRegs
**
** Function(s) invoked   : EthIf_TxConfirmation, Eth_ReleaseTxBuffer
**
** Registers Used        : ETNCnEDTRR, ETNCnEESR
**
** Reference ID          : ETH_DUD_ACT_152, ETH_DUD_ACT_152_CRT001,
** Reference ID          : ETH_DUD_ACT_152_CRT002, ETH_DUD_ACT_152_GBL001,
** Reference ID          : ETH_DUD_ACT_152_GBL002, ETH_DUD_ACT_152_GBL003,
** Reference ID          : ETH_DUD_ACT_152_GBL004, ETH_DUD_ACT_152_GBL005,
** Reference ID          : ETH_DUD_ACT_152_GBL006
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_HwTxIsrHandler(CONST(uint32, AUTOMATIC) LulCtrlIdx)                                    /* PRQA S 1505, 1503 # JV-01, JV-01 */
{
  uint32 LulDescIdx;
  uint32 LulBufIdx;
  uint32 LulDescIdxCompleted;
  uint32 LulDescIdxQueued;
  uint32 LulTimeoutCount;

  #if (ETH_REGISTER_CHECK_RUNTIME == STD_ON)
  uint32 LulResult;
  uint32 LulExpectedTD0;
  /* Initialize the register error check result (Error flag) */
  LulResult = 0UL;
  #endif

  /* Clear Tx Descriptor Write-Back Complete Flag */
  Eth_GaaETNCRegs[LulCtrlIdx]->ulEESR = ETH_ETNC_TWB;                                                                   /* PRQA S 2844, 2814 # JV-01, JV-01 */

  /* Synchronization should be considered when interrupt mode only */
  Eth_GaaETNCRegs[LulCtrlIdx]->ulEESR;
  EXECUTE_SYNCP();
  /* Wait for EDMAC to stop completely */
  LulTimeoutCount = 0UL;
  while ((0UL != Eth_GaaETNCRegs[LulCtrlIdx]->ulEDTRR) && (LulTimeoutCount <= ETH_TIMEOUT_COUNT))
  {
    LulTimeoutCount++;
    /* To stabilize the time of a loop, execute SYNCP for each loop */
    EXECUTE_SYNCP();
  }
  #if (ETH_REGISTER_CHECK_RUNTIME == STD_ON)
  /* Confirm EESR.TWB has been cleared */
  LulResult |= (Eth_GaaETNCRegs[LulCtrlIdx]->ulEESR & ETH_ETNC_TWB);
  /* Confirm EDTRR.TR has been cleared */
  LulResult |= Eth_GaaETNCRegs[LulCtrlIdx]->ulEDTRR;
  #endif

  /* Critical section to exclusive control between Eth_HwTransmit */
  ETH_ENTER_CRITICAL_SECTION(ETH_RAM_DATA_PROTECTION);

  /* All descriptors requested in the previous call should be completed */
  LulDescIdxCompleted =
    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.ucTxDescIdxRequested;                                                          /* PRQA S 2844 # JV-01 */
  /* If there are pending frames, set TACT bit and start transmission */
  LulDescIdx = LulDescIdxCompleted;
  LulDescIdxQueued = Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.ucTxDescIdxQueued;
  if (LulDescIdx != LulDescIdxQueued)
  {
    /* In interrupt mode, send 1 frame per 1 ISR. */
    /* Set TACT and set TWBI to enable completion interrupt. */
    Eth_GaaTxDescriptor[LulDescIdx].ulTD0 |= ETH_ETNC_TACT | ETH_ETNC_TWBI;                                             /* PRQA S 2844 # JV-01 */
    /* Increment the descriptor index */
    LulDescIdx = (LulDescIdx + 1UL) % (uint32)ETH_TX_BUF_TOTAL_0;                                                       /* PRQA S 3383 # JV-01 */
    /* Synchronize the memory and start the HW */
    EXECUTE_SYNCM();
    Eth_GaaETNCRegs[LulCtrlIdx]->ulEDTRR = ETH_ETNC_TR;
    /* Update the descriptor index */
    /* Since the maximum value of descriptor index is 254, casting to uint8 does no problem. */
    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.ucTxDescIdxRequested = (uint8)LulDescIdx;                                      /* PRQA S 2844 # JV-01 */
  }
  else
  {
    /* If there is no pending frame, clear the ongoing flag */
    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.blTxOngoing = ETH_FALSE;                                                       /* PRQA S 2844 # JV-01 */
  }
  /* Perform TxConfirmation for completed descriptors */
  LulDescIdx = Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.ucTxDescIdxConfirmed;                                               /* PRQA S 2844 # JV-01 */

  ETH_EXIT_CRITICAL_SECTION(ETH_RAM_DATA_PROTECTION);

  do
  {
    {
      #if (ETH_REGISTER_CHECK_RUNTIME == STD_ON)
      /* Confirm TxDescriptor has not been corrupted */
      if ((uint32)(ETH_TX_BUF_TOTAL_0 - 1UL) == LulDescIdx)
      {
        LulExpectedTD0 = ETH_ETNC_TDLE | ETH_ETNC_TFP_ALL;
      }
      else
      {
        LulExpectedTD0 = ETH_ETNC_TFP_ALL;
      }
      LulResult |=
        ((LulExpectedTD0 ^ Eth_GaaTxDescriptor[LulDescIdx].ulTD0) &                                                     /* PRQA S 2844 # JV-01 */
          ETH_ETNC_TD0_CHECK_MASK);
      #endif /* (ETH_REGISTER_CHECK_RUNTIME == STD_ON) */

      LulBufIdx = Eth_GaaTxDescriptor[LulDescIdx].ucBufIdx;
      /*
       * If this buffer was requested with TxConfirmation=true,
       * inform EthIf
       */
      if (0U != (Eth_GaaTxDescriptor[LulDescIdx].ucAttributes &
        ETH_ETNC_ATTR_CONFIRMATION))
      {
        /* Since the maximum value of controller index is 0, casting to uint8 does no problem. */
        EthIf_TxConfirmation((uint8)LulCtrlIdx, (Eth_BufIdxType)LulBufIdx);
      }
      else
      {
        /* No action required */
      }

      /* Release tx buffer */
      Eth_ReleaseTxBuffer(LulCtrlIdx, LulBufIdx);

      /* Increment the descriptor index */
      LulDescIdx = (LulDescIdx + 1UL) % (uint32)ETH_TX_BUF_TOTAL_0;                                                     /* PRQA S 3383 # JV-01 */
    }
  } while (LulDescIdx != LulDescIdxCompleted);
  /* Update the descriptor index */
  /* Since the maximum value of descriptor index is 254, casting to uint8 does no problem. */
  Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.ucTxDescIdxConfirmed = (uint8)LulDescIdx;                                        /* PRQA S 2844 # JV-01 */

  #if (ETH_REGISTER_CHECK_RUNTIME == STD_ON)
  /* If any object didn't have expected valuds, raise the error flag */
  if (0UL != LulResult)
  {
    /* Set the flag, DEM error will be reported at the next MainFunction */
    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.blErrDetected = ETH_TRUE;                                                      /* PRQA S 2844 # JV-01 */
  }
  else
  {
    /* No action required */
  }
  #endif /* (ETH_REGISTER_CHECK_RUNTIME == STD_ON) */

}

/***********************************************************************************************************************
** Function Name         : Eth_HwRxIsrHandler (ETNC)
**
** Service ID            : N/A
**
** Description           : Perform Rx ISR handling.
**                         When polling mode:
**                           Receive one frame and indicate it EthIf
**                         When interrupt mode:
**                           Receive all frames and indicate them to EthIf
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LulCtrlIdx     : controller index
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : When polling mode:
**                           ETH_NOT_RECEIVED : there was no received frame
**                           ETH_RECEIVED     : there was one received frame
**                           ETH_RECEIVED_MORE_DATA_AVAILABLE:
**                                 there were more than one received frames
**                         When interrupt mode:
**                           always returns ETH_RECEIVED
**
**                         Note that this function returns RECEIVED even if
**                         a received frame was discarded.
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat, Eth_GaaRxDescriptor,
**                         Eth_GaaETNCRegs
**
** Function(s) invoked   : Eth_RxIndication
**
** Registers Used        : ETNCnEDRRR, ETNCnEESR
**
** Reference ID          : ETH_DUD_ACT_153, ETH_DUD_ACT_153_GBL001,
** Reference ID          : ETH_DUD_ACT_153_GBL002, ETH_DUD_ACT_153_GBL003,
** Reference ID          : ETH_DUD_ACT_153_GBL004
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_HwRxIsrHandler(                                                                        /* PRQA S 1505, 1503 # JV-01, JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  uint32 LulDescIdx;
  uint32 LulRD0;
  uint32 LulFrameLenByte;
  P2VAR(volatile Eth_ETNCRxDescriptorType, AUTOMATIC, ETH_VAR_NO_INIT)
    LpDesc;
  #if (ETH_REGISTER_CHECK_RUNTIME == STD_ON)
  uint32 LulResult;
  uint32 LulExpectedRD0;

  /* Initialize the register error check result */
  LulResult = 0UL;
  #endif

  /* EESR.FR flag is referred when interrupt mode only */
  /* Clear Frame Receive Flag */
  Eth_GaaETNCRegs[0]->ulEESR = ETH_ETNC_FR;                                                                             /* PRQA S 2814 # JV-01 */

  /* Do dummy read and SYNCP */
  Eth_GaaETNCRegs[0]->ulEESR;
  EXECUTE_SYNCP();

  /* Load the global var to the local var to improve performance */
  LulDescIdx = Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.ucRxDescIdxHead;                                                    /* PRQA S 2844 # JV-01 */

  /* Scan received frames */
  LulRD0 = Eth_GaaRxDescriptor[LulDescIdx].ulRD0;                                                                       /* PRQA S 2844 # JV-01 */
  /* When interrupt mode, scan all received frames */
  while ((0UL == (LulRD0 & ETH_ETNC_RACT)) &&
    (ETH_ETNC_RFL_WORKAROUND_VALUE !=
      ETH_ETNC_RFL_GET(Eth_GaaRxDescriptor[LulDescIdx].ulRD1)))                                                         /* PRQA S 3469 # JV-01 */
  {
    LpDesc = &Eth_GaaRxDescriptor[LulDescIdx];                                                                          /* PRQA S 2934 # JV-01 */

    #if (ETH_REGISTER_CHECK_RUNTIME == STD_ON)
    /* Confirm RD0, RDLE has not be changed,
       RACT is cleared, RFP0, RFP1 are set by the HW */
    if (((uint32)ETH_RX_BUF_TOTAL_0 - 1UL) == LulDescIdx)
    {
      LulExpectedRD0 = ETH_ETNC_RDLE | ETH_ETNC_RFP1 | ETH_ETNC_RFP0;
    }
    else
    {
      LulExpectedRD0 = ETH_ETNC_RFP1 | ETH_ETNC_RFP0;
    }
    LulResult |= ((LulExpectedRD0 ^ LpDesc->ulRD0)                                                                      /* PRQA S 2844 # JV-01 */
      & ETH_ETNC_RD0_CHECK_MASK);
    #endif /* (ETH_REGISTER_CHECK_RUNTIME == STD_ON) */

    /* When the following error occurs, discard this frame:
       - CRC error
       - PHY-LSI receive error
       - Too long frame received
       - Too short frame received
       - Receive abort
       - Rx FIFO overflow
       The AUTOSAR only requires to discard a frame when CRC error occurs.
       However rx data is not available when the above errors occur,
       discard it also the above case. */
    if (0UL != (LulRD0 &
      (ETH_ETNC_RFS9_RFOF | ETH_ETNC_RFS8_RABT | ETH_ETNC_RFS3_RTLF |
        ETH_ETNC_RFS2_RTSF | ETH_ETNC_RFS1_PRE | ETH_ETNC_RFS0_CERF)))
    {
      /* If any error occurred, discard this frame */
    }
    else if ((0UL != (LulRD0 & ETH_ETNC_RFS7_RMAF))
    #if (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON)
      && (0UL == Eth_GaaCtrlStat[LulCtrlIdx].ulActiveFilterBits)                                                        /* PRQA S 2844 # JV-01 */
      && (ETH_FALSE == Eth_GaaCtrlStat[LulCtrlIdx].blPromiscuous)
    #endif
      )
    {
      /* Discard multicast frames when the filtering is off */
    }
    else
    {
      /* Indicate a frame to EthIf */
      LulFrameLenByte = ETH_ETNC_RFL_GET(LpDesc->ulRD1);                                                                /* PRQA S 3469 # JV-01 */
      /* The 2nd parameter is BufferIdx, it is same as DescIdx on Rx side */
      Eth_RxIndication(LulCtrlIdx, LulDescIdx, LulFrameLenByte);
    }
    /* Set 0xFFFF to RFL field as the workaround */
    LpDesc->ulRD1 =
      ETH_ETNC_RBL(ETH_INTERNAL_RX_BUFFER_BYTE) |                                                                       /* PRQA S 3469 # JV-01 */
      ETH_ETNC_RFL_WORKAROUND_VALUE;
    /* Set RACT bit to mark this descriptor is empty */
    LpDesc->ulRD0 = (LpDesc->ulRD0 & ~ETH_ETNC_RFS_MASK) | ETH_ETNC_RACT;
    /* Increment the ring pointer */
    LulDescIdx = (LulDescIdx + 1UL) % (uint32)ETH_RX_BUF_TOTAL_0;                                                       /* PRQA S 3383 # JV-01 */
    /* Load RD0 of the next descriptor */
    LulRD0 = Eth_GaaRxDescriptor[LulDescIdx].ulRD0;
  }

  /* Update the global var */
  /* Since the maximum value of descriptor index is 254, casting to uint8 does no problem. */
  Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.ucRxDescIdxHead = (uint8)LulDescIdx;                                             /* PRQA S 2844 # JV-01 */

  /* If descriptor overflow has been occurred, the receive operation is stopped.
     Set RR bit hear in case of such case. */
  Eth_GaaETNCRegs[LulCtrlIdx]->ulEDRRR = ETH_ETNC_RR;                                                                   /* PRQA S 2844 # JV-01 */


  #if (ETH_REGISTER_CHECK_RUNTIME == STD_ON)
  /* If any object didn't have expected values, set the error flag */
  if (0UL != LulResult)
  {
    /* Set the flag, DEM error will be reported at the next MainFunction */
    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.blErrDetected = ETH_TRUE;                                                      /* PRQA S 2844 # JV-01 */
  }
  else
  {
    /* No action required */
  }
  #endif /* (ETH_REGISTER_CHECK_RUNTIME == STD_ON) */
}

/***********************************************************************************************************************
**                                            End of File                                                             **
***********************************************************************************************************************/
