/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Icu_LLDriver.c                                                                                      */
/* SW-VERSION   = 1.5.0                                                                                               */
/*====================================================================================================================*/
/*                                                     COPYRIGHT                                                      */
/*====================================================================================================================*/
/* (c) 2020-2022 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* Define common Internal Function for Hardware Units.                                                                */
/*                                                                                                                    */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s) and/or */
/* the Application.                                                                                                   */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs of  */
/* program errors, compliance with applicable laws, damage to or loss of data, programs or equipment, and             */
/* unavailability or interruption of operations.                                                                      */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*              Devices:        X2x                                                                                   */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                              Revision Control History                                              **
***********************************************************************************************************************/
/*
 * 1.4.3:  09/05/2022  : Remove "else" statement when no action required.
 *         17/04/2022  : Merge QAC message from U2Ax, E2x and U2Bx
 * 1.4.1:  18/10/2021  : Add QAC message 9.5.0
 * 1.3.2:  30/08/2021  : Change from #include "Icu_MemMap.h" to #include "Icu_Mapping.h".
 * 1.3.1:  02/07/2021  : Add QAC message 9.5.0
 *                       Format source code to 120 characters
 *                       Improve Violation tag (remove START/END)
 * 1.2.0:  17/07/2020  : Release
 * 1.1.0:  19/06/2020  : Release
 * 1.0.0:  25/03/2020  : Initial Version
 */
/**********************************************************************************************************************/
/***********************************************************************************************************************
**                                             Margin for Control History                                             **
***********************************************************************************************************************/

/***********************************************************************************************************************
**                                                  Include Section                                                   **
***********************************************************************************************************************/

/* Included for macro definitions */
#include "Icu_LLDriver.h"                                                                                               /* PRQA S 0857 # JV-01 */

/***********************************************************************************************************************
**                                                Version Information                                                 **
***********************************************************************************************************************/
/* AUTOSAR Release version information */
#define ICU_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION    ICU_AR_RELEASE_MAJOR_VERSION_VALUE
#define ICU_LLDRIVER_C_AR_RELEASE_MINOR_VERSION    ICU_AR_RELEASE_MINOR_VERSION_VALUE
#define ICU_LLDRIVER_C_AR_RELEASE_REVISION_VERSION ICU_AR_RELEASE_REVISION_VERSION_VALUE
/* File version information */
#define ICU_LLDRIVER_C_SW_MAJOR_VERSION            ICU_SW_MAJOR_VERSION_VALUE
#define ICU_LLDRIVER_C_SW_MINOR_VERSION            ICU_SW_MINOR_VERSION_VALUE
/***********************************************************************************************************************
**                                                   Version Check                                                    **
***********************************************************************************************************************/
#if (ICU_LLDRIVER_AR_RELEASE_MAJOR_VERSION != ICU_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION)
  #error "Icu_LLDriver.c : Mismatch in Release Major Version"
#endif

#if (ICU_LLDRIVER_AR_RELEASE_MINOR_VERSION != ICU_LLDRIVER_C_AR_RELEASE_MINOR_VERSION)
  #error "Icu_LLDriver.c : Mismatch in Release Minor Version"
#endif

#if (ICU_LLDRIVER_AR_RELEASE_REVISION_VERSION != ICU_LLDRIVER_C_AR_RELEASE_REVISION_VERSION)
  #error "Icu_LLDriver.c : Mismatch in Release Revision Version"
#endif

#if (ICU_LLDRIVER_SW_MAJOR_VERSION != ICU_LLDRIVER_C_SW_MAJOR_VERSION)
  #error "Icu_LLDriver.c : Mismatch in Software Major Version"
#endif

#if (ICU_LLDRIVER_SW_MINOR_VERSION != ICU_LLDRIVER_C_SW_MINOR_VERSION)
  #error "Icu_LLDriver.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (4:0312)    : Dangerous pointer cast results in loss of volatile qualification.                            */
/* Rule                : CERTCCM EXP32, MISRA C:2012 Rule-11.8                                                        */
/* JV-01 Justification : According to the coding guide, all global variables must have volatile attribute, but        */
/*                       volatile must be removed when return these variables from APIs according to the AUTOSAR      */
/*                       specification.                                                                               */
/*       Verification  : Since these are normal memory objects, any problem doesn't occurs.                           */
/**********************************************************************************************************************/
/* Message (2:0751)    : Casting to char pointer type.                                                                */
/* Rule                : CERTCCM EXP11, EXP39                                                                         */
/* JV-01 Justification : Using void due to specific requirement of input parameter. So, this can be skipped           */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0857)    : Number of macro definitions exceeds 1024 - program does not conform strictly to ISO:C90.     */
/* Rule                : MISRA C:2012 Dir-1.1                                                                         */
/* JV-01 Justification : The number of macro depend on module code size. There is no issue when number of macro is    */
/*                       over 1024                                                                                    */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : This is accepted, due to the implementation is following hardware specification or it was    */
/*                       checked or do not dereference to NULL pointer.                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : It is specific for device register accessing and confirmed has no issue in software behavior.*/
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : Use of #include directive after code fragment.                                               */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is done as per Memory Requirement, (MEMMAP003 - Specification of Memory Mapping).       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                    Global Data                                                     **
***********************************************************************************************************************/

/***********************************************************************************************************************
**                                                Function Definitions                                                **
***********************************************************************************************************************/

/***********************************************************************************************************************
** Function Name         : Icu_HW_GetInputLevel
**
** Service ID            : NA
**
** Description           : This API service returns the logic voltage level of the input pin related to the ICU channel
**
** Sync/Async            : NA
**
** Reentrancy            : NA
**
** Input Parameters      : LddChannel
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Icu_LevelType
**
** Preconditions         : The Icu Driver must be initialized, ICU_GET_INPUT_LEVEL_API must be STD_ON
**
** Global Variables      : Icu_GpChannelConfig
**
** Functions invoked     : None
**
** Registers Used        : PPRn.
**
** Reference ID          : ICU_DUD_ACT_083
***********************************************************************************************************************/
#if (ICU_GET_INPUT_LEVEL_API == STD_ON)

#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"

FUNC(Icu_LevelType, ICU_PRIVATE_CODE) Icu_HW_GetInputLevel(const Icu_ChannelType LddChannel)
{
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;
  /* Local variable to store the return level value  */
  Icu_LevelType LenInputLevel;
  LenInputLevel = ICU_LOW;

  /* Update the channel configuration pointer to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LddChannel];

  if (NULL_PTR != LpChannelConfig->pPPRAddress)                                                                         /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    /* Check the port type configured */
    if (ICU_NORMAL_PORT == LpChannelConfig->ucIcuPortType)
    {
      /* Access data from 16 bit register and compare with the mask value */
      if ((*(LpChannelConfig->pPPRAddress) & (LpChannelConfig->usPortMaskValue)) == (LpChannelConfig->usPortMaskValue)) /* PRQA S 2814 # JV-01 */
      {
        LenInputLevel = ICU_HIGH;
      } /* else No action required */
    }
    else
    {
      /* Access data from 8 bit register and compare with the mask value */
      if ((uint16)(*((uint8 *)(LpChannelConfig->pPPRAddress)) &                                                         /* PRQA S 0312, 0751 # JV-01, JV-01 */
                                              (LpChannelConfig->usPortMaskValue)) == (LpChannelConfig->usPortMaskValue))
      {
        LenInputLevel = ICU_HIGH;
      } /* else No action required */
    }
  } /* else No action required */
   /* End of LpChannelConfig->pPPRAddress != NULL_PTR */
  return (LenInputLevel);
} /* End of API Icu_HW_Pin_GetInputLevel */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* #if (ICU_GET_INPUT_LEVEL_API == STD_ON) */
/***********************************************************************************************************************
**                                                    End of File                                                     **
***********************************************************************************************************************/
