/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Eth_ETNB_Irq.c                                                                                      */
/* SW-VERSION   = 1.6.0                                                                                               */
/*====================================================================================================================*/
/*                                             COPYRIGHT                                                              */
/*====================================================================================================================*/
/* (c) 2020-2022 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* This file contains Interrupt Service Routine for  Ethernet Driver Component                                        */
/*                                                                                                                    */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s)        */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs     */
/* of program errors, compliance with applicable laws, damage to or loss of data, programs or equipment,              */
/* and unavailability or interruption of operations.                                                                  */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*                                      Devices:        X2x                                                           */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                       Revision Control History                                                     **
***********************************************************************************************************************/
/*
 * 1.5.2: 08/02/2022    : Update QA-C comments
 *        24/01/2022    : Added the process to execute SYNCP after clear the interrupt status.
 *                        Fixed an issue in the TX BUFFER error check process.
 *                        Fixed the problem that transmission stops when LAN is repeatedly plugged and unplugged
 *                        with high transmission load
 *                        Fix the interrupt consistency check progress in the ETH_AVBnMNGISR function
 *                        Fixed an issue in the executing of the pipeline progress of the ETH_AVBnMNGISR function
 * 1.4.2: 24/08/2021    : Modify the format to 120 characters
 * 1.4.1: 01/04/2021    : Remove vendor ID from the file name, API names
 *                        and parameters according to BSW00347 requirement
 *        09/06/2021    : Remove QA-C Warning 4397 according to QA-C 9.5.0
 *        06/07/2021    : Updated QA-C 9.5.0 comments.
 * 1.2.0: 13/07/2020    : Release
 *        29/07/2020    : Add QAC 9.3.1 comment.
 * 1.1.0: 19/06/2020    : Release
 * 1.0.1: 04/06/2020    : To support Transmit/Receive interrupts for each controller.
 *                        Added ETH_59_AVBnMNGISR.
 *                        Update Reentrancy for ISR.
 *                        Add ETH_59_TSR_TFFL_MASK macro.
 *                        Updated static analysis result.
 *                        Changed include file structure.
 * 1.0.0: 25/03/2020    : Initial Version
 */
/******************************************************************************/

/***********************************************************************************************************************
**                                       Include Section                                                              **
***********************************************************************************************************************/
#include "Eth.h"
#include "Eth_ETNB_Irq.h"
#include "Eth_ETNB_Dma.h"
#include "Eth_ETNB_Ram.h"
/* Included for declaration of the function Dem_ReportErrorStatus() */
#include "Dem.h"
/***********************************************************************************************************************
**                                      Version Information                                                           **
***********************************************************************************************************************/

/* AUTOSAR release version information */
#define ETH_ETNB_IRQ_C_AR_RELEASE_MAJOR_VERSION ETH_AR_RELEASE_MAJOR_VERSION_VALUE
#define ETH_ETNB_IRQ_C_AR_RELEASE_MINOR_VERSION ETH_AR_RELEASE_MINOR_VERSION_VALUE
#define ETH_ETNB_IRQ_C_AR_RELEASE_REVISION_VERSION ETH_AR_RELEASE_REVISION_VERSION_VALUE

/* File version information */
#define ETH_ETNB_IRQ_C_SW_MAJOR_VERSION    ETH_SW_MAJOR_VERSION_VALUE
#define ETH_ETNB_IRQ_C_SW_MINOR_VERSION    ETH_SW_MINOR_VERSION_VALUE

/***********************************************************************************************************************
**                                      Version Check                                                                 **
***********************************************************************************************************************/

/* Functionality related to R4.0 */
#if (ETH_AR_RELEASE_MAJOR_VERSION != ETH_ETNB_IRQ_C_AR_RELEASE_MAJOR_VERSION)
  #error "Eth_ETNB_Irq.c : Mismatch in Release Major Version"
#endif
#if (ETH_AR_RELEASE_MINOR_VERSION != ETH_ETNB_IRQ_C_AR_RELEASE_MINOR_VERSION)
  #error "Eth_ETNB_Irq.c : Mismatch in Release Minor Version"
#endif
#if (ETH_AR_RELEASE_REVISION_VERSION != ETH_ETNB_IRQ_C_AR_RELEASE_REVISION_VERSION)
  #error "Eth_ETNB_Irq.c : Mismatch in Release Revision Version"
#endif

#if (ETH_SW_MAJOR_VERSION != ETH_ETNB_IRQ_C_SW_MAJOR_VERSION)
  #error "Eth_ETNB_Irq.c : Mismatch in Software Major Version"
#endif
#if (ETH_SW_MINOR_VERSION != ETH_ETNB_IRQ_C_SW_MINOR_VERSION)
  #error "Eth_ETNB_Irq.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (2:3408)    : %s' has external linkage and is being defined without any previous declaration.              */
/* Rule                : CERTCCM DCL07, MISRA C:2012 Rule-8.4                                                         */
/* JV-01 Justification : It is accepted, due to the declaration will be taken care by Os                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1503)    : The function 'name' is defined but is not used within this project.                          */
/* Rule                : CERTCCM MSC07, MISRA C:2012 Rule-2.1                                                         */
/* JV-01 Justification : This is accepted, due to the module's API is exported for user's usage.                      */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3432)    : Simple macro argument expression is not parenthesized.                                       */
/* Rule                : MISRA C:2012 Rule-20.7                                                                       */
/*                       REFERENCE - ISO:C90-6.3.1 Primary Expressions                                                */
/* JV-01 Justification : Compiler keyword (macro) is defined and used followed AUTOSAR standard rule. It is           */
/*                       accepted.                                                                                    */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3469)    : This usage of a function-like macro looks like it could be replaced by an equivalent         */
/*                       function call.                                                                               */
/* Rule                : MISRA C:2012 Dir-4.9                                                                         */
/* JV-01 Justification : To prevent overhead and speed up the processing, there is no need to call a function         */
/*                       here for such a small operation.                                                             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2934)    : Possible: Computing an invalid pointer value.                                                */
/* Rule                : CERTCCM ARR30, ARR37, ARR38, EXP08                                                           */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3383)    : Cannot identify wraparound guard for unsigned arithmetic expression.                         */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : Wraparound can't be occur in this unsigned integer arithmetic so there is no need to add     */
/*                       a wraparound guard here.                                                                     */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : (4:5087) #include statements in a file should only be preceded by other preprocessor         */
/*                       directives or comments.                                                                      */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is accepted, due to implementation for include memmap is following AUTOSAR rule.        */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                              Global Data                                                           **
***********************************************************************************************************************/
/***********************************************************************************************************************
**                                              Function Definitions                                                  **
***********************************************************************************************************************/
/***********************************************************************************************************************
** Function Name         : ETH_AVB0DATAISR
**
** Service ID            : NA
**
** Description           : Data Interrupt Service Handler
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETNBEICRegs, Eth_GaaDemEventIntInconsistent
**
** Function(s) invoked   : Eth_DemConfigCheck, Eth_HwTxConfirmation, Eth_Hw_Etnb_RxIrqHdlr
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_089,
** Reference ID          : ETH_DUD_ACT_089_ERR001
***********************************************************************************************************************/
#define ETH_START_SEC_CODE_FAST
#include "Eth_MemMap.h"

#if (ETH_AVB0_DATA_ISR == STD_ON)
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_AVB0DATAISR_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_AVB0DATAISR_CAT2)                                                                                               /* PRQA S 3408, 1503 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_AVB0DATAISR(void)                                                             /* PRQA S 1503 # JV-01 */
#endif
{
  #if (ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if (((ETH_EIC_CONSISTENCY_MASK & ~ETH_EIC_EIMK_MASK) !=
      (Eth_GaaETNBEICRegs[ETHAVB0]->usDATA & ETH_EIC_CONSISTENCY_MASK))                                                 /* PRQA S 2814 # JV-01 */

  #if (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
    || ((0UL != (Eth_GaaETNBRegs[ETHAVB0]->ulRIS0 & (uint32)(~ETH_RX_QUEUE_CONFIG_0)))                                  /* PRQA S 2814 # JV-01 */
    && (ETH_ENABLE == Eth_GpEthConfigPtr[ETHAVB0].enRxInterruptMode))
  #endif
  #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON)
    || ((0UL != (Eth_GaaETNBRegs[ETHAVB0]->ulDIS & (uint16)(~ETH_TX_QUEUE_CONFIG_0)))                                   /* PRQA S 2814 # JV-01 */
    && (ETH_ENABLE == Eth_GpEthConfigPtr[ETHAVB0].enTxInterruptMode))
  #endif
    )
  {
    Eth_DemConfigCheck(Eth_GaaDemEventIntInconsistent[ETHAVB0], DEM_EVENT_STATUS_FAILED);
  }
  else
#endif
  {
    if (ETH_ENABLE == Eth_GpEthConfigPtr[ETHAVB0].enTxInterruptMode)
    {
      Eth_HwTxConfirmation(ETHAVB0);
    }
    else
    {
      /* No action required */
    }

    #if (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
    if (ETH_ENABLE == Eth_GpEthConfigPtr[ETHAVB0].enRxInterruptMode)
    {
      Eth_Hw_Etnb_RxIrqHdlr(ETHAVB0);
    }
    else
    {
      /* No action required */
    }
    #endif
  }
}
#endif /* (ETH_AVB0_DATA_ISR == STD_ON) */

/***********************************************************************************************************************
** Function Name         : ETH_AVB0ERRISR
**
** Service ID            : NA
**
** Description           : Error Interrupt Service Handler
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETNBEICRegs, Eth_GaaDemEventIntInconsistent,
**                         Eth_GaaDemEventDmaError, Eth_GaaETNBRegs
**
** Function(s) invoked   : Eth_DemConfigCheck
**
** Registers Used        : EIC, ETNBnESR, ETNBnEIS
**
** Reference ID          : ETH_DUD_ACT_090,
** Reference ID          : ETH_DUD_ACT_090_ERR001, ETH_DUD_ACT_090_ERR002
** Reference ID          : ETH_DUD_ACT_090_REG001
***********************************************************************************************************************/
#if (ETH_AVB0_ERR_ISR == STD_ON)
#if defined (Os_ETH_AVB0ERRISR_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_AVB0ERRISR_CAT2)                                                                                                /* PRQA S 1503, 3408 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_AVB0ERRISR(void)                                                              /* PRQA S 1503 # JV-01 */
#endif
{
  uint32 LulRegVal;
  #if (ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if (((ETH_EIC_CONSISTENCY_MASK & ~ETH_EIC_EIMK_MASK) !=
      (Eth_GaaETNBEICRegs[ETHAVB0]->usERR & ETH_EIC_CONSISTENCY_MASK)) ||                                               /* PRQA S 2814 # JV-01 */
      (0UL == (Eth_GaaETNBRegs[ETHAVB0]->ulEIS & ETH_EIS_QEF_MASK)))                                                    /* PRQA S 2814 # JV-01 */
  {
    Eth_DemConfigCheck(Eth_GaaDemEventIntInconsistent[ETHAVB0], DEM_EVENT_STATUS_FAILED);
  }
  else
  #endif
  {
    LulRegVal = Eth_GaaETNBRegs[ETHAVB0]->ulESR & ETH_ETNB_ESR_ERROR_MASK;                                              /* PRQA S 2814 # JV-01 */
    if (ETH_ETNB_ESR_ERROR_TX_BUFFER != LulRegVal)
    {
      /* No action required */
    }
    else
    {
      /* Critical error detection on the MAC controller */
      Eth_DemConfigCheck(Eth_GaaDemEventDmaError[ETHAVB0], DEM_EVENT_STATUS_FAILED);
    }

    /* Clear Error Flag */
    LulRegVal = Eth_GaaETNBRegs[ETHAVB0]->ulEIS;
    Eth_GaaETNBRegs[ETHAVB0]->ulEIS = ~(LulRegVal & ETH_ETNB_EIC_ERROR);
    /* Execute the pipeline to avoid multiple interrupts */
    (void)Eth_GaaETNBRegs[ETHAVB0]->ulEIS;
    EXECUTE_SYNCP();
  }
}

#endif /* (ETH_AVB0_ERR_ISR == STD_ON) */

/***********************************************************************************************************************
** Function Name         : ETH_AVB0MNGISR
**
** Service ID            : NA
**
** Description           : Other Management Interrupt
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETNBRegs, Eth_GaaTxBufferIndex, Eth_GaaETNBEICRegs
**
** Function(s) invoked   : None
**
** Registers Used        : ETNBnTFA2, ETNBnTFA0, ETNBnTFA1, ETNBnTCCR, ETNBnTSR,
**                         ETNBnTIS
**
** Reference ID          : ETH_DUD_ACT_176,
** Reference ID          : ETH_DUD_ACT_176_ERR001, ETH_DUD_ACT_176_GLB001
** Reference ID          : ETH_DUD_ACT_176_GLB002, ETH_DUD_ACT_176_REG004
***********************************************************************************************************************/
#if (ETH_AVB0_MNG_ISR == STD_ON)
#if defined (Os_ETH_AVB0MNGISR_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_AVB0MNGISR_CAT2)                                                                                                /* PRQA S 3408, 1503 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_AVB0MNGISR(void)                                                              /* PRQA S 1503 # JV-01 */
#endif
{
  #if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
  uint8 LucFifoCount;
  uint16 LusTxBufferIdx;
  P2VAR(Eth_TxBufferType, AUTOMATIC, ETH_APPL_DATA) LpTxBuffer;                                                         /* PRQA S 3432 # JV-01 */
  #endif

  #if (ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if (((ETH_EIC_CONSISTENCY_MASK & ~ETH_EIC_EIMK_MASK) !=
      (Eth_GaaETNBEICRegs[ETHAVB0]->usMNG & ETH_EIC_CONSISTENCY_MASK)) ||                                               /* PRQA S 2814 # JV-01 */
      (0UL == (Eth_GaaETNBRegs[ETHAVB0]->ulTIS & ETH_TIS_TFUF_MASK)))                                                   /* PRQA S 2814 # JV-01 */
  {
    Eth_DemConfigCheck(Eth_GaaDemEventIntInconsistent[ETHAVB0], DEM_EVENT_STATUS_FAILED);
  }
  else
  #endif
  {
    #if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
    if (0UL != (Eth_GaaETNBRegs[ETHAVB0]->ulTIS & ETH_TIS_TFUF_MASK))
    {
      /* Get timestamp fifo counter */
      /* ETNBnTSR.TFFL is 8-10bit. so, casting to uint8 does no problem. */
      LucFifoCount = (uint8)(ETH_ETNB_GET_TFFL(Eth_GaaETNBRegs[ETHAVB0]->ulTSR));                                       /* PRQA S 3469 # JV-01 */

      do
      {
        /* Read buffer index from ETNBnTFA2.TST */
        LusTxBufferIdx = (uint16)(ETH_ETNB_GET_TST(Eth_GaaETNBRegs[ETHAVB0]->ulTFA2));                                  /* PRQA S 3469 # JV-01 */

        /* Initialize pointer to timestamp buffer */
        LpTxBuffer = (Eth_TxBufferType *)&Eth_GaaTxBufferIndex[ETHAVB0][LusTxBufferIdx];                                /* PRQA S 2934 # JV-01 */

        /* Read timestamp value to Tx buffer */
        LpTxBuffer->stTimeStamp.nanoseconds = Eth_GaaETNBRegs[ETHAVB0]->ulTFA0;                                         /* PRQA S 2844 # JV-01 */
        LpTxBuffer->stTimeStamp.seconds = Eth_GaaETNBRegs[ETHAVB0]->ulTFA1;
        LpTxBuffer->stTimeStamp.secondsHi = (uint16)(Eth_GaaETNBRegs[ETHAVB0]->ulTFA2 & ETH_TFA2_TSV_MASK);

        /* Release oldest entry in Timestamp FIFO - TCCR.TFR = 1 */
        Eth_GaaETNBRegs[ETHAVB0]->ulTCCR |= ETH_ETNB_TCCR_TFR;

        /* Set timestamp quality information as ETH_VALID */
        LpTxBuffer->enTimeQual = ETH_VALID;

        LucFifoCount--;                                                                                                 /* PRQA S 3383 # JV-01 */
      } while (0U < LucFifoCount);

      /* Execute the pipeline to avoid multiple interrupts */
      (void)Eth_GaaETNBRegs[ETHAVB0]->ulTCCR;
      EXECUTE_SYNCP();
    }
    else
    #endif
    {
      /* No action */
    }
  }
}
#endif /* (ETH_AVB0_ERR_ISR == STD_ON) */

/***********************************************************************************************************************
** Function Name         : ETH_AVB1DATAISR
**
** Service ID            : NA
**
** Description           : Data Interrupt Service Handler
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETNBEICRegs, Eth_GaaDemEventIntInconsistent
**
** Function(s) invoked   : Eth_DemConfigCheck, Eth_HwTxConfirmation, Eth_Hw_Etnb_RxIrqHdlr
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_093,
** Reference ID          : ETH_DUD_ACT_093_ERR001
***********************************************************************************************************************/
#if (ETH_AVB1_DATA_ISR == STD_ON)
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_AVB1DATAISR_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_AVB1DATAISR_CAT2)                                                                                               /* PRQA S 3408, 1503 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_AVB1DATAISR(void)                                                             /* PRQA S 1503 # JV-01 */
#endif
{
  #if (ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if (((ETH_EIC_CONSISTENCY_MASK & ~ETH_EIC_EIMK_MASK) !=
      (Eth_GaaETNBEICRegs[ETHAVB1]->usDATA & ETH_EIC_CONSISTENCY_MASK))                                                 /* PRQA S 2814 # JV-01 */
  #if (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
    || ((0UL != (Eth_GaaETNBRegs[ETHAVB1]->ulRIS0 & (uint32)(~ETH_RX_QUEUE_CONFIG_1)))                                  /* PRQA S 2814 # JV-01 */
    && (ETH_ENABLE == Eth_GpEthConfigPtr[ETHAVB1].enRxInterruptMode))
  #endif
  #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON)
    || ((0UL != (Eth_GaaETNBRegs[ETHAVB1]->ulDIS & (uint16)(~ETH_TX_QUEUE_CONFIG_1)))                                   /* PRQA S 2814 # JV-01 */
    && (ETH_ENABLE == Eth_GpEthConfigPtr[ETHAVB1].enTxInterruptMode))
  #endif
    )
  {
    Eth_DemConfigCheck(Eth_GaaDemEventIntInconsistent[ETHAVB1], DEM_EVENT_STATUS_FAILED);
  }
  else
#endif
  {
    if (ETH_ENABLE == Eth_GpEthConfigPtr[ETHAVB1].enTxInterruptMode)
    {
      Eth_HwTxConfirmation(ETHAVB1);
    }
    else
    {
      /* No action required */
    }

    #if (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
    if (ETH_ENABLE == Eth_GpEthConfigPtr[ETHAVB1].enRxInterruptMode)
    {
      Eth_Hw_Etnb_RxIrqHdlr(ETHAVB1);
    }
    else
    {
      /* No action required */
    }
    #endif /* (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON) */
  }
}
#endif /* (ETH_AVB1_DATA_ISR == STD_ON) */

/***********************************************************************************************************************
** Function Name         : ETH_AVB1ERRISR
**
** Service ID            : NA
**
** Description           : Error Interrupt Service Handler
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETNBEICRegs, Eth_GaaDemEventIntInconsistent,
**                         Eth_GaaDemEventDmaError, Eth_GaaETNBRegs
**
** Function(s) invoked   : Eth_DemConfigCheck
**
** Registers Used        : EIC, ETNBnESR, ETNBnEIS
**
** Reference ID          : ETH_DUD_ACT_094,
** Reference ID          : ETH_DUD_ACT_094_ERR001, ETH_DUD_ACT_094_ERR002
** Reference ID          : ETH_DUD_ACT_094_REG001
***********************************************************************************************************************/
#if (ETH_AVB1_ERR_ISR == STD_ON)
#if defined (Os_ETH_AVB1ERRISR_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_AVB1ERRISR_CAT2)                                                                                                /* PRQA S 3408, 1503 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_AVB1ERRISR(void)                                                              /* PRQA S 1503 # JV-01 */
#endif
{
  uint32 LulRegVal;
  #if (ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if (((ETH_EIC_CONSISTENCY_MASK & ~ETH_EIC_EIMK_MASK) !=
      (Eth_GaaETNBEICRegs[ETHAVB1]->usERR & ETH_EIC_CONSISTENCY_MASK)) ||                                               /* PRQA S 2814 # JV-01 */
      (0UL == (Eth_GaaETNBRegs[ETHAVB1]->ulEIS & ETH_EIS_QEF_MASK)))                                                    /* PRQA S 2814 # JV-01 */
  {
    Eth_DemConfigCheck(Eth_GaaDemEventIntInconsistent[ETHAVB1], DEM_EVENT_STATUS_FAILED);
  }
  else
  #endif
  {
    LulRegVal = Eth_GaaETNBRegs[ETHAVB1]->ulESR & ETH_ETNB_ESR_ERROR_MASK;
    if (ETH_ETNB_ESR_ERROR_TX_BUFFER != LulRegVal)
    {
      /* No action required */
    }
    else
    {
      /* Critical error detection on the MAC controller */
      Eth_DemConfigCheck(Eth_GaaDemEventDmaError[ETHAVB1], DEM_EVENT_STATUS_FAILED);
    }

    /* Clear Error Flag */
    LulRegVal = Eth_GaaETNBRegs[ETHAVB1]->ulEIS;
    Eth_GaaETNBRegs[ETHAVB1]->ulEIS = ~(LulRegVal & ETH_ETNB_EIC_ERROR);
    /* Execute the pipeline to avoid multiple interrupts */
    (void)Eth_GaaETNBRegs[ETHAVB1]->ulEIS;
    EXECUTE_SYNCP();
  }
}
#endif /* (ETH_AVB1_ERR_ISR == STD_ON) */

/***********************************************************************************************************************
** Function Name         : ETH_AVB1MNGISR
**
** Service ID            : NA
**
** Description           : Other Management Interrupt
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETNBRegs, Eth_GaaTxBufferIndex, Eth_GaaETNBEICRegs
**
** Function(s) invoked   : None
**
** Registers Used        : ETNBnTFA2, ETNBnTFA0, ETNBnTFA1, ETNBnTCCR, ETNBnTSR,
**                         ETNBnTIS
**
** Reference ID          : ETH_DUD_ACT_177,
** Reference ID          : ETH_DUD_ACT_177_ERR001, ETH_DUD_ACT_177_GLB001
** Reference ID          : ETH_DUD_ACT_177_GLB002, ETH_DUD_ACT_177_REG004
***********************************************************************************************************************/
#if (ETH_AVB1_MNG_ISR == STD_ON)
#if defined (Os_ETH_AVB1MNGISR_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_AVB1MNGISR_CAT2)                                                                                                /* PRQA S 3408, 1503 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_AVB1MNGISR(void)                                                              /* PRQA S 1503 # JV-01 */
#endif
{
  #if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
  uint8 LucFifoCount;
  uint16 LusTxBufferIdx;
  P2VAR(Eth_TxBufferType, AUTOMATIC, ETH_APPL_DATA) LpTxBuffer;                                                         /* PRQA S 3432 # JV-01 */
  #endif

  #if (ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if (((ETH_EIC_CONSISTENCY_MASK & ~ETH_EIC_EIMK_MASK) !=
      (Eth_GaaETNBEICRegs[ETHAVB1]->usMNG & ETH_EIC_CONSISTENCY_MASK)) ||                                               /* PRQA S 2814 # JV-01 */
      (0UL == (Eth_GaaETNBRegs[ETHAVB1]->ulTIS & ETH_TIS_TFUF_MASK)))                                                   /* PRQA S 2814 # JV-01 */
  {
    Eth_DemConfigCheck(Eth_GaaDemEventIntInconsistent[ETHAVB1], DEM_EVENT_STATUS_FAILED);
  }
  else
  #endif
  {
    #if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
    if (0UL != (Eth_GaaETNBRegs[ETHAVB1]->ulTIS & ETH_TIS_TFUF_MASK))
    {
      /* Get timestamp fifo counter */
      /* ETNBnTSR.TFFL is 8-10bit. so, casting to uint8 does no problem. */
      LucFifoCount = (uint8)(ETH_ETNB_GET_TFFL(Eth_GaaETNBRegs[ETHAVB1]->ulTSR));                                       /* PRQA S 3469 # JV-01 */

      do
      {
        /* Read buffer index from ETNBnTFA2.TST */
        LusTxBufferIdx = (uint16)(ETH_ETNB_GET_TST(Eth_GaaETNBRegs[ETHAVB1]->ulTFA2));                                  /* PRQA S 3469 # JV-01 */

        /* Initialize pointer to timestamp buffer */
        LpTxBuffer = (Eth_TxBufferType *)&Eth_GaaTxBufferIndex[ETHAVB1][LusTxBufferIdx];                                /* PRQA S 2934 # JV-01 */

        /* Read timestamp value to Tx buffer */
        LpTxBuffer->stTimeStamp.nanoseconds = Eth_GaaETNBRegs[ETHAVB1]->ulTFA0;                                         /* PRQA S 2844 # JV-01 */
        LpTxBuffer->stTimeStamp.seconds = Eth_GaaETNBRegs[ETHAVB1]->ulTFA1;
        LpTxBuffer->stTimeStamp.secondsHi = (uint16)(Eth_GaaETNBRegs[ETHAVB1]->ulTFA2 & ETH_TFA2_TSV_MASK);

        /* Release oldest entry in Timestamp FIFO - TCCR.TFR = 1 */
        Eth_GaaETNBRegs[ETHAVB1]->ulTCCR |= ETH_ETNB_TCCR_TFR;

        /* Set timestamp quality information as ETH_VALID */
        LpTxBuffer->enTimeQual = ETH_VALID;

        LucFifoCount--;                                                                                                 /* PRQA S 3383 # JV-01 */
      } while (0U < LucFifoCount);

      /* Execute the pipeline to avoid multiple interrupts */
      (void)Eth_GaaETNBRegs[ETHAVB1]->ulTCCR;
      EXECUTE_SYNCP();
    }
    else
    #endif
    {
      /* No action */
    }
  }
}
#endif /* (ETH_AVB0_ERR_ISR == STD_ON) */

/***********************************************************************************************************************
** Function Name         : ETH_AVB1STAISR
**
** Service ID            : NA
**
** Description           : Set ETNB1GECMR according to the STA interrupt.
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETNBEICRegs, Eth_GaaDemEventIntInconsistent
**                         Eth_GaaETNBRegs
**
** Function(s) invoked   : Eth_DemConfigCheck
**
** Registers Used        : EIC, ETNB1SGINTS, ETNB1SGOPMS, ETNB1GECMR, ETNB1SGINTS
**
** Reference ID          : ETH_DUD_ACT_097,
** Reference ID          : ETH_DUD_ACT_097_ERR001, ETH_DUD_ACT_097_REG001
** Reference ID          : ETH_DUD_ACT_097_REG002, ETH_DUD_ACT_097_REG003
***********************************************************************************************************************/
#if (ETH_AVB1_STA_ISR == STD_ON)
#if defined (Os_ETH_AVB1STAISR_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_AVB1STAISR_CAT2)                                                                                                /* PRQA S 3408, 1503 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_AVB1STAISR(void)                                                              /* PRQA S 1503 # JV-01 */
#endif
{
  uint32 LulLinkState;
  uint32 LulRegVal;

  #if (ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if (((ETH_EIC_CONSISTENCY_MASK & ~ETH_EIC_EIMK_MASK) !=
      (Eth_GaaETNBEICRegs[ETHAVB1]->usSTA & ETH_EIC_CONSISTENCY_MASK)) ||                                               /* PRQA S 2814 # JV-01 */
      (0UL == (Eth_GaaETNBRegs[ETHAVB1]->ulSGINTS & ETH_SGINTS_MASK)))                                                  /* PRQA S 2814 # JV-01 */
  {
    Eth_DemConfigCheck(Eth_GaaDemEventIntInconsistent[ETHAVB1], DEM_EVENT_STATUS_FAILED);
  }
  else
  #endif
  {
    LulRegVal = Eth_GaaETNBRegs[ETHAVB1]->ulSGINTS;
    LulLinkState = Eth_GaaETNBRegs[ETHAVB1]->ulSGOPMS;
    if ((ETH_ETNB_SGINTS_RDY == LulRegVal) ||
       ((ETH_ETNB_SGINTS_RDY_UNRDY == LulRegVal) && (ETH_SGOPMS_DATA_COM == (LulLinkState & ETH_SGOPMS_XMIT_MASK))))
    {
      if (0UL != (LulLinkState & ETH_SGOPMS_GSPEED_MASK))
      {
        Eth_GaaETNBRegs[ETHAVB1]->ulGECMR |= ETH_ETNB_GECMR_SPEED;
      }
      else
      {
        Eth_GaaETNBRegs[ETHAVB1]->ulGECMR &= ~ETH_ETNB_GECMR_SPEED;
      }
    }
    else
    {
      Eth_GaaETNBRegs[ETHAVB1]->ulGECMR &= ~ETH_ETNB_GECMR_SPEED;
    }
    Eth_GaaETNBRegs[ETHAVB1]->ulSGINTS = LulRegVal;
    /* Execute the pipeline to avoid multiple interrupts */
    (void)Eth_GaaETNBRegs[ETHAVB1]->ulSGINTS;
    EXECUTE_SYNCP();
  }
}
#endif /* (ETH_AVB1_STA_ISR == STD_ON) */

#define ETH_STOP_SEC_CODE_FAST
#include "Eth_MemMap.h"
/***********************************************************************************************************************
**                                           End of File                                                              **
***********************************************************************************************************************/
