/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Eth_Filter.c                                                                                        */
/* SW-VERSION   = 1.6.0                                                                                               */
/*====================================================================================================================*/
/*                                  COPYRIGHT                                                                         */
/*====================================================================================================================*/
/* (c) 2020,2021 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* This file contains software filter functions implementation of Eth Driver Component.                               */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s)        */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs     */
/* of program errors, compliance with applicable laws, damage to or loss of data, programs or equipment,              */
/* and unavailability or interruption of operations.                                                                  */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*                                        Devices:                X2x                                                 */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                          Revision Control History                                                  **
***********************************************************************************************************************/
/*
 * 1.5.1: 18/10/2021    : Commonized Eth_ETNE_Ram.h and Eth_ETND_Ram.h.
 *        12/01/2021    : Update QA-C 9.5.0 comments
 * 1.4.2: 24/08/2021    : Modify the format to 120 characters
 *        20/08/2021    : Update include file for support U2Bx devices.
 * 1.4.1: 07/05/2021    : Changes to support U2Bx devices.
 *                        Remove vendor ID from the file name, API names
 *        09/06/2021    : Remove QA-C Warning 0404 and 3432 according to QA-C 9.5.0
 *                        and parameters according to BSW00347 requirement
 *        06/07/2021    : Updated QA-C 9.5.0 comments.
 * 1.2.0: 13/07/2020    : Release
 *        29/07/2020    : Add QAC 9.3.1 comment.
 * 1.1.0: 19/06/2020    : Release
 * 1.0.0: 25/03/2020    : Initial Version
 */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                       Include Section                                                              **
***********************************************************************************************************************/
#include "Eth.h"
#if (ETH_USING_MACRO == ETH_MACRO_ETNC)
#include "Eth_ETNC_Ram.h"
#elif (ETH_USING_MACRO == ETH_MACRO_ETNB)
#include "Eth_ETNB_Ram.h"
#elif (ETH_USING_MACRO == ETH_MACRO_ETND || ETH_USING_MACRO == ETH_MACRO_ETNE)
#include "Eth_ETNDE_Ram.h"
#endif
#include "Eth_Filter.h"

/***********************************************************************************************************************
**                      Version Information                                                                           **
***********************************************************************************************************************/
/* AUTOSAR release version information */
#define ETH_FILTER_C_AR_RELEASE_MAJOR_VERSION ETH_AR_RELEASE_MAJOR_VERSION_VALUE
#define ETH_FILTER_C_AR_RELEASE_MINOR_VERSION ETH_AR_RELEASE_MINOR_VERSION_VALUE
#define ETH_FILTER_C_AR_RELEASE_REVISION_VERSION ETH_AR_RELEASE_REVISION_VERSION_VALUE

/* File version information */
#define ETH_FILTER_C_SW_MAJOR_VERSION    ETH_SW_MAJOR_VERSION_VALUE
#define ETH_FILTER_C_SW_MINOR_VERSION    ETH_SW_MINOR_VERSION_VALUE

/***********************************************************************************************************************
**                      Version Check                                                                                 **
***********************************************************************************************************************/
#if (ETH_FILTER_AR_RELEASE_MAJOR_VERSION != ETH_FILTER_C_AR_RELEASE_MAJOR_VERSION)
  #error "Eth_Filter.c : Mismatch in Release Major Version"
#endif
#if (ETH_FILTER_AR_RELEASE_MINOR_VERSION != ETH_FILTER_C_AR_RELEASE_MINOR_VERSION)
  #error "Eth_Filter.c : Mismatch in Release Minor Version"
#endif
#if (ETH_FILTER_AR_RELEASE_REVISION_VERSION != ETH_FILTER_C_AR_RELEASE_REVISION_VERSION)
  #error "Eth_Filter.c : Mismatch in Release Revision Version"
#endif

#if (ETH_FILTER_SW_MAJOR_VERSION != ETH_FILTER_C_SW_MAJOR_VERSION)
  #error "Eth_Filter.c : Mismatch in Software Major Version"
#endif
#if (ETH_FILTER_SW_MINOR_VERSION != ETH_FILTER_C_SW_MINOR_VERSION)
  #error "Eth_Filter.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (1:1533)    : The object '%1s' is only referenced by function '%2s'.                                       */
/* Rule                : MISRA C:2012 Rule-8.9                                                                        */
/* JV-01 Justification : This is accepted, due to the object is defined in separated source C file to followed        */
/*                       coding rule                                                                                  */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1532)    : The function '%1s' is only referenced in one translation unit - but not the one in which it  */
/*                       is defined.                                                                                  */
/* Rule                : CERTCCM DCL19, MISRA C:2012 Rule-8.7                                                         */
/* JV-01 Justification : This is accepted, due to the specific coding rule, function of each hardware unit is         */
/*                       implemented in separated files for this hardware unit. Could not be static function.         */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3469)    : This usage of a function-like macro looks like it could be replaced by an equivalent         */
/*                       function call.                                                                               */
/* Rule                : MISRA C:2012 Dir-4.9                                                                         */
/* JV-01 Justification : To prevent overhead and speed up the processing, there is no need to call a function here    */
/*                       for such a small operation.                                                                  */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : Use of #include directive after code fragment.                                               */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is accepted, due to implementation for include memmap is following AUTOSAR rule.        */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                      Global Data                                                                                   **
***********************************************************************************************************************/
#define ETH_START_SEC_CONST_32
#include "Eth_MemMap.h"

/* Const values of broadcast and null address to be compared */
CONST(Eth_MacAddressType, ETH_CONST) Eth_GstBroadcastAddr =
{
  0xFFFFFFFFUL, 0x0000FFFFU
};
CONST(Eth_MacAddressType, ETH_CONST) Eth_GstNullAddr =                                                                  /* PRQA S 1533 # JV-01 */
{
  0x00000000UL, 0x00000000U
};
#define ETH_STOP_SEC_CONST_32
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */


/***********************************************************************************************************************
**                                         Function Definitions                                                       **
***********************************************************************************************************************/
#define ETH_START_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

#if (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_GetFilterIndex
**
** Service ID            : N/A
**
** Description           : Get the index of the specified MAC address if exists
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LulCtrlIdx: Index of a controller
**                         LpMacAddr:  Pointer to a MAC address to be searched
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat, Eth_GaaAddressFilters
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_032
***********************************************************************************************************************/
FUNC(uint32, ETH_PRIVATE_CODE) Eth_GetFilterIndex(                                                                      /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONSTP2CONST(Eth_MacAddressType, AUTOMATIC, AUTOMATIC) LpMacAddr)
{
  uint32 LulFilterIdx;
  uint32 LulFoundIdx;

  LulFoundIdx = ETH_INVALID_FILTER_INDEX;
  /* Check the specified address is already registered */
  for (LulFilterIdx = 0UL; (ETH_INVALID_FILTER_INDEX == LulFoundIdx) &&
    (LulFilterIdx < ETH_MAX_FILTERS); LulFilterIdx++)
  {
    if ((0UL != (Eth_GaaCtrlStat[LulCtrlIdx].ulActiveFilterBits & (1UL << LulFilterIdx))) &&                            /* PRQA S 2844 # JV-01 */
      (0UL == ETH_COMPARE_MAC(*LpMacAddr, Eth_GaaAddressFilters[LulCtrlIdx][LulFilterIdx])))                            /* PRQA S 2814, 2844, 3469 # JV-01, JV-01, JV-01 */
    {
      LulFoundIdx = LulFilterIdx;
    }
    else
    {
      /* No action required */
    }
  }

  return LulFoundIdx;
}

/***********************************************************************************************************************
** Function Name         : Eth_ClearAllAddressFilters
**
** Service ID            : N/A
**
** Description           : Remove all filters from the filter list and turn off promiscuous mode
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LulCtrlIdx: Index of a controller
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_033, ETH_DUD_ACT_033_GBL001,
** Reference ID          : ETH_DUD_ACT_033_GBL002
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_ClearAllAddressFilters(                                                                /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  Eth_GaaCtrlStat[LulCtrlIdx].ulActiveFilterBits = 0UL;                                                                 /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].blPromiscuous = ETH_FALSE;                                                                /* PRQA S 2844 # JV-01 */
}

/***********************************************************************************************************************
** Function Name         : Eth_RemoveAddressFilter
**
** Service ID            : N/A
**
** Description           : Remove a filter from the filter list
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LulCtrlIdx  : Index of a controller
**                         LulFilterIdx: Index of an address in the filter list
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_034, ETH_DUD_ACT_034_GBL001
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_RemoveAddressFilter(                                                                   /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulFilterIdx)
{
  /* Clear an active bit */
  Eth_GaaCtrlStat[LulCtrlIdx].ulActiveFilterBits &= ~(1UL << LulFilterIdx);                                             /* PRQA S 2844 # JV-01 */
}

/***********************************************************************************************************************
** Function Name         : Eth_AddAddressFilter
**
** Service ID            : N/A
**
** Description           : Add a address filter to the filter list
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LulCtrlIdx: Index of a controller
**                         LpMacAddr : Pointer to a MAC address to be added
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : At least there is an empty entry in the FilterArray
**
** Global Variable(s)    : Eth_GaaCtrlStat, Eth_GaaAddressFilters
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_035, ETH_DUD_ACT_035_GBL001,
** Reference ID          : ETH_DUD_ACT_035_GBL002
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_AddAddressFilter(                                                                      /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONSTP2CONST(Eth_MacAddressType, AUTOMATIC, AUTOMATIC) LpMacAddr)
{
  uint32 LulFilterIdx;
  boolean LblRegistered;

  LblRegistered = ETH_FALSE;
  for (LulFilterIdx = 0UL; (ETH_FALSE == LblRegistered) && (LulFilterIdx < ETH_MAX_FILTERS); LulFilterIdx++)
  {
    if (0UL == (Eth_GaaCtrlStat[LulCtrlIdx].ulActiveFilterBits & (1UL << (LulFilterIdx))))                              /* PRQA S 2844 # JV-01 */
    {
      Eth_GaaAddressFilters[LulCtrlIdx][LulFilterIdx] = *LpMacAddr;                                                     /* PRQA S 2844, 2814 # JV-01, JV-01 */
      Eth_GaaCtrlStat[LulCtrlIdx].ulActiveFilterBits |= (1UL << LulFilterIdx);
      LblRegistered = ETH_TRUE;
    }
    else
    {
      /* No action required */
    }
  }
}
#endif /* (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON) */

#define ETH_STOP_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
**                                            End of File                                                             **
***********************************************************************************************************************/
