/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Eth_ETNDE_Irq.c                                                                                     */
/* SW-VERSION   = 1.5.4                                                                                               */
/*====================================================================================================================*/
/*                                             COPYRIGHT                                                              */
/*====================================================================================================================*/
/* (c) 2022 Renesas Electronics Corporation. All rights reserved.                                                     */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* This file contains Interrupt Service Routine for  Ethernet Driver Component                                        */
/*                                                                                                                    */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s)        */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs     */
/* of program errors, compliance with applicable laws, damage to or loss of data, programs or equipment,              */
/* and unavailability or interruption of operations.                                                                  */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*                                      Devices:        X2x                                                           */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                       Revision Control History                                                     **
***********************************************************************************************************************/
/*
 * 1.5.3: 14/04/2022    : Update QA-C comments.
 * 1.5.1: 11/01/2022    : Initial Version
 */
/******************************************************************************/

/***********************************************************************************************************************
**                                       Include Section                                                              **
***********************************************************************************************************************/
#include "Eth.h"
#include "Eth_ETNDE_Irq.h"
#include "Eth_ETNDE_Ram.h"
/* Included for declaration of the function Dem_ReportErrorStatus() */
#include "Dem.h"
/***********************************************************************************************************************
**                                      Version Information                                                           **
***********************************************************************************************************************/

/* AUTOSAR release version information */
#define ETH_ETNDE_IRQ_C_AR_RELEASE_MAJOR_VERSION    ETH_AR_RELEASE_MAJOR_VERSION_VALUE
#define ETH_ETNDE_IRQ_C_AR_RELEASE_MINOR_VERSION    ETH_AR_RELEASE_MINOR_VERSION_VALUE
#define ETH_ETNDE_IRQ_C_AR_RELEASE_REVISION_VERSION ETH_AR_RELEASE_REVISION_VERSION_VALUE

/* File version information */
#define ETH_ETNDE_IRQ_C_SW_MAJOR_VERSION    ETH_SW_MAJOR_VERSION_VALUE
#define ETH_ETNDE_IRQ_C_SW_MINOR_VERSION    ETH_SW_MINOR_VERSION_VALUE

/***********************************************************************************************************************
**                                      Version Check                                                                 **
***********************************************************************************************************************/

/* Functionality related to R4.0 */
#if (ETH_AR_RELEASE_MAJOR_VERSION != ETH_ETNDE_IRQ_C_AR_RELEASE_MAJOR_VERSION)
  #error "Eth_ETNDE_Irq.c : Mismatch in Release Major Version"
#endif
#if (ETH_AR_RELEASE_MINOR_VERSION != ETH_ETNDE_IRQ_C_AR_RELEASE_MINOR_VERSION)
  #error "Eth_ETNDE_Irq.c : Mismatch in Release Minor Version"
#endif
#if (ETH_AR_RELEASE_REVISION_VERSION != ETH_ETNDE_IRQ_C_AR_RELEASE_REVISION_VERSION)
  #error "Eth_ETNDE_Irq.c : Mismatch in Release Revision Version"
#endif

#if (ETH_SW_MAJOR_VERSION != ETH_ETNDE_IRQ_C_SW_MAJOR_VERSION)
  #error "Eth_ETNDE_Irq.c : Mismatch in Software Major Version"
#endif
#if (ETH_SW_MINOR_VERSION != ETH_ETNDE_IRQ_C_SW_MINOR_VERSION)
  #error "Eth_ETNDE_Irq.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (2:3432)    : Simple macro argument expression is not parenthesized.                                       */
/* Rule                : MISRA C:2012 Rule-20.7                                                                       */
/* JV-01 Justification : Compiler keyword (macro) is defined and used followed AUTOSAR standard rule. It is accepted. */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0316)    : Cast from a pointer to void to a pointer to object type.                                     */
/* Rule                : MISRA C:2012 Rule-11.5                                                                       */
/* JV-01 Justification : Typecasting from void* is necessary to hide internal types from the header files which are   */
/*                       exposed to user.                                                                             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2824)    : Possible: Arithmetic operation on NULL pointer.                                              */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3383)    : Cannot identify wraparound guard for unsigned arithmetic expression.                         */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : Wraparound can't be occur in this unsigned integer arithmetic so there is no need to add a   */
/*                       wraparound guard here.                                                                       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2834)    : Possible: Division by zero.                                                                  */
/* Rule                : CERTCCM INT33                                                                                */
/* JV-01 Justification : Value generated by generation tool is always different from zero.                            */
/*       Verification  : It's not impact to driver source code implementation                                         */
/**********************************************************************************************************************/
/* Message (2:0306)    : Cast between a pointer to object and an integral type.                                       */
/* Rule                : CERTCCM INT36, MISRA C:2012 Rule-11.4                                                        */
/* JV-01 Justification : Typecasting is done as per the register size or hardware specific structure.                 */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3200)    : '%s' returns a value which is not being used.                                                */
/* Rule                : CERTCCM ERR33, EXP12, POS54, MISRA C:2012 Rule-17.7                                          */
/* JV-01 Justification : The usage of the return value of this function is not required for the interrupt handler.    */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1503)    : The function '%1s' is defined but is not used within this project.                           */
/* Rule                : CERTCCM MSC07, MISRA C:2012 Rule-2.1                                                         */
/* JV-01 Justification : This is accepted, due to the module's API is exported for user's usage.                      */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : Use of #include directive after code fragment.                                               */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is done as per Memory Requirement, (MEMMAP003 - Specification of Memory Mapping).       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0488)    : Performing pointer arithmetic.                                                               */
/* Rule                : CERTCCM EXP08, MISRA C:2012 Rule-18.4                                                        */
/* JV-01 Justification : To simplify the register checking by using the table.                                        */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:0312)    : Dangerous pointer cast results in loss of volatile qualification.                            */
/* Rule                : CERTCCM EXP32, MISRA C:2012 Rule-11.8                                                        */
/* JV-01 Justification : To simplify the register checking by using the table.                                        */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3408)    : %s' has external linkage and is being defined without any previous declaration.              */
/* Rule                : CERTCCM DCL07, MISRA C:2012 Rule-8.4                                                         */
/* JV-01 Justification : It is accepted, due to the declaration will be taken care by Os                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0310)    : Casting to different object pointer type.                                                    */
/* Rule                : CERTCCM EXP11, EXP39, MISRA C:2012 Rule-11.3                                                 */
/* JV-01 Justification : There is no problem because it is as designed.                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:1006)    : [E] This in-line assembler construct is a language extension. The code has been ignored.     */
/* Rule                : CERTCCM MSC14, MISRA C:2012 Dir-1.1, Rule-1.2, Dir-4.2                                       */
/* JV-01 Justification : SYNCP instruction is required to avoid multiple interrupts which cause incomplete instruction*/
/*                       to the register.                                                                             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/***********************************************************************************************************************
**                                              Global Data                                                           **
***********************************************************************************************************************/
/***********************************************************************************************************************
**                                              Function Definitions                                                  **
***********************************************************************************************************************/
#define ETH_START_SEC_CODE_FAST
#include "Eth_MemMap.h"

/* Prototypes for internal functions */
#if ((ETH_TSN0_DATA1_ISR == STD_ON) || (ETH_TSN0_DATA2_ISR == STD_ON) || (ETH_TSN0_DATA3_ISR == STD_ON) || \
     (ETH_TSN1_DATA1_ISR == STD_ON) || (ETH_TSN1_DATA2_ISR == STD_ON) || (ETH_TSN1_DATA3_ISR == STD_ON) || \
     (ETH_RSW_DATA1_ISR  == STD_ON) || (ETH_RSW_DATA2_ISR  == STD_ON) || (ETH_RSW_DATA3_ISR  == STD_ON))
#define ETH_CONTROLLER_DATA_INTERRUPT_ON
STATIC FUNC(void, ETH_CODE_FAST) Eth_DataIsr(CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulIsr);
#endif

#if ((ETH_TSN0_DATA0_ISR == STD_ON) || (ETH_TSN1_DATA0_ISR == STD_ON) || (ETH_RSW_DATA0_ISR == STD_ON))
#define ETH_CONTROLLER_DATA0_INTERRUPT_ON
STATIC FUNC(void, ETH_CODE_FAST) Eth_DataIsr0(CONST(uint32, AUTOMATIC) LulCtrlIdx);
#endif

#if ((ETH_TSN0_GENERAL_ISR == STD_ON) || (ETH_TSN1_GENERAL_ISR == STD_ON) || (ETH_RSW_GENERAL_ISR == STD_ON))
#define ETH_CONTROLLER_GENERAL_INTERRUPT_ON
STATIC FUNC(void, ETH_CODE_FAST) Eth_GeneralIsr(CONST(uint32, AUTOMATIC) LulCtrlIdx);
#endif

/***********************************************************************************************************************
** Function Name         : Eth_DataIsr0
**
** Service ID            : NA
**
** Description           : Data Interrupt Service Handler
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETND_EICRegs, Eth_GaaDemEventIntInconsistent,
**                         Eth_GstETND_Regs, Eth_GaaETNE_EICRegs, Eth_GpEthConfigPtr
**                         Eth_GpCtrlConfigPtr, Eth_GaaDescTsChainTable, Eth_GaaDescTsChain
**                         Eth_GpETNE_MFWDRegs
**
** Function(s) invoked   : Eth_DemConfigCheck, Eth_HwTxConfirmation, Eth_HwReceive
**
** Registers Used        : EICn, TDISi, RDISi, TSDIS
**
** Reference ID          : ETH_DUD_ACT_348
** Reference ID          : ETH_DUD_ACT_348_GBL001, ETH_DUD_ACT_348_GBL002, ETH_DUD_ACT_348_GBL003
** Reference ID          : ETH_DUD_ACT_348_GBL004, ETH_DUD_ACT_348_GBL005
** Reference ID          : ETH_DUD_ACT_348_GBL007, ETH_DUD_ACT_348_GBL008, ETH_DUD_ACT_348_GBL009
** Reference ID          : ETH_DUD_ACT_348_GBL010, ETH_DUD_ACT_348_GBL011, ETH_DUD_ACT_348_GBL012
** Reference ID          : ETH_DUD_ACT_348_GBL013, ETH_DUD_ACT_348_GBL014, ETH_DUD_ACT_348_GBL015
** Reference ID          : ETH_DUD_ACT_348_GBL016, ETH_DUD_ACT_348_GBL017, ETH_DUD_ACT_348_GBL018
** Reference ID          : ETH_DUD_ACT_348_GBL019, ETH_DUD_ACT_348_GBL020, ETH_DUD_ACT_348_REG001
** Reference ID          : ETH_DUD_ACT_348_REG002, ETH_DUD_ACT_348_ERR001
***********************************************************************************************************************/
#ifdef ETH_CONTROLLER_DATA0_INTERRUPT_ON
STATIC FUNC(void, ETH_CODE_FAST) Eth_DataIsr0(CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON || ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
  P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;
  uint32 LulIntStat;
  #endif

  #if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
  P2VAR(Eth_BufHandlerType, AUTOMATIC, ETH_APPL_DATA) LpTxBufferNode;                                                   /* PRQA S 3432 # JV-01 */
  P2VAR(Eth_ExtDescWithTsType, AUTOMATIC, ETH_APPL_DATA) LpDescTsChain;                                                 /* PRQA S 3432 # JV-01 */
  P2VAR(Eth_ExtDescWithTsType, AUTOMATIC, ETH_APPL_DATA) LpLinkDesc;                                                    /* PRQA S 3432 # JV-01 */
  Eth_BufIdxType LulBufIdx;
  uint32 LulRingIdx;
  #endif

  #if (ETH_USING_MACRO == ETH_MACRO_ETND)
  CONSTP2VAR(volatile Eth_ETNDE_AXIBMIRegType, AUTOMATIC, REGSPACE) LpAxiRegs =
    &Eth_GstETND_Regs.pES[LulCtrlIdx]->stAXIBMI;                                                                        /* PRQA S 2844 # JV-01 */
  #else
  CONSTP2VAR(volatile Eth_ETNDE_AXIBMIRegType, AUTOMATIC, REGSPACE) LpAxiRegs = Eth_GstETNE_Regs.pAXIBMI;
  #endif

  #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON || ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
  LpHwUnitConfig =
    (P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA))Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;              /* PRQA S 0316 # JV-01 */
  #endif

  #if (ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if (((ETH_EIC_CONSISTENCY_MASK & ~ETH_EIC_EIMK_MASK) !=
      #if (ETH_USING_MACRO == ETH_MACRO_ETND)
      (Eth_GaaETND_EICRegs[LulCtrlIdx]->usINTETNDw0 & ETH_EIC_CONSISTENCY_MASK))                                        /* PRQA S 2814, 2844 # JV-01, JV-01 */
      #else
      (Eth_GpETNE_EICRegs->usINTETNE00 & ETH_EIC_CONSISTENCY_MASK))
      #endif
  #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON)
    || ((0UL != ((LpAxiRegs->stTXDI[0].ulTDISi & LpHwUnitConfig->stQueueConfig.ulTxIntConfig)                           /* PRQA S 2814 # JV-01 */
     & (uint32)(~ETH_ETNDE_DIS_MASK(LulCtrlIdx, ETH_DATAISR0, LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue))))       /* PRQA S 2814 # JV-01 */
    && (ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enTxInterruptMode))
  #endif
  #if (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
    || ((0UL != ((LpAxiRegs->stRXDI[0].ulRDISi & LpHwUnitConfig->stQueueConfig.ulRxIntConfig)
     & (uint32)(~ETH_ETNDE_DIS_MASK(LulCtrlIdx, ETH_DATAISR0, LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue))))
    && (ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enRxInterruptMode))
  #endif
  #if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
    || (0UL == (LpAxiRegs->ulTSDIS & ETH_ETNDE_TSDIS_0_MASK))
  #endif
    )
  {
    Eth_DemConfigCheck(Eth_GaaDemEventIntInconsistent[LulCtrlIdx], DEM_EVENT_STATUS_FAILED);                            /* PRQA S 2844 # JV-01 */
  }
  else
  #endif /* ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON */
  {
    #if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
    if (0UL != (LpAxiRegs->ulTSDIS & ETH_ETNDE_TSDIS_0_MASK))                                                           /* PRQA S 2814 # JV-01 */
    {
      #if (ETH_USING_MACRO == ETH_MACRO_ETND)
      LpLinkDesc = &Eth_GaaDescTsChainTable[LulCtrlIdx][Eth_GaaTsDescTotal[LulCtrlIdx]];                                /* PRQA S 2844, 2824 # JV-01, JV-01 */
      #else
      LpLinkDesc = &Eth_GaaDescTsChain[(uint32)ETH_MAX_TS_DESCRIPTOR];
      #endif
      
      /* Check status of link descriptor */
      if (ETH_DESC_LEMPTY == LpLinkDesc->stHeader.ulDt)                                                                 /* PRQA S 2814 # JV-01 */
      {
        /* Reset link descriptor status */
        LpLinkDesc->stHeader.ulDt = (uint32)ETH_DESC_LINK;
      } /* else: No action required */

      /* Search timestamp descriptor */
      #if (ETH_USING_MACRO == ETH_MACRO_ETND)
      LulRingIdx = Eth_GaaCtrlStat[LulCtrlIdx].ulTsDescTail;                                                            /* PRQA S 2844 # JV-01 */
      LpDescTsChain = &Eth_GaaDescTsChainTable[LulCtrlIdx][LulRingIdx];
      #else
      LulRingIdx = Eth_GulTsDescTail;                                                                                   /* PRQA S 2844 # JV-01 */
      LpDescTsChain = &Eth_GaaDescTsChain[LulRingIdx];
      #endif

      while (ETH_DESC_FSINGLE == LpDescTsChain->stHeader.ulDt)                                                          /* PRQA S 2814 # JV-01 */
      {
        #if (ETH_USING_MACRO == ETH_MACRO_ETND)
        LulBufIdx = (LpDescTsChain->stHeader.ulInfo0 & ETH_TX_DESCR_TSUN_MASK) >> 4UL;                                  /* PRQA S 2814 # JV-01 */
        #else
        LulBufIdx = ((Eth_TxCtrlInfo1Type *)&LpDescTsChain->ulInfo1[0])->ulTsun -
                     (LulCtrlIdx * ETH_TX_DESCR_TSUN_OFFSET);
        #endif
        LpTxBufferNode = (Eth_BufHandlerType *)Eth_GaaTxBufferMgrTable[LulCtrlIdx][LulBufIdx].pBufferHdr;               /* PRQA S 2844, 2824 # JV-01, JV-01 */

        /* Read timestamp value to Tx buffer */
        LpTxBufferNode->stTimeStamp.nanoseconds = LpDescTsChain->ulTsns;                                                /* PRQA S 2814 # JV-01 */
        LpTxBufferNode->stTimeStamp.seconds     = LpDescTsChain->ulTss;
        LpTxBufferNode->stTimeStamp.secondsHi   = (uint16)0U;

        /* Set timestamp quality information as ETH_VALID */
        LpTxBufferNode->enTimeQual = ETH_VALID;

        /* Release timestamp descriptor */
        LpDescTsChain->stHeader.ulDie   = (uint32)ETH_TS_DESC_DIE_ENABLE;
        LpDescTsChain->stHeader.ulAxie  = (uint32)0UL;
        LpDescTsChain->stHeader.ulDse   = (uint32)0UL;
        LpDescTsChain->stHeader.ulInfo0 = (uint32)0UL;
        LpDescTsChain->stHeader.ulDs    = (uint32)0UL;
        LpDescTsChain->ulDptr     = 0UL;
        LpDescTsChain->ulInfo1[0] = 0UL;
        LpDescTsChain->ulInfo1[1] = 0UL;
        LpDescTsChain->ulTsns     = 0UL;
        LpDescTsChain->ulTss      = 0UL;
        do 
        {
          LpDescTsChain->stHeader.ulDt = (uint32)ETH_DESC_FEMPTY_ND;
        } while ((uint32)ETH_DESC_FEMPTY_ND != LpDescTsChain->stHeader.ulDt);
        
        #if (ETH_USING_MACRO == ETH_MACRO_ETND)
        LulRingIdx = (LulRingIdx + 1UL) % (uint32)(Eth_GaaTsDescTotal[LulCtrlIdx] + ETH_CYCLIC_DESC_NUM);               /* PRQA S 3383, 2834 # JV-01, JV-01 */
        LpDescTsChain = &Eth_GaaDescTsChainTable[LulCtrlIdx][LulRingIdx];
        #else
        LulRingIdx = (LulRingIdx + 1UL) % (uint32)((uint32)ETH_MAX_TS_DESCRIPTOR + (uint32)ETH_CYCLIC_DESC_NUM);
        LpDescTsChain = &Eth_GaaDescTsChain[LulRingIdx];
        #endif

        while ((ETH_DESC_LINK == LpDescTsChain->stHeader.ulDt) || (ETH_DESC_LEMPTY == LpDescTsChain->stHeader.ulDt))    /* PRQA S 2814 # JV-01 */
        {
          LpDescTsChain->stHeader.ulDt = (uint32)ETH_DESC_LINK;                                                         /* PRQA S 2814 # JV-01 */
          LpDescTsChain = (Eth_ExtDescWithTsType *)LpDescTsChain->ulDptr;                                               /* PRQA S 0306 # JV-01 */
          #if (ETH_USING_MACRO == ETH_MACRO_ETND)
          LulRingIdx = (LulRingIdx + 1UL) % (uint32)(Eth_GaaTsDescTotal[LulCtrlIdx] + ETH_CYCLIC_DESC_NUM);             /* PRQA S 3383, 2834 # JV-01, JV-01 */
          #else
          LulRingIdx = (LulRingIdx + 1UL) % (uint32)((uint32)ETH_MAX_TS_DESCRIPTOR + (uint32)ETH_CYCLIC_DESC_NUM);
          #endif
        }
      }

      #if (ETH_USING_MACRO == ETH_MACRO_ETND)
      Eth_GaaCtrlStat[LulCtrlIdx].ulTsDescTail = LulRingIdx;                                                            /* PRQA S 2844 # JV-01 */
      #else
      Eth_GulTsDescTail = LulRingIdx;
      #endif

      /* Clear the TS Descriptor Data Interrupt Status bit */
      LpAxiRegs->ulTSDIS |= ETH_ETNDE_TSDIS_0_MASK;
      /* Execute the pipeline to avoid multiple interrupts */
      (void)LpAxiRegs->ulTSDIS;
      EXECUTE_SYNCP();                                                                                                  /* PRQA S 1006 # JV-01 */
    } /* else: No action required */
    #endif /* ETH_GLOBAL_TIME_SUPPORT == STD_ON */

    #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON)
    LulIntStat = LpAxiRegs->stTXDI[0].ulTDISi &
      ETH_ETNDE_DIS_MASK(LulCtrlIdx, ETH_DATAISR0, LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue);                    /* PRQA S 2814 # JV-01 */
    if ((ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enTxInterruptMode) && (0UL != LulIntStat))
    {
      /* Clear interrupt status */
      LpAxiRegs->stTXDI[0].ulTDISi = LulIntStat;
      Eth_HwTxConfirmation(LulCtrlIdx);
    } /* else: No action required */
    #endif /* ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON */

    #if (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
    LulIntStat = LpAxiRegs->stRXDI[0].ulRDISi &
      ETH_ETNDE_DIS_MASK(LulCtrlIdx, ETH_DATAISR0, LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue);
    if ((ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enRxInterruptMode) && (0UL != LulIntStat))
    {
      Eth_HwReceive(LulCtrlIdx, 0UL);                                                                                   /* PRQA S 3200 # JV-01 */
    } /* else: No action required */
    #endif /* ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON */
  }
}
#endif /* ETH_CONTROLLER_DATA0_INTERRUPT_ON */

/***********************************************************************************************************************
** Function Name         : Eth_DataIsr
**
** Service ID            : Not Applicable
**
** Description           : Common part of each  ETH_DATAISR.
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx: The index of the controller
**                       : LulIsr: The index of the interrupt handler
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : The availability of the index must be guaranteed
**                         by the upper layer.
**
** Global Variables Used : Eth_GaaETND_EICRegs, Eth_GaaDemEventIntInconsistent,
**                         Eth_GstETND_Regs, Eth_GaaETNE_EICRegs, Eth_GpETNE_MFWDRegs,
**                         Eth_GpCtrlConfigPtr, Eth_GpEthConfigPtr
**
** Function(s) invoked   : Eth_DemConfigCheck, Eth_HwTxConfirmation, Eth_HwReceive
**
** Registers Used        : EICn, TDISi, RDISi
**
** Reference ID          : ETH_DUD_ACT_349
** Reference ID          : ETH_DUD_ACT_349_ERR001, ETH_DUD_ACT_349_REG001
***********************************************************************************************************************/
#ifdef ETH_CONTROLLER_DATA_INTERRUPT_ON
STATIC FUNC(void, ETH_CODE_FAST) Eth_DataIsr(CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulIsr)
{
  P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;
  uint32 LulIntStat;

  #if (ETH_USING_MACRO == ETH_MACRO_ETND)
  CONSTP2VAR(volatile Eth_ETNDE_AXIBMIRegType, AUTOMATIC, REGSPACE) LpAxiRegs =
    &Eth_GstETND_Regs.pES[LulCtrlIdx]->stAXIBMI;                                                                        /* PRQA S 2844 # JV-01 */
  #else
  CONSTP2VAR(volatile Eth_ETNDE_AXIBMIRegType, AUTOMATIC, REGSPACE) LpAxiRegs = Eth_GstETNE_Regs.pAXIBMI;
  #endif

  LpHwUnitConfig =
    (P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA))Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;              /* PRQA S 0316 # JV-01 */

  #if (ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if (((ETH_EIC_CONSISTENCY_MASK & ~ETH_EIC_EIMK_MASK) !=
     #if (ETH_USING_MACRO == ETH_MACRO_ETND)
     ((*((uint16 *)Eth_GaaETND_EICRegs[LulCtrlIdx] + LulIsr)) & ETH_EIC_CONSISTENCY_MASK))                              /* PRQA S 0310, 0312, 2844, 0488, 2824 # JV-01, JV-01, JV-01, JV-01, JV-01 */
     #else
     ((*((uint16 *)Eth_GpETNE_EICRegs + LulIsr)) & ETH_EIC_CONSISTENCY_MASK))
     #endif
  #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON)
    || ((0UL != ((LpAxiRegs->stTXDI[0].ulTDISi & LpHwUnitConfig->stQueueConfig.ulTxIntConfig)                           /* PRQA S 2814 # JV-01 */
     & (uint32)(~ETH_ETNDE_DIS_MASK(LulCtrlIdx, LulIsr, LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue))))
    && (ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enTxInterruptMode))
  #endif
  #if (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
    || ((0UL != ((LpAxiRegs->stRXDI[0].ulRDISi & LpHwUnitConfig->stQueueConfig.ulRxIntConfig)
     & (uint32)(~ETH_ETNDE_DIS_MASK(LulCtrlIdx, LulIsr, LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue))))
    && (ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enRxInterruptMode))
  #endif
    )
  {
    Eth_DemConfigCheck(Eth_GaaDemEventIntInconsistent[LulCtrlIdx], DEM_EVENT_STATUS_FAILED);                            /* PRQA S 2844 # JV-01 */
  }
  else
  #endif /* ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON */
  {
    #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON)
    LulIntStat = LpAxiRegs->stTXDI[0].ulTDISi &
      ETH_ETNDE_DIS_MASK(LulCtrlIdx, LulIsr, LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue);
    if ((ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enTxInterruptMode) && (0UL != LulIntStat))
    {
      /* Clear interrupt status */
      LpAxiRegs->stTXDI[0].ulTDISi = LulIntStat;
      Eth_HwTxConfirmation(LulCtrlIdx);
    } /* else: No action required */
    #endif /* ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON */

    #if (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
    LulIntStat = LpAxiRegs->stRXDI[0].ulRDISi &
      ETH_ETNDE_DIS_MASK(LulCtrlIdx, LulIsr, LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue);
    if ((ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enRxInterruptMode) && (0UL != LulIntStat))
    {
      Eth_HwReceive(LulCtrlIdx, 0UL);                                                                                   /* PRQA S 3200 # JV-01 */
    } /* else: No action required */
    #endif /* ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON */
  }
}
#endif /* ETH_CONTROLLER_DATA_INTERRUPT_ON */

/***********************************************************************************************************************
** Function Name         : Eth_GeneralIsr
**
** Service ID            : NA
**
** Description           : General Interrupt Service Handler for ETND/ETNE controllers
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETND_EICRegs, Eth_GaaDemEventIntInconsistent,
**                         Eth_GaaDemEventDmaError, Eth_GstETND_Regs
**                         Eth_GaaDemEventEccError
**
** Function(s) invoked   : Eth_DemConfigCheck
**
** Registers Used        : TCIS, TIS2, RIS, EIS0, EICn, ETNzSGINTS, SGOPMS, MPIC
**
** Reference ID          : ETH_DUD_ACT_350
** Reference ID          : ETH_DUD_ACT_350_ERR001, ETH_DUD_ACT_350_ERR002, ETH_DUD_ACT_350_ERR003
** Reference ID          : ETH_DUD_ACT_350_ERR004, ETH_DUD_ACT_350_ERR005, ETH_DUD_ACT_350_ERR006
** Reference ID          : ETH_DUD_ACT_350_REG001, ETH_DUD_ACT_350_REG002, ETH_DUD_ACT_350_REG003
** Reference ID          : ETH_DUD_ACT_350_REG004, ETH_DUD_ACT_350_REG005, ETH_DUD_ACT_350_REG006
** Reference ID          : ETH_DUD_ACT_350_REG007
***********************************************************************************************************************/
#ifdef ETH_CONTROLLER_GENERAL_INTERRUPT_ON
STATIC FUNC(void, ETH_CODE_FAST) Eth_GeneralIsr(CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  #if ((ETH_TSN0_SGMII_ISR == STD_ON) || (ETH_TSN1_SGMII_ISR == STD_ON) || (ETH_RSW_SGMII_ISR == STD_ON))
  uint32 LulLinkState;
  uint32 LulRegVal;
  #if (ETH_USING_MACRO == ETH_MACRO_ETND)
  CONSTP2VAR(volatile Eth_ETNDE_SGMIIRegType, AUTOMATIC, REGSPACE) LpSgmiiRegs = Eth_GstETND_Regs.pSGMII;
  CONSTP2VAR(volatile Eth_ETNDE_RMACRegType, AUTOMATIC, REGSPACE) LpRmacRegs =
    &Eth_GstETND_Regs.pES[LulCtrlIdx]->stRMACSys.stRMAC;
  #else
  CONSTP2VAR(volatile Eth_ETNDE_SGMIIRegType, AUTOMATIC, REGSPACE) LpSgmiiRegs = Eth_GstETNE_Regs.pSGMII[LulCtrlIdx];
  CONSTP2VAR(volatile Eth_ETNDE_RMACRegType, AUTOMATIC, REGSPACE) LpRmacRegs =
    &Eth_GstETNE_Regs.pEA[LulCtrlIdx]->stRMACSys.stRMAC;
  #endif
  #endif

  #if (ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if (((ETH_EIC_CONSISTENCY_MASK & ~ETH_EIC_EIMK_MASK) !=
    #if (ETH_USING_MACRO == ETH_MACRO_ETND)
    (Eth_GaaETND_EICRegs[LulCtrlIdx]->usINTETNDw6 & ETH_EIC_CONSISTENCY_MASK))                                          /* PRQA S 2844, 2814 # JV-01, JV-01 */
    || ((0UL == (Eth_GstETND_Regs.pES[LulCtrlIdx]->stMHD.ulTCIS    & (uint32)ETH_ETND_TCIS_ERROR_MASK))
    &&  (0UL == (Eth_GstETND_Regs.pES[LulCtrlIdx]->stMHD.ulTIS2    & (uint32)ETH_ETND_TIS2_ERROR_MASK))
    &&  (0UL == (Eth_GstETND_Regs.pES[LulCtrlIdx]->stMHD.ulRIS     & (uint32)ETH_ETND_RIS_ERROR_MASK))
    &&  (0UL == (Eth_GstETND_Regs.pES[LulCtrlIdx]->stAXIBMI.ulEIS0 & (uint32)ETH_ETND_EIS0_ECCFES_MASK))
    &&  (0UL == (Eth_GstETND_Regs.pES[LulCtrlIdx]->stAXIBMI.ulEIS0 & (uint32)ETH_ETND_EIS0_AEIS_MASK))
    &&  (0UL == (Eth_GstETND_Regs.pSGMII->ulETNDEzSGINTS           & (uint32)ETH_ETNDE_SGINTS_RDY_UNRDY)))
    #else
    (Eth_GpETNE_EICRegs->usINTETNE08 & ETH_EIC_CONSISTENCY_MASK))
    || ((0UL == (Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulEAEIS0 & (uint32)ETH_ETNE_EAEIE0_ERROR_MASK))
    &&  (0UL == (Eth_GstETNE_Regs.pSGMII[LulCtrlIdx]->ulETNDEzSGINTS & ETH_ETNDE_SGINTS_RDY_UNRDY)))
    #endif
    )
  {
    Eth_DemConfigCheck(Eth_GaaDemEventIntInconsistent[LulCtrlIdx], DEM_EVENT_STATUS_FAILED);                            /* PRQA S 2844 # JV-01 */
  }
  else
  #endif
  {
    #if ((ETH_TSN0_SGMII_ISR == STD_ON) || (ETH_TSN1_SGMII_ISR == STD_ON) || (ETH_RSW_SGMII_ISR == STD_ON))
    LulRegVal = LpSgmiiRegs->ulETNDEzSGINTS;
    LulLinkState = LpSgmiiRegs->ulETNDEzSGOPMS;
    if ((ETH_ETNDE_SGINTS_RDY == LulRegVal) ||
       ((ETH_ETNDE_SGINTS_RDY_UNRDY == LulRegVal) && (ETH_SGOPMS_DATA_COM == (LulLinkState & ETH_SGOPMS_XMIT_MASK))))
    {
      LpRmacRegs->ulMPIC = (LpRmacRegs->ulMPIC & ~ETH_SGOPMS_SPEED_MASK) | (LulLinkState & ETH_SGOPMS_SPEED_MASK);
    }
    else
    {
      LpRmacRegs->ulMPIC &= ~ETH_SGOPMS_SPEED_MASK;
    }
    LpSgmiiRegs->ulETNDEzSGINTS = LulRegVal;
    #endif

    #if (ETH_USING_MACRO == ETH_MACRO_ETND)
    if (0UL != (Eth_GstETND_Regs.pES[LulCtrlIdx]->stMHD.ulTCIS & (uint32)ETH_ETND_TCIS_ERROR_MASK))
    {
      Eth_DemConfigCheck(Eth_GaaDemEventEccError[LulCtrlIdx], DEM_EVENT_STATUS_FAILED);                                 /* PRQA S 2844 # JV-01 */

      /* Clear Error Flag */
      Eth_GstETND_Regs.pES[LulCtrlIdx]->stMHD.ulTCIS &= (uint32)(~ETH_ETND_TCIS_ERROR_MASK);
    } /* else: No action required */

    if (0UL != (Eth_GstETND_Regs.pES[LulCtrlIdx]->stMHD.ulTIS2 & (uint32)ETH_ETND_TIS2_ERROR_MASK))
    {
      Eth_DemConfigCheck(Eth_GaaDemEventEccError[LulCtrlIdx], DEM_EVENT_STATUS_FAILED);                                 /* PRQA S 2844 # JV-01 */

      /* Clear Error Flag */
      Eth_GstETND_Regs.pES[LulCtrlIdx]->stMHD.ulTIS2 &= (uint32)(~ETH_ETND_TIS2_ERROR_MASK);
    } /* else: No action required */

    if (0UL != (Eth_GstETND_Regs.pES[LulCtrlIdx]->stMHD.ulRIS & (uint32)ETH_ETND_RIS_ERROR_MASK))
    {
      Eth_DemConfigCheck(Eth_GaaDemEventEccError[LulCtrlIdx], DEM_EVENT_STATUS_FAILED);                                 /* PRQA S 2844 # JV-01 */

      /* Clear Error Flag */
      Eth_GstETND_Regs.pES[LulCtrlIdx]->stMHD.ulRIS &= (uint32)(~ETH_ETND_RIS_ERROR_MASK);
    } /* else: No action required */

    if (0UL != (Eth_GstETND_Regs.pES[LulCtrlIdx]->stAXIBMI.ulEIS0 & (uint32)ETH_ETND_EIS0_ECCFES_MASK))                 /* PRQA S 2814 # JV-01 */
    {
      Eth_DemConfigCheck(Eth_GaaDemEventEccError[LulCtrlIdx], DEM_EVENT_STATUS_FAILED);                                 /* PRQA S 2844 # JV-01 */

      /* Clear Error Flag */
      Eth_GstETND_Regs.pES[LulCtrlIdx]->stAXIBMI.ulEIS0 |= (uint32)ETH_ETND_EIS0_ECCFES_MASK;
    } /* else: No action required */

    if (0UL != (Eth_GstETND_Regs.pES[LulCtrlIdx]->stAXIBMI.ulEIS0 & (uint32)ETH_ETND_EIS0_AEIS_MASK))
    {
      Eth_DemConfigCheck(Eth_GaaDemEventDmaError[LulCtrlIdx], DEM_EVENT_STATUS_FAILED);                                 /* PRQA S 2844 # JV-01 */

      /* Clear Error Flag */
      Eth_GstETND_Regs.pES[LulCtrlIdx]->stAXIBMI.ulEIS0 |= (uint32)ETH_ETND_EIS0_AEIS_MASK;
    } /* else: No action required */

    /* Execute the pipeline to avoid multiple interrupts */
    (void)Eth_GstETND_Regs.pES[LulCtrlIdx]->stAXIBMI.ulEIS0;
    EXECUTE_SYNCP();                                                                                                    /* PRQA S 1006 # JV-01 */

    #else
    if (0UL != (Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulEAEIS0 & (uint32)ETH_ETNE_EAEIE0_ERROR_MASK))
    {
      Eth_DemConfigCheck(Eth_GaaDemEventEccError[LulCtrlIdx], DEM_EVENT_STATUS_FAILED);                                 /* PRQA S 2844 # JV-01 */

      /* Clear Error Flag */
      Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulEAEIS0 |= (uint32)ETH_ETNE_EAEIE0_ERROR_MASK;
    } /* else: No action required */

    /* Execute the pipeline to avoid multiple interrupts */
    (void)Eth_GstETNE_Regs.pEA[LulCtrlIdx]->ulEAEIS0;
    EXECUTE_SYNCP();                                                                                                    /* PRQA S 1006 # JV-01 */
    #endif /* ETH_USING_MACRO == ETH_MACRO_ETND */
  }
}
#endif /* ETH_CONTROLLER_GENERAL_INTERRUPT_ON */

/***********************************************************************************************************************
** Function Name         : ETH_TSN0_DATAISR0
**
** Service ID            : NA
**
** Description           : Data Interrupt Service Handler
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpETNE_MFWDRegs, Eth_GpCtrlConfigPtr
**
** Function(s) invoked   : Eth_DataIsr0
**
** Registers Used        : TDISi, RDISi
**
** Reference ID          : ETH_DUD_ACT_351
***********************************************************************************************************************/
#if ((ETH_TSN0_DATA0_ISR == STD_ON) || (ETH_RSW_DATA0_ISR == STD_ON))
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_TSN0_DATAISR0_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_TSN0_DATAISR0_CAT2)                                                                                             /* PRQA S 1503, 3408 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_TSN0_DATAISR0(void)                                                           /* PRQA S 1503 # JV-01 */
#endif
{
  #if (ETH_USING_MACRO == ETH_MACRO_ETNE)
  P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;

  LpHwUnitConfig =
    (P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA))Eth_GpCtrlConfigPtr[0UL].pHwUnitConfig;

  if ((0UL != (Eth_GstETNE_Regs.pAXIBMI->stTXDI[0].ulTDISi & LpHwUnitConfig->stQueueConfig.ulTxIntConfig)) ||
      (0UL != (Eth_GstETNE_Regs.pAXIBMI->stRXDI[0].ulRDISi & LpHwUnitConfig->stQueueConfig.ulRxIntConfig)))
  {
    Eth_DataIsr0((uint32)ETHTSN0);
  } /* else: No action required */

  if ((0UL != (Eth_GstETNE_Regs.pAXIBMI->stTXDI[0].ulTDISi >> LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue)) ||
      (0UL != (Eth_GstETNE_Regs.pAXIBMI->stRXDI[0].ulRDISi >> LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue)))
  {
    Eth_DataIsr0((uint32)ETHTSN1);
  } /* else: No action required */

  #else
  Eth_DataIsr0((uint32)ETHTSN0);
  #endif /* ETH_USING_MACRO == ETH_MACRO_ETNE */
}
#endif /* ETH_TSN0_DATA0_ISR == STD_ON */

/***********************************************************************************************************************
** Function Name         : ETH_TSN1_DATAISR0
**
** Service ID            : NA
**
** Description           : Data Interrupt Service Handler (only for ETND)
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : None
**
** Function(s) invoked   : Eth_DataIsr0
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_352
***********************************************************************************************************************/
#if (ETH_TSN1_DATA0_ISR == STD_ON)
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_TSN1_DATAISR0_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_TSN1_DATAISR0_CAT2)
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_TSN1_DATAISR0(void)
#endif
{
  Eth_DataIsr0((uint32)ETHTSN1);
}
#endif /* ETH_TSN1_DATA0_ISR == STD_ON */

/***********************************************************************************************************************
** Function Name         : ETH_TSN0_GENERALISR
**
** Service ID            : NA
**
** Description           : General Interrupt Service Handler for ETND controllers
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : None
**
** Function(s) invoked   : Eth_GeneralIsr
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_353
***********************************************************************************************************************/
#if (ETH_TSN0_GENERAL_ISR == STD_ON)
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_TSN0_GENERALISR_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_TSN0_GENERALISR_CAT2)                                                                                           /* PRQA S 1503, 3408 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_TSN0_GENERALISR(void)                                                         /* PRQA S 1503 # JV-01 */
#endif
{
  Eth_GeneralIsr((uint32)ETHTSN0);
}
#endif /* ETH_TSN0_GENERAL_ISR == STD_ON */

/***********************************************************************************************************************
** Function Name         : ETH_TSN1_GENERALISR
**
** Service ID            : NA
**
** Description           : General Interrupt Service Handler for ETND controllers
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : None
**
** Function(s) invoked   : Eth_GeneralIsr
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_354
***********************************************************************************************************************/
#if (ETH_TSN1_GENERAL_ISR == STD_ON)
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_TSN1_GENERALISR_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_TSN1_GENERALISR_CAT2)
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_TSN1_GENERALISR(void)
#endif
{
  Eth_GeneralIsr((uint32)ETHTSN1);
}
#endif /* ETH_TSN1_GENERAL_ISR == STD_ON */

/***********************************************************************************************************************
** Function Name         : ETH_RSW_GENERALISR1
**
** Service ID            : NA
**
** Description           : General Interrupt Service Handler for ETNE controllers
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : None
**
** Function(s) invoked   : Eth_GeneralIsr
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_364
***********************************************************************************************************************/
#if (ETH_RSW_GENERAL_ISR == STD_ON)
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_RSW_GENERALISR1_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_RSW_GENERALISR1_CAT2)                                                                                           /* PRQA S 1503, 3408 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_RSW_GENERALISR1(void)                                                         /* PRQA S 1503 # JV-01 */
#endif
{
  if ((0UL != (Eth_GstETNE_Regs.pSGMII[(uint32)ETHTSN0]->ulETNDEzSGINTS & ETH_ETNDE_SGINTS_RDY_UNRDY)) ||
       0UL != (Eth_GstETNE_Regs.pEA[(uint32)ETHTSN0]->ulEAEIS0 & (uint32)ETH_ETNE_EAEIE0_ERROR_MASK))
  {
    Eth_GeneralIsr((uint32)ETHTSN0);
  } /* else: No action required */

  if ((0UL != (Eth_GstETNE_Regs.pSGMII[(uint32)ETHTSN1]->ulETNDEzSGINTS & ETH_ETNDE_SGINTS_RDY_UNRDY)) ||
       0UL != (Eth_GstETNE_Regs.pEA[(uint32)ETHTSN1]->ulEAEIS0 & (uint32)ETH_ETNE_EAEIE0_ERROR_MASK))
  {
    Eth_GeneralIsr((uint32)ETHTSN1);
  } /* else: No action required */
}
#endif /* (ETH_TSN0_GENERAL_ISR == STD_ON) || (ETH_TSN1_GENERAL_ISR == STD_ON) && (ETH_USING_MACRO == ETH_MACRO_ETNE) */

/***********************************************************************************************************************
** Function Name         : ETH_RSW_GENERALISR2
**
** Service ID            : NA
**
** Description           : General Interrupt Service Handler for ETNE controllers
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : None
**
** Function(s) invoked   : None
**
** Registers Used        : EICn, FWEIS1
**
** Reference ID          : ETH_DUD_ACT_365
** Reference ID          : ETH_DUD_ACT_365_ERR001, ETH_DUD_ACT_365_ERR002, ETH_DUD_ACT_365_REG001
***********************************************************************************************************************/
#if (ETH_RSW_GENERAL_ISR == STD_ON)
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_RSW_GENERALISR2_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_RSW_GENERALISR2_CAT2)                                                                                           /* PRQA S 1503, 3408 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_RSW_GENERALISR2(void)                                                         /* PRQA S 1503 # JV-01 */
#endif
{
  #if (ETH_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if (((ETH_EIC_CONSISTENCY_MASK & ~ETH_EIC_EIMK_MASK) !=
       (Eth_GpETNE_EICRegs->usINTETNE07 & ETH_EIC_CONSISTENCY_MASK)) ||
       (0UL == (Eth_GstETNE_Regs.pMFWD->ulFWEIS1 & (uint32)ETH_ETNE_FWEIS1_ERROR_MASK)))
  {
    Eth_DemConfigCheck(Eth_GaaDemEventIntInconsistent[0UL], DEM_EVENT_STATUS_FAILED);                                   /* PRQA S 2844 # JV-01 */
  }
  else
  #endif
  {
    if (0UL != (Eth_GstETNE_Regs.pMFWD->ulFWEIS1 & (uint32)ETH_ETNE_FWEIS1_ERROR_MASK))
    {
      Eth_DemConfigCheck(Eth_GaaDemEventEccError[0UL], DEM_EVENT_STATUS_FAILED);                                        /* PRQA S 2844 # JV-01 */

      /* Clear Error Flag */
      Eth_GstETNE_Regs.pMFWD->ulFWEIS1 |= (uint32)ETH_ETNE_FWEIS1_ERROR_MASK;

      /* Execute the pipeline to avoid multiple interrupts */
      (void)Eth_GstETNE_Regs.pMFWD->ulFWEIS1;
      EXECUTE_SYNCP();                                                                                                  /* PRQA S 1006 # JV-01 */
    } /* else: No action required */
  }
}
#endif /* (ETH_TSN0_GENERAL_ISR == STD_ON) || (ETH_TSN1_GENERAL_ISR == STD_ON) && (ETH_USING_MACRO == ETH_MACRO_ETNE) */

/***********************************************************************************************************************
** Function Name         : ETH_TSN0_DATAISR1
**
** Service ID            : NA
**
** Description           : Data Interrupt Service Handler
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpETNE_MFWDRegs, Eth_GpCtrlConfigPtr
**
** Function(s) invoked   : Eth_DataIsr
**
** Registers Used        : TDISi, RDISi
**
** Reference ID          : ETH_DUD_ACT_355
***********************************************************************************************************************/
#if ((ETH_TSN0_DATA1_ISR == STD_ON) || (ETH_RSW_DATA1_ISR == STD_ON))
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_TSN0_DATAISR1_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_TSN0_DATAISR1_CAT2)                                                                                             /* PRQA S 1503, 3408 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_TSN0_DATAISR1(void)                                                           /* PRQA S 1503 # JV-01 */
#endif
{
  #if (ETH_USING_MACRO == ETH_MACRO_ETNE)
  P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;

  LpHwUnitConfig =
    (P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA))Eth_GpCtrlConfigPtr[0UL].pHwUnitConfig;

  if ((0UL != (Eth_GstETNE_Regs.pAXIBMI->stTXDI[0].ulTDISi & LpHwUnitConfig->stQueueConfig.ulTxIntConfig)) ||
      (0UL != (Eth_GstETNE_Regs.pAXIBMI->stRXDI[0].ulRDISi & LpHwUnitConfig->stQueueConfig.ulRxIntConfig)))
  {
    Eth_DataIsr((uint32)ETHTSN0, ETH_DATAISR1);
  } /* else: No action required */

  if ((0UL != (Eth_GstETNE_Regs.pAXIBMI->stTXDI[0].ulTDISi >> LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue)) ||
      (0UL != (Eth_GstETNE_Regs.pAXIBMI->stRXDI[0].ulRDISi >> LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue)))
  {
    Eth_DataIsr((uint32)ETHTSN1, ETH_DATAISR1);
  } /* else: No action required */

  #else
  Eth_DataIsr((uint32)ETHTSN0, ETH_DATAISR1);
  #endif /* ETH_USING_MACRO == ETH_MACRO_ETNE */
}
#endif /* ETH_TSN0_DATA1_ISR == STD_ON */

/***********************************************************************************************************************
** Function Name         : ETH_TSN0_DATAISR2
**
** Service ID            : NA
**
** Description           : Data Interrupt Service Handler
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpETNE_MFWDRegs, Eth_GpCtrlConfigPtr
**
** Function(s) invoked   : Eth_DataIsr
**
** Registers Used        : TDISi, RDISi
**
** Reference ID          : ETH_DUD_ACT_357
***********************************************************************************************************************/
#if ((ETH_TSN0_DATA2_ISR == STD_ON) || (ETH_RSW_DATA2_ISR == STD_ON))
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_TSN0_DATAISR2_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_TSN0_DATAISR2_CAT2)                                                                                             /* PRQA S 1503, 3408 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_TSN0_DATAISR2(void)                                                           /* PRQA S 1503 # JV-01 */
#endif
{
  #if (ETH_USING_MACRO == ETH_MACRO_ETNE)
  P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;

  LpHwUnitConfig =
    (P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA))Eth_GpCtrlConfigPtr[0UL].pHwUnitConfig;

  if ((0UL != (Eth_GstETNE_Regs.pAXIBMI->stTXDI[0].ulTDISi & LpHwUnitConfig->stQueueConfig.ulTxIntConfig)) ||
      (0UL != (Eth_GstETNE_Regs.pAXIBMI->stRXDI[0].ulRDISi & LpHwUnitConfig->stQueueConfig.ulRxIntConfig)))
  {
    Eth_DataIsr((uint32)ETHTSN0, ETH_DATAISR2);
  } /* else: No action required */

  if ((0UL != (Eth_GstETNE_Regs.pAXIBMI->stTXDI[0].ulTDISi >> LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue)) ||
      (0UL != (Eth_GstETNE_Regs.pAXIBMI->stRXDI[0].ulRDISi >> LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue)))
  {
    Eth_DataIsr((uint32)ETHTSN1, ETH_DATAISR2);
  } /* else: No action required */

  #else
  Eth_DataIsr((uint32)ETHTSN0, ETH_DATAISR2);
  #endif /* ETH_USING_MACRO == ETH_MACRO_ETNE */
}
#endif /* (ETH_TSN0_DATA2_ISR == STD_ON) */

/***********************************************************************************************************************
** Function Name         : ETH_TSN0_DATAISR3
**
** Service ID            : NA
**
** Description           : Data Interrupt Service Handler
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpETNE_MFWDRegs, Eth_GpCtrlConfigPtr
**
** Function(s) invoked   : Eth_DataIsr
**
** Registers Used        : TDISi, RDISi
**
** Reference ID          : ETH_DUD_ACT_359
***********************************************************************************************************************/
#if ((ETH_TSN0_DATA3_ISR == STD_ON) || (ETH_RSW_DATA3_ISR == STD_ON))
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_TSN0_DATAISR3_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_TSN0_DATAISR3_CAT2)                                                                                             /* PRQA S 1503, 3408 # JV-01, JV-01 */
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_TSN0_DATAISR3(void)                                                           /* PRQA S 1503 # JV-01 */
#endif
{
  #if (ETH_USING_MACRO == ETH_MACRO_ETNE)
  P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;

  LpHwUnitConfig =
    (P2CONST(Eth_ETNDEConfigType, AUTOMATIC, ETH_APPL_DATA))Eth_GpCtrlConfigPtr[0UL].pHwUnitConfig;

  if ((0UL != (Eth_GstETNE_Regs.pAXIBMI->stTXDI[0].ulTDISi & LpHwUnitConfig->stQueueConfig.ulTxIntConfig)) ||
      (0UL != (Eth_GstETNE_Regs.pAXIBMI->stRXDI[0].ulRDISi & LpHwUnitConfig->stQueueConfig.ulRxIntConfig)))
  {
    Eth_DataIsr((uint32)ETHTSN0, ETH_DATAISR3);
  } /* else: No action required */

  if ((0UL != (Eth_GstETNE_Regs.pAXIBMI->stTXDI[0].ulTDISi >> LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue)) ||
      (0UL != (Eth_GstETNE_Regs.pAXIBMI->stRXDI[0].ulRDISi >> LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue)))
  {
    Eth_DataIsr((uint32)ETHTSN1, ETH_DATAISR3);
  } /* else: No action required */

  #else
  Eth_DataIsr((uint32)ETHTSN0, ETH_DATAISR3);
  #endif /* ETH_USING_MACRO == ETH_MACRO_ETNE */
}
#endif /* (ETH_TSN0_DATA3_ISR == STD_ON) */

/***********************************************************************************************************************
** Function Name         : ETH_TSN1_DATAISR1
**
** Service ID            : NA
**
** Description           : Data Interrupt Service Handler (only for ETND)
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : None
**
** Function(s) invoked   : Eth_DataIsr
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_356
***********************************************************************************************************************/
#if (ETH_TSN1_DATA1_ISR == STD_ON)
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_TSN1_DATAISR1_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_TSN1_DATAISR1_CAT2)
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_TSN1_DATAISR1(void)
#endif
{
  Eth_DataIsr((uint32)ETHTSN1, ETH_DATAISR1);
}
#endif /* (ETH_TSN1_DATA1_ISR == STD_ON) */

/***********************************************************************************************************************
** Function Name         : ETH_TSN1_DATAISR2
**
** Service ID            : NA
**
** Description           : Data Interrupt Service Handler (only for ETND)
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : None
**
** Function(s) invoked   : Eth_DataIsr
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_358
***********************************************************************************************************************/
#if (ETH_TSN1_DATA2_ISR == STD_ON)
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_TSN1_DATAISR2_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_TSN1_DATAISR2_CAT2)
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_TSN1_DATAISR2(void)
#endif
{
  Eth_DataIsr((uint32)ETHTSN1, ETH_DATAISR2);
}
#endif /* (ETH_TSN0_DATA2_ISR == STD_ON) */

/***********************************************************************************************************************
** Function Name         : ETH_TSN1_DATAISR3
**
** Service ID            : NA
**
** Description           : Data Interrupt Service Handler (only for ETND)
**
** Sync/Async            : NA
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : None
**
** Function(s) invoked   : Eth_DataIsr
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_360
***********************************************************************************************************************/
#if (ETH_TSN1_DATA3_ISR == STD_ON)
/* Defines the CAT2 interrupt mapping */
#if defined (Os_ETH_TSN1_DATAISR3_CAT2) || (ETH_ISR_CATEGORY_2 == STD_ON)
ISR(ETH_TSN1_DATAISR3_CAT2)
#else
_INTERRUPT_ FUNC(void, ETH_CODE_FAST) ETH_TSN1_DATAISR3(void)
#endif
{
  Eth_DataIsr((uint32)ETHTSN1, ETH_DATAISR3);
}
#endif /* (ETH_TSN0_DATA3_ISR == STD_ON) */

#define ETH_STOP_SEC_CODE_FAST
#include "Eth_MemMap.h"
/***********************************************************************************************************************
**                                           End of File                                                              **
***********************************************************************************************************************/
