/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Eth_Common_LLDriver.h                                                                               */
/* SW-VERSION   = 1.6.0                                                                                               */
/*====================================================================================================================*/
/*                                                     COPYRIGHT                                                      */
/*====================================================================================================================*/
/* (c) 2020-2022 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* This file contains common definitions among all low level drivers of Eth                                           */
/* Driver Component.                                                                                                  */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s) and/or */
/* the Application.                                                                                                   */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs of  */
/* program errors, compliance with applicable laws, damage to or loss of data, programs or equipment, and             */
/* unavailability or interruption of operations.                                                                      */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*              Devices:        X2x                                                                                   */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                              Revision Control History                                              **
***********************************************************************************************************************/
/*
 * 1.5.3: 05/04/2022    : Add control index to parameter of Eth_HwCheckFifoIndex.
 *                        Add function prototype for Eth_HwDeInit.
 * 1.5.2: 03/03/2022    : Add control index to parameter of Eth_HwGetTxErrorCounterValues.
 *        28/01/2022    : Move the ETH_HEADER_SIZE, ETH_MACADDR_SIZE, ETH_SRC_DST_ADDRESS_SIZE,
 *                        ETH_ETHERTYPE_SIZE, ETH_VLAN_SIZE to Eth_Types.h
 *        25/01/2022    : Changed the management variable of the ring buffer of Tx buffer from uint8 to uint32.
 * 1.5.1: 14/12/2021    : Changed ON / OFF of function definition of Eth_HwReceive for support U2Bx devices.
 *        13/12/2021    : Add ulTsDescTail member variable to Eth_ControllerStatusType.
 *        28/10/2021    : Update include file for support U2Bx devices.
 *        12/01/2021    : Update QA-C 9.5.0 comments
 * 1.4.2: 24/08/2021    : Modify the format to 120 characters
 *        20/08/2021    : Update include file for support U2Bx devices.
 * 1.4.1: 07/05/2021    : Changes to support U2Bx devices.
 *                        Remove vendor ID from the file name, API names
 *                        and parameters according to BSW00347 requirement
 *        09/06/2021    : Remove QA-C Warning 3432 according to QA-C 9.5.0
 *        06/07/2021    : Updated QA-C 9.5.0 comments.
 * 1.2.0: 13/07/2020    : Release
 *        29/07/2020    : Add QAC 9.3.1 comment.
 * 1.1.0: 19/06/2020    : Release
 * 1.0.1: 04/06/2020    : To support Transmit/Receive interrupts for each controller.
 *                        Added ETH_59_COPY_MAC_ADDRESS macro.
 *                        Moved ETNB specific function.
 *                        Update for improvement by the QA-C.
 * 1.0.0: 25/03/2020    : Initial Version
 */
/**********************************************************************************************************************/
#ifndef ETH_COMMON_LLDRIVER_H
#define ETH_COMMON_LLDRIVER_H

/***********************************************************************************************************************
**                                               Macro type definitions                                               **
***********************************************************************************************************************/

/***********************************************************************************************************************
**                                                  Include Section                                                   **
***********************************************************************************************************************/
#include "Eth_Types.h"
#include "Eth_Filter.h"
/* Include the LLDriver header file */
#if (ETH_USING_MACRO == ETH_MACRO_ETNC)
#include "Eth_ETNC_LLDriver.h"
#elif (ETH_USING_MACRO == ETH_MACRO_ETNB)
#include "Eth_ETNB_LLDriver.h"
#elif (ETH_USING_MACRO == ETH_MACRO_ETND)
#include "Eth_ETNDE_LLDriver.h"
#include "Eth_ETND_LLDriver.h"
#elif (ETH_USING_MACRO == ETH_MACRO_ETNE)
#include "Eth_ETNDE_LLDriver.h"
#include "Eth_ETNE_LLDriver.h"
#endif

/***********************************************************************************************************************
**                                                General definitions                                                 **
***********************************************************************************************************************/
/* Bit length of byte */
#define ETH_BYTE_BITS            8UL

/* Maximum value of uint32 */
#define ETH_UINT32_MAXVALUE      0xFFFFFFFFUL

/* EIMK bit mask of EICn register */
#define ETH_EIC_EIMK_MASK        (uint16)0x0080U
/* EIRF bit mask of EICn register */
#define ETH_EIC_EIRF_MASK        (uint16)0x1000U
/* EIRF and EIMK bit mask of EICn register */
#define ETH_EIC_CONSISTENCY_MASK (uint16)(ETH_EIC_EIRF_MASK | ETH_EIC_EIMK_MASK)

/* Copy the MAC address */
#define ETH_COPY_MAC_ADDRESS(src, dst) \
  do                                   \
  {                                    \
    (dst)[0] = (src)[0];               \
    (dst)[1] = (src)[1];               \
    (dst)[2] = (src)[2];               \
    (dst)[3] = (src)[3];               \
    (dst)[4] = (src)[4];               \
    (dst)[5] = (src)[5];               \
  } while (0)                                                                                                           /* PRQA S 3472 # JV-01 */

/***********************************************************************************************************************
**                                      PHY management data reletive definitions                                      **
***********************************************************************************************************************/
#define ETH_PHY_MAX_PHYAD_IDX    31U
#define ETH_PHY_MAX_REGAD_IDX    31U
#define ETH_PHY_PREAMBLE_SIZE    32UL
#define ETH_PHY_HEADER_SIZE      14UL
#define ETH_PHY_DATA_SIZE        16UL
/* Create a header for read operation */
#define ETH_PHY_RHEADER(phyad, regad) \
  (0x1800UL | ((uint32)(phyad) << 5UL) | ((uint32)(regad)))                                                             /* PRQA S 3472 # JV-01 */
/* Create a header for write operation */
#define ETH_PHY_WHEADER(phyad, regad) \
  (0x1400UL | ((uint32)(phyad) << 5UL) | ((uint32)(regad)))                                                             /* PRQA S 3472 # JV-01 */

/***********************************************************************************************************************
**                                             MDIO reletive definitions                                              **
***********************************************************************************************************************/
#define ETH_MDIO_SETUP_TIME      10UL

/***********************************************************************************************************************
**                                      Offset addressed from the top of buffer                                       **
**                                  These are depending on each MACRO specification                                   **
***********************************************************************************************************************/
/* Destination MAC address */
#define ETH_DST_MACADDR_OFFSET ETH_TXRX_BUFFER_PRE_PADDING
/* Src MAC address */
#define ETH_SRC_MACADDR_OFFSET   (ETH_TXRX_BUFFER_PRE_PADDING + ETH_MACADDR_SIZE)
/* Frame type */
#define ETH_FRAMETYPE_OFFSET     (ETH_SRC_MACADDR_OFFSET + ETH_MACADDR_SIZE)
/* Payload */
#define ETH_PAYLOAD_OFFSET       (ETH_TXRX_BUFFER_PRE_PADDING + ETH_HEADER_SIZE + ETH_TXRX_BUFFER_IN_PADDING)

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (2:3472)    : All toplevel uses of this function-like macro look like they could be replaced by            */
/*                       equivalent function calls.                                                                   */
/* Rule                : MISRA C:2012 Dir-4.9                                                                         */
/* JV-01 Justification : To prevent overhead and speed up the processing, there is no need to use function call for   */
/*                       such a small operation.                                                                      */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3432)    : Simple macro argument expression is not parenthesized.                                       */
/* Rule                : MISRA C:2012 Rule-20.7                                                                       */
/* JV-01 Justification : Compiler keyword (macro) is defined and used followed AUTOSAR standard rule. It is accepted. */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                 Global Data Types                                                  **
***********************************************************************************************************************/
/* Order of the drop count information for Eth_GetDropCount */
typedef enum ETag_Eth_EtherDropType
{
  ETH_DROP_BUFF_OVER_RUN = 0,
  ETH_DROP_CRC_ERR,
  ETH_DROP_UNDERSIZE,
  ETH_DROP_OVERSIZE,
  ETH_DROP_ALIGNMENT_ERR,
  ETH_DROP_SQE_TEST_ERR,
  ETH_DROP_RECEIVE_DISCARD,
  ETH_DROP_ERRONEOUS_INBOUND,
  ETH_DROP_TRANSMIT_DISCARD,
  ETH_DROP_ERRONESOUS_OUTBOUND,
  ETH_DROP_SINGLE_COLLISION,
  ETH_DROP_MULTIPLE_COLLISTION,
  ETH_DROP_DEFERRED_TRANSMIT,
  ETH_DROP_LATE_COLLISION
} Eth_EtherDropType;

/* Order of the error statuses for Eth_GetEtherStats */
typedef enum ETag_Eth_EtherStatsType
{
  ETH_STATS_DROP_EVENTS = 0,
  ETH_STATS_OCTETS,
  ETH_STATS_PKTS,
  ETH_STATS_BROADCAST_PKTS,
  ETH_STATS_MULTICAST_PKTS,
  ETH_STATS_CRC_ALIGN_ERRORS,
  ETH_STATS_UNDERSIZE_PKTS,
  ETH_STATS_OVERSIZE_PKTS,
  ETH_STATS_FRAGMENTS,
  ETH_STATS_JABBERS,
  ETH_STATS_COLLISIONS,
  ETH_STATS_PKTS_64_OCTETS,
  ETH_STATS_PKTS_65_TO_127_OCTETS,
  ETH_STATS_PKTS_128_TO_255_OCTETS,
  ETH_STATS_PKTS_256_TO_511_OCTETS,
  ETH_STATS_PKTS_512_TO_1023_OCTETS,
  ETH_STATS_PKTS_1024_TO_1518_OCTETS,
  ETH_STATS_MAX_VALUE /* stopper */
} Eth_EtherStatsType;

#if (ETH_GET_ETHER_STATS_API == STD_ON || ETH_GET_COUNTER_VALUES_API == STD_ON || \
     ETH_GET_TX_STATS_API == STD_ON || ETH_GET_RX_STATS_API == STD_ON)
typedef struct Stag_Eth_TxRxStatsType
{
  /* The total number of good packets received that were directed to the broadcast address.*/
  uint32 ulStatsBroadcastPkts;
  /* The total number of packets (including bad packets, broadcast packets, and multicast packets) received. */
  uint32 ulRxStatsPkts;
  /* The total number of octets of data (including those in bad packets) received on the network
     (excluding framing bits but including FCS octets). */
  uint32 ulRxStatsOctets;
  /* The total number of packets received that had a length of bertween 64 and 1518 octets 
     that had either a bad Frame Check Sequence (FCS) with an integral number of octets 
     (FCS Error) or a bad FCS with a non-integral number of octets (Alignment Error). */
  uint32 ulRxStatsCrcAlignErrors;
  /* The total number of packets received that were less than 64 octets long (excluding 
     framing bits, but including FCS octets) and were otherwise well formed. 
     Also described in IETF RFC 2819 MIB etherStatsUndersizePkts. */
  uint32 ulRxStatsUndersizePkts;
  /* The total number of packets received that were longer than 1518 octets (excluding 
     framing bits, but including FCS octets) and were otherwise well formed. 
     Also described in IETF RFC 2819 MIB etherStatsOversizePkts */
  uint32 ulRxStatsOversizePkts;
  /* The total number of packets (including bad packets) received that were 64 octets in length.  */
  uint32 ulRxStatsPkts64Octets;
  /* The total number of packets (including bad packets) received that were between 65 and 127 octets in length. */
  uint32 ulRxStatsPkts65to127Octets;
  /* The total number of packets (including bad packets) received that were between 128 and 255 octets in length. */
  uint32 ulRxStatsPkts128to255Octets;
  /* The total number of packets (including bad packets) received that were between 256 and 511 octets in length. */
  uint32 ulRxStatsPkts256to511Octets;
  /* The total number of packets (including bad packets) received that were between 512 and 1023 octets in length. */
  uint32 ulRxStatsPkts512to1023Octets;
  /* The total number of packets (including bad packets) received that were between 1024 and 1518 octets in length. */
  uint32 ulRxStatsPkts1024to1518Octets;
  /* The number of subnetwork-unicast packets delivered to a higher-layer protocol. */
  uint32 ulRxUnicastFrames;
  /* The total number of octets transmitted out of the interface, including framing characters. */
  uint32 ulTxNumberOfOctets;
  /* The total number of packets that higher-level protocols requested be transmitted to a
     non-unicast address, including those that were discarded or not sent. */
  uint32 ulTxNUcastPkts;
  /* The total number of packets that higher-level protocols requested be transmitted to a subnetwork-unicast address,
     including those that were discarded or not sent. */
  uint32 ulTxUniCastPkts;
  /* dropped packets due to CRC errors */
  uint32 ulDropPktCrc;
  /* number of undersize packets which were less than 64 octets long
    (excluding framing bits, but including FCS octets) and were otherwise well formed. (see IETF RFC 1757) */
  uint32 ulUndersizePkt;
  /* number of oversize packets which are longer than 1518 octets
    (excluding framing bits, but including FCS octets) and were otherwise well formed. (see IETF RFC 1757) */
  uint32 ulOversizePkt;
  /* number of alignment errors, i.e.packets which are received and
     are not an integral number of octets in length and do not pass the CRC. */
  uint32 ulAlgnmtErr;
} Eth_TxRxStatsType;

#endif

/* Status of each controller */
typedef struct Stag_Eth_ControllerStatusType
{
  /* MAC address of this controller */
  Eth_MacAddressType stMacAddr;
  /* Current mode, DOWN or ACTIVE */
  Eth_ModeType enMode;
#if (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON)
  /* Whehter a controller is in promiscuous mode.
  This is the logical state, not necessarily equal to the HW state. */
  boolean blPromiscuous;
  /* Whether the filter operation is activated */
  uint32 ulActiveFilterBits;
#endif
  /* Pointer to the head of tx buffer ring */
  uint32 ulTxBufHead;
  /* Pointer to the tail of tx buffer ring */
  uint32 ulTxBufTail;
#if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
  /* Pointer to the tail of ts descriptor ring */
  uint32 ulTsDescTail;
#endif
#if (ETH_GET_ETHER_STATS_API == STD_ON || ETH_GET_COUNTER_VALUES_API == STD_ON || \
     ETH_GET_TX_STATS_API == STD_ON || ETH_GET_RX_STATS_API == STD_ON)
  Eth_TxRxStatsType stTxRxStat;
#endif
  /* LLDriver specific status */
  Eth_HwStatusType stHwStat;

} Eth_ControllerStatusType;

/* A dummy value for unsupported items by HW */
#define ETH_NOT_AVAILABLE 0xFFFFFFFFUL

/***********************************************************************************************************************
**                                                Function Prototypes                                                 **
***********************************************************************************************************************/
#define ETH_START_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"

#ifdef ETH_HW_COMMON_INIT
extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_HwCommonInit(void);
#endif

#ifdef ETH_HW_COMMON_DEINIT
extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_HwCommonDeInit(void);
#endif

extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_HwInit(CONST(uint32, AUTOMATIC) LulCtrlIdx);
#if (ETH_DEINIT_API == STD_ON)
extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_HwDeInit(
    CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(boolean, AUTOMATIC) ForceReset);
#endif
extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_HwDisableController(CONST(uint32, AUTOMATIC) LulCtrlIdx);
extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_HwEnableController(CONST(uint32, AUTOMATIC) LulCtrlIdx);

extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE)
    Eth_HwTransmit(CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulBufIdx,
                   CONST(uint32, AUTOMATIC) LulLenByte, CONST(boolean, AUTOMATIC) LblConfirmation);

#if (ETH_GET_DROP_COUNT_API == STD_ON)
extern FUNC(void, ETH_PRIVATE_CODE)
    Eth_HwGetDropCount(CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulCountValues,
                       CONSTP2VAR(uint32, AUTOMATIC, ETH_APPL_DATA) LpDropCount);                                       /* PRQA S 3432 # JV-01 */
#endif

#if (ETH_GET_COUNTER_VALUES_API == STD_ON)
extern FUNC(void, ETH_PRIVATE_CODE)
    Eth_HwGetCounterValues(CONST(uint32, AUTOMATIC) LulCtrlIdx,
                           CONSTP2VAR(Eth_CounterType, AUTOMATIC, ETH_APPL_DATA) LpCounterPtr);                         /* PRQA S 3432 # JV-01 */
#endif

#if (ETH_GET_ETHER_STATS_API == STD_ON)
extern FUNC(void, ETH_PRIVATE_CODE)
    Eth_HwGetEtherStats(CONST(uint32, AUTOMATIC) LulCtrlIdx,
                        CONSTP2VAR(uint32, AUTOMATIC, ETH_APPL_DATA) LpEtherStats);                                     /* PRQA S 3432 # JV-01 */
#endif

#if (ETH_GET_RX_STATS_API == STD_ON)
extern FUNC(void, ETH_PRIVATE_CODE)
    Eth_HwGetRxStats(CONST(uint32, AUTOMATIC) LulCtrlIdx,
                     CONSTP2VAR(Eth_RxStatsType, AUTOMATIC, ETH_APPL_DATA) LpRxStats);                                  /* PRQA S 3432 # JV-01 */
#endif

#if (ETH_GET_TX_STATS_API == STD_ON)
extern FUNC(void, ETH_PRIVATE_CODE)
    Eth_HwGetTxStats(CONST(uint32, AUTOMATIC) LulCtrlIdx,
                     CONSTP2VAR(Eth_TxStatsType, AUTOMATIC, ETH_APPL_DATA) LpTxStats);                                  /* PRQA S 3432 # JV-01 */
#endif

#if (ETH_GET_TX_ERROR_COUNTER_VALUES_API == STD_ON)
extern FUNC(void, ETH_PRIVATE_CODE) Eth_HwGetTxErrorCounterValues(
    CONST(uint32, AUTOMATIC) LulCtrlIdx,
    CONSTP2VAR(Eth_TxErrorCounterValuesType, AUTOMATIC, ETH_APPL_DATA) LpTxErrorCounterValues);                         /* PRQA S 3432 # JV-01 */
#endif

#if (ETH_CTRL_ENABLE_MII == STD_ON)
#ifdef ETH_HW_NOT_LEGACY_MDIO
extern FUNC(uint16, ETH_PRIVATE_CODE)
    Eth_HwReadMii(CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint8, AUTOMATIC) LucTrcvIdx,
                  CONST(uint8, AUTOMATIC) LucRegIdx);
extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE)
    Eth_HwWriteMii(CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint8, AUTOMATIC) LucTrcvIdx,
                   CONST(uint8, AUTOMATIC) LucRegIdx, CONST(uint16, AUTOMATIC) LusRegVal);
#else
extern FUNC(void, ETH_PRIVATE_CODE)
    Eth_HwWriteMiiBit(CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulBit);
extern FUNC(uint32, ETH_PRIVATE_CODE) Eth_HwReadMiiBit(CONST(uint32, AUTOMATIC) LulCtrlIdx);
#endif
#endif

extern FUNC(void, ETH_PRIVATE_CODE) Eth_HwMainFunction(void);

extern FUNC(void, ETH_PRIVATE_CODE) Eth_HwTxConfirmation(CONST(uint32, AUTOMATIC) LulCtrlIdx);

#if (ETH_CTRL_ENABLE_RX_POLLING == STD_ON)
#if (ETH_AR_VERSION >= ETH_AR_431_VERSION)
extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE)
    Eth_HwCheckFifoIndex(CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulFifoIdx);
#endif
#endif

#if (ETH_USING_MACRO == ETH_MACRO_ETNC || ETH_USING_MACRO == ETH_MACRO_ETNB)
#if (ETH_CTRL_ENABLE_RX_POLLING == STD_ON)
#if (ETH_AR_VERSION >= ETH_AR_431_VERSION)
extern FUNC(Eth_RxStatusType, ETH_PRIVATE_CODE)
    Eth_HwReceive(CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulFifoIdx);
#else
extern FUNC(Eth_RxStatusType, ETH_PRIVATE_CODE) Eth_HwReceive(CONST(uint32, AUTOMATIC) LulCtrlIdx);
#endif
#endif
#else
extern FUNC(Eth_RxStatusType, ETH_PRIVATE_CODE)
    Eth_HwReceive(CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulFifoIdx);
#endif

#if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
FUNC(Std_ReturnType, ETH_PRIVATE_CODE)
    Eth_HwGetCurrentTime(CONST(uint32, AUTOMATIC) LulCtrlIdx,
                         CONSTP2VAR(Eth_TimeStampQualType, AUTOMATIC, ETH_APPL_DATA) LpTimeQualPtr,                     /* PRQA S 3432 # JV-01 */
                         CONSTP2VAR(Eth_TimeStampType, AUTOMATIC, ETH_APPL_DATA) LpTimeStampPtr);                       /* PRQA S 3432 # JV-01 */

FUNC(void, ETH_PRIVATE_CODE) Eth_HwEnableEgressTimeStamp(CONST(uint32, AUTOMATIC) LulCtrlIdx, uint8 BufIdx);

FUNC(void, ETH_PRIVATE_CODE)
    Eth_HwGetEgressTimeStamp(CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(Eth_BufIdxType, AUTOMATIC) LulBufIdx,
                             CONSTP2VAR(Eth_TimeStampQualType, AUTOMATIC, ETH_APPL_DATA) LpTimeQualPtr,                 /* PRQA S 3432 # JV-01 */
                             CONSTP2VAR(Eth_TimeStampType, AUTOMATIC, ETH_APPL_DATA) LpTimeStampPtr);                   /* PRQA S 3432 # JV-01 */

FUNC(void, ETH_PRIVATE_CODE)
    Eth_HwGetIngressTimeStamp(CONST(uint32, AUTOMATIC) LulCtrlIdx,
                              CONSTP2CONST(Eth_DataType, AUTOMATIC, ETH_APPL_DATA) LpDataPtr,
                              CONSTP2VAR(Eth_TimeStampQualType, AUTOMATIC, ETH_APPL_DATA) LpTimeQualPtr,                /* PRQA S 3432 # JV-01 */
                              CONSTP2VAR(Eth_TimeStampType, AUTOMATIC, ETH_APPL_DATA) LpTimeStampPtr);                  /* PRQA S 3432 # JV-01 */
#if (ETH_AR_VERSION == ETH_AR_422_VERSION)
FUNC(Std_ReturnType, ETH_PRIVATE_CODE)
    Eth_HwSetCorrectionTime(CONST(uint32, AUTOMATIC) LucCtrlIdx,
                            P2CONST(Eth_TimeIntDiffType, AUTOMATIC, ETH_APPL_DATA) LpTimeOffsetPtr);

FUNC(Std_ReturnType, ETH_PRIVATE_CODE)
    Eth_HwSetGlobalTime(CONST(uint32, AUTOMATIC) LulCtrlIdx,
                        P2CONST(Eth_TimeStampType, AUTOMATIC, ETH_APPL_DATA) timeStampPtr);
#endif
#endif

#define ETH_STOP_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"

#endif /* !ETH_COMMON_LLDIVER_H */
/***********************************************************************************************************************
**                                                    End of File                                                     **
***********************************************************************************************************************/
