/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Eth.c                                                                                               */
/* SW-VERSION   = 1.6.0                                                                                               */
/*====================================================================================================================*/
/*                                                  COPYRIGHT                                                         */
/*====================================================================================================================*/
/* (c) 2020-2022 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose: This file contains API implementations of Ethernet Driver Component.                                      */
/*                                                                                                                    */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s)        */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs     */
/* of program errors, compliance with applicable laws, damage to or loss of data, programs or equipment,              */
/* and unavailability or interruption of operations.                                                                  */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*                                        Devices:                X2x                                                 */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                          Revision Control History                                                  **
***********************************************************************************************************************/
/*
 * 1.5.3: 05/04/2022    : Add control index to parameter of Eth_HwCheckFifoIndex.
 *                        Add Eth_DeInit API
 *        13/04/2022    : Update QA-C comments.
 *        18/05/2022    : Limit the parameter IncVal checking of Eth_SetIncrementTimeForGptp to U2Ax devices only.
 * 1.5.2: 08/03/2022    : Add inclusion of SchM_Eth.h
 *        03/03/2022    : Add control index to parameter of Eth_HwGetTxErrorCounterValues.
 *        07/02/2022    : Update QA-C comment.
 *        24/01/2022    : Fixed the executable condition of Eth_SetIncrementTimeForGptp
 *                        from ETH_MODE_ACTIVE or ETH_MODE_DOWN to ETH_MODE_ACTIVE.
 * 1.5.1: 13/12/2021    : Update the check process of the tx buffer index of Eth_Transmit for each controller.
 *        18/10/2021    : Commonized Eth_ETNE_Ram.h and Eth_ETND_Ram.h.
 *                      : Change the memory section of the Eth_DemConfigCheck function to ETH_PRIVATE_CODE.
 *        27/10/2021    : Implementing the return value checking for Eth_HwCommonInit
 *                        (Report DEM_EVENT_STATUS_PREFAILED to DEM if the return value is E_NOT_OK)
 *        12/01/2021    : Update QA-C 9.5.0 comments
 * 1.4.2: 20/08/2021    : Update include file for support U2Bx devices.
 *        23/08/2021    : Fix an issue in which wrong event ID is reported to Diagnostic Event Manager in case of
 *                        failing access to the Ethernet Controller
 *        24/08/2021    : Modify the format to 120 characters
 *        26/08/2021    : Updated QA-C 9.5.0 comments
 * 1.4.1: 24/06/2021    : Update Eth_GetEgressTimeStamp and Eth_GetIngressTimeStamp for R19-11 support.
 *        07/05/2021    : Update WriteMii/ReadMii function for supporting U2Bx
 *                        Changes to support U2Bx devices.
 *                        Remove vendor ID from the file name, API names and parameters according to 
 *                        BSW00347 requirement
 *        09/06/2021    : Remove QA-C Warning 1339, 3214 and 3432 according to QA-C 9.5.0
 *        06/07/2021    : Updated QA-C 9.5.0 comments.
 * 1.2.0: 13/07/2020    : Release
 *        29/07/2020    : Add QAC 9.3.1 comment.
 * 1.1.0: 19/06/2020    : Release
 * 1.0.1: 04/06/2020    : To support Transmit/Receive interrupts for each controller.
 *                        Removed redundant error reports in Eth_59_EnableEgressTimeStamp.
 *                        Commented on a cast from a large type to a small type.
 *                        Change File version information from hard-code to macro.
 *                        Update by full check of the critical section.
 *                        Updated static analysis result.
 *                        Check range of parameter IncVal in API Eth_59_SetIncrementTimeForGptp.
 *                        Changed include file structure.
 *                        Fixed the error check of Eth_59_Receive.
 *                        Changed the parameter of Eth_59_SetOffsetTimeForGptp.
 *                        Fixed violation of AUTOSAR specifications for EthSwt.
 * 1.0.0: 25/03/2020    : Initial Version
 */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                       Include Section                                                              **
***********************************************************************************************************************/
/* Included for module version information and other types declarations                                               */
#include "Eth.h"

#if (ETH_USING_MACRO == ETH_MACRO_ETNC)
#include "Eth_ETNC_Ram.h"
#elif (ETH_USING_MACRO == ETH_MACRO_ETNB)
#include "Eth_ETNB_Ram.h"
#elif (ETH_USING_MACRO == ETH_MACRO_ETND || ETH_USING_MACRO == ETH_MACRO_ETNE)
#include "Eth_ETNDE_Ram.h"
#endif

#include "EthIf_Cbk.h"
#include "EthTrcv_Cbk.h"

#if (ETH_DEV_ERROR_DETECT == STD_ON)
/* Included for the declaration of Det_ReportError() */
#include "Det.h"
#endif

/* Included for declaration of the function Dem_ReportErrorStatus() */
#include "Dem.h"

#include "SchM_Eth.h"

/***********************************************************************************************************************
**                      Version Information                                                                           **
***********************************************************************************************************************/

/* AUTOSAR release version information */
#define ETH_C_AR_RELEASE_MAJOR_VERSION ETH_AR_RELEASE_MAJOR_VERSION_VALUE
#define ETH_C_AR_RELEASE_MINOR_VERSION ETH_AR_RELEASE_MINOR_VERSION_VALUE
#define ETH_C_AR_RELEASE_REVISION_VERSION ETH_AR_RELEASE_REVISION_VERSION_VALUE

/* File version information */
#define ETH_C_SW_MAJOR_VERSION      ETH_SW_MAJOR_VERSION_VALUE
#define ETH_C_SW_MINOR_VERSION      ETH_SW_MINOR_VERSION_VALUE

#define ETH_DBTOC_VALUE \
                         (((uint32)ETH_VENDOR_ID_VALUE << 22UL) | \
                         ((uint32)ETH_MODULE_ID_VALUE << 14UL) | \
                         ((uint32)ETH_SW_MAJOR_VERSION_VALUE << 8UL) | \
                         ((uint32)ETH_SW_MINOR_VERSION_VALUE << 3UL))                                                   /* PRQA S 3214 # JV-01 */

/***********************************************************************************************************************
**                      Version Check                                                                                 **
***********************************************************************************************************************/

#if (ETH_AR_RELEASE_MAJOR_VERSION != ETH_C_AR_RELEASE_MAJOR_VERSION)
  #error "Eth.c : Mismatch in Release Major Version"
#endif

#if (ETH_AR_RELEASE_MINOR_VERSION != ETH_C_AR_RELEASE_MINOR_VERSION)
  #error "Eth.c : Mismatch in Release Minor Version"
#endif

#if (ETH_AR_RELEASE_REVISION_VERSION != ETH_C_AR_RELEASE_REVISION_VERSION)
  #error "Eth.c : Mismatch in Release Revision Version"
#endif


#if (ETH_SW_MAJOR_VERSION != ETH_C_SW_MAJOR_VERSION)
  #error "Eth.c : Mismatch in Software Major Version"
#endif

#if (ETH_SW_MINOR_VERSION != ETH_C_SW_MINOR_VERSION)
  #error "Eth.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (2:3214)    : The macro '%s' is not used and could be removed.                                             */
/* Rule                : MISRA C:2012 Rule-2.5                                                                        */
/* JV-01 Justification : This can be accepted, it will be used in some case of configuration.                         */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1503)    : The function '%1s' is defined but is not used within this project.                           */
/* Rule                : CERTCCM MSC07, MISRA C:2012 Rule-2.1                                                         */
/* JV-01 Justification : This is accepted, due to the module's API is exported for user's usage.                      */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2982)    : This assignment is redundant. The value of this object is never used before being modified.  */
/* Rule                : CERTCCM MSC07, MSC13, MISRA C:2012 Rule-2.2                                                  */
/* JV-01 Justification : The variable needs to be initialized before using it.                                        */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (6:2877)    : This loop will never be executed more than once.                                             */
/* Rule                : CERTCCM MSC07, MISRA C:2012 Dir-4.1                                                          */
/* JV-01 Justification : This loop will only be executed at least once, depends on user configuration.                */
/*       Verification  : This is Hardware Specification, X2x only provides 1 Unit. So it is not having any impact.    */
/**********************************************************************************************************************/
/* Message (2:3469)    : This usage of a function-like macro looks like it could be replaced by an equivalent         */
/*                       function call.                                                                               */
/* Rule                : MISRA C:2012 Dir-4.9                                                                         */
/* JV-01 Justification : To prevent overhead and speed up the processing, there is no need to call a function here    */
/*                       for such a small operation.                                                                  */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3464)    : Argument to macro '%s' contains a side effect that will be evaluated more than once.         */
/* Rule                : CERTCCM PRE31                                                                                */
/* JV-01 Justification : This message is only emitted for expressions expanded from argument tokens written out at    */
/*                       the top level, that did not themselves originate from a macro expansion.                     */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2824)    : Possible: Arithmetic operation on NULL pointer.                                              */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3432)    : Simple macro argument expression is not parenthesized.                                       */
/* Rule                : MISRA C:2012 Rule-20.7                                                                       */
/* JV-01 Justification : Compiler keyword (macro) is defined and used followed AUTOSAR standard rule. It is accepted. */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3383)    : Cannot identify wraparound guard for unsigned arithmetic expression.                         */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : Wraparound can't be occur in this unsigned integer arithmetic so there is no need to add     */
/*                       a wraparound guard here.                                                                     */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:3206)    : The parameter '%s' is not used in this function.                                             */
/* Rule                : CERTCCM MSC07, MSC13, MISRA C:2012 Rule-2.7                                                  */
/* JV-01 Justification : This is done as per implementation requirement                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:1339)    : Evaluating the address of the parameter '%s'.                                                */
/* Rule                : MISRA C:2012 Rule-17.8                                                                       */
/* JV-01 Justification : This is accepted, there is no issue at this LOC with current implementation                  */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : Use of #include directive after code fragment.                                               */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is accepted, due to implementation for include memmap is following AUTOSAR rule.        */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                      Function Definitions                                                                          **
***********************************************************************************************************************/

/* Static internal functions */
#define ETH_START_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"

STATIC FUNC(void, ETH_PRIVATE_CODE) Eth_SetStatus(
  CONST(Eth_StateType, AUTOMATIC) LenStatus);

#if (ETH_DEV_ERROR_DETECT == STD_ON)
STATIC FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_CommonDetCheck(
  CONST(uint8, AUTOMATIC) LucSID, CONST(uint8, AUTOMATIC) LucCtrlIdx);
#endif

#define ETH_STOP_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */


#define ETH_START_SEC_PUBLIC_CODE
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name         : Eth_Init
**
** Service ID            : 0x01
**
** Description           : This API performs the initialization of the Ethernet Driver by initializing the configuration
**                         structure for subsequent API calls.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CfgPtr: Pointer to ETH Driver configuration set
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variables Used : Eth_GaaCtrlStat, Eth_GaaDemEventAccess
**
** Function(s) invoked   : Det_ReportError, Eth_DemConfigCheck, Eth_SetStatus, Eth_HwInit,
**                         Eth_InitializeBuffer, Eth_ClearAllAddressFilters
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_001
** Reference ID          : ETH_DUD_ACT_001_ERR001, ETH_DUD_ACT_001_ERR002, ETH_DUD_ACT_001_ERR003
** Reference ID          : ETH_DUD_ACT_001_ERR004, ETH_DUD_ACT_001_ERR005
** Reference ID          : ETH_DUD_ACT_001_GBL001, ETH_DUD_ACT_001_GBL002
** Reference ID          : ETH_DUD_ACT_001_GBL003, ETH_DUD_ACT_001_GBL004
***********************************************************************************************************************/
FUNC(void, ETH_PUBLIC_CODE) Eth_Init(                                                                                   /* PRQA S 1503 # JV-01 */
  P2CONST(Eth_ConfigType, AUTOMATIC, ETH_APPL_CONST) CfgPtr)
{
  uint32 LulCtrlIdx;
  Std_ReturnType LucResult;
  LucResult = E_OK;                                                                                                     /* PRQA S 2982 # JV-01 */

  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  if (NULL_PTR == CfgPtr)
  {
    /* Report Error to DET */
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_INIT_SID, ETH_E_PARAM_POINTER);
  }
  /* Check whether the existing database is correct */
  else if (ETH_DBTOC_VALUE != CfgPtr->ulStartOfDbToc)
  {
    /* Report Error to DET */
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_INIT_SID, ETH_E_INVALID_DATABASE);
  }
  else
  #endif /* (ETH_DEV_ERROR_DETECT == STD_ON) */
  {
    /* Assign the config pointer value to global pointer */
    Eth_GpCtrlConfigPtr = CfgPtr->pCtrlConfig;                                                                          /* PRQA S 2814 # JV-01 */
    Eth_GpEthConfigPtr = Eth_GpCtrlConfigPtr->pEthConfig;

    #ifdef ETH_HW_COMMON_INIT
    LucResult = Eth_HwCommonInit();
    for (LulCtrlIdx = 0UL; (LulCtrlIdx < (uint32)ETH_TOTAL_CTRL_CONFIG) && (E_NOT_OK == LucResult); LulCtrlIdx++)       /* PRQA S 2877 # JV-01 */
    {
      Eth_DemConfigCheck(Eth_GaaDemEventAccess[LulCtrlIdx], DEM_EVENT_STATUS_PREFAILED);
    }
    #endif

    for (LulCtrlIdx = 0UL; (LulCtrlIdx < (uint32)ETH_TOTAL_CTRL_CONFIG) && (E_OK == LucResult); LulCtrlIdx++)           /* PRQA S 2877 # JV-01 */
    {
      /* Copy the configured MAC address as inital value */
      ETH_PACK_ADDRESS_FROM_8(Eth_GpCtrlConfigPtr[LulCtrlIdx].aaEthMACAddr,                                             /* PRQA S 3469, 3464 # JV-01, JV-01 */
        Eth_GaaCtrlStat[LulCtrlIdx].stMacAddr);
      /* Initialize controller state as DOWN */
      Eth_GaaCtrlStat[LulCtrlIdx].enMode = ETH_MODE_DOWN;

      /* Initialize address filter */
      #if (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON)
      Eth_ClearAllAddressFilters(LulCtrlIdx);
      #endif
      /* Initialize Tx/Rx buffer pool */
      Eth_InitializeBuffer(LulCtrlIdx);

      /* Initialize low-level driver */
      LucResult = Eth_HwInit(LulCtrlIdx);
      if (E_OK == LucResult)
      {
        /* Report PASS to DEM */
        Eth_DemConfigCheck(Eth_GaaDemEventAccess[LulCtrlIdx], DEM_EVENT_STATUS_PREPASSED);
      }
      else
      {
        Eth_DemConfigCheck(Eth_GaaDemEventAccess[LulCtrlIdx], DEM_EVENT_STATUS_PREFAILED);
      }
    }

    if (E_OK == LucResult)
    {
      /* Initialize the Driver State to INIT */
      Eth_SetStatus(ETH_STATE_INIT);
    }
    else
    {
      /* No action required */
    }
  }
} /* End of API Eth_Init */

#if (ETH_DEINIT_API == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_DeInit
**
** Service ID            : 0xA3
**
** Description           : This API performs the SW Reset process needed for the initialization of the Ethernet Driver
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : ForceReset: Indicate the reset flow to follow (normal/emergency)
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Std_ReturnType
**                         E_OK: Success to SW Reset
**                         E_NOT_OK: Fail to SW Reset
**
** Preconditions         : The function requires previous controller initialization Eth_Init().
**
** Global Variables Used : Eth_GenDriverState
**
** Function(s) invoked   : Det_ReportError, Eth_SetStatus, Eth_HwDeInit
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_052
** Reference ID          : ETH_DUD_ACT_052_ERR001
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE) Eth_DeInit(CONST(boolean, AUTOMATIC) ForceReset)                                  /* PRQA S 1503 # JV-01 */
{
  uint32 LulCtrlIdx;
  Std_ReturnType LucResult;
  LucResult = E_OK;

  (void)ForceReset;

  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  if (ETH_STATE_INIT != Eth_GenDriverState)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_DEINIT_SID, ETH_E_UNINIT);
    LucResult = E_NOT_OK;
  }
  if (E_OK == LucResult)
  #endif /* (ETH_DEV_ERROR_DETECT == STD_ON) */
  {
    #ifdef ETH_HW_COMMON_DEINIT
    LucResult = Eth_HwCommonDeInit();
    #endif

    for (LulCtrlIdx = 0UL; (LulCtrlIdx < (uint32)ETH_TOTAL_CTRL_CONFIG) && (E_OK == LucResult); LulCtrlIdx++)           /* PRQA S 2877 # JV-01 */
    {
      /* Initialize low-level driver */
      LucResult = Eth_HwDeInit(LulCtrlIdx, ForceReset);
    }

    if (E_OK == LucResult)
    {
      Eth_SetStatus(ETH_STATE_UNINIT);
    } /* else: No action required */
  } /* else: No action required */

  return LucResult;
} /* End of API Eth_DeInit */
#endif /* (ETH_DEINIT_API == STD_ON) */

/***********************************************************************************************************************
** Function Name         : Eth_SetControllerMode
**
** Service ID            : 0x03
**
** Description           : This API performs enabling / disabling  of the indexed controller.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx:  Index of an ether controller
**                         CtrlMode: Enable/Disable controller
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Std_ReturnType
**                         E_OK: success.
**                         E_NOT_OK: controller mode could not be changed.
**
** Preconditions         : Component Requires previous controller initialization using Eth_Init. This API must not 
**                         preempt Eth_MainFunction, to avoid the false detection of the registercorruption check.
**
** Global Variables Used : Eth_GaaCtrlStat, Eth_GaaDemEventAccess, Eth_GaaDemEventRxFramesLost,
**                         Eth_GaaDemEventCRC, Eth_GaaDemEventUnderSizeFrame,
**                         Eth_GaaDemEventOverSizeFrame, Eth_GaaDemEventAlignment,
**                         Eth_GaaDemEventSinglecollision, Eth_GaaDemEventMultiplecollision,
**                         Eth_GaaDemEventLatecollision
**
** Function(s) invoked   : Eth_DemConfigCheck, Eth_CommonDetCheck, Eth_HwEnableController,
**                         Eth_HwDisableController, Eth_InitializeBuffer,
**                         Eth_PreprocessBuffer, EthIf_CtrlModeIndication
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_002,
** Reference ID          : ETH_DUD_ACT_002_ERR001, ETH_DUD_ACT_002_ERR002
** Reference ID          : ETH_DUD_ACT_002_ERR003, ETH_DUD_ACT_002_ERR004
** Reference ID          : ETH_DUD_ACT_002_ERR005, ETH_DUD_ACT_002_ERR006
** Reference ID          : ETH_DUD_ACT_002_ERR007, ETH_DUD_ACT_002_ERR008
** Reference ID          : ETH_DUD_ACT_002_ERR009, ETH_DUD_ACT_002_ERR010
** Reference ID          : ETH_DUD_ACT_002_ERR011, ETH_DUD_ACT_002_GBL001
** Reference ID          : ETH_DUD_ACT_002_GBL002
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE) Eth_SetControllerMode(                                                            /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, Eth_ModeType CtrlMode)
{
  Std_ReturnType LucReturnValue;

  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_SETCONTROLLERMODE_SID, CtrlIdx);
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    LucReturnValue = E_OK;
    if (Eth_GaaCtrlStat[CtrlIdx].enMode != CtrlMode)                                                                    /* PRQA S 2844 # JV-01 */
    {
      if (ETH_MODE_ACTIVE == CtrlMode)
      {
        /* Activate HW */
        LucReturnValue = Eth_HwEnableController((uint32)CtrlIdx);

        if (E_OK == LucReturnValue)
        {
          /* Set source MAC address to all Tx Buffers in advance */
          Eth_PreprocessBuffer((uint32)CtrlIdx);
          /* To prevent MainFunction be executed during the transition,
          set the controller state as ACTIVE after the transition. */
          Eth_GaaCtrlStat[CtrlIdx].enMode = ETH_MODE_ACTIVE;                                                            /* PRQA S 2844 # JV-01 */
        }
        else
        {
          Eth_DemConfigCheck(Eth_GaaDemEventAccess[CtrlIdx], DEM_EVENT_STATUS_PREFAILED);
        }
      }
      else
      {
        /* To prevent MainFunction be executed during the transition,
        set the controller state as DOWN before the transition. */
        Eth_GaaCtrlStat[CtrlIdx].enMode = CtrlMode;                                                                     /* PRQA S 2844 # JV-01 */
        /* Stop HW */
        LucReturnValue = Eth_HwDisableController((uint32)CtrlIdx);
        /* Retreive all provided Tx Buffers */
        if (E_OK == LucReturnValue)
        {
          Eth_InitializeBuffer((uint32)CtrlIdx);
        }
        else
        {
          Eth_DemConfigCheck(Eth_GaaDemEventAccess[CtrlIdx], DEM_EVENT_STATUS_PREFAILED);
        }
      }
      if (E_OK == LucReturnValue)
      {
        /* Since this function is implemented as synchronous, indicate a mode switch to EthIf here */
        EthIf_CtrlModeIndication(CtrlIdx, CtrlMode);
        Eth_DemConfigCheck(Eth_GaaDemEventAccess[CtrlIdx], DEM_EVENT_STATUS_PREPASSED);
        Eth_DemConfigCheck(Eth_GaaDemEventRxFramesLost[CtrlIdx], DEM_EVENT_STATUS_PREPASSED);
        Eth_DemConfigCheck(Eth_GaaDemEventCRC[CtrlIdx], DEM_EVENT_STATUS_PREPASSED);
        Eth_DemConfigCheck(Eth_GaaDemEventUnderSizeFrame[CtrlIdx], DEM_EVENT_STATUS_PREPASSED);
        Eth_DemConfigCheck(Eth_GaaDemEventOverSizeFrame[CtrlIdx], DEM_EVENT_STATUS_PREPASSED);
        Eth_DemConfigCheck(Eth_GaaDemEventAlignment[CtrlIdx], DEM_EVENT_STATUS_PREPASSED);
        Eth_DemConfigCheck(Eth_GaaDemEventSinglecollision[CtrlIdx], DEM_EVENT_STATUS_PREPASSED);
        Eth_DemConfigCheck(Eth_GaaDemEventMultiplecollision[CtrlIdx], DEM_EVENT_STATUS_PREPASSED);
        Eth_DemConfigCheck(Eth_GaaDemEventLatecollision[CtrlIdx], DEM_EVENT_STATUS_PREPASSED);
      }
      else
      {
        /* No action required */
      }
    }
    else
    {
       /* No action required */
    }
  }

  return(LucReturnValue);
} /* End of API Eth_SetControllerMode */

/***********************************************************************************************************************
** Function Name         : Eth_GetControllerMode
**
** Service ID            : 0x04
**
** Description           : This API Obtains the state of the indexed controller
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**
** InOut Parameters      : None
**
** Output Parameters     : CtrlModePtr: Pointer of the controller mode
**
** Return parameter      : Std_ReturnType
**                         E_OK: success.
**                         E_NOT_OK: controller mode could not be obtained.
**
** Preconditions         : The function requires previous controller initialization Eth_Init().
**
** Global Variables Used : Eth_GaaCtrlStat
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck,
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_003,
** Reference ID          : ETH_DUD_ACT_003_ERR001, ETH_DUD_ACT_003_GBL001
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE)Eth_GetControllerMode(                                                             /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, P2VAR(Eth_ModeType, AUTOMATIC, ETH_APPL_DATA) CtrlModePtr)                                             /* PRQA S 3432 # JV-01 */
{
  Std_ReturnType LucReturnValue;

  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_GETCONTROLLERMODE_SID, CtrlIdx);
  /* Report Error to DET, if the CtrlModePtr pointer value is NULL */
  if (NULL_PTR == CtrlModePtr)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GETCONTROLLERMODE_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif /* (ETH_DEV_ERROR_DETECT == STD_ON) */
  {
    *CtrlModePtr = Eth_GaaCtrlStat[CtrlIdx].enMode;                                                                     /* PRQA S 2844 # JV-01 */
    LucReturnValue = E_OK;
  }

  return(LucReturnValue);
} /* End of API Eth_GetControllerMode */

/***********************************************************************************************************************
** Function Name         : Eth_GetPhysAddr
**
** Service ID            : 0x08
**
** Description           : This API obtains the physical source address(MAC Address) configured for the 
**                         indexed controller.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**
** InOut Parameters      : None
**
** Output Parameters     : PhysAddrPtr: Pointer to memory containing the physical source address (MAC address) in 
**                         network byte order.
**
** Return parameter      : None
**
** Preconditions         : Component Requires previous controller initialization using Eth_Init.
**
** Global Variables Used : Eth_GaaCtrlStat
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_004,
** Reference ID          : ETH_DUD_ACT_004_ERR001
***********************************************************************************************************************/
FUNC(void, ETH_PUBLIC_CODE) Eth_GetPhysAddr(                                                                            /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, P2VAR(uint8, AUTOMATIC, ETH_APPL_DATA) PhysAddrPtr)                                                    /* PRQA S 3432 # JV-01 */
{
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  Std_ReturnType LucReturnValue;
  LucReturnValue = Eth_CommonDetCheck(ETH_GETPHYSADDR_SID, CtrlIdx);
  /* Report Error to DET, if the PhysAddrPtr pointer value is NULL */
  if (NULL_PTR == PhysAddrPtr)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GETPHYSADDR_SID, ETH_E_PARAM_POINTER);
     LucReturnValue  = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif /* (ETH_DEV_ERROR_DETECT == STD_ON) */
  {
    ETH_UNPACK_ADDRESS_TO_8(Eth_GaaCtrlStat[CtrlIdx].stMacAddr, PhysAddrPtr);                                           /* PRQA S 2824, 2844, 3469 # JV-01, JV-01, JV-01 */
  }
} /* End of API Eth_GetPhysAddr */

/***********************************************************************************************************************
** Function Name         : Eth_SetPhysAddr
**
** Service ID            : 0x13
**
** Description           : Sets the physical source address used by the indexed controller.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant for the same CtrlIdx,
**                         reentrant for different
**
** Input Parameters      : CtrlIdx:     Index of the controller
**                         PhysAddrPtr: Pointer to memory containing the physical source address (MAC address)
**                                      in network byte order.
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : Component Requires previous controller initialization using Eth_Init.
**
** Global Variables Used : Eth_GaaCtrlStat
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_005,
** Reference ID          : ETH_DUD_ACT_005_ERR001, ETH_DUD_ACT_005_ERR002
** Reference ID          : ETH_DUD_ACT_005_GBL001
***********************************************************************************************************************/
FUNC(void, ETH_PUBLIC_CODE) Eth_SetPhysAddr(                                                                            /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, P2CONST(uint8, AUTOMATIC, ETH_APPL_DATA) PhysAddrPtr)
{
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  Std_ReturnType LucReturnValue;
  LucReturnValue = Eth_CommonDetCheck(ETH_SETPHYSADDR_SID, CtrlIdx);
  /* Report Error to DET, if the PhysAddrPtr pointer value is NULL */
  if (NULL_PTR == PhysAddrPtr)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_SETPHYSADDR_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    if (ETH_MODE_ACTIVE == Eth_GaaCtrlStat[CtrlIdx].enMode)                                                             /* PRQA S 2844 # JV-01 */
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_SETPHYSADDR_SID, ETH_E_INV_MODE);
      #endif
    }
    else
    {
      ETH_PACK_ADDRESS_FROM_8(PhysAddrPtr, Eth_GaaCtrlStat[CtrlIdx].stMacAddr);                                         /* PRQA S 2824, 2844, 3469 # JV-01, JV-01, JV-01 */
    }
  }
} /* End of API Eth_SetPhysAddr */

#if (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_UpdatePhysAddrFilter
**
** Service ID            : 0x12
**
** Description           : Update the physical source address to/from the indexed controller filter.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant for the same CtrlIdx,
**                         reentrant for different
**
** Input Parameters      : CtrlIdx:    Index of the controller
**                         PhyAddrPtr: Pointer to memory containing the physical destination address (MAC address)
**                                     in network byte order.
**                         Action:     Add or remove the address from the Ethernet controllers
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Std_ReturnType
**                         E_OK: filter was successfully changed
**                         E_NOT_OK: filter could not be changed and the DET is ON
**
** Preconditions         : Component Requires previous controller initialization using Eth_Init.
**
** Global Variables Used : Eth_GaaCtrlStat, Eth_GstBroadcastAddr, Eth_GstNullAddr
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_AddAddressFilter, Eth_RemoveAddressFilter,
**                         Eth_GetFilterIndex, Eth_ClearAllAddressFilters
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_006,
** Reference ID          : ETH_DUD_ACT_006_ERR001, ETH_DUD_ACT_006_ERR002
** Reference ID          : ETH_DUD_ACT_006_ERR003, ETH_DUD_ACT_006_ERR004
** Reference ID          : ETH_DUD_ACT_006_GBL001, ETH_DUD_ACT_006_GBL002
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE) Eth_UpdatePhysAddrFilter(                                                         /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, P2CONST(uint8, AUTOMATIC, ETH_APPL_DATA) PhysAddrPtr, Eth_FilterActionType Action)
{
  Eth_MacAddressType LstMacAddr;
  Std_ReturnType LucReturnValue;
  uint32 LulFilterIdx;

  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_UPDATEPHYSADDRFILTER_SID, CtrlIdx);
  /* Report Error to DET, if the PhysAddrPtr pointer value is NULL */
  if (NULL_PTR == PhysAddrPtr)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID,
      ETH_UPDATEPHYSADDRFILTER_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }

  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    /* Pack MAC address from user to the structure to be compared */
    ETH_PACK_ADDRESS_FROM_8(PhysAddrPtr, LstMacAddr);                                                                   /* PRQA S 2824, 3469 # JV-01, JV-01 */

    /* FF:..:FF and 00:..:00 are not allowed while the controller is ACTVIE */
    if ((ETH_MODE_ACTIVE == Eth_GaaCtrlStat[CtrlIdx].enMode) &&                                                         /* PRQA S 2844 # JV-01 */
      ((0UL == ETH_COMPARE_MAC(LstMacAddr, Eth_GstBroadcastAddr)) ||                                                    /* PRQA S 3469 # JV-01 */
      (0UL == ETH_COMPARE_MAC(LstMacAddr, Eth_GstNullAddr))))                                                           /* PRQA S 3469 # JV-01 */
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_UPDATEPHYSADDRFILTER_SID, ETH_E_INV_MODE);
      #endif
      LucReturnValue = E_NOT_OK;
    }
    else
    {
      LucReturnValue = E_OK;

      if (0UL == ETH_COMPARE_MAC(LstMacAddr, Eth_GstBroadcastAddr))                                                     /* PRQA S 3469 # JV-01 */
      {
        /* If FF:..:FF is specified, enter promiscuous mode */
        Eth_GaaCtrlStat[CtrlIdx].blPromiscuous = ETH_TRUE;                                                              /* PRQA S 2844 # JV-01 */
      }
      else if (0UL == ETH_COMPARE_MAC(LstMacAddr, Eth_GstNullAddr))                                                     /* PRQA S 3469 # JV-01 */
      {
        /* If 00:..:00 is specified, clear promisoucs mode and all filters */
        Eth_GaaCtrlStat[CtrlIdx].blPromiscuous = ETH_FALSE;                                                             /* PRQA S 2844 # JV-01 */
        Eth_ClearAllAddressFilters((uint32)CtrlIdx);
      }
      else
      {
        if (ETH_ADD_TO_FILTER == Action)
        {
          #if (ETH_DEV_ERROR_DETECT == STD_ON)
          /*
             Report DET error when:
             - The specified address is not a multicast address
             - The specified address has been already registered
             - The filter array is full
          */
          LulFilterIdx = Eth_GetFilterIndex((uint32)CtrlIdx, &LstMacAddr);
          if ((0UL == ETH_CHECK_MULTICAST(LstMacAddr)) ||                                                               /* PRQA S 3469 # JV-01 */
            (ETH_INVALID_FILTER_INDEX != LulFilterIdx) ||
            (ETH_FILTER_FULL_VALUE == Eth_GaaCtrlStat[CtrlIdx].ulActiveFilterBits))                                     /* PRQA S 2844 # JV-01 */
          {
            (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID,
              ETH_UPDATEPHYSADDRFILTER_SID, ETH_E_INV_PARAM);
            LucReturnValue = E_NOT_OK;
          }
          else
          #endif
          {
            Eth_AddAddressFilter((uint32)CtrlIdx, &LstMacAddr);
          }
        }
        else
        {
          LulFilterIdx = Eth_GetFilterIndex((uint32)CtrlIdx, &LstMacAddr);
          #if (ETH_DEV_ERROR_DETECT == STD_ON)
          /* If the specified address is not registered, report error */
          if (ETH_INVALID_FILTER_INDEX == LulFilterIdx)
          {
            (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_UPDATEPHYSADDRFILTER_SID, ETH_E_INV_PARAM);
            LucReturnValue = E_NOT_OK;
          }
          else
          #endif
          {
            Eth_RemoveAddressFilter((uint32)CtrlIdx, LulFilterIdx);
          }
        }
      }
    }
  }

  return LucReturnValue;
}
#endif /* (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON) */

#if (ETH_CTRL_ENABLE_MII == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_WriteMii
**
** Service ID            : 0x05
**
** Description           : This API Configures a transceiver register or triggers a function offered by the receiver by
**                         writing the specified transceiver register through the MII of the indexed controller
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**                         TrcvIdx: Index of the transceiver on the MII
**                         RegIdx:  Index of the transceiver register on the MII
**                         RegVal:  Value to be written into the indexed register
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Std_ReturnType
**                         E_OK: No error has occurred during execution of the API
**                         E_NOT_OK: An error has occurred during execution of the API
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration 
**                         parameter EthCtrlEnableMii.
**
** Global Variables Used : None
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_HwWriteMiiBit
**                         Eth_HwWriteMii, EthTrcv_WriteMiiIndication, Eth_HwReadMiiBit
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_007,
** Reference ID          : ETH_DUD_ACT_007_ERR001, ETH_DUD_ACT_007_ERR002
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE)Eth_WriteMii(                                                                      /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, uint8 TrcvIdx, uint8 RegIdx, uint16 RegVal)
{
  Std_ReturnType LucReturnValue;

  #ifndef ETH_HW_NOT_LEGACY_MDIO
  /* Local variable only used in E2x and U2Ax devices */
  uint32 LulBitCount;
  uint32 LulHeader;
  uint32 LulBit;
  #endif

  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_WRITEMII_SID, CtrlIdx);
  /* Check for the valid TrcvIdx  */
  if (ETH_PHY_MAX_PHYAD_IDX < TrcvIdx)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_WRITEMII_SID, ETH_E_INV_PARAM);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  /* Check for the valid RegIdx  */
  if (ETH_PHY_MAX_REGAD_IDX < RegIdx)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_WRITEMII_SID, ETH_E_INV_PARAM);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }

  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif /* (ETH_DEV_ERROR_DETECT == STD_ON) */

  #ifdef ETH_HW_NOT_LEGACY_MDIO
  {
    LucReturnValue = Eth_HwWriteMii((uint32)CtrlIdx, TrcvIdx, RegIdx, RegVal);

    if (E_OK == LucReturnValue)
    {
      /* Since this function is implemented as synchronous, call EthTrcv here */
      EthTrcv_WriteMiiIndication(CtrlIdx, TrcvIdx, RegIdx);
    }
    else
    {
      /* No action required */
    }
  }
  #else
  {
    /* PRE */
    for (LulBitCount = (uint32)ETH_PHY_PREAMBLE_SIZE; LulBitCount > 0UL; LulBitCount--)
    {
      Eth_HwWriteMiiBit((uint32)CtrlIdx, 1UL);
    }

    /* ST, OP, PHYAD, REGAD */
    LulHeader = ETH_PHY_WHEADER(TrcvIdx, RegIdx);                                                                       /* PRQA S 3469 # JV-01 */
    for (LulBitCount = (uint32)ETH_PHY_HEADER_SIZE; LulBitCount > 0UL; LulBitCount--)
    {
      /* Pickup the bit which should be output next, MSB first */
      LulBit = (LulHeader >> (LulBitCount - 1UL)) & 1UL;                                                                /* PRQA S 3383 # JV-01 */
      Eth_HwWriteMiiBit((uint32)CtrlIdx, LulBit);
    }

    /* TA: 10 */
    Eth_HwWriteMiiBit((uint32)CtrlIdx, 1UL);
    Eth_HwWriteMiiBit((uint32)CtrlIdx, 0UL);

    /* DATA */
    for (LulBitCount = (uint32)ETH_PHY_DATA_SIZE; LulBitCount > 0UL; LulBitCount--)
    {
      /* Pickup the bit which should be output next, MSB first */
      LulBit = ((uint32)RegVal >> (LulBitCount - 1UL)) & 1UL;                                                           /* PRQA S 3383 # JV-01 */
      Eth_HwWriteMiiBit((uint32)CtrlIdx, LulBit);
    }

    /* Bus release (MDIO = Z) and IDLE cycle, it is same as read cycle */
    (void)Eth_HwReadMiiBit((uint32)CtrlIdx);

    /* Since this function is implemented as synchronous, call EthTrcv here */
    EthTrcv_WriteMiiIndication(CtrlIdx, TrcvIdx, RegIdx);

    LucReturnValue = E_OK;
  }
  #endif /* #ifdef ETH_HW_NOT_LEGACY_MDIO */

  return LucReturnValue;
} /* End of API Eth_WriteMii */

/***********************************************************************************************************************
** Function Name         : Eth_ReadMii
**
** Service ID            : 0x06
**
** Description           : This API Reads the specified transceiver register through the MII of the indexed controller.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**                         TrcvIdx: Index of the transceiver on the MII
**                         RegIdx:  Index of the transceiver register on the MII
**
** InOut Parameters      : None
**
** Output Parameters     : RegValPtr: Filled with the register content of the indexed register
**
** Return parameter      : Std_ReturnType
**                         E_OK: No error has occurred during execution of the API
**                         E_NOT_OK: An error has occurred during execution of the API
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration 
**                         parameter EthCtrlEnableMii.
**
** Global Variables Used : None
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_HwReadMiiBit,
**                         Eth_HwReadMii, EthTrcv_ReadMiiIndication, Eth_HwWriteMiiBit
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_008, ETH_DUD_ACT_008_ERR001,
** Reference ID          : ETH_DUD_ACT_008_ERR002, ETH_DUD_ACT_008_ERR003,
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE)Eth_ReadMii(                                                                       /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, uint8 TrcvIdx, uint8 RegIdx,
  P2VAR(uint16, AUTOMATIC, ETH_APPL_DATA)RegValPtr)                                                                     /* PRQA S 3432 # JV-01 */
{
  /* Local variable to hold the DET return value */
  Std_ReturnType LucReturnValue;

  #ifndef ETH_HW_NOT_LEGACY_MDIO
  /* Local variable only used in E2x and U2Ax devices */
  uint32 LulBitCount;
  uint32 LulHeader;
  uint32 LulBit;
  /* uint16 LulRegVal; */
  uint32 LulRegVal;
  #endif

  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_READMII_SID, CtrlIdx);
  if (NULL_PTR == RegValPtr)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_READMII_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  /* Check for the valid TrcvIdx  */
  if (ETH_PHY_MAX_PHYAD_IDX < TrcvIdx)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_READMII_SID, ETH_E_INV_PARAM);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  /* Check for the valid RegIdx  */
  if (ETH_PHY_MAX_REGAD_IDX < RegIdx)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_READMII_SID, ETH_E_INV_PARAM);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }

  /* Check if any DET was reported */
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif /* (ETH_DEV_ERROR_DETECT == STD_ON) */

  #ifdef ETH_HW_NOT_LEGACY_MDIO
  {
    /* Read data from PHY */
    *RegValPtr = Eth_HwReadMii((uint32)CtrlIdx, TrcvIdx, RegIdx);

    /* Since this function is implemented as synchronous, call EthTrcv here */
    EthTrcv_ReadMiiIndication(CtrlIdx, TrcvIdx, RegIdx, *RegValPtr);

    LucReturnValue = E_OK;
  }
  #else
  {
    /* PRE */
    for (LulBitCount = (uint32)ETH_PHY_PREAMBLE_SIZE; LulBitCount > 0UL; LulBitCount--)
    {
      Eth_HwWriteMiiBit((uint32)CtrlIdx, 1UL);
    }

    /* ST, OP, PHYAD, REGAD */
    LulHeader = ETH_PHY_RHEADER(TrcvIdx, RegIdx);                                                                       /* PRQA S 3469 # JV-01 */
    for (LulBitCount = (uint32)ETH_PHY_HEADER_SIZE; LulBitCount > 0UL; LulBitCount--)
    {
      /* Pickup the MSB which should be output next, MSB first */
      LulBit = (LulHeader >> (LulBitCount - 1UL)) & 1UL;                                                                /* PRQA S 3383 # JV-01 */
      Eth_HwWriteMiiBit((uint32)CtrlIdx, LulBit);
    }

    /* TA[0]: Turn MDIO to Z, it is same as Read cycle */
    (void)Eth_HwReadMiiBit((uint32)CtrlIdx);
    /* TA[1]: Read 0 from PHY, ignore it */
    (void)Eth_HwReadMiiBit((uint32)CtrlIdx);

    /* DATA */
    LulRegVal = 0UL;
    for (LulBitCount = (uint32)ETH_PHY_DATA_SIZE; LulBitCount > 0UL; LulBitCount--)
    {
      /* Since read data is MSB first, shift left the result */
      LulRegVal = LulRegVal << 1UL;
      /* Read data from PHY */
      LulRegVal |= Eth_HwReadMiiBit((uint32)CtrlIdx);
    }

    *RegValPtr = (uint16)LulRegVal;

    /* Since this function is implemented as synchronous, call EthTrcv here */
    EthTrcv_ReadMiiIndication(CtrlIdx, TrcvIdx, RegIdx, *RegValPtr);

    LucReturnValue = E_OK;
  }
  #endif /* #ifdef ETH_HW_NOT_LEGACY_MDIO */

  return LucReturnValue;
} /* End of API Eth_ReadMii */
#endif /* (ETH_CTRL_ENABLE_MII == STD_ON) */

#if (ETH_GET_DROP_COUNT_API == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_GetDropCount
**
** Service ID            : 0x14
**
** Description           : Reads a list with drop counter values of the corresponding controller.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx:     Index of the controller
**                         CountValues: Maximal number of values which can be written from DropCount
**
** InOut Parameters      : None
**
** Output Parameters     : DropCount: Pointer to where is stored the version information of this module
**
** Return parameter      : Std_ReturnType
**                         E_OK: success
**                         E_NOT_OK: Transmission failed
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration 
**                         parameter EthGetDropCountApi.
**
** Global Variables Used : None
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_HwGetDropCount
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_009, ETH_DUD_ACT_009_ERR001
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE)Eth_GetDropCount(                                                                  /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, uint8 CountValues, P2VAR(uint32, AUTOMATIC, ETH_APPL_DATA) DropCount)                                  /* PRQA S 3432 # JV-01 */
{
  Std_ReturnType LucReturnValue;
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_GETDROPCOUNT_SID, CtrlIdx);
  /* Report Error to DET, if the stherStats pointer value is NULL */
  if (NULL_PTR == DropCount)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GETDROPCOUNT_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    Eth_HwGetDropCount((uint32)CtrlIdx, (uint32)CountValues, DropCount);
    LucReturnValue = E_OK;
  }

  return (LucReturnValue);
} /* End of API Eth_GetDropCount */
#endif

#if (ETH_GET_COUNTER_VALUES_API == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_GetCounterValues
**
** Service ID            : 0x14
**
** Description           : This API Reads a list with drop counter values of the corresponding controller.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**
** InOut Parameters      : None
**
** Output Parameters     : CounterPtr: counter values according to IETF RFC 1757, RFC 1643 and RFC 2233.
**
** Return parameter      : Std_ReturnType
**                         E_OK: success
**                         E_NOT_OK: counter values read failure
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration 
**                         parameter EthGetCounterValuesApi
**
** Global Variables Used : None
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_HwGetCounterValues
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_010, ETH_DUD_ACT_010_ERR001
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE)Eth_GetCounterValues(                                                              /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, P2VAR(Eth_CounterType, AUTOMATIC, ETH_APPL_DATA) CounterPtr)                                           /* PRQA S 3432 # JV-01 */
{
  Std_ReturnType LucReturnValue;
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_GETCOUNTERVALUES_SID, CtrlIdx);
  /* Report Error to DET, if the stherStats pointer value is NULL */
  if (NULL_PTR == CounterPtr)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GETCOUNTERVALUES_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    Eth_HwGetCounterValues((uint32)CtrlIdx, CounterPtr);
    LucReturnValue = E_OK;
  }

  return (LucReturnValue);
} /* End of API Eth_GetCounterValues */
#endif

#if (ETH_GET_ETHER_STATS_API == STD_ON)
/**********************************************************************************************************************
** Function Name         : Eth_GetEtherStats
**
** Service ID            : 0x15
**
** Description           : This API get the ethernet statistics information.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**
** InOut Parameters      : None
**
** Output Parameters     : etherStats: List of values according to IETF RFC 2819
**
** Return parameter      : Std_ReturnType
**                         E_OK: Success
**                         E_NOT_OK: Transmission failed
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration 
**                         parameter EthGetEtherStatsApi.
**
** Global Variables Used : None
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_HwGetEtherStats
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_011, ETH_DUD_ACT_011_ERR001
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE)Eth_GetEtherStats(                                                                 /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, P2VAR(uint32, AUTOMATIC, ETH_APPL_DATA) etherStats)                                                    /* PRQA S 3432 # JV-01 */
{
  Std_ReturnType LucReturnValue;
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_GETETHERSTATS_SID, CtrlIdx);
  /* Report Error to DET, if the stherStats pointer value is NULL */
  if (NULL_PTR == etherStats)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GETETHERSTATS_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    Eth_HwGetEtherStats((uint32)CtrlIdx, etherStats);
    LucReturnValue = E_OK;
  }

  return (LucReturnValue);
} /* End of API Eth_GetEtherStats */
#endif

#if (ETH_GET_RX_STATS_API == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_GetRxStats
**
** Service ID            : 0x15
**
** Description           : This API Returns the following list according to IETF RFC2819, where the maximal possible 
**                         value shall denote an invalid value.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**
** InOut Parameters      : None
**
** Output Parameters     : RxStats: List of values according to IETF RFC 2819 (Remote Network Monitoring Management
**                                  Information Base).
**
** Return parameter      : Std_ReturnType
**                         E_OK: success
**                         E_NOT_OK: drop counter could not be obtained
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration 
**                         parameter EthGetRxStatsApi.
**
** Global Variables Used : None
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_HwGetRxStats
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_012, ETH_DUD_ACT_012_ERR001
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE)Eth_GetRxStats(                                                                    /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, P2VAR(Eth_RxStatsType, AUTOMATIC, ETH_APPL_DATA) RxStats)                                              /* PRQA S 3432 # JV-01 */
{
  Std_ReturnType LucReturnValue;
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_GETRXSTATS_SID, CtrlIdx);
  /* Report Error to DET, if the stherStats pointer value is NULL */
  if (NULL_PTR == RxStats)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GETRXSTATS_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    Eth_HwGetRxStats((uint32)CtrlIdx, RxStats);
    LucReturnValue = E_OK;
  }

  return (LucReturnValue);
} /* End of API Eth_GetRxStats */
#endif

#if (ETH_GET_TX_STATS_API == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_GetTxStats
**
** Service ID            : 0x1C
**
** Description           : This API Returns the list of Transmission Statistics out of IETF RFC1213 defined with 
**                         Eth_TxStatsType, where the  maximal possible value shall denote an invalid value.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**
** InOut Parameters      : None
**
** Output Parameters     : TxStats: List of values to read statistic values for
**                                  transmission.
**
** Return parameter      : Std_ReturnType
**                         E_OK: success
**                         E_NOT_OK: Tx-statistics could not be obtained
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration 
**                         parameter EthGetTxStatsApi.
**
** Global Variables Used : None
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_HwGetTxStats
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_013, ETH_DUD_ACT_013_ERR001
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE)Eth_GetTxStats(                                                                    /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, P2VAR(Eth_TxStatsType, AUTOMATIC, ETH_APPL_DATA) TxStats)                                              /* PRQA S 3432 # JV-01 */
{
  Std_ReturnType LucReturnValue;
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_GETTXSTATS_SID, CtrlIdx);
  /* Report Error to DET, if the stherStats pointer value is NULL */
  if (NULL_PTR == TxStats)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GETTXSTATS_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    Eth_HwGetTxStats((uint32)CtrlIdx, TxStats);
    LucReturnValue = E_OK;
  }

  return (LucReturnValue);
} /* End of API Eth_GetTxStats */
#endif

#if (ETH_GET_TX_ERROR_COUNTER_VALUES_API == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_GetTxErrorCounterValues
**
** Service ID            : 0x1D
**
** Description           : This API Returns the list of Transmission Error Counters out of IETF RFC1213 and RFC1643 
**                         defined with Eth_TxErrorCounterValuesType, where the maximal possible value shall denote 
**                         an invalid value.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**
** InOut Parameters      : None
**
** Output Parameters     : TxErrorCounterValues: List of values to read statistic values for transmission.
**
** Return parameter      : Std_ReturnType
**                         E_OK: success
**                         E_NOT_OK: Tx-statistics could not be obtained
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration 
**                         parameter EthGetTxErrorCounterValuesApi.
**
** Global Variables Used : None
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_HwGetTxErrorCounterValues
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_014, ETH_DUD_ACT_014_ERR001
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE)Eth_GetTxErrorCounterValues(                                                       /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, P2VAR(Eth_TxErrorCounterValuesType, AUTOMATIC, ETH_APPL_DATA) TxErrorCounterValues)                    /* PRQA S 3432 # JV-01 */
{
  Std_ReturnType LucReturnValue;
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_GETTXERRORCOUNTERVALUES_SID, CtrlIdx);
  /* Report Error to DET, if the stherStats pointer value is NULL */
  if (NULL_PTR == TxErrorCounterValues)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID,
      ETH_GETTXERRORCOUNTERVALUES_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    Eth_HwGetTxErrorCounterValues((uint32)CtrlIdx, TxErrorCounterValues);
    LucReturnValue = E_OK;
  }

  return (LucReturnValue);
} /* End of API Eth_GetTxErrorCounterValues */
#endif

/***********************************************************************************************************************
** Function Name         : Eth_ProvideTxBuffer
**
** Service ID            : 0x09
**
** Description           : This API Provides access to a transmit buffer of the specified controller
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : CtrlIdx:  Index of the controller
**                         Priority: Frame priority for transmit buffer FIFO selection *This parameter is valid for
**                                   AUTOSAR R4.3.1 later
**
** InOut Parameters      : LenBytePtr: In:  desired length in bytes,
**                                     Out: granted length in bytes
**
** Output Parameters     : BufIdxPtr: Index to the granted buffer resource
**                         BufPtr:    Pointer to the granted buffer
**
** Return parameter      : BufReq_ReturnType
**                         BUFREQ_OK:       success
**                         BUFREQ_E_NOT_OK: development error detected
**                         BUFREQ_E_BUSY:   all buffers in use
**                         BUFREQ_E_OVFL:   the requested size is larger than the buffer
**
** Preconditions         : Component Requires previous controller initialization using Eth_Init.
**
** Global Variables Used : Eth_GaaCtrlStat
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_GetTxBuffer
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_015, ETH_DUD_ACT_015_ERR001,
** Reference ID          : ETH_DUD_ACT_015_ERR002, ETH_DUD_ACT_015_ERR003,
** Reference ID          : ETH_DUD_ACT_015_ERR004
***********************************************************************************************************************/
FUNC(BufReq_ReturnType, ETH_PUBLIC_CODE) Eth_ProvideTxBuffer(                                                           /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx,
  #if (ETH_AR_VERSION >= ETH_AR_431_VERSION)
  uint8 Priority,
  #endif
  P2VAR(Eth_BufIdxType, AUTOMATIC, ETH_APPL_DATA) BufIdxPtr,                                                            /* PRQA S 3432 # JV-01 */
  P2VAR(uint8*, AUTOMATIC, ETH_APPL_DATA) BufPtr,                                                                       /* PRQA S 3432 # JV-01 */
  P2VAR(uint16, AUTOMATIC, ETH_APPL_DATA) LenBytePtr)                                                                   /* PRQA S 3432 # JV-01 */
{
  BufReq_ReturnType LenReturnValue;

  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  Std_ReturnType LucCommonResult;
  LenReturnValue = BUFREQ_OK;
  LucCommonResult = Eth_CommonDetCheck(ETH_PROVIDETXBUFFER_SID, CtrlIdx);
  if (E_OK != LucCommonResult)
  {
    LenReturnValue = BUFREQ_E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  /* Report Error to DET, if the BufIdxPtr pointer value is NULL */
  if (NULL_PTR == BufIdxPtr)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_PROVIDETXBUFFER_SID, ETH_E_PARAM_POINTER);
    LenReturnValue = BUFREQ_E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (NULL_PTR == BufPtr)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_PROVIDETXBUFFER_SID, ETH_E_PARAM_POINTER);
    LenReturnValue = BUFREQ_E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (NULL_PTR == LenBytePtr)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_PROVIDETXBUFFER_SID, ETH_E_PARAM_POINTER);
    LenReturnValue = BUFREQ_E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (BUFREQ_OK != LenReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    /* Report Error to DET, if the controller mode Is not Active */
    if (ETH_MODE_ACTIVE != Eth_GaaCtrlStat[CtrlIdx].enMode)                                                             /* PRQA S 2844 # JV-01 */
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_PROVIDETXBUFFER_SID, ETH_E_INV_MODE);
      #endif
      LenReturnValue = BUFREQ_E_NOT_OK;
    }
    else
    {
      /* Get a buffer from the TxBufferRing */
      #if (ETH_AR_VERSION >= ETH_AR_431_VERSION)
      LenReturnValue = Eth_GetTxBuffer((uint32)CtrlIdx, Priority, BufIdxPtr, BufPtr, LenBytePtr);
      #else
      LenReturnValue = Eth_GetTxBuffer((uint32)CtrlIdx, BufIdxPtr, BufPtr, LenBytePtr);
      #endif
    }
  }
  return(LenReturnValue);
} /* End of API Eth_ProvideTxBuffer */

/***********************************************************************************************************************
** Function Name         : Eth_Transmit
**
** Service ID            : 0x0A
**
** Description           : This API triggers transmission of a previously filled transmit buffer
**
** Sync/Async            : Asynchronous
**
** Reentrancy            : Reentrant for different buffer indexes and
**                         Ctrl indexes
**
** Input Parameters      : CtrlIdx:        Index of the controller
**                         BufIdx:         Index of the buffer resource
**                         FrameType:      Ethernet frame type
**                         TxConfirmation: Activates transmission confirmation
**                         LenByte:        Data length in byte
**                         PhysAddrPtr:    Physical target address (MAC address)
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Std_ReturnType
**                         E_OK: Success.
**                         E_NOT_OK: Transmission failed.
**
** Preconditions         : Component Requires previous controller initialization using Eth_Init.
**
** Global Variables Used : Eth_GaaCtrlStat, Eth_GaaTxBufferTotal
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck,
**                         Eth_CheckProvideBuffer, Eth_PreprocessFrame,
**                         Eth_HwTransmit
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_016, ETH_DUD_ACT_016_ERR001,
** Reference ID          : ETH_DUD_ACT_016_ERR002, ETH_DUD_ACT_016_ERR003,
** Reference ID          : ETH_DUD_ACT_016_ERR004
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE) Eth_Transmit(                                                                     /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, Eth_BufIdxType BufIdx, Eth_FrameType FrameType, boolean TxConfirmation,
  uint16 LenByte, P2CONST(uint8, AUTOMATIC, ETH_APPL_DATA) PhysAddrPtr)
{
  Std_ReturnType LucReturnValue;
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_TRANSMIT_SID, CtrlIdx);
  /* Report Error to DET, if the PhysAddrPtr pointer value is NULL */
  if (NULL_PTR == PhysAddrPtr)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_TRANSMIT_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  /* ETNC need #define ETH_TX_DESC_MAX ETH_TX_BUF_TOTAL_0 */
  /* Report Error to DET, if the BufIdx is out of range */
  #if (ETH_USING_MACRO == ETH_MACRO_ETNC || ETH_USING_MACRO == ETH_MACRO_ETNB)
  if ((uint32)ETH_TX_DESC_MAX <= BufIdx)
  #else
  if (Eth_GaaTxBufferTotal[CtrlIdx] <= BufIdx)
  #endif
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_TRANSMIT_SID, ETH_E_INV_PARAM);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    /* Report Error to DET, if the controller mode Is not Active */
    if (ETH_MODE_ACTIVE != Eth_GaaCtrlStat[CtrlIdx].enMode)                                                             /* PRQA S 2844 # JV-01 */
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_TRANSMIT_SID, ETH_E_INV_MODE);
      #endif
      LucReturnValue = E_NOT_OK;
    }
    else
    {
      LucReturnValue = Eth_CheckProvideBuffer((uint32)CtrlIdx, (uint32)BufIdx);
      /* If the buffer has not been provided, report the error */
      if (E_OK != LucReturnValue)
      {
        #if (ETH_DEV_ERROR_DETECT == STD_ON)
        (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_TRANSMIT_SID, ETH_E_INV_PARAM);
        #endif
      }
      else
      {
        /* Prepare a frame for transmission.*/
        #if (ETH_AR_VERSION >= ETH_AR_431_VERSION)
        Eth_PreprocessFrame((uint32)CtrlIdx, (uint32)BufIdx, (uint32)FrameType, PhysAddrPtr, &LenByte);                 /* PRQA S 1339 # JV-01 */
        #else
        Eth_PreprocessFrame((uint32)CtrlIdx, (uint32)BufIdx, (uint32)FrameType, PhysAddrPtr);
        #endif
        /* Send a transmit request to the low level driver */
        LucReturnValue = Eth_HwTransmit((uint32)CtrlIdx, (uint32)BufIdx, (uint32)LenByte, TxConfirmation);
      }
    }
  }
  return LucReturnValue;
} /* End of API Eth_Transmit */

#if (ETH_CTRL_ENABLE_RX_POLLING == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_Receive
**
** Service ID            : 0x0B
**
** Description           : This API Triggers frame reception
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant for different FIFOs.
**                         Non Reentrant for the same FIFO.
**
** Input Parameters      : CtrlIdx: Index of the controller
**                         FifoIdx: Specifies the related FIFO
**                             *This parameter is valid for AUTOSAR R4.3.1 later
**
** InOut Parameters      : None
**
** Output Parameters     : RxStatusPtr: Indicates whether a frame has been received and if so, whether more frames
**                                      are available or frames got lost.
**
** Return parameter      : None
**
** Preconditions         : Component Requires previous controller initialization using Eth_Init.
**
** Global Variables Used : Eth_GaaCtrlStat
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck,
**                         Eth_HwCheckFifoIndex, Eth_HwReceive
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_017, ETH_DUD_ACT_017_ERR001,
** Reference ID          : ETH_DUD_ACT_017_ERR002, ETH_DUD_ACT_017_ERR003
***********************************************************************************************************************/
FUNC(void, ETH_PUBLIC_CODE) Eth_Receive(                                                                                /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx,
  #if (ETH_AR_VERSION >= ETH_AR_431_VERSION)
  uint8 FifoIdx,
  #endif
  P2VAR(Eth_RxStatusType, AUTOMATIC, ETH_APPL_DATA) RxStatusPtr)                                                        /* PRQA S 3432 # JV-01 */
{
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  Std_ReturnType LucReturnValue;
  LucReturnValue = Eth_CommonDetCheck(ETH_RECEIVE_SID, CtrlIdx);
  #if (ETH_AR_VERSION >= ETH_AR_431_VERSION)
  if (E_OK == LucReturnValue)
  {
    LucReturnValue = Eth_HwCheckFifoIndex((uint32)CtrlIdx, (uint32)FifoIdx);
    if (E_OK != LucReturnValue)
    {
      /* Report Error to DET, if the FifoIdx is out of range */
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_RECEIVE_SID, ETH_E_INV_PARAM);
    }
    else
    {
      /* No action required */
    }
  }
  else
  {
    /* No action required */
  }
  #endif
  /* Report Error to DET, if the RxStatusPtr pointer value is NULL */
  if (NULL_PTR == RxStatusPtr)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_RECEIVE_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* Initial Value Set */
    *RxStatusPtr = ETH_NOT_RECEIVED;
  }
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif /* (ETH_DEV_ERROR_DETECT == STD_ON) */
  {
    /* Report Error to DET, if the controller mode Is not Active */
    if (ETH_MODE_ACTIVE != Eth_GaaCtrlStat[CtrlIdx].enMode)                                                             /* PRQA S 2844 # JV-01 */
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_RECEIVE_SID, ETH_E_INV_MODE);
      #endif
      *RxStatusPtr = ETH_NOT_RECEIVED;                                                                                  /* PRQA S 2814 # JV-01 */
    }
    else
    {
      /* Update Descriptor */
      #if (ETH_AR_VERSION >= ETH_AR_431_VERSION)
      *RxStatusPtr = Eth_HwReceive((uint32)CtrlIdx, (uint32)FifoIdx);
      #else
      *RxStatusPtr = Eth_HwReceive((uint32)CtrlIdx);
      #endif
    }
  }
}
#endif /* (ETH_CTRL_ENABLE_RX_POLLING == STD_ON) */

#if (ETH_CTRL_ENABLE_TX_POLLING == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_TxConfirmation
**
** Service ID            : 0x0C
**
** Description           : This API Triggers frame transmission confirmation
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : Component Requires previous controller initialization
**                         using Eth_Init.
**
** Global Variables Used : Eth_GaaCtrlStat
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck,
**                         Eth_HwTxConfirmation
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_018, ETH_DUD_ACT_018_ERR001
***********************************************************************************************************************/
FUNC(void, ETH_PUBLIC_CODE) Eth_TxConfirmation(                                                                         /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx)
{
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  Std_ReturnType LucReturnValue;
  LucReturnValue = Eth_CommonDetCheck(ETH_TXCONFIRMATION_SID, CtrlIdx);
  /* Do not execute at ETH_E_INV_CTRL_IDX */
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    /* Report Error to DET, if the controller mode Is not Active */
    if (ETH_MODE_ACTIVE != Eth_GaaCtrlStat[CtrlIdx].enMode)                                                             /* PRQA S 2844 # JV-01 */
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_TXCONFIRMATION_SID, ETH_E_INV_MODE);
      #endif
    }
    else
    {
      Eth_HwTxConfirmation((uint32)CtrlIdx);
    }
  }
}
#endif /* (ETH_CTRL_ENABLE_TX_POLLING == STD_ON) */

#if (ETH_VERSION_INFO_API == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_GetVersionInfo
**
** Service ID            : 0x0D
**
** Description           : This API returns the version information of Ethernet driver component.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : VersionInfoPtr : Pointer to where is stored the version information of this module
**
** Return parameter      : None
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration 
**                         parameter ETH_VERSION_INFO_API.
**
** Global Variables      : None
**
** Functions invoked     : Det_ReportError
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_019, ETH_DUD_ACT_019_ERR001
***********************************************************************************************************************/
FUNC(void, ETH_PUBLIC_CODE) Eth_GetVersionInfo(                                                                         /* PRQA S 1503 # JV-01 */
  P2VAR(Std_VersionInfoType, AUTOMATIC, ETH_APPL_DATA)VersionInfoPtr)                                                   /* PRQA S 3432 # JV-01 */
{
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  /* Check if parameter passed is equal to Null pointer */
  if (NULL_PTR == VersionInfoPtr)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GET_VERSION_INFO_SID, ETH_E_PARAM_POINTER);
  }
  else
  #endif
  {
    VersionInfoPtr->vendorID = (uint16)ETH_VENDOR_ID;
    VersionInfoPtr->moduleID = (uint16)ETH_MODULE_ID;
    VersionInfoPtr->sw_major_version = (uint8)ETH_SW_MAJOR_VERSION;
    VersionInfoPtr->sw_minor_version = (uint8)ETH_SW_MINOR_VERSION;
    VersionInfoPtr->sw_patch_version = (uint8)ETH_SW_PATCH_VERSION;
  }
}
#endif /* (STD_ON == ETH_VERSION_INFO_API ) */

/***********************************************************************************************************************
** Function Name         : Eth_MainFunction
**
** Service ID            : 0x20
**
** Description           : The function checks for controller errors and lost frames. Used for polling state changes.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variables      : Eth_GenDriverState
**
** Functions invoked     : Eth_HwMainFunction
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_020
***********************************************************************************************************************/
FUNC(void, ETH_PUBLIC_CODE) Eth_MainFunction(void)                                                                      /* PRQA S 1503 # JV-01 */
{
  Eth_StateType LenState;

  LenState = Eth_GenDriverState;
  if (ETH_STATE_INIT != LenState)
  {
    /* No Action required */
  }
  else
  {
    /* Error check and device specific operations */
    Eth_HwMainFunction();
  }
}

#if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_GetCurrentTime
**
** Service ID            : 0x16
**
** Description           : Returns a time value out of the HW registers according to the capability of the HW
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**
** InOut Parameters      : None
**
** Output Parameters     : timeQualPtr:  Quality of the HW TimeStamp returned
**                         timeStampPtr: Current TimeStamp value
**
** Return parameter      : Std_ReturnType
**                         E_OK: Success
**                         E_NOT_OK: Failed
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration 
**                         parameter EthGlobalTimeSupport.
**
** Global Variables Used : Eth_GaaCtrlStat,
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_HwGetCurrentTime
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_021, ETH_DUD_ACT_021_ERR001,
** Reference ID          : ETH_DUD_ACT_021_ERR002
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE) Eth_GetCurrentTime(                                                               /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, P2VAR(Eth_TimeStampQualType, AUTOMATIC, ETH_APPL_DATA) timeQualPtr,                                    /* PRQA S 3432 # JV-01 */
  P2VAR(Eth_TimeStampType, AUTOMATIC, ETH_APPL_DATA) timeStampPtr)                                                      /* PRQA S 3432 # JV-01 */
{
  Std_ReturnType LucReturnValue;
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_GETCURRENTTIME_SID, CtrlIdx);
  if ((NULL_PTR == timeQualPtr) || (NULL_PTR == timeStampPtr))
  {
    /* Report Error to DET */
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GETCURRENTTIME_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No Action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No Action required */
  }
  else
  #endif
  {
    if (ETH_MODE_ACTIVE != Eth_GaaCtrlStat[CtrlIdx].enMode)                                                             /* PRQA S 2844 # JV-01 */
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      /* Report Error to DET */
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GETCURRENTTIME_SID, ETH_E_INV_MODE);
      #endif
      LucReturnValue = E_NOT_OK;
    }
    else
    {
      LucReturnValue = Eth_HwGetCurrentTime((uint32)CtrlIdx, timeQualPtr, timeStampPtr);
    }
  }
  return (LucReturnValue);
}

/***********************************************************************************************************************
** Function Name         : Eth_EnableEgressTimeStamp
**
** Service ID            : 0x17
**
** Description           : Enable TimeStamp capture for the message that will be transmitted
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**                         BufIdx:  Index of the message buffer
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration 
**                         parameter EthGlobalTimeSupport.
**
** Global Variables Used : None
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_FindTxBufferHandler, Eth_CheckProvideBuffer
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_022,
** Reference ID          : ETH_DUD_ACT_022_ERR001
***********************************************************************************************************************/
FUNC(void, ETH_PUBLIC_CODE) Eth_EnableEgressTimeStamp(                                                                  /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, Eth_BufIdxType BufIdx)
{
  P2VAR(Eth_BufHandlerType, AUTOMATIC, ETH_APPL_DATA) LpBufHandlerPtr;                                                  /* PRQA S 3432 # JV-01 */
  Std_ReturnType LucErrorValue;

  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  /* Local variable to hold the DET value */
  LucErrorValue = Eth_CommonDetCheck(ETH_ENEGRESSTS_SID, CtrlIdx);
  if (E_OK != LucErrorValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    LucErrorValue = Eth_CheckProvideBuffer((uint32)CtrlIdx, (uint32)BufIdx);
    /* Invalid Buffer */
    if (E_OK != LucErrorValue)
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_ENEGRESSTS_SID, ETH_E_INV_PARAM);
      #endif
    }
    else
    {
      /* This will be written in the Ram Descriptor on the next Eth_Transmit call */
      LpBufHandlerPtr = Eth_FindTxBufferHandler((uint32)CtrlIdx, (uint32)BufIdx);
      if (NULL_PTR != LpBufHandlerPtr)
      {
        LpBufHandlerPtr->blbenableTS = ETH_TRUE;
      }
      else
      {
        /* No action required */
      }
    }
  }
}

/***********************************************************************************************************************
** Function Name         : Eth_GetEgressTimeStamp
**
** Service ID            : 0x18
**
** Description           : Reads back the egress time stamp on a dedicated message object.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**                         BufIdx:  Index of the message buffer
**
** InOut Parameters      : None
**
** Output Parameters     : timeQualPtr:  Quality of HW time stamp
**                         timeStampPtr: Current time stamp
**
** Return parameter      : None
**                         For R19-11 or later, Std_ReturnType
**                                              E_OK: Success
**                                              E_NOT_OK: Failed
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration 
**                         parameter EthGlobalTimeSupport.
**
** Global Variables Used :
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck,
**                         Eth_HwGetEgressTimeStamp
**
** Registers Used        : Eth_GaaCtrlStat
**
** Reference ID          : ETH_DUD_ACT_023, ETH_DUD_ACT_023_ERR001,
** Reference ID          : ETH_DUD_ACT_023_ERR002
***********************************************************************************************************************/
#if (ETH_AR_VERSION >= ETH_AR_R19_11_VERSION)
FUNC(Std_ReturnType, ETH_PUBLIC_CODE) Eth_GetEgressTimeStamp(
  uint8 CtrlIdx, Eth_BufIdxType BufIdx,
  P2VAR(Eth_TimeStampQualType, AUTOMATIC, ETH_APPL_DATA) timeQualPtr,
  P2VAR(Eth_TimeStampType, AUTOMATIC, ETH_APPL_DATA) timeStampPtr)
#else
FUNC(void, ETH_PUBLIC_CODE) Eth_GetEgressTimeStamp(                                                                     /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, Eth_BufIdxType BufIdx,
  P2VAR(Eth_TimeStampQualType, AUTOMATIC, ETH_APPL_DATA) timeQualPtr,                                                   /* PRQA S 3432 # JV-01 */
  P2VAR(Eth_TimeStampType, AUTOMATIC, ETH_APPL_DATA) timeStampPtr)                                                      /* PRQA S 3432 # JV-01 */
#endif
{
  #if (ETH_AR_VERSION >= ETH_AR_R19_11_VERSION)
  Std_ReturnType LucReturnValue;
  #endif

  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  Std_ReturnType LucErrorValue;
  /* Local variable to hold the DET value */
  LucErrorValue = Eth_CommonDetCheck(ETH_GETEGRESSTS_SID, CtrlIdx);
  if ((NULL_PTR == timeQualPtr) || (NULL_PTR == timeStampPtr))
  {
    /* Report Error to DET */
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GETEGRESSTS_SID, ETH_E_PARAM_POINTER);
    LucErrorValue = E_NOT_OK;
  }
  else
  {
    /* No Action required */
  }
  if (E_OK != LucErrorValue)
  {
    #if (ETH_AR_VERSION >= ETH_AR_R19_11_VERSION)
    LucReturnValue = E_NOT_OK;
    #endif
  }
  else
  #endif
  {
    if (ETH_MODE_ACTIVE != Eth_GaaCtrlStat[CtrlIdx].enMode)                                                             /* PRQA S 2844 # JV-01 */
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      /* Report Error to DET */
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GETEGRESSTS_SID, ETH_E_INV_MODE);
      #endif
      #if (ETH_AR_VERSION >= ETH_AR_R19_11_VERSION)
      LucReturnValue = E_NOT_OK;
      #endif
    }
    else
    {
      #if (ETH_AR_VERSION >= ETH_AR_R19_11_VERSION)
      LucReturnValue = Eth_HwGetEgressTimeStamp((uint32)CtrlIdx, BufIdx, timeQualPtr, timeStampPtr);
      #else
      Eth_HwGetEgressTimeStamp((uint32)CtrlIdx, BufIdx, timeQualPtr, timeStampPtr);
      #endif
    }
  }

  #if (ETH_AR_VERSION >= ETH_AR_R19_11_VERSION)
  return LucReturnValue;
  #endif
}

/***********************************************************************************************************************
** Function Name         : Eth_GetIngressTimeStamp
**
** Service ID            : 0x19
**
** Description           : Return TimeStamp from the HW register
**                         previously started
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**                         DataPtr: Pointer to the message buffer
**
** InOut Parameters      : None
**
** Output Parameters     : timeQualPtr:  Quality of HW time stamp
**                         timeStampPtr: Current time stamp
**
** Return parameter      : None
**                         For R19-11 or later, Std_ReturnType
**                                              E_OK: Success
**                                              E_NOT_OK: Failed
**
** Preconditions         : This function is pre-compile time configurable
**                        (STD_ON/STD_OFF) by the configuration parameter
**                         EthGlobalTimeSupport.
**
** Global Variables Used : Eth_GaaCtrlStat
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck,
**                         Eth_HwGetIngressTimeStamp
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_024, ETH_DUD_ACT_024_ERR001,
** Reference ID          : ETH_DUD_ACT_024_ERR002
***********************************************************************************************************************/
#if (ETH_AR_VERSION >= ETH_AR_R19_11_VERSION)
FUNC(Std_ReturnType, ETH_PUBLIC_CODE) Eth_GetIngressTimeStamp(
  uint8 CtrlIdx, P2CONST(Eth_DataType, AUTOMATIC, ETH_APPL_DATA) DataPtr,
  P2VAR(Eth_TimeStampQualType, AUTOMATIC, ETH_APPL_DATA) timeQualPtr,
  P2VAR(Eth_TimeStampType, AUTOMATIC, ETH_APPL_DATA) timeStampPtr)
#else
FUNC(void, ETH_PUBLIC_CODE) Eth_GetIngressTimeStamp(                                                                    /* PRQA S 1503 # JV-01 */
  uint8 CtrlIdx, P2CONST(Eth_DataType, AUTOMATIC, ETH_APPL_DATA) DataPtr,
  P2VAR(Eth_TimeStampQualType, AUTOMATIC, ETH_APPL_DATA) timeQualPtr,                                                   /* PRQA S 3432 # JV-01 */
  P2VAR(Eth_TimeStampType, AUTOMATIC, ETH_APPL_DATA) timeStampPtr)                                                      /* PRQA S 3432 # JV-01 */
#endif
{
  #if (ETH_AR_VERSION >= ETH_AR_R19_11_VERSION)
  Std_ReturnType LucReturnValue;
  #endif

  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  Std_ReturnType LucErrorValue;
  /* Local variable to hold the DET value */
  LucErrorValue = Eth_CommonDetCheck(ETH_GETINGRESSTS_SID, CtrlIdx);
  if ((NULL_PTR == DataPtr) || (NULL_PTR == timeQualPtr) || (NULL_PTR == timeStampPtr))
  {
    /* Report Error to DET */
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GETINGRESSTS_SID, ETH_E_PARAM_POINTER);
    LucErrorValue = E_NOT_OK;
  }
  else
  {
    /* No Action required */
  }
  if (E_OK != LucErrorValue)
  {
    #if (ETH_AR_VERSION >= ETH_AR_R19_11_VERSION)
    LucReturnValue = E_NOT_OK;
    #endif
  }
  else
  #endif
  {
    if (ETH_MODE_ACTIVE != Eth_GaaCtrlStat[CtrlIdx].enMode)                                                             /* PRQA S 2844 # JV-01 */
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      /* Report Error to DET */
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_GETINGRESSTS_SID, ETH_E_INV_MODE);
      #endif
      #if (ETH_AR_VERSION >= ETH_AR_R19_11_VERSION)
      LucReturnValue = E_NOT_OK;
      #endif
    }
    else
    {
      #if (ETH_AR_VERSION >= ETH_AR_R19_11_VERSION)
      LucReturnValue = Eth_HwGetIngressTimeStamp((uint32)CtrlIdx, DataPtr, timeQualPtr, timeStampPtr);
      #else
      Eth_HwGetIngressTimeStamp((uint32)CtrlIdx, DataPtr, timeQualPtr, timeStampPtr);
      #endif
    }
  }

  #if (ETH_AR_VERSION >= ETH_AR_R19_11_VERSION)
  return LucReturnValue;
  #endif
}

#if (ETH_AR_VERSION == ETH_AR_422_VERSION)
/***********************************************************************************************************************
** Function Name         : Eth_SetCorrectionTime
**
** Service ID            : 0x1A
**
** Description           : Allows the Time Slave to adjust the local ETH
**                         Reference clock in HW
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant

** Input Parameters      : CtrlIdx:       Index of the controller
**                         timeOffsetPtr: Offset between time stamp grandmaster and time stamp by local clock
**                         rateRatioPtr:  Time elements to calculate and to modify the ratio of the frequency of the
**                                      grandmaster in relation to the frequency of the Local Clock.
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration
**                         parameter EthGlobalTimeSupport.
**
** Global Variables Used : Eth_GaaCtrlStat
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_HwSetCorrectionTime
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_025, ETH_DUD_ACT_025_ERR001,
** Reference ID          : ETH_DUD_ACT_025_ERR002
***********************************************************************************************************************/
FUNC(void, ETH_PUBLIC_CODE) Eth_SetCorrectionTime(
  uint8 CtrlIdx, P2CONST(Eth_TimeIntDiffType, AUTOMATIC, ETH_APPL_DATA) timeOffsetPtr,
  P2CONST(Eth_RateRatioType,AUTOMATIC, ETH_APPL_DATA) rateRatioPtr)
{
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  Std_ReturnType LucErrorValue;
  /* Local variable to hold the DET value */
  LucErrorValue = Eth_CommonDetCheck(ETH_SETCORRECTIONTIME_SID, CtrlIdx);
  if ((NULL_PTR == timeOffsetPtr) || (NULL_PTR == rateRatioPtr))
  {
    /* Report Error to DET */
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_SETCORRECTIONTIME_SID, ETH_E_PARAM_POINTER);
    LucErrorValue = E_NOT_OK;
  }
  else
  {
    /* No Action required */
  }
  if (E_OK != LucErrorValue)
  {
    /* No Action required */
  }
  else
  #endif
  {
    if (ETH_MODE_ACTIVE != Eth_GaaCtrlStat[CtrlIdx].enMode)
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      /* Report Error to DET */
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_SETCORRECTIONTIME_SID, ETH_E_INV_MODE);
      #endif
    }
    else
    {
      (void)Eth_HwSetCorrectionTime(CtrlIdx, timeOffsetPtr);
    }
  }
}

/***********************************************************************************************************************
** Function Name         : Eth_SetGlobalTime
**
** Service ID            : 0x1B
**
** Description           : Allows the Time Master to adjust the global ETH
**                         Reference clock in HW.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**                         timeStampPtr: New time stamp
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Std_ReturnType
**                         E_OK: Success
**                         E_NOT_OK: Failed
**
** Preconditions         : This function is pre-compile time configurable (STD_ON/STD_OFF) by the configuration 
**                         parameter EthGlobalTimeSupport.
**
** Global Variables Used : Eth_GaaCtrlStat
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_HwSetGlobalTime
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_026, ETH_DUD_ACT_026_ERR001,
** Reference ID          : ETH_DUD_ACT_026_ERR002
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE) Eth_SetGlobalTime(
  uint8 CtrlIdx, P2CONST(Eth_TimeStampType, AUTOMATIC, ETH_APPL_DATA) timeStampPtr)
{
  Std_ReturnType LucReturnValue;
  LucReturnValue = E_OK;
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_SETGLOBALTIME_SID, CtrlIdx);
  if (NULL_PTR == timeStampPtr)
  {
    /* Report Error to DET */
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_SETGLOBALTIME_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No Action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No Action required */
  }
  else
  #endif
  {
    if (ETH_MODE_ACTIVE != Eth_GaaCtrlStat[CtrlIdx].enMode)
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      /* Report Error to DET */
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_SETGLOBALTIME_SID, ETH_E_INV_MODE);
      #endif
      LucReturnValue = E_NOT_OK;
    }
    else
    {
      LucReturnValue = Eth_HwSetGlobalTime(CtrlIdx, timeStampPtr);
    }
  }
  return (LucReturnValue);
}
#endif /* (ETH_AR_VERSION == ETH_AR_422_VERSION) */

#if (ETH_AR_VERSION >= ETH_AR_431_VERSION)
/***********************************************************************************************************************
** Function Name         : Eth_SetIncrementTimeForGptp
**
** Service ID            : 0xA1
**
** Description           : Sets the gPTP timer increment with the specified value.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx: Index of the controller
**                         IncVal:  Increment value for gPTP timer
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Std_ReturnType
**                         E_OK: gPTP increment register was successfully changed
**                         E_NOT_OK: gPTP increment register could not be changed and the DET is ON
**
** Preconditions         : Component Requires previous controller initialization using Eth_Init.
**
** Global Variables Used : Eth_GaaCtrlStat
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck,
**                         Eth_HwSetIncrementTimeForGptp
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_027,
** Reference ID          : ETH_DUD_ACT_027_ERR002, ETH_DUD_ACT_027_ERR003
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE) Eth_SetIncrementTimeForGptp(                                                      /* PRQA S 1503 # JV-01 */
  CONST(uint8, AUTOMATIC) CtrlIdx, CONST(uint32, AUTOMATIC) IncVal)
{
  Std_ReturnType LucReturnValue;

  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_SETINCREMENTTIMEFORGPTP_SID, CtrlIdx);
  if (E_OK != LucReturnValue)
  {
    /* No Action required */
  }
  else
  #endif
  {
    if (ETH_MODE_ACTIVE != Eth_GaaCtrlStat[CtrlIdx].enMode)                                                             /* PRQA S 2844 # JV-01 */
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      /* Report Error to DET, if the controller mode Is Standby */
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_SETINCREMENTTIMEFORGPTP_SID, ETH_E_INV_MODE);
      #endif
      LucReturnValue = E_NOT_OK;
    }
    #if (ETH_USING_MACRO == ETH_MACRO_ETNB)
    else if ((ETH_GPTP_INC_MINVALUE > IncVal) || (ETH_GPTP_INC_MAXVALUE < IncVal))
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      /* Report Error to DET, if the increment value for gPTP timer out of range */
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_SETINCREMENTTIMEFORGPTP_SID, ETH_E_INV_PARAM);
      #endif
      LucReturnValue = E_NOT_OK;
    }
    #endif
    else
    {
      LucReturnValue = Eth_HwSetIncrementTimeForGptp((uint32)CtrlIdx, IncVal);
    }
  }
  return LucReturnValue;
}

/***********************************************************************************************************************
** Function Name         : Eth_SetOffsetTimeForGptp
**
** Service ID            : 0xA2
**
** Description           : Sets the gPTP timer offset with the specified value.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx:        Index of the controller
**                         pTimeOffsetPtr: Offset value for gPTP timer
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Std_ReturnType
**                         E_OK: gPTP offset register was successfully changed
**                         E_NOT_OK: gPTP offset register could not be changed and the DET is ON
**
** Preconditions         : Component Requires previous controller initialization using Eth_Init.
**
** Global Variables Used : Eth_GaaCtrlStat
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_HwSetOffsetTimeForGptp
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_028,
** Reference ID          : ETH_DUD_ACT_028_ERR002
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE) Eth_SetOffsetTimeForGptp(                                                         /* PRQA S 1503 # JV-01 */
  CONST(uint8, AUTOMATIC) CtrlIdx, CONSTP2CONST(Eth_TimeStampType, AUTOMATIC, ETH_APPL_DATA) pTimeOffsetPtr)
{
  Std_ReturnType LucReturnValue;
  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  /* Local variable to hold the DET value */
  LucReturnValue = Eth_CommonDetCheck(ETH_SETOFFSETTIMEFORGPTP_SID, CtrlIdx);
  if (NULL_PTR == pTimeOffsetPtr)
  {
    /* Report Error to DET */
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID,
      ETH_SETOFFSETTIMEFORGPTP_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    if (ETH_MODE_ACTIVE != Eth_GaaCtrlStat[CtrlIdx].enMode)                                                             /* PRQA S 2844 # JV-01 */
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      /* Report Error to DET */
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_SETOFFSETTIMEFORGPTP_SID, ETH_E_INV_MODE);
      #endif
      LucReturnValue = E_NOT_OK;
    }
    else
    {
      LucReturnValue = Eth_HwSetOffsetTimeForGptp((uint32)CtrlIdx, pTimeOffsetPtr);
    }
  }
  return LucReturnValue;
}
#endif /* (ETH_AR_VERSION >= ETH_AR_431_VERSION) */
#endif /* (ETH_GLOBAL_TIME_SUPPORT == STD_ON) */

#if (ETH_STREAM_FILTERING == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_UpdateStreamFilter
**
** Service ID            : 0xA0
**
** Description           : Update the stream ID for separate filtering with the specified AVB stream receive queue.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : CtrlIdx:     Index of the controller
**                         QueIdx :     Index of the receive queue
**                         StreamIdPtr: Update the stream id for separate filtering
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Std_ReturnType
**                         E_OK: filter was successfully changed
**                         E_NOT_OK: filter could not be changed and the DET is ON
**
** Preconditions         : Component Requires previous controller initialization using Eth_Init.
**
** Global Variables Used : Eth_GenDriverState
**
** Function(s) invoked   : Det_ReportError, Eth_CommonDetCheck, Eth_HwCheckRxStreamQueueIndex
**                         Eth_HwUpdateStreamFilter
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_029, ETH_DUD_ACT_029_ERR001
** Reference ID          : ETH_DUD_ACT_029_ERR002, ETH_DUD_ACT_029_ERR003
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PUBLIC_CODE) Eth_UpdateStreamFilter(                                                           /* PRQA S 1503 # JV-01 */
  CONST(uint8, AUTOMATIC) CtrlIdx, CONST(uint8, AUTOMATIC) QueIdx,
  CONSTP2CONST(uint8, AUTOMATIC, ETH_APPL_DATA) StreamIdPtr)
{
  Std_ReturnType LucReturnValue;

  #if (ETH_DEV_ERROR_DETECT == STD_ON)
  LucReturnValue = Eth_CommonDetCheck(ETH_UPDATESTREAMFILTER_SID, CtrlIdx);
  if (E_OK == LucReturnValue)
  {
    LucReturnValue = Eth_HwCheckRxStreamQueueIndex((uint32)CtrlIdx, (uint32)QueIdx);
    if (E_OK != LucReturnValue)
    {
      /* Report Error to DET */
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_UPDATESTREAMFILTER_SID, ETH_E_INV_PARAM);
    }
    else
    {
      /* No action required */
    }
  }
  else
  {
    /* No action required */
  }
  if (NULL_PTR == StreamIdPtr)
  {
    /* Report Error to DET */
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_UPDATESTREAMFILTER_SID, ETH_E_PARAM_POINTER);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* No action required */
  }
  if (E_OK != LucReturnValue)
  {
    /* No action required */
  }
  else
  #endif
  {
    if ((ETH_MODE_ACTIVE != Eth_GaaCtrlStat[CtrlIdx].enMode)                                                            /* PRQA S 2844 # JV-01 */
       && (ETH_MODE_DOWN != Eth_GaaCtrlStat[CtrlIdx].enMode))
    {
      #if (ETH_DEV_ERROR_DETECT == STD_ON)
      /* Report Error to DET, if the controller mode Is Standby */
      (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, ETH_UPDATESTREAMFILTER_SID, ETH_E_INV_MODE);
      #endif
      LucReturnValue = E_NOT_OK;
    }
    else
    {
      /* Update Filter Pattern for the specific Rx queue */
      Eth_HwUpdateStreamFilter((uint32)CtrlIdx, (uint32)QueIdx, StreamIdPtr);
      LucReturnValue = E_OK;
    }
  }
  return LucReturnValue;
}
#endif /* (ETH_STREAM_FILTERING == STD_ON) */

#define ETH_STOP_SEC_PUBLIC_CODE
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

#define ETH_START_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */
/***********************************************************************************************************************
** Function Name         : Eth_DemConfigCheck
**
** Service ID            : N/A
**
** Description           : Checks if the EventId is configured, and if so, reports a DemEventReport.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : EventId : Identification of an event by assigned EventId.
**                         EventStatus : Monitor test result
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variables      : None
**
** Functions invoked     : Det_ReportError
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_174,
** Reference ID          : ETH_DUD_ACT_174_ERR001
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_DemConfigCheck(
  CONST(Dem_EventIdType, AUTOMATIC) EventId,
  CONST(Dem_EventStatusType, AUTOMATIC) EventStatus)
{
  /* Report PASS to DEM */
  if ((Dem_EventIdType)ETH_DEM_NOT_CONFIGURED == EventId)
  {
    /* No action required */
  }
  else
  {
    ETH_DEM_REPORT_ERROR(EventId, EventStatus);
  }
}

/***********************************************************************************************************************
** Function Name       : Eth_SetStatus
**
** Service ID          : N/A
**
** Description         : This function updates Eth_GenDriverState.
**                       The purpose of this function is to prevent the order of
**                       instructions being changed by the compiler.
**
** Sync/Async          : Synchronous
**
** Reentrancy          : Non Reentrant
**
** Input Parameters    : LenStatus: New status value
**
** InOut Parameters    : None
**
** Output Parameters   : None
**
** Return parameter    : None
**
** Preconditions       : None
**
** Global Variable     : Eth_GenDriverState
**
** Function invoked    : None
**
** Registers Used      : None
**
** Reference ID        : ETH_DUD_ACT_030, ETH_DUD_ACT_030_GBL001
***********************************************************************************************************************/
STATIC FUNC(void, ETH_PRIVATE_CODE) Eth_SetStatus(
  CONST(Eth_StateType, AUTOMATIC) LenStatus)
{
  Eth_GenDriverState = LenStatus;
}

/***********************************************************************************************************************
** Function Name         : Eth_CommonDetCheck
**
** Service ID            : N/A
**
** Description           : This function performs DET checks which are common among almost all APIs.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LucSID:     Service ID of the caller
**                         LucCtrlIdx: Index of the controller
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Std_ReturnType
**                         E_OK     : No error is detected
**                         E_NOT_OK : Any error is detected
**
** Preconditions         : None
**
** Global Variables      : Eth_GenDriverState
**
** Functions invoked     : Det_ReportError
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_031, ETH_DUD_ACT_031_ERR002,
** Reference ID          : ETH_DUD_ACT_031_ERR003
***********************************************************************************************************************/
#if (ETH_DEV_ERROR_DETECT == STD_ON)
STATIC FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_CommonDetCheck(
  CONST(uint8, AUTOMATIC) LucSID, CONST(uint8, AUTOMATIC) LucCtrlIdx)
{
  Std_ReturnType LucReturnValue;
  Eth_StateType LenState;

  LenState = Eth_GenDriverState;
  if (ETH_STATE_INIT != LenState)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, LucSID, ETH_E_UNINIT);
    LucReturnValue = E_NOT_OK;
  }
  else if ((uint32)ETH_TOTAL_CTRL_CONFIG <= LucCtrlIdx)
  {
    (void)Det_ReportError(ETH_MODULE_ID, ETH_INSTANCE_ID, LucSID, ETH_E_INV_CTRL_IDX);
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    LucReturnValue = E_OK;
  }

  return LucReturnValue;
}
#endif /* (ETH_DEV_ERROR_DETECT == STD_ON) */

#define ETH_STOP_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 2814 # JV-01 */
/***********************************************************************************************************************
**                                             End of File                                                            **
***********************************************************************************************************************/
