/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Eth_ETNB_Dma.c                                                                                      */
/* SW-VERSION   = 1.6.0                                                                                               */
/*====================================================================================================================*/
/*                                            COPYRIGHT                                                               */
/*====================================================================================================================*/
/* (c) 2020-2022 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* This file contains DMAC functions implementation of Ethernet Driver  Component.                                    */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s)        */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs     */
/* of program errors, compliance with applicable laws, damage to or loss of data, programs or equipment,              */
/* and unavailability or interruption of operations.                                                                  */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*                                        Devices:        X2x                                                         */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                     Revision Control History                                                       **
***********************************************************************************************************************/
/*
 * 1.5.4: 25/07/2022    : Fix a bug in the implementation of RxBeProcess, RxNcProcess, RxSProcess
 * 1.5.2: 03/03/2022    : Revert the call timing of EthSwt_EthRxFinishedIndication.
 *        09/02/2022    : Fixed an issue that the call timing of the EthSwt_EthRxFinishedIndication was incorrect.
 *        08/02/2022    : Update QA-C comments
 *        28/01/2022    : Change the name of Eth_GaaTempTimeStamp to Eth_GaaRxFrame.
 *        27/01/2022    : Changed the get timestamp method of RxBeProcess, RxNcProcess, RxSProcess
 *        24/01/2022    : Changed location to set the descriptor type.
 *                        Improved the logic for support ETH_ETHSWITCH_MANAGEMENT_SUPPORT.
 * 1.4.2: 06/09/2021    : Fixed an issue where the Eth_Hw_Etnb_DescConfig and SetDescChain function
 *                        would cause out-of-range access.
 *        26/08/2021    : Updated QA-C 9.5.0 comments
 *        25/08/2021    : Fixed an issue where the AllocMemForDesc function would cause out-of-range access.
 *        24/08/2021    : Modify the format to 120 characters
 *        08/09/2021    : Updated QA-C 9.5.0 comments.
 * 1.4.1: 07/05/2021    : Remove vendor ID from the file name, API names
 *                        and parameters according to BSW00347 requirement
 *        09/06/2021    : Remove QA-C Warning 0404 and 3432 according to QA-C 9.5.0
 *                        Initializing parameters input into GetCounterValue function
 *        06/07/2021    : Updated QA-C 9.5.0 comments.
 * 1.4.0: 13/04/2021    : Fixed the algorithm for collecting Rx statistics.
 * 1.3.0: 10/03/2021    : Corrected implement of broadcast massage checking in RxCallEthIf.
 *        09/11/2020    : Corrected implement of Tx/Rx ISR for each controller.
 *        19/11/2020    : Removed RIC2 and RIC3 process in TxRxIntConfig.
 * 1.2.0: 18/08/2020    : Release
 *        18/08/2020    : Updated TxRxConfig() to correct writing value to ETNBnTGC.TQP[1:0].
 *        29/07/2020    : Add QAC 9.3.1 comment.
 * 1.1.0: 19/06/2020    : Release
 * 1.0.1: 04/06/2020    : 1. Removed enBufMode, aaRxBufMode, Eth_59_RxBufModeType.
 *                        Because only use single buffer descripter.
 *                        Removed multi buffer descripter process in
 *                        RxBeProcess, RxNcProcess, RxSProcess, DescUpdate and DescTsUpdate.
 *                        2. End of Eth_59_Hw_Etnb_DisableController, write ETNBnCCC.DTSR
 *                        because suspend mode is a temporary mode and must be released.
 *                        Updated Eth_59_Hw_Etnb_RxIrqHdlr to not clear RIS0 since it will be
 *                        cleared automatically when the Rx queue becomes empty.
 *                        Commented on a cast from a large type to a small type.
 *                        Removed setting for TIC. Because set default value.
 *                        Handles the controller index in uint32.
 *                        Added set TFUE bit of TIC register.
 *                        Update type usage in GetCounterValue().
 *                        Replace Eth_59_GetStatusTxBuffer to Eth_59_CheckProvideBuffer.
 *                        Replace ETH_OS_COUNTER_MAX to ETH_OS_COUNTER_MAX_VALUE.
 *                        Update by full check of the critical section.
 *                        Fixed a bug in Transmit Queue Priority setting.
 *                        Removed ulPHLength, ulPLength.
 *                        Fixed an issue where statistics were not updated when using EthSwt.
 *                        Removed LpChConfig in DescUpdate and DescTsUpdate.
 *                        Updated static analysis result.
 *                        Changed include file structure.
 *                        Fixed violation of AUTOSAR specifications for EthSwt.
 * 1.0.0: 25/03/2020    : Initial Version
 */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                        Include Section                                                             **
***********************************************************************************************************************/
/* Included for module version information and other types declarations */
#include "Eth.h"
/* Included for prototypes for internal functions of Ethernet Component */
#include "Eth_ETNB_Dma.h"
/* Included for Ethernet Component register types used within the module */
#include "Eth_ETNB_Ram.h"

#include "EthIf_Cbk.h"

#if (ETH_CRITICAL_SECTION_PROTECTION == STD_ON)
#include "SchM_Eth.h"
#endif

#if (ETH_ETHSWITCH_MANAGEMENT_SUPPORT == STD_ON)
#include "EthSwt_Cbk.h"
#endif

/***********************************************************************************************************************
**                                          Version Information                                                       **
***********************************************************************************************************************/

/* AUTOSAR release version information */
#define ETH_ETNB_DMA_C_AR_RELEASE_MAJOR_VERSION  ETH_AR_RELEASE_MAJOR_VERSION_VALUE
#define ETH_ETNB_DMA_C_AR_RELEASE_MINOR_VERSION  ETH_AR_RELEASE_MINOR_VERSION_VALUE
#define ETH_ETNB_DMA_C_AR_RELEASE_REVISION_VERSION  ETH_AR_RELEASE_REVISION_VERSION_VALUE

/* File version information */
#define ETH_ETNB_DMA_C_SW_MAJOR_VERSION    ETH_SW_MAJOR_VERSION_VALUE
#define ETH_ETNB_DMA_C_SW_MINOR_VERSION    ETH_SW_MINOR_VERSION_VALUE

/***********************************************************************************************************************
**                                          Version Check                                                             **
***********************************************************************************************************************/

/* Functionality related to R4.0 */
#if (ETH_ETNB_DMA_AR_RELEASE_MAJOR_VERSION !=  ETH_ETNB_DMA_C_AR_RELEASE_MAJOR_VERSION)
  #error "Eth_ETNB_Dma.c : Mismatch in Release Major Version"
#endif
#if (ETH_ETNB_DMA_AR_RELEASE_MINOR_VERSION !=  ETH_ETNB_DMA_C_AR_RELEASE_MINOR_VERSION)
  #error "Eth_ETNB_Dma.c : Mismatch in Release Minor Version"
#endif
#if (ETH_ETNB_DMA_AR_RELEASE_REVISION_VERSION !=  ETH_ETNB_DMA_C_AR_RELEASE_REVISION_VERSION)
  #error "Eth_ETNB_Dma.c : Mismatch in Release Revision Version"
#endif

#if (ETH_ETNB_DMA_SW_MAJOR_VERSION != ETH_ETNB_DMA_C_SW_MAJOR_VERSION)
  #error "Eth_ETNB_Dma.c : Mismatch in Software Major Version"
#endif
#if (ETH_ETNB_DMA_SW_MINOR_VERSION != ETH_ETNB_DMA_C_SW_MINOR_VERSION)
  #error "Eth_ETNB_Dma.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (2:3432)    : Simple macro argument expression is not parenthesized.                                       */
/* Rule                : MISRA C:2012 Rule-20.7                                                                       */
/*                       REFERENCE - ISO:C90-6.3.1 Primary Expressions                                                */
/* JV-01 Justification : Compiler keyword (macro) is defined and used followed AUTOSAR standard rule. It is           */
/*                       accepted.                                                                                    */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3678)    : The object referenced by '%1s' is not modified through it, so '%1s' could be declared        */
/*                       with type '%2s'.                                                                             */
/* Rule                : MISRA C:2012 Rule-8.13                                                                       */
/* JV-01 Justification : The object addressed by this pointer change so it can not be of type "pointer to const"      */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0306)    : [I] Cast between a pointer to object and an integral type.                                   */
/* Rule                : CERTCCM INT36, MISRA C:2012 Rule-11.4                                                        */
/*                       REFERENCE - ISO:C90-6.3.4  Semantics                                                         */
/* JV-01 Justification : The address is cast into a field provided by the hardware. The address passed has been       */
/*                       reviewed and found to be okay.                                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2824)    : Possible: Arithmetic operation on NULL pointer.                                              */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1532)    : The function '%1s' is only referenced in one translation unit - but not the one in which     */
/*                       it is defined.                                                                               */
/* Rule                : CERTCCM DCL19, MISRA C:2012 Rule-8.7                                                         */
/* JV-01 Justification : This is accepted, due to the specific coding rule, function of each hardware unit is         */
/*                       implemented in separated files for this hardware unit. Could not be static function.         */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0316)    : [I] Cast from a pointer to void to a pointer to object type.                                 */
/* Rule                : MISRA C:2012 Rule-11.5                                                                       */
/* JV-01 Justification : The information required for each IP is different, so it is declared as void pointer.        */
/*       Verification  : Out-of-range access does not occur because Generation Tool outputs with Eth_ETNBConfigType.  */
/**********************************************************************************************************************/
/* Message (2:3469)    : This usage of a function-like macro looks like it could be replaced by an equivalent         */
/*                       function call.                                                                               */
/* Rule                : MISRA C:2012 Dir-4.9                                                                         */
/* JV-01 Justification : To prevent overhead and speed up the processing, there is no need to call a function         */
/*                       here for such a small operation.                                                             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3673)    : The object addressed by the pointer parameter '%s' is not modified and so the pointer        */
/*                       could be of type 'pointer to const'.                                                         */
/* Rule                : CERTCCM DCL00, DCL13, MISRA C:2012 Rule-8.13                                                 */
/* JV-01 Justification : Pointer variable is used to modify the value at the address so the pointer cannot be         */
/*                       declared as 'pointer to const' type.                                                         */
/*       Verification  : It cannot be declared in cosnt because it may be overwritten.                                */
/**********************************************************************************************************************/
/* Message (2:0759)    : An object of union type has been defined.                                                    */
/* Rule                : MISRA C:2012 Rule-19.2                                                                       */
/* JV-01 Justification : Data access of larger data types is used to achieve better throughput.                       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3383)    : Cannot identify wraparound guard for unsigned arithmetic expression.                         */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : Wraparound can't be occur in this unsigned integer arithmetic so there is no need to add     */
/*                       a wraparound guard here.                                                                     */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2934)    : Possible: Computing an invalid pointer value.                                                */
/* Rule                : CERTCCM ARR30, ARR37, ARR38, EXP08                                                           */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1505)    : The function 'name' is only referenced in the translation unit where it is defined.          */
/* Rule                : CERTCCM DCL19, MISRA C:2012 Rule-8.7                                                         */
/* JV-01 Justification : This is accepted, due to following coding rule, internal function can be defined in          */
/*                       other C source files                                                                         */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3384)    : Cannot identify wraparound guard for dependent unsigned arithmetic expression.               */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : Wraparound can't be occur in this unsigned integer arithmetic so there is no need to add     */
/*                       a wraparound guard here.                                                                     */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (5:2976)    : Definite: Passing address of partially initialized object '%s' to a function parameter       */
/*                       declared as a pointer to const.                                                              */
/* Rule                : CERTCCM EXP33                                                                                */
/* JV-01 Justification : The message propose that address should be initialized before passed to function             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:2469)    : Loop control variable in this 'for' statement LucCount is modified in the body of the        */
/*                       loop.                                                                                        */
/* Rule                : MISRA C:2012 Rule-14.2                                                                       */
/* JV-01 Justification : This is to exit from the linear search loop when the required channel Id is found.           */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:4391)    : A composite expression of 'essentially unsigned' type (unsigned char) is being cast to a     */
/*                       wider unsigned type 'unsigned short'.                                                        */
/* Rule                : MISRA C:2012 Rule-10.8                                                                       */
/* JV-01 Justification : This is necessary to support configuration. The actuall array can't be decided statically.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0488)    : Performing pointer arithmetic.                                                               */
/* Rule                : CERTCCM EXP08, MISRA C:2012 Rule-18.4                                                        */
/* JV-01 Justification : This implementation is required for heap memory allocation algorithm                         */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:3206)    : The parameter 'ucApiId' is not used in this function.                                        */
/* Rule                : CERTCCM MSC07, MSC13, MISRA C:2012 Rule-2.7                                                  */
/* JV-01 Justification : This variable is used when using interrupt mode. This cannot be avoided because it is        */
/*                       switched by the compile switch by the preprocessor and this variable is an argument          */
/*                       of the function.                                                                             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (7:0404)    : [U] More than one read access to volatile objects between sequence points.                   */
/* Rule                : CERTCCM EXP10, EXP30, MISRA C:2012 Rule 1.3, 13.2                                            */
/* JV-01 Justification : This is to get element in array of struct, volatile of counter variable of 'for' loop is     */
/*                       used to ensure no optimization. It is accepted                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                              Global Data                                                           **
***********************************************************************************************************************/

/***********************************************************************************************************************
**                                              Function Definitions                                                  **
***********************************************************************************************************************/
#define ETH_START_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"

STATIC FUNC(void, ETH_PRIVATE_CODE) RxBeProcess(CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2VAR(Eth_RxSingleFrameType, AUTOMATIC, ETH_APPL_DATA) LpSingleRxFramePtr,                                       /* PRQA S 3432 # JV-01 */
  CONST(uint32, AUTOMATIC) LulDescPtr);

STATIC FUNC(void, ETH_PRIVATE_CODE) RxNcProcess(CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2VAR(Eth_RxSingleFrameType, AUTOMATIC, ETH_APPL_DATA) LpSingleRxFramePtr,                                       /* PRQA S 3432 # JV-01 */
  CONST(uint32, AUTOMATIC) LulDescPtr);

STATIC FUNC(void, ETH_PRIVATE_CODE) RxSProcess(CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2VAR(Eth_RxSingleFrameType, AUTOMATIC, ETH_APPL_DATA) LpSingleRxFramePtr,                                       /* PRQA S 3432 # JV-01 */
  CONST(uint32, AUTOMATIC) LulChannelNum, CONST(uint32, AUTOMATIC) LulDescPtr);

STATIC FUNC(void, ETH_PRIVATE_CODE) SetDescChain(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulAddr, CONST(uint8, AUTOMATIC) LucQidx,
  CONST(Eth_DirectionType, AUTOMATIC) LenDir);

STATIC FUNC(void, ETH_PRIVATE_CODE) SetFempty(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONSTP2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA) LpDescr,                  /* PRQA S 3432 # JV-01 */
  CONST(uint8, AUTOMATIC) LucQidx, CONST(Eth_DirectionType, AUTOMATIC) LenQdir,
  CONST(Eth_BufIdxType, AUTOMATIC) LulBufIdx);

STATIC FUNC(void, ETH_PRIVATE_CODE) SetExtFempty(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONSTP2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA) LpDescr,               /* PRQA S 3432 # JV-01 */
  CONST(uint8, AUTOMATIC) LucQidx, CONST(Eth_DirectionType, AUTOMATIC) LenQdir,
  CONST(Eth_BufIdxType, AUTOMATIC) LulBufIdx);

STATIC FUNC(void, ETH_PRIVATE_CODE) SetLinkFix(
  CONST(uint32, AUTOMATIC) LulLinkDesc, CONST(uint32, AUTOMATIC) LulAddrToLink);

STATIC FUNC(void, ETH_PRIVATE_CODE) TxRxConfig(
  CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2CONST(Eth_TxConfigType, AUTOMATIC, ETH_APPL_DATA) LpTxConfig,
  CONSTP2CONST(Eth_RxConfigType, AUTOMATIC, ETH_APPL_DATA) LpRxConfig);

STATIC FUNC(uint32, ETH_PRIVATE_CODE) DescUpdate(
  CONST(uint32, AUTOMATIC) LulDescPtr);

STATIC FUNC(uint32, ETH_PRIVATE_CODE) DescTsUpdate(
  CONST(uint32, AUTOMATIC) LulDescPtr);

STATIC FUNC(void, ETH_PRIVATE_CODE) RxQueueSet(
  CONST(uint32, AUTOMATIC) LulCtrlIdx);

STATIC FUNC(uint32, ETH_PRIVATE_CODE) RxDescChainUpdate(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulDescPtr,
  CONSTP2CONST(Eth_RxChConfigType, AUTOMATIC, ETH_APPL_DATA) LpChConfig);

STATIC FUNC(void, ETH_PRIVATE_CODE) RxCallEthIf(
  CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2CONST(Eth_RxSingleFrameType, AUTOMATIC,ETH_APPL_DATA) LpFrame);

STATIC FUNC(void, ETH_PRIVATE_CODE) TxRxIntConfig(
  CONST(uint32, AUTOMATIC) LulCtrlIdx);

STATIC FUNC(uint32, ETH_PRIVATE_CODE) AllocMemForDesc(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint8, AUTOMATIC) LucQidx,
  CONST(Eth_DirectionType, AUTOMATIC) LenQdir);

STATIC FUNC(void, ETH_PRIVATE_CODE) FillDescMemory(
  CONSTP2VAR(Eth_LinkDescType, AUTOMATIC, ETH_APPL_DATA) LpLinkDescr, CONST(uint32, AUTOMATIC) LulDescAddr);            /* PRQA S 3432 # JV-01 */

STATIC FUNC(boolean, ETH_PRIVATE_CODE) IsQueueConfigured(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint8, AUTOMATIC) LucQidx,
  CONST(Eth_DirectionType, AUTOMATIC) LenDir);

#if (ETH_STREAM_FILTERING == STD_ON)
STATIC FUNC(void,  ETH_PRIVATE_CODE) SetRxFilter(
  CONST(uint32, AUTOMATIC) LulCtrlIdx);
#endif

#if (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON)
STATIC FUNC(boolean, ETH_PRIVATE_CODE) IsRxFrameValid(
  CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2CONST(Eth_RxSingleFrameType, AUTOMATIC, ETH_APPL_DATA) LpRxFrame);
#endif

#if (ETH_QOS_SUPPORT == STD_ON)
STATIC FUNC(void, ETH_PRIVATE_CODE) SetCbsParameter(
  CONST(uint32, AUTOMATIC) LulCtrlIdx);
#endif

/***********************************************************************************************************************
** Function Name         : RxBeProcess
**
** Service ID            : NA
**
** Description           : This process best effort queue
**                         (The PS bit isn't checked.)
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Ethernet channel number
**                       : LulRxFramePtr - pointer to rx frame
**                       : LulDescPtr - pointer to rx Descriptor
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_052,
** Reference ID          : ETH_DUD_ACT_052_GBL001, ETH_DUD_ACT_052_GBL002
** Reference ID          : ETH_DUD_ACT_052_GBL003
***********************************************************************************************************************/
STATIC FUNC(void, ETH_PRIVATE_CODE) RxBeProcess(
  CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2VAR(Eth_RxSingleFrameType, AUTOMATIC, ETH_APPL_DATA) LpSingleRxFramePtr,                                       /* PRQA S 3432 # JV-01 */
  CONST(uint32, AUTOMATIC) LulDescPtr)
{
  P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA) LpDataDesc;                                                         /* PRQA S 3432, 3678 # JV-01, JV-01 */
  P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA) LpExtDataDesc;                                                   /* PRQA S 3432, 3678 # JV-01, JV-01 */
  Eth_OptionType LenRxBeTimestamp;

  LenRxBeTimestamp = Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enRxBeTimestamp;                                              /* PRQA S 2844 # JV-01 */

  /* single frame single buffer */
  if (ETH_ENABLE == LenRxBeTimestamp)
  {
    /* timestamp enabled */
    LpExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA)) LulDescPtr;                                  /* PRQA S 0306, 3432 # JV-01, JV-01 */

    if (ETH_DESC_FSINGLE == LpExtDataDesc->stHeader.ulDt)                                                               /* PRQA S 2814 # JV-01 */
    {
      /* descriptor type correct */
      LpSingleRxFramePtr->ulFrameAddr   = LpExtDataDesc->ulDptr;                                                        /* PRQA S 2814 # JV-01 */
      LpSingleRxFramePtr->ulEthTypeAddr = LpExtDataDesc->ulDptr + ETH_SRC_DST_ADDRESS_SIZE;                             /* PRQA S 3383 # JV-01 */
      LpSingleRxFramePtr->ulFrameLength = LpExtDataDesc->stHeader.ulDs;
      LpSingleRxFramePtr->stTimestamp   = LpExtDataDesc->stTimestamp;
      LpSingleRxFramePtr->enTimeQual    = ETH_VALID;
    }
    else
    {
      /* descriptor type incorrect */
      LpSingleRxFramePtr->ulFrameAddr   = 0UL;
      LpSingleRxFramePtr->ulFrameLength = 0UL;
    }
    LpExtDataDesc++;                                                                                                    /* PRQA S 2824 # JV-01 */
    while ((LpExtDataDesc->stHeader.ulDt == ETH_DESC_LINKFIX) ||                                                        /* PRQA S 2814 # JV-01 */
      (LpExtDataDesc->stHeader.ulDt == ETH_DESC_LINK))
    {
      LpExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA)) LpExtDataDesc->ulDptr;                     /* PRQA S 0306, 3432, 2814 # JV-01, JV-01, JV-01 */

    }

    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaNextRxDesc[(uint32)ETH_BECHANNEL] = (uint32)LpExtDataDesc;                   /* PRQA S 2844, 0306 # JV-01, JV-01 */
  }
  else
  {
    /* timestamp disabled */
    LpDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA)) LulDescPtr;                                        /* PRQA S 0306, 3432 # JV-01, JV-01 */

    if (ETH_DESC_FSINGLE == LpDataDesc->stHeader.ulDt)                                                                  /* PRQA S 2814 # JV-01 */
    {
      /* descriptor type correct */
      LpSingleRxFramePtr->ulFrameAddr   = LpDataDesc->ulDptr;
      LpSingleRxFramePtr->ulEthTypeAddr = LpDataDesc->ulDptr + ETH_SRC_DST_ADDRESS_SIZE;                                /* PRQA S 3383 # JV-01 */
      LpSingleRxFramePtr->ulFrameLength = LpDataDesc->stHeader.ulDs;
      LpSingleRxFramePtr->enTimeQual    = ETH_INVALID;
    }
    else
    {
      /* descriptor type incorrect */
      LpSingleRxFramePtr->ulFrameAddr   = 0UL;
      LpSingleRxFramePtr->ulFrameLength = 0UL;
    }
    LpDataDesc++;                                                                                                       /* PRQA S 2824 # JV-01 */

    while ((LpDataDesc->stHeader.ulDt == ETH_DESC_LINKFIX) ||                                                           /* PRQA S 2814 # JV-01 */
      (LpDataDesc->stHeader.ulDt == ETH_DESC_LINK))
    {
      LpDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA)) LpDataDesc->ulDptr;                              /* PRQA S 0306, 3432, 2814 # JV-01, JV-01, JV-01 */
    }

    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaNextRxDesc[(uint32)ETH_BECHANNEL] = (uint32)LpDataDesc;                      /* PRQA S 2844, 0306 # JV-01, JV-01 */
  }

}

/***********************************************************************************************************************
** Function Name         : RxNcProcess
**
** Service ID            : NA
**
** Description           : This  process network control queue
**                         (The PS bit isn't checked.)
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Ethernet channel number
**                       : LulRxFramePtr - pointer to rx frame
**                       : LulDescPtr - pointer to rx Descriptor
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_053,
** Reference ID          : ETH_DUD_ACT_053_GBL001, ETH_DUD_ACT_053_GBL002
***********************************************************************************************************************/
STATIC FUNC(void, ETH_PRIVATE_CODE) RxNcProcess(
  CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2VAR(Eth_RxSingleFrameType, AUTOMATIC, ETH_APPL_DATA) LpSingleRxFramePtr,                                       /* PRQA S 3432 # JV-01 */
  CONST(uint32, AUTOMATIC) LulDescPtr)
{
  P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA) LpExtDataDesc;                                                   /* PRQA S 3432, 3678 # JV-01, JV-01 */

  LpExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA)) LulDescPtr;                                    /* PRQA S 0306, 3432 # JV-01, JV-01 */

  if (ETH_DESC_FSINGLE == LpExtDataDesc->stHeader.ulDt)                                                                 /* PRQA S 2814 # JV-01 */
  {
    /* descriptor type correct */
    LpSingleRxFramePtr->ulFrameAddr   = LpExtDataDesc->ulDptr;                                                          /* PRQA S 2814 # JV-01 */
    LpSingleRxFramePtr->ulEthTypeAddr = LpExtDataDesc->ulDptr + ETH_SRC_DST_ADDRESS_SIZE;                               /* PRQA S 3383 # JV-01 */
    LpSingleRxFramePtr->ulFrameLength = LpExtDataDesc->stHeader.ulDs;
    LpSingleRxFramePtr->stTimestamp   = LpExtDataDesc->stTimestamp;
    LpSingleRxFramePtr->enTimeQual    = ETH_VALID;
  }
  else
  {
    /* descriptor type incorrect */
    LpSingleRxFramePtr->ulFrameAddr   = 0UL;
    LpSingleRxFramePtr->ulFrameLength = 0UL;
  }

  LpExtDataDesc++;                                                                                                      /* PRQA S 2824 # JV-01 */
  while ((LpExtDataDesc->stHeader.ulDt == ETH_DESC_LINKFIX) ||                                                          /* PRQA S 2814 # JV-01 */
    (LpExtDataDesc->stHeader.ulDt == ETH_DESC_LINK))
  {
    LpExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA)) LpExtDataDesc->ulDptr;                       /* PRQA S 0306, 3432, 2814 # JV-01, JV-01, JV-01 */
  }

  Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaNextRxDesc[(uint32)ETH_NCCHANNEL] = (uint32)LpExtDataDesc;                     /* PRQA S 2844, 0306 # JV-01, JV-01 */
}

/***********************************************************************************************************************
** Function Name         : RxSProcess
**
** Service ID            : NA
**
** Description           : This  process stream queue
**                         (The PS bit isn't checked.)
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx       - Instance number
**                       : LulRxFramePtr - pointer to rx frame
**                       : LulChannelNum - stream channel number
**                       : LulDescPtr - pointer to rx Descriptor
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_054,
** Reference ID          : ETH_DUD_ACT_054_GBL001, ETH_DUD_ACT_054_GBL002
** Reference ID          : ETH_DUD_ACT_054_GBL003
***********************************************************************************************************************/
STATIC FUNC(void, ETH_PRIVATE_CODE) RxSProcess(
  CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2VAR(Eth_RxSingleFrameType, AUTOMATIC, ETH_APPL_DATA) LpSingleRxFramePtr,                                       /* PRQA S 3432 # JV-01 */
  CONST(uint32, AUTOMATIC) LulChannelNum, CONST(uint32, AUTOMATIC) LulDescPtr)
{
  P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA) LpDataDesc;                                                         /* PRQA S 3432, 3678 # JV-01, JV-01 */
  P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA) LpExtDataDesc;                                                   /* PRQA S 3432, 3678 # JV-01, JV-01 */
  Eth_OptionType LenRxSTimestamp;

  LenRxSTimestamp = Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enRxSTimestamp;                                                /* PRQA S 2844 # JV-01 */

  /* single frame single buffer */
  if (ETH_ENABLE == LenRxSTimestamp)
  {
    /* timestamp enabled */
    LpExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA)) LulDescPtr;                                  /* PRQA S 0306, 3432 # JV-01, JV-01 */

    if (ETH_DESC_FSINGLE == LpExtDataDesc->stHeader.ulDt)                                                               /* PRQA S 2814 # JV-01 */
    {
      /* descriptor type correct */
      LpSingleRxFramePtr->ulFrameAddr   = LpExtDataDesc->ulDptr;                                                        /* PRQA S 2814 # JV-01 */
      LpSingleRxFramePtr->ulEthTypeAddr = LpExtDataDesc->ulDptr + ETH_SRC_DST_ADDRESS_SIZE;                             /* PRQA S 3383 # JV-01 */
      LpSingleRxFramePtr->ulFrameLength = LpExtDataDesc->stHeader.ulDs;
      LpSingleRxFramePtr->stTimestamp   = LpExtDataDesc->stTimestamp;
      LpSingleRxFramePtr->enTimeQual    = ETH_VALID;
    }
    else
    {
      /* descriptor type incorrect */
      LpSingleRxFramePtr->ulFrameAddr   = 0UL;
      LpSingleRxFramePtr->ulFrameLength = 0UL;
    }
    LpExtDataDesc++;                                                                                                    /* PRQA S 2824 # JV-01 */
    while ((LpExtDataDesc->stHeader.ulDt == ETH_DESC_LINKFIX) ||                                                        /* PRQA S 2814 # JV-01 */
      (LpExtDataDesc->stHeader.ulDt == ETH_DESC_LINK))
    {
      LpExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA)) LpExtDataDesc->ulDptr;                     /* PRQA S 0306, 3432, 2814 # JV-01, JV-01, JV-01 */
    }

    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaNextRxDesc[LulChannelNum] = (uint32)LpExtDataDesc;                           /* PRQA S 2844, 0306 # JV-01, JV-01 */
  }
  else
  {
    /* timestamp disabled */
    LpDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA)) LulDescPtr;                                        /* PRQA S 0306, 3432 # JV-01, JV-01 */

    if (ETH_DESC_FSINGLE == LpDataDesc->stHeader.ulDt)                                                                  /* PRQA S 2814 # JV-01 */
    {
      /* descriptor type correct */
      LpSingleRxFramePtr->ulFrameAddr   = LpDataDesc->ulDptr;
      LpSingleRxFramePtr->ulEthTypeAddr = LpDataDesc->ulDptr + ETH_SRC_DST_ADDRESS_SIZE;                                /* PRQA S 3383 # JV-01 */
      LpSingleRxFramePtr->ulFrameLength = LpDataDesc->stHeader.ulDs;
      LpSingleRxFramePtr->enTimeQual    = ETH_INVALID;
    }
    else
    {
      /* descriptor type incorrect */
      LpSingleRxFramePtr->ulFrameAddr   = 0UL;
      LpSingleRxFramePtr->ulFrameLength = 0UL;
    }

    LpDataDesc++;                                                                                                       /* PRQA S 2824 # JV-01 */
    while ((LpDataDesc->stHeader.ulDt == ETH_DESC_LINKFIX) ||                                                           /* PRQA S 2814 # JV-01 */
      (LpDataDesc->stHeader.ulDt == ETH_DESC_LINK))
    {
      LpDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA)) LpDataDesc->ulDptr;                              /* PRQA S 0306, 3432, 2814 # JV-01, JV-01, JV-01 */
    }

    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaNextRxDesc[LulChannelNum] = (uint32)LpDataDesc;                              /* PRQA S 2844, 0306 # JV-01, JV-01 */
  }

}

/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_DMACStructConfig
**
** Service ID            : NA
**
** Description           : This Initializes and enables Ethernet peripheral.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Index of a controller
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaAvbConfig,
**                       : Eth_GaaRxConfig, Eth_GaaQConfig
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_055,
** Reference ID          : ETH_DUD_ACT_055_GBL001, ETH_DUD_ACT_055_GBL002
** Reference ID          : ETH_DUD_ACT_055_GBL003, ETH_DUD_ACT_055_GBL004
** Reference ID          : ETH_DUD_ACT_055_GBL005, ETH_DUD_ACT_055_GBL009
** Reference ID          : ETH_DUD_ACT_055_GBL010, ETH_DUD_ACT_055_GBL011
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE)Eth_Hw_Etnb_DMACStructConfig(                                                               /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  uint32 LulIdx;
  uint8 LucQidx;
  P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;

  LpHwUnitConfig = (P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA))                                              /* PRQA S 0316 # JV-01 */
    Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;
  /* AVB part */
  Eth_GaaAvbConfig[LulCtrlIdx].stRxConfig.enEncf = LpHwUnitConfig->stRxConfig.enEncf;                                   /* PRQA S 2844, 2814 # JV-01, JV-01 */
  Eth_GaaAvbConfig[LulCtrlIdx].stRxConfig.enEsf  = LpHwUnitConfig->stRxConfig.enEsf;                                    /* PRQA S 2844 # JV-01 */
  Eth_GaaAvbConfig[LulCtrlIdx].stRxConfig.enEts0 = LpHwUnitConfig->stRxConfig.enEts0;                                   /* PRQA S 2844 # JV-01 */
  Eth_GaaAvbConfig[LulCtrlIdx].stRxConfig.enEts2 = LpHwUnitConfig->stRxConfig.enEts2;                                   /* PRQA S 2844 # JV-01 */
  Eth_GaaAvbConfig[LulCtrlIdx].stRxConfig.ulRfcl = ETH_RX_FIFO_CRIT_LVL;                                                /* PRQA S 2844 # JV-01 */
  #if (ETH_STREAM_FILTERING == STD_ON)
  Eth_GaaAvbConfig[LulCtrlIdx].stRxConfig.enSRPTalkerFiltering = LpHwUnitConfig->stRxConfig.enSRPTalkerFiltering;       /* PRQA S 2844 # JV-01 */
  #endif
  Eth_GaaAvbConfig[LulCtrlIdx].stTxConfig.enTsm0 = ETH_TXNORMAL;                                                        /* PRQA S 2844 # JV-01 */
  Eth_GaaAvbConfig[LulCtrlIdx].stTxConfig.enTsm1 = ETH_TXNORMAL;                                                        /* PRQA S 2844 # JV-01 */
  Eth_GaaAvbConfig[LulCtrlIdx].stTxConfig.enTsm2 = ETH_TXNORMAL;                                                        /* PRQA S 2844 # JV-01 */
  Eth_GaaAvbConfig[LulCtrlIdx].stTxConfig.enTsm3 = ETH_TXNORMAL;                                                        /* PRQA S 2844 # JV-01 */
  Eth_GaaAvbConfig[LulCtrlIdx].stTxConfig.enEcbs = LpHwUnitConfig->enEcbs;                                              /* PRQA S 2844 # JV-01 */
  Eth_GaaAvbConfig[LulCtrlIdx].stTxConfig.enTqp = LpHwUnitConfig->enTxConfig;                                           /* PRQA S 2844 # JV-01 */


  /* Read the number of Rx Queue Configured */
  for (LulIdx = 0UL; LulIdx < (uint32)LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue; LulIdx++)
  {
    LucQidx = LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulIdx].ucEthRxQueueId;                                      /* PRQA S 2824 # JV-01 */

    Eth_GaaRxConfig[LucQidx].ucChNum    = LucQidx;                                                                      /* PRQA S 2844 # JV-01 */

    if (ETH_BECHANNEL == Eth_GaaRxConfig[LucQidx].ucChNum)
    {
      /* This is Rx BE */
      Eth_GaaRxConfig[LucQidx].enChType = ETH_RX_BE;                                                                    /* PRQA S 2844 # JV-01 */
    }
    else if (ETH_NCCHANNEL == Eth_GaaRxConfig[LucQidx].ucChNum)
    {
      /* This is Rx NC */
      Eth_GaaRxConfig[LucQidx].enChType = ETH_RX_NC;                                                                    /* PRQA S 2844 # JV-01 */
    }
    else
    {
      /* Rx Stream */
      Eth_GaaRxConfig[LucQidx].enChType = ETH_RX_S;                                                                     /* PRQA S 2844 # JV-01 */
    }

    /* Queue Configuration */
    Eth_GaaQConfig[LulIdx].enPia = ETH_GAP32;                                                                           /* PRQA S 2844 # JV-01 */
    Eth_GaaQConfig[LulIdx].enUfcc = ETH_UFCC0;                                                                          /* PRQA S 2844 # JV-01 */
    Eth_GaaQConfig[LulIdx].enRsm = ETH_RXNORMAL;                                                                        /* PRQA S 2844 # JV-01 */

    #if (ETH_STREAM_FILTERING == STD_ON)
    /* Save Pattern Address if Filtering is enabled */
    ETH_COPY_STREAM_ADDRESS(LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulIdx].aaEthPatternStream,                    /* PRQA S 3469, 2844 # JV-01, JV-01 */
      Eth_GaaRxConfig[LucQidx].aaPatternStream);
    #endif
  }
}

/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_SingleDescFrameSend
**
** Service ID            : NA
**
** Description           : send frame (single frame single buffer)
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Instance number
**                       : LenQIndex - Queue Index
**                       : LpFrame - Tx Frame Pointer
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : LucReturnValue - E_OK / E_NOT_OK
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat, Eth_GaaETNBRegs
**
** Function(s) invoked   : ETH_ENTER_CRITICAL_SECTION,
**                         ETH_EXIT_CRITICAL_SECTION,
**                         Eth_CheckProvideBuffer
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_056,
** Reference ID          : ETH_DUD_ACT_056_CRT001, ETH_DUD_ACT_056_CRT002
** Reference ID          : ETH_DUD_ACT_056_GBL001, ETH_DUD_ACT_056_REG001
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_Hw_Etnb_SingleDescFrameSend(                                                 /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2VAR(Eth_BufHandlerType, AUTOMATIC, ETH_APPL_DATA) LpTxBufHdr)                                                  /* PRQA S 3432, 3673 # JV-01, JV-01 */

{
  uint32 LulBufIdx;
  Std_ReturnType LucTxBufferProvide;
  P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA) LpDataDesc;                                                         /* PRQA S 3432 # JV-01 */
  Eth_TxRxCtrl  LunTxCtrl;                                                                                              /* PRQA S 0759 # JV-01 */
  Std_ReturnType LucReturnValue;

  LucTxBufferProvide = E_NOT_OK;
  LucReturnValue = E_OK;

  /* Get next free descriptor */
  LpDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA))                                                      /* PRQA S 0306, 3432 # JV-01, JV-01 */
    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaLastTxDesc[LpTxBufHdr->enChannel];                                           /* PRQA S 2844, 2814 # JV-01, JV-01 */

  if (0UL != LpDataDesc->ulDptr)                                                                                        /* PRQA S 2814 # JV-01 */
  {
    /* Check previous TxBuffer release status (The memory leak protect) */
    LulBufIdx = (uint32)((LpDataDesc->stHeader.ulCtrl) & ETH_DESCR_TAG_MASK);
    LucTxBufferProvide = Eth_CheckProvideBuffer(LulCtrlIdx, LulBufIdx);
  }
  else
  {
    /* No action required */
  }

  if ((ETH_DESC_FEMPTY == LpDataDesc->stHeader.ulDt) && (E_NOT_OK == LucTxBufferProvide))
  {
    LunTxCtrl.ulWord = 0UL;
    if (ETH_TRUE == LpTxBufHdr->blbenableTS)
    {
      LunTxCtrl.stTxCtrl.ulTsr = ETH_DESC_RETAIN_TS;
    }
    else
    {
      /* No action required */
    }
    /* Justify typecast to smaller datatype */
    LunTxCtrl.stTxCtrl.ulTag = LpTxBufHdr->ulbufIdx;
    /* build the descriptor */
    LpDataDesc->stHeader.ulDie  = (uint32)LpTxBufHdr->enChannel + 1UL;                                                  /* PRQA S 3383 # JV-01 */
    LpDataDesc->stHeader.ulCtrl = LunTxCtrl.ulWord;
    LpDataDesc->stHeader.ulDs   = LpTxBufHdr->ulTxLength;
    LpDataDesc->ulDptr          = LpTxBufHdr->ulbufAddr;
    LpDataDesc->stHeader.ulDt   = ETH_DESC_FSINGLE;

    /* Set next descriptor */
    LpDataDesc++;                                                                                                       /* PRQA S 2824 # JV-01 */
    while ((LpDataDesc->stHeader.ulDt == ETH_DESC_LINKFIX) || (LpDataDesc->stHeader.ulDt == ETH_DESC_LINK))             /* PRQA S 2814 # JV-01 */
    {
      LpDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA)) LpDataDesc->ulDptr;                              /* PRQA S 0306, 3432, 2814 # JV-01, JV-01, JV-01 */
    }

    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaLastTxDesc[LpTxBufHdr->enChannel] = (uint32)LpDataDesc;                      /* PRQA S 2844, 0306 # JV-01, JV-01 */

    ETH_ENTER_CRITICAL_SECTION(ETH_RAM_DATA_PROTECTION);

    /* Increase the number of buffer of current Tx queue */
    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaBufTxCnt[LpTxBufHdr->enChannel]++;                                           /* PRQA S 2844, 3383 # JV-01, JV-01 */

    ETH_EXIT_CRITICAL_SECTION(ETH_RAM_DATA_PROTECTION);

    /* Transmit start request */
    Eth_GaaETNBRegs[LulCtrlIdx]->ulTCCR |= (uint32)(1UL << (uint32)LpTxBufHdr->enChannel);                              /* PRQA S 2844, 2814 # JV-01, JV-01 */
  }
  else
  {
    LucReturnValue = E_NOT_OK;
  }

  return LucReturnValue;
}

/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_RxQueueProcess
**
** Service ID            : NA
**
** Description           : Process Receive Queue.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx  - Instance number
**                       : LucQidx - Rx queue index
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat, Eth_GaaRxConfig,
**                         Eth_GaaETNBRegs, Eth_GaaRxFrame
**
** Function(s) invoked   : RxBeProcess, RxNcProcess,
**                       : RxSProcess, RxDescChainUpdate,
**                       : IsRxFrameValid, RxCallEthIf
**
** Registers Used        : UFCD
**
** Reference ID          : ETH_DUD_ACT_057,
** Reference ID          : ETH_DUD_ACT_057_REG001
** Reference ID          : ETH_DUD_ACT_057_GBL001, ETH_DUD_ACT_057_GBL002
** Reference ID          : ETH_DUD_ACT_057_GBL003, ETH_DUD_ACT_057_GBL004
** Reference ID          : ETH_DUD_ACT_057_GBL005, ETH_DUD_ACT_057_GBL006
** Reference ID          : ETH_DUD_ACT_057_GBL007
***********************************************************************************************************************/
FUNC(boolean, ETH_PRIVATE_CODE) Eth_Hw_Etnb_RxQueueProcess(                                                             /* PRQA S 1505, 1532 # JV-01, JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint8, AUTOMATIC) LucQidx)
{
  uint32 LulDescAddr;
  boolean LblRxFrameValid;
  P2CONST(Eth_RxChConfigType, AUTOMATIC, ETH_APPL_DATA) LpChConfig;

  #if (ETH_GET_RX_STATS_API == STD_ON)
  uint32 LulLengthWithFCS;
  #endif

  #if (ETH_ETHSWITCH_MANAGEMENT_SUPPORT == STD_ON)
  Eth_BufIdxType LulBufIdx;
  P2VAR(uint8, AUTOMATIC, ETH_APPL_DATA) LpDataPtr;                                                                     /* PRQA S 3432 # JV-01 */
  boolean LblIsMgmtFrameOnlyPtr;
  uint16 LusLength;
  Std_ReturnType LucReturnValue;
  #endif

  LblRxFrameValid = ETH_TRUE;
  /*Get Rx queue configuration */
  LpChConfig = &Eth_GaaRxConfig[LucQidx];                                                                               /* PRQA S 2934 # JV-01 */

  LulDescAddr = Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaNextRxDesc[LpChConfig->ucChNum];                                 /* PRQA S 2844 # JV-01 */


  if (ETH_BECHANNEL == LpChConfig->ucChNum)
  {
    /* best effort channel */
    RxBeProcess(LulCtrlIdx, &Eth_GaaRxFrame[LulCtrlIdx], LulDescAddr);                                                  /* PRQA S 2934 # JV-01 */

    #if (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON)
    /* check whether received frame is valid or not */
    LblRxFrameValid = IsRxFrameValid(LulCtrlIdx, &Eth_GaaRxFrame[LulCtrlIdx]);                                          /* PRQA S 2934 # JV-01 */
    #endif
  }
  else if (ETH_NCCHANNEL == LpChConfig->ucChNum)
  {
    /* network control channel */
    RxNcProcess(LulCtrlIdx, &Eth_GaaRxFrame[LulCtrlIdx], LulDescAddr);
  }
  else
  {
    /* stream channel */
    RxSProcess(LulCtrlIdx, &Eth_GaaRxFrame[LulCtrlIdx], (uint32)LpChConfig->ucChNum, LulDescAddr);

    #if (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON)
    /* check whether received frame is valid or not */
    LblRxFrameValid = IsRxFrameValid(LulCtrlIdx, &Eth_GaaRxFrame[LulCtrlIdx]);                                          /* PRQA S 2934 # JV-01 */
    #endif
  }
  /* Decrement ETNBnUFCD.DVr by 1 */
  Eth_GaaETNBRegs[LulCtrlIdx]->ulUFCD[ETH_ETNB_GET_UFCDi(LucQidx)] = ETH_ETNB_SET_UFCDi_DVr(1UL, LucQidx);              /* PRQA S 2844, 2814, 3469, 3384 # JV-01, JV-01, JV-01, JV-01 */

  if (ETH_TRUE == LblRxFrameValid)
  {
    #if (ETH_GET_RX_STATS_API == STD_ON)
    Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts++;                                                             /* PRQA S 2844, 3383 # JV-01, JV-01 */
    LulLengthWithFCS = Eth_GaaRxFrame[LulCtrlIdx].ulFrameLength + ETH_FCS_LENGTH;                                       /* PRQA S 2844, 3383 # JV-01, JV-01 */
    Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsOctets += LulLengthWithFCS;                                         /* PRQA S 2844, 3383 # JV-01, JV-01 */
    if (LulLengthWithFCS <= 64UL)
    {
      /* Since the receive data size that does not include padding data is set in the receive descriptor,
         frames of 64 or less are collected by the statistical counter. */
      Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts64Octets++;                                                   /* PRQA S 2844, 3383 # JV-01, JV-01 */
    }
    else if (LulLengthWithFCS <= 127UL)
    {
      Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts65to127Octets++;                                              /* PRQA S 3383 # JV-01 */
    }
    else if (LulLengthWithFCS <= 255UL)
    {
      Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts128to255Octets++;                                             /* PRQA S 3383 # JV-01 */
    }
    else if (LulLengthWithFCS <= 511UL)
    {
      Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts256to511Octets++;                                             /* PRQA S 3383 # JV-01 */
    }
    else if (LulLengthWithFCS <= 1023UL)
    {
      Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts512to1023Octets++;                                            /* PRQA S 3383 # JV-01 */
    }
    else
    {
      /* When VLAN tag is supported, the maximum data size is 1522. */
      /* The maximum frame size that HW can receive is set to 1522. */
      Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxStatsPkts1024to1518Octets++;                                           /* PRQA S 3383 # JV-01 */
    }
    #endif

    #if (ETH_ETHSWITCH_MANAGEMENT_SUPPORT == STD_ON)
    LulBufIdx = *(P2VAR(Eth_BufIdxType, AUTOMATIC, ETH_APPL_DATA))                                                      /* PRQA S 0306, 2814, 3432 # JV-01, JV-01, JV-01 */
      (Eth_GaaRxFrame[LulCtrlIdx].ulFrameAddr - sizeof(Eth_BufIdxType) - ETH_RX_DPTR_OFFSET);                           /* PRQA S 3383, 3384 # JV-01, JV-01 */

    LpDataPtr = (P2VAR(uint8, AUTOMATIC, ETH_APPL_DATA))Eth_GaaRxFrame[LulCtrlIdx].ulEthTypeAddr;                       /* PRQA S 0306, 3432 # JV-01, JV-01 */
    /* Since the maximum value of buffer size is 1518, casting to uint16 does no problem. */
    LusLength = (uint16)(Eth_GaaRxFrame[LulCtrlIdx].ulFrameLength - ETH_HEADER_SIZE);                                   /* PRQA S 3383 # JV-01 */
    LblIsMgmtFrameOnlyPtr = ETH_FALSE;
    /* Since the maximum value of controller index is 1, casting to uint8 does no problem. */
    LucReturnValue = EthSwt_EthRxProcessFrame((uint8)LulCtrlIdx, LulBufIdx, &LpDataPtr,
      &LusLength, &LblIsMgmtFrameOnlyPtr);
    if (E_OK == LucReturnValue)
    {
      Eth_GaaRxFrame[LulCtrlIdx].ulEthTypeAddr = (uint32)LpDataPtr;                                                     /* PRQA S 0306, 2844 # JV-01, JV-01 */
      Eth_GaaRxFrame[LulCtrlIdx].ulFrameLength = (uint32)(LusLength + ETH_HEADER_SIZE);                                 /* PRQA S 3383, 2844 # JV-01, JV-01 */

      if (ETH_FALSE == LblIsMgmtFrameOnlyPtr)
      {
        /* Call EthIf if the Frame Received is valid */
        RxCallEthIf(LulCtrlIdx, &Eth_GaaRxFrame[LulCtrlIdx]);                                                           /* PRQA S 2934 # JV-01 */
      }
      else
      {
        /* Must not be the Rx processed */
        /* Since the maximum value of controller index is 1, casting to uint8 does no problem. */
        (void)EthSwt_EthRxFinishedIndication((uint8)LulCtrlIdx, LulBufIdx);
      }
    }
    else
    {
      /* Normal operation if E_NOT_OK */
      RxCallEthIf(LulCtrlIdx, &Eth_GaaRxFrame[LulCtrlIdx]);                                                             /* PRQA S 2934 # JV-01 */
    }
    #else
    /* Call EthIf if the Frame Received is valid */
    RxCallEthIf(LulCtrlIdx, &Eth_GaaRxFrame[LulCtrlIdx]);                                                               /* PRQA S 2934 # JV-01 */
    #endif
  }
  else
  {
    /* Frame Invalid - E.g. Multicast to be discarded */
    /* No Call of EthIf */
  }

  /* update descriptor chain */
  (void)RxDescChainUpdate(LulCtrlIdx, LulDescAddr, LpChConfig);

  return LblRxFrameValid;
}

/***********************************************************************************************************************
** Function Name         : RxCallEthIf
**
** Service ID            : NA
**
** Description           : Wrapper for the Callback to the Eth Interface for each frame received
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Controller / Channel Index
**                       : LpFrame - Address of the Received Frame in URAM
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : NA
**
** Global Variables Used : Eth_GstBroadcastAddr, Eth_GaaCtrlStat
**
** Function(s) invoked   : EthIf_RxIndication, EthSwt_EthRxFinishedIndication,
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_058,
** Reference ID          : ETH_DUD_ACT_058_GBL002, ETH_DUD_ACT_058_GBL009
***********************************************************************************************************************/
STATIC FUNC (void, ETH_PRIVATE_CODE) RxCallEthIf(
  CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2CONST(Eth_RxSingleFrameType, AUTOMATIC, ETH_APPL_DATA) LpFrame)
{
  boolean LblBroadcast;
  P2CONST(Eth_DataType, AUTOMATIC, ETH_APPL_DATA) LpFrameType;
  Eth_FrameType LddFrameType;
  Eth_MacAddressType LstMacAddr;
  #if (ETH_ETHSWITCH_MANAGEMENT_SUPPORT == STD_ON)
  Eth_BufIdxType LulBufIdx;
  #endif
  P2CONST(Eth_EtherFrameType, AUTOMATIC, ETH_APPL_DATA) LstEtherFrame;

  LstEtherFrame = (P2CONST(Eth_EtherFrameType, AUTOMATIC, ETH_APPL_DATA)) LpFrame->ulFrameAddr;                         /* PRQA S 0306, 2814 # JV-01, JV-01 */
  ETH_PACK_ADDRESS_FROM_8(LstEtherFrame->ucDstAddr, LstMacAddr);                                                        /* PRQA S 2814, 3469 # JV-01, JV-01 */

  if (0UL == ETH_COMPARE_MAC(LstMacAddr, Eth_GstBroadcastAddr))                                                         /* PRQA S 3469 # JV-01 */
  {
    LblBroadcast = ETH_TRUE;
    #if (ETH_GET_RX_STATS_API == STD_ON)
    Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulStatsBroadcastPkts++;                                                      /* PRQA S 2844, 3383 # JV-01, JV-01 */
    #endif
  }
  else
  {
    LblBroadcast = ETH_FALSE;
    #if (ETH_GET_RX_STATS_API == STD_ON)
    if (0UL == ETH_CHECK_MULTICAST(LstMacAddr))                                                                         /* PRQA S 3469 # JV-01 */
    {
      /* Unicast frame */
      Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulRxUnicastFrames++;                                                       /* PRQA S 3383 # JV-01 */
    }
    else
    {
      /* No action required */
    }
    #endif
  }

  LpFrameType = (const Eth_DataType *)LpFrame->ulEthTypeAddr;                                                           /* PRQA S 0306 # JV-01 */
  LddFrameType = (Eth_FrameType)((uint32)LpFrameType[0] << ETH_BYTE_BITS);                                              /* PRQA S 2824 # JV-01 */
  LddFrameType |= (Eth_FrameType)LpFrameType[1];

  /* Since the maximum value of Controller Index is 1, casting to uint8 does no problem. */
  EthIf_RxIndication((uint8)LulCtrlIdx, LddFrameType, LblBroadcast, LstEtherFrame->ucSrcAddr,
    LpFrameType + ETH_ETHERTYPE_SIZE, (uint16)(LpFrame->ulFrameLength - ETH_HEADER_SIZE));                              /* PRQA S 0488, 3383 # JV-01, JV-01 */

  #if (ETH_ETHSWITCH_MANAGEMENT_SUPPORT == STD_ON)
  LulBufIdx = *(P2VAR(Eth_BufIdxType, AUTOMATIC, ETH_APPL_DATA))                                                        /* PRQA S 2814, 0306, 3432 # JV-01, JV-01, JV-01 */
    (LpFrame->ulFrameAddr - sizeof(Eth_BufIdxType) - ETH_RX_DPTR_OFFSET);                                               /* PRQA S 3383, 3384 # JV-01, JV-01 */
  (void)EthSwt_EthRxFinishedIndication((uint8)LulCtrlIdx, LulBufIdx);
  #endif
}

/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_RxMemAlloc
**
** Service ID            : NA
**
** Description           : Allocate buffer memory for Rx queues.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Instance number
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpCtrlConfigPtr, Eth_GaaCtrlStat,
**                         Eth_GaaRxBufferIndex
**
** Function(s) invoked   : Eth_Ram_Alloc,
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_059,
** Reference ID          : ETH_DUD_ACT_059_GBL001, ETH_DUD_ACT_059_GBL002
** Reference ID          : ETH_DUD_ACT_059_GBL003, ETH_DUD_ACT_059_GBL004
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_Hw_Etnb_RxMemAlloc(                                                          /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  uint8 LucInc;
  uint8 LucQid;
  uint16 LusBufIdx;
  uint32 LulBufAddress;
  P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;
  Std_ReturnType LucReturnValue;
  uint32 LulRxDptrOffset;
  LucReturnValue = E_OK;
  LulRxDptrOffset = (uint32)sizeof(Eth_BufIdxType) + ETH_RX_DPTR_OFFSET;
  LpHwUnitConfig = (P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA))                                              /* PRQA S 0316 # JV-01 */
    Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;
  for (LucInc = 0U; LucInc < LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue; LucInc++)                                 /* PRQA S 2814 # JV-01 */
  {
    LucQid = LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LucInc].ucEthRxQueueId;                                       /* PRQA S 2824 # JV-01 */
    LusBufIdx = 0U;
    do
    {
      LulBufAddress = Eth_Ram_Alloc(LulCtrlIdx,
        (uint32)ETH_RX_BUF_LENGTH + (uint32)sizeof(Eth_BufIdxType) + ETH_RX_DPTR_OFFSET);
      if (0UL == LulBufAddress)
      {
        /* Allocation memory error due to lack of resources */
        LucReturnValue = E_NOT_OK;
        LucInc = LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue;                                                       /* PRQA S 2469 # JV-01 */
        break;
      }
      else
      {
        /* Add receive buffer index */
        *(P2VAR(Eth_BufIdxType, AUTOMATIC, ETH_APPL_DATA)) LulBufAddress = Eth_GaaRxBufferIndex[LulCtrlIdx];            /* PRQA S 0306, 2814, 3432, 2844 # JV-01, JV-01, JV-01, JV-01 */
        Eth_GaaRxBufferIndex[LulCtrlIdx]++;                                                                             /* PRQA S 3383 # JV-01 */
      }

      switch (LucQid)
      {
      case ETH_BECHANNEL:
        /* Best Effort Queue */
        Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaRxBeQueueBuf[LusBufIdx] = LulBufAddress + LulRxDptrOffset;               /* PRQA S 2844, 3383 # JV-01, JV-01 */
        break;

      case ETH_NCCHANNEL:
        /* Network Control Queue */
        Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaRxNcQueueBuf[LusBufIdx] = LulBufAddress + LulRxDptrOffset;               /* PRQA S 2844, 3383 # JV-01, JV-01 */
        break;

      default :
        /* Stream Queue */
        Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaRxSQueueBuf[LucQid][LusBufIdx] = LulBufAddress + LulRxDptrOffset;        /* PRQA S 2844, 3383 # JV-01, JV-01 */
        break;
      }
      LusBufIdx++;                                                                                                      /* PRQA S 3383 # JV-01 */
    } while (LusBufIdx < LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LucInc].usEthRxQueueBufs);
  }

  return LucReturnValue;
}

/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_DescConfig
**
** Service ID            : NA
**
** Description           : Configure descriptor.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Controlled Id
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat, Eth_GpCtrlConfigPtr
**
** Function(s) invoked   : SetDescChain
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_060
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_Hw_Etnb_DescConfig(                                                                    /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  uint8 LucQid; /* Index for the Queue */
  uint8 LucCnt;
  P2VAR(Eth_LinkDescType, AUTOMATIC, ETH_APPL_DATA) LpLinkAddr;                                                         /* PRQA S 3432, 3678 # JV-01, JV-01 */
  P2VAR(Eth_LinkDescType, AUTOMATIC, ETH_APPL_DATA) LpTargetAddr;                                                       /* PRQA S 3432, 3678 # JV-01, JV-01 */
  P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;

  LpHwUnitConfig = (P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA))                                              /* PRQA S 0316 # JV-01 */
    Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;
  /* init link descriptor */
  LpLinkAddr = (P2VAR(Eth_LinkDescType, AUTOMATIC, ETH_APPL_DATA))                                                      /* PRQA S 0306, 3432 # JV-01, JV-01 */
    (Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.ulDescTableAddr);                                                             /* PRQA S 2844 # JV-01 */

  /*======TX Queues ========*/
  for (LucCnt = 0U; LucCnt < LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue; LucCnt++)                                 /* PRQA S 2814 # JV-01 */
  {
    /* Calculate target descriptor table address */
    LpTargetAddr = LpLinkAddr + LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LucCnt].ucEthTxQueueId;                    /* PRQA S 2824, 0488 # JV-01, JV-01 */
    if (0UL != LpTargetAddr->ulDptr)                                                                                    /* PRQA S 2814 # JV-01 */
    {
      /* Get Tx LucQid */
      LucQid = LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LucCnt].ucEthTxQueueId;
      SetDescChain(LulCtrlIdx, (LpTargetAddr->ulDptr), LucQid, ETH_TX);
    }
    else
    {
      /* This Queue is not configured */
    }
  }

  /* Set Rx Queues start position */
  LpLinkAddr = LpLinkAddr + ETH_RXBEQ_OFFSET;                                                                           /* PRQA S 0488 # JV-01 */

  /*======RX Queues ========*/
  for (LucCnt = 0U; LucCnt < LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue; LucCnt++)
  {
    /* Calculate target descriptor table address */
    LpTargetAddr = LpLinkAddr + LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LucCnt].ucEthRxQueueId;                    /* PRQA S 0488, 2824 # JV-01, JV-01 */
    if (0UL != LpTargetAddr->ulDptr)                                                                                    /* PRQA S 2814 # JV-01 */
    {
      LucQid = LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LucCnt].ucEthRxQueueId;
      SetDescChain(LulCtrlIdx, (LpTargetAddr->ulDptr), LucQid, ETH_RX);
    }
    else
    {
      /* This Queue is not configured */
    }
  }
}

/***********************************************************************************************************************
** Function Name         : SetDescChain
**
** Service ID            : NA
**
** Description           : Configured the Descriptor chain for each Queue (Circular Chain).
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Controlled Id
**                       : LulAddr - Addr of the first descriptor in Queue
**                       : LucQidx - Index of the Queue (0..3 for Tx 0..17 for Rx)
**                       : LenDir - Queue Direction (e.g. TX or RX)
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat, Eth_GpCtrlConfigPtr
**
** Function(s) invoked   : SetFempty, SetLinkFix, SetExtFempty,
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_061
***********************************************************************************************************************/
STATIC FUNC(void, ETH_PRIVATE_CODE) SetDescChain(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulAddr,
  CONST(uint8, AUTOMATIC) LucQidx, CONST(Eth_DirectionType, AUTOMATIC) LenDir)
{
  uint32 LulIdx;
  uint32 LulCfgIdx;
  Eth_OptionType LenRxTimestamp;
  P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA) LpCurrentDataDesc;                                                  /* PRQA S 3432 # JV-01 */
  P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA) LpCurrentExtDataDesc;                                            /* PRQA S 3432 # JV-01 */
  P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;

  LpHwUnitConfig = (P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA))                                              /* PRQA S 0316 # JV-01 */
    Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;
  /* Initialize the counter for any Queue */
  LulIdx = 0UL;

  if (ETH_TX == LenDir)
  {
    /* Search for Tx configuration index */
    for (LulCfgIdx = 0; LulCfgIdx < LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue; LulCfgIdx++)                       /* PRQA S 2814 # JV-01 */
    {
      if (LucQidx == LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulCfgIdx].ucEthTxQueueId)                            /* PRQA S 2824 # JV-01 */
      {
        break;
      }
      else
      {
        /* No action required */
      }
    }
  }
  else
  {
    /* Search for Rx configuration index */
    for (LulCfgIdx = 0; LulCfgIdx < LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue; LulCfgIdx++)
    {
      if (LucQidx == LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].ucEthRxQueueId)                            /* PRQA S 2824 # JV-01 */
      {
        break;
      }
      else
      {
        /* No action required */
      }
    }
  }

  if (0UL != LulAddr)
  {
    if (ETH_TX == LenDir)
    {
      /* TX */
      LpCurrentDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA)) (LulAddr);                                /* PRQA S 0306, 3432 # JV-01, JV-01 */
      do
      {
        if ((uint32)(LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulCfgIdx].usEthTxQueueBufs -                         /* PRQA S 4391, 3383 # JV-01, JV-01 */
          (uint16)ETH_CYCLIC_DESC_NUM) >= LulIdx)
        {
          /* Set FEmpty */
          SetFempty(LulCtrlIdx, LpCurrentDataDesc, LucQidx, ETH_TX, LulIdx);
        }
        else
        {
          /* Set LinkFix */
          SetLinkFix((uint32)LpCurrentDataDesc, LulAddr);                                                               /* PRQA S 0306 # JV-01 */
        }

        LpCurrentDataDesc++;                                                                                            /* PRQA S 2824 # JV-01 */
        LulIdx++;                                                                                                       /* PRQA S 3383 # JV-01 */
      }
      while (LulIdx <= (uint32)LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulCfgIdx].usEthTxQueueBufs);
    }
    else if ((ETH_BECHANNEL == LucQidx) && (ETH_RX == LenDir))
    {
      LenRxTimestamp = Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enRxBeTimestamp;                                            /* PRQA S 2844 # JV-01 */
      /* RX BE */
      if (ETH_DISABLE == LenRxTimestamp)
      {
        /* No TS for RX BE */
        LpCurrentDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA)) (LulAddr);                              /* PRQA S 0306, 3432 # JV-01, JV-01 */
        do
        {
          if ((uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs > LulIdx)
          {
            SetFempty(LulCtrlIdx, LpCurrentDataDesc, LucQidx, ETH_RX, LulIdx);
          }
          else
          {
            SetLinkFix((uint32)LpCurrentDataDesc,LulAddr);                                                              /* PRQA S 0306 # JV-01 */
          }
          LpCurrentDataDesc++;                                                                                          /* PRQA S 2824 # JV-01 */
          LulIdx++;                                                                                                     /* PRQA S 3383 # JV-01 */
        }
        while (LulIdx <= (uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs);
        /* Better to repeat this code for any Queue */
        /* as in future we may have different No of buffers */
      }
      else
      {
        /* TS enabled for RX BE */
        LpCurrentExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA)) (LulAddr);                        /* PRQA S 0306, 3432 # JV-01, JV-01 */

        do
        {
          if ((uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs > LulIdx)
          {
            /* Extended Data Descriptor */
            SetExtFempty(LulCtrlIdx, LpCurrentExtDataDesc, LucQidx, ETH_RX, LulIdx);
          }
          else
          {
            SetLinkFix((uint32)LpCurrentExtDataDesc, LulAddr);                                                          /* PRQA S 0306 # JV-01 */
          }

          LulIdx++;                                                                                                     /* PRQA S 3383 # JV-01 */
          LpCurrentExtDataDesc++;                                                                                       /* PRQA S 2824 # JV-01 */
        }
        while (LulIdx <= (uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs);
      }
    }
    else if ((ETH_NCCHANNEL == LucQidx) && (ETH_RX == LenDir))
    {
      /* RX NC */
      /* Only Extended Descriptor for a RX Network Queue */

      LpCurrentExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA)) (LulAddr);                          /* PRQA S 0306, 3432 # JV-01, JV-01 */

      do
      {
        if ((uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs > LulIdx)
        {
          /* Extended Data Descriptor */
          SetExtFempty(LulCtrlIdx, LpCurrentExtDataDesc, LucQidx, ETH_RX, LulIdx);
        }
        else
        {
          SetLinkFix((uint32)LpCurrentExtDataDesc, LulAddr);                                                            /* PRQA S 0306 # JV-01 */
        }

        LulIdx++;                                                                                                       /* PRQA S 3383 # JV-01 */
        LpCurrentExtDataDesc++;                                                                                         /* PRQA S 2824 # JV-01 */
      }
      while (LulIdx <= (uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs);
    }
    else
    {
      LenRxTimestamp = Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enRxSTimestamp;
      /* RX S */
      if (ETH_DISABLE == LenRxTimestamp)
      {
        /* No TS for RX S */
        LpCurrentDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA)) (LulAddr);                              /* PRQA S 0306, 3432 # JV-01, JV-01 */
        do
        {
          if ((uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs > LulIdx)
          {
            SetFempty(LulCtrlIdx, LpCurrentDataDesc, LucQidx, ETH_RX, LulIdx);
          }
          else
          {
            SetLinkFix((uint32)LpCurrentDataDesc, LulAddr);                                                             /* PRQA S 0306 # JV-01 */
          }
          LpCurrentDataDesc++;                                                                                          /* PRQA S 2824 # JV-01 */

          LulIdx++;                                                                                                     /* PRQA S 3383 # JV-01 */
        }
        while (LulIdx <= (uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs);
        /* Better to repeat this code for any Queue */
        /* as in future we may have different No of buffers */
      }
      else
      {
        /* TS enabled for RX S */
        LpCurrentExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA)) (LulAddr);                        /* PRQA S 0306, 3432 # JV-01, JV-01 */

        do
        {
          if ((uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs > LulIdx)
          {
            /* Extended Data Descriptor */
            SetExtFempty(LulCtrlIdx, LpCurrentExtDataDesc, LucQidx, ETH_RX, LulIdx);
          }
          else
          {
            SetLinkFix((uint32)LpCurrentExtDataDesc, LulAddr);                                                          /* PRQA S 0306 # JV-01 */
          }

          LulIdx++;                                                                                                     /* PRQA S 3383 # JV-01 */
          LpCurrentExtDataDesc++;                                                                                       /* PRQA S 2824 # JV-01 */
        }
        while (LulIdx <= (uint32)LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs);
      }
    }
  }
  else
  {
    /* No action required */
    /* Cannot set a descriptor chain which address is invalid */
  }

}


/***********************************************************************************************************************
** Function Name         : SetFempty
**
** Service ID            : NA
**
** Description           : Set Fempty in the standard (8Bytes) descriptor
**                         chain for passed Queue and Buffer Index
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Controlled Id
**                       : LpDescr - Pointer to a Data Descriptor
**                       : LucQidx-Index of the Queue (0..3 for Tx 0..17 for Rx)
**                       : LenQdir - Queue Direction (e.g. TX or RX)
**                       : LusBufidx - Index ot the TX or RX buffer
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_062
***********************************************************************************************************************/
STATIC FUNC(void, ETH_PRIVATE_CODE) SetFempty(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONSTP2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA) LpDescr,                  /* PRQA S 3432 # JV-01 */
  CONST(uint8, AUTOMATIC) LucQidx, CONST(Eth_DirectionType, AUTOMATIC) LenQdir,
  CONST(Eth_BufIdxType, AUTOMATIC) LulBufIdx)
{
  if (NULL_PTR != LpDescr)
  {
    if (ETH_TX == LenQdir)
    {
      /* TX Queue */
      LpDescr->stHeader.ulDie = 0U;
      LpDescr->stHeader.ulCtrl = 0U;
      /* It will be set only during Transmission at run time */
      LpDescr->stHeader.ulDs = 0U;
      LpDescr->ulDptr = (uint32)0x0U;
      LpDescr->stHeader.ulDt = ETH_DESC_FEMPTY;
    }
    else if (ETH_BECHANNEL == LucQidx)
    {
      /* RX BE */
      LpDescr->stHeader.ulDie = 0U;
      LpDescr->stHeader.ulCtrl = 0U;
      LpDescr->stHeader.ulDs = ETH_RX_BUF_LENGTH;
      LpDescr->ulDptr = Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaRxBeQueueBuf[LulBufIdx];                                 /* PRQA S 2844 # JV-01 */
      LpDescr->stHeader.ulDt = ETH_DESC_FEMPTY;
    }
    else if (ETH_NCCHANNEL == LucQidx)
    {
      /* RX NC */
      /* This is an Error as RX NC is always Extended Descriptor */
    }
    else
    {
      /* RX S */
      LpDescr->stHeader.ulDie = 0U;
      LpDescr->stHeader.ulCtrl = 0U;
      LpDescr->stHeader.ulDs = ETH_RX_BUF_LENGTH;
      /* Attention Q index!*/
      LpDescr->ulDptr = Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaRxSQueueBuf[LucQidx][LulBufIdx];
      LpDescr->stHeader.ulDt = ETH_DESC_FEMPTY;
    }
  }
  else
  {
    /* Cannot Set Fempty for NULL Address */
  }
}


/***********************************************************************************************************************
** Function Name       : SetExtFempty
**
** Service ID          : NA
**
** Description         : Set Fempty in the extended (20Bytes) descriptor
**                       chain for passed Queue and Buffer Index
**
** Sync/Async          : Synchronous
**
** Reentrancy          : Non-Reentrant
**
** Input Parameters    : LulCtrlIdx - Controlled Id
**                     : LpDescr - Pointer to a Data Descriptor
**                     : LucQidx - Index of the Queue (0..3 for Tx 0..17 for Rx)
**                     : LenQdir - Queue Direction (e.g. TX or RX)
**                     : LulBufIdx - Index ot the TX or RX buffer
**
** InOut Parameters    : None
**
** Output Parameters   : None
**
** Return parameter    : None
**
** Preconditions       : None
**
** Global Variable(s)  : Eth_GaaCtrlStat
**
** Function(s) invoked : None
**
** Registers Used      : None
**
** Reference ID        : ETH_DUD_ACT_063
***********************************************************************************************************************/
STATIC FUNC(void, ETH_PRIVATE_CODE) SetExtFempty(
  CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA) LpDescr,                                                    /* PRQA S 3432 # JV-01 */
  CONST(uint8, AUTOMATIC) LucQidx, CONST(Eth_DirectionType, AUTOMATIC) LenQdir,
  CONST(Eth_BufIdxType, AUTOMATIC) LulBufIdx)
{
  if (((NULL_PTR != LpDescr) && (ETH_TX != LenQdir)))
  {
    LpDescr->stHeader.ulDie = 0U;
    LpDescr->stHeader.ulCtrl = 0U;
    LpDescr->stHeader.ulDs = ETH_RX_BUF_LENGTH;
    LpDescr->stTimestamp.ulTimestamp0 = 0x0U;
    LpDescr->stTimestamp.ulTimestamp1 = 0x0U;
    LpDescr->stTimestamp.usTimestamp2 = 0x0U;
    if (ETH_BECHANNEL == LucQidx)
    {
      /* RX BE */
      LpDescr->ulDptr = Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaRxBeQueueBuf[LulBufIdx];                                 /* PRQA S 2844 # JV-01 */
    }
    else if (ETH_NCCHANNEL == LucQidx)
    {
      /* RX NC */
      LpDescr->ulDptr = Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaRxNcQueueBuf[LulBufIdx];
    }
    else
    {
      /* RX S */
      /* Attention Q index!*/
      LpDescr->ulDptr = Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaRxSQueueBuf[LucQidx][LulBufIdx];
    }
    LpDescr->stHeader.ulDt = ETH_DESC_FEMPTY;
  }
  else
  {
    /* Cannot Set Fempty for NULL Address */
    /* Error as TX Descr are always Standard and not extended */
  }
}


/***********************************************************************************************************************
** Function Name         : SetLinkFix
**
** Service ID            : NA
**
** Description           : Set LinkFix in the descriptor chain passed as first Argument
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulLinkDesc - Pointer to the Descriptor to be set
**                       : LulAddrToLink - Link Address to be set
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : None
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_064
***********************************************************************************************************************/
STATIC FUNC(void, ETH_PRIVATE_CODE) SetLinkFix(
  CONST(uint32, AUTOMATIC) LulLinkDesc, CONST(uint32, AUTOMATIC) LulAddrToLink)
{
  P2VAR(Eth_LinkDescType, AUTOMATIC, ETH_APPL_DATA) LpLinkDesc;                                                         /* PRQA S 3432 # JV-01 */
  LpLinkDesc = (P2VAR(Eth_LinkDescType, AUTOMATIC, ETH_APPL_DATA)) LulLinkDesc;                                         /* PRQA S 0306, 3432 # JV-01, JV-01 */

  LpLinkDesc->stHeader.ulDie = 0U;                                                                                      /* PRQA S 2814 # JV-01 */                                    
  LpLinkDesc->stHeader.ulRes = 0U;
  LpLinkDesc->ulDptr         = LulAddrToLink;
  LpLinkDesc->stHeader.ulDt  = ETH_DESC_LINKFIX;
}

/***********************************************************************************************************************
** Function Name         : DescUpdate
**
** Service ID            : NA
**
** Description           : update descriptor chain (without timestamp support).
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulDescPtr - pointer to descriptor to be updated
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : LpDataDesc - last descriptor address
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_065
***********************************************************************************************************************/
STATIC FUNC(uint32, ETH_PRIVATE_CODE) DescUpdate(
  CONST(uint32, AUTOMATIC) LulDescPtr)
{
  P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA) LpDataDesc;                                                         /* PRQA S 3432 # JV-01 */
  LpDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA)) LulDescPtr;                                          /* PRQA S 0306, 3432 # JV-01, JV-01 */

  LpDataDesc->stHeader.ulDie  = 0U;                                                                                     /* PRQA S 2814 # JV-01 */
  LpDataDesc->stHeader.ulCtrl = 0U;
  LpDataDesc->stHeader.ulDs   = ETH_RX_BUF_LENGTH;
  LpDataDesc->stHeader.ulDt   = ETH_DESC_FEMPTY;

  return ((uint32)LpDataDesc);                                                                                          /* PRQA S 0306 # JV-01 */
}

/***********************************************************************************************************************
** Function Name         : DescTsUpdate
**
** Service ID            : NA
**
** Description           : update descriptor chain (with timestamp support).
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulDescPtr - pointer to descriptor to be updated
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : LpExtDataDesc - last descriptor address
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_066
***********************************************************************************************************************/
STATIC FUNC(uint32, ETH_PRIVATE_CODE) DescTsUpdate(
  CONST(uint32, AUTOMATIC) LulDescPtr)
{
  P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA) LpExtDataDesc;                                                   /* PRQA S 3432 # JV-01 */
  LpExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA)) LulDescPtr;                                    /* PRQA S 0306, 3432 # JV-01, JV-01 */

  LpExtDataDesc->stHeader.ulDie  = 0U;                                                                                  /* PRQA S 2814 # JV-01 */
  LpExtDataDesc->stHeader.ulCtrl = 0U;
  LpExtDataDesc->stHeader.ulDs   = ETH_RX_BUF_LENGTH;
  LpExtDataDesc->stTimestamp.ulTimestamp0 = 0UL;
  LpExtDataDesc->stTimestamp.ulTimestamp1 = 0UL;
  LpExtDataDesc->stTimestamp.usTimestamp2 = 0U;
  LpExtDataDesc->stHeader.ulDt   = ETH_DESC_FEMPTY;

  return ((uint32)LpExtDataDesc);                                                                                       /* PRQA S 0306 # JV-01 */
}

/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_DMACInit
**
** Service ID            : NA
**
** Description           : Initialize DMAC
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Index of a controller
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaAvbConfig, Eth_GaaETNBRegs
**
** Function(s) invoked   : TxRxConfig, TxRxIntConfig, SetCbsParameter,
**                         RxQueueSet, SetRxFilter
**
** Registers Used        : CCC, EIC, GIC, RPC
**
** Reference ID          : ETH_DUD_ACT_067,
** Reference ID          : ETH_DUD_ACT_067_GBL001, ETH_DUD_ACT_067_GBL002
** Reference ID          : ETH_DUD_ACT_067_GBL003, ETH_DUD_ACT_067_GBL004
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_Hw_Etnb_DMACInit(                                                                      /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  /* configure CCC register */
  if (ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enInternalLoopBackMode)
  {
    Eth_GaaETNBRegs[LulCtrlIdx]->ulCCC |= ETH_ETNB_CCC_LBME;                                                            /* PRQA S 2844, 2814 # JV-01, JV-01 */
  }
  else
  {
    /* No action required */
  }

  /* configure Error interrupt mask - EIC register */
  Eth_GaaETNBRegs[LulCtrlIdx]->ulEIC = ETH_ETNB_EIC_ERROR;
  /* configure reception part - RCR register */
  /* configure transmission part - TGC register */
  TxRxConfig(LulCtrlIdx, &(Eth_GaaAvbConfig[LulCtrlIdx].stTxConfig), &(Eth_GaaAvbConfig[LulCtrlIdx].stRxConfig));       /* PRQA S 2844 # JV-01 */

  TxRxIntConfig(LulCtrlIdx);

  #if (ETH_QOS_SUPPORT == STD_ON)
  /* Set CBS Counter parameters (CIV and CDV)  */
  SetCbsParameter(LulCtrlIdx);
  #endif

  /* set GIC interrupt */
  Eth_GaaETNBRegs[LulCtrlIdx]->ulGIC = 0UL;

  /* Set RQCi register */
  RxQueueSet(LulCtrlIdx);

  /* Set padding to zero - Padding Not used */
  Eth_GaaETNBRegs[LulCtrlIdx]->ulRPC = 0UL;

  /* Set Common RX Filter if enabled */
  #if (ETH_STREAM_FILTERING == STD_ON)
  SetRxFilter(LulCtrlIdx);
  #endif
}

/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_OpModeChange
**
** Service ID            : NA
**
** Description           : To change Operating Mode.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Instance number
**                       : LenMode - Mode to change
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Ethernet Error
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat, Eth_GaaETNBRegs
**
** Function(s) invoked   : GetCounterValue, Eth_GetTimeOutValue
**
** Registers Used        : CCC, CSR
**
** Reference ID          : ETH_DUD_ACT_068,
** Reference ID          : ETH_DUD_ACT_068_GBL001, ETH_DUD_ACT_068_REG001
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_Hw_Etnb_OpModeChange(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(Eth_OpStatusType, AUTOMATIC) LenMode)
{
  Std_ReturnType LucReturnValue;
  uint32 LulRegVal;
  TickType LulTimeOutCountInit;
  TickType LulTimeoutCountElap;

  LulTimeOutCountInit = 0UL;

  if (Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enDevStatus != LenMode)                                                      /* PRQA S 2844 # JV-01 */
  {
    LulRegVal = Eth_GaaETNBRegs[LulCtrlIdx]->ulCCC & ~ETH_ETNB_CCC_OPC_MASK;                                            /* PRQA S 2844, 2814 # JV-01, JV-01 */
    Eth_GaaETNBRegs[LulCtrlIdx]->ulCCC = LulRegVal | (uint32)LenMode;

    (void)GetCounterValue((CounterType)ETH_OS_COUNTER_ID, &LulTimeOutCountInit);
    do
    {
      LulTimeoutCountElap = Eth_GetTimeOutValue(LulTimeOutCountInit);
      LulRegVal = Eth_GaaETNBRegs[LulCtrlIdx]->ulCSR & ETH_ETNB_CSR_OPS;
    } while ((LulRegVal != (1UL << (uint32)LenMode)) && (LulTimeoutCountElap <= ETH_TIMEOUT_COUNT));

    if ((1UL << (uint32)LenMode) == LulRegVal)
    {
      /* No Time-out - mode changed  */
      Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enDevStatus = LenMode;                                                       /* PRQA S 2844 # JV-01 */
      LucReturnValue = E_OK;
    }
    else
    {
      /* Return Error - as mode not changed */
      LucReturnValue = E_NOT_OK;
    }
  }
  else
  {
    /* Controller is already in the requested state */
    LucReturnValue = E_OK;
  }

  return LucReturnValue;
}

/***********************************************************************************************************************
** Function Name         : RxQueueSet
**
** Service ID            : NA
**
** Description           : Set Receive Queue
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Instance number
**                       : LpQconfig  - Queue config structure pointer
**                       : LulQIndex - Queue Index
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpCtrlConfigPtr, Eth_GaaETNBRegs,
**                         Eth_GaaQConfig
**
** Function(s) invoked   : None
**
** Registers Used        : RQCi
**
** Reference ID          : ETH_DUD_ACT_257, ETH_DUD_ACT_257_GBL001
***********************************************************************************************************************/
STATIC FUNC(void, ETH_PRIVATE_CODE) RxQueueSet(
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  uint32 LulIdx;
  uint32 LulQIndex;
  uint32 LulConfig;
  P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;

  LpHwUnitConfig = (P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA))                                              /* PRQA S 0316 # JV-01 */
    Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;

  for (LulIdx = 0UL; LulIdx < (uint32)LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue; LulIdx++)                        /* PRQA S 2814 # JV-01 */
  {
    LulQIndex = (uint32) (LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulIdx].ucEthRxQueueId);                         /* PRQA S 2824 # JV-01 */

    LulConfig = (uint32)(ETH_ETNB_SET_RQCi_RSMr(Eth_GaaQConfig[LulIdx].enRsm, LulQIndex) |                              /* PRQA S 3384, 3469, 2844 # JV-01, JV-01, JV-01 */
      ETH_ETNB_SET_RQCi_UFCCr(Eth_GaaQConfig[LulIdx].enUfcc, LulQIndex) |                                               /* PRQA S 3384, 3383, 3469 # JV-01, JV-01, JV-01 */
      ETH_ETNB_SET_RQCi_PIAr(Eth_GaaQConfig[LulIdx].enPia, LulQIndex));                                                 /* PRQA S 3383, 3469, 3384 # JV-01, JV-01, JV-01 */
    Eth_GaaETNBRegs[LulCtrlIdx]->ulRQC[ETH_ETNB_GET_RQCi(LulQIndex)] = LulConfig;                                       /* PRQA S 2844, 2814, 3469 # JV-01, JV-01, JV-01 */
  }
}

/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_UFCounterGet
**
** Service ID            : NA
**
** Description           : Get Unread Frame Counter
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Instance number
**                       : LulQIndex - Queue Index
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : LulUFCount - Number of unread frames
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETNBRegs
**
** Function(s) invoked   : None
**
** Registers Used        : UFCVi
**
** Reference ID          : ETH_DUD_ACT_069
***********************************************************************************************************************/
FUNC(uint32, ETH_PRIVATE_CODE) Eth_Hw_Etnb_UFCounterGet(                                                                /* PRQA S 1505, 1532 # JV-01, JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulQIndex)
{
  uint32  LulUfcv;
  uint32  LulUFCount;
  LulUfcv = Eth_GaaETNBRegs[LulCtrlIdx]->ulUFCV[ETH_ETNB_GET_UFCVi(LulQIndex)];                                         /* PRQA S 2844, 2814, 3469 # JV-01, JV-01, JV-01 */
  LulUFCount = ETH_ETNB_GET_UFCVi_CVr(LulUfcv, LulQIndex);                                                              /* PRQA S 3384 # JV-01 */
  return (LulUFCount);
}

/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_RxDescChainConfig
**
** Service ID            : NA
**
** Description           : configure Rx descriptor chain
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Instance number
**                       : LpChConfig - Channel Config Pointer Index
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat, Eth_GaaETNBRegs
**
** Function(s) invoked   : RxDescChainUpdate,
**
** Registers Used        : DLR
**
** Reference ID          : ETH_DUD_ACT_071,
** Reference ID          : ETH_DUD_ACT_071_GBL002, ETH_DUD_ACT_071_GBL003
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_Hw_Etnb_RxDescChainConfig(                                                             /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONSTP2CONST(Eth_RxChConfigType, AUTOMATIC, ETH_APPL_DATA) LpChConfig)
{
  P2VAR(Eth_LinkDescType, AUTOMATIC, ETH_APPL_DATA) LpLinkAddr;                                                         /* PRQA S 3432, 3678 # JV-01, JV-01 */
  P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA) LpDataDesc;                                                         /* PRQA S 3432, 3678 # JV-01, JV-01 */
  P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA) LpExtDataDesc;                                                   /* PRQA S 3432, 3678 # JV-01, JV-01 */
  LpLinkAddr = (P2VAR(Eth_LinkDescType, AUTOMATIC, ETH_APPL_DATA))                                                      /* PRQA S 0306, 3432 # JV-01, JV-01 */
    (Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.ulDescTableAddr);                                                             /* PRQA S 2844 # JV-01 */

  LpLinkAddr += (ETH_RXBEQ_OFFSET + LpChConfig->ucChNum);                                                               /* PRQA S 0488, 2824, 2814 # JV-01, JV-01, JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaNextRxDesc[LpChConfig->ucChNum] = (uint32)(LpLinkAddr->ulDptr);                /* PRQA S 2844, 2814 # JV-01, JV-01 */

  if (ETH_BECHANNEL == LpChConfig->ucChNum)
  {
    /* best effort channel */
    if (ETH_DISABLE == Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enRxBeTimestamp)
    {
      LpDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA))                                                  /* PRQA S 0306, 3432 # JV-01, JV-01 */
        Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaNextRxDesc[LpChConfig->ucChNum];
      do
      {
        LpDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA))                                                /* PRQA S 0306, 3432 # JV-01, JV-01 */
          RxDescChainUpdate(LulCtrlIdx, (uint32)LpDataDesc, LpChConfig);                                                /* PRQA S 0306 # JV-01 */
        LpDataDesc++;                                                                                                   /* PRQA S 2824 # JV-01 */
      } while (LpDataDesc->stHeader.ulDt != ETH_DESC_LINKFIX);                                                          /* PRQA S 2814 # JV-01 */
    }
    else
    {
      LpExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA))                                            /* PRQA S 0306, 3432 # JV-01, JV-01 */
        Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaNextRxDesc[LpChConfig->ucChNum];
      do
      {
        LpExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA))                                          /* PRQA S 0306, 3432 # JV-01, JV-01 */
          RxDescChainUpdate(LulCtrlIdx, (uint32)LpExtDataDesc, LpChConfig);                                             /* PRQA S 0306 # JV-01 */
        LpExtDataDesc++;                                                                                                /* PRQA S 2824 # JV-01 */
      } while (LpExtDataDesc->stHeader.ulDt != ETH_DESC_LINKFIX);                                                       /* PRQA S 2814 # JV-01 */
    }
  }
  else if (ETH_NCCHANNEL == LpChConfig->ucChNum)
  {
    /* network control channel */
    LpExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA))                                              /* PRQA S 0306, 3432 # JV-01, JV-01 */
      Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaNextRxDesc[LpChConfig->ucChNum];
    do
    {
      LpExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA))                                            /* PRQA S 0306, 3432 # JV-01, JV-01 */
        RxDescChainUpdate(LulCtrlIdx, (uint32)LpExtDataDesc, LpChConfig);                                               /* PRQA S 0306 # JV-01 */
      LpExtDataDesc++;                                                                                                  /* PRQA S 2824 # JV-01 */
    } while (LpExtDataDesc->stHeader.ulDt != ETH_DESC_LINKFIX);                                                         /* PRQA S 2814 # JV-01 */

  }
  else
  {
    /* stream channel */
    if (ETH_DISABLE == Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enRxSTimestamp)
    {
      LpDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA))                                                  /* PRQA S 0306, 3432 # JV-01, JV-01 */
        Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaNextRxDesc[LpChConfig->ucChNum];
      do
      {
        LpDataDesc = (P2VAR(Eth_DataDescType, AUTOMATIC, ETH_APPL_DATA))                                                /* PRQA S 0306, 3432 # JV-01, JV-01 */
          RxDescChainUpdate(LulCtrlIdx, (uint32)LpDataDesc, LpChConfig);                                                /* PRQA S 0306 # JV-01 */

        LpDataDesc++;                                                                                                   /* PRQA S 2824 # JV-01 */
      } while (LpDataDesc->stHeader.ulDt != ETH_DESC_LINKFIX);                                                          /* PRQA S 2814 # JV-01 */
    }
    else
    {
      LpExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA))                                            /* PRQA S 0306, 3432 # JV-01, JV-01 */
        Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaNextRxDesc[LpChConfig->ucChNum];
      do
      {
        LpExtDataDesc = (P2VAR(Eth_ExtDataDescType, AUTOMATIC, ETH_APPL_DATA))                                          /* PRQA S 0306, 3432 # JV-01, JV-01 */
          RxDescChainUpdate(LulCtrlIdx, (uint32)LpExtDataDesc, LpChConfig);                                             /* PRQA S 0306 # JV-01 */

        LpExtDataDesc++;                                                                                                /* PRQA S 2824 # JV-01 */
      } while (LpExtDataDesc->stHeader.ulDt != ETH_DESC_LINKFIX);                                                       /* PRQA S 2814 # JV-01 */
    }
  }
  /* reload desc. base address */
  if (ETH_OPERATION == Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enDevStatus)                                                /* PRQA S 2844 # JV-01 */
  {
    Eth_GaaETNBRegs[LulCtrlIdx]->ulDLR = (1UL << (LpChConfig->ucChNum + ETH_RXBEQ_OFFSET));                             /* PRQA S 2844, 2814 # JV-01, JV-01 */
  }
  else
  {
    /* No action required */
  }
}

/***********************************************************************************************************************
** Function Name         : RxDescChainUpdate
**
** Service ID            : NA
**
** Description           : Update receive descriptor chain
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Instance number
**                       : LulDescPtr - Descriptor pointer
**                       : LpChConfig - Channel Config Pointer
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : LulLastDesc - Last processed descriptor address
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat
**
** Function(s) invoked   : DescUpdate, DescTsUpdate
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_072
***********************************************************************************************************************/
STATIC FUNC(uint32, ETH_PRIVATE_CODE) RxDescChainUpdate(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulDescPtr,
  CONSTP2CONST(Eth_RxChConfigType, AUTOMATIC, ETH_APPL_DATA) LpChConfig)
{
  uint32 LulLastDesc;

  switch (LpChConfig->enChType)                                                                                         /* PRQA S 2814 # JV-01 */
  {
  case ETH_RX_BE:
    if (ETH_DISABLE == Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enRxBeTimestamp)                                            /* PRQA S 2844 # JV-01 */
    {
      LulLastDesc = DescUpdate(LulDescPtr);
    }
    else
    {
      LulLastDesc = DescTsUpdate(LulDescPtr);
    }
    break;
  case ETH_RX_S:
    if (ETH_DISABLE == Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enRxSTimestamp)
    {
      LulLastDesc = DescUpdate(LulDescPtr);
    }
    else
    {
      LulLastDesc = DescTsUpdate(LulDescPtr);
    }
    break;
  /* ETH_RX_NC */
  default:
    LulLastDesc = DescTsUpdate(LulDescPtr);
    break;
  }
  return (LulLastDesc);
}

/***********************************************************************************************************************
** Function Name         : TxRxConfig
**
** Service ID            : NA
**
** Description           : Tx parameters Config
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Instance number
**                       : LpTxConfig - Tx Config Pointer
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat, Eth_GaaETNBRegs
**
** Function(s) invoked   : None
**
** Registers Used        : RCR, TGC
**
** Reference ID          : ETH_DUD_ACT_073,
** Reference ID          : ETH_DUD_ACT_073_GBL001, ETH_DUD_ACT_073_GBL002
** Reference ID          : ETH_DUD_ACT_073_GBL003
***********************************************************************************************************************/
STATIC FUNC(void, ETH_PRIVATE_CODE) TxRxConfig(
  CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2CONST(Eth_TxConfigType, AUTOMATIC, ETH_APPL_DATA) LpTxConfig,
  CONSTP2CONST(Eth_RxConfigType, AUTOMATIC, ETH_APPL_DATA) LpRxConfig)
{
  uint32 LulRegVal;
  LulRegVal = (uint32)(((uint32)LpRxConfig->enEncf << ETH_ETNB_RCR_ENCF_SHIFT) |                                        /* PRQA S 2814 # JV-01 */
              ((uint32)LpRxConfig->enEsf << ETH_ETNB_RCR_ESF_SHIFT) |
              ((uint32)LpRxConfig->enEts0 << ETH_ETNB_RCR_ETS0_SHIFT) |
              ((uint32)LpRxConfig->enEts2 << ETH_ETNB_RCR_ETS2_SHIFT) |
              ((uint32)LpRxConfig->ulRfcl << ETH_ETNB_RCR_RFCL_SHIFT));

  Eth_GaaETNBRegs[LulCtrlIdx]->ulRCR = LulRegVal;                                                                       /* PRQA S 2844, 2814 # JV-01, JV-01 */

  Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enRxBeTimestamp = LpRxConfig->enEts0;                                            /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enRxSTimestamp = LpRxConfig->enEts2;                                             /* PRQA S 2844 # JV-01 */
  LulRegVal = (uint32)(((uint32)LpTxConfig->enTsm0) |                                                                   /* PRQA S 2814 # JV-01 */
              ((uint32)LpTxConfig->enTsm1 << ETH_ETNB_TGC_TSM1_SHIFT) |
              ((uint32)LpTxConfig->enTsm2 << ETH_ETNB_TGC_TSM2_SHIFT) |
              ((uint32)LpTxConfig->enTsm3 << ETH_ETNB_TGC_TSM3_SHIFT) |
              ((uint32)LpTxConfig->enEcbs << ETH_ETNB_TGC_TQP0_SHIFT) |
              ((uint32)LpTxConfig->enTqp << ETH_ETNB_TGC_TQP1_SHIFT) |
               ETH_ETNB_TGC_TBDn_VALUE);

  Eth_GaaETNBRegs[LulCtrlIdx]->ulTGC = LulRegVal;
}

/***********************************************************************************************************************
** Function Name         : TxRxIntConfig
**
** Service ID            : NA
**
** Description           : Tx/Rx Interrupt config
**
** Sync/Async            : NA
**
** Reentrancy            : NA
**
** Input Parameters      : LulCtrlIdx - Instance number
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpCtrlConfigPtr, Eth_GaaETNBRegs
**                         Eth_GpEthConfigPtr
**
** Function(s) invoked   : None
**
** Registers Used        : RIC0, RIC2, RIC3, DIC, TIC
**
** Reference ID          : ETH_DUD_ACT_074,
** Reference ID          : ETH_DUD_ACT_074_GBL001, ETH_DUD_ACT_074_GBL002
** Reference ID          : ETH_DUD_ACT_074_GBL003
***********************************************************************************************************************/
STATIC FUNC(void, ETH_PRIVATE_CODE) TxRxIntConfig(
  CONST(uint32, AUTOMATIC) LulCtrlIdx)                                                                                  /* PRQA S 3206 # JV-01 */
{
  #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON || ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
  P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;
  uint8 LucIdx;
  #endif
  #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON)
  uint8 LucQIdx;
  #endif
  #if (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
  uint8 LucBitNo;
  #endif

  #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON || ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
  LpHwUnitConfig = (P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA))                                              /* PRQA S 0316 # JV-01 */
    Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;
  #endif

  #if (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
  if (ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enRxInterruptMode)
  {
    /* set RIC0 interrupt */
    for (LucIdx = 0U; LucIdx < LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue; LucIdx++)                               /* PRQA S 2814 # JV-01 */
    {
      LucBitNo = LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LucIdx].ucEthRxQueueId;                                   /* PRQA S 2824 # JV-01 */
      Eth_GaaETNBRegs[LulCtrlIdx]->ulRIC0 |= (1UL << LucBitNo);                                                         /* PRQA S 2844, 2814 # JV-01, JV-01 */
    }
  }
  else
  {
    /* No action required */
  }
  #endif

  #if (ETH_CTRL_ENABLE_TX_INTERRUPT == STD_ON)
  if (ETH_ENABLE == Eth_GpEthConfigPtr[LulCtrlIdx].enTxInterruptMode)
  {
    /* set descriptor interrupt */
    for (LucIdx = 0U; LucIdx < LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue; LucIdx++)                               /* PRQA S 2814 # JV-01 */
    {
      /* Get Tx queue id */
      LucQIdx = LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LucIdx].ucEthTxQueueId;                                    /* PRQA S 2824 # JV-01 */
      /* Enable descriptor interrupt
      Queue n (n = 0..3): corresponding to descriptor interrupt (n + 1) */
      Eth_GaaETNBRegs[LulCtrlIdx]->ulDIC |= (1UL << (LucQIdx + 1U));                                                    /* PRQA S 3383, 2844, 2814 # JV-01, JV-01, JV-01 */
    }
  }
  else
  {
    /* No action required */
  }
  #endif

  #if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
  Eth_GaaETNBRegs[LulCtrlIdx]->ulTIC = ETH_TIC_TFUE;                                                                    /* PRQA S 2844, 2814 # JV-01, JV-01 */
  #endif
}

/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_SYS_AllocDescBuffer
**
** Service ID            : NA
**
** Description           : Allocate memory for DBA (Descriptor Base Address)
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Instance number
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Std_ReturnType
**
** Preconditions         : None
**
** Global Variable(s)    : None
**
** Function(s) invoked   : Eth_Ram_Alloc, AllocMemForDesc, FillDescMemory
**                         IsQueueConfigured
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_075
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_Hw_Etnb_SYS_AllocDescBuffer(                                                 /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  Eth_DirectionType LenQdirection; /* Direction Tx or Rx of the Queue */
  uint8 LucQid;                   /* Index of the Queue  */
  uint8 LucIdx;
  uint32 LulDescAddr;
  boolean LblQueueConfig;
  Std_ReturnType LucReturnValue;

  P2VAR(Eth_LinkDescType, AUTOMATIC, ETH_APPL_DATA) LpLinkDesc;                                                         /* PRQA S 3432 # JV-01 */

  LucReturnValue = E_OK;

  LpLinkDesc  = (P2VAR(Eth_LinkDescType, AUTOMATIC, ETH_APPL_DATA))                                                     /* PRQA S 0306, 3432 # JV-01, JV-01 */
    Eth_Ram_Alloc(LulCtrlIdx, (ETH_TXQ_NUM + ETH_RXQ_NUM) * (uint32)ETH_NORMAL_DESC_SIZE);
  if (NULL_PTR != LpLinkDesc)
  {
    for (LucIdx = 0U; LucIdx < (ETH_TXQ_NUM + ETH_RXQ_NUM); LucIdx++)
    {
      /* !!!!!!!Attention index need to be reduced by 4 for a RX Queue !!!!! */
      if (ETH_TXQ_NUM > LucIdx)
      {
        /* TX Queue */
        LenQdirection = ETH_TX;
        LucQid = LucIdx;
      }
      else
      {
        /* Rx Queue */
        LenQdirection = ETH_RX;
        LucQid = (LucIdx - (uint8)ETH_TXQ_NUM);
      }

      LblQueueConfig = IsQueueConfigured(LulCtrlIdx, LucQid, LenQdirection);

      if (ETH_TRUE == LblQueueConfig)
      {
        /* Queue has been configured */
        /* Alloc Desc Chain Memory */

        LulDescAddr = AllocMemForDesc(LulCtrlIdx, LucQid, LenQdirection);
        if (0UL != LulDescAddr)
        {
          FillDescMemory(LpLinkDesc, LulDescAddr);
        }
        else
        {
          /* Memory allocation error due to lack of resources */
          LucReturnValue = E_NOT_OK;
          break;
        }
      }
      else
      {
        /* Queue not configured */
        /* Set Null Dptr field in Link Descriptor */
        FillDescMemory(LpLinkDesc, (uint32)0x0U);
      }
      /* Go to the next DBA element */
      LpLinkDesc++;
    } /* end loop for all Tx and Rx Queues  */
  }
  else
  {
    /* Memory allocation error due to lack of resources */
    LucReturnValue = E_NOT_OK;
  }

  return LucReturnValue;
}

/***********************************************************************************************************************
** Function Name         : AllocMemForDesc
**
** Service ID            : NA
**
** Description           : Alloc Memory for Descriptor Chain
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Controller Index
**                       : LucQidx - Queue Index
**                       : LenQdir - Queue Direction
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : LulDescAddr
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpCtrlConfigPtr, Eth_GaaCtrlStat
**
** Function(s) invoked   : Eth_Ram_Alloc,
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_076,
** Reference ID          : ETH_DUD_ACT_076_GBL001
***********************************************************************************************************************/
STATIC FUNC(uint32, ETH_PRIVATE_CODE) AllocMemForDesc(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint8, AUTOMATIC) LucQidx, CONST(Eth_DirectionType, AUTOMATIC) LenQdir)
{
  uint32 LulDescAddr;
  uint32 LulCfgIdx;
  P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;

  LpHwUnitConfig = (P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA))                                              /* PRQA S 0316 # JV-01 */
    Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;
  if (ETH_TX == LenQdir)
  {
    /* Search for Tx configuration index */
    for (LulCfgIdx = 0; LulCfgIdx < LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue; LulCfgIdx++)                       /* PRQA S 2814 # JV-01 */
    {
      if (LucQidx == LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulCfgIdx].ucEthTxQueueId)                            /* PRQA S 2824 # JV-01 */
      {
        break;
      }
      else
      {
        /* No action required */
      }
    }

    /* TX */
    LulDescAddr = Eth_Ram_Alloc(LulCtrlIdx,
      (uint32)((LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulCfgIdx].usEthTxQueueBufs + ETH_CYCLIC_DESC_NUM)         /* PRQA S 4391, 3383 # JV-01, JV-01 */
      * ETH_NORMAL_DESC_SIZE));                                                                                         /* PRQA S 3384 # JV-01 */
    /* Store the current Tx descriptor address */
    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaLastTxDesc[LucQidx] = LulDescAddr;                                           /* PRQA S 2844 # JV-01 */
    Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.aaHeadTxDesc[LucQidx] = LulDescAddr;                                           /* PRQA S 2844 # JV-01 */
  }
  else
  {
    /* Search for Rx configuration index */
    for (LulCfgIdx = 0; LulCfgIdx < LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue; LulCfgIdx++)
    {
      if (LucQidx == LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].ucEthRxQueueId)                            /* PRQA S 2824 # JV-01 */
      {
        break;
      }
      else
      {
        /* No action required */
      }
    }

    /* Rx */
    switch (LucQidx)
    {
    case ETH_BECHANNEL:
      /* RX BE */
      if (ETH_ENABLE == LpHwUnitConfig->stRxConfig.enEts0)
      {
        /* TS Enabled for RX BE */
        LulDescAddr = Eth_Ram_Alloc(LulCtrlIdx,
          (uint32)((LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs + ETH_CYCLIC_DESC_NUM)     /* PRQA S 4391, 3383 # JV-01, JV-01 */
          * ETH_SPECIAL_DESC_SIZE));                                                                                    /* PRQA S 3384 # JV-01 */
      }
      else
      {
        /* TS Disabled for RX BE */
        LulDescAddr = Eth_Ram_Alloc(LulCtrlIdx,
          (uint32)((LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs + ETH_CYCLIC_DESC_NUM)     /* PRQA S 4391, 3383 # JV-01, JV-01 */
          * ETH_NORMAL_DESC_SIZE));                                                                                     /* PRQA S 3384 # JV-01 */
      }

      break;

    case ETH_NCCHANNEL:
      /* RX NC */
      LulDescAddr = Eth_Ram_Alloc(LulCtrlIdx,
        (uint32)((LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs + ETH_CYCLIC_DESC_NUM)       /* PRQA S 4391, 3383 # JV-01, JV-01 */
        * ETH_SPECIAL_DESC_SIZE));                                                                                      /* PRQA S 3384 # JV-01 */
      break;

    default:
      /* RX Stream */
      if (ETH_ENABLE == LpHwUnitConfig->stRxConfig.enEts2)
      {
        /* TS enabled for Rx Stream */
        LulDescAddr = Eth_Ram_Alloc(LulCtrlIdx,
          (uint32)(((LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs + ETH_CYCLIC_DESC_NUM)    /* PRQA S 4391, 3383 # JV-01, JV-01 */
          * ETH_SPECIAL_DESC_SIZE)));                                                                                   /* PRQA S 3384 # JV-01 */
      }
      else
      {
        /* TS disabled for Rx Stream  */
        LulDescAddr = Eth_Ram_Alloc(LulCtrlIdx,
          (uint32)((LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulCfgIdx].usEthRxQueueBufs + ETH_CYCLIC_DESC_NUM)     /* PRQA S 4391, 3383 # JV-01, JV-01 */
          * ETH_NORMAL_DESC_SIZE));                                                                                     /* PRQA S 3384 # JV-01 */
      }

      break;
    }
  }

  return (LulDescAddr);
}

/***********************************************************************************************************************
** Function Name         : FillDescMemory
**
** Service ID            : NA
**
** Description           : Set LinkFix for all DBA for each Queue
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulDescAddr
**
** InOut Parameters      : LpLinkDescr
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : None
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_077
***********************************************************************************************************************/
STATIC FUNC(void, ETH_PRIVATE_CODE) FillDescMemory(
  CONSTP2VAR(Eth_LinkDescType, AUTOMATIC, ETH_APPL_DATA) LpLinkDescr, CONST(uint32, AUTOMATIC) LulDescAddr)             /* PRQA S 3432 # JV-01 */
{
  if (NULL_PTR != LpLinkDescr)
  {
    LpLinkDescr->stHeader.ulDie  = 0U;
    LpLinkDescr->stHeader.ulRes  = 0U;
    LpLinkDescr->ulDptr      = LulDescAddr;
    LpLinkDescr->stHeader.ulDt   = ETH_DESC_LINKFIX;
  }
  else
  {
    /* No action at this level */
  }
}

/***********************************************************************************************************************
** Function Name         : IsQueueConfigured
**
** Service ID            : NA
**
** Description           : Use the Binary Search Algo to discover if
**                         a Queue is configured and in this case return
**                         Direction otherwise return false and
**                         Direction is irrelevant
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LucQidx - Queue Index to see if configured
**                       : LenDir - Direction Type
**
** InOut Parameters      : None
**
** Output Parameters     : pDir Direction of the Queue Configured
**
** Return parameter      : True: Qidx found in config structure
**                       : False: Qidx not found in config struct
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GpCtrlConfigPtr
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_078
***********************************************************************************************************************/
STATIC FUNC(boolean, ETH_PRIVATE_CODE) IsQueueConfigured(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint8, AUTOMATIC) LucQidx, CONST(Eth_DirectionType, AUTOMATIC) LenDir)
{
  uint8 LucVal;
  boolean LblReturnVal;
  P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;

  LpHwUnitConfig = (P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA))                                              /* PRQA S 0316 # JV-01 */
    Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;
  LucVal = 0U;
  LblReturnVal = ETH_FALSE;

  if (ETH_TX == LenDir)
  {
    /* Tx Queues */
    do
    {
      if (LucQidx == LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LucVal].ucEthTxQueueId)                               /* PRQA S 2814, 2824 # JV-01, JV-01 */
      {
        LblReturnVal = ETH_TRUE;
      }
      else
      {
        LucVal++;                                                                                                       /* PRQA S 3383 # JV-01 */
      }
    }
    while ((ETH_FALSE == LblReturnVal) && (LucVal < LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue));
  }
  else
  {
    /* Rx Queues */
    do
    {
      if (LucQidx == LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LucVal].ucEthRxQueueId)                               /* PRQA S 2824 # JV-01 */
      {
        LblReturnVal = ETH_TRUE;
      }
      else
      {
        LucVal++;                                                                                                       /* PRQA S 3383 # JV-01 */
      }
    }
    while ((ETH_FALSE == LblReturnVal) && (LucVal < LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue));
  }

  return (LblReturnVal);
}

/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_DisableController
**
** Service ID            : NA
**
** Description           : This Disables the indexed Ethernet Controller
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Ethernet Controller index
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETNBRegs, Eth_GaaCtrlStat
**
** Function(s) invoked   : GetCounterValue, Eth_GetTimeOutValue,
**                         Eth_Hw_Etnb_OpModeChange
**
** Registers Used        : ETNBnCCC, ETNBnECMR, ETNBnTCCR, ETNBnCSR
**
** Reference ID          : ETH_DUD_ACT_079,
** Reference ID          : ETH_DUD_ACT_079_GBL001, ETH_DUD_ACT_079_REG001
** Reference ID          : ETH_DUD_ACT_079_REG002, ETH_DUD_ACT_079_REG003
** Reference ID          : ETH_DUD_ACT_079_REG005, ETH_DUD_ACT_079_REG007
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_Hw_Etnb_DisableController(                                                   /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  Std_ReturnType LucErrorValue;
  TickType LulTimeOutCountInit;
  TickType LulTimeoutCountElap;
  uint32 LulRegVal;

  if (ETH_OPERATION == Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enDevStatus)                                                /* PRQA S 2844 # JV-01 */
  {
    /* Disable the DMAC and MAC of Controller for pending Tx or Rx */
    Eth_GaaETNBRegs[LulCtrlIdx]->ulECMR &= ~ETH_ETNB_ECMR_RE_TE;                                                        /* PRQA S 2844, 2814 # JV-01, JV-01 */

    /* Wait for completion of transmission request */
    LulTimeOutCountInit = 0UL;
    (void)GetCounterValue((CounterType)ETH_OS_COUNTER_ID, &LulTimeOutCountInit);
    do
    {
      LulTimeoutCountElap = Eth_GetTimeOutValue(LulTimeOutCountInit);
      LulRegVal = Eth_GaaETNBRegs[LulCtrlIdx]->ulTCCR & ETH_TCCR_TSRQX_MASK;
    } while ((LulRegVal != 0UL) && (LulTimeoutCountElap <= ETH_TIMEOUT_COUNT));
    /* Wait for completion of reception/transmit process */
    LulTimeOutCountInit = 0UL;
    (void)GetCounterValue((CounterType)ETH_OS_COUNTER_ID, &LulTimeOutCountInit);
    do
    {
      LulTimeoutCountElap = Eth_GetTimeOutValue(LulTimeOutCountInit);
      LulRegVal = Eth_GaaETNBRegs[LulCtrlIdx]->ulCSR & ETH_CSR_RPO_TPOX_MASK;
    } while ((LulRegVal != 0UL) && (LulTimeoutCountElap <= ETH_TIMEOUT_COUNT));

    /* Process all required status information */
    /* In ETNB, necessary information is not reset in CONFIG and STANDBY mode.*/
    Eth_GaaETNBRegs[LulCtrlIdx]->ulCCC |= ETH_ETNB_CCC_DTSR;

    LulTimeOutCountInit = 0UL;
    (void)GetCounterValue((CounterType)ETH_OS_COUNTER_ID, &LulTimeOutCountInit);
    do
    {
      LulTimeoutCountElap = Eth_GetTimeOutValue(LulTimeOutCountInit);
      LulRegVal = Eth_GaaETNBRegs[LulCtrlIdx]->ulCSR & ETH_ETNB_CSR_DTS;
    } while ((LulRegVal != ETH_ETNB_CSR_DTS) && (LulTimeoutCountElap <= ETH_TIMEOUT_COUNT));
  }
  else
  {
    /* No action required */
  }

  LucErrorValue = Eth_Hw_Etnb_OpModeChange(LulCtrlIdx, ETH_CONFIG);

  if (ETH_CONFIG != Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enDevStatus)                                                   /* PRQA S 2844 # JV-01 */
  {
    LulRegVal = Eth_GaaETNBRegs[LulCtrlIdx]->ulCCC & ~ETH_ETNB_CCC_OPC_MASK;
    Eth_GaaETNBRegs[LulCtrlIdx]->ulCCC = LulRegVal | ETH_ETNB_CCC_OPC_OPERATION;
    Eth_GaaETNBRegs[LulCtrlIdx]->ulCCC &= ~ETH_ETNB_CCC_DTSR;

    /* Wait for completion of reception/transmit process */
    LulTimeOutCountInit = 0UL;
    (void)GetCounterValue((CounterType)ETH_OS_COUNTER_ID, &LulTimeOutCountInit);
    do
    {
      LulTimeoutCountElap = Eth_GetTimeOutValue(LulTimeOutCountInit);
      LulRegVal = Eth_GaaETNBRegs[LulCtrlIdx]->ulCSR & ETH_ETNB_CSR_RPO;
    } while ((LulRegVal != 0UL) && (LulTimeoutCountElap <= ETH_TIMEOUT_COUNT));

    Eth_GaaETNBRegs[LulCtrlIdx]->ulCCC |= ETH_ETNB_CCC_DTSR;

    LulTimeOutCountInit = 0UL;
    (void)GetCounterValue((CounterType)ETH_OS_COUNTER_ID, &LulTimeOutCountInit);
    do
    {
      LulTimeoutCountElap = Eth_GetTimeOutValue(LulTimeOutCountInit);
      LulRegVal = Eth_GaaETNBRegs[LulCtrlIdx]->ulCSR & ETH_ETNB_CSR_DTS;
    } while ((LulRegVal != ETH_ETNB_CSR_DTS) && (LulTimeoutCountElap <= ETH_TIMEOUT_COUNT));

    LucErrorValue = Eth_Hw_Etnb_OpModeChange(LulCtrlIdx, ETH_CONFIG);
  }
  else
  {
    /* No action required */
  }
  Eth_GaaETNBRegs[LulCtrlIdx]->ulCCC &= ~ETH_ETNB_CCC_DTSR;

  return LucErrorValue;
}

/***********************************************************************************************************************
** Function Name         : Eth_GetTimeOutValue
**
** Service ID            : NA
**
** Description           : This returns the elapsed timeout value of  Os timer
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : LusTimeOutCount_Init
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : LulTimeOutCount_Elapsed - TickType
**
** Preconditions         : None
**
** Global Variable(s)    : None
**
** Function(s) invoked   : GetCounterValue
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_080
***********************************************************************************************************************/
FUNC(TickType, ETH_PRIVATE_CODE) Eth_GetTimeOutValue(
  CONST(TickType, AUTOMATIC) LusTimeOutCount_Init)
{
  TickType LulTimeOutCount_Curr;
  TickType LulTimeOutCount_Elapsed;

  LulTimeOutCount_Curr = 0UL;

  /* Get the current tick value from OS Counter */
  (void)GetCounterValue((CounterType)ETH_OS_COUNTER_ID, &LulTimeOutCount_Curr);

  /* Check whether current time out exceeds inital time out value */
  if (LulTimeOutCount_Curr > LusTimeOutCount_Init)
  {
    /* Get elapsed time out count */
    LulTimeOutCount_Elapsed = LulTimeOutCount_Curr - LusTimeOutCount_Init;                                              /* PRQA S 3383 # JV-01 */
  }
  else
  {
    /* Get elapsed time out count */
    LulTimeOutCount_Elapsed = ((uint32)ETH_OS_COUNTER_MAX_VALUE - LusTimeOutCount_Init) + LulTimeOutCount_Curr;         /* PRQA S 3383 # JV-01 */
  }

  return (LulTimeOutCount_Elapsed);
}

#if (ETH_STREAM_FILTERING == STD_ON)
/***********************************************************************************************************************
** Function Name         : SetRxFilter
**
** Service ID            : NA
**
** Description           : Configure SFO, SFM, SFP
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Ethernet Controller index
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : void
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETNBRegs, Eth_GaaAvbConfig
**                         Eth_GpCtrlConfigPtr
**
** Function(s) invoked   : Eth_Hw_Etnb_WriteIntoSFPReg
**
** Registers Used        : SFO, SFM0, SFM1
**
** Reference ID          : ETH_DUD_ACT_081,
** Reference ID          : ETH_DUD_ACT_081_GBL001, ETH_DUD_ACT_081_GBL002
** Reference ID          : ETH_DUD_ACT_081_GBL003, ETH_DUD_ACT_081_GBL004
** Reference ID          : ETH_DUD_ACT_081_GBL005
***********************************************************************************************************************/
STATIC FUNC (void,  ETH_PRIVATE_CODE) SetRxFilter(
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  #if (ETH_STREAM_FILTERING == STD_ON)
  VAR(uint8, AUTOMATIC) LucNumberOfRxQueue;
  VAR(uint8, AUTOMATIC) LucCount;
  VAR(uint8, AUTOMATIC) LucQidx;
  #endif
  P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;

  LpHwUnitConfig = (P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA))                                              /* PRQA S 0316 # JV-01 */
    Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;
  Eth_GaaETNBRegs[LulCtrlIdx]->ulSFO = ETH_ETNB_SFO_FBP_VALUE;                                                          /* PRQA S 2844, 2814 # JV-01, JV-01 */
  if (ETH_ENABLE == Eth_GaaAvbConfig[LulCtrlIdx].stRxConfig.enSRPTalkerFiltering)                                       /* PRQA S 2844 # JV-01 */
  {
    Eth_GaaETNBRegs[LulCtrlIdx]->ulSFM[0] = ETH_ETNB_SFM0_TALKERFILTER;
    Eth_GaaETNBRegs[LulCtrlIdx]->ulSFM[1] = ETH_ETNB_SFM1_TALKERFILTER;
  }
  else
  {
    Eth_GaaETNBRegs[LulCtrlIdx]->ulSFM[0] = ETH_ETNB_SFM0_LISNERFILTER;
    Eth_GaaETNBRegs[LulCtrlIdx]->ulSFM[1] = ETH_ETNB_SFM1_LISNERFILTER;
  }

  #if (ETH_STREAM_FILTERING == STD_ON)
  /* Get number of configured queues */
  LucNumberOfRxQueue = LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue;                                                 /* PRQA S 2814 # JV-01 */
  for (LucCount = 0U; LucCount < LucNumberOfRxQueue; LucCount++)
  {
    LucQidx = LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LucCount].ucEthRxQueueId;                                    /* PRQA S 2824 # JV-01 */

    /* If Rx Queue is Stream set corresponding SPFI register */
    if (ETH_NCCHANNEL < LucQidx)
    {
      /* Write into SFP Register */
      Eth_Hw_Etnb_WriteIntoSFPReg(
        LulCtrlIdx, LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LucCount].aaEthPatternStream, LucQidx);
    }
    else
    {
      /* NO action as this is not a RX Stream Queue */
    }
  }
  #endif
}

/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_WriteIntoSFPReg
**
** Service ID            : NA
**
** Description           : Write In corresponding SFPi the passed address
**                         (for Rx Filter)
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LulCtrlIdx - Ethernet Controller index
**                       : LpMacAddrPtr - Pointer to the address to be stored
**                       : LucQueueIdx - In case of clear this is the index of
**                         the SFP register minus offset
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : AVB shall be into Config Mode
**
** Global Variable(s)    : Eth_GaaAvbConfig, Eth_GaaETNBRegs
**
** Function(s) invoked   : None
**
** Registers Used        : CCC, SFV, SFL, SFPi i = 0,1..31
**
** Reference ID          : ETH_DUD_ACT_082,
** Reference ID          : ETH_DUD_ACT_082_GBL001, ETH_DUD_ACT_082_GBL002
** Reference ID          : ETH_DUD_ACT_082_GBL003, ETH_DUD_ACT_082_GBL004
** Reference ID          : ETH_DUD_ACT_082_GBL005
***********************************************************************************************************************/
FUNC (void, ETH_PRIVATE_CODE) Eth_Hw_Etnb_WriteIntoSFPReg(
  CONST(uint32,AUTOMATIC) LulCtrlIdx, CONSTP2CONST(uint8, AUTOMATIC, ETH_APPL_DATA) LpMacAddrPtr,
  CONST(uint8,AUTOMATIC) LucQueueIdx)
{
  uint32 LulValHigh;
  uint32 LulValLow;
  uint32 LulVal;
  uint8 LucSFPidx;
  /************ Little Endian Core **************/
  LucSFPidx = LucQueueIdx - ETH_RX_QUEUE_INDEX_AVAILABLE_FILTER;                                                        /* PRQA S 3383 # JV-01 */
  /* Write into SFP1,3,...31 */
  if (ETH_ENABLE == Eth_GaaAvbConfig[LulCtrlIdx].stRxConfig.enSRPTalkerFiltering)                                       /* PRQA S 2844 # JV-01 */
  {
    LulValHigh = ((uint32)LpMacAddrPtr[5] << 8U) | ((uint32)LpMacAddrPtr[4]);                                           /* PRQA S 2824 # JV-01 */
  }
  else
  {
    LulValHigh = ((uint32)LpMacAddrPtr[7] << 24U) | ((uint32)LpMacAddrPtr[6] << 16U) |
      ((uint32)LpMacAddrPtr[5] << 8U) | ((uint32)LpMacAddrPtr[4]);
  }
  /* Write into SFP0,2,..30 */
  LulValLow = ((uint32)LpMacAddrPtr[3] << 24U) | ((uint32)LpMacAddrPtr[2] << 16U) |
    ((uint32)LpMacAddrPtr[1] << 8U) | ((uint32)LpMacAddrPtr[0]);
  /* Read Opc */
  if (ETH_OPERATION == Eth_GaaCtrlStat[LulCtrlIdx].stHwStat.enDevStatus)                                                /* PRQA S 2844 # JV-01 */
  {
    Eth_GaaETNBRegs[LulCtrlIdx]->ulSFV[0] = LulValLow;                                                                  /* PRQA S 2844, 2814 # JV-01, JV-01 */
    Eth_GaaETNBRegs[LulCtrlIdx]->ulSFV[1] = LulValHigh;
    /* Check it can be updated */

    do
    {
      LulVal = Eth_GaaETNBRegs[LulCtrlIdx]->ulSFL;
    } while (ETH_NO_LOAD_REQ != (LulVal & ETH_SFL_MASK));

    /* SFP update request */
    LulVal = (uint32)LucSFPidx;
    Eth_GaaETNBRegs[LulCtrlIdx]->ulSFL = LulVal;
  }
  else
  {
    Eth_GaaETNBRegs[LulCtrlIdx]->ulSFP[ETH_ETNB_GET_LOW_SFPi(LucSFPidx)] = LulValLow;                                   /* PRQA S 3469, 3383 # JV-01, JV-01 */
    Eth_GaaETNBRegs[LulCtrlIdx]->ulSFP[ETH_ETNB_GET_HIGH_SFPi(LucSFPidx)] = LulValHigh;                                 /* PRQA S 3469, 3383 # JV-01, JV-01 */
  }

}
#endif /* (ETH_STREAM_FILTERING == STD_ON) */

#if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_WaitPTPRequestIsComplete
**
** Service ID            : NA
**
** Description           : Service Api to wait a Gptp request is complete
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant for the same Ctrl ,
**                         Re-entrant for different
**
** Input Parameters      : LulCtrlIdx   Index of the controller within the
**                         context of the Ethernet Driver
**                       : LucBit Which Bit to poll in GCCR Register
**
** InOut Parameters      : None
**
** Output Parameters     : E_OK   gPTP Request is completed w/o timeout
**                       : E_NOT_OK gPTP Request not completed due to timeout
**
** Return parameter      : LucReturnValue E_OK / E_NOT_OK
**
** Preconditions         : Component Requires previous controller
**                         initialization using Eth_ControllerInit..
**
** Global Variable(s)    : Eth_GaaETNBRegs
**
** Function(s) invoked   : GetCounterValue, Eth_GetTimeOutValue
**
** Registers Used        : GCCR
**
** Reference ID          : ETH_DUD_ACT_083
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_Hw_Etnb_WaitPTPRequestIsComplete(                                            /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32,AUTOMATIC) LulMask)
{
  Std_ReturnType LucReturnValue;
  uint32 LulRegValue;
  TickType LulTimeOutCountInit;
  TickType LulTimeoutCountElap;

  LulTimeOutCountInit = 0UL;

  (void)GetCounterValue((CounterType)ETH_OS_COUNTER_ID, &LulTimeOutCountInit);
  do
  {
    LulTimeoutCountElap = Eth_GetTimeOutValue(LulTimeOutCountInit);
    LulRegValue = Eth_GaaETNBRegs[LulCtrlIdx]->ulGCCR & LulMask;                                                        /* PRQA S 2844, 2814 # JV-01, JV-01 */
  } while ((LulRegValue != 0UL) && (LulTimeoutCountElap <= ETH_TIMEOUT_COUNT));

  if (ETH_TIMEOUT_COUNT < LulTimeoutCountElap)
  {
    /* Timeout Occurred */
    LucReturnValue = E_NOT_OK;
  }
  else
  {
    /* Success */
    LucReturnValue = E_OK;
  }
 return (LucReturnValue);
}

#endif /* ETH_GLOBAL_TIME_SUPPORT */

#if (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON)
/***********************************************************************************************************************
** Function Name         : IsRxFrameValid
**
** Service ID            : NA
**
** Description           : Compare Mac Address Frame just received with
**                         Broadcast and Controller Mac Address.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LpRxFrame. Pointer to the Frame just received
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : LblPass
**                         True : Frame DA is Broadcast or
**                         matches with the Source Address of this Controller
**                         False : Otherwise
**
** Preconditions         : None
**
** Global Variables Used : Eth_GaaCtrlStat, Eth_GaaCtrlFilterAddr,
**                         Eth_GaaAddressFilters
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_085
***********************************************************************************************************************/
STATIC FUNC(boolean, ETH_PRIVATE_CODE) IsRxFrameValid(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONSTP2CONST(Eth_RxSingleFrameType, AUTOMATIC, ETH_APPL_DATA) LpRxFrame)
{
  Eth_MacAddressType LstDstAddress;
  boolean LblPass;
  uint32 LulFilterIdx;
  uint32 LulRemainBits;

  LblPass = ETH_FALSE;
  ETH_PACK_ADDRESS_FROM_8(((P2VAR(uint8, AUTOMATIC, ETH_APPL_DATA))LpRxFrame->ulFrameAddr), LstDstAddress);             /* PRQA S 3432, 2824, 3469, 2814, 0306 # JV-01, JV-01, JV-01, JV-01, JV-01 */

  /* Check whether promiscuous mode, broadcast, own unicast, multicast address is not registed */
  if ((ETH_TRUE == Eth_GaaCtrlStat[LulCtrlIdx].blPromiscuous) ||                                                        /* PRQA S 2844 # JV-01 */
      (0UL == ETH_COMPARE_MAC(LstDstAddress, Eth_GstBroadcastAddr)) ||                                                  /* PRQA S 3469 # JV-01 */
      (0UL == ETH_COMPARE_MAC(LstDstAddress, Eth_GaaCtrlStat[LulCtrlIdx].stMacAddr)))                                   /* PRQA S 3469 # JV-01 */
  {
    LblPass = ETH_TRUE;
  }
  else
  {
    /* Otherwise, do the filter operation */

    /* To skip empty filters efficiently, copy active bits to the local var */
    LulRemainBits = Eth_GaaCtrlStat[LulCtrlIdx].ulActiveFilterBits;
    LulFilterIdx = 0UL;
    /* Loop until any filter hits or pass all active filters */
    while (0UL != LulRemainBits)
    {
      if ((0UL != (LulRemainBits & (1UL << LulFilterIdx))) &&
          (0UL == ETH_COMPARE_MAC(LstDstAddress, Eth_GaaAddressFilters[LulCtrlIdx][LulFilterIdx])))                     /* PRQA S 2844, 3469 # JV-01, JV-01 */
      {
        LblPass = ETH_TRUE;
        break;
      }
      else
      {
        /* No action required */
      }
      /* Clear the compared filter bit */
      LulRemainBits &= ~(1UL << LulFilterIdx);
      LulFilterIdx++;                                                                                                   /* PRQA S 3383 # JV-01 */
    }
  }

  return (LblPass);
}
#endif  /* (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON) */

#if (ETH_QOS_SUPPORT == STD_ON)
/***********************************************************************************************************************
** Function Name         : SetCbsParameter
**
** Service ID            : NA
**
** Description           : Set Tx queues configured
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LulCtrlIdx - Index of the controller within the
**                         context of the Ethernet Driver
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaETNBRegs, Eth_GpCtrlConfigPtr
**
** Function(s) invoked   : None
**
** Registers Used        : CIVR0..1, CDVR0..1, CUL0..1, CLL..10
**
** Reference ID          : ETH_DUD_ACT_086,
** Reference ID          : ETH_DUD_ACT_086_GBL001, ETH_DUD_ACT_086_GBL002
** Reference ID          : ETH_DUD_ACT_086_GBL003, ETH_DUD_ACT_086_GBL004
** Reference ID          : ETH_DUD_ACT_086_GBL005, ETH_DUD_ACT_086_GBL006
** Reference ID          : ETH_DUD_ACT_086_GBL007, ETH_DUD_ACT_086_GBL008
***********************************************************************************************************************/
STATIC FUNC(void, ETH_PRIVATE_CODE) SetCbsParameter(
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  uint32 LulQidx;

  uint32 LulCIV_tmp;
  uint32 LulCDV_tmp; /* negative number represented in 2'complement */
  P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;

  LpHwUnitConfig = (P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA))                                              /* PRQA S 0316 # JV-01 */
    Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;
  /* loop through all Tx Queues configured */
  LulQidx = 0UL;
  do
  {
    if (((uint8)(ETH_TX_SB) == LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulQidx].ucEthTxQueueId) &&                 /* PRQA S 2814, 2824 # JV-01, JV-01 */
      (ETH_CBS == LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulQidx].stTxQueueShaper.enTxQPolicy))
    {
      /* Q2 */
      LulCIV_tmp = (uint32)(LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulQidx].stTxQueueShaper.ulCiv);

      LulCDV_tmp = (uint32)(LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulQidx].stTxQueueShaper.slCdv);
      /* Write into CIVR0 , CDVR0 */
      Eth_GaaETNBRegs[LulCtrlIdx]->ulCIVR[0] = LulCIV_tmp;                                                              /* PRQA S 2844, 2814 # JV-01, JV-01 */
      Eth_GaaETNBRegs[LulCtrlIdx]->ulCDVR[0] = LulCDV_tmp;
      /* Set Upper Limit => 65535 */
      Eth_GaaETNBRegs[LulCtrlIdx]->ulCUL[0] = ETH_MAX_CIV;
      /* Set Lower Limit => -65536 */
      Eth_GaaETNBRegs[LulCtrlIdx]->ulCLL[0] = ETH_MIN_CDV;
    }
    else if (((uint8)(ETH_TX_SA) == LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulQidx].ucEthTxQueueId) &&
      (ETH_CBS == LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulQidx].stTxQueueShaper.enTxQPolicy))
    {
      /* Q3 */
      LulCIV_tmp = (uint32)(LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulQidx].stTxQueueShaper.ulCiv);

      LulCDV_tmp = (uint32)(LpHwUnitConfig->stQueueConfig.pTxQueueConfig[LulQidx].stTxQueueShaper.slCdv);

      /* Write into CIVR1 , CDVR1 */
      Eth_GaaETNBRegs[LulCtrlIdx]->ulCIVR[1] = LulCIV_tmp;
      Eth_GaaETNBRegs[LulCtrlIdx]->ulCDVR[1] = LulCDV_tmp;
      /* Set Upper Limit => 65535 */
      Eth_GaaETNBRegs[LulCtrlIdx]->ulCUL[1] = ETH_MAX_CIV;
      /* Set Lower Limit => -65536 */
      Eth_GaaETNBRegs[LulCtrlIdx]->ulCLL[1] = ETH_MIN_CDV;
    }
    else
    {
      /* Nothing to set for this Tx Queue */
    }

    LulQidx++;                                                                                                          /* PRQA S 3383 # JV-01 */
  } while (LulQidx < (uint32)LpHwUnitConfig->stQueueConfig.ucNumberOfTxQueue);
}
#endif

#if (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON)
/***********************************************************************************************************************
** Function Name         : Eth_Hw_Etnb_RxIrqHdlr
**
** Service ID            : NA
**
** Description           : This API Handles frame reception interrupts
**                         of the indexed controller
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : Component Requires previous controller
**                         initialization using Eth_ControllerInit..
**
** Global Variables Used : Eth_GpCtrlConfigPtr, Eth_GaaETNBRegs
**
** Function(s) invoked   : Eth_Hw_Etnb_UFCounterGet,
**                         Eth_Hw_Etnb_RxQueueProcess
**
** Registers Used        : RIS0
**
** Reference ID          : ETH_DUD_ACT_087
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_Hw_Etnb_RxIrqHdlr(                                                                     /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  uint32 LulRIS0;
  uint32 LulIdx;
  uint8 LucQidx;
  uint32 LulCnt;
  uint32 LulUnreadFrameCnt;
  P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA) LpHwUnitConfig;
  LpHwUnitConfig = (P2CONST(Eth_ETNBConfigType, AUTOMATIC, ETH_APPL_DATA))                                              /* PRQA S 0316 # JV-01 */
    Eth_GpCtrlConfigPtr[LulCtrlIdx].pHwUnitConfig;
  /* Initialize the counter */
  LulIdx = 0UL;
  /* Get receive interrupt status */
  LulRIS0 = Eth_GaaETNBRegs[LulCtrlIdx]->ulRIS0 & ETH_RX_INT_MASK;                                                      /* PRQA S 2844, 2814 # JV-01, JV-01 */

  while (0UL != LulRIS0)
  {
    LucQidx = LpHwUnitConfig->stQueueConfig.pRxQueueConfig[LulIdx].ucEthRxQueueId;                                      /* PRQA S 2814, 2824 # JV-01, JV-01 */
    if (ETH_BIT1_SET_32 == ((LulRIS0 >> LucQidx) & ETH_BIT1_SET_32))
    {
      LulUnreadFrameCnt = Eth_Hw_Etnb_UFCounterGet(LulCtrlIdx, (uint32)LucQidx);

      /* Queue that has received some frame/s */
      for (LulCnt = 0UL; LulCnt < LulUnreadFrameCnt; LulCnt++)
      {
        /* Rx frame data processing */
        (void)Eth_Hw_Etnb_RxQueueProcess(LulCtrlIdx, LucQidx);
      }
    }
    else
    {
      /* No action required */
    }

    if (LulIdx < (uint32)(LpHwUnitConfig->stQueueConfig.ucNumberOfRxQueue - 1U))                                        /* PRQA S 4391, 3383 # JV-01, JV-01 */
    {
      /* Next configured queue */
      LulIdx++;
    }
    else
    {
      /* Clear the flag to exit loop */
      LulRIS0 = 0UL;
    }
  }
}
#endif /* (ETH_CTRL_ENABLE_RX_INTERRUPT == STD_ON) */

#define ETH_STOP_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"
/***********************************************************************************************************************
**                                               End of File                                                          **
***********************************************************************************************************************/
