/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Eth_ETNC_LLDriver.h                                                                                 */
/* SW-VERSION   = 1.6.0                                                                                               */
/*====================================================================================================================*/
/*                                                     COPYRIGHT                                                      */
/*====================================================================================================================*/
/* (c) 2020,2021 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* This file contains ETNC specific definitions of Eth Driver Component.                                              */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s) and/or */
/* the Application.                                                                                                   */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs of  */
/* program errors, compliance with applicable laws, damage to or loss of data, programs or equipment, and             */
/* unavailability or interruption of operations.                                                                      */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*              Devices:        X2x                                                                                   */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                              Revision Control History                                              **
***********************************************************************************************************************/
/*
 * 1.4.2: 24/08/2021    : Modify the format to 120 characters
 * 1.4.1: 07/05/2021    : Remove vendor ID from the file name, API names
 *                        and parameters according to BSW00347 requirement
 *        06/07/2021    : Updated QA-C 9.5.0 comments.
 * 1.2.0: 13/07/2020    : Release
 *        29/07/2020    : Add QAC 9.3.1 comment.
 * 1.1.0: 19/06/2020    : Release
 * 1.0.1: 04/06/2020    : Removed unnecessary code.
 *                        Updated static analysis result.
 * 1.0.0: 25/03/2020    : Initial Version
 */
/**********************************************************************************************************************/
#ifndef ETH_ETNC_LLDRIVER_H
#define ETH_ETNC_LLDRIVER_H

#include "Eth_Types.h"

/***********************************************************************************************************************
**                                                Version Information                                                 **
***********************************************************************************************************************/
/* AUTOSAR release version information */
#define ETH_ETNC_AR_RELEASE_MAJOR_VERSION    ETH_AR_RELEASE_MAJOR_VERSION
#define ETH_ETNC_AR_RELEASE_MINOR_VERSION    ETH_AR_RELEASE_MINOR_VERSION
#define ETH_ETNC_AR_RELEASE_REVISION_VERSION ETH_AR_RELEASE_REVISION_VERSION

/* Module Software version information */
#define ETH_ETNC_SW_MAJOR_VERSION            ETH_SW_MAJOR_VERSION
#define ETH_ETNC_SW_MINOR_VERSION            ETH_SW_MINOR_VERSION

/***********************************************************************************************************************
**                            HW specific parameters referred from the upper-level driver                             **
***********************************************************************************************************************/
/* Insert paddings to the top of buffers to align payload location as 4 byte */
/* Paddings before header */
#define ETH_TXRX_BUFFER_PRE_PADDING          2UL
/* Paddings between header and payload */
#define ETH_TXRX_BUFFER_IN_PADDING           0UL

/* Required byte alignment for the top of Tx buffers */
#define ETH_TX_BUFFER_ALIGNMENT              4UL
/* Required byte alignment for the top of Rx buffers */
#define ETH_RX_BUFFER_ALIGNMENT              32UL

/* Minimum payload length allowed by the HW */
#define ETH_MIN_PAYLOAD_SIZE                 19UL

/* Maximum number of tx descriptors */
#define ETH_TX_DESC_MAX                      ETH_TX_BUF_TOTAL_0

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (2:3630)    : The implementation of this struct/union type should be hidden.                               */
/* Rule                : MISRA C:2012 Dir-4.8                                                                         */
/* JV-01 Justification : The pointers point to this struct are dereferenced in other .c files, so this                */
/*                       implementation should not be hidden                                                          */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3472)    : All toplevel uses of this function-like macro look like they could be replaced by            */
/*                       equivalent function calls.                                                                   */
/* Rule                : MISRA C:2012 Dir-4.9                                                                         */
/* JV-01 Justification : To prevent overhead and speed up the processing, there is no need to use function call       */
/*                       for such a small operation.                                                                  */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                 Global Data Types                                                  **
***********************************************************************************************************************/
/* For polling operation on Rx side:
  EDMAC updates a Rx descriptor in the order: RD0 -> RD1.
  Thus, if CPU reads RD1.RFL right after RD0.RACT is cleared by EDMAC,
  CPU can get the old value of RD1.RFL incorrectly.
  To prevent this issue, set 0xFFFF to RD1.RFL of empty descriptors,
  and CPU polls whether RD0.RACT == 0 && RD1.RFL != 0xFFFF. */
#define ETH_ETNC_RFL_WORKAROUND_VALUE        0x0000FFFFUL

/* ETNC LLDriver specific status variables */
typedef struct STag_Eth_HwStatusType
{
  /* Index of tx descriptor which should be confirmed next */
  uint8 ucTxDescIdxConfirmed;
  /* Index of tx descriptor which should be transmitted next */
  uint8 ucTxDescIdxRequested;
  /* Index of tx descriptor which should be written next */
  uint8 ucTxDescIdxQueued;
  /* Index of rx descriptor which should be indicated next */
  uint8 ucRxDescIdxHead;
  /* HW is BUSY */
  boolean blTxOngoing;
  /* Previous value of RMFCR (frame lost counter) */
  uint32 ulPrevRMFCRVal;
#if (ETH_REGISTER_CHECK_RUNTIME == STD_ON)
  /* Backup of the value of ECMR register, used for the verification */
  uint32 ulECMRCopy;
  /* Whether register error is detected by the ISR or the MainFunction */
  boolean blErrDetected;
  /* Whether the DEM error of register corruption is already reported */
  boolean blErrReported;
#endif
} Eth_HwStatusType;

/* Wait cycles after a software reset is issued.
   It takes 64 cycles of the peripheral
   module clock (PCLKA) to initialize the ETHERC and EDMAC. Complete the
   initialization process before accessing registers in the ETHERC and EDMAC.
   PCLKA = 100MHz = Over 640ns
   6 = about 1315ns at 400MHz (640+margin)
   This value is acquired by measuring actual loop time with TSCOUNT register
*/
#define ETH_ETNC_CYCLES_AFTER_RESET 6UL

/* Contains data to be used for register read/write check */
typedef struct STag_Eth_ETNCCheckRWRegType
{
  uint32 ulOffset;
  uint32 ulMaskValue;
  uint32 ulWritePattern0;
  uint32 ulWritePattern1;
} Eth_ETNCCheckRWRegType;

/* Contains data to be used for register read check */
typedef struct STag_Eth_ETNCCheckRORegType
{
  uint32 ulOffset;
  uint32 ulMaskValue;
  uint32 ulExpectedValue;
} Eth_ETNCCheckRORegType;

/* Tx descriptor, its size must be 16 bytes */
typedef struct STag_Eth_ETNCTxDescriptorType                                                                            /* PRQA S 3630 # JV-01 */
{
  uint32 ulTD0;
  uint32 ulTD1;
  P2CONST(volatile uint8, TYPEDEF, TYPEDEF) pTD2;
  /* The following members are for software use */
  uint8 ucAttributes;
  uint8 ucBufIdx;
  uint8 aaPadding[2];

} Eth_ETNCTxDescriptorType;

/* TD0 */
#define ETH_ETNC_TACT              0x80000000UL
#define ETH_ETNC_TDLE              0x40000000UL
#define ETH_ETNC_TFP_ALL           0x30000000UL
#define ETH_ETNC_TWBI              0x04000000UL

/* TD1 */
#define ETH_ETNC_TBL(length)       ((uint32)(length) << 16UL)                                                           /* PRQA S 3472 # JV-01 */

/* Attributes */
#define ETH_ETNC_ATTR_CONFIRMATION (uint8)0x01U

/* Rx descriptor, its size must be 16 bytes */
typedef struct STag_Eth_ETNCRxDescriptorType                                                                            /* PRQA S 3630 # JV-01 */
{
  uint32 ulRD0;
  uint32 ulRD1;
  P2VAR(volatile uint8, TYPEDEF, TYPEDEF) pRD2;
  /* The following members are for software use */
  uint32 ulPadding;
} Eth_ETNCRxDescriptorType;

/* RD0 */
#define ETH_ETNC_RACT              0x80000000UL
#define ETH_ETNC_RDLE              0x40000000UL
#define ETH_ETNC_RFP1              0x20000000UL
#define ETH_ETNC_RFP0              0x10000000UL
#define ETH_ETNC_RFS_MASK          0x07FFFFFFUL
#define ETH_ETNC_RFS9_RFOF         0x00000200UL
#define ETH_ETNC_RFS8_RABT         0x00000100UL
#define ETH_ETNC_RFS7_RMAF         0x00000080UL
#define ETH_ETNC_RFS3_RTLF         0x00000008UL
#define ETH_ETNC_RFS2_RTSF         0x00000004UL
#define ETH_ETNC_RFS1_PRE          0x00000002UL
#define ETH_ETNC_RFS0_CERF         0x00000001UL
/* RD1 */
#define ETH_ETNC_RBL(length)       ((uint32)(length) << 16UL)                                                           /* PRQA S 3472 # JV-01 */
#define ETH_ETNC_RFL_GET(val)      ((uint32)(val)&0xFFFFUL)                                                             /* PRQA S 3472 # JV-01 */

/* ETNC register definitions */
typedef struct STag_Eth_ETNCRegType                                                                                     /* PRQA S 3630 # JV-01 */
{
  uint32 ulEDMR;                             /* +0000 EDMR      */
  uint32 ulReserved0;                        /* Reserved        */
  uint32 ulEDTRR;                            /* +0008 EDTRR     */
  uint32 ulReserved1;                        /* Reserved        */
  uint32 ulEDRRR;                            /* +0010 EDRRR     */
  uint32 ulReserved2;                        /* Reserved        */
  volatile Eth_ETNCTxDescriptorType *pTDLAR; /* +0018 TDLAR     */
  uint32 ulReserved3;                        /* Reserved        */
  volatile Eth_ETNCRxDescriptorType *pRDLAR; /* +0020 RDLAR     */
  uint32 ulReserved4;                        /* Reserved        */
  uint32 ulEESR;                             /* +0028 EESR      */
  uint32 ulReserved5;                        /* Reserved        */
  uint32 ulEESIPR;                           /* +0030 EESIPR    */
  uint32 ulReserved6;                        /* Reserved        */
  uint32 ulTRSCER;                           /* +0038 TRSCER    */
  uint32 ulReserved7;                        /* Reserved        */
  uint32 ulRMFCR;                            /* +0040 RMFCR     */
  uint32 ulReserved8;                        /* Reserved        */
  uint32 ulTFTR;                             /* +0048 TFTR      */
  uint32 ulReserved9;                        /* Reserved        */
  uint32 ulFDR;                              /* +0050 FDR       */
  uint32 ulReserved10;                       /* Reserved        */
  uint32 ulRMCR;                             /* +0058 RMCR      */
  uint32 aaReserved11[2];                    /* Reserved        */
  uint32 ulTFUCR;                            /* +0064 TFUCR     */
  uint32 ulRFOCR;                            /* +0068 RFOCR     */
  uint32 ulIOSR;                             /* +006C IOSR      */
  uint32 ulFCFTR;                            /* +0070 FCFTR     */
  uint32 ulReserved12;                       /* Reserved        */
  uint32 ulRPADIR;                           /* +0078 RPADIR    */
  uint32 ulTRIMD;                            /* +007C TRIMD     */
  uint32 aaReserved13[18];                   /* Reserved        */
  uint32 ulRBWAR;                            /* +00C8 RBWAR     */
  uint32 ulRDFAR;                            /* +00CC RDFAR     */
  uint32 ulReserved14;                       /* Reserved        */
  uint32 ulTBRAR;                            /* +00D4 TBRAR     */
  uint32 ulTDFAR;                            /* +00D8 TDFAR     */
  uint32 aaReserved14[9];                    /* Reserved        */
  uint32 ulECMR;                             /* +0100 ECMR      */
  uint32 ulReserved15;                       /* Reserved        */
  uint32 ulRFLR;                             /* +0108 RFLR      */
  uint32 ulReserved16;                       /* Reserved        */
  uint32 ulECSR;                             /* +0110 ECSR      */
  uint32 ulReserved17;                       /* Reserved        */
  uint32 ulECSIPR;                           /* +0118 ECSIPR    */
  uint32 ulReserved18;                       /* Reserved        */
  uint32 ulPIR;                              /* +0120 PIR       */
  uint32 ulReserved19;                       /* Reserved        */
  uint32 ulPSR;                              /* +0128 PSR       */
  uint32 aaReserved20[5];                    /* Reserved        */
  uint32 ulRDMLR;                            /* +0140 RDMLR     */
  uint32 aaReserved21[3];                    /* Reserved        */
  uint32 ulIPGR;                             /* +0150 IPGR      */
  uint32 ulAPR;                              /* +0154 APR       */
  uint32 ulMPR;                              /* +0158 MPR       */
  uint32 ulReserved22;                       /* Reserved        */
  uint32 ulRFCF;                             /* +0160 RFCF      */
  uint32 ulTPAUSER;                          /* +0164 TPAUSER   */
  uint32 ulTPAUSECR;                         /* +0168 TPAUSECR  */
  uint32 ulBCFRR;                            /* +016C BCFRR     */
  uint32 aaReserved23[20];                   /* Reserved        */
  uint32 ulMAHR;                             /* +01C0 MAHR      */
  uint32 ulReserved24;                       /* Reserved        */
  uint32 ulMALR;                             /* +01C8 MALR      */
  uint32 ulReserved25;                       /* Reserved        */
  uint32 ulTROCR;                            /* +01D0 TROCR     */
  uint32 ulCDCR;                             /* +01D4 CDCR      */
  uint32 ulLCCR;                             /* +01D8 LCCR      */
  uint32 ulCNDCR;                            /* +01DC CNDCR     */
  uint32 ulReserved26;                       /* Reserved        */
  uint32 ulCEFCR;                            /* +01E4 CEFCR     */
  uint32 ulFRECR;                            /* +01E8 FRECR     */
  uint32 ulTSFRCR;                           /* +01EC TSFRCR    */
  uint32 ulTLFRCR;                           /* +01F0 TLFRCR    */
  uint32 ulRFCR;                             /* +01F4 RFCR      */
  uint32 ulMAFCR;                            /* +01F8 MAFCR     */
} Eth_ETNCRegType;

/* ETNCnECMR */
#define ETH_ETNC_RE                 0x00000040UL
#define ETH_ETNC_TE                 0x00000020UL
#define ETH_ETNC_ILB                0x00000008UL
#define ETH_ETNC_RTM                0x00000004UL
#define ETH_ETNC_DM                 0x00000002UL
#define ETH_ETNC_PRM                0x00000001UL

/* ETNCnPIR */
#define ETH_ETNC_GET_MDI(regval)    (((uint32)(regval) >> 3UL) & 1UL)                                                   /* PRQA S 3472 # JV-01 */
#define ETH_ETNC_MDO                0x00000004UL
#define ETH_ETNC_MMD                0x00000002UL
#define ETH_ETNC_MDC                0x00000001UL

/* ETNCnIPGR */
#define ETH_ETNC_IPGR_DEFAULT       0x00000014UL

/* ETNCnEDMR */
#define ETH_ETNC_DE_LITTLE          0x00000040UL
#define ETH_ETNC_DL_16              0x00000000UL
#define ETH_ETNC_SWR                0x00000001UL

/* ETNCnEDTRR */
#define ETH_ETNC_TR                 0x00000001UL

/* ETNCnEDRRR */
#define ETH_ETNC_RR                 0x00000001UL

/* ETNCnEESIPR/ETNCnEEIPR */
#define ETH_ETNC_TWB                0x40000000UL
#define ETH_ETNC_RFCOF              0x01000000UL
#define ETH_ETNC_ADE                0x00800000UL
#define ETH_ETNC_FR                 0x00040000UL
#define ETH_ETNC_CD                 0x00000200UL
#define ETH_ETNC_RRF                0x00000010UL
#define ETH_ETNC_RTLF               0x00000008UL
#define ETH_ETNC_RTSF               0x00000004UL
#define ETH_ETNC_CERF               0x00000001UL

/* ETNCnFDR */
#define ETH_ETNC_FDR_DEFAULT        0x00000707UL

/* ETNCnRMCR */
#define ETH_ETNC_RNR                0x00000001UL

/* ETNCnRPADIR */
#define ETH_ETNC_RPADIR_4BYTE_ALIGN 0x00020000UL

/* ETNCnTRIMD */
#define ETH_ETNC_TIM                0x00000010UL
#define ETH_ETNC_TIS                0x00000001UL

/* Offset addresses for registers */
#define ETH_ETNC_OFFSET_EDMR        0x00000000UL
#define ETH_ETNC_OFFSET_EDTRR       0x00000008UL
#define ETH_ETNC_OFFSET_EDRRR       0x00000010UL
#define ETH_ETNC_OFFSET_TDLAR       0x00000018UL
#define ETH_ETNC_OFFSET_RDLAR       0x00000020UL
#define ETH_ETNC_OFFSET_EESR        0x00000028UL
#define ETH_ETNC_OFFSET_EESIPR      0x00000030UL
#define ETH_ETNC_OFFSET_TRSCER      0x00000038UL
#define ETH_ETNC_OFFSET_RMFCR       0x00000040UL
#define ETH_ETNC_OFFSET_TFTR        0x00000048UL
#define ETH_ETNC_OFFSET_FDR         0x00000050UL
#define ETH_ETNC_OFFSET_RMCR        0x00000058UL
#define ETH_ETNC_OFFSET_TFUCR       0x00000064UL
#define ETH_ETNC_OFFSET_RFOCR       0x00000068UL
#define ETH_ETNC_OFFSET_FCFTR       0x00000070UL
#define ETH_ETNC_OFFSET_RPADIR      0x00000078UL
#define ETH_ETNC_OFFSET_TRIMD       0x0000007CUL
#define ETH_ETNC_OFFSET_RBWAR       0x000000C8UL
#define ETH_ETNC_OFFSET_RDFAR       0x000000CCUL
#define ETH_ETNC_OFFSET_TBRAR       0x000000D4UL
#define ETH_ETNC_OFFSET_TDFAR       0x000000D8UL
#define ETH_ETNC_OFFSET_ECMR        0x00000100UL
#define ETH_ETNC_OFFSET_RFLR        0x00000108UL
#define ETH_ETNC_OFFSET_ECSR        0x00000110UL
#define ETH_ETNC_OFFSET_ECSIPR      0x00000118UL
#define ETH_ETNC_OFFSET_RDMLR       0x00000140UL
#define ETH_ETNC_OFFSET_IPGR        0x00000150UL
#define ETH_ETNC_OFFSET_APR         0x00000154UL
#define ETH_ETNC_OFFSET_RFCF        0x00000160UL
#define ETH_ETNC_OFFSET_TPAUSER     0x00000164UL
#define ETH_ETNC_OFFSET_TPAUSECR    0x00000168UL
#define ETH_ETNC_OFFSET_BCFRR       0x0000016CUL
#define ETH_ETNC_OFFSET_MAHR        0x000001C0UL
#define ETH_ETNC_OFFSET_MALR        0x000001C8UL
#define ETH_ETNC_OFFSET_TROCR       0x000001D0UL
#define ETH_ETNC_OFFSET_CDCR        0x000001D4UL
#define ETH_ETNC_OFFSET_LCCR        0x000001D8UL
#define ETH_ETNC_OFFSET_CNDCR       0x000001DCUL
#define ETH_ETNC_OFFSET_CEFCR       0x000001E4UL
#define ETH_ETNC_OFFSET_FRECR       0x000001E8UL
#define ETH_ETNC_OFFSET_TSFRCR      0x000001ECUL
#define ETH_ETNC_OFFSET_TLFRCR      0x000001F0UL
#define ETH_ETNC_OFFSET_RFCR        0x000001F4UL
#define ETH_ETNC_OFFSET_MAFCR       0x000001F8UL

/* Mask patterns to be used for the register corruption checking */
#define ETH_ETNC_ECMR_CHECK_MASK    0x001F116FUL
#define ETH_ETNC_TD0_CHECK_MASK     0xF0000000UL
#define ETH_ETNC_RD0_CHECK_MASK     0xF0000000UL

/***********************************************************************************************************************
**                                                   Global Symbols                                                   **
***********************************************************************************************************************/
#define ETH_START_SEC_CONST_UNSPECIFIED
#include "Eth_MemMap.h"

/* ETNC Registers per channel */
extern CONSTP2VAR(volatile Eth_ETNCRegType, ETH_CONST, REGSPACE) Eth_GaaETNCRegs[ETH_MAX_CTRLS_SUPPORTED];

/* EIC registers per channel */
extern CONSTP2VAR(volatile uint16, ETH_CONST, REGSPACE) Eth_GaaETNCEICRegs[ETH_MAX_CTRLS_SUPPORTED];

#define ETH_STOP_SEC_CONST_UNSPECIFIED
#include "Eth_MemMap.h"

#endif /* !ETH_ETNC_LLDRIVER_H_ */
/***********************************************************************************************************************
**                                                    End of File                                                     **
***********************************************************************************************************************/
