/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Eth_ETNC_Ram.c                                                                                      */
/* SW-VERSION   = 1.6.0                                                                                               */
/*====================================================================================================================*/
/*                                                   COPYRIGHT                                                        */
/*====================================================================================================================*/
/* (c) 2020-2022 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* This file contains global variables and buffer management functions for                                            */
/* Eth Driver Component.                                                                                              */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s)        */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs     */
/* of program errors, compliance with applicable laws, damage to or loss of data, programs or equipment,              */
/* and unavailability or interruption of operations.                                                                  */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*                                      Devices:        X2x                                                           */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                            Revision Control History                                                **
***********************************************************************************************************************/
/*
 * 1.5.2: 07/02/2022    : Update QA-C comment.
 *        25/01/2022    : Changed the prefixes for the TxBufHead and TxBufTail variables.
 * 1.4.2: 24/08/2021    : Modify the format to 120 characters
 * 1.4.1: 07/05/2021    : Remove vendor ID from the file name, API names
 *                        and parameters according to BSW00347 requirement
 *        09/06/2021    : Add QAC 9.5.0 comment.
 *        06/07/2021    : Updated QA-C 9.5.0 comments.
 * 1.2.0: 13/07/2020    : Release
 *        29/07/2020    : Add QAC 9.3.1 comment.
 * 1.1.0: 19/06/2020    : Release
 * 1.0.1: 04/06/2020    : To support Transmit/Receive interrupts for each controller.
 *                        Fixed the cast of invoke parameters
 *                        of the EthIf_RxIndication.
 *                        Commented on a cast from a large type to a small type.
 *                        Removed unnecessary code.
 *                        Reviewed the memory section.
 *                        Updated static analysis result.
 *                        Removed #if (ETH_59_AR_VERSION >= ETH_59_AR_431_VERSION) in Eth_59_GetTxBuffer
 *                        Changed include file structure.
 * 1.0.0: 25/03/2020    : Initial Version
 */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                        Include Section                                                             **
***********************************************************************************************************************/
#include "Eth.h"
#include "Eth_ETNC_Ram.h"
#include "EthIf_Cbk.h"
#if (ETH_CRITICAL_SECTION_PROTECTION == STD_ON)
/* Included for the declaration of the critical section protection functions */
#include "SchM_Eth.h"
#endif
#include "Dem.h"


/***********************************************************************************************************************
**                                        Version Information                                                         **
***********************************************************************************************************************/
/* AUTOSAR release version information */
#define ETH_ETNC_RAM_C_AR_RELEASE_MAJOR_VERSION \
                                      ETH_AR_RELEASE_MAJOR_VERSION_VALUE
#define ETH_ETNC_RAM_C_AR_RELEASE_MINOR_VERSION \
                                      ETH_AR_RELEASE_MINOR_VERSION_VALUE
#define ETH_ETNC_RAM_C_AR_RELEASE_REVISION_VERSION \
                                   ETH_AR_RELEASE_REVISION_VERSION_VALUE

/* File version information */
#define ETH_ETNC_RAM_C_SW_MAJOR_VERSION         ETH_SW_MAJOR_VERSION_VALUE
#define ETH_ETNC_RAM_C_SW_MINOR_VERSION         ETH_SW_MINOR_VERSION_VALUE

/***********************************************************************************************************************
**                                       Version Check                                                                **
***********************************************************************************************************************/
#if (ETH_ETNC_RAM_AR_RELEASE_MAJOR_VERSION != \
                                          ETH_ETNC_RAM_C_AR_RELEASE_MAJOR_VERSION)
#error "Eth_ETNC_Ram.c : Mismatch in Release Major Version"
#endif
#if (ETH_ETNC_RAM_AR_RELEASE_MINOR_VERSION != \
                                          ETH_ETNC_RAM_C_AR_RELEASE_MINOR_VERSION)
#error "Eth_ETNC_Ram.c : Mismatch in Release Minor Version"
#endif
#if (ETH_ETNC_RAM_AR_RELEASE_REVISION_VERSION != \
                                       ETH_ETNC_RAM_C_AR_RELEASE_REVISION_VERSION)
#error "Eth_ETNC_Ram.c : Mismatch in Release Revision Version"
#endif


#if (ETH_ETNC_RAM_SW_MAJOR_VERSION != ETH_ETNC_RAM_C_SW_MAJOR_VERSION)
#error "Eth_ETNC_Ram.c : Mismatch in Software Major Version"
#endif
#if (ETH_ETNC_RAM_SW_MINOR_VERSION != ETH_ETNC_RAM_C_SW_MINOR_VERSION)
#error "Eth_ETNC_Ram.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (1:1531)    : The object 'name' is referenced in only one translation unit - but not the one in which      */
/*                       it is defined.                                                                               */
/* Rule                : CERTCCM DCL15, CERTCCM DCL19, MISRA C:2012 Rule-8.7                                          */
/* JV-01 Justification : Followed coding rules, the objects (global variable/const) is define in seperated source     */
/*                       file, so this could be accepted                                                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : (4:5087) #include statements in a file should only be preceded by other preprocessor         */
/*                       directives or comments.                                                                      */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is accepted, due to implementation for include memmap is following AUTOSAR rule.        */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1504)    : The object 'name' is only referenced in the translation unit where it is defined.            */
/* Rule                : CERTCCM DCL15, DCL19, MISRA C:2012 Rule-8.7                                                  */
/* JV-01 Justification : This is accepted, due to following coding rule, internal function can be defined in          */
/*                       other C source files                                                                         */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1533)    : The object 'entity' is only referenced by function 'func'.                                   */
/* Rule                : MISRA C:2012 Rule-8.9                                                                        */
/* JV-01 Justification : This is accepted, due to the object is defined in seperated cource C file to followed        */
/*                       coding rule                                                                                  */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1532)    : The function '%1s' is only referenced in one translation unit - but not the one in which     */
/*                       it is defined.                                                                               */
/* Rule                : CERTCCM DCL19, MISRA C:2012 Rule-8.7                                                         */
/* JV-01 Justification : This is accepted, due to the specific coding rule, function of each hardware unit is         */
/*                       implemented in separated files for this hardware unit. Could not be static function.         */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3469)    : This usage of a function-like macro looks like it could be replaced by an equivalent         */
/*                       function call.                                                                               */
/* Rule                : MISRA C:2012 Dir-4.9                                                                         */
/* JV-01 Justification : To prevent overhead and speed up the processing, there is no need to call a function         */
/*                       here for such a small operation.                                                             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3432)    : Simple macro argument expression is not parenthesized.                                       */
/* Rule                : MISRA C:2012 Rule-20.7                                                                       */
/*                       REFERENCE - ISO:C90-6.3.1 Primary Expressions                                                */
/* JV-01 Justification : Compiler keyword (macro) is defined and used followed AUTOSAR standard rule. It is           */
/*                       accepted.                                                                                    */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3383)    : Cannot identify wraparound guard for unsigned arithmetic expression.                         */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : Wraparound can't be occur in this unsigned integer arithmetic so there is no need to add     */
/*                       a wraparound guard here.                                                                     */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:0312)    : Dangerous pointer cast results in loss of volatile qualification.                            */
/* Rule                : CERTCCM EXP32, MISRA C:2012 Rule 11.8                                                        */
/* JV-01 Justification : According to the coding guide, all global variables must have volatile attribute, but        */
/*                       volatile must be removed when return these variables from APIs according to the AUTOSAR      */
/*                       specification.                                                                               */
/*       Verification  : Since these are normal memory objects, any problem doesn't occurs.                           */
/**********************************************************************************************************************/
/* Message (3:3206)    : The parameter 'ucApiId' is not used in this function.                                        */
/* Rule                : CERTCCM MSC07, MSC13, MISRA C:2012 Rule-2.7                                                  */
/* JV-01 Justification : This argument is not used since only one controller is used. However, it is prepared         */
/*                       because it is used by another IP. There is no problem because it is as designed.             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2824)    : Possible: Arithmetic operation on NULL pointer.                                              */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : To prevent overhead and excessive nesting of condition statements, this pointer has been     */
/*                       tested valid with many test cases, so there is no problem with the current implementation.   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (7:0404)    : [U] More than one read access to volatile objects between sequence points.                   */
/* Rule                : CERTCCM EXP10, EXP30, MISRA C:2012 Rule 1.3, 13.2                                            */
/* JV-01 Justification : This is to get element in array of struct, volatile of counter variable of 'for' loop is     */
/*                       used to ensure no optimization. It is accepted                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3464)    : Argument to macro '%s' contains a side effect that will be evaluated more than once.         */
/* Rule                : CERTCCM PRE31                                                                                */
/* JV-01 Justification : After reviewing coding, it is concluded that this implementation is okay as it is since the  */
/*                       memory used to access this volatile variable is user RAM, not a specific HW register, so     */
/*                       there is no worry about potential side effects in this case.                                 */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:3387)    : A full expression containing an increment (++) or decrement (--) operator should have no     */
/*                       potential side effects other than that caused by the increment or decrement operator.        */
/* Rule                : MISRA C:2012 Rule-13.3                                                                       */
/*                       REFERENCE - ISO:C90-5.1.2.3 Program Execution                                                */
/* JV-01 Justification : After reviewing coding, it is concluded that this implementation is okay as it is since the  */
/*                       memory used to access this volatile variable is user RAM, not a specific HW register, so     */
/*                       there is no worry about potential side effects in this case.                                 */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (7:2801)    : Definite: Overflow in signed arithmetic operation.                                           */
/* Rule                : MISRA-C:2012 Dir 4.1                                                                         */
/* JV-01 Justification : This is the buffer pool array used for the entire system of this module.                     */
/*       Verification  : There is no problem because it is an intended design considering the space-time tradeoff,    */
/*                       that is, scalability.                                                                        */
/**********************************************************************************************************************/
/* Message (7:2841)    : Definite: Dereference of an invalid pointer value.                                           */
/* Rule                : MISRA-C:2012 Dir 4.1                                                                         */
/* JV-01 Justification : This is the buffer pool array used for the entire system this module.                        */
/*       Verification  : There is no problem because it is an intended design considering the space-time tradeoff,    */
/*                       that is, scalability.                                                                        */
/**********************************************************************************************************************/
/* Message (2:0612)    : [L] Size of object '%s' exceeds 32767 bytes - program does not conform strictly to ISO:C90.  */
/* Rule                : MISRA-C:2012 Dir 1.1                                                                         */
/* JV-01 Justification : This is the buffer pool array used for the entire system of this module.                     */
/*       Verification  : There is no problem because it is an intended design considering the space-time tradeoff,    */
/*                       that is, scalability.                                                                        */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                   Global Data                                                      **
***********************************************************************************************************************/
#define ETH_START_SEC_VAR_INIT_UNSPECIFIED
#include "Eth_MemMap.h"

/* Global variable to store  Controller State*/
volatile VAR(Eth_StateType, ETH_VAR_INIT) Eth_GenDriverState =                                                          /* PRQA S 1531 # JV-01 */
  ETH_STATE_UNINIT;

#define ETH_STOP_SEC_VAR_INIT_UNSPECIFIED
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */


#define ETH_START_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/* Status of controller(s) */
volatile VAR(Eth_ControllerStatusType, ETH_VAR_NO_INIT)
  Eth_GaaCtrlStat[ETH_MAX_CTRLS_SUPPORTED];

#define ETH_STOP_SEC_VAR_NO_INIT_UNSPECIFIED
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */


#define ETH_START_SEC_VAR_NO_INIT_8
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/* Ring buffer which contains free tx buffer indexes */
volatile VAR(uint8, ETH_VAR_NO_INIT)
  Eth_GaaTxBufferRing[ETH_TX_BUF_TOTAL_0];                                                                              /* PRQA S 1504 # JV-01 */

/* Whether each Tx buffer is provided to the user.
   Set as TRUE at Eth_ProvideTxBuffer, set as FALSE at Eth_Transmit. */
volatile VAR(boolean, ETH_VAR_NO_INIT)
  Eth_GaaTxBufferProvided[ETH_TX_BUF_TOTAL_0];                                                                          /* PRQA S 1504 # JV-01 */

#define ETH_STOP_SEC_VAR_NO_INIT_8
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */


#define ETH_START_SEC_VAR_NO_INIT_32
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

#if (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON)
/* Multicast address filter list */
volatile VAR(Eth_MacAddressType, ETH_VAR_NO_INIT)
  Eth_GaaAddressFilters[ETH_MAX_CTRLS_SUPPORTED][ETH_MAX_FILTERS];
#endif

#define ETH_STOP_SEC_VAR_NO_INIT_32
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */


#define ETH_START_SEC_VAR_INIT_16
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

VAR(uint16, ETH_VAR_INIT)Eth_GaaDemEventAccess[ETH_MAX_CTRLS_SUPPORTED] =                                               /* PRQA S 1531 # JV-01 */
  {ETH_E_ACCESS_0};
VAR(uint16, ETH_VAR_INIT)Eth_GaaDemEventRxFramesLost[ETH_MAX_CTRLS_SUPPORTED] =
  {ETH_E_RX_FRAMES_LOST_0};
VAR(uint16, ETH_VAR_INIT)Eth_GaaDemEventCRC[ETH_MAX_CTRLS_SUPPORTED] =
  {ETH_E_CRC_0};
VAR(uint16, ETH_VAR_INIT)Eth_GaaDemEventUnderSizeFrame[ETH_MAX_CTRLS_SUPPORTED] =
  {ETH_E_UNDERSIZEFRAME_0};
VAR(uint16, ETH_VAR_INIT)Eth_GaaDemEventOverSizeFrame[ETH_MAX_CTRLS_SUPPORTED] =
  {ETH_E_OVERSIZEFRAME_0};
VAR(uint16, ETH_VAR_INIT)Eth_GaaDemEventAlignment[ETH_MAX_CTRLS_SUPPORTED] =
  {ETH_E_ALIGNMENT_0};
VAR(uint16, ETH_VAR_INIT)Eth_GaaDemEventSinglecollision[ETH_MAX_CTRLS_SUPPORTED] =                                      /* PRQA S 1533 # JV-01 */
  {ETH_E_SINGLECOLLISION_0};
VAR(uint16, ETH_VAR_INIT)Eth_GaaDemEventMultiplecollision[ETH_MAX_CTRLS_SUPPORTED] =                                    /* PRQA S 1533 # JV-01 */
  {ETH_E_MULTIPLECOLLISION_0};
VAR(uint16, ETH_VAR_INIT)Eth_GaaDemEventLatecollision[ETH_MAX_CTRLS_SUPPORTED] =
  {ETH_E_LATECOLLISION_0};

VAR(uint16, ETH_VAR_INIT)Eth_GaaDemEventIntInconsistent[ETH_MAX_CTRLS_SUPPORTED] =                                      /* PRQA S 1533 # JV-01 */
  {ETH_E_INT_INCONSISTENT_0};
VAR(uint16, ETH_VAR_INIT)Eth_GaaDemEventRegisterCorruption[ETH_MAX_CTRLS_SUPPORTED] =                                   /* PRQA S 1531 # JV-01 */
  {ETH_E_REGISTER_CORRUPTION_0};
VAR(uint16, ETH_VAR_INIT)Eth_GaaDemEventDmaError[ETH_MAX_CTRLS_SUPPORTED] =                                             /* PRQA S 1533 # JV-01 */
  {ETH_E_DMA_ERROR_0};

#define ETH_STOP_SEC_VAR_INIT_16
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */


#define ETH_START_SEC_VAR_NO_INIT_PTR
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/* Global variable to store pointer to Configuration */
P2CONST(Eth_CtrlConfigType, ETH_CONST, ETH_CONFIG_CONST) volatile Eth_GpCtrlConfigPtr;                                  /* PRQA S 1533 # JV-01 */
P2CONST(Eth_EthConfigType, ETH_CONST, ETH_CONFIG_CONST) volatile Eth_GpEthConfigPtr;
/* Status of controller(s) */

#define ETH_STOP_SEC_VAR_NO_INIT_PTR
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */


#define ETH_START_SEC_VAR_DESC_BUFFER
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/* Descriptors */
volatile VAR(Eth_ETNCTxDescriptorType, ETH_VAR_NO_INIT)
  Eth_GaaTxDescriptor[ETH_TX_BUF_TOTAL_0];
volatile VAR(Eth_ETNCRxDescriptorType, ETH_VAR_NO_INIT)
  Eth_GaaRxDescriptor[ETH_RX_BUF_TOTAL_0];

#define ETH_STOP_SEC_VAR_DESC_BUFFER
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */


#define ETH_START_SEC_VAR_TX_BUFFER
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/* Tx buffer */
volatile VAR(uint8, ETH_VAR_NO_INIT)
  Eth_GaaTxBuffer[ETH_TX_BUF_TOTAL_0][ETH_INTERNAL_TX_BUFFER_BYTE];                                                     /* PRQA S 0612 # JV-01 */

#define ETH_STOP_SEC_VAR_TX_BUFFER
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */


#define ETH_START_SEC_VAR_RX_BUFFER
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/* Rx buffer */
volatile VAR(uint8, ETH_VAR_NO_INIT)
  Eth_GaaRxBuffer[ETH_MAX_CTRLS_SUPPORTED][ETH_RX_BUF_TOTAL_0][ETH_INTERNAL_RX_BUFFER_BYTE];                            /* PRQA S 0612 # JV-01 */

#define ETH_STOP_SEC_VAR_RX_BUFFER
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */


/***********************************************************************************************************************
**                                                Function Definitions                                                **
***********************************************************************************************************************/
#define ETH_START_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name         : Eth_InitializeBuffer
**
** Service ID            : N/A
**
** Description           : Initialize the Tx buffer ring
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LulCtrlIdx     : controller index
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variables Used : Eth_GaaCtrlStat, Eth_GaaTxBufferRing,
**                         Eth_GaaTxBufferProvided
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_167, ETH_DUD_ACT_167_GBL001,
** Reference ID          : ETH_DUD_ACT_167_GBL002, ETH_DUD_ACT_167_GBL003,
** Reference ID          : ETH_DUD_ACT_167_GBL004
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_InitializeBuffer(                                                                      /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  uint32 LulBufIdx;

  /* Store all buffers to the tx buffer ring */
  for (LulBufIdx = 0U; LulBufIdx < (uint32)ETH_TX_BUF_TOTAL_0; LulBufIdx++)
  {
    /* Since the maximum value of buffer index is 254, casting to uint8 does no problem. */
    Eth_GaaTxBufferRing[LulBufIdx] = (uint8)LulBufIdx;
    Eth_GaaTxBufferProvided[LulBufIdx] = ETH_FALSE;
  }

  /* Initialize pointers */
  Eth_GaaCtrlStat[LulCtrlIdx].ulTxBufHead = 0UL;                                                                        /* PRQA S 2844 # JV-01 */
  Eth_GaaCtrlStat[LulCtrlIdx].ulTxBufTail = 0UL;                                                                        /* PRQA S 2844 # JV-01 */
}

/***********************************************************************************************************************
** Function Name         : Eth_PreprocessBuffer
**
** Service ID            : N/A
**
** Description           : Write source address to all Tx Buffers in advance
**                         because source address is never changed while
**                         a controller mode is ACTIVE.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LulCtrlIdx     : controller index
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variables Used : Eth_GaaCtrlStat, Eth_GaaTxBuffer
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_168, ETH_DUD_ACT_168_GBL001
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_PreprocessBuffer(                                                                      /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx)
{
  uint32 LulBufIdx;

  for (LulBufIdx = 0U; LulBufIdx < (uint32)ETH_TX_BUF_TOTAL_0; LulBufIdx++)
  {
    ETH_UNPACK_ADDRESS_TO_8(Eth_GaaCtrlStat[LulCtrlIdx].stMacAddr,                                                      /* PRQA S 2801, 2841, 3469, 2844 # JV-01, JV-01, JV-01, JV-01 */
      &Eth_GaaTxBuffer[LulBufIdx][ETH_SRC_MACADDR_OFFSET]);
  }
}

/***********************************************************************************************************************
** Function Name         : Eth_GetTxBuffer
**
** Service ID            : N/A
**
** Description           : Get a buffer from the Tx buffer ring
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LulCtrlIdx     : controller index
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : A index of acquired buffer
**                         When the ring is empty, ETH_INVALID_BUFFER_INDEX
**
** Preconditions         : None
**
** Global Variables Used : Eth_GaaCtrlStat, Eth_GaaTxBufferRing,
**                         Eth_GaaTxBufferProvided
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_169, ETH_DUD_ACT_169_GBL001,
** Reference ID          : ETH_DUD_ACT_169_GBL002, ETH_DUD_ACT_169_GBL003
***********************************************************************************************************************/
FUNC(BufReq_ReturnType , ETH_PRIVATE_CODE) Eth_GetTxBuffer(                                                             /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONSTP2VAR(Eth_BufIdxType, AUTOMATIC, ETH_APPL_DATA) BufIdxPtr,                                                       /* PRQA S 3432 # JV-01 */
  CONSTP2VAR(uint8*, AUTOMATIC, ETH_APPL_DATA) BufPtr,                                                                  /* PRQA S 3432 # JV-01 */
  CONSTP2VAR(uint16, AUTOMATIC, ETH_APPL_DATA) LenBytePtr)                                                              /* PRQA S 3432 # JV-01 */
{
  uint32 LulBufIdx;
  uint32 LulRingIdx;
  BufReq_ReturnType LenReturnValue;
  /* Get the current tail pointer */
  /* If the requested size is larger than the buffer, return error */
  /* Since the maximum value of buffer index is 1024, casting to uint16 does no problem. */
  if (*LenBytePtr > (uint16)((uint32)ETH_CTRL_TX_BUF_LEN_BYTE_0 - (uint32)ETH_HEADER_SIZE))                             /* PRQA S 2814 # JV-01 */
  {
    *LenBytePtr = (uint16)((uint32)ETH_CTRL_TX_BUF_LEN_BYTE_0 - (uint32)ETH_HEADER_SIZE);
    LenReturnValue = BUFREQ_E_OVFL;
  }
  else
  {
    /* If the requested size is smaller than the minimum size,
       expand it to the minimum size */
    if ((uint16)ETH_MIN_PAYLOAD_SIZE >= *LenBytePtr)
    {
      *LenBytePtr = (uint16)ETH_MIN_PAYLOAD_SIZE;
    }
    else
    {
      /*Do Nothing*/
    }

    LulRingIdx = Eth_GaaCtrlStat[LulCtrlIdx].ulTxBufTail;                                                               /* PRQA S 2844 # JV-01 */

    LulBufIdx = (uint32)Eth_GaaTxBufferRing[LulRingIdx];                                                                /* PRQA S 2844 # JV-01 */

    /* If the ring is empty, return the invalid index */
    if (ETH_INVALID_BUFFER_INDEX == LulBufIdx)
    {
      /* Nothin to do */
    }
    else
    {
      /* Mark this slot as empty */
      /* Since value of invalid buffer index is 255, casting to uint8 does no problem. */
      Eth_GaaTxBufferRing[LulRingIdx] = (uint8)ETH_INVALID_BUFFER_INDEX;                                                /* PRQA S 2844 # JV-01 */

      /* Increment the tail pointer */
      /* Since the maximum value of buffer index is 254, casting to uint8 does no problem. */
      Eth_GaaCtrlStat[LulCtrlIdx].ulTxBufTail = (uint32)((LulRingIdx + 1UL) % (uint32)ETH_TX_BUF_TOTAL_0);              /* PRQA S 2844, 3383 # JV-01, JV-01 */

      /* Mark this buffer as provided */
      Eth_GaaTxBufferProvided[LulBufIdx] = ETH_TRUE;                                                                    /* PRQA S 2844 # JV-01 */
    }

    if (ETH_INVALID_BUFFER_INDEX == LulBufIdx)
    {
      LenReturnValue = BUFREQ_E_BUSY;
    }
    else
    {
      *BufIdxPtr = (Eth_BufIdxType)LulBufIdx;                                                                           /* PRQA S 2814 # JV-01 */
      /* Store the top address of payload area to BufPtr */
      *BufPtr = (P2VAR(uint8, AUTOMATIC, ETH_APPL_DATA)) &Eth_GaaTxBuffer[*BufIdxPtr][ETH_PAYLOAD_OFFSET];              /* PRQA S 2814, 0312, 3432, 2844 # JV-01, JV-01, JV-01, JV-01 */
      LenReturnValue = BUFREQ_OK;
    }
  }
  return LenReturnValue;
}

/***********************************************************************************************************************
** Function Name         : Eth_ReleaseTxBuffer
**
** Service ID            : N/A
**
** Description           : Release a Tx buffer and store it to the tail of the
**                         Tx buffer ring
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant when interrupt mode
**                         Reentrant when polling mode
**
** Input Parameters      : LulCtrlIdx     : controller index
**                         LulBufIdx      : index to the buffer
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variables Used : Eth_GaaCtrlStat, Eth_GaaTxBufferRing
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_170, ETH_DUD_ACT_170_CRT001,
** Reference ID          : ETH_DUD_ACT_170_CRT002, ETH_DUD_ACT_170_GBL001,
** Reference ID          : ETH_DUD_ACT_170_GBL002
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_ReleaseTxBuffer(                                                                       /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulBufIdx)
{
  uint32 LulRingIdx;

  /* When the polling mode, this function can be invoked from
     Eth_Transmit and Eth_TxConfirmation at same time.
     So the critical section is required.
     When the interrupt mode, this function is invoked from TxISR only. */
  #if (ETH_CTRL_ENABLE_TX_POLLING == STD_ON)
  ETH_ENTER_CRITICAL_SECTION(ETH_RAM_DATA_PROTECTION);
  #endif
  /* Get the current head pinter */
  LulRingIdx = Eth_GaaCtrlStat[LulCtrlIdx].ulTxBufHead;                                                                 /* PRQA S 2844 # JV-01 */

  /* Store a buffer to the head of the tx buffer ring */
  /* Since the maximum value of buffer index is 254, casting to uint8 does no problem. */
  Eth_GaaTxBufferRing[LulRingIdx] = (uint8)LulBufIdx;                                                                   /* PRQA S 2844 # JV-01 */

  /* Increment the head pointer */
  /* Since the maximum value of buffer index is 254, casting to uint8 does no problem. */
  Eth_GaaCtrlStat[LulCtrlIdx].ulTxBufHead = (uint32)((LulRingIdx + 1UL) % (uint32)ETH_TX_BUF_TOTAL_0);                  /* PRQA S 2844, 3383 # JV-01, JV-01 */

  #if (ETH_CTRL_ENABLE_TX_POLLING == STD_ON)
  ETH_EXIT_CRITICAL_SECTION(ETH_RAM_DATA_PROTECTION);
  #endif
}

/***********************************************************************************************************************
** Function Name         : Eth_PreprocessFrame
**
** Service ID            : N/A
**
** Description           :
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant when interrupt mode
**                         Reentrant when polling mode
**
** Input Parameters      : LulCtrlIdx     : controller index
**                         LulBufIdx      : index to the buffer
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variables Used : Eth_GaaCtrlStat, Eth_GaaTxBufferRing
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_171, ETH_DUD_ACT_171_GBL001,
** Reference ID          : ETH_DUD_ACT_171_GBL002, ETH_DUD_ACT_171_GBL003,
** Reference ID          : ETH_DUD_ACT_171_GBL004
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_PreprocessFrame(                                                                       /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulBufIdx,                                              /* PRQA S 3206 # JV-01 */
  CONST(uint32, AUTOMATIC) LulFrameType, CONSTP2CONST(uint8, AUTOMATIC, ETH_APPL_DATA) LpPhysAddrPtr)
{
  uint32 LulInc;
  /* Mark this buffer as not provided to prevent re-used */
  Eth_GaaTxBufferProvided[LulBufIdx] = ETH_FALSE;                                                                       /* PRQA S 2844 # JV-01 */

  /* Write destination MAC address */
  for (LulInc = 0U; LulInc < ETH_MACADDR_SIZE; LulInc++)
  {
    Eth_GaaTxBuffer[LulBufIdx][ETH_DST_MACADDR_OFFSET + LulInc] = LpPhysAddrPtr[LulInc];                                /* PRQA S 2844, 3383, 2824 # JV-01, JV-01, JV-01 */
  }

  /* Source MAC address is already written by Eth_PreprocessBuffer */

  /* Write FrameType with big endian */
  /* Casted to uint8 to extract the required 1 byte. */
  Eth_GaaTxBuffer[LulBufIdx][ETH_FRAMETYPE_OFFSET] = (uint8)(LulFrameType >> ETH_BYTE_BITS);                            /* PRQA S 2844 # JV-01 */
  Eth_GaaTxBuffer[LulBufIdx][ETH_FRAMETYPE_OFFSET + 1UL] = (uint8)LulFrameType;                                         /* PRQA S 2844 # JV-01 */

}

/***********************************************************************************************************************
** Function Name         : Eth_CheckProvideBuffer
**
** Service ID            : N/A
**
** Description           :
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant when interrupt mode
**                         Reentrant when polling mode
**
** Input Parameters      : LulCtrlIdx     : controller index
**                         LulBufIdx      : index to the buffer
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variables Used : Eth_GaaCtrlStat, Eth_GaaTxBufferRing
**
** Function(s) invoked   : None
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_172
***********************************************************************************************************************/
FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_CheckProvideBuffer(                                                          /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulBufIdx)                                              /* PRQA S 3206 # JV-01 */
{
  Std_ReturnType LucReturnValue;
  LucReturnValue = E_OK;
  if (ETH_FALSE == Eth_GaaTxBufferProvided[LulBufIdx])                                                                  /* PRQA S 2844 # JV-01 */
  {
    LucReturnValue = E_NOT_OK;
  }
  else
  {
     /* No action is required */
  }
  return LucReturnValue;
}

#define ETH_STOP_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"                                                                                                 /* PRQA S 5087 # JV-01 */


/***********************************************************************************************************************
** Function Name         : Eth_RxIndication
**
** Service ID            : N/A
**
** Description           : Filters the specified frame and indicates it to EthIf
**                         if it is acceptable
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LulCtrlIdx   : controller index
**                         LpBuffer     : received frame data
**                         LulLenByte   : byte length of a frame data
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variable(s)    : Eth_GaaCtrlStat, Eth_GaaAddressFilters,
**                         Eth_GaaRxBuffer, Eth_GstBroadcastAddr
**
** Function(s) invoked   : EthIf_RxIndication
**
** Registers Used        : None
**
** Reference ID          : ETH_DUD_ACT_173, ETH_DUD_ACT_173_GBL001
***********************************************************************************************************************/
FUNC(void, ETH_PRIVATE_CODE) Eth_RxIndication(                                                                          /* PRQA S 1532 # JV-01 */
  CONST(uint32, AUTOMATIC) LulCtrlIdx,
  CONST(uint32, AUTOMATIC) LulBufferIdx, CONST(uint32, AUTOMATIC) LulLenByte)
{
  Eth_MacAddressType LstDstAddress;
  boolean LblBroadcast;
  Eth_FrameType LusFrameType;
#if (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON)
  boolean LblPass;
  uint32 LulFilterIdx;
  uint32 LulRemainBits;
#endif

  /* Pack MAC address from BYTE to WORD to quicken comparing operation */
  ETH_PACK_ADDRESS_FROM_8(&Eth_GaaRxBuffer[LulCtrlIdx][LulBufferIdx][ETH_DST_MACADDR_OFFSET], LstDstAddress);           /* PRQA S 0404, 3469, 3464, 2844 # JV-01, JV-01, JV-01, JV-01 */

  /* Check whether broadcast */
  /* Since this value is passed to EthIf,
     use TRUE/FALSE instead of ETH_TRUE/ETH_FALSE */
  if (0UL == ETH_COMPARE_MAC(LstDstAddress, Eth_GstBroadcastAddr))                                                      /* PRQA S 3469 # JV-01 */
  {
    LblBroadcast = (boolean)TRUE;
    /* Increment the total number of bradcast frames */
#if (ETH_GET_ETHER_STATS_API == STD_ON)
    if (Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulStatsBroadcastPkts < ETH_UINT32_MAXVALUE)                              /* PRQA S 2844 # JV-01 */
    {
      Eth_GaaCtrlStat[LulCtrlIdx].stTxRxStat.ulStatsBroadcastPkts++;                                                    /* PRQA S 3383, 3387 # JV-01, JV-01 */
    }
    else
    {
      /* No action required */
    }
#endif
  }
  else
  {
    LblBroadcast = (boolean)FALSE;
  }

#if (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON)
  LblPass = ETH_FALSE;

  /* In the following case, accept unconditionally:
     - Filtering is off (unicast only, multicast is discarded by LLDriver)
     - Promiscuous mode
     - Broadcast frame */
  if ((0UL == Eth_GaaCtrlStat[LulCtrlIdx].ulActiveFilterBits) ||                                                        /* PRQA S 2844 # JV-01 */
    (ETH_TRUE == Eth_GaaCtrlStat[LulCtrlIdx].blPromiscuous) ||
    ((boolean)TRUE == LblBroadcast))
  {
    LblPass = ETH_TRUE;
  }
  else
  {
    /* When unicast frame for this node, accept it */
    if (0UL == ETH_COMPARE_MAC(LstDstAddress, Eth_GaaCtrlStat[LulCtrlIdx].stMacAddr))                                   /* PRQA S 3469, 3464 # JV-01, JV-01 */
    {
      LblPass = ETH_TRUE;
    }
    else
    {
      /* Otherwise, do the filter operation */

      /* To skip empty filters efficiently, copy active bits to the local var */
      LulRemainBits = Eth_GaaCtrlStat[LulCtrlIdx].ulActiveFilterBits;
      LulFilterIdx = 0UL;
      /* Loop until any filter hits or pass all active filters */
      while ((0UL != LulRemainBits) && (LblPass == ETH_FALSE))
      {
        if ((0UL != (LulRemainBits & (1UL << LulFilterIdx))) &&                                                         /* PRQA S 0404 # JV-01 */
          (0UL == ETH_COMPARE_MAC(LstDstAddress, Eth_GaaAddressFilters[LulCtrlIdx][LulFilterIdx])))                     /* PRQA S 3469, 3464, 2844 # JV-01, JV-01, JV-01 */
        {
          LblPass = ETH_TRUE;
        }
        else
        {
          /* No action required */
        }
        /* Clear the compared filter bit */
        LulRemainBits &= ~(1UL << LulFilterIdx);
        LulFilterIdx++;                                                                                                 /* PRQA S 3383 # JV-01 */
      }
    }
  }

  if (ETH_FALSE == LblPass)
  {
    /* Discard this frame */
  }
  else
#endif /* (ETH_UPDATE_PHYS_ADDR_FILTER == STD_ON) */
  {
    /* Get FrameType with little endian  */
    /* Since the frame type is 16bit, casting to uint16 does no problem. */
    LusFrameType = (uint16)((uint32)Eth_GaaRxBuffer[LulCtrlIdx][LulBufferIdx][ETH_FRAMETYPE_OFFSET]
      << (uint32)ETH_BYTE_BITS);
    LusFrameType |= Eth_GaaRxBuffer[LulCtrlIdx][LulBufferIdx][ETH_FRAMETYPE_OFFSET + 1UL];

    /* Indicate to the upper layer */
    /* Since the maximum value of buffer size is 1518, casting to uint16 does no problem. */
    /* Since the maximum value of controller index is 1, casting to uint8 does no problem. */
    EthIf_RxIndication((uint8)LulCtrlIdx, LusFrameType, LblBroadcast,
      (P2CONST(uint8, AUTOMATIC, ETH_APPL_DATA)) &Eth_GaaRxBuffer[LulCtrlIdx][LulBufferIdx][ETH_SRC_MACADDR_OFFSET],    /* PRQA S 0312 # JV-01 */
      (P2VAR(Eth_DataType, AUTOMATIC, ETH_APPL_DATA)) &Eth_GaaRxBuffer[LulCtrlIdx][LulBufferIdx][ETH_PAYLOAD_OFFSET],   /* PRQA S 0312, 3432 # JV-01, JV-01 */
      (uint16)(LulLenByte - (uint32)ETH_HEADER_SIZE));                                                                  /* PRQA S 3383 # JV-01 */
  }
}

/***********************************************************************************************************************
**                                                   End of File                                                      **
***********************************************************************************************************************/
