/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Icu_TAUJ_LLDriver.c                                                                                 */
/* SW-VERSION   = 1.5.0                                                                                               */
/*====================================================================================================================*/
/*                                                     COPYRIGHT                                                      */
/*====================================================================================================================*/
/* (c) 2020-2022 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* This file contains Low Level function implementations of ICU Driver Component                                      */
/*                                                                                                                    */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s) and/or */
/* the Application.                                                                                                   */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs of  */
/* program errors, compliance with applicable laws, damage to or loss of data, programs or equipment, and             */
/* unavailability or interruption of operations.                                                                      */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*              Devices:        U2A                                                                                   */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                              Revision Control History                                              **
***********************************************************************************************************************/
/*
 * 1.5.0:  17/11/2022  : As per ticket ARDAACA-1266 and ARDAACA-1302, update "Input Parameters", "Global Variables", 
 *                       "Functions invoked" for all functions.
 *                       Add ICU_DUD_ACT_048_REG004 in description of Icu_HW_Tauj_ChannelInit.
 *         15/11/2022  : Update Icu_HW_Tauj_UnitDeInit to access all TAUJnTT registers
 * 1.4.4:  16/06/2022  : Update QAC message and QAC header
 * 1.4.3:  23/05/2022  : Fix QAC header and message.
 *         09/05/2022  : Remove "else" statement when no action required; add QAC message 2004.
 *         17/04/2022  : Merge QAC message from U2Ax, E2x and U2Bx
 *         13/04/2022  : Update function Icu_HW_Tauj_GetTAUInCountValue, get Tau counter value from new pTAUJCNTAddress
 *                       and add Null check before read pTAUJCNTAddress counter value
 * 1.4.2:  02/03/2022  : Fix QAC header and message
 * 1.4.1:  20/11/2021  : Add QAC message 9.5.0
 *         07/10/2021  : Remove code segment of usTAUJnTPS and ucTAUJnBRS in Icu_HW_Tauj_UnitDeInit
 *         18/10/2021  : Add QAC message 9.5.0
 *         16/09/2021  : Correct preprocessor directives
 *                       Correct if conditions
 *                       Correct names, datatypes of used variables
 *                       Removed redundant switch/case branches
 * 1.3.2:  30/08/2021  : Change from #include "Icu_MemMap.h" to #include "Icu_Mapping.h".
 *                       Update pre-compile parameter ICU_E_INT_INCONSISTENT_CONFIGURED for ICU_DEM_REPORT_ERROR
 * 1.3.1:  02/07/2021  : Add QAC message 9.5.0
 *                       Format source code to 120 characters
 *                       Improve Violation tag (remove START/END)
 * 1.3.0:  20/01/2021  : Changed CONSTP2CONST to P2CONST for argument ConfigPtr
 * 1.2.0:  17/07/2020  : Release
 * 1.1.0:  19/06/2020  : Release
 * 1.0.0:  25/03/2020  : Initial Version
 */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (2:0310)    : Casting to different object pointer type.                                                    */
/* Rule                : CERTCCM EXP11, EXP39, MISRA C:2012 Rule-11.3                                                 */
/* JV-01 Justification : For accessing 8-bit and 16-bit PNOT and JPNOT register respectively, the 32-bit pointer is   */
/*                       typecasted.                                                                                  */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0316)    : Cast from a pointer to void to a pointer to object type.                                     */
/* Rule                : MISRA C:2012 Rule-11.5                                                                       */
/* JV-01 Justification : A cast should not be performed between a pointer to object type and a different pointer to   */
/*                       object type.                                                                                 */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0317)    : Implicit conversion from a pointer to void to a pointer to object type.                      */
/* Rule                : MISRA C:2012 Rule-11.5                                                                       */
/* JV-01 Justification : This is done as per implementation requirement.                                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact               */
/**********************************************************************************************************************/
/* Message (2:0488)    : Performing pointer arithmetic.                                                               */
/* Rule                : CERTCCM EXP08, MISRA C:2012 Rule-18.4                                                        */
/* JV-01 Justification : This is to get the ID in the data structure in the code.                                     */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0751)    : Casting to char pointer type.                                                                */
/* Rule                : CERTCCM EXP11, EXP39                                                                         */
/* JV-01 Justification : Using void due to specific requirement of input parameter. So, this can be skipped           */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (7:0791)    : Macro identifier does not differ from other macro identifier(s) (e.g. '%s') within the       */
/*                       specified number of significant characters.                                                  */
/* Rule                : CERTCCM DCL23, MISRA C:2012 Rule-5.4                                                         */
/* JV-01 Justification : This macro identifier is following AUTOSAR standard rule (Symbolic Name or Published         */
/*                       Macro's name), so this is accepted.                                                          */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0857)    : Number of macro definitions exceeds 1024 - program does not conform strictly to ISO:C90.     */
/* Rule                : MISRA C:2012 Dir-1.1                                                                         */
/* JV-01 Justification : The number of macro depend on module code size. There is no issue when number of macro is    */
/*                       over 1024                                                                                    */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1281)    : Integer literal constant is of an unsigned type but does not include a "U" suffix.           */
/* Rule                : MISRA C:2012 Rule-7.2                                                                        */
/* JV-01 Justification : Integer literal constant is of an unsigned type but does not include a "U" suffix.           */
/*       Verification  : Lack of using "U" is accepted since it cause no problem in software behavior.                */
/**********************************************************************************************************************/
/* Message (1:1503)    : The function '%1s' is defined but is not used within this project.                           */
/* Rule                : CERTCCM MSC07, MISRA C:2012 Rule-2.1                                                         */
/* JV-01 Justification : This is accepted, due to the module's API is exported for user's usage.                      */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1505)    : The function '%1s' is only referenced in the translation unit where it is defined.           */
/* Rule                : CERTCCM DCL19, MISRA C:2012 Rule-8.7                                                         */
/* JV-01 Justification : This is accepted, due to following coding rule, internal function can be defined in other C  */
/*                       source files                                                                                 */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1532)    : The function '%1s' is only referenced in one translation unit - but not the one in which it  */
/*                       is defined.                                                                                  */
/* Rule                : CERTCCM DCL19, MISRA C:2012 Rule-8.7                                                         */
/* JV-01 Justification : This macro identifier is following AUTOSAR standard rule  (Symbolic Name or Published        */
/*                       Macro's name), so this is accepted                                                           */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:2004)    : No concluding 'else' exists in this 'if'-'else'-'if' statement.                              */
/* Rule                : CERTCCM MSC01, MISRA C:2012 Rule-15.7                                                        */
/* JV-01 Justification : The "else" statement with empty content is removed to improve readability.                   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : This is accepted, due to the implementation is following hardware specification or it was    */
/*                       checked or do not dereference to NULL pointer.                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2824)    : Possible: Arithmetic operation on NULL pointer.                                              */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : This is accepted, due to the implementation is following hardware specification.             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : It is specific for device register accessing and confirmed has no issue in software behavior.*/
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (6:3305)    : Pointer cast to stricter alignment.                                                          */
/* Rule                : CERTCCM EXP36, EXP39, MISRA C:2012 Rule-11.3                                                 */
/* JV-01 Justification : Pointer alignment is changed by casting, but it's necessary for embedded programming         */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3383)    : Cannot identify wraparound guard for unsigned arithmetic expression.                         */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : It can still result in values that are out of range for the intended use, as intuitive       */
/*                       "invariants" may not hold                                                                    */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3408)    : %s' has external linkage and is being defined without any previous declaration.              */
/* Rule                : CERTCCM DCL07, MISRA C:2012 Rule-8.4                                                         */
/* JV-01 Justification : It is accepted, due to the declaration will be taken care by Os                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3432)    : Simple macro argument expression is not parenthesized.                                       */
/* Rule                : MISRA C:2012 Rule-20.7                                                                       */
/* JV-01 Justification : Compiler keyword (macro) is defined and used followed AUTOSAR standard rule. It is accepted. */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3678)    : The object referenced by '%1s' is not modified through it, so '%1s' could be declared with   */
/*                       type '%2s'.                                                                                  */
/* Rule                : MISRA C:2012 Rule-8.13                                                                       */
/* JV-01 Justification : This is accepted. It just an advise for improve safety by reducing the possibility that the  */
/*                       referenced data is unintentionally modified through an unexpected alias and improves         */
/*                       clarity by indicating that the referenced data is not intended to be modified through this   */
/*                       alias or those depending on it                                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:4342)    : An expression of 'essentially unsigned' type (%1s) is being cast to enum type '%2s'.         */
/* Rule                : MISRA C:2012 Rule-10.5                                                                       */
/* JV-01 Justification : It is assigned to a variable with no conflict in the data.                                   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : Use of #include directive after code fragment.                                               */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is done as per Memory Requirement, (MEMMAP003 - Specification of Memory Mapping).       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                  Include Section                                                   **
***********************************************************************************************************************/

/* Included for Header file inclusion */
#include "Icu_TAU_LLDriver.h"                                                                                           /* PRQA S 0857 # JV-01 */
#include "Icu_TAUJ_LLDriver.h"
/* Included for RAM variable declarations */
#include "Icu_Ram.h"
#include "Icu_TAUJ_Ram.h"

#if (ICU_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
/* Include for interrupt consistency check */
#include "Dem.h"                                                                                                        /* PRQA S 0857 # JV-01 */
#endif
/***********************************************************************************************************************
**                                                Version Information                                                 **
***********************************************************************************************************************/
/* AUTOSAR release version information */
#define ICU_TAUJ_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION    ICU_AR_RELEASE_MAJOR_VERSION_VALUE
#define ICU_TAUJ_LLDRIVER_C_AR_RELEASE_MINOR_VERSION    ICU_AR_RELEASE_MINOR_VERSION_VALUE                              /* PRQA S 0791 # JV-01 */
#define ICU_TAUJ_LLDRIVER_C_AR_RELEASE_REVISION_VERSION ICU_AR_RELEASE_REVISION_VERSION_VALUE                           /* PRQA S 0791 # JV-01 */

/* File version information */
#define ICU_TAUJ_LLDRIVER_C_SW_MAJOR_VERSION            ICU_SW_MAJOR_VERSION_VALUE
#define ICU_TAUJ_LLDRIVER_C_SW_MINOR_VERSION            ICU_SW_MINOR_VERSION_VALUE

/***********************************************************************************************************************
**                                                   Version Check                                                    **
***********************************************************************************************************************/
#if (ICU_TAUJ_LLDRIVER_AR_RELEASE_MAJOR_VERSION != ICU_TAUJ_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION)
#error "Icu_TAUJ_LLDriver.c : Mismatch in Release Major Version"
#endif

#if (ICU_TAUJ_LLDRIVER_AR_RELEASE_MINOR_VERSION != ICU_TAUJ_LLDRIVER_C_AR_RELEASE_MINOR_VERSION)
#error "Icu_TAUJ_LLDriver.c : Mismatch in Release Minor Version"
#endif

#if (ICU_TAUJ_LLDRIVER_AR_RELEASE_REVISION_VERSION != ICU_TAUJ_LLDRIVER_C_AR_RELEASE_REVISION_VERSION)
#error "Icu_TAUJ_LLDriver.c : Mismatch in Release Revision Version"
#endif

#if (ICU_TAUJ_LLDRIVER_SW_MAJOR_VERSION != ICU_TAUJ_LLDRIVER_C_SW_MAJOR_VERSION)
#error "Icu_TAUJ_LLDriver.c : Mismatch in Software Major Version"
#endif

#if (ICU_TAUJ_LLDRIVER_SW_MINOR_VERSION != ICU_TAUJ_LLDRIVER_C_SW_MINOR_VERSION)
#error "Icu_TAUJ_LLDriver.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                                    Global Data                                                     **
***********************************************************************************************************************/

/***********************************************************************************************************************
**                                                Function Definitions                                                **
***********************************************************************************************************************/

/***********************************************************************************************************************
** Function Name        : Icu_HW_Tauj_UnitInit
**
** Service ID           : None
**
** Description          : This service initializes the hardware for all the
**                        configured channels based on the measurement mode.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Non Reentrant
**
** Input Parameters     : ConfigPtr
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpTAUUnitConfig, Icu_GpSignalMeasurementData.
**
** Functions invoked    : None
**
** Registers Used       : TAUJnTT, TAUJnTPS, TAUJnBRS.
**
** Reference ID         : ICU_DUD_ACT_047, ICU_DUD_ACT_047_GBL002, ICU_DUD_ACT_047_GBL003,
** Reference ID         : ICU_DUD_ACT_047_REG001, ICU_DUD_ACT_047_REG002,
** Reference ID         : ICU_DUD_ACT_047_REG003, ICU_DUD_ACT_047_REG004
***********************************************************************************************************************/
#if (ICU_TAUJ_UNIT_USED == STD_ON)
#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Tauj_UnitInit(P2CONST(Icu_ConfigType, AUTOMATIC, ICU_APPL_CONST) ConfigPtr)         /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to the TAU configuration parameters */
  P2CONST(Icu_TAUUnitConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpTAUUnitConfig;
  /* Declare a Local variable to Unit Type */
  uint8 LucUnitType;
  /* Local variable to hold the channel number */
  uint8 LucCount;

  #if (ICU_PRESCALER_CONFIGURED == STD_ON)
  /* Pointer pointing to the TAUJ Unit control registers */
  P2VAR(volatile Icu_TAUJUnitOsRegs, AUTOMATIC, REGSPACE) LpTAUJUnitOsReg;

  P2VAR(volatile Icu_TAUJUnitUserRegs, AUTOMATIC, REGSPACE) LpTAUJUnitUserReg;
  #endif

  /* Update the global pointer with the first channel's address of Signal Measurement mode type channel's RAM data */

  #if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)
  Icu_GpSignalMeasurementData = (P2VAR(volatile Icu_SignalMeasurementModeRamDataType, ICU_VAR_NO_INIT,                  /* PRQA S 0316 # JV-01 */
                                                                      ICU_APPL_CONST))ConfigPtr->pSignalMeasureAddress; /* PRQA S 2814 # JV-01 */
  #endif
  /* Store the global pointer to first Unit Configuration */
  Icu_GpTAUUnitConfig = ConfigPtr->pTAUUnitConfig;                                                                      /* PRQA S 0317 # JV-01 */

  /* Update the Timer channel configuration pointer to point to the current channel */
  LpTAUUnitConfig = Icu_GpTAUUnitConfig;
  /* Update the TAU configuration pointer to point to the current TAU */
  for (LucCount = (uint8)ICU_ZERO; LucCount < (uint8)ICU_TOTAL_TAU_UNITS_CONFIGURED; LucCount++)
  {
    /* Read the Unit Type for given unit  */
    LucUnitType = LpTAUUnitConfig->ucIcuUnitType;                                                                       /* PRQA S 2814 # JV-01 */
    /* Invoke low level driver for initializing the hardware */
    if(ICU_HW_TAUJ == LucUnitType)
    {
      LpTAUUnitConfig = &Icu_GpTAUUnitConfig[LucCount];                                                                 /* PRQA S 2824 # JV-01 */

      /* Initialize pointer to the base address of the current timer unit */
      #if (ICU_PRESCALER_CONFIGURED == STD_ON)
      LpTAUJUnitOsReg = (P2VAR(volatile Icu_TAUJUnitOsRegs, AUTOMATIC, REGSPACE))LpTAUUnitConfig->pTAUnitOsCntlRegs;    /* PRQA S 0310, 2814, 3305 # JV-01, JV-01, JV-01 */
                               
      LpTAUJUnitUserReg = (P2VAR(volatile Icu_TAUJUnitUserRegs, AUTOMATIC,                                              /* PRQA S 0310, 3305 # JV-01, JV-01 */
                                                                         REGSPACE))LpTAUUnitConfig->pTAUnitUserCntlRegs;

      /* Set the values of prescaler and baud rate to TPS and BRS registers respectively */
      if ((uint8)ICU_TRUE == LpTAUUnitConfig->ucConfigurePrescaler)
      {
        LpTAUJUnitUserReg->ucTAUJnTT = ICU_TAUJ_TIMER_RESET_VAL;                                                        /* PRQA S 2814 # JV-01 */
        LpTAUJUnitOsReg->usTAUJnTPS = LpTAUUnitConfig->usPrescaler;                                                     /* PRQA S 2814 # JV-01 */
        #if (ICU_TAUJ_UNIT_CONFIG == STD_ON)
        LpTAUJUnitOsReg->ucTAUJnBRS = LpTAUUnitConfig->ucBaudRate;
        #endif
      } /* End of ICU_TRUE == LpTAUUnitConfig->ucConfigurePrescaler */
      else
      {
        /* Set the bit corresponding to the channel number in TAUJnTT register */
        LpTAUJUnitUserReg->ucTAUJnTT = (uint8)LpTAUUnitConfig->usTAUChannelMaskValue;
      }
      #endif /* End of (ICU_PRESCALER_CONFIGURED == STD_ON) */
    } /* else No action required */
    /* Increment the pointer to the next unit */
    LpTAUUnitConfig++;                                                                                                  /* PRQA S 2824 # JV-01 */
  }
} /* End of Internal function Icu_HW_Tauj_UnitInit */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (ICU_TAUJ_UNIT_USED == STD_ON) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Tauj_ChannelInit
**
** Service ID           : None
**
** Description          : This service initializes the hardware for all the
**                        configured channels based on the measurement mode.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Non Reentrant
**
** Input Parameters     : LucChannelIndex
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelConfig, Icu_GpChannelUserRegConfig, Icu_GpChannelRamData.
**
** Functions invoked    : Icu_HW_Tauj_TimestampInit, Icu_HW_Tauj_SignalMeasurementInit, RH850_SV_SET_ICR_SYNCP
**
** Registers Used       : TAUJnCMORm, TAUJnCMURm, TAUJnCSCm.
**
** Reference ID         : ICU_DUD_ACT_048, ICU_DUD_ACT_048_REG001, ICU_DUD_ACT_048_REG002, ICU_DUD_ACT_048_REG003
** Reference ID         : ICU_DUD_ACT_048_REG004
** Reference ID         : ICU_DUD_ACT_048_GBL003, ICU_DUD_ACT_048_GBL004, ICU_DUD_ACT_048_GBL001
** Reference ID         : ICU_DUD_ACT_048_GBL005, ICU_DUD_ACT_048_GBL006, ICU_DUD_ACT_048_GBL002
***********************************************************************************************************************/
#if (ICU_TAUJ_UNIT_USED == STD_ON)
#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Tauj_ChannelInit(const Icu_ChannelType LucChannelIndex)                             /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;
  /* Defining a pointer to the Channel Ram Data */
  P2VAR(volatile Icu_ChannelRamDataType, AUTOMATIC, ICU_CONFIG_DATA) LpRamData;

  /* Defining a pointer to the timer channel configuration parameters */
  P2CONST(Icu_ChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelUserRegConfig;

  /* Defining a pointer to the timer TAU channel configuration parameters */
  P2CONST(Icu_TAUChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpTAUChannelUserRegConfig;

  /* Pointer pointing to the TAUJ channel control registers */
  P2VAR(volatile Icu_TAUJChannelUserRegs, AUTOMATIC, REGSPACE) LpTAUJChannelReg;

  #if ((ICU_TIMESTAMP_API == STD_ON) || (ICU_SIGNAL_MEASUREMENT_API == STD_ON))
  /* Local variable to store the measurement mode of a channel */
  Icu_MeasurementModeType LenMeasurementMode;
  #endif

  /* Update the channel configuration pointer to point to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LucChannelIndex];
  /* Update the global pointer with the first channel's ram address */
  LpRamData = &Icu_GpChannelRamData[LucChannelIndex];
  /* Initialize all the configured Icu Channels */
  /* Initialize each channel status as idle */
  LpRamData->ucChannelStatus = (uint8)ICU_IDLE;                                                                         /* PRQA S 2814, 2844 # JV-01, JV-01 */
  #if ((ICU_TIMESTAMP_API == STD_ON) || (STD_ON == ICU_BUSY_OPERATION_DET_CHECK) || \
                                              (ICU_SIGNAL_MEASUREMENT_API == STD_ON) || (ICU_EDGE_DETECT_API == STD_ON))
  /* Initialize running status for each channel */
  LpRamData->ucChannelRunning = (uint8)ICU_FALSE;
  #endif

  #if ((ICU_DISABLE_WAKEUP_API == STD_ON) || (ICU_ENABLE_WAKEUP_API == STD_ON) || \
                                                                               (ICU_WAKEUP_FUNCTIONALITY_API == STD_ON))
  /* Disable each channel wakeup from sleep mode */
  LpRamData->ucWakeupEnable = (uint8)ICU_FALSE;
  /* Initialize wakeup occurrence for each channel */
  LpRamData->ucWakeupOccurrence = (uint8)ICU_FALSE;
  #endif
  /* Disable notification for each channel */
  LpRamData->ucNotificationEnable = (uint8)ICU_FALSE;
  /* Initialize result complete status for each channel */
  LpRamData->blResultComplete = (uint8)ICU_FALSE;

  /* Update the Timer channel configuration pointer to point to the current channel */
  LpChannelUserRegConfig = &Icu_GpChannelUserRegConfig[LucChannelIndex];

  LpTAUChannelUserRegConfig = Icu_GpChannelUserRegConfig[LucChannelIndex].pHWChannelUserRegConfig;                      /* PRQA S 0317 # JV-01 */

  #if ((ICU_TIMESTAMP_API == STD_ON) || (ICU_SIGNAL_MEASUREMENT_API == STD_ON))
  /* Read the channel's measurement mode */
  LenMeasurementMode = (Icu_MeasurementModeType)(LpChannelConfig->ucIcuMeasurementMode);                                /* PRQA S 2814, 2844, 4342 # JV-01, JV-01, JV-01 */
  #endif
  /* Initialize pointer to the base address of the current channel */
  LpTAUJChannelReg = (P2VAR(volatile Icu_TAUJChannelUserRegs, AUTOMATIC, REGSPACE)) LpChannelConfig->pCntlRegs;         /* PRQA S 0316 # JV-01 */

  /* Read the value of Channel Mode OS Register configured */
  *(LpTAUChannelUserRegConfig->pCMORReg) = LpTAUChannelUserRegConfig->usChannelModeOSRegSettings;                       /* PRQA S 2814 # JV-01 */

  /* Read the value of Channel Mode User Register configured */
  LpTAUJChannelReg->ucTAUJnCMURm = LpTAUChannelUserRegConfig->ucChannelModeUserRegSettings;                             /* PRQA S 2814 # JV-01 */

  /* Reset Channel Status Clear Trigger Register */
  LpTAUJChannelReg->ucTAUJnCSCm = ICU_TAUJ_CSC_RESET_VAL;

  if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)
  {
    /* Disable the Interrupt processing of the current channel */
    RH850_SV_SET_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_DISABLE_EIMK_MASK);                              /* PRQA S 0751, 2814 # JV-01, JV-01 */
  } /* else No action required */

  #if ((ICU_TIMESTAMP_API == STD_ON) || (ICU_SIGNAL_MEASUREMENT_API == STD_ON))
  if ((Icu_MeasurementModeType)ICU_MODE_TIMESTAMP == LenMeasurementMode)
  {
    #if (ICU_TIMESTAMP_API == STD_ON)
    /* Configure the channel in Timestamp Mode */
    Icu_HW_Tauj_TimestampInit(LpChannelUserRegConfig);
    #endif
  }
  #if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)
  else if ((Icu_MeasurementModeType)ICU_MODE_SIGNAL_MEASUREMENT == LenMeasurementMode)                                  /* PRQA S 2004 # JV-01 */
  {
    /* Configure the channel in Signal Measurement Mode */
    Icu_HW_Tauj_SignalMeasurementInit(LpChannelUserRegConfig);
  } /* else No action required */
  #endif
  #endif /* End of ((ICU_TIMESTAMP_API == STD_ON) || (ICU_SIGNAL_MEASUREMENT_API == STD_ON)) */
} /* End of Internal function Icu_HW_Tauj_ChannelInit */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (ICU_TAUJ_UNIT_USED == STD_ON) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Tauj_UnitDeInit
**
** Service ID           : None
**
** Description          : This service De-Initializes the hardware for all the
**                        configured channels based on the measurement mode.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Non Reentrant
**
** Input Parameters     : None
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpTAUUnitConfig
**
** Functions invoked    : None
**
** Registers Used       : TAUJnTT.
**
** Reference ID         : ICU_DUD_ACT_049, ICU_DUD_ACT_049_REG001
***********************************************************************************************************************/
#if (ICU_TAUJ_UNIT_USED == STD_ON)
#if (ICU_DE_INIT_API == STD_ON)

#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Tauj_UnitDeInit(void)                                                               /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to the TAU configuration parameters */
  P2CONST(Icu_TAUUnitConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpTAUUnitConfig;

  /* Pointer pointing to the TAUJ channel control registers */
  P2VAR(volatile Icu_TAUJUnitUserRegs, AUTOMATIC, REGSPACE) LpTAUJUnitUserReg;

  /* Declare a Local variable to Timer Type */
  VAR(uint8, AUTOMATIC) LucTimerType;

  /* Declare a Local variable to counting loop */
  VAR(uint8, AUTOMATIC) LucCount;

  /* Update the TAU configuration pointer to point to the current TAU */
  LpTAUUnitConfig = Icu_GpTAUUnitConfig;

  for (LucCount = (uint8)ICU_ZERO; LucCount < (uint8)ICU_TOTAL_TAU_UNITS_CONFIGURED; LucCount++)
  {
    /* Read the Timer Type for given channel */
    LucTimerType = LpTAUUnitConfig[LucCount].ucIcuUnitType;                                                             /* PRQA S 2824 # JV-01 */
    if (ICU_HW_TAUJ == LucTimerType)
    {
      /* Initialize pointer to the base address of the current channel */
      LpTAUJUnitUserReg =
            (P2VAR(volatile Icu_TAUJUnitUserRegs, AUTOMATIC, REGSPACE))(LpTAUUnitConfig[LucCount].pTAUnitUserCntlRegs); /* PRQA S 0310, 3305 # JV-01, JV-01 */
                                                                            
      /* Set the bit corresponding to the channel number in TAUJnTT register */
      LpTAUJUnitUserReg->ucTAUJnTT = (uint8)(LpTAUUnitConfig[LucCount].usTAUChannelMaskValue);                          /* PRQA S 2814 # JV-01 */
    } /* else No action required */
  }
} /* End of Internal function Icu_HW_Tauj_UnitDeInit */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* End of (ICU_DE_INIT_API == STD_ON) */
#endif /* End of (ICU_TAUJ_UNIT_USED == STD_ON) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Tauj_ChannelDeInit
**
** Service ID           : None
**
** Description          : This service De-Initializes the hardware for all the
**                        configured channels based on the measurement mode.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Non Reentrant
**
** Input Parameters     : LucChannelIndex
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelConfig, Icu_GpChannelUserRegConfig, Icu_GpChannelRamData.
**
** Functions invoked    : None
**
** Registers Used       : TAUJnCMORm, TAUJnCDRm, TAUJnCSCm, TAUJnCMURm.
**
** Reference ID         : ICU_DUD_ACT_050, ICU_DUD_ACT_050_REG001, ICU_DUD_ACT_050_REG002, ICU_DUD_ACT_050_REG003
** Reference ID         : ICU_DUD_ACT_050_REG004, ICU_DUD_ACT_050_GBL001, ICU_DUD_ACT_050_GBL002, ICU_DUD_ACT_050_GBL003
** Reference ID         : ICU_DUD_ACT_050_GBL004, ICU_DUD_ACT_050_GBL005, ICU_DUD_ACT_050_GBL006, ICU_DUD_ACT_050_REG005
***********************************************************************************************************************/
#if (ICU_TAUJ_UNIT_USED == STD_ON)
#if (ICU_DE_INIT_API == STD_ON)

#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Tauj_ChannelDeInit(const Icu_ChannelType LucChannelIndex)                           /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to the timer TAU channel configuration parameters */
  P2CONST(Icu_TAUChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpTAUChannelUserRegConfig;
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;
  /* Defining a pointer to the Timer registers */
  P2VAR(volatile Icu_ChannelRamDataType, AUTOMATIC, ICU_CONFIG_DATA) LpRamData;

  /* Pointer pointing to the TAUJ channel control registers */
  P2VAR(volatile Icu_TAUJChannelUserRegs, AUTOMATIC, REGSPACE) LpTAUJChannelReg;

  /* Update the channel configuration pointer to point to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LucChannelIndex];
  LpRamData = &Icu_GpChannelRamData[LucChannelIndex];
  /* Initialize each channel status as idle */
  LpRamData->ucChannelStatus = (uint8)ICU_IDLE;                                                                         /* PRQA S 2814, 2844 # JV-01, JV-01 */
  #if ((ICU_DISABLE_WAKEUP_API == STD_ON) || (ICU_ENABLE_WAKEUP_API == STD_ON) || \
                                                                               (ICU_WAKEUP_FUNCTIONALITY_API == STD_ON))
  /* Disable each channel wakeup from sleep mode */
  LpRamData->ucWakeupEnable = (uint8)ICU_FALSE;
  /* Initialize wakeup occurrence for each channel */
  LpRamData->ucWakeupOccurrence = (uint8)ICU_FALSE;
  #endif
  /* Disable notification for each channel */
  LpRamData->ucNotificationEnable = (uint8)ICU_FALSE;
  /* Initialize result complete status for each channel */
  LpRamData->blResultComplete = (uint8)ICU_FALSE;
  #if ((ICU_TIMESTAMP_API == STD_ON) || (STD_ON == ICU_BUSY_OPERATION_DET_CHECK) || \
                                              (ICU_SIGNAL_MEASUREMENT_API == STD_ON) || (ICU_EDGE_DETECT_API == STD_ON))
  /* Initialize running status for each channel */
  LpRamData->ucChannelRunning = (uint8)ICU_FALSE;
  #endif

  LpTAUChannelUserRegConfig = Icu_GpChannelUserRegConfig[LucChannelIndex].pHWChannelUserRegConfig;                      /* PRQA S 0317 # JV-01 */

  /* Initialize pointer to the base address of the current channel */
  LpTAUJChannelReg = (P2VAR(volatile Icu_TAUJChannelUserRegs, AUTOMATIC, REGSPACE)) LpChannelConfig->pCntlRegs;         /* PRQA S 0316, 2814, 2844 # JV-01, JV-01, JV-01 */

  /* Reset the value of Channel Mode OS Register */
  *(LpTAUChannelUserRegConfig->pCMORReg) = ICU_TAUJ_CMOR_RESET_VAL;                                                     /* PRQA S 2814 # JV-01 */

  /* Reset the value of Channel Data Register */
  LpTAUJChannelReg->ulTAUJnCDRm = ICU_TAUJ_CDR_RESET_VAL;                                                               /* PRQA S 2814 # JV-01 */

  /* Reset the value of Channel Mode User Register */
  LpTAUJChannelReg->ucTAUJnCMURm = ICU_TAUJ_CMUR_RESET_VAL;

  /* Reset Channel Status Clear Trigger Register */
  LpTAUJChannelReg->ucTAUJnCSCm = ICU_TAUJ_CSC_RESET_VAL;

  if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)
  {
    /* Disabling the Interrupt processing */
    RH850_SV_SET_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_DISABLE_EIMK_MASK);                              /* PRQA S 0751, 2814 # JV-01, JV-01 */
  } /* else No action required */

} /* End of Internal function Icu_HW_Tauj_ChannelDeInit */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* End of (ICU_DE_INIT_API == STD_ON) */
#endif /* End of (ICU_TAUJ_UNIT_USED == STD_ON) */

/***********************************************************************************************************************
** Function Name         : Icu_HW_Tauj_TimestampInit
**
** Service ID            : None
**
** Description           : This service is used to initialize the channel configured for Timestamp Measurement mode.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LpChannelUserRegConfig
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variables      : Icu_GpTimeStampData.
**
** Functions invoked     : None
**
** Registers Used        : None
**
** Reference ID          : ICU_DUD_ACT_051, ICU_DUD_ACT_051_GBL001
***********************************************************************************************************************/
#if (ICU_TAUJ_UNIT_USED == STD_ON)
#if (ICU_TIMESTAMP_API == STD_ON)
#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE)
    Icu_HW_Tauj_TimestampInit(P2CONST(Icu_ChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelUserRegConfig) /* PRQA S 1505 # JV-01 */
{
  /* Local variable used to store the channel index */
  uint8 LucIndex;
  /* Read the Timestamp Channel ram index */
  LucIndex = LpChannelUserRegConfig->ucRamDataIndex;                                                                    /* PRQA S 2814 # JV-01 */

  Icu_GpTimeStampData[LucIndex].usTimestampIndex = (uint16)ICU_BUFFER_IDX_INIT_VAL;
} /* End of Internal function Icu_HW_Tauj_TimestampInit */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (ICU_TIMESTAMP_API == STD_ON) */
#endif /* End of (ICU_TAUJ_UNIT_USED == STD_ON) */

/***********************************************************************************************************************
** Function Name         : Icu_HW_Tauj_SignalMeasurementInit
**
** Service ID            : None
**
** Description           : This service initializes the channel configured in Signal Measurement Mode.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non Reentrant
**
** Input Parameters      : LpChannelUserRegConfig
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variables      : Icu_GpSignalMeasurementData.
**
** Functions invoked     : None
**
** Registers Used        : None
**
** Reference ID          : ICU_DUD_ACT_052, ICU_DUD_ACT_052_GBL001,
** Reference ID          : ICU_DUD_ACT_052_GBL002, ICU_DUD_ACT_052_GBL003, ICU_DUD_ACT_052_GBL004
***********************************************************************************************************************/
#if (ICU_TAUJ_UNIT_USED == STD_ON)
#if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)
#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE)Icu_HW_Tauj_SignalMeasurementInit                                                           /* PRQA S 1505 # JV-01 */
                             (P2CONST(Icu_ChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelUserRegConfig)
{
  /* Local variable used to store the Timer Instance */
  uint8 LucIndex;

  /* Read the Signal Measurement Channel ram index */
  LucIndex = LpChannelUserRegConfig->ucRamDataIndex;                                                                    /* PRQA S 2814 # JV-01 */

  /* Initialize the Signal Active time to zero */
  Icu_GpSignalMeasurementData[LucIndex].ulSignalActiveTime = (uint32)ICU_ACTIVE_TIME_RESET_VAL;                         /* PRQA S 2824 # JV-01 */
  /* Initialize the Signal Period time to zero */
  Icu_GpSignalMeasurementData[LucIndex].ulSignalPeriodTime = (uint32)ICU_PERIOD_TIME_RESET_VAL;
  /* Initialize the Signal Period time to zero */
  Icu_GpSignalMeasurementData[LucIndex].ulPrevSignalActiveTime = (uint32)ICU_ACTIVE_TIME_RESET_VAL;
  /* Initialize the Period measurement status */
  Icu_GpSignalMeasurementData[LucIndex].ucPeriodMeasurementStatus = (uint8)ICU_PERIOD_IDLE;
} /* End of Internal function Icu_HW_Tauj_SignalMeasurementInit */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (ICU_SIGNAL_MEASUREMENT_API == STD_ON) */
#endif /* End of (ICU_TAUJ_UNIT_USED == STD_ON) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Tauj_SetActivation
**
** Service ID           : None
**
** Description          : This service configures the hardware to the active edges of the requested channel.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Reentrant
**
** Input Parameters     : LddChannel, LenActiveEdge
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : NA
**
** Global Variables     : Icu_GpChannelConfig, Icu_GpChannelRamData.
**
** Functions invoked    : RH850_SV_CLEAR_ICR_SYNCP.
**
** Registers Used       : TAUJnCMURm
**
** Reference ID         : ICU_DUD_ACT_053, ICU_DUD_ACT_053_REG001, ICU_DUD_ACT_053_GBL001, ICU_DUD_ACT_053_REG002
***********************************************************************************************************************/
#if (ICU_TAUJ_UNIT_USED == STD_ON)
#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE)
                    Icu_HW_Tauj_SetActivation(const Icu_ChannelType LddChannel, const Icu_ActivationType LenActiveEdge) /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;

  /* Defining a pointer to point to the TAUJ registers */
  P2VAR(volatile Icu_TAUJChannelUserRegs, AUTOMATIC, REGSPACE) LpTAUJChannelReg;

  /* To store Channel Measurement Mode */
  Icu_MeasurementModeType LenMeasurementMode;
  /* Update the channel configuration pointer to point to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LddChannel];

  /* Read measurement mode */
  LenMeasurementMode = (Icu_MeasurementModeType)(LpChannelConfig->ucIcuMeasurementMode);                                /* PRQA S 2814, 2844, 4342 # JV-01, JV-01, JV-01 */

  /* TAUJ doesn't support edge counter mode. so, set the active edge for timestamp mode only */
  if (ICU_MODE_TIMESTAMP == LenMeasurementMode)
  {
    /* Initialize pointer to the base address of the current channel */
    LpTAUJChannelReg = (P2VAR(volatile Icu_TAUJChannelUserRegs, AUTOMATIC, REGSPACE)) LpChannelConfig->pCntlRegs;       /* PRQA S 0316 # JV-01 */

    LpTAUJChannelReg->ucTAUJnCMURm = (uint8)LenActiveEdge;                                                              /* PRQA S 2814 # JV-01 */
  } /* else No action required  */

  #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON)

  /* Check if the Interrupt register is present for particular channel */
  if (NULL_PTR != LpChannelConfig->pIntrCntlAddress)
  {
    /* Clear the pending interrupts */
    RH850_SV_CLEAR_ICR_SYNCP(16, LpChannelConfig->pIntrCntlAddress, ICU_CLEAR_PENDING_INTR_MASK);                       /* PRQA S 1281, 2814 # JV-01, JV-01 */
  } /* else No action required */
  /* End of  NULL_PTR != LpChannelConfig->pIntrCntlAddress */
  #endif /* End of ICU_CLEAR_PENDING_INTERRUPT == STD_ON */

  /* Initialize channel status as idle */
  Icu_GpChannelRamData[LddChannel].ucChannelStatus = (uint8)ICU_IDLE;

} /* End of Internal function Icu_HW_Tauj_SetActivation */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (ICU_TAUJ_UNIT_USED == STD_ON) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Tauj_StartCountMeasurement
**
** Service ID           : None
**
** Description          : This service routine starts the count measurement
**                        for starting edge counting or signal measurement or timestamp measurement.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Reentrant
**
** Input Parameters     : LddChannel
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelUserRegConfig, Icu_GpChannelConfig, Icu_GpTAUUnitConfig, 
**                        Icu_GpSignalMeasurementData, Icu_GpChannelRamData.
**
** Functions invoked    : ICU_ENTER_CRITICAL_SECTION, ICU_EXIT_CRITICAL_SECTION, Icu_HW_Tau_StartCountMeasurement.
**
** Registers Used       : TAUJnTT, TAUJnTS.
**
** Reference ID         : ICU_DUD_ACT_054, ICU_DUD_ACT_054_CRT001, ICU_DUD_ACT_054_CRT002, ICU_DUD_ACT_054_GBL001
** Reference ID         : ICU_DUD_ACT_054_REG001, ICU_DUD_ACT_054_REG002, ICU_DUD_ACT_054_GBL002
***********************************************************************************************************************/
#if (ICU_TAUJ_UNIT_USED == STD_ON)
#if ((ICU_TIMESTAMP_API == STD_ON) || (ICU_SIGNAL_MEASUREMENT_API == STD_ON))
#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Tauj_StartCountMeasurement(const Icu_ChannelType LddChannel)                        /* PRQA S 1532 # JV-01 */
{
  #if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)
  /* Local variable to store the RAM index */
  uint8 LucIndex;
  /* To store Channel Measurement Mode */
  Icu_MeasurementModeType LenMeasurementMode;
  #endif

  /* Defining a pointer to the TAU configuration parameters */
  P2CONST(Icu_TAUUnitConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpTAUUnitConfig;
  /* Defining a pointer to the timer channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;
  /* Defining a pointer to the timer channel configuration parameters */
  P2CONST(Icu_TAUChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpTAUChannelUserRegConfig;
  /* Defining a pointer to point to the TAUJ user registers */
  P2VAR(volatile Icu_TAUJUnitUserRegs, AUTOMATIC, REGSPACE) LpTAUJUnitUserReg;

  /* Read channel configuration pointer */
  LpChannelConfig = &Icu_GpChannelConfig[LddChannel];

  /* Invoke service start count measurement and enable interrupt*/
  Icu_HW_Tau_StartCountMeasurement(LddChannel);

  /* Read timer channel configuration pointer */
  LpTAUChannelUserRegConfig = Icu_GpChannelUserRegConfig[LddChannel].pHWChannelUserRegConfig;                           /* PRQA S 0317 # JV-01 */

  /* Update the TAU configuration pointer to point to the current TAU */
  LpTAUUnitConfig = &Icu_GpTAUUnitConfig[LpTAUChannelUserRegConfig->ucTimerUnitIndex];                                  /* PRQA S 2814, 2824 # JV-01, JV-01 */

  /* Initialize pointer to the base address of the current timer unit */
  LpTAUJUnitUserReg = (P2VAR(volatile Icu_TAUJUnitUserRegs, AUTOMATIC, REGSPACE)) LpTAUUnitConfig->pTAUnitUserCntlRegs; /* PRQA S 0310, 2814, 3305 # JV-01, JV-01, JV-01 */

  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Enter Critical Section */
  ICU_ENTER_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

  /* Set the bit status running of channel is ICU_TRUE*/
  Icu_GpChannelRamData[LddChannel].ucChannelRunning = (uint8)ICU_TRUE;

  /* Set the bit corresponding to the channel number in TAUJnTT register */
  LpTAUJUnitUserReg->ucTAUJnTT = (uint8)LpChannelConfig->usChannelMaskValue;                                            /* PRQA S 2814, 2844 # JV-01, JV-01 */
  /* Set the bit corresponding to the channel number in TAUJnTS register */
  LpTAUJUnitUserReg->ucTAUJnTS = (uint8)LpChannelConfig->usChannelMaskValue;

  #if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)

  LenMeasurementMode = (Icu_MeasurementModeType)(LpChannelConfig->ucIcuMeasurementMode);                                /* PRQA S 4342 # JV-01 */

  if (ICU_MODE_SIGNAL_MEASUREMENT == LenMeasurementMode)
  {
    /* Read the current activation edge from RAM */
    LucIndex = Icu_GpChannelUserRegConfig[LddChannel].ucRamDataIndex;

    /* Initialize period measurement status as idle */
    Icu_GpSignalMeasurementData[LucIndex].ucPeriodMeasurementStatus = (uint8)ICU_PERIOD_IDLE;                           /* PRQA S 2824 # JV-01 */
  } /* else No action required */
  #endif /* End of #if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)*/

  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Exit Critical Section */
  ICU_EXIT_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

} /* End of Internal function Icu_HW_Tauj_StartCountMeasurement */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of ((ICU_TIMESTAMP_API == STD_ON) || (ICU_SIGNAL_MEASUREMENT_API == STD_ON) */
#endif /* End of (ICU_TAUJ_UNIT_USED == STD_ON) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Tauj_StopCountMeasurement
**
** Service ID           : None
**
** Description          : This service routine stops the count measurement
**                        for starting edge counting or signal measurement or timestamp measurement.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Reentrant
**
** Input Parameters     : LddChannel
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelUserRegConfig, Icu_GpChannelConfig, Icu_GpTAUUnitConfig, 
**                        Icu_GpChannelRamData, Icu_GpSignalMeasurementData.
**
** Functions invoked    : ICU_ENTER_CRITICAL_SECTION, ICU_EXIT_CRITICAL_SECTION, Icu_HW_Tau_StopCountMeasurement.
**
** Registers Used       : TAUJnTT
**
** Reference ID         : ICU_DUD_ACT_055, ICU_DUD_ACT_055_CRT001, ICU_DUD_ACT_055_CRT002
** Reference ID         : ICU_DUD_ACT_055_REG001, ICU_DUD_ACT_055_GBL002, ICU_DUD_ACT_055_GBL001
***********************************************************************************************************************/
#if (ICU_TAUJ_UNIT_USED == STD_ON)
#if ((ICU_TIMESTAMP_API == STD_ON) || (ICU_SIGNAL_MEASUREMENT_API == STD_ON))
#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Tauj_StopCountMeasurement(const Icu_ChannelType LddChannel)                         /* PRQA S 1532 # JV-01 */
{
  #if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)
  /* Local variable to store the RAM index */
  uint8 LucIndex;
  /* To store Channel Measurement Mode */
  Icu_MeasurementModeType LenMeasurementMode;
  #endif

  /* Defining a pointer to the timer channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;
  /* Defining a pointer to the timer channel configuration parameters */
  P2CONST(Icu_TAUChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpTAUChannelUserRegConfig;
  /* Defining a pointer to the TAU configuration parameters */
  P2CONST(Icu_TAUUnitConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpTAUUnitConfig;

  /* Defining a pointer to point to the TAUJ registers */
  P2VAR(volatile Icu_TAUJUnitUserRegs, AUTOMATIC, REGSPACE) LpTAUJUnitUserReg;

  /* Read channel configuration pointer */
  LpChannelConfig = &Icu_GpChannelConfig[LddChannel];

  /* Invoke service stop count measurement and disable interrupt*/
  Icu_HW_Tau_StopCountMeasurement(LddChannel);

  /* Read timer channel configuration pointer */
  LpTAUChannelUserRegConfig = Icu_GpChannelUserRegConfig[LddChannel].pHWChannelUserRegConfig;                           /* PRQA S 0317 # JV-01 */

  /* Update the TAU configuration pointer to point to the current TAU */
  LpTAUUnitConfig = &Icu_GpTAUUnitConfig[LpTAUChannelUserRegConfig->ucTimerUnitIndex];                                  /* PRQA S 2814, 2824 # JV-01, JV-01 */

  /* Initialize pointer to the base address of the current timer unit */
  LpTAUJUnitUserReg = (P2VAR(volatile Icu_TAUJUnitUserRegs, AUTOMATIC, REGSPACE)) LpTAUUnitConfig->pTAUnitUserCntlRegs; /* PRQA S 0310, 2814, 3305 # JV-01, JV-01, JV-01 */

  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Enter Critical Section */
  ICU_ENTER_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

  /* Set bit flag channel running is ICU_FALSE */
  Icu_GpChannelRamData[LddChannel].ucChannelRunning = (uint8)ICU_FALSE;

  /* Set the bit corresponding to the channel number in TAUJnTT register */
  LpTAUJUnitUserReg->ucTAUJnTT = (uint8)LpChannelConfig->usChannelMaskValue;                                            /* PRQA S 2814, 2844 # JV-01, JV-01 */

  #if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)

  LenMeasurementMode = (Icu_MeasurementModeType)(LpChannelConfig->ucIcuMeasurementMode);                                /* PRQA S 4342 # JV-01 */

  if (ICU_MODE_SIGNAL_MEASUREMENT == LenMeasurementMode)
  {
    /* Read the current activation edge from RAM */
    LucIndex = Icu_GpChannelUserRegConfig[LddChannel].ucRamDataIndex;

    /* Initialize period measurement status as idle */
    Icu_GpSignalMeasurementData[LucIndex].ucPeriodMeasurementStatus = (uint8)ICU_PERIOD_IDLE;                           /* PRQA S 2824 # JV-01 */
  } /* else No action required */
  #endif /* End of #if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)*/

  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Exit Critical Section */
  ICU_EXIT_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

} /* End of Internal function Icu_HW_Tauj_StopCountMeasurement */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of ((ICU_TIMESTAMP_API == STD_ON) || (ICU_SIGNAL_MEASUREMENT_API == STD_ON) */
#endif /* End of (ICU_TAUJ_UNIT_USED == STD_ON) */

/***********************************************************************************************************************
** Function Name         : Icu_HW_Tauj_TimerIsr
**
** Service ID            : None
**
** Description           : This service routine invokes the required function
**                         based on the channel configuration for further calculations.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Non-Reentrant
**
** Input Parameters      : LddChannel
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variables      : Icu_GpChannelConfig, Icu_GpChannelUserRegConfig,
**                         Icu_GpChannelRamData, Icu_GenModuleMode, Icu_GpSignalMeasurementData.
**
** Functions invoked     : Icu_HW_Tauj_ServiceSignalMeasurement,
**                         Icu_HW_Tauj_ServiceTimestamp,
**                         ICU_ENTER_CRITICAL_SECTION, ICU_EXIT_CRITICAL_SECTION
**                         EcuM_CheckWakeup, ICU_DEM_REPORT_ERROR.
**
** Registers Used        : TAUDnCDRm.
**
** Reference ID          : ICU_DUD_ACT_044, ICU_DUD_ACT_044_CRT001, ICU_DUD_ACT_044_CRT002
** Reference ID          : ICU_DUD_ACT_044_GBL002, ICU_DUD_ACT_044_GBL003, ICU_DUD_ACT_044_GBL006
** Reference ID          : ICU_DUD_ACT_044_GBL004, ICU_DUD_ACT_044_GBL005, ICU_DUD_ACT_044_GBL001
***********************************************************************************************************************/
#if (ICU_TAUJ_UNIT_USED == STD_ON)
#define ICU_START_SEC_CODE_FAST
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_CODE_FAST) Icu_HW_Tauj_TimerIsr(const Icu_ChannelType LddChannel)                                        /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to point to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;

  #if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)
  /* Define a pointer to point to the timer channel configuration parameters */
  P2CONST(Icu_ChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelUserRegConfig;
  #endif

  #if (ICU_TIMESTAMP_API == STD_ON)
  /* Defining a pointer to point to the TAUJ registers */
  P2VAR(volatile Icu_TAUJChannelUserRegs, AUTOMATIC, REGSPACE) LpTAUJChannelReg;                                        /* PRQA S 3678 # JV-01 */
  #endif

  Icu_MeasurementModeType LenMeasurementMode;
  #if (ICU_TIMESTAMP_API == STD_ON)
  uint32 LulCapturedTimestampVal;
  #endif

  #if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)
  uint8 LucIndex;
  uint8 LucMeasureProperty;
  uint8 LucPeriodMeasurementStatus;
  /* Update the timer channel pointer to point to the current channel */
  LpChannelUserRegConfig = &Icu_GpChannelUserRegConfig[LddChannel];
  /* Read the edge count RAM index */
  LucIndex = LpChannelUserRegConfig->ucRamDataIndex;                                                                    /* PRQA S 2814, 2844 # JV-01, JV-01 */
  #endif /* End of #if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)*/

  /* Update the channel pointer to point to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LddChannel];

  /* Read the channel's measurement property */
  LenMeasurementMode = (Icu_MeasurementModeType)(LpChannelConfig->ucIcuMeasurementMode);                                /* PRQA S 2814, 2844, 4342 # JV-01, JV-01, JV-01 */
  #if (ICU_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if ((uint16)ICU_ZERO != (uint16)(*LpChannelConfig->pIntrCntlAddress & ICU_EIC_EIMK_MASK))                             /* PRQA S 2814 # JV-01 */
  {
    #if (ICU_E_INT_INCONSISTENT_CONFIGURED == STD_ON)
    ICU_DEM_REPORT_ERROR(ICU_E_INT_INCONSISTENT, DEM_EVENT_STATUS_FAILED);
    #endif
  }
  else
  #endif
  {
    /* Check whether the channel is configured for Signal Measurement */
    if (ICU_MODE_SIGNAL_MEASUREMENT == LenMeasurementMode)
    {
      #if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)
      /* Process Signal Measurement */
      Icu_HW_Tauj_ServiceSignalMeasurement(LddChannel);

      /* Read channel measurement property */
      LucMeasureProperty = LpChannelUserRegConfig->ucChannelProperties;
      #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
      /* Enter Critical Section */
      ICU_ENTER_CRITICAL_SECTION(ICU_RAM_DATA_PROTECTION);
      #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

      /* Check measurement property is period */
      if (ICU_PERIOD_CHANNEL_ENABLED_MASK == (uint8)(LucMeasureProperty & ICU_HIGH_LOW_TIME_MASK))
      {
        LucPeriodMeasurementStatus = Icu_GpSignalMeasurementData[LucIndex].ucPeriodMeasurementStatus;                   /* PRQA S 2824 # JV-01 */
        if ((uint8)ICU_PERIOD_IDLE == LucPeriodMeasurementStatus)
        {
          /* Set period measurement status as ongoing */
          Icu_GpSignalMeasurementData[LucIndex].ucPeriodMeasurementStatus = (uint8)ICU_PERIOD_ONGOING;
        }
        else /* if ((uint8)ICU_PERIOD_ONGOING == LucPeriodMeasurementStatus) */
        {
          /* Set channel input status as active */
          Icu_GpChannelRamData[LddChannel].ucChannelStatus = (uint8)ICU_ACTIVE;

          /* Set channel result status as avaible */
          Icu_GpChannelRamData[LddChannel].blResultComplete = (uint8)ICU_TRUE;
        }
      }
      else /* (LucMeasureProperty != (ICU_PERIOD_TIME)) */
      {
        /* Set channel input status as active */
        Icu_GpChannelRamData[LddChannel].ucChannelStatus = (uint8)ICU_ACTIVE;

        /* Set channel result status as avaible */
        Icu_GpChannelRamData[LddChannel].blResultComplete = (uint8)ICU_TRUE;
      }
      #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
      /* Exit Critical Section */
      ICU_EXIT_CRITICAL_SECTION(ICU_RAM_DATA_PROTECTION);
      #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

      #endif /* End of (ICU_SIGNAL_MEASUREMENT_API == STD_ON) */
    }
    /* Check whether the channel is configured for Timestamp */
    else if (ICU_MODE_TIMESTAMP == LenMeasurementMode)                                                                  /* PRQA S 2004 # JV-01 */
    {
      #if (ICU_TIMESTAMP_API == STD_ON)
      if ((uint8)ICU_ACTIVE == Icu_GpChannelRamData[LddChannel].ucChannelStatus)
      {
        /* Initialize pointer to the base address of the current channel */
        LpTAUJChannelReg = (P2VAR(volatile Icu_TAUJChannelUserRegs, AUTOMATIC, REGSPACE)) LpChannelConfig->pCntlRegs;   /* PRQA S 0316 # JV-01 */

        /* Calculate captured value */
        LulCapturedTimestampVal = (uint32)(LpTAUJChannelReg->ulTAUJnCDRm + (uint32)ICU_ONE);                            /* PRQA S 2814, 3383 # JV-01, JV-01 */

        /* Process Timestamp */
        Icu_HW_Tauj_ServiceTimestamp(LddChannel, LulCapturedTimestampVal);
      } /* else No action required */

      #endif /* End of (ICU_TIMESTAMP_API == STD_ON) */
    } /* else No action required */
    /* End of (ICU_MODE_SIGNAL_MEASUREMENT == LenMeasurementMode) */

    #if (ICU_REPORT_WAKEUP_SOURCE == STD_ON)
    /* If Module was in SLEEP mode and reporting wakeup is enabled */
    if ((ICU_MODE_SLEEP == Icu_GenModuleMode) &&
                                    ((uint8)ICU_FALSE == LpChannelConfig->ucIcuDisableEcumWakeupNotification) &&
                                                             (ICU_NOWAKEUP != LpChannelConfig->ucEcuMChannelWakeupInfo))
    {
      /* Set the wakeup status to true */
      Icu_GpChannelRamData[LddChannel].ucWakeupOccurrence = (uint8)ICU_TRUE;

      /* Report Wakeup Event to EcuM */
      EcuM_CheckWakeup((EcuM_WakeupSourceType)(ICU_DOUBLE_ONE << (LpChannelConfig->ucEcuMChannelWakeupInfo)));
    } /* else No action required */
    #endif /* End of (ICU_REPORT_WAKEUP_SOURCE == STD_ON) */
  }
} /* End of Internal function Icu_HW_Tauj_TimerIsr */

#define ICU_STOP_SEC_CODE_FAST
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* End of (ICU_TAUJ_UNIT_USED == STD_ON) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Tauj_ServiceSignalMeasurement
**
** Service ID           : None
**
** Description          : This service routine calculates the channel's Signal
**                        Time (Low, High, Period or Duty) based on its configuration.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Reentrant
**
** Input Parameters     : LddChannel
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelConfig, Icu_GpSignalMeasurementData, Icu_GpChannelUserRegConfig.
**
** Functions invoked    : ICU_ENTER_CRITICAL_SECTION, ICU_EXIT_CRITICAL_SECTION.
**
** Registers Used       : TAUJnCDRm
**
** Reference ID         : ICU_DUD_ACT_045, ICU_DUD_ACT_045_CRT001,
** Reference ID         : ICU_DUD_ACT_045_GBL001, ICU_DUD_ACT_045_GBL002, ICU_DUD_ACT_045_CRT002
***********************************************************************************************************************/
#if (ICU_TAUJ_UNIT_USED == STD_ON)
#if (ICU_SIGNAL_MEASUREMENT_API == STD_ON)
#define ICU_START_SEC_CODE_FAST
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_CODE_FAST) Icu_HW_Tauj_ServiceSignalMeasurement(const Icu_ChannelType LddChannel)                        /* PRQA S 1505 # JV-01 */
{
  /* Defining a pointer to point to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;

  /* Define a pointer to point to the timer channel configuration parameters */
  P2CONST(Icu_ChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelUserRegConfig;

  /* Defining a pointer to point to the TAUJ registers */
  P2VAR(volatile Icu_TAUJChannelUserRegs, AUTOMATIC, REGSPACE) LpTAUJChannelReg;                                        /* PRQA S 3678 # JV-01 */

  uint8 LucMeasureProperty;
  uint8 LucRamIndex;

  /* Read channel configuration pointer */
  LpChannelConfig = &Icu_GpChannelConfig[LddChannel];

  /* Read timer channel configuration pointer */
  LpChannelUserRegConfig = &Icu_GpChannelUserRegConfig[LddChannel];

  /* Read the channel ram index */
  LucRamIndex = LpChannelUserRegConfig->ucRamDataIndex;                                                                 /* PRQA S 2814, 2844 # JV-01, JV-01 */
  /* Read channel measurement property */
  LucMeasureProperty = LpChannelUserRegConfig->ucChannelProperties;

  /* Initialize pointer to the base address of the current channel */
  LpTAUJChannelReg = (P2VAR(volatile Icu_TAUJChannelUserRegs, AUTOMATIC, REGSPACE)) LpChannelConfig->pCntlRegs;         /* PRQA S 0316, 2814, 2844 # JV-01, JV-01, JV-01 */

  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Enter Critical Section */
  ICU_ENTER_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

  if ((ICU_HIGH_TIME_MASK == (uint8)(LucMeasureProperty & ICU_HIGH_LOW_TIME_MASK)) ||
                                            (ICU_LOW_TIME_MASK == (uint8)(LucMeasureProperty & ICU_HIGH_LOW_TIME_MASK)))
  /* Calculate Active time */
  {
    Icu_GpSignalMeasurementData[LucRamIndex].ulSignalActiveTime =                                                       /* PRQA S 2824 # JV-01 */
                                                           (uint32)((LpTAUJChannelReg->ulTAUJnCDRm) + (uint32)ICU_ONE); /* PRQA S 2814, 3383 # JV-01, JV-01 */
  }
  else /* (LucMeasureProperty == (ICU_PERIOD_TIME)) */
  {
    Icu_GpSignalMeasurementData[LucRamIndex].ulPrevSignalActiveTime =
                                                            Icu_GpSignalMeasurementData[LucRamIndex].ulSignalActiveTime;
    /* Calculate Period time */
    Icu_GpSignalMeasurementData[LucRamIndex].ulSignalPeriodTime =
                                                             (uint32)(LpTAUJChannelReg->ulTAUJnCDRm + (uint32)ICU_ONE); /* PRQA S 3383 # JV-01 */
  }

  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Exit Critical Section */
  ICU_EXIT_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

} /* End of Internal function Icu_HW_Tauj_ServiceSignalMeasurement */

#define ICU_STOP_SEC_CODE_FAST
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (ICU_SIGNAL_MEASUREMENT_API == STD_ON) */
#endif /* End of (ICU_TAUJ_UNIT_USED == STD_ON) */

/***********************************************************************************************************************
** Function Name         : Icu_HW_Tauj_ServiceTimestamp
**
** Service ID            : None
**
** Description           : This service routine captures the channel's Timestamp
**                         data based on its active edge configuration.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : LddChannel, LulCapturedTimestampVal
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Preconditions         : None
**
** Global Variables      : Icu_GpChannelConfig, Icu_GpTimeStampData, Icu_GpChannelUserRegConfig, Icu_GpChannelRamData.
**
** Functions invoked     : pIcuNotificationPointer.
**
** Registers Used        : None
**
** Reference ID          : ICU_DUD_ACT_046, ICU_DUD_ACT_046_GBL001, ICU_DUD_ACT_046_GBL002
***********************************************************************************************************************/
#if (ICU_TAUJ_UNIT_USED == STD_ON)
#if (ICU_TIMESTAMP_API == STD_ON)
#define ICU_START_SEC_CODE_FAST
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_CODE_FAST)
                   Icu_HW_Tauj_ServiceTimestamp(const Icu_ChannelType LddChannel, const uint32 LulCapturedTimestampVal) /* PRQA S 1505 # JV-01 */
{
  #if (ICU_NOTIFICATION_CONFIG == STD_ON)
  /* Defining a pointer to point to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;
  #endif /* if (ICU_NOTIFICATION_CONFIG == STD_ON) */
  /* Defining a pointer to point to the timer channel configuration parameters */
  P2CONST(Icu_ChannelUserRegConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelUserRegConfig;

  P2VAR(Icu_TimeStampChannelRamDataType, AUTOMATIC, ICU_CONFIG_DATA) LpTimestampdata;                                   /* PRQA S 3432 # JV-01 */

  /* Local variable used to store the ram index of the channel */
  uint8 LucRamIndex;
  #if (ICU_NOTIFICATION_CONFIG == STD_ON)
  /* Update the channel pointer to point to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LddChannel];
  #endif /* if (ICU_NOTIFICATION_CONFIG == STD_ON) */
  /* Load timer channel configuration pointer and channel properties */
  LpChannelUserRegConfig = &Icu_GpChannelUserRegConfig[LddChannel];
  /* Read channel ram index */
  LucRamIndex = LpChannelUserRegConfig->ucRamDataIndex;                                                                 /* PRQA S 2814, 2844 # JV-01, JV-01 */

  /* Read the Timestamp channel data pointer */
  LpTimestampdata = &Icu_GpTimeStampData[LucRamIndex];

  /* Check if timestamp capturing reached end of the buffer */
  if ((LpTimestampdata->usTimestampIndex) < (LpTimestampdata->usBufferSize))                                            /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    /* Update the buffer pointer with current timestamp */
    *(LpTimestampdata->pBufferPointer) = LulCapturedTimestampVal;
    (LpTimestampdata->usTimestampIndex)++;
    (LpTimestampdata->pBufferPointer)++;                                                                                /* PRQA S 2824 # JV-01 */

    /* Check whether the timestamp index has reached the end of buffer */
    if ((LpTimestampdata->usTimestampIndex) >= (LpTimestampdata->usBufferSize))
    {
      /*  Check if buffer is configured as circular */
      if (ICU_TS_CIRCULAR_BUFFER_MASK ==
                                    (uint8)(LpChannelUserRegConfig->ucChannelProperties & ICU_TS_CIRCULAR_BUFFER_MASK))
      {
        /* Reset buffer pointer and index */
        LpTimestampdata->usTimestampIndex = (uint16)ICU_BUFFER_IDX_INIT_VAL;
        LpTimestampdata->pBufferPointer = LpTimestampdata->pBufferPointer - LpTimestampdata->usBufferSize;              /* PRQA S 0488 # JV-01 */
      } /* else No action required */
    } /* else No action required */

    /* Notify if the configured number of timestamps are captured and notification is enabled */
    #if (ICU_NOTIFICATION_CONFIG == STD_ON)

    if ((uint8)ICU_TRUE == Icu_GpChannelRamData[LddChannel].ucNotificationEnable)
    {
      (LpTimestampdata->usTimestampsCounter)++;                                                                         /* PRQA S 3383 # JV-01 */
      if ((LpTimestampdata->usTimestampsCounter) == (LpTimestampdata->usNotifyInterval))
      {
        /* Reset the number of timestamp captured counter */
        LpTimestampdata->usTimestampsCounter = (uint16)ICU_TIMESTAMP_RESET_CNT_VAL;
        /* Check if the function pointer address is present for particular channel */
        if (NULL_PTR != LpChannelConfig->pIcuNotificationPointer)                                                       /* PRQA S 2814, 2844 # JV-01, JV-01 */
        {
          /* callback notification */
          LpChannelConfig->pIcuNotificationPointer();
        } /* else No action required */
      } /* else No action required */
    } /* else No action required*/

#endif /* if (ICU_NOTIFICATION_CONFIG == STD_ON) */
  } /* else No action required */

} /* End of Internal function Icu_HW_Tauj_ServiceTimestamp */

#define ICU_STOP_SEC_CODE_FAST
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (ICU_TIMESTAMP_API == STD_ON) */
#endif /* End of (ICU_TAUJ_UNIT_USED == STD_ON) */

/***********************************************************************************************************************
** Function Name        : Icu_HW_Tauj_SetMode
**
** Service ID           : None
**
** Description          : This service sets the operating mode of the ICU
**                        Driver.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Non Reentrant
**
** Input Parameters     : LucChannel, LenMode.
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Preconditions        : None
**
** Global Variables     : Icu_GpChannelConfig, Icu_GpChannelRamData.
**
** Functions invoked    : ICU_ENTER_CRITICAL_SECTION, ICU_EXIT_CRITICAL_SECTION,
**                        RH850_SV_SET_ICR_SYNCP, RH850_SV_CLEAR_ICR_SYNCP.
**
** Registers Used       : EICn.
**
** Reference ID         : ICU_DUD_ACT_093, ICU_DUD_ACT_093_CRT001, ICU_DUD_ACT_093_CRT004
** Reference ID         : ICU_DUD_ACT_093_REG002, ICU_DUD_ACT_093_REG003, ICU_DUD_ACT_093_REG001
***********************************************************************************************************************/
#if (ICU_SET_MODE_API == STD_ON)

#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, ICU_PRIVATE_CODE) Icu_HW_Tauj_SetMode(const Icu_ChannelType LucChannel, const Icu_ModeType LenMode)          /* PRQA S 1503, 1532, 3408 # JV-01, JV-01, JV-01 */
{
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_ChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpChannelConfig;

  /* Update the channel configuration pointer to the current channel */
  LpChannelConfig = &Icu_GpChannelConfig[LucChannel];

  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Enter Critical Section */
  ICU_ENTER_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

  /* Check for the current wakeup status of the channel */
  if (
  #if ((ICU_ENABLE_WAKEUP_API == STD_ON) || (ICU_NOTIFY_WAKEUP_INTERRUPT == STD_ON))
  (ICU_FALSE == (uint8)Icu_GpChannelRamData[LucChannel].ucWakeupEnable) &&
  #endif
  (NULL_PTR != LpChannelConfig->pIntrCntlAddress))                                                                      /* PRQA S 2814, 2844 # JV-01, JV-01 */
  {
    if (ICU_MODE_SLEEP == LenMode)
    {
      /* Disable Interrupt */
      RH850_SV_SET_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_DISABLE_EIMK_MASK);                            /* PRQA S 0751, 2814 # JV-01, JV-01 */

    } /* End of LenMode = ICU_MODE_SLEEP */

    else /* LenMode = ICU_MODE_NORMAL */
    {
      #if (ICU_CLEAR_PENDING_INTERRUPT == STD_ON)
      /* Check if the Interrupt register is present for particular channel */
      /* Clear the pending interrupts */
      RH850_SV_CLEAR_ICR_SYNCP(16, LpChannelConfig->pIntrCntlAddress, ICU_CLEAR_PENDING_INTR_MASK);                     /* PRQA S 1281 # JV-01 */
      #endif

      /* Enable Interrupt for on-going channels */
      if ((uint8)ICU_TRUE == Icu_GpChannelRamData[LucChannel].ucChannelRunning)
      {
        RH850_SV_CLEAR_ICR_SYNCP(8, (LpChannelConfig->pIntrCntlAddress), ICU_ENABLE_EIMK_MASK);                         /* PRQA S 0751 # JV-01 */
      } /* else No action required */

    } /* End of LenMode = ICU_MODE_NORMAL */

  } /* End of (ICU_FALSE == (uint8)Icu_GpChannelRamData[LucChannel].ucWakeupEnable) && 
                                                                        NULL_PTR != LpChannelConfig->pIntrCntlAddress */
  /* else No action required */
  #if (ICU_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Exit Critical Section */
  ICU_EXIT_CRITICAL_SECTION(ICU_INTERRUPT_CONTROL_PROTECTION);
  #endif /* End of (ICU_CRITICAL_SECTION_PROTECTION == STD_ON) */

} /* End of Internal function Icu_HW_Pin_SetMode */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* End of (ICU_SET_MODE_API == STD_ON) */

/***********************************************************************************************************************
** Function Name         : Icu_HW_Tauj_GetTAUInCountValue
**
** Service ID            : NA
**
** Description           : This API service returns the counter value of TAU timer.
**
** Sync/Async            : NA
**
** Reentrancy            : NA
**
** Input Parameters      : LddChannel
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Icu_CounterValueType
**
** Preconditions         : The Icu Driver must be initialized, ICU_GET_TAU_COUNT_VALUE_API must be STD_ON
**
** Global Variables      : Icu_GpChannelConfig
**
** Functions invoked     : None
**
** Registers Used        : None
**
** Reference ID          : ICU_DUD_ACT_056
***********************************************************************************************************************/
#if (ICU_TAUJ_UNIT_USED == STD_ON)
#if (ICU_GET_TAU_COUNT_VALUE_API == STD_ON)

#define ICU_START_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(Icu_CounterValueType, ICU_PRIVATE_CODE) Icu_HW_Tauj_GetTAUInCountValue(const Icu_ChannelType LddChannel)           /* PRQA S 1532 # JV-01 */
{
  /* Defining a pointer to the channel configuration parameters */
  P2CONST(Icu_TAUChannelConfigType, AUTOMATIC, ICU_CONFIG_DATA) LpTAUChannelConfig;

  /* Local variable to store the return level value  */
  Icu_CounterValueType LddTAUInCounterValue;

  /* Initialize the return value */
  LddTAUInCounterValue = ICU_TAU_CNT_INIT_VAL;

  LpTAUChannelConfig = Icu_GpChannelConfig[LddChannel].pHWIPChannelConfig;                                              /* PRQA S 0317 # JV-01 */

  /* Get the TAUIn counter value of particular channel */
  if (NULL_PTR != LpTAUChannelConfig->pTAUJCNTAddress)                                                                  /* PRQA S 2814 # JV-01 */
  {
    LddTAUInCounterValue = *(LpTAUChannelConfig->pTAUJCNTAddress);
  } /* else No action required */

  /* Return the TAUIn counter value of particular channel */
  return (LddTAUInCounterValue);
} /* End of API Icu_HW_Tauj_GetTAUInCountValue */

#define ICU_STOP_SEC_PRIVATE_CODE
#include "Icu_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* End of (ICU_GET_TAU_COUNT_VALUE_API == STD_ON) */
#endif /* End of (ICU_TAUJ_UNIT_USED == STD_ON) */

/***********************************************************************************************************************
**                                                    End of File                                                     **
***********************************************************************************************************************/
