/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Spi_MSPI_Irq.c                                                                                      */
/* SW-VERSION   = 1.5.0                                                                                               */
/*====================================================================================================================*/
/*                                                     COPYRIGHT                                                      */
/*====================================================================================================================*/
/* (c) 2020,2021 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* This file contains the ISR implementations for SPI(MSPI)                                                           */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s) and/or */
/* the Application.                                                                                                   */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs of  */
/* program errors, compliance with applicable laws, damage to or loss of data, programs or equipment, and             */
/* unavailability or interruption of operations.                                                                      */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*              Devices:        X2x                                                                                   */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                              Revision Control History                                              **
***********************************************************************************************************************/
/* 1.3.2:  04/09/2021  : Replace the inclusion of Spi_MemMap.h with Spi_Mapping.h
 * 1.3.1:  02/07/2021  : Add QAC message 9.5.0
 *                       Format source code to 120 characters
 *                       Improve Violation tag (remove START/END)
 *         19/05/2021  : Update QAC contents.
 * 1.2.0:  30/07/2020  : Release
 *         28/07/2020  : Add QAC 9.3.1 comment.
 * 1.1.0:  19/06/2020  : Release
 * 1.0.0:  24/03/2020  : Initial Version
 */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                  Include Section                                                   **
***********************************************************************************************************************/
#include "Spi.h"                                                                                                        /* PRQA S 0857 # JV-01 */
#include "Spi_Ram.h"
#include "Spi_Irq.h"
#include "Spi_MSPI_Irq.h"
#include "Spi_MSPI_LLDriver.h"
#include "Dem.h"

/***********************************************************************************************************************
**                                                Version Information                                                 **
***********************************************************************************************************************/
/* AUTOSAR release version information */
#define SPI_MSPI_IRQ_C_AR_RELEASE_MAJOR_VERSION    SPI_AR_RELEASE_MAJOR_VERSION_VALUE
#define SPI_MSPI_IRQ_C_AR_RELEASE_MINOR_VERSION    SPI_AR_RELEASE_MINOR_VERSION_VALUE
#define SPI_MSPI_IRQ_C_AR_RELEASE_REVISION_VERSION SPI_AR_RELEASE_REVISION_VERSION_VALUE

/* File version information */
#define SPI_MSPI_IRQ_C_SW_MAJOR_VERSION            SPI_SW_MAJOR_VERSION_VALUE
#define SPI_MSPI_IRQ_C_SW_MINOR_VERSION            SPI_SW_MINOR_VERSION_VALUE

/***********************************************************************************************************************
**                                                   Version Check                                                    **
***********************************************************************************************************************/

#if (SPI_MSPI_IRQ_AR_RELEASE_MAJOR_VERSION != SPI_MSPI_IRQ_C_AR_RELEASE_MAJOR_VERSION)
#error "Spi_MSPI_Irq.c : Mismatch in Release Major Version"
#endif

#if (SPI_MSPI_IRQ_AR_RELEASE_MINOR_VERSION != SPI_MSPI_IRQ_C_AR_RELEASE_MINOR_VERSION)
#error "Spi_MSPI_Irq.c : Mismatch in Release Minor Version"
#endif

#if (SPI_MSPI_IRQ_AR_RELEASE_REVISION_VERSION != SPI_MSPI_IRQ_C_AR_RELEASE_REVISION_VERSION)
#error "Spi_MSPI_Irq.c : Mismatch in Release Revision Version"
#endif

#if (SPI_MSPI_IRQ_SW_MAJOR_VERSION != SPI_MSPI_IRQ_C_SW_MAJOR_VERSION)
#error "Spi_MSPI_Irq.c : Mismatch in Software Major Version"
#endif

#if (SPI_MSPI_IRQ_SW_MINOR_VERSION != SPI_MSPI_IRQ_C_SW_MINOR_VERSION)
#error "Spi_MSPI_Irq.c : Mismatch in Software Minor Version"
#endif

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (2:0857)    : Number of macro definitions exceeds 1024 - program does not conform strictly to ISO:C90.     */
/* Rule                : MISRA C:2012 Dir-1.1                                                                         */
/* JV-01 Justification : The number of macro depend on module code size. There is no issue when number of macro is    */
/*                       over 1024                                                                                    */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0342)    : Using the glue operator '##'.                                                                */
/* Rule                : CERTCCM PRE05, MISRA C:2012 Rule-20.10                                                       */
/* JV-01 Justification : This can be accepted, due to the implementation is following AUTOSAR standard rule for SchM  */
/*                       module's name.                                                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3214)    : The macro '%s' is not used and could be removed.                                             */
/* Rule                : MISRA C:2012 Rule-2.5                                                                        */
/* JV-01 Justification : This can be accepted, it will be used in some case of configuration.                         */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3408)    : '%s' has external linkage and is being defined without any previous declaration.             */
/* Rule                : CERTCCM DCL07, MISRA C:2012 Rule-8.4                                                         */
/* JV-01 Justification : It is accepted, due to the declaration will be taken care by Os                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1503)    : The function '%1s' is defined but is not used within this project.                           */
/* Rule                : CERTCCM MSC07, MISRA C:2012 Rule-2.1                                                         */
/* JV-01 Justification : This is accepted, due to the module's API is exported for user's usage.                      */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : This is accepted, due to the implementation is following hardware specification.             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2824)    : Possible: Arithmetic operation on NULL pointer.                                              */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : This is accepted, due to the implementation is following hardware specification.             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : Use of #include directive after code fragment.                                               */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is done as per Memory Requirement, (MEMMAP003 - Specification of Memory Mapping).       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                    Global Data                                                     **
***********************************************************************************************************************/

/***********************************************************************************************************************
**                                                Function Definitions                                                **
***********************************************************************************************************************/
/* Code snipet to perform interrupt consistency checking */
#if (SPI_INTERRUPT_CONSISTENCY_CHECK == STD_ON)                                                                         
#define SPI_MSPI_INT_ASSERTION(index, intname)                             \
  if (0U != (*Spi_GaaMSPIRegs[(index)].pIC##intname & SPI_EIC_EIMK_MASK))                                               /* PRQA S 0342 # JV-01 */ \
  {                                                                        \
    SPI_DEM_REPORT_ERROR(SPI_E_INT_INCONSISTENT, DEM_EVENT_STATUS_FAILED); \
  }                                                                        \
  else
#else
#define SPI_MSPI_INT_ASSERTION(index, intname)                                                                          /* PRQA S 3214 # JV-01 */
#endif

/* All functions in this file have same attribute */
#define SPI_START_SEC_CODE_FAST
#include "Spi_Mapping.h"

/***********************************************************************************************************************
** Function Name     : SPI_MSPIn_TX_ISR
**
** Service ID        : Not Applicable
**
** Description       : Interrupt Service Routine for MSPI transmit interrupt
**
** Sync/Async        : Synchronous
**
** Re-entrancy       : Non-Reentrant
**
** Input Parameters  : None
**
** InOut Parameters  : None
**
** Output Parameters : None
**
** Return Value      : void
**
** Pre-condition     : Not Applicable
**
** Global Variable   : None
**
** Function invoked  : Spi_MSPITransmitISR, SPI_DEM_REPORT_ERROR
**
** Registers Used    : None
**
** Reference ID      : SPI_DUD_ACT_025, SPI_DUD_ACT_025_ERR001
***********************************************************************************************************************/
/* CH0 ****************************************************************************************************************/
#if (SPI_MSPI0_TX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI0_TX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI0_TX_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI0_TX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI0, TX)                                                                          /* PRQA S 2814 # JV-01 */
  {
    Spi_MSPITransmitISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI0]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI0_TX_ISR_API == STD_ON) */

/* CH1 ****************************************************************************************************************/
#if (SPI_MSPI1_TX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI1_TX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI1_TX_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI1_TX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI1, TX)                                                                          /* PRQA S 2814 # JV-01 */
  {
    Spi_MSPITransmitISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI1]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI1_TX_ISR_API == STD_ON) */

/* CH2 ****************************************************************************************************************/
#if (SPI_MSPI2_TX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI2_TX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI2_TX_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI2_TX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI2, TX)
  {
    Spi_MSPITransmitISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI2]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI2_TX_ISR_API == STD_ON) */

/* CH3 ****************************************************************************************************************/
#if (SPI_MSPI3_TX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI3_TX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI3_TX_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI3_TX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI3, TX)
  {
    Spi_MSPITransmitISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI3]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI3_TX_ISR_API == STD_ON) */

/* CH4 ****************************************************************************************************************/
#if (SPI_MSPI4_TX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI4_TX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI4_TX_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI4_TX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI4, TX)
  {
    Spi_MSPITransmitISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI4]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI4_TX_ISR_API == STD_ON) */

/* CH5 ****************************************************************************************************************/
#if (SPI_MSPI5_TX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI5_TX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI5_TX_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI5_TX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI5, TX)
  {
    Spi_MSPITransmitISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI5]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI5_TX_ISR_API == STD_ON) */

/* CH6 ****************************************************************************************************************/
#if (SPI_MSPI6_TX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI6_TX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI6_TX_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI6_TX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI6, TX)
  {
    Spi_MSPITransmitISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI6]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI6_TX_ISR_API == STD_ON) */

/* CH7 ****************************************************************************************************************/
#if (SPI_MSPI7_TX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI7_TX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI7_TX_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI7_TX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI7, TX)
  {
    Spi_MSPITransmitISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI7]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI7_TX_ISR_API == STD_ON) */

/* CH8 ****************************************************************************************************************/
#if (SPI_MSPI8_TX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI8_TX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI8_TX_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI8_TX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI8, TX)
  {
    Spi_MSPITransmitISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI8]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI8_TX_ISR_API == STD_ON) */

/* CH9 ****************************************************************************************************************/
#if (SPI_MSPI9_TX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI9_TX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI9_TX_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI9_TX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI9, TX)
  {
    Spi_MSPITransmitISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI9]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI9_TX_ISR_API == STD_ON) */

/***********************************************************************************************************************
** Function Name     : SPI_MSPIn_RX_ISR
**
** Service ID        : Not Applicable
**
** Description       : Interrupt Service Routine for MSPI receive interrupt
**
** Sync/Async        : Synchronous
**
** Re-entrancy       : Non-Reentrant
**
** Input Parameters  : None
**
** InOut Parameters  : None
**
** Output Parameters : None
**
** Return Value      : void
**
** Pre-condition     : Not Applicable
**
** Global Variable   : None
**
** Function invoked  : Spi_MSPIReceiveISR, SPI_DEM_REPORT_ERROR
**
** Registers Used    : None
**
** Reference ID      : SPI_DUD_ACT_026, SPI_DUD_ACT_026_ERR001
***********************************************************************************************************************/
/* CH0 ****************************************************************************************************************/
#if (SPI_MSPI0_RX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI0_RX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI0_RX_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI0_RX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI0, RX)                                                                          /* PRQA S 2814 # JV-01 */
  {
    Spi_MSPIReceiveISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI0]);                          /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI0_RX_ISR_API == STD_ON) */

/* CH1 ****************************************************************************************************************/
#if (SPI_MSPI1_RX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI1_RX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI1_RX_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI1_RX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI1, RX)                                                                          /* PRQA S 2814 # JV-01 */
  {
    Spi_MSPIReceiveISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI1]);                          /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI1_RX_ISR_API == STD_ON) */

/* CH2 ****************************************************************************************************************/
#if (SPI_MSPI2_RX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI2_RX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI2_RX_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI2_RX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI2, RX)
  {
    Spi_MSPIReceiveISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI2]);                          /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI2_RX_ISR_API == STD_ON) */

/* CH3 ****************************************************************************************************************/
#if (SPI_MSPI3_RX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI3_RX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI3_RX_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI3_RX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI3, RX)
  {
    Spi_MSPIReceiveISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI3]);                          /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI3_RX_ISR_API == STD_ON) */

/* CH4 ****************************************************************************************************************/
#if (SPI_MSPI4_RX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI4_RX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI4_RX_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI4_RX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI4, RX)
  {
    Spi_MSPIReceiveISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI4]);                          /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI4_RX_ISR_API == STD_ON) */

/* CH5 ****************************************************************************************************************/
#if (SPI_MSPI5_RX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI5_RX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI5_RX_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI5_RX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI5, RX)
  {
    Spi_MSPIReceiveISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI5]);                          /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI5_RX_ISR_API == STD_ON) */

/* CH6 ****************************************************************************************************************/
#if (SPI_MSPI6_RX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI6_RX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI6_RX_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI6_RX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI6, RX)
  {
    Spi_MSPIReceiveISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI6]);                          /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI6_RX_ISR_API == STD_ON) */

/* CH7 ****************************************************************************************************************/
#if (SPI_MSPI7_RX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI7_RX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI7_RX_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI7_RX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI7, RX)
  {
    Spi_MSPIReceiveISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI7]);                          /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI7_RX_ISR_API == STD_ON) */

/* CH8 ****************************************************************************************************************/
#if (SPI_MSPI8_RX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI8_RX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI8_RX_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI8_RX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI8, RX)
  {
    Spi_MSPIReceiveISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI8]);                          /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI8_RX_ISR_API == STD_ON) */

/* CH9 ****************************************************************************************************************/
#if (SPI_MSPI9_RX_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI9_RX_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI9_RX_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI9_RX_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI9, RX)
  {
    Spi_MSPIReceiveISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI9]);                          /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI9_RX_ISR_API == STD_ON) */

/***********************************************************************************************************************
** Function Name     : SPI_MSPIn_FE_ISR
**
** Service ID        : Not Applicable
**
** Description       : Interrupt Service Routine for MSPI frame count end interrupt
**
** Sync/Async        : Synchronous
**
** Re-entrancy       : Non-Reentrant
**
** Input Parameters  : None
**
** InOut Parameters  : None
**
** Output Parameters : None
**
** Return Value      : void
**
** Pre-condition     : Not Applicable
**
** Global Variable   : None
**
** Function invoked  : Spi_MSPIFrameEndISR, SPI_DEM_REPORT_ERROR
**
** Registers Used    : None
**
** Reference ID      : SPI_DUD_ACT_027, SPI_DUD_ACT_027_ERR001
***********************************************************************************************************************/
/* CH0 ****************************************************************************************************************/
#if (SPI_MSPI0_FE_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI0_FE_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI0_FE_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI0_FE_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI0, FE)                                                                          /* PRQA S 2814 # JV-01 */
  {
    Spi_MSPIFrameEndISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI0]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI0_FE_ISR_API == STD_ON) */

/* CH1 ****************************************************************************************************************/
#if (SPI_MSPI1_FE_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI1_FE_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI1_FE_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI1_FE_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI1, FE)
  {
    Spi_MSPIFrameEndISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI1]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI1_FE_ISR_API == STD_ON) */

/* CH2 ****************************************************************************************************************/
#if (SPI_MSPI2_FE_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI2_FE_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI2_FE_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI2_FE_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI2, FE)
  {
    Spi_MSPIFrameEndISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI2]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI2_FE_ISR_API == STD_ON) */

/* CH3 ****************************************************************************************************************/
#if (SPI_MSPI3_FE_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI3_FE_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI3_FE_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI3_FE_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI3, FE)
  {
    Spi_MSPIFrameEndISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI3]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI3_FE_ISR_API == STD_ON) */

/* CH4 ****************************************************************************************************************/
#if (SPI_MSPI4_FE_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI4_FE_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI4_FE_CAT2_ISR)                                                                                              /* PRQA S 1503, 3408 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI4_FE_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI4, FE)
  {
    Spi_MSPIFrameEndISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI4]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI4_FE_ISR_API == STD_ON) */

/* CH5 ****************************************************************************************************************/
#if (SPI_MSPI5_FE_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI5_FE_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI5_FE_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI5_FE_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI5, FE)
  {
    Spi_MSPIFrameEndISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI5]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI5_FE_ISR_API == STD_ON) */

/* CH6 ****************************************************************************************************************/
#if (SPI_MSPI6_FE_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI6_FE_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI6_FE_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI6_FE_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI6, FE)
  {
    Spi_MSPIFrameEndISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI6]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI6_FE_ISR_API == STD_ON) */

/* CH7 ****************************************************************************************************************/
#if (SPI_MSPI7_FE_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI7_FE_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI7_FE_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI7_FE_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI7, FE)
  {
    Spi_MSPIFrameEndISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI7]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI7_FE_ISR_API == STD_ON) */

/* CH8 ****************************************************************************************************************/
#if (SPI_MSPI8_FE_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI8_FE_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI8_FE_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI8_FE_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI8, FE)
  {
    Spi_MSPIFrameEndISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI8]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI8_FE_ISR_API == STD_ON) */

/* CH9 ****************************************************************************************************************/
#if (SPI_MSPI9_FE_ISR_API == STD_ON)
/* Defines the CAT2interrupt mapping */
#if defined(Os_SPI_MSPI9_FE_CAT2_ISR) || (SPI_ISR_CATEGORY_2 == STD_ON)
ISR(SPI_MSPI9_FE_CAT2_ISR)                                                                                              /* PRQA S 3408, 1503 # JV-01, JV-01 */
/* Defines the CAT1 interrupt mapping */
#else
_INTERRUPT_ FUNC(void, SPI_CODE_FAST) SPI_MSPI9_FE_ISR(void)                                                            /* PRQA S 1503 # JV-01 */
#endif
{
  SPI_MSPI_INT_ASSERTION(SPI_PHYIDX_MSPI9, FE)
  {
    Spi_MSPIFrameEndISR((uint32)Spi_GpHwConfig[SPI_MACRO_MSPI].pPhyToHWUnit[SPI_PHYIDX_MSPI9]);                         /* PRQA S 2824 # JV-01 */
  }
}
#endif /* End of (SPI_MSPI9_FE_ISR_API == STD_ON) */

#define SPI_STOP_SEC_CODE_FAST
#include "Spi_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
**                                                    End of File                                                     **
***********************************************************************************************************************/
