/*====================================================================================================================*/
/* Project      = AUTOSAR Renesas X2x MCAL Components                                                                 */
/* Module       = Eth_ETNDE_LLDriver.h                                                                                */
/* SW-VERSION   = 1.5.4                                                                                               */
/*====================================================================================================================*/
/*                                                     COPYRIGHT                                                      */
/*====================================================================================================================*/
/* (c) 2021,2022 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* This file contains ETNDE specific definitions of Eth Driver Component.                                             */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s) and/or */
/* the Application.                                                                                                   */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs of  */
/* program errors, compliance with applicable laws, damage to or loss of data, programs or equipment, and             */
/* unavailability or interruption of operations.                                                                      */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*              Devices:        X2x                                                                                   */
/*====================================================================================================================*/

/***********************************************************************************************************************
**                                              Revision Control History                                              **
***********************************************************************************************************************/
/*
 * 1.5.4: 03/08/2022    : Change the setting of the maximum size of received frames from each queue to each MAC
 * 1.5.3: 11/04/2022    : Update QA-C comments
 * 1.5.1: 20/10/2021    : Initial Version
 */
/**********************************************************************************************************************/
#ifndef ETH_ETNDE_LLDRIVER_H
#define ETH_ETNDE_LLDRIVER_H

#include "Eth_Types.h"
#include "Eth_Util.h"
#include "Os.h"
#include <rh850_Types.h>
/***********************************************************************************************************************
**                                                Version Information                                                 **
***********************************************************************************************************************/
/* AUTOSAR release version information */
#define ETH_ETNDE_AR_RELEASE_MAJOR_VERSION    ETH_AR_RELEASE_MAJOR_VERSION
#define ETH_ETNDE_AR_RELEASE_MINOR_VERSION    ETH_AR_RELEASE_MINOR_VERSION
#define ETH_ETNDE_AR_RELEASE_REVISION_VERSION ETH_AR_RELEASE_REVISION_VERSION

/* Module Software version information */
#define ETH_ETNDE_SW_MAJOR_VERSION            ETH_SW_MAJOR_VERSION
#define ETH_ETNDE_SW_MINOR_VERSION            ETH_SW_MINOR_VERSION

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (2:1055)    : [C99] The keyword 'inline' has been used.                                                    */
/* Rule                : MISRA C:2012 Dir-1.1, Rule-1.2                                                               */
/* JV-01 Justification : In this case, inline functions are considered more optimized for ECODE than function-like    */
/*                       macros. Usage of inline functions is accepted.                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3630)    : The implementation of this struct/union type should be hidden.                               */
/* Rule                : MISRA C:2012 Dir-4.8                                                                         */
/* JV-01 Justification : The pointers point to this struct are dereferenced in other .c files, so this                */
/*                       implementation should not be hidden.                                                         */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0635)    : Bit-field %s in %s has been declared with a type not explicitly supported.                   */
/* Rule                : CERTCCM INT12, MSC14, MISRA C:2012 Dir-1.1,  Rule-1.2, Rule-6.1                              */
/* JV-01 Justification : To access bit wise from register.                                                            */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0750)    : A union type specifier has been defined.                                                     */
/* Rule                : MISRA C:2012 Rule-19.2                                                                       */
/* JV-01 Justification : This union type is used for descriptor accessing and there is no issue with this usage       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3432)    : Simple macro argument expression is not parenthesized.                                       */
/* Rule                : MISRA C:2012 Rule-20.7                                                                       */
/* JV-01 Justification : Compiler keyword (macro) is defined and used followed AUTOSAR standard rule. It is accepted. */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0639)    : [E] Treating array of length one as potentially flexible member.                             */
/* Rule                : MISRA-C:2012 Dir 1.1                                                                         */
/* JV-01 Justification : This warning does not indicate a mistake in the code, it is a message to call attention to   */
/*                       the processing power of the compiler. Because the compiler in use has been confirmed to have */
/*                       the ability to handle this, no problem.                                                      */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3383)    : Cannot identify wraparound guard for unsigned arithmetic expression.                         */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : Wraparound can't be occur in this unsigned integer arithmetic so there is no need to add a   */
/*                       wraparound guard here.                                                                       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3384)    : Cannot identify wraparound guard for dependent unsigned arithmetic expression.               */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : Wraparound can't be occur in this unsigned integer arithmetic so there is no need to add a   */
/*                       wraparound guard here.                                                                       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:4395)    : A composite expression of 'essentially floating' type (double) is being cast to a different  */
/*                       type category, 'unsigned'.                                                                   */
/* Rule                : MISRA C:2012 Rule-10.8                                                                       */
/* JV-01 Justification : This calculation is handled with care to preserve the accuracy of the result. It is confirmed*/
/*                       that there is no problem with this calculation.                                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:4394)    : A composite expression of 'essentially unsigned' type (unsigned long) is being cast to a     */
/*                       different type category, 'floating'.                                                         */
/* Rule                : MISRA C:2012 Rule-10.8                                                                       */
/* JV-01 Justification : This calculation is handled with care to preserve the accuracy of the result. It is confirmed*/
/*                       that there is no problem with this calculation.                                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (8:0644)    : [C] Width of bit-field must be no bigger than the width of an 'int'.                         */
/* Rule                : MISRA C:2012 Rule-1.1                                                                        */
/* JV-01 Justification : This is specific for memory accessing and there is no issue with this usage.                 */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                            HW specific parameters referred from the upper-level driver                             **
***********************************************************************************************************************/
#define ETH_CPUCLK_MHZ                        400UL

/* Insert paddings to the top of buffers to align payload location as 4 byte */
/* Paddings before header */
#define ETH_TXRX_BUFFER_PRE_PADDING           2UL
/* Paddings between header and payload */
#define ETH_TXRX_BUFFER_IN_PADDING            0UL

/* Minimum payload length allowed by the HW */
#define ETH_MIN_PAYLOAD_SIZE                  2UL

/* Use gPTP timer domain 1 */
#define ETH_GPTP_TIMER_DOMAIN                 0UL

/* Enable gPTP timer domain 1 */
#define ETH_GPTP_TIMER_DOMAIN_ENABLE          1UL
#define ETH_GPTP_TIMER_DOMAIN_MASK            (1UL << ETH_GPTP_TIMER_DOMAIN)

#define ETH_GPTP_OFFSET_NANOSEC_MAXVALUE      0x3B9AC9FFUL

/***********************************************************************************************************************
**                                                  Descriptor Types                                                  **
***********************************************************************************************************************/
#define ETH_DESC_FEMPTY_ND                    3UL
#define ETH_DESC_FEMPTY                       4UL
#define ETH_DESC_FSINGLE                      8UL
#define ETH_DESC_FSTART                       9UL
#define ETH_DESC_FMID                         10UL
#define ETH_DESC_FEND                         11UL
#define ETH_DESC_LEMPTY                       12UL
#define ETH_DESC_LINK                         14UL
#define ETH_DESC_EOS                          15UL

/***********************************************************************************************************************
**                                             Descriptor Attribute Types                                             **
***********************************************************************************************************************/
#define ETH_TX_DESC_TYPE_DIRECT               1UL
#define ETH_TX_DESC_TCX_ENABLE                1UL
#define ETH_TX_DESC_TN_DOMAIN                 ETH_GPTP_TIMER_DOMAIN
#define ETH_TX_DESC_DIE_ENABLE                1UL
#define ETH_RX_DESC_DIE_ENABLE                1UL
#define ETH_TS_DESC_DIE_ENABLE                1UL

/***********************************************************************************************************************
**                                                 Size of descriptor                                                 **
***********************************************************************************************************************/
#define ETH_EXT_DESC_SIZE                     16UL /* Extended descriptor size(in byte) */
#define ETH_EXT_DESC_TS_SIZE                  24UL /* Extended descriptor with Timestamp size(in byte) */

/***********************************************************************************************************************
**                                         General Ethernet MAC Address Size                                          **
***********************************************************************************************************************/
#define ETH_RX_DPTR_OFFSET                    2UL

/***********************************************************************************************************************
**                                                 Descriptor related                                                 **
***********************************************************************************************************************/
#define ETH_CYCLIC_DESC_NUM                   1U
/* TxRAM initialization */
#define ETH_ETND_2KB_FRAME_SIZE               0x00000800UL
#define ETH_ETND_INIT_FRAME_LENGTH            0x00004000UL

/***********************************************************************************************************************
** Queues related definitions
***********************************************************************************************************************/
#define ETH_TAS_QUEUE_NUM                     8UL /* Maximum TAS queue number */

/* Queue ID mask */
#define ETH_QUEUE_ID_0_4_MASK                 0x00000011UL
#define ETH_QUEUE_ID_1_5_MASK                 0x00000022UL
#define ETH_QUEUE_ID_2_6_MASK                 0x00000033UL
#define ETH_QUEUE_ID_3_7_MASK                 0x00000044UL

/***********************************************************************************************************************
** Mask for the Status Register
***********************************************************************************************************************/
/* RAM reset in descriptor */
#define ETH_ETNDE_RATRR_RESET                 0x00000001UL
#define ETH_ETNDE_TATRR_RESET                 0x00000002UL
#define ETH_ETNDE_RMSRR_RESET                 0x00000001UL

/* AXI Bus Configuration */
#define ETH_ETNDE_AXIWC_CONFIG                0x00014411UL
#define ETH_ETNDE_AXIRC_CONFIG                0x00011144UL

/* Tx Descriptor Address Table Learning */
#define ETH_ETNDE_TATLS0_CONFIG               0x00000002UL
#define ETH_ETNDE_TATLR_LEARNING              0x80000000UL
#define ETH_ETNDE_TATLR_FAILED                0x00000001UL

/* Rx Descriptor Address Table Learning */
#define ETH_ETNDE_RATLS0_CONFIG               0x0000000CUL
#define ETH_ETNDE_RATLR_LEARNING              0x80000000UL
#define ETH_ETNDE_RATLR_FAILED                0x00000001UL
#define ETH_ETNDE_RATSR2_SEARCHING            0x80000000UL

/* Ts Descriptor Address Table Learning */
#define ETH_ETNDE_TSS_CONFIG                  0x00000005UL

/* TAS Control List Table Learning */
#define ETH_ETNDE_TAEN_POSITION               4UL
#define ETH_ETNDE_TAEN_OFFSET                 8UL
#define ETH_ETNDE_TACLL2_LEARNING             0x80000000UL
#define ETH_ETNDE_TCC_ENABLE                  0x00000001UL
#define ETH_ETNDE_TCC_DISABLE                 0x00000000UL

/* MAC automatic timestamp configuration */
#define ETH_ETNDE_MTATC_CONFIG                0x02000000UL

/* MAC Timestamp reception configuration */
#if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
#define ETH_ETNDE_MTRC_CONFIG                 0x00020000UL
#else
#define ETH_ETNDE_MTRC_CONFIG                 0x03020000UL
#endif

/* MAC reception address filter configuration */
#define ETH_ETNDE_MRAFC_CONFIG                0x00660066UL
#define ETH_ETNDE_PROMISCUOUS_MODE            0x00010001UL

#define ETH_ETNDE_GECMR_SPEED                 0x00000001UL
#define ETH_ETNDE_SGSRST_SRST                 0x00000001UL
#define ETH_ETNDE_SGINTM_URDYIM               0x0002U
#define ETH_ETNDE_SGINTM_RDYIM                0x0001U
#define ETH_ETNDE_SGSDS_SUC                   0x04U
#define ETH_ETNDE_SGSDS_PWS_ON                0x03U
#define ETH_ETNDE_SGCLKSEL_SEL                0x01U
#define ETH_ETNDE_SGRCIE_RCIE                 0x01U

/* MAC Loopback configuration */
#define ETH_ETNDE_MLBC_CONFIG                 0x00000001UL

/* AXIBMI Data Interrupt */
#define ETH_ETNDE_DIE_CONFIG                   0x0001FFFFUL
#define ETH_ETNDE_TSDIE_CONFIG                 0x00000001UL

/* Link Verification Settings */
#define ETH_ETNDE_MLVC_PASE_MASK               0x00000100UL
#define ETH_ETNDE_MLVC_PLV_MASK                0x00020000UL
#define ETH_ETNDE_MLVC_PSE_MASK                0x00010000UL

/***********************************************************************************************************************
**                                             MDIO related definitions                                               **
***********************************************************************************************************************/
/* MDIO Config:                                                                                                       */
/* PHY Station Management Capture time:         User define                                                           */
/* PHY Station Management Hold time adjustment: User define                                                           */
/* PHY Station Management Preamble:             Enabled                                                               */
/* PHY Station Management Clock Selection:      User define                                                           */
/* PHY Link Status Pin Plugged:                 Unplugged                                                             */
/* PHY Interrupt Pin Plugged:                   Unplugged                                                             */
/* PHY Interrupt Polarity:                      Active low                                                            */
/* RMII error pin plugged:                      Unplugged                                                             */
/* Link Speed Configuration:                    Gentool variable                                                      */
/* PHY Interface Select:                        Gentool variable                                                      */
/**********************************************************************************************************************/
#define ETH_ETNDE_PHY_CONFIG_REGISTER_MASK 0x1UL

#define ETH_ETNDE_MPSM_READ_OPERATION      0x1UL
#define ETH_ETNDE_MPSM_WRITE_DATA_SET      0x2UL
#define ETH_ETNDE_MPSM_WRITE_OPERATION     0x3UL

#define ETH_ETNDE_MPIC_PSMCS_MASK          0xFFC0FFFFUL

/***********************************************************************************************************************
**                                             SGMII related definitions                                              **
***********************************************************************************************************************/
#define ETH_ETNDE_WAIT_FOR_SGMII                4000UL /* nanosecond */

#define ETH_ETNDE_PIR_MDC                       0x00000001UL
#define ETH_ETNDE_PIR_MMD                       0x00000002UL
#define ETH_ETNDE_PIR_MDO                       0x00000004UL
#define ETH_ETNDE_GECMR_SPEED                   0x00000001UL
#define ETH_ETNDE_SGSRST_SRST                   0x00000001UL
#define ETH_ETNDE_SGINTM_URDYIM                 0x0002U
#define ETH_ETNDE_SGINTM_RDYIM                  0x0001U
#define ETH_ETNDE_SGINTS_URDY                   0x0002U
#define ETH_ETNDE_SGINTS_RDY                    0x0001U
#define ETH_ETNDE_SGINTS_RDY_UNRDY              (ETH_ETNDE_SGINTS_RDY | ETH_ETNDE_SGINTS_URDY)
#define ETH_ETNDE_SGSDS_SUC                     0x04U
#define ETH_ETNDE_SGSDS_PWS_ON                  0x03U
#define ETH_ETNDE_SGCLKSEL_SEL                  0x01U
#define ETH_ETNDE_SGRCIE_RCIE                   0x01U

#define ETH_ETNDE_SGOPMC_10M                    0x00000003UL
#define ETH_ETNDE_SGOPMC_100M                   0x00000007UL
#define ETH_ETNDE_SGOPMC_1G                     0x0000000BUL

/***********************************************************************************************************************
**                                              GTO related definitions                                               **
***********************************************************************************************************************/
#define ETH_ETNDE_GTO_MINUS                0xFFFFU

/***********************************************************************************************************************
**                                              Register address offset                                               **
***********************************************************************************************************************/

/***********************************************************************************************************************
**                                                 Global Data Types                                                  **
***********************************************************************************************************************/

/* TSN frame type */
typedef enum ETag_Eth_TSN_FrameType
{
  ETH_EXPRESS_FRAME,
  ETH_PREEMPTION_FRAME
} Eth_TSN_FrameType;

typedef enum ETag_Eth_TxFragmentType
{
  ETH_FRAGMENT_64_BYTE,
  ETH_FRAGMENT_128_BYTE,
  ETH_FRAGMENT_192_BYTE,
  ETH_FRAGMENT_256_BYTE
} Eth_TxFragmentType;

typedef enum ETag_Eth_TASGateStateType
{
  ETH_TAS_OPEN,
  ETH_TAS_CLOSE
} Eth_TASGateStateType;

typedef enum ETag_Eth_RxExtStatusType
{
  ETH_EXT_RECEIVED = 0,
  ETH_EXT_NOT_RECEIVED,
  ETH_EXT_RECEIVED_MORE_DATA_AVAILABLE,
  ETH_EXT_NOT_RECEIVED_MORE_DATA_AVAILABLE
} Eth_ExtRxStatusType;

/* Credit Based Shaper */
typedef struct STag_Eth_CBSConfigType
{
  Eth_OptionType enCE;
  uint32 ulCIV; /* Credit Increment Value */
  uint32 ulCUL; /* Credit Upper Limit */
} Eth_CBSConfigType;

typedef struct STag_Eth_TASEntryType                                                                                    /* PRQA S 3630 # JV-01 */
{
  Eth_TASGateStateType enGateState;
  uint32 ulTimeInterval;
} Eth_TASEntryType;

/* Time Aware Shaper */
typedef struct STag_Eth_TASConfigType
{
  Eth_OptionType enTasEnable;
  uint32 ulTxMiniLatency;
  uint32 ulTxJitter;
  uint32 aaNumberOfTasTable[ETH_TAS_QUEUE_NUM];
  P2CONST(Eth_TASEntryType, TYPEDEF, ETH_APPL_CONST) aaTasAdminTable[ETH_TAS_QUEUE_NUM];
} Eth_TASConfigType;

/* Tx Queue Struct */
typedef struct STag_Eth_TxQueueType                                                                                     /* PRQA S 3630 # JV-01 */
{
  uint32 ulQueueBufs;
  uint32 ulQueueId;
  Eth_TSN_FrameType enFrameType;
  uint32 ulMaxFrameSize;
  Eth_CBSConfigType stCBSConfig;
} Eth_TxQueueType;

/* Rx Queue struct */
typedef struct STag_Eth_RxQueueType                                                                                     /* PRQA S 3630 # JV-01 */
{
  uint32 ulQueueBufs;
  uint32 ulQueueId;
  uint8 aaMacFilterPattern[ETH_MACADDR_SIZE];
} Eth_RxQueueType;

typedef struct STag_Eth_QueueConfigType
{
  P2CONST(Eth_TxQueueType, TYPEDEF, ETH_APPL_CONST) pTxQueueConfig;
  P2CONST(Eth_RxQueueType, TYPEDEF, ETH_APPL_CONST) pRxQueueConfig;
  uint8 ucNumberOfTxQueue;
  uint8 ucNumberOfRxQueue;
  uint32 ulTxIntConfig;
  uint32 ulRxIntConfig;
} Eth_QueueConfigType;

/***********************************************************************************************************************
**  Type: Eth_CaptureTimeAdjustment                                                                                   **
**                                                                                                                    **
**  Capture Time option                                                                                               **
**                                                                                                                    **
**  Members:                                                                                                          **
**    MDIO_CAPTURE_TIME_0_CLK_CYCLE  -  No adjusted capture time                                                      **
**    MDIO_CAPTURE_TIME_1_CLK_CYCLE  -  Capture before 1 clk                                                          **
**    MDIO_CAPTURE_TIME_2_CLK_CYCLE  -  Capture before 2 clk                                                          **
**    MDIO_CAPTURE_TIME_3_CLK_CYCLE  -  Capture before 3 clk                                                          **
**    MDIO_CAPTURE_TIME_4_CLK_CYCLE  -  Capture before 4 clk                                                          **
**    MDIO_CAPTURE_TIME_5_CLK_CYCLE  -  Capture before 5 clk                                                          **
**    MDIO_CAPTURE_TIME_6_CLK_CYCLE  -  Capture before 6 clk                                                          **
**    MDIO_CAPTURE_TIME_7_CLK_CYCLE  -  Capture before 7 clk                                                          **
***********************************************************************************************************************/
typedef enum ETag_Eth_CaptureTimeAdjustmentType
{
  MDIO_CAPTURE_TIME_0_CLK_CYCLE = 0,
  MDIO_CAPTURE_TIME_1_CLK_CYCLE,
  MDIO_CAPTURE_TIME_2_CLK_CYCLE,
  MDIO_CAPTURE_TIME_3_CLK_CYCLE,
  MDIO_CAPTURE_TIME_4_CLK_CYCLE,
  MDIO_CAPTURE_TIME_5_CLK_CYCLE,
  MDIO_CAPTURE_TIME_6_CLK_CYCLE,
  MDIO_CAPTURE_TIME_7_CLK_CYCLE
} Eth_CaptureTimeAdjustmentType;

/***********************************************************************************************************************
**  Type: Eth_HoldTimeAdjustment                                                                                      **
**                                                                                                                    **
**  Hold Time option                                                                                                  **
**                                                                                                                    **
**  Members:                                                                                                          **
**    MDIO_HOLD_TIME_0_CLK_CYCLE  -  No adjusted hold time                                                            **
**    MDIO_HOLD_TIME_1_CLK_CYCLE  -  Hold extra 1 clk                                                                 **
**    MDIO_HOLD_TIME_2_CLK_CYCLE  -  Hold extra 2 clk                                                                 **
**    MDIO_HOLD_TIME_3_CLK_CYCLE  -  Hold extra 3 clk                                                                 **
**    MDIO_HOLD_TIME_4_CLK_CYCLE  -  Hold extra 4 clk                                                                 **
**    MDIO_HOLD_TIME_5_CLK_CYCLE  -  Hold extra 5 clk                                                                 **
**    MDIO_HOLD_TIME_6_CLK_CYCLE  -  Hold extra 6 clk                                                                 **
**    MDIO_HOLD_TIME_7_CLK_CYCLE  -  Hold extra 7 clk                                                                 **
***********************************************************************************************************************/
typedef enum ETag_Eth_HoldTimeAdjustmentType
{
  MDIO_HOLD_TIME_0_CLK_CYCLE = 0,
  MDIO_HOLD_TIME_1_CLK_CYCLE,
  MDIO_HOLD_TIME_2_CLK_CYCLE,
  MDIO_HOLD_TIME_3_CLK_CYCLE,
  MDIO_HOLD_TIME_4_CLK_CYCLE,
  MDIO_HOLD_TIME_5_CLK_CYCLE,
  MDIO_HOLD_TIME_6_CLK_CYCLE,
  MDIO_HOLD_TIME_7_CLK_CYCLE
} Eth_HoldTimeAdjustmentType;

/***********************************************************************************************************************
**  Type: Eth_RxConfigType                                                                                            **
**                                                                                                                    **
**  reception configuration structure                                                                                 **
**                                                                                                                    **
**  Members:                                                                                                          **
**    enEncf                  - enable network control filtering                                                      **
**    enEsf                   - enable stream filtering                                                               **
**    enEts0                  - enable timestamp storage (best effort)                                                **
**    enEts2                  - enable timestamp storage (stream)                                                     **
**    ulRfcl                  - reception FIFO critical level                                                         **
**    enSRPTalkerFiltering    - enable stream filtering for talker only                                               **
***********************************************************************************************************************/
typedef struct STag_Eth_RxConfigType
{
  uint32 ulBcastThreshold;
  uint32 ulMcastThreshold;
  Eth_OptionType enBcastStormFilter;
  Eth_OptionType enMcastStormFilter;
  uint32 ulMaxFrameSize;
} Eth_RxConfigType;

/***********************************************************************************************************************
**  Type: Eth_MACConfigType                                                                                           **
**                                                                                                                    **
**  MAC configuration structure                                                                                       **
**                                                                                                                    **
**  Members:                                                                                                          **
**    enPauseFrame                   - enable/disable flow control                                                    **
**    ulRetransmissionTime           - pause frame retransmission time                                                **
**    ulPauseTime                    - stop time at pause frame                                                       **
**    enPauseTimeZero                - enable/disable receive pause frame zero                                        **
**    ulLVTimer                      - Link Verification Timer                                                        **
***********************************************************************************************************************/
typedef struct STag_Eth_MACConfigType
{
  Eth_OptionType enPauseFrame;
  uint32 ulRetransmissionTime;
  uint32 ulPauseTime;
  Eth_OptionType enPauseTimeZero;
  uint32 ulLVTimer;
} Eth_MACConfigType;

/***********************************************************************************************************************
**  Type: Eth_TimeAdjustment                                                                                          **
**                                                                                                                    **
**  Time adjustment structure                                                                                         **
**                                                                                                                    **
**  Members:                                                                                                          **
**    enPSMCaptureTime        - PHY Station Management Capture time                                                   **
**    enPSMHoldTime           - PHY Station Management Hold time                                                      **
**    ulPSMClockSelection     - PHY Station Management Clock Selection                                                **
***********************************************************************************************************************/
typedef struct STag_Eth_PHYConfigType
{
  Eth_CaptureTimeAdjustmentType enPSMCaptureTime;
  Eth_HoldTimeAdjustmentType enPSMHoldTime;
  uint32 ulPSMClockSelection;
} Eth_PHYConfigType;

/* ETNDE LLDriver specific hardware configuration */
typedef struct STag_Eth_ETNDEConfigType
{
  uint32 ulEthPortId;
  #if (ETH_USING_MACRO == ETH_MACRO_ETNE)
  uint32 ulFwdVectorId;
  #endif
  uint32 ulGateLatency;
  uint32 ulGateJitter;
  Eth_QueueConfigType stQueueConfig;
  Eth_RxConfigType stRxConfig;
  Eth_TxFragmentType enTxFragment;
  Eth_TASConfigType stTASConfig;
  Eth_PHYConfigType stPHYConfig;
  Eth_MACConfigType stMACConfig;
  Eth_OptionType enBypassMode;
} Eth_ETNDEConfigType;

/***********************************************************************************************************************
**  Type: Eth_TxCtrlInfo0Type                                                                                         **
**                                                                                                                    **
**  Tx specific control field structure                                                                               **
**                                                                                                                    **
**  Members End Station:                                                                                              **
**    ulFi   - FCS in                                                                                                 **
**    ulFr   - FCS Replacement                                                                                        **
**    ulTxc  - Tx Capture                                                                                             **
**    ulIet  - Insert egress timestamp                                                                                **
**    ulTsun - Timestamp unique number                                                                                **
**    ulTn   - Timer number                                                                                           **
**  Members R-Switch:                                                                                                 **
**    ulFmt  - Frame format                                                                                           **
**    ulErr  - Descriptor error                                                                                       **
**    ulFi   - FCS                                                                                                    **
**    ulVctr - VLAN control                                                                                           **
**    ulReserved  - Reserved                                                                                          **
***********************************************************************************************************************/
typedef struct STag_Eth_TxCtrlInfo0Type
{
  #if (ETH_USING_MACRO == ETH_MACRO_ETND)
  uint32 ulFi : 1;                                                                                                      /* PRQA S 0635 # JV-01 */
  uint32 ulFr : 1;                                                                                                      /* PRQA S 0635 # JV-01 */
  uint32 ulTxc : 1;                                                                                                     /* PRQA S 0635 # JV-01 */
  uint32 ulIet : 1;                                                                                                     /* PRQA S 0635 # JV-01 */
  uint32 ulTsun : 8;                                                                                                    /* PRQA S 0635 # JV-01 */
  uint32 ulTn : 1;                                                                                                      /* PRQA S 0635 # JV-01 */
  #else
  uint32 ulFmt : 2;                                                                                                     /* PRQA S 0635 # JV-01 */
  uint32 ulErr : 1;                                                                                                     /* PRQA S 0635 # JV-01 */
  uint32 ulFi : 1;                                                                                                      /* PRQA S 0635 # JV-01 */
  uint32 ulVctr : 4;                                                                                                    /* PRQA S 0635 # JV-01 */
  uint32 ulReserved : 5;                                                                                                /* PRQA S 0635 # JV-01 */
  #endif
} Eth_TxCtrlInfo0Type;

/***********************************************************************************************************************
**  Union: Eth_TxCtrlInfo0                                                                                            **
**                                                                                                                    **
**  Simplifies Bit and Byte access                                                                                    **
**                                                                                                                    **
**  Members:                                                                                                          **
**    stTxCtrl  - Tx control field                                                                                    **
**    ulWord    - Allow access of the whole word.                                                                     **
***********************************************************************************************************************/
typedef union UTag_Eth_TxCtrlInfo0
{                                                                                                                       /* PRQA S 0750 # JV-01 */
  Eth_TxCtrlInfo0Type stTxCtrl;
  uint32 ulWord;
} Eth_TxCtrlInfo0;

/***********************************************************************************************************************
**  Type: Eth_TxCtrlInfo1Type                                                                                         **
**                                                                                                                    **
**  Tx specific control field structure                                                                               **
**                                                                                                                    **
**  Members End Station:                                                                                              **
**    ulTfl        - Total Frame Length                                                                               **
**    ulCrt        - Calculate residence time (This bit should be always set to 0)                                    **
**    ulReserved0  - Reserved                                                                                         **
**  Members R-Switch:                                                                                                 **
**    ulTsun       - Timestamp Unique Number for IEEE 1588, AS, AS_rev                                                **
**    ulTxc        - TX Capture                                                                                       **
**    ulIet        - Insert Egress Timestamp for IEEE 1588, AS, AS_rev                                                **
**    ulCrt        - Calculation Residence Time for IEEE 1588, AS, AS_rev                                             **
**    ulReserved0  - Reserved                                                                                         **
**    ulTn         - Timestamp domain Number                                                                          **
**    ulReserved1  - Reserved                                                                                         **
***********************************************************************************************************************/
typedef struct STag_Eth_TxCtrlInfo1Type
{
  #if (ETH_USING_MACRO == ETH_MACRO_ETND)
  uint32 ulTfl : 16;                                                                                                    /* PRQA S 0635 # JV-01 */
  uint32 ulCrt : 1;                                                                                                     /* PRQA S 0635 # JV-01 */
  uint32 ulReserved0 : 15;                                                                                              /* PRQA S 0635 # JV-01 */
  #else
  uint32 ulTsun : 8;                                                                                                    /* PRQA S 0635 # JV-01 */
  uint32 ulTxc : 1;                                                                                                     /* PRQA S 0635 # JV-01 */
  uint32 ulIet : 1;                                                                                                     /* PRQA S 0635 # JV-01 */
  uint32 ulCrt : 1;                                                                                                     /* PRQA S 0635 # JV-01 */
  uint32 ulReserved0 : 1;                                                                                               /* PRQA S 0635 # JV-01 */
  uint32 ulTn : 3;                                                                                                      /* PRQA S 0635 # JV-01 */
  uint32 ulReserved1 : 17;                                                                                              /* PRQA S 0635, 0644 # JV-01, JV-01 */
  #endif
} Eth_TxCtrlInfo1Type;

/***********************************************************************************************************************
**  Union: Eth_TxCtrlInfo1                                                                                            **
**                                                                                                                    **
**  Simplifies Bit and Byte access                                                                                    **
**                                                                                                                    **
**  Members:                                                                                                          **
**    stTxCtrl  - Tx control field                                                                                    **
**    ulWord    - Allow access of the whole word.                                                                     **
***********************************************************************************************************************/
typedef union UTag_Eth_TxCtrlInfo1
{                                                                                                                       /* PRQA S 0750 # JV-01 */
  Eth_TxCtrlInfo1Type stTxCtrl;
  uint32 ulWord;
} Eth_TxCtrlInfo1;

/***********************************************************************************************************************
**  Type: Eth_DataDescLType                                                                                           **
**                                                                                                                    **
**  Data descriptor structure low word                                                                                **
**                                                                                                                    **
**  Members:                                                                                                          **
**    ulDs    - Descriptor size                                                                                       **
**    ulInfo0 - Information 0 between CPU and MHD                                                                     **
**    ulDse   - Data Size Error                                                                                       **
**    ulAxie  - AXI Bus Error                                                                                         **
**    ulDie   - Descriptor Interrupt Enable                                                                           **
**    ulDt    - Descriptor Type                                                                                       **
***********************************************************************************************************************/
typedef struct STag_Eth_DataDescLType
{
  uint32 ulDs : 12;                                                                                                     /* PRQA S 0635 # JV-01 */
  uint32 ulInfo0 : 13;                                                                                                  /* PRQA S 0635 # JV-01 */
  uint32 ulDse : 1;                                                                                                     /* PRQA S 0635 # JV-01 */
  uint32 ulAxie : 1;                                                                                                    /* PRQA S 0635 # JV-01 */
  uint32 ulDie : 1;                                                                                                     /* PRQA S 0635 # JV-01 */
  uint32 ulDt : 4;                                                                                                      /* PRQA S 0635 # JV-01 */
} Eth_DataDescLType;

/***********************************************************************************************************************
**  Type: Eth_ExtDescType                                                                                             **
**                                                                                                                    **
**  Extended data descriptor structure                                                                                **
**                                                                                                                    **
**  Members:                                                                                                          **
**    stHeader    - low word of the data descriptor                                                                   **
**    ulDptr      - Descriptor data                                                                                   **
**    ulInfo1     - Information 1 between CPU and MHD                                                                 **
***********************************************************************************************************************/
typedef struct STag_Eth_ExtDescType                                                                                     /* PRQA S 3630 # JV-01 */
{
  Eth_DataDescLType stHeader;
  uint32 ulDptr;
  uint32 ulInfo1[2];
} Eth_ExtDescType;

/***********************************************************************************************************************
**  Type: Eth_ExtDescWithTsType                                                                                       **
**                                                                                                                    **
**  Extended data descriptor with Timestamp structure                                                                 **
**                                                                                                                    **
**  Members:                                                                                                          **
**    stHeader    - low word of the data descriptor                                                                   **
**    ulDptr      - Descriptor data                                                                                   **
**    ulInfo1     - Information 1 between CPU and MHD                                                                 **
**    ulTsns      - Timestamps nanosecond                                                                             **
**    ulTss       - Timestamp second                                                                                  **
***********************************************************************************************************************/
typedef struct STag_Eth_ExtDescWithTsType                                                                               /* PRQA S 3630 # JV-01 */
{
  Eth_DataDescLType stHeader;
  uint32 ulDptr;
  uint32 ulInfo1[2];
  uint32 ulTsns;
  uint32 ulTss;
} Eth_ExtDescWithTsType;

/***********************************************************************************************************************
**  Type: Eth_RxFrameType                                                                                             **
**                                                                                                                    **
**  Rx frame information structure                                                                                    **
**                                                                                                                    **
**  Members:                                                                                                          **
**    ulFrameAddr    - frame address                                                                                  **
**    ulEthTypeAddr  - ether type address                                                                             **
**    ulFrameLength  - frame length                                                                                   **
**    ulTsns         - Timestamps nanosecond                                                                          **
**    ulTss          - Timestamp second                                                                               **
***********************************************************************************************************************/
typedef struct STag_Eth_RxFrameType
{
  uint32 ulFrameAddr;
  uint32 ulEthTypeAddr;
  uint32 ulFrameLength;
  uint32 ulTsns;
  uint32 ulTss;
} Eth_RxFrameType;

/***********************************************************************************************************************
**  Type: Eth_BufHandlerType                                                                                          **
**                                                                                                                    **
**  Setting option                                                                                                    **
**                                                                                                                    **
**  Members:                                                                                                          **
**    stNode      - the Node for linked list management                                                               **
**    ulbufIdx    - the Index of buffer                                                                               **
**    ulbufAddr   - Pointer to the Frame in URAM  to be transmitted                                                   **
**    ulTxLength  - Transmit request size                                                                             **
**    enChannel   - Tx Queue used to transmit the associated buffer                                                   **
**    blEnableTS  - Flag to store if TimeStamp shall be stored                                                        **
**    blTxConfirm - the Tx Confirmation flag                                                                          **
***********************************************************************************************************************/
typedef struct STag_Eth_BufHandlerType                                                                                  /* PRQA S 3630 # JV-01 */
{
  uint32 ulbufIdx;
  uint32 ulbufAddr;
  uint32 ulTxLength;
  uint32 ulEthTypeAddr;
  uint8 ucPriority;
  boolean blbenableTS;
  boolean blTxConfirm;
  uint8 ucDummy0;
  #if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)
  Eth_TimeStampType stTimeStamp;
  Eth_TimeStampQualType enTimeQual;
  uint8 ucDummy1[3];
  #endif
} Eth_BufHandlerType;

/***********************************************************************************************************************
**  Type: Eth_ETNDEStatusType                                                                                         **
**                                                                                                                    **
**  Device instance specific data.                                                                                    **
**                                                                                                                    **
**  All instance specific data is kept within that structure.                                                         **
**                                                                                                                    **
**  Members:                                                                                                          **
**    pBufTxCnt             - Buffer Tx counter                                                                       **
**    ppHeadTxDesc          - Head tx descriptor address                                                              **
**    ppNextTxDesc          - Last tx descriptor address                                                              **
**    ppHeadRxDesc          - Head rx descriptor address                                                              **
***********************************************************************************************************************/
typedef struct STag_Eth_HwStatusType
{
  P2VAR(uint32, AUTOMATIC, ETH_APPL_DATA) pBufTxCnt;                                                                    /* PRQA S 3432 # JV-01 */
  P2VAR(uint32, AUTOMATIC, ETH_APPL_DATA) pTxAllocCnt;                                                                  /* PRQA S 3432 # JV-01 */
  P2VAR(Eth_ExtDescType *, AUTOMATIC, ETH_APPL_DATA) ppHeadTxDesc;                                                      /* PRQA S 3432 # JV-01 */
  P2VAR(Eth_ExtDescType *, AUTOMATIC, ETH_APPL_DATA) ppNextTxDesc;                                                      /* PRQA S 3432 # JV-01 */
  P2VAR(Eth_ExtDescWithTsType *, AUTOMATIC, ETH_APPL_DATA) ppHeadRxDesc;                                                /* PRQA S 3432 # JV-01 */
} Eth_HwStatusType;

/***********************************************************************************************************************
**                                                   Global Symbols                                                   **
***********************************************************************************************************************/

/***********************************************************************************************************************
**  Type: Eth_EtherFrameType                                                                                          **
**                                                                                                                    **
**  Ethernet Frame structure                                                                                          **
**                                                                                                                    **
**  Members:                                                                                                          **
**    ucDstAddr    - Destination address                                                                              **
**    ucSrcAddr    - Source address                                                                                   **
**    ucFrameType  - EtherType                                                                                        **
**    ucPayload    - Payload                                                                                          **
***********************************************************************************************************************/
typedef struct STag_Eth_EtherFrameType
{
  uint8 ucDstAddr[6];
  uint8 ucSrcAddr[6];
  uint8 ucFrameType[2];
  Eth_DataType ucPayload;
} Eth_EtherFrameType;

/* ETNDE register definitions */
typedef struct STag_Eth_ETNDE_TBFRegType
{
  uint32 ulTBFCRi;                      /* CONFIG/OPERATION                  */
  uint32 ulTBFVRi;                      /* CONFIG/OPERATION                  */
  uint32 Reserved0[2];                  /* Reserved                          */
} Eth_ETNDE_TBFRegType;

typedef struct STag_Eth_ETNDE_FBFRegType
{
  uint32 ulFBFCRi;                      /* CONFIG/OPERATION                  */
  uint32 ulFBFV0Ri;                     /* CONFIG/OPERATION                  */
  uint32 ulFBFV1Ri;                     /* CONFIG/OPERATION                  */
  uint32 Reserved0;                     /* Reserved                          */
} Eth_ETNDE_FBFRegType;

typedef struct STag_Eth_ETNDE_CFSFRegType
{
  uint32 ulCFCRi;                       /* CONFIG/OPERATION                  */
  uint32 ulSFN0Ri;                      /* CONFIG/OPERATION                  */
  uint32 ulSFN1Ri;                      /* CONFIG/OPERATION                  */
  uint32 Reserved0;                     /* Reserved                          */
} Eth_ETNDE_CFSFRegType;

typedef struct STag_Eth_ETNDE_RMACRegType
{
  uint32 ulMCC;                         /* Write inhibit CONFIG/OPERATION    */
  uint32 Reserved0[3];                  /* Reserved                          */
  uint32 ulMPSM;                        /* Any                               */
  uint32 ulMPIC;                        /* CONFIG                            */
  uint32 Reserved1[2];                  /* Reserved                          */
  uint32 ulMTFFC;                       /* CONFIG                            */
  uint32 ulMTPFC;                       /* CONFIG/OPERATION                  */
  uint32 Reserved2[6];                  /* Reserved                          */
  uint32 ulMTATC[16];                   /* CONFIG                            */
  uint32 ulMRGC;                        /* CONFIG                            */
  uint32 ulMRMAC0;                      /* CONFIG                            */
  uint32 ulMRMAC1;                      /* CONFIG                            */
  uint32 ulMRAFC;                       /* CONFIG                            */
  uint32 ulMRSCE;                       /* CONFIG                            */
  uint32 ulMRSCP;                       /* CONFIG                            */
  uint32 ulMRSCC;                       /* CONFIG/OPERATION                  */
  uint32 ulMRFSCE;                      /* CONFIG                            */
  uint32 ulMRFSCP;                      /* CONFIG                            */
  uint32 ulMTRC;                        /* CONFIG                            */
  uint32 Reserved3[22];                 /* Reserved                          */
  uint32 ulMPFC[16];                    /* CONFIG                            */
  uint32 Reserved4[128];                /* Reserved                          */
  uint32 ulMLVC;                        /* CONFIG/OPERATION                  */
  uint32 Reserved5[3];                  /* Reserved                          */
  uint32 ulMEEEC;                       /* CONFIG/OPERATION                  */
  uint32 Reserved6[3];                  /* Reserved                          */
  uint32 ulMLBC;                        /* CONFIG                            */
  uint32 Reserved7[39];                 /* Reserved                          */
  uint32 ulMGMR;                        /* Read only                         */
  uint32 Reserved8[3];                  /* Reserved                          */
  uint32 ulMMPFTCT;                     /* Read only                         */
  uint32 ulMAPFTCT;                     /* Read only                         */
  uint32 ulMPFRCT;                      /* Read only                         */
  uint32 ulMFCICT;                      /* Read only                         */
  uint32 ulMEEECT;                      /* Read only                         */
  uint32 Reserved9[55];                 /* Reserved                          */
  uint32 ulMEIS;                        /* Any                               */
  uint32 ulMEIE;                        /* Any                               */
  uint32 ulMEID;                        /* Any                               */
  uint32 Reserved10;                    /* Reserved                          */
  uint32 ulMMIS0;                       /* Any                               */
  uint32 ulMMIE0;                       /* Any                               */
  uint32 ulMMID0;                       /* Any                               */
  uint32 Reserved11;                    /* Reserved                          */
  uint32 ulMMIS1;                       /* Any                               */
  uint32 ulMMIE1;                       /* Any                               */
  uint32 ulMMID1;                       /* Any                               */
  uint32 Reserved12;                    /* Reserved                          */
  uint32 ulMMIS2;                       /* Any                               */
  uint32 ulMMIE2;                       /* Any                               */
  uint32 ulMMID2;                       /* Any                               */
} Eth_ETNDE_RMACRegType;

typedef struct STag_Eth_ETNDE_RMACSRegType
{
  Eth_ETNDE_TBFRegType stTBF[64];       /* CONFIG/OPERATION                  */
  Eth_ETNDE_FBFRegType stFBF[64];       /* CONFIG/OPERATION                  */
  Eth_ETNDE_CFSFRegType stCFSF[64];     /* CONFIG/OPERATION                  */
  uint32 ulVTCCR;                       /* CONFIG                            */
  uint32 Reserved0[3];                  /* Reserved                          */
  uint32 ulFMSCR;                       /* CONFIG                            */
  uint32 Reserved1[59];                 /* Reserved                          */
  uint32 ulFVLSR;                       /* Read only                         */
  uint32 Reserved2[3];                  /* Reserved                          */
  uint32 ulMVCR;                        /* Read only                         */
  uint32 ulMV0R;                        /* Read only                         */
  uint32 ulMV1R;                        /* Read only                         */
  uint32 Reserved3[57];                 /* Reserved                          */
  uint32 ulINTSR;                       /* CONFIG/DISABLE/OPERATION          */
  uint32 ulINTER;                       /* CONFIG/DISABLE/OPERATION          */
  uint32 ulINTDR;                       /* CONFIG/DISABLE/OPERATION          */
  uint32 Reserved4[125];                /* Reserved                          */
  uint32 ulRXBCE;                       /* Read only                         */
  uint32 ulRXBCP;                       /* Read only                         */
  uint32 ulRGFCE;                       /* Read only                         */
  uint32 ulRGFCP;                       /* Read only                         */
  uint32 ulRBFC;                        /* Read only                         */
  uint32 ulRMFC;                        /* Read only                         */
  uint32 ulRVFC;                        /* Read only                         */
  uint32 ulRPEFC;                       /* Read only                         */
  uint32 ulRNEFC;                       /* Read only                         */
  uint32 ulRFMEFC;                      /* Read only                         */
  uint32 ulRFFMEFC;                     /* Read only                         */
  uint32 ulRCFCEFC;                     /* Read only                         */
  uint32 ulRFCEFC;                      /* Read only                         */
  uint32 ulRRCFEFC;                     /* Read only                         */
  uint32 ulRRXFEFC;                     /* Read only                         */
  uint32 ulRUEFC;                       /* Read only                         */
  uint32 ulROEFC;                       /* Read only                         */
  uint32 ulRBOEC;                       /* Read only                         */
  uint32 Reserved5[46];                 /* Reserved                          */
  uint32 ulTXBCE;                       /* Read only                         */
  uint32 ulTXBCP;                       /* Read only                         */
  uint32 ulTGFCE;                       /* Read only                         */
  uint32 ulTGFCP;                       /* Read only                         */
  uint32 ulTBFC;                        /* Read only                         */
  uint32 ulTMFC;                        /* Read only                         */
  uint32 ulTVFC;                        /* Read only                         */
  uint32 ulTEFC;                        /* Read only                         */
  uint32 Reserved6[440];                /* Reserved                          */
  Eth_ETNDE_RMACRegType stRMAC;         /* RMAC regs                         */
} Eth_ETNDE_RMACSRegType;

typedef struct STag_Eth_ETNDE_TSDRegType
{
  uint32 ulTSAs;                        /* TS Descriptor Chain Address s */
  uint32 ulTSSs;                        /* TS Descriptor Chain Setting s */
} Eth_ETNDE_TSDRegType;

typedef struct STag_Eth_ETNDE_RIDRegType
{
  uint32 ulRIDASMi;                      /* RX Incremental Data Area Size Monitoring i              */
  uint32 ulRIDASAMi;                     /* RX Incremental Data Area Start Address Monitoring i     */
  uint32 ulRIDACAMi;                     /* RX Incremental Data Area Current Address Monitoring i   */
  uint32 Reserved0;                      /* Reserved                                                */
} Eth_ETNDE_RIDRegType;

typedef struct STag_Eth_ETNDE_TCERegType
{
  uint32 ulTCEISi;                       /* TX Descriptor Configuration Error Interrupt Status i    */
  uint32 ulTCEIEi;                       /* TX Descriptor Configuration Error Interrupt Enable i    */
  uint32 ulTCEIDi;                       /* TX Descriptor Configuration Error Interrupt Disable i   */
  uint32 Reserved0;                      /* Reserved                                                */
} Eth_ETNDE_TCERegType;

typedef struct STag_Eth_ETNDE_RFSRegType
{
  uint32 ulRFSEISi;                      /* RX Frame Size Error Interrupt Status i                  */
  uint32 ulRFSEIEi;                      /* RX Frame Size Error Interrupt Enable i                  */
  uint32 ulRFSEIDi;                      /* RX Frame Size Error Interrupt Disable i                 */
  uint32 Reserved0;                      /* Reserved                                                */
} Eth_ETNDE_RFSRegType;

typedef struct STag_Eth_ETNDE_RFERegType
{
  uint32 ulRFEISi;                       /* RX Descriptor Full Error Interrupt Status i             */
  uint32 ulRFEIEi;                       /* RX Descriptor Full Error Interrupt Enable i             */
  uint32 ulRFEIDi;                       /* RX Descriptor Full Error Interrupt Disable i            */
  uint32 Reserved0;                      /* Reserved                                                */
} Eth_ETNDE_RFERegType;

typedef struct STag_Eth_ETNDE_RCERegType
{
  uint32 ulRCEISi;                       /* RX Descriptor Configuration Error Interrupt Status i    */
  uint32 ulRCEIEi;                       /* RX Descriptor Configuration Error Interrupt Enable i    */
  uint32 ulRCEIDi;                       /* RX Descriptor Configuration Error Interrupt Disable i   */
  uint32 Reserved0;                      /* Reserved                                                */
} Eth_ETNDE_RCERegType;

typedef struct STag_Eth_ETNDE_TDIRegType
{
  uint32 ulTDISi;                        /* TX Descriptor Data Interrupt Status i                   */
  uint32 ulTDIEi;                        /* TX Descriptor Data Interrupt Enable i                   */
  uint32 ulTDIDi;                        /* TX Descriptor Data Interrupt Disable i                  */
  uint32 Reserved0;                      /* Reserved                                                */
} Eth_ETNDE_TDIRegType;

typedef struct STag_Eth_ETNDE_RDIRegType
{
  uint32 ulRDISi;                        /* RX Descriptor Data Interrupt Status i                   */
  uint32 ulRDIEi;                        /* RX Descriptor Data Interrupt Enable i                   */
  uint32 ulRDIDi;                        /* RX Descriptor Data Interrupt Disable i                  */
  uint32 Reserved0;                      /* Reserved                                                */
} Eth_ETNDE_RDIRegType;

typedef struct STag_Eth_ETNDE_AXIBMIRegType
{
  uint32 ulAXIWC;                       /* AXI Bus Write Configuration                              */
  uint32 ulAXIRC;                       /* AXI Bus Read Configuration                               */
  uint32 Reserved0[2];                  /* Reserved                                                 */
  uint32 ulTDPC[32];                    /* TX Descriptor chain Priority Configuration i             */
  uint32 ulTFT;                         /* TX Descriptor Chain Type                                 */
  uint32 Reserved1[3];                  /* Reserved                                                 */
  uint32 ulTATLS0;                      /* TX Descriptor Address Table Learning Setting 0           */
  uint32 ulTATLS1;                      /* TX Descriptor Address Table Learning Setting 1           */
  uint32 ulTATLR;                       /* TX Descriptor Address Table Learning Result              */
  uint32 Reserved2;                     /* Reserved                                                 */
  uint32 ulRATLS0;                      /* RX Descriptor Address Table Learning Setting 0           */
  uint32 ulRATLS1;                      /* RX Descriptor Address Table Learning Setting 1           */
  uint32 ulRATLR;                       /* RX Descriptor Address Table Learning Result              */
  uint32 Reserved3;                     /* Reserved                                                 */
  Eth_ETNDE_TSDRegType stTSD[16];       /* TS Regs                                                  */
  uint32 ulTRCR[8];                     /* Transmission Request Configuration Register i            */
  uint32 Reserved4[8];                  /* Reserved                                                 */
  uint32 ulRIDAUAS[8];                  /* RX Incremental Data Area Used Area Size                  */
  uint32 Reserved5[24];                 /* Reserved                                                 */
  uint32 ulRR;                          /* RAM Reset                                                */
  uint32 Reserved6[3];                  /* Reserved                                                 */
  uint32 ulTATS;                        /* TX Descriptor Address Table Searching                    */
  uint32 ulTATSR0;                      /* TX Descriptor Address Table Searching Result 0           */
  uint32 ulTATSR1;                      /* TX Descriptor Address Table Searching Result 1           */
  uint32 ulTATSR2;                      /* TX Descriptor Address Table Searching Result 2           */
  uint32 ulRATS;                        /* RX Descriptor Address Table Searching                    */
  uint32 ulRATSR0;                      /* RX Descriptor Address Table Searching Result 0           */
  uint32 ulRATSR1;                      /* RX Descriptor Address Table Searching Result 1           */
  uint32 ulRATSR2;                      /* RX Descriptor Address Table Searching Result 2           */
  uint32 Reserved7[4];                  /* Reserved                                                 */
  Eth_ETNDE_RIDRegType stRID[8];        /* RX Regs                                                  */
  uint32 Reserved8[16];                 /* Reserved                                                 */
  uint32 ulEIS0;                        /* Error Interrupt Status 0                                 */
  uint32 ulEIE0;                        /* Error Interrupt Enable 0                                 */
  uint32 ulEID0;                        /* Error Interrupt Disable 0                                */
  uint32 Reserved9;                     /* Reserved                                                 */
  uint32 ulEIS1;                        /* Error Interrupt Status 1                                 */
  uint32 ulEIE1;                        /* Error Interrupt Enable 1                                 */
  uint32 ulEID1;                        /* Error Interrupt Disable 1                                */
  uint32 Reserved10[9];                 /* Reserved                                                 */
  Eth_ETNDE_TCERegType stTXDCEI[8];     /* TX Descriptor Configuration Error Interrupt Regs         */
  uint32 Reserved11[64];                /* Reserved                                                 */
  Eth_ETNDE_RFSRegType stRXFSEI[8];     /* RX Frame Size Error Interrupt Regs                       */
  Eth_ETNDE_RFERegType stRXDFEI[8];     /* RX Descriptor Full Error Interrupt Regs                  */
  Eth_ETNDE_RCERegType stRXDCEI[8];     /* RX Descriptor Configuration Error Interrupt Regs         */
  uint32 ulRIDAOIS;                     /* RX Incremental Data Area Overflow Interrupt Error        */
  uint32 ulRIDAOIE;                     /* RX Incremental Data Area Overflow Interrupt Enable       */
  uint32 ulRIDAOID;                     /* RX Incremental Data Area Overflow Interrupt Disable      */
  uint32 Reserved12[29];                /* Reserved                                                 */
  uint32 ulTSFEIS;                      /* TS Descriptor Full Error Interrupt Status                */
  uint32 ulTSFEIE;                      /* TS Descriptor Full Error Interrupt Enable                */
  uint32 ulTSFEID;                      /* TS Descriptor Full Error Interrupt Disable               */
  uint32 Reserved13;                    /* Reserved                                                 */
  uint32 ulTSCEIS;                      /* TS Descriptor Configuration Error Interrupt Status       */
  uint32 ulTSCEIE;                      /* TS Descriptor Configuration Error Interrupt Enable       */
  uint32 ulTSCEID;                      /* TS Descriptor Configuration Error Interrupt Disable      */
  uint32 Reserved14[265];               /* Reserved                                                 */
  uint32 ulDIS;                         /* Data Interrupt Status                                    */
  uint32 ulDIE;                         /* Data Interrupt Enable                                    */
  uint32 ulDID;                         /* Data Interrupt Disable                                   */
  uint32 Reserved15;                    /* Reserved                                                 */
  Eth_ETNDE_TDIRegType stTXDI[8];       /* TX Descriptor Data Interrupt Regs                        */
  Eth_ETNDE_RDIRegType stRXDI[8];       /* RX Descriptor Data Interrupt Regs                        */
  uint32 ulTSDIS;                       /* TS Descriptor Data Interrupt Status                      */
  uint32 ulTSDIE;                       /* TS Descriptor Data Interrupt Enable                      */
  uint32 ulTSDID;                       /* TS Descriptor Data Interrupt Disable                     */
} Eth_ETNDE_AXIBMIRegType;

typedef struct STag_Eth_ETNDE_QCISRegType
{
  uint32 ulQSTMACUs;                    /* CONFIG/OPERATION                  */
  uint32 ulQSTMACDs;                    /* CONFIG/OPERATION                  */
  uint32 ulQSTMAMUs;                    /* CONFIG/OPERATION                  */
  uint32 ulQSTMAMDs;                    /* CONFIG/OPERATION                  */
  uint32 ulQSFTVLs;                     /* CONFIG/OPERATION                  */
  uint32 ulQSFTVLMs;                    /* CONFIG/OPERATION                  */
  uint32 ulQSFTMSDs;                    /* CONFIG/OPERATION                  */
  uint32 ulQSFTGMIs;                    /* CONFIG/OPERATION                  */
} Eth_ETNDE_QCISRegType;

typedef struct STag_Eth_ETNDE_MHDRegType
{
  uint32 ulOCR;                         /* Any                               */
  uint32 ulOSR;                         /* N/A                               */
  uint32 ulSWR;                         /* DISABLE                           */
  uint32 ulSIS;                         /* N/A                               */
  uint32 ulGIS;                         /* Any                               */
  uint32 ulGIE;                         /* Any                               */
  uint32 ulGID;                         /* Any                               */
  uint32 Reserved0;                     /* Reserved                          */
  uint32 ulTIS1;                        /* Any                               */
  uint32 ulTIE1;                        /* Any                               */
  uint32 ulTID1;                        /* Any                               */
  uint32 Reserved1;                     /* Reserved                          */
  uint32 ulTIS2;                        /* Any                               */
  uint32 ulTIE2;                        /* Any                               */
  uint32 ulTID2;                        /* Any                               */
  uint32 Reserved2;                     /* Reserved                          */
  uint32 ulRIS;                         /* Any                               */
  uint32 ulRIE;                         /* Any                               */
  uint32 ulRID;                         /* Any                               */
  uint32 Reserved3;                     /* Reserved                          */
  uint32 ulTGC1;                        /* CONFIG                            */
  uint32 ulTGC2;                        /* CONFIG                            */
  uint32 Reserved4[2];                  /* Reserved                          */
  uint32 ulTFS[4];                      /* CONFIG                            */
  uint32 ulTCF[4];                      /* CONFIG                            */
  uint32 ulTCR1;                        /* CONFIG                            */
  uint32 ulTCR2;                        /* CONFIG                            */
  uint32 ulTCR3;                        /* CONFIG                            */
  uint32 ulTCR4;                        /* CONFIG                            */
  uint32 ulTMS[8];                      /* CONFIG                            */
  uint32 ulTSR1;                        /* N/A                               */
  uint32 ulTSR2;                        /* N/A                               */
  uint32 ulTSR3;                        /* N/A                               */
  uint32 ulTSR4;                        /* N/A                               */
  uint32 ulTSR5;                        /* N/A                               */
  uint32 Reserved5[3];                  /* Reserved                          */
  uint32 ulRGC;                         /* CONFIG                            */
  uint32 ulRDFCR;                       /* CONFIG                            */
  uint32 ulRCFCR;                       /* CONFIG                            */
  uint32 ulREFCNCR;                     /* CONFIG                            */
  uint32 ulRSR1;                        /* N/A                               */
  uint32 ulRSR2;                        /* N/A                               */
  uint32 ulRSR3;                        /* N/A                               */
  uint32 Reserved6[61];                 /* Reserved                          */
  uint32 ulTCIS;                        /* Any                               */
  uint32 ulTCIE;                        /* Any                               */
  uint32 ulTCID;                        /* Any                               */
  uint32 Reserved7;                     /* Reserved                          */
  uint32 ulTPTPC;                       /* CONFIG                            */
  uint32 ulTTML;                        /* CONFIG                            */
  uint32 ulTTJ;                         /* CONFIG                            */
  uint32 Reserved8;                     /* Reserved                          */
  uint32 ulTCC;                         /* CONFIG/OPERATION                  */
  uint32 ulTCS;                         /* CONFIG/OPERATION                  */
  uint32 Reserved9;                     /* Reserved                          */
  uint32 ulTGS;                         /* CONFIG/OPERATION                  */
  uint32 ulTACST0;                      /* CONFIG/OPERATION                  */
  uint32 ulTACST1;                      /* CONFIG/OPERATION                  */
  uint32 ulTACST2;                      /* CONFIG/OPERATION                  */
  uint32 Reserved10;                    /* Reserved                          */
  uint32 ulTALIT0;                      /* CONFIG/OPERATION                  */
  uint32 ulTALIT1;                      /* CONFIG/OPERATION                  */
  uint32 ulTALIT2;                      /* CONFIG/OPERATION                  */
  uint32 Reserved11;                    /* Reserved                          */
  uint32 ulTAEN[2];                     /* CONFIG/OPERATION                  */
  uint32 Reserved12[2];                 /* Reserved                          */
  uint32 ulTASFE;                       /* CONFIG/OPERATION                  */
  uint32 Reserved13[3];                 /* Reserved                          */
  uint32 ulTACLL0;                      /* CONFIG/OPERATION                  */
  uint32 ulTACLL1;                      /* CONFIG/OPERATION                  */
  uint32 ulTACLL2;                      /* CONFIG/OPERATION                  */
  uint32 Reserved14;                    /* Reserved                          */
  uint32 ulCACC;                        /* CONFIG/OPERATION                  */
  uint32 ulCCS;                         /* CONFIG/OPERATION                  */
  uint32 Reserved15[2];                 /* Reserved                          */
  uint32 ulCAIV[8];                     /* CONFIG/OPERATION                  */
  uint32 ulCAUL[8];                     /* CONFIG/OPERATION                  */
  uint32 Reserved16[20];                /* Reserved                          */
  uint32 ulTOCST0;                      /* N/A                               */
  uint32 ulTOCST1;                      /* N/A                               */
  uint32 ulTOCST2;                      /* N/A                               */
  uint32 Reserved17;                    /* Reserved                          */
  uint32 ulTOLIT0;                      /* N/A                               */
  uint32 ulTOLIT1;                      /* N/A                               */
  uint32 ulTOLIT2;                      /* N/A                               */
  uint32 Reserved18;                    /* Reserved                          */
  uint32 ulTOEN0;                       /* N/A                               */
  uint32 ulTOEN1;                       /* N/A                               */
  uint32 Reserved19[2];                 /* Reserved                          */
  uint32 ulTOSFE;                       /* N/A                               */
  uint32 Reserved20[3];                 /* Reserved                          */
  uint32 ulTCLR0;                       /* N/A                               */
  uint32 ulTCLR1;                       /* N/A                               */
  uint32 ulTCLR2;                       /* N/A                               */
  uint32 Reserved21;                    /* Reserved                          */
  uint32 ulTSMS;                        /* N/A                               */
  uint32 Reserved22[3];                 /* Reserved                          */
  uint32 ulCOCC;                        /* N/A                               */
  uint32 Reserved23[19];                /* Reserved                          */
  uint32 ulCOIV[8];                     /* N/A                               */
  uint32 ulCOUL[8];                     /* N/A                               */
  uint32 Reserved24[4];                 /* Reserved                          */
  Eth_ETNDE_QCISRegType stQCIS[16];     /* CONFIG/OPERATION                  */
  uint32 ulQSFTLS;                      /* N/A                               */
  uint32 ulQSFTLIS;                     /* Any                               */
  uint32 ulQSFTLIE;                     /* Any                               */
  uint32 ulQSFTLID;                     /* Any                               */
  uint32 ulQSMSMC;                      /* CONFIG/OPERATION                  */
  uint32 ulQSGTMC;                      /* CONFIG/OPERATION                  */
  uint32 ulQSEIS;                       /* Any                               */
  uint32 ulQSEIE;                       /* Any                               */
  uint32 ulQSEID;                       /* Any                               */
  uint32 Reserved25[3];                 /* Reserved                          */
  uint32 ulQGACST0;                     /* CONFIG/OPERATION                  */
  uint32 ulQGACST1;                     /* CONFIG/OPERATION                  */
  uint32 ulQGACST2;                     /* CONFIG/OPERATION                  */
  uint32 ulQGALIT0;                     /* CONFIG/OPERATION                  */
  uint32 ulQGALIT1;                     /* CONFIG/OPERATION                  */
  uint32 ulQGALIT2;                     /* CONFIG/OPERATION                  */
  uint32 ulQGAEN0;                      /* CONFIG/OPERATION                  */
  uint32 ulQGAEN1;                      /* CONFIG/OPERATION                  */
  uint32 ulQGIGS;                       /* CONFIG/OPERATION                  */
  uint32 ulQGJC;                        /* CONFIG/OPERATION                  */
  uint32 ulQGTL;                        /* CONFIG/OPERATION                  */
  uint32 ulQGSE;                        /* CONFIG/OPERATION                  */
  uint32 ulQGCC;                        /* CONFIG/OPERATION                  */
  uint32 ulQGATL0;                      /* CONFIG/OPERATION                  */
  uint32 ulQGATL1;                      /* CONFIG/OPERATION                  */
  uint32 ulQGATL2;                      /* CONFIG/OPERATION                  */
  uint32 ulQGOCST0;                     /* N/A                               */
  uint32 ulQGOCST1;                     /* N/A                               */
  uint32 ulQGOCST2;                     /* N/A                               */
  uint32 ulQGOLIT0;                     /* N/A                               */
  uint32 ulQGOLIT1;                     /* N/A                               */
  uint32 ulQGOLIT2;                     /* N/A                               */
  uint32 ulQGOEN0;                      /* N/A                               */
  uint32 ulQGOEN1;                      /* N/A                               */
  uint32 ulQGTR0;                       /* CONFIG/OPERATION                  */
  uint32 ulQGTR1;                       /* CONFIG/OPERATION                  */
  uint32 ulQGTR2;                       /* N/A                               */
  uint32 ulQGFSMS;                      /* N/A                               */
  uint32 Reserved26[16];                /* Reserved                          */
  uint32 ulQTMIS;                       /* Any                               */
  uint32 ulQTMIE;                       /* Any                               */
  uint32 ulQTMID;                       /* Any                               */
  uint32 ulQTEIS;                       /* Any                               */
  uint32 ulQTEIE;                       /* Any                               */
  uint32 ulQTEID;                       /* Any                               */
  uint32 Reserved27[2];                 /* Reserved                          */
  uint32 ulQMEC;                        /* CONFIG/OPERATION                  */
  uint32 ulQMMC;                        /* CONFIG/OPERATION                  */
  uint32 ulQRFDC;                       /* CONFIG/OPERATION                  */
  uint32 ulQYFDC;                       /* CONFIG/OPERATION                  */
  uint32 ulQVTCMC[16];                  /* CONFIG/OPERATION                  */
  uint32 ulQMCBSC[16];                  /* CONFIG/OPERATION                  */
  uint32 ulQMCIRC[16];                  /* CONFIG/OPERATION                  */
  uint32 ulQMEBSC[16];                  /* CONFIG/OPERATION                  */
  uint32 ulQMEIRC[16];                  /* CONFIG/OPERATION                  */
  uint32 ulQMCFC;                       /* CONFIG/OPERATION                  */
  uint32 Reserved28[3];                 /* Reserved                          */
  uint32 ulQMEIS;                       /* Any                               */
  uint32 ulQMEIE;                       /* Any                               */
  uint32 ulQMEID;                       /* Any                               */
  uint32 Reserved29;                    /* Reserved                          */
  uint32 ulQSMFC[16];                   /* N/A                               */
  uint32 ulQMSPPC[16];                  /* N/A                               */
  uint32 ulQMSRPC[16];                  /* N/A                               */
  uint32 ulQGPPC[8];                    /* N/A                               */
  uint32 ulQGRPC[8];                    /* N/A                               */
  uint32 ulQMDPC[16];                   /* N/A                               */
  uint32 ulQMGPC[16];                   /* N/A                               */
  uint32 ulQMYPC[16];                   /* N/A                               */
  uint32 ulQMRPC[16];                   /* N/A                               */
  uint32 ulMQSTMACU;                    /* N/A                               */
  uint32 ulMQSTMACD;                    /* N/A                               */
  uint32 ulMQSTMAMU;                    /* N/A                               */
  uint32 ulMQSTMAMD;                    /* N/A                               */
  uint32 ulMQSFTVL;                     /* N/A                               */
  uint32 ulMQSFTVLM;                    /* N/A                               */
  uint32 ulMQSFTMSD;                    /* N/A                               */
  uint32 ulMQSFTGMI;                    /* N/A                               */
  uint32 Reserved30[92];                /* Reserved                          */
  uint32 ulDCR;                         /* Any                               */
  uint32 Reserved31[7];                 /* Reserved                          */
  uint32 ulTDRRR;                       /* Any                               */
  uint32 ulTDRRD;                       /* Any                               */
  uint32 ulTCRRR;                       /* Any                               */
  uint32 ulTCRRD;                       /* Any                               */
  uint32 ulTCD;                         /* N/A                               */
  uint32 Reserved32[3];                 /* Reserved                          */
  uint32 ulRDRRR;                       /* Any                               */
  uint32 ulRDRRD;                       /* Any                               */
  uint32 ulRCRRR;                       /* Any                               */
  uint32 ulRCRRD;                       /* Any                               */
  uint32 ulVRR;                         /* Any                               */
} Eth_ETNDE_MHDRegType;                                                                                                 /* PRQA S 0639 # JV-01 */

typedef struct STag_Eth_ETNDE_GPTPCFGRegType
{
  uint32 ulGTIVt;                       /* Any                               */
  uint32 ulGTOVt0;                      /* Any                               */
  uint32 ulGTOVt1;                      /* Any                               */
  uint32 ulGTOVt2;                      /* Any                               */
  uint32 ulGAVTPt;                      /* Any                               */
  uint32 ulGCTt0;                       /* Any                               */
  uint32 ulGCTt1;                       /* Any                               */
  uint32 ulGCTt2;                       /* Any                               */
} Eth_ETNDE_GPTPCFGRegType;

typedef struct STag_Eth_ETNDE_GPTPCMPRegType
{
  uint32 ulTCDg;                        /* Any                               */
  uint32 ulTCMg;                        /* Any                               */
  uint32 ulGTMg;                        /* Any                               */
  uint32 ulTLSCVg;                      /* Any                               */
  uint32 ulTNVg;                        /* Any                               */
  uint32 ulTPVg;                        /* Any                               */
  uint32 ulTFFCECg;                     /* Any                               */
  uint32 ulTCSTg;                       /* Any                               */
} Eth_ETNDE_GPTPCMPRegType;

typedef struct STag_Eth_ETNDE_GPTPDBGRegType
{
  uint32 ulTCDFRg;                      /* When debug_mode signal is 1       */
  uint32 ulTCDFRRg;                     /* When debug_mode signal is 1       */
} Eth_ETNDE_GPTPDBGRegType;

typedef struct STag_Eth_ETNDE_GPTMARegType
{
  uint32 ulTME;                         /* Any                               */
  uint32 ulTMD;                         /* Any                               */
  uint32 Reserved0[2];                  /* Reserved                          */
  Eth_ETNDE_GPTPCFGRegType stGptpCfg[8];/* 0x0810 - 0x090F                   */
  uint32 Reserved1[64];                 /* Reserved                          */
  Eth_ETNDE_GPTPCMPRegType stGptpCmp[8];/* 0x0A10 - 0x0B0F                   */
  uint32 Reserved2[64];                 /* Reserved                          */
  uint32 ulGIS;                         /* Any                               */
  uint32 ulGIE;                         /* Any                               */
  uint32 ulGID;                         /* Any                               */
  uint32 Reserved3;                     /* Reserved                          */
  Eth_ETNDE_GPTPDBGRegType stGptpDbg[8];/* When debug_mode signal is 1       */
} Eth_ETNDE_GPTMARegType;

typedef struct STag_Eth_ETNE_GWCARegType
{
  uint32 ulRRR0;                        /* Any                               */
  uint32 ulRRR1;                        /* Any                               */
  uint32 Reserved0[62];                 /* Reserved                          */
  uint32 ulRIPV;                        /* Any                               */
  uint32 Reserved1[63];                 /* Reserved                          */
  uint32 ulRMIS0;                       /* Any                               */
  uint32 ulRMIE0;                       /* Any                               */
  uint32 ulRMID0;                       /* Any                               */
  uint32 Reserved2;                     /* Reserved                          */
  uint32 ulRMIS1;                       /* Any                               */
  uint32 ulRMIE1;                       /* Any                               */
  uint32 ulRMID1;                       /* Any                               */
  uint32 Reserved3;                     /* Reserved                          */
  uint32 ulREIS0;                       /* Any                               */
  uint32 ulREIE0;                       /* Any                               */
  uint32 ulREID0;                       /* Any                               */
  uint32 Reserved4;                     /* Reserved                          */
  uint32 ulREIS1;                       /* Any                               */
  uint32 ulREIE1;                       /* Any                               */
  uint32 ulREID1;                       /* Any                               */
  uint32 Reserved5;                     /* Reserved                          */
  uint32 ulRDIS;                        /* Any                               */
  uint32 ulRDIE;                        /* Any                               */
  uint32 ulRDID;                        /* Any                               */
  uint32 Reserved6;                     /* Reserved                          */
  uint32 ulRMDIOIS;                     /* Any                               */
  uint32 ulRMDIOIE;                     /* Any                               */
  uint32 ulRMDIOID;                     /* Any                               */
  uint32 Reserved7[41];                 /* Reserved                          */
  uint32 ulRDASK;                       /* Any                               */
  uint32 ulRDAU;                        /* Any                               */
  uint32 ulRDASKC;                      /* DEBUG                             */
  uint32 Reserved8[61];                 /* Reserved                          */
  uint32 ulGWMC;                        /* Any                               */
  uint32 Reserved9[3];                  /* Reserved                          */
  uint32 ulGWRTC[8];                    /* CONFIG                            */
  uint32 ulGWERDQD[8];                  /* CONFIG                            */
  uint32 ulGWPRDQD[8];                  /* CONFIG                            */
  uint32 ulGWEWMC[8];                   /* CONFIG                            */
  uint32 ulGWPWMC[8];                   /* CONFIG                            */
  uint32 ulGWRMSTLS;                    /* CONFIG                            */
  uint32 ulGWRMSTLR;                    /* CONFIG                            */
  uint32 ulGWRGC;                       /* CONFIG                            */
  uint32 Reserved10[17];                /* Reserved                          */
  uint32 ulGWMS;                        /* Any                               */
  uint32 ulGWRR;                        /* CONFIG                            */
  uint32 Reserved11[2];                 /* Reserved                          */
  uint32 ulGWERDQM0;                    /* Any                               */
  uint32 ulGWERDQM1;                    /* Any                               */
  uint32 ulGWERDQM2;                    /* Any                               */
  uint32 ulGWERDQM3;                    /* Any                               */
  uint32 ulGWERDQM4;                    /* Any                               */
  uint32 ulGWERDQM5;                    /* Any                               */
  uint32 ulGWERDQM6;                    /* Any                               */
  uint32 ulGWERDQM7;                    /* Any                               */
  uint32 ulGWPRDQM0;                    /* Any                               */
  uint32 ulGWPRDQM1;                    /* Any                               */
  uint32 ulGWPRDQM2;                    /* Any                               */
  uint32 ulGWPRDQM3;                    /* Any                               */
  uint32 ulGWPRDQM4;                    /* Any                               */
  uint32 ulGWPRDQM5;                    /* Any                               */
  uint32 ulGWPRDQM6;                    /* Any                               */
  uint32 ulGWPRDQM7;                    /* Any                               */
  uint32 ulGWERDQML0;                   /* Any                               */
  uint32 ulGWERDQML1;                   /* Any                               */
  uint32 ulGWERDQML2;                   /* Any                               */
  uint32 ulGWERDQML3;                   /* Any                               */
  uint32 ulGWERDQML4;                   /* Any                               */
  uint32 ulGWERDQML5;                   /* Any                               */
  uint32 ulGWERDQML6;                   /* Any                               */
  uint32 ulGWERDQML7;                   /* Any                               */
  uint32 ulGWPRDQML0;                   /* Any                               */
  uint32 ulGWPRDQML1;                   /* Any                               */
  uint32 ulGWPRDQML2;                   /* Any                               */
  uint32 ulGWPRDQML3;                   /* Any                               */
  uint32 ulGWPRDQML4;                   /* Any                               */
  uint32 ulGWPRDQML5;                   /* Any                               */
  uint32 ulGWPRDQML6;                   /* Any                               */
  uint32 ulGWPRDQML7;                   /* Any                               */
  uint32 ulGWRDQM;                      /* Any                               */
  uint32 ulGWRDQML;                     /* Any                               */
  uint32 Reserved12[2];                 /* Reserved                          */
  uint32 ulGWRMSTSS;                    /* Any                               */
  uint32 ulGWRMSTSR0;                   /* Any                               */
  uint32 ulGWRMSTSR1;                   /* Any                               */
  uint32 Reserved13[21];                /* Reserved                          */
  uint32 ulGWEIS0;                      /* Any                               */
  uint32 ulGWEIE0;                      /* Any                               */
  uint32 ulGWEID0;                      /* Any                               */
  uint32 Reserved14;                    /* Reserved                          */
  uint32 ulGWEIS1;                      /* Any                               */
  uint32 ulGWEIE1;                      /* Any                               */
  uint32 ulGWEID1;                      /* Any                               */
  uint32 Reserved15[57];                /* Reserved                          */
  uint32 ulGWTSRR;                      /* DEBUG                             */
  uint32 ulGWTSRRR0;                    /* DEBUG                             */
  uint32 ulGWTSRRR1;                    /* DEBUG                             */
  uint32 ulGWTSRRR2;                    /* DEBUG                             */
  uint32 ulGWDRR;                       /* DEBUG                             */
  uint32 ulGWDRRR0;                     /* DEBUG                             */
  uint32 ulGWDRRR1;                     /* DEBUG                             */
  uint32 ulGWDRRR2;                     /* DEBUG                             */
  uint32 Reserved16;                    /* Reserved                          */
  uint32 Reserved17;                    /* Reserved                          */
  uint32 ulGWDRRR5;                     /* DEBUG                             */
} Eth_ETNE_GWCARegType;

typedef struct STag_Eth_ETNE_MFWDRegType
{
  uint32 ulFWCEPC;                      /* Any                               */
  uint32 ulFWCLPC;                      /* Any                               */
  uint32 Reserved0[2];                  /* Reserved                          */
  uint32 ulFWMPC0;                      /* Any                               */
  uint32 ulFWMPC1;                      /* Any                               */
  uint32 ulFWMPC2;                      /* Any                               */
  uint32 ulFWMPC3;                      /* Any                               */
  uint32 ulFWMPC4;                      /* Any                               */
  uint32 Reserved1[3];                  /* Reserved                          */
  uint32 ulFWASPLC;                     /* Any                               */
  uint32 ulFWSTPDBC;                    /* Any                               */
  uint32 ulFWSTPBC;                     /* Any                               */
  uint32 ulFWSTPLEC;                    /* Any                               */
  uint32 ulFWSTPFC;                     /* Any                               */
  uint32 ulFWSTPFLC;                    /* Any                               */
  uint32 ulFWMC;                        /* Any                               */
  uint32 ulFWAC;                        /* Any                               */
  uint32 ulFWPVPC;                      /* Any                               */
  uint32 ulFWPVIC;                      /* Any                               */
  uint32 Reserved2[2];                  /* Reserved                          */
  uint32 ulFWPVCC[3];                   /* Any                               */
  uint32 Reserved3[13];                 /* Reserved                          */
  uint32 ulFWSOVPC0;                    /* Any                               */
  uint32 ulFWSOVPC1;                    /* Any                               */
  uint32 ulFWSOVPC2;                    /* Any                               */
  uint32 ulFWSOVPC3;                    /* Any                               */
  uint32 ulFWSOVPC4;                    /* Any                               */
  uint32 Reserved4[3];                  /* Reserved                          */
  uint32 ulFWMDANFV[6];                 /* Any                               */
  uint32 Reserved5[26];                 /* Reserved                          */
  uint32 ulFWMDANFCSD[6];               /* Any                               */
  uint32 Reserved6[26];                 /* Reserved                          */
  uint32 ulFWSPBFE;                     /* Any                               */
  uint32 ulFWSPBFC0;                    /* Any                               */
  uint32 ulFWSPBFC1;                    /* Any                               */
  uint32 Reserved7[17];                 /* Reserved                          */
  uint32 ulFWSPBFC2[6];                 /* Any                               */
  uint32 Reserved8[26];                 /* Reserved                          */
  uint32 ulFWSPBFC3[6];                 /* Any                               */
  uint32 Reserved9[26];                 /* Reserved                          */
  uint32 ulFWMAFC0;                     /* Any                               */
  uint32 Reserved10;                    /* Reserved                          */
  uint32 ulFWMAFC1;                     /* Any                               */
  uint32 Reserved11;                    /* Reserved                          */
  uint32 ulFWMDFC00;                    /* Any                               */
  uint32 ulFWMDFC01;                    /* Any                               */
  uint32 ulFWMDFC02;                    /* Any                               */
  uint32 ulFWMDFC03;                    /* Any                               */
  uint32 ulFWMDFC04;                    /* Any                               */
  uint32 ulFWMDFC05;                    /* Any                               */
  uint32 Reserved12[2];                 /* Reserved                          */
  uint32 ulFWMDFC10;                    /* Any                               */
  uint32 ulFWMDFC11;                    /* Any                               */
  uint32 ulFWMDFC12;                    /* Any                               */
  uint32 ulFWMDFC13;                    /* Any                               */
  uint32 ulFWMDFC14;                    /* Any                               */
  uint32 ulFWMDFC15;                    /* Any                               */
  uint32 Reserved13[2];                 /* Reserved                          */
  uint32 ulFWBFC0;                      /* Any                               */
  uint32 ulFWBFC1;                      /* Any                               */
  uint32 ulFWBFC2;                      /* Any                               */
  uint32 ulFWBFC3;                      /* Any                               */
  uint32 ulFWBFC4;                      /* Any                               */
  uint32 ulFWBFC5;                      /* Any                               */
  uint32 Reserved14[2];                 /* Reserved                          */
  uint32 ulFWPVC0;                      /* Any                               */
  uint32 ulFWPVC1;                      /* Any                               */
  uint32 ulFWPVC2;                      /* Any                               */
  uint32 ulFWPVC3;                      /* Any                               */
  uint32 ulFWPVC4;                      /* Any                               */
  uint32 Reserved15[3];                 /* Reserved                          */
  uint32 ulFWDTTC;                      /* Any                               */
  uint32 ulFWAFCE;                      /* Any                               */
  uint32 Reserved16[54];                /* Reserved                          */
  uint32 ulFWQMEC[5];                   /* Any                               */
  uint32 Reserved17[27];                /* Reserved                          */
  uint32 ulFWQMMC[5];                   /* Any                               */
  uint32 Reserved18[27];                /* Reserved                          */
  uint32 ulFWQRFDC[5];                  /* Any                               */
  uint32 Reserved19[27];                /* Reserved                          */
  uint32 ulFWQYFDC[5];                  /* Any                               */
  uint32 Reserved20[27];                /* Reserved                          */
  uint32 ulFWQVTCMC[20];                /* Any                               */
  uint32 Reserved21[140];               /* Reserved                          */
  uint32 ulFWQMCBSC[40];                /* Any                               */
  uint32 Reserved22[216];               /* Reserved                          */
  uint32 ulFWQMCIRC[40];                /* Any                               */
  uint32 Reserved23[216];               /* Reserved                          */
  uint32 ulFWQMEBSC[40];                /* Any                               */
  uint32 Reserved24[216];               /* Reserved                          */
  uint32 ulFWQMEIRC[40];                /* Any                               */
  uint32 Reserved25[216];               /* Reserved                          */
  uint32 ulFWQMCFC[5];                  /* Any                               */
  uint32 Reserved26[27];                /* Reserved                          */
  uint32 ulFWQIUEC[5];                  /* Any                               */
  uint32 Reserved27[27];                /* Reserved                          */
  uint32 ulFWQGMC[5];                   /* Any                               */
  uint32 Reserved28[27];                /* Reserved                          */
  uint32 ulBPWMC1[6];                   /* Any                               */
  uint32 Reserved29[26];                /* Reserved                          */
  uint32 ulBPWMC2;                      /* Any                               */
  uint32 ulBPWMC3;                      /* Any                               */
  uint32 ulBPWML;                       /* Any                               */
  uint32 ulBPPFL;                       /* Any                               */
  uint32 ulFWATC;                       /* Any                               */
  uint32 Reserved30[3];                 /* Reserved                          */
  uint32 ulFWQTL0;                      /* Any                               */
  uint32 ulFWQTL1;                      /* Any                               */
  uint32 ulFWQTL2;                      /* Any                               */
  uint32 ulFWQTL3;                      /* Any                               */
  uint32 ulFWQTLR;                      /* Any                               */
  uint32 Reserved31[3];                 /* Reserved                          */
  uint32 ulFWMTL0;                      /* Any                               */
  uint32 ulFWMTL1;                      /* Any                               */
  uint32 ulFWMTL2;                      /* Any                               */
  uint32 ulFWMTL3;                      /* Any                               */
  uint32 ulFWMTLR;                      /* Any                               */
  uint32 Reserved32[3];                 /* Reserved                          */
  uint32 ulFWVTL0;                      /* Any                               */
  uint32 ulFWVTL1;                      /* Any                               */
  uint32 ulFWVTL2;                      /* Any                               */
  uint32 ulFWVTL3;                      /* Any                               */
  uint32 ulFWVTLR;                      /* Any                               */
  uint32 Reserved33[3];                 /* Reserved                          */
  uint32 ulFWSTL0;                      /* Any                               */
  uint32 ulFWSTL1;                      /* Any                               */
  uint32 ulFWSTL2;                      /* Any                               */
  uint32 ulFWSTL3;                      /* Any                               */
  uint32 ulFWSTLR;                      /* Any                               */
  uint32 Reserved34[3];                 /* Reserved                          */
  uint32 ulHFSR;                        /* Any                               */
  uint32 ulMHC0;                        /* Any                               */
  uint32 ulMHC1;                        /* Any                               */
  uint32 ulVHC;                         /* Any                               */
  uint32 ulSHC0;                        /* Any                               */
  uint32 ulSHC1;                        /* Any                               */
  uint32 ulQHC0;                        /* Any                               */
  uint32 ulQHC1;                        /* Any                               */
  uint32 ulQHC2;                        /* Any                               */
  uint32 Reserved35[271];               /* Reserved                          */
  uint32 ulFWGGTS[5];                   /* Any                               */
  uint32 Reserved36[27];                /* Reserved                          */
  uint32 ulFWGACST0[5];                 /* Any                               */
  uint32 Reserved37[27];                /* Reserved                          */
  uint32 ulFWGACST1[5];                 /* Any                               */
  uint32 Reserved38[27];                /* Reserved                          */
  uint32 ulFWGACST2[5];                 /* Any                               */
  uint32 Reserved39[27];                /* Reserved                          */
  uint32 ulFWGALIT0[5];                 /* Any                               */
  uint32 Reserved40[27];                /* Reserved                          */
  uint32 ulFWGALIT1[5];                 /* Any                               */
  uint32 Reserved41[27];                /* Reserved                          */
  uint32 ulFWGALIT2[5];                 /* Any                               */
  uint32 Reserved42[27];                /* Reserved                          */
  uint32 ulFWGAEN0[5];                  /* Any                               */
  uint32 Reserved43[27];                /* Reserved                          */
  uint32 ulFWGAEN1[5];                  /* Any                               */
  uint32 Reserved44[27];                /* Reserved                          */
  uint32 ulFWGIGS[5];                   /* Any                               */
  uint32 Reserved45[27];                /* Reserved                          */
  uint32 ulFWGJC[5];                    /* Any                               */
  uint32 Reserved46[27];                /* Reserved                          */
  uint32 ulFWGTL[5];                    /* Any                               */
  uint32 Reserved47[27];                /* Reserved                          */
  uint32 ulFWGSE[5];                    /* Any                               */
  uint32 Reserved48[27];                /* Reserved                          */
  uint32 ulFWGCC[5];                    /* Any                               */
  uint32 Reserved49[27];                /* Reserved                          */
  uint32 ulFWGATL0;                     /* Any                               */
  uint32 ulFWGATL1;                     /* Any                               */
  uint32 ulFWGATL2;                     /* Any                               */
  uint32 Reserved50[61];                /* Reserved                          */
  uint32 ulFWMFV[40];                   /* Any                               */
  uint32 Reserved51[216];               /* Reserved                          */
  uint32 ulFWQMFMC[5];                  /* Any                               */
  uint32 Reserved52[251];               /* Reserved                          */
  uint32 ulFWARA0;                      /* Any                               */
  uint32 ulFWARA1;                      /* Any                               */
  uint32 Reserved53[2];                 /* Reserved                          */
  uint32 ulFWTEN0;                      /* Any                               */
  uint32 ulFWTEN1;                      /* Any                               */
  uint32 Reserved54[2];                 /* Reserved                          */
  uint32 ulFWQTS0;                      /* Any                               */
  uint32 ulFWQTS1;                      /* Any                               */
  uint32 ulFWQTS2;                      /* Any                               */
  uint32 ulFWQTSR0;                     /* Any                               */
  uint32 ulFWQTSR1;                     /* Any                               */
  uint32 Reserved55[3];                 /* Reserved                          */
  uint32 ulFWMTS0;                      /* Any                               */
  uint32 ulFWMTS1;                      /* Any                               */
  uint32 ulFWMTSR0;                     /* Any                               */
  uint32 ulFWMTSR1;                     /* Any                               */
  uint32 ulFWMTSR2;                     /* Any                               */
  uint32 Reserved56[3];                 /* Reserved                          */
  uint32 ulFWVTS;                       /* Any                               */
  uint32 ulFWVTSR0;                     /* Any                               */
  uint32 ulFWVTSR1;                     /* Any                               */
  uint32 ulFWVTSR2;                     /* Any                               */
  uint32 ulFWVTSR3;                     /* Any                               */
  uint32 Reserved57[3];                 /* Reserved                          */
  uint32 ulFWSTS0;                      /* Any                               */
  uint32 ulFWSTS1;                      /* Any                               */
  uint32 ulFWSTSR0;                     /* Any                               */
  uint32 ulFWSTSR1;                     /* Any                               */
  uint32 ulFWSTSR2;                     /* Any                               */
  uint32 Reserved58[3];                 /* Reserved                          */
  uint32 ulFWMSAR;                      /* Any                               */
  uint32 ulFWDRFDP;                     /* Any                               */
  uint32 ulFWEFS;                       /* Any                               */
  uint32 ulFWRR;                        /* Any                               */
  uint32 Reserved59[20];                /* Reserved                          */
  uint32 ulFWCEP;                       /* Any                               */
  uint32 ulFWCMP;                       /* Any                               */
  uint32 ulFWCRP;                       /* Any                               */
  uint32 ulFWBPC;                       /* Any                               */
  uint32 ulFWMPC;                       /* Any                               */
  uint32 ulFWRDC;                       /* Any                               */
  uint32 ulFWBPPC;                      /* Any                               */
  uint32 ulFWBPLC;                      /* Any                               */
  uint32 Reserved60[56];                /* Reserved                          */
  uint32 ulFWQMSRPC[40];                /* Any                               */
  uint32 Reserved61[216];               /* Reserved                          */
  uint32 ulFWQGRPC[40];                 /* Any                               */
  uint32 Reserved62[216];               /* Reserved                          */
  uint32 ulFWQMDPC[40];                 /* Any                               */
  uint32 Reserved63[216];               /* Reserved                          */
  uint32 ulFWQMGPC[30];                 /* Any                               */
  uint32 Reserved64[216];               /* Reserved                          */
  uint32 ulFWQMYPC[40];                 /* Any                               */
  uint32 Reserved65[216];               /* Reserved                          */
  uint32 ulFWQMRPC[40];                 /* Any                               */
  uint32 Reserved66[216];               /* Reserved                          */
  uint32 ulFWGOCST0[5];                 /* Any                               */
  uint32 Reserved67[27];                /* Reserved                          */
  uint32 ulFWGOCST1[5];                 /* Any                               */
  uint32 Reserved68[27];                /* Reserved                          */
  uint32 ulFWGOCST2[5];                 /* Any                               */
  uint32 Reserved69[27];                /* Reserved                          */
  uint32 ulFWGOLIT0[5];                 /* Any                               */
  uint32 Reserved70[27];                /* Reserved                          */
  uint32 ulFWGOLIT1[5];                 /* Any                               */
  uint32 Reserved71[27];                /* Reserved                          */
  uint32 ulFWGOLIT2[5];                 /* Any                               */
  uint32 Reserved72[27];                /* Reserved                          */
  uint32 ulFWGOEN0[5];                  /* Any                               */
  uint32 Reserved73[27];                /* Reserved                          */
  uint32 ulFWGOEN1[5];                  /* Any                               */
  uint32 Reserved74[27];                /* Reserved                          */
  uint32 ulFWGTR0;                      /* Any                               */
  uint32 ulFWGTR1;                      /* Any                               */
  uint32 ulFWGTR2;                      /* Any                               */
  uint32 Reserved75;                    /* Reserved                          */
  uint32 ulFWGFSMS;                     /* Any                               */
  uint32 Reserved76[123];               /* Reserved                          */
  uint32 ulFWEIS0;                      /* Any                               */
  uint32 ulFWEIE0;                      /* Any                               */
  uint32 ulFWEID0;                      /* Any                               */
  uint32 Reserved77;                    /* Reserved                          */
  uint32 ulFWEIS1;                      /* Any                               */
  uint32 ulFWEIE1;                      /* Any                               */
  uint32 ulFWEID1;                      /* Any                               */
  uint32 Reserved78[25];                /* Reserved                          */
  uint32 ulFWEIS2[5];                   /* Any                               */
  uint32 Reserved79[27];                /* Reserved                          */
  uint32 ulFWEIE2[5];                   /* Any                               */
  uint32 Reserved80[27];                /* Reserved                          */
  uint32 ulFWEID2[5];                   /* Any                               */
  uint32 Reserved81[27];                /* Reserved                          */
  uint32 ulFWEIS3;                      /* Any                               */
  uint32 ulFWEIE3;                      /* Any                               */
  uint32 ulFWEID3;                      /* Any                               */
  uint32 Reserved82;                    /* Reserved                          */
  uint32 ulFWEIS4;                      /* Any                               */
  uint32 ulFWEIE4;                      /* Any                               */
  uint32 ulFWEID4;                      /* Any                               */
  uint32 Reserved83[25];                /* Reserved                          */
  uint32 ulFWEIS5[5];                   /* Any                               */
  uint32 Reserved84[27];                /* Reserved                          */
  uint32 ulFWEIE5[5];                   /* Any                               */
  uint32 Reserved85[27];                /* Reserved                          */
  uint32 ulFWEID5[5];                   /* Any                               */
  uint32 Reserved86[27];                /* Reserved                          */
  uint32 ulFWEIS6[5];                   /* Any                               */
  uint32 Reserved87[27];                /* Reserved                          */
  uint32 ulFWEIE6[5];                   /* Any                               */
  uint32 Reserved88[27];                /* Reserved                          */
  uint32 ulFWEID6[5];                   /* Any                               */
  uint32 Reserved89[27];                /* Reserved                          */
  uint32 ulFWMIS0;                      /* Any                               */
  uint32 ulFWMIE0;                      /* Any                               */
  uint32 ulFWMID0;                      /* Any                               */
  uint32 Reserved90;                    /* Reserved                          */
  uint32 ulFWMIS1;                      /* Any                               */
  uint32 ulFWMIE1;                      /* Any                               */
  uint32 ulFWMID1;                      /* Any                               */
  uint32 Reserved91;                    /* Reserved                          */
  uint32 ulFWMIS2;                      /* Any                               */
  uint32 ulFWMIE2;                      /* Any                               */
  uint32 ulFWMID2;                      /* Any                               */
  uint32 Reserved92[21];                /* Reserved                          */
  uint32 ulFWQTR;                       /* When FWRR.QTRR = 1                */
  uint32 ulFWQTRR0;                     /* When FWRR.QTRR = 1                */
  uint32 ulFWQTRR1;                     /* When FWRR.QTRR = 1                */
  uint32 ulFWQTRR2;                     /* When FWRR.QTRR = 1                */
  uint32 ulFWQTRR3;                     /* When FWRR.QTRR = 1                */
  uint32 Reserved93[3];                 /* Reserved                          */
  uint32 ulFWMTR;                       /* When FWRR.MTRR = 1                */
  uint32 ulFWMTRR0;                     /* When FWRR.MTRR = 1                */
  uint32 ulFWMTRR1;                     /* When FWRR.MTRR = 1                */
  uint32 ulFWMTRR2;                     /* When FWRR.MTRR = 1                */
  uint32 ulFWVTR;                       /* When FWRR.VTRR = 1                */
  uint32 ulFWVTRR;                      /* When FWRR.VTRR = 1                */
  uint32 Reserved94[2];                 /* Reserved                          */
  uint32 ulFWSTR;                       /* When FWRR.STRR = 1                */
  uint32 ulFWSTRR0;                     /* When FWRR.STRR = 1                */
  uint32 ulFWSTRR1;                     /* When FWRR.STRR = 1                */
  uint32 ulFWSTRR2;                     /* When FWRR.STRR = 1                */
  uint32 ulFWFBTR;                      /* When FWRR.BPRR = 1                */
  uint32 ulFWFBTRR;                     /* When FWRR.BPRR = 1                */
} Eth_ETNE_MFWDRegType;                                                                                                 /* PRQA S 0639 # JV-01 */

typedef struct STag_Eth_ETNE_ETHARegType
{
  uint32 ulEAMC;                        /* Any                               */
  uint32 Reserved0[3];                  /* Reserved                          */
  uint32 ulEATMTC;                      /* CONFIG                            */
  uint32 ulEATAFS;                      /* CONFIG                            */
  uint32 ulEAPRC;                       /* CONFIG                            */
  uint32 Reserved1;                     /* Reserved                          */
  uint32 ulEATMS[8];                    /* CONFIG                            */
  uint32 ulEATDQD[8];                   /* CONFIG                            */
  uint32 ulEATWMC[8];                   /* CONFIG                            */
  uint32 ulEAVTMC;                      /* CONFIG                            */
  uint32 ulEAVTC;                       /* CONFIG                            */
  uint32 Reserved2[18];                 /* Reserved                          */
  uint32 ulTPTPC;                       /* CONFIG                            */
  uint32 ulTTML;                        /* CONFIG                            */
  uint32 ulTTJ;                         /* CONFIG                            */
  uint32 Reserved3[9];                  /* Reserved                          */
  uint32 ulTCC;                         /* CONFIG/OPERATION                  */
  uint32 ulTCS;                         /* CONFIG/OPERATION                  */
  uint32 Reserved4;                     /* Reserved                          */
  uint32 ulTGS;                         /* CONFIG/OPERATION                  */
  uint32 ulTACST0;                      /* CONFIG/OPERATION                  */
  uint32 ulTACST1;                      /* CONFIG/OPERATION                  */
  uint32 ulTACST2;                      /* CONFIG/OPERATION                  */
  uint32 Reserved5;                     /* Reserved                          */
  uint32 ulTALIT0;                      /* CONFIG/OPERATION                  */
  uint32 ulTALIT1;                      /* CONFIG/OPERATION                  */
  uint32 ulTALIT2;                      /* CONFIG/OPERATION                  */
  uint32 Reserved6;                     /* Reserved                          */
  uint32 ulTAEN[2];                     /* CONFIG/OPERATION                  */
  uint32 Reserved7[2];                  /* Reserved                          */
  uint32 ulTASFE;                       /* CONFIG/OPERATION                  */
  uint32 Reserved8[3];                  /* Reserved                          */
  uint32 ulTACLL0;                      /* CONFIG/OPERATION                  */
  uint32 ulTACLL1;                      /* CONFIG/OPERATION                  */
  uint32 ulTACLL2;                      /* CONFIG/OPERATION                  */
  uint32 Reserved9;                     /* Reserved                          */
  uint32 ulCACC;                        /* CONFIG/OPERATION                  */
  uint32 ulCCS;                         /* CONFIG/OPERATION                  */
  uint32 Reserved10[2];                 /* Reserved                          */
  uint32 ulCAIV[8];                     /* CONFIG OPERATION                  */
  uint32 ulCAUL[8];                     /* CONFIG OPERATION                  */
  uint32 Reserved11[20];                /* Reserved                          */
  uint32 ulEAMS;                        /* Read only                         */
  uint32 ulEAOTM;                       /* Read only                         */
  uint32 Reserved12[2];                 /* Reserved                          */
  uint32 ulEADQM[8];                    /* Read only                         */
  uint32 Reserved13[4];                 /* Reserved                          */
  uint32 ulTOCST0;                      /* Read only                         */
  uint32 ulTOCST1;                      /* Read only                         */
  uint32 ulTOCST2;                      /* Read only                         */
  uint32 Reserved14;                    /* Reserved                          */
  uint32 ulTOLIT0;                      /* Read only                         */
  uint32 ulTOLIT1;                      /* Read only                         */
  uint32 ulTOLIT2;                      /* Read only                         */
  uint32 Reserved15;                    /* Reserved                          */
  uint32 ulTOEN0;                       /* Read only                         */
  uint32 ulTOEN1;                       /* Read only                         */
  uint32 Reserved16[2];                 /* Reserved                          */
  uint32 ulTOSFE;                       /* Read only                         */
  uint32 Reserved17[3];                 /* Reserved                          */
  uint32 ulTCLR0;                       /* Read only                         */
  uint32 ulTCLR1;                       /* Read only                         */
  uint32 ulTCLR2;                       /* Read only                         */
  uint32 Reserved18;                    /* Reserved                          */
  uint32 ulTSMS;                        /* Read only                         */
  uint32 Reserved19[3];                 /* Reserved                          */
  uint32 ulCOCC;                        /* Read only                         */
  uint32 Reserved20[3];                 /* Reserved                          */
  uint32 ulCOIV[8];                     /* Read only                         */
  uint32 ulCOUL[8];                     /* Read only                         */
  uint32 Reserved21[4];                 /* Reserved                          */
  uint32 ulEAEIS0;                      /* Any                               */
  uint32 ulEAEIE0;                      /* Any                               */
  uint32 ulEAEID0;                      /* Any                               */
  uint32 Reserved22;                    /* Reserved                          */
  uint32 ulEAEIS1;                      /* Any                               */
  uint32 ulEAEIE1;                      /* Any                               */
  uint32 ulEAEID1;                      /* Any                               */
  uint32 Reserved23;                    /* Reserved                          */
  uint32 ulEAEIS2;                      /* Any                               */
  uint32 ulEAEIE2;                      /* Any                               */
  uint32 ulEAEID2;                      /* Any                               */
  uint32 Reserved24;                    /* Reserved                          */
  uint32 ulEAMIS;                       /* Any                               */
  uint32 ulEAMIE;                       /* Any                               */
  uint32 ulEAMID;                       /* Any                               */
  uint32 Reserved25;                    /* Reserved                          */
  uint32 ulTCD;                         /* Read only                         */
  uint32 ulEATQD;                       /* CONFIG OPERATION                  */
  uint32 Reserved26[2];                 /* Reserved                          */
  uint32 ulEADRR;                       /* CONFIG OPERATION                  */
  uint32 ulEADRRR0;                     /* Read only                         */
  uint32 ulEADRRR1;                     /* Read only                         */
  uint32 Reserved27[1833];              /* Reserved                          */
  Eth_ETNDE_RMACSRegType stRMACSys;     /* RMACSys & RMAC regs               */
} Eth_ETNE_ETHARegType;

typedef struct STag_Eth_ETND_ESRegType                                                                                  /* PRQA S 3630 # JV-01 */
{
  Eth_ETNDE_AXIBMIRegType stAXIBMI;     /* AXIBMI regs                       */
  uint32 Reserved0[249];                /* Reserved                          */
  Eth_ETNDE_MHDRegType stMHD;           /* MHD regs                          */
  uint32 Reserved1[235];                /* Reserved                          */
  Eth_ETNDE_RMACSRegType stRMACSys;     /* RMACSys & RMAC regs               */
} Eth_ETND_ESRegType;

typedef struct STag_Eth_ETNDE_PWRCTLRegType
{
  uint32 ulETNDEzSGSDS;                 /* <PWRCTLz_base> + 0000H            */
  uint32 ulETNDEzSGCLKSEL;              /* <PWRCTLz_base> + 0004H            */
  uint32 ulETNDEzSGRCIE;                /* <PWRCTLz_base> + 0008H            */
} Eth_ETNDE_PWRCTLRegType;

typedef struct STag_Eth_ETNDE_SGMIIRegType                                                                              /* PRQA S 3630 # JV-01 */
{
  uint32 ulETNDEzSGOPMC;                /* <SGMIIz_base> + 0000H             */
  uint32 ulETNDEzSGOPMS;                /* <SGMIIz_base> + 0004H             */
  uint32 ulETNDEzSGSRST;                /* <SGMIIz_base> + 0008H             */
  uint32 ulETNDEzSGINTS;                /* <SGMIIz_base> + 000CH             */
  uint32 ulETNDEzSGINTM;                /* <SGMIIz_base> + 0010H             */
  uint32 ulETNDEzSGLTVC;                /* <SGMIIz_base> + 0014H             */
  uint32 ulETNDEzSGCECT;                /* <SGMIIz_base> + 0018H             */
  uint32 ulETNDEzSGRECT;                /* <SGMIIz_base> + 001CH             */
  uint32 Reserved0[120];                /* Reserved                          */
  Eth_ETNDE_PWRCTLRegType stPWRCTL;     /* PWRCTL regs                       */
} Eth_ETNDE_SGMIIRegType;

typedef struct STag_Eth_ETND_RegType
{
  Eth_ETND_ESRegType *pES[ETH_TOTAL_CTRL_CONFIG];
  Eth_ETNDE_SGMIIRegType *pSGMII;
  Eth_ETNDE_GPTMARegType *pGPTMA;
} Eth_ETND_RegType;

typedef struct STag_Eth_ETNE_RegType
{
  Eth_ETNE_MFWDRegType *pMFWD;
  Eth_ETNE_GWCARegType *pGWCA;
  Eth_ETNDE_GPTMARegType *pGPTMA;
  Eth_ETNDE_AXIBMIRegType *pAXIBMI;
  Eth_ETNE_ETHARegType *pEA[ETH_TOTAL_CTRL_CONFIG];
  Eth_ETNDE_SGMIIRegType *pSGMII[ETH_TOTAL_CTRL_CONFIG];
} Eth_ETNE_RegType;

typedef struct STag_Eth_TxBufferType                                                                                    /* PRQA S 3630 # JV-01 */
{
  Eth_BufHandlerType *pBufferHdr;
  boolean blTxOngoing;
  uint8 ucDummy[3];
} Eth_TxBufferType;

typedef struct STag_Eth_ETND_EICRegType                                                                                 /* PRQA S 3630 # JV-01 */
{
  uint16 usINTETNDw0;                    /* CPU data request 0 (TX[0,4]/RX[0,4]/TS) */
  uint16 usINTETNDw1;                    /* CPU data request 1 (TX[1,5]/RX[1,5])    */
  uint16 usINTETNDw2;                    /* CPU data request 2 (TX[2,6]/RX[0,6])    */
  uint16 usINTETNDw3;                    /* CPU data request 3 (TX[3,7]/RX[0,7])    */
  uint16 usINTETNDw4;                    /* MDIO related PHY control                */
  uint16 usINTETNDw5;                    /* MagicPacket/LPI                         */
  uint16 usINTETNDw6;                    /* General                                 */
  uint16 usINTETNDw7;                    /* GPTP interrupt                          */
} Eth_ETND_EICRegType;

typedef struct STag_Eth_ETNE_EICRegType
{
  uint16 usINTETNE00;                    /* CPU data request 0 (TX[0,4]/RX[0,4]/TS)                */
  uint16 usINTETNE01;                    /* CPU data request 1 (TX[1,5]/RX[1,5])                   */
  uint16 usINTETNE02;                    /* CPU data request 2 (TX[2,6]/RX[0,6])                   */
  uint16 usINTETNE03;                    /* CPU data request 3 (TX[3,7]/RX[0,7])                   */
  uint16 usINTETNE04;                    /* GPTP timer compare flag                                */
  uint16 usINTETNE05;                    /* Phy control of MDIO write read access completed status */
  uint16 usINTETNE06;                    /* MagicPacket/LPI                                        */
  uint16 usINTETNE07;                    /* Error state                                            */
  uint16 usINTETNE08;                    /* Internal state                                         */
} Eth_ETNE_EICRegType;

/***********************************************************************************************************************
**                                                Inline Macros                                                       **
***********************************************************************************************************************/

LOCAL_INLINE uint32 ETH_NS2HZ(const uint32 nano_sec)                                                                    /* PRQA S 1055 # JV-01 */
{
  return (ETH_CPUCLK_MHZ * nano_sec) / 1000UL;                                                                          /* PRQA S 3383 # JV-01 */
}

/* Caculate gPTP timer increment value */
LOCAL_INLINE uint32 ETH_GPTP_SET_GTIV(const uint32 clock_hz)                                                            /* PRQA S 1055 # JV-01 */
{
  return (uint32)0x08000000 * ((uint32)1000000000 / clock_hz);                                                          /* PRQA S 3384 # JV-01 */
}

/* AXI Transmit Status */
LOCAL_INLINE uint32 ETH_GET_TDIS_STAT(const uint32 idx, const uint32 val, const uint32 base)                            /* PRQA S 1055 # JV-01 */
{
  #if (ETH_USING_MACRO == ETH_MACRO_ETND)
  (void)idx;
  (void)base;
  return val;
  #else
  return val & (0x000000FFUL << (idx * base));
  #endif
}

LOCAL_INLINE uint32 ETH_GET_TX_INT_STAT(const uint32 idx, const uint32 que, const uint32 val, const uint32 base)        /* PRQA S 1055 # JV-01 */
{
  #if (ETH_USING_MACRO == ETH_MACRO_ETND)
  (void)idx;
  (void)base;
  return val & (1UL << que);
  #else
  return val & ((1UL << que) << (idx * base));
  #endif
}

/* AXI Receive Status */
LOCAL_INLINE uint32 ETH_GET_RX_INT_STAT(const uint32 idx, const uint32 que, const uint32 val, const uint32 base)        /* PRQA S 1055 # JV-01 */
{
  #if (ETH_USING_MACRO == ETH_MACRO_ETND)
  (void)idx;
  (void)base;
  return val & (1UL << que);
  #else
  return val & ((1UL << que) << (idx * base));
  #endif
}

/* Rx descriptor full Status */
LOCAL_INLINE uint32 ETH_GET_RX_FULL_INT_STAT(const uint32 idx, const uint32 que, const uint32 val, const uint32 base)   /* PRQA S 1055 # JV-01 */
{
  #if (ETH_USING_MACRO == ETH_MACRO_ETND)
  (void)idx;
  (void)base;
  return val & (1UL << que);
  #else
  return val & ((1UL << que) << (idx * base));
  #endif
}

/* TAS startup time calculation */
LOCAL_INLINE float64 ETH_GET_TAS_STARTUP_TIME(const volatile Eth_ETNDE_GPTMARegType * const gptp)                       /* PRQA S 1055 # JV-01 */
{
  /* gPTP timer + 1 seconds */
  return (float64)gptp->stGptpCfg[ETH_GPTP_TIMER_DOMAIN].ulGCTt0 +
         ((float64)gptp->stGptpCfg[ETH_GPTP_TIMER_DOMAIN].ulGCTt1 * (float64)1000000000.0) +
         ((float64)gptp->stGptpCfg[ETH_GPTP_TIMER_DOMAIN].ulGCTt2 * (float64)1000000000.0 * (float64)4294967296.0) +
          (float64)1000000000.0;
}

LOCAL_INLINE uint32 ETH_CALC_TAS_TACST0(const float64 base)                                                             /* PRQA S 1055 # JV-01 */
{
  const float64 LdbCalcTemp = base / 4294967296.0 / 4294967296.0;
  return (uint32)(LdbCalcTemp - (float64)((uint64)(LdbCalcTemp / (float64)32768.0) * (uint64)32768.0));                 /* PRQA S 3384, 4395, 4394 # JV-01, JV-01, JV-01 */
}

LOCAL_INLINE uint32 ETH_CALC_TAS_TACST1(const float64 base)                                                             /* PRQA S 1055 # JV-01 */
{
  const float64 LdbCalcTemp = base / 4294967296.0;
  return (uint32)(LdbCalcTemp - (float64)((uint64)(LdbCalcTemp / (float64)4294967296.0) * (uint64)4294967296.0));       /* PRQA S 3384, 4395, 4394 # JV-01, JV-01, JV-01 */
}

LOCAL_INLINE uint32 ETH_CALC_TAS_TACST2(const float64 base)                                                             /* PRQA S 1055 # JV-01 */
{
  return (uint32)(base - (float64)((uint64)(base / (float64)4294967296.0) * (uint64)4294967296.0));                     /* PRQA S 3384, 4395, 4394 # JV-01, JV-01, JV-01 */
}

LOCAL_INLINE void ETH_TXRAM_INIT_DESC_CONFIG(Eth_ExtDescType * const desc_chain, const uint32 dptr)                     /* PRQA S 1055 # JV-01 */
{
  desc_chain->stHeader.ulDie   = (uint32)0UL;
  desc_chain->stHeader.ulAxie  = (uint32)0UL;
  desc_chain->stHeader.ulDse   = (uint32)0UL;
  desc_chain->stHeader.ulInfo0 = (uint32)0UL;
  desc_chain->stHeader.ulDs = (uint32)ETH_ETND_2KB_FRAME_SIZE;
  desc_chain->ulDptr = dptr;
  desc_chain->ulInfo1[0] = (uint32)ETH_ETND_INIT_FRAME_LENGTH;
  desc_chain->ulInfo1[1] = 0UL;
}

LOCAL_INLINE void ETH_TXRAM_INIT_EOS_DESC_CONFIG(Eth_ExtDescType * const desc_chain, const uint32 dptr)                 /* PRQA S 1055 # JV-01 */
{
  desc_chain->stHeader.ulDie   = (uint32)ETH_TX_DESC_DIE_ENABLE;
  desc_chain->stHeader.ulAxie  = (uint32)0UL;
  desc_chain->stHeader.ulDse   = (uint32)0UL;
  desc_chain->stHeader.ulInfo0 = (uint32)0UL;
  desc_chain->stHeader.ulDs = (uint32)0UL;
  desc_chain->ulDptr = dptr;
  desc_chain->ulInfo1[0] = 0UL;
  desc_chain->ulInfo1[1] = 0UL;
}

/* Mask patterns to be used for the register corruption checking */
#define ETH_ETNDE_ECMR_CHECK_MASK 0x04BF02C3UL


/***********************************************************************************************************************
**                                                Function Prototypes                                                 **
***********************************************************************************************************************/
#define ETH_START_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"

extern FUNC(Eth_ExtRxStatusType, ETH_PRIVATE_CODE)
  Eth_RxQueueProcess(CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulFifoIdx);

extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_TxRxDescConfig(CONST(uint32, AUTOMATIC) LulCtrlIdx);

extern FUNC(TickType, ETH_PRIVATE_CODE) Eth_GetTimeOutValue(CONST(TickType, AUTOMATIC) LusTimeOutCount_Init);

extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_TxDescLearning(
  CONSTP2VAR(volatile Eth_ETNDE_AXIBMIRegType, AUTOMATIC, REGSPACE) LpAxiRegs,
  CONST(uint32, AUTOMATIC) LulEntryNum,
  CONST(uint32, AUTOMATIC) LulChainAddr);

extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE) Eth_SGMIIInit(
  CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(Eth_OptionType, AUTOMATIC) LenBypassMode);

#if (ETH_GLOBAL_TIME_SUPPORT == STD_ON)

extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE)
  Eth_HwSetIncrementTimeForGptp(CONST(uint32, AUTOMATIC) LulCtrlIdx, CONST(uint32, AUTOMATIC) LulIncVal);

extern FUNC(Std_ReturnType, ETH_PRIVATE_CODE)
  Eth_HwSetOffsetTimeForGptp(CONST(uint32, AUTOMATIC) LulCtrlIdx,
                             CONSTP2CONST(Eth_TimeStampType, AUTOMATIC, ETH_APPL_DATA) LpTimeOffsetPtr);

#if (ETH_USING_MACRO == ETH_MACRO_ETND)
extern FUNC(void, ETH_PRIVATE_CODE) Eth_TsDescConfig(CONST(uint32, AUTOMATIC) LulCtrlIdx);
#else
extern FUNC(void, ETH_PRIVATE_CODE) Eth_TsDescConfig(void);
#endif

#endif

#define ETH_STOP_SEC_PRIVATE_CODE
#include "Eth_MemMap.h"

#endif /* !ETH_ETNDE_LLDRIVER_H_ */
/***********************************************************************************************************************
**                                                    End of File                                                     **
***********************************************************************************************************************/
