/*====================================================================================================================*/
/* Project      = RH850/X2x AR4 MCAL Development                                                                      */
/* Module       = Pwm_ATU_LLDriver.c                                                                                  */
/* SW-VERSION   = 1.5.0                                                                                               */
/*                                                                                                                    */
/*====================================================================================================================*/
/*                                                       COPYRIGHT                                                    */
/*====================================================================================================================*/
/* (c) 2020-2022 Renesas Electronics Corporation. All rights reserved.                                                */
/*====================================================================================================================*/
/* Purpose:                                                                                                           */
/* API function implementations of PWM Driver                                                                         */
/*                                                                                                                    */
/*====================================================================================================================*/
/*                                                                                                                    */
/* Unless otherwise agreed upon in writing between your company and Renesas Electronics Corporation the following     */
/* shall apply!                                                                                                       */
/*                                                                                                                    */
/* Warranty Disclaimer                                                                                                */
/*                                                                                                                    */
/* There is no warranty of any kind whatsoever granted by Renesas. Any warranty is expressly disclaimed and excluded  */
/* by Renesas, either expressed or implied, including but not limited to those for non-infringement of intellectual   */
/* property, merchantability and/or fitness for the particular purpose.                                               */
/*                                                                                                                    */
/* Renesas shall not have any obligation to maintain, service or provide bug fixes for the supplied Product(s) and/or */
/* the Application.                                                                                                   */
/*                                                                                                                    */
/* Each User is solely responsible for determining the appropriateness of using the Product(s) and assumes all risks  */
/* associated with its exercise of rights under this Agreement, including, but not limited to the risks and costs of  */
/* program errors, compliance with applicable laws, damage to or loss of data, programs or equipment, and             */
/* unavailability or interruption of operations.                                                                      */
/*                                                                                                                    */
/* Limitation of Liability                                                                                            */
/*                                                                                                                    */
/* In no event shall Renesas be liable to the User for any incidental, consequential, indirect, or punitive damage    */
/* (including but not limited to lost profits) regardless of whether such liability is based on breach of contract,   */
/* tort, strict liability, breach of warranties, failure of essential purpose or otherwise and even if advised of the */
/* possibility of such damages. Renesas shall not be liable for any services or products provided by third party      */
/* vendors, developers or consultants identified or referred to the User by Renesas in connection with the Product(s) */
/* and/or the Application.                                                                                            */
/*                                                                                                                    */
/*====================================================================================================================*/
/* Environment:                                                                                                       */
/*                                              Devices:        X2x                                                   */
/*====================================================================================================================*/
/***********************************************************************************************************************
**                                              Revision Control History                                              **
***********************************************************************************************************************/
/*                                                                                                                    **
 ** 1.5.0: 14/11/2022 : Update header description for all function                                                    **
 ** 1.4.4: 13/07/2022 : Remove QAC message 1006, 2841                                                                 **
 **        29/06/2022 : Set cycle value to one when period is zero in Pwm_HW_SetPeriodAndDuty                         **
 **        22/06/2022 : Added QAC message and header for message 1006, 2841                                           **
 **        10/06/2022 : Update cycle value to register CRLDExy when period = 0 in Pwm_HW_SetPeriodAndDuty             **
 ** 1.4.3: 23/05/2022 : Remove redundant QAC header of message 1281, 0488                                             **
 **        09/05/2022 : Removed "else" statement don't use when no action required, updated QAC message               **
 **        18/04/2022 : Changed MAX_DUTY_CYCLE to PWM_MAX_DUTY_CYCLE_VALUE and                                        **
 **                             MIN_DUTY_CYCLE to PWM_MIN_DUTY_CYCLE_VALUE                                            **
 **        08/04/2022 : Corrected error content from "Pwm_ATU_LLDriver.c : Mismatch in Release Patch Version" to      **
 **                     "Pwm_ATU_LLDriver.c : Mismatch in Release Revision Version"                                   **
 ** 1.4.2  09/03/2022 : Add QAC message and header for message 2962, 2993, 3416, 2982, 2877, 2996, 2983               **
 **        26/02/2022 : Updated SW VERSION to 1.4.2                                                                   **
 ** 1.4.1: 06/12/2021 : Add condition check ucTimerUnitType for HW IP TAU in all loop                                 **
 **                     "for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_CHANNELS_CONFIG; LulCount++)" **
 **        11/12/2021 : Added loop for check  HWIPType before call HWUnit in function Pwm_HW_ATU_InitUnit             **
 ** 1.4.0: 15/10/2021 : Added ELSE branch for IF-ELSE-IF in Pwm_HW_SetDutyToIdleState, Pwm_HW_EnableNotification to   **
 **                     fix QAC message Msg(3:2004)                                                                   **
 **        11/11/2021 : Replace ATU5 with ATU                                                                         **
 **        20/11/2021 : Add condition macro PWM_ATU_UNIT_USED to support U2Bx                                         **
 ** 1.3.2: 06/09/2021 : Removed Critical Section in Pwm_HW_SynchronousStart, Pwm_HW_SynchronousStop.                  **
 **        04/08/2021 : Device specific implementation moved from Pwm common file.                                    **
 **                     Error checking at Pwm_HW_SetPeriodAndDuty move to Pwm_SetPeriodAndDuty API.                   **
 **                     Format source code unified within module.                                                     **
 ** 1.3.1: 02/07/2021 : Add QAC message 9.5.0                                                                         **
 **                     Format source code to 120 characters                                                          **
 **                     Improve Violation tag (remove START/END)                                                      **
 **        22/05/2021 : Fixed QAC CWM, removed Msg 2985                                                               **
 **        15/05/2021 : Fixed Coding Rule and tailor space                                                            **
 **        12/05/2021 : Changed the IF condition in Pwm_HW_ATU_InitChannel to follow the Coding Rule                  **
 **        29/04/2021 : Updated the IF condition in Pwm_HW_ATU_InitChannel to avoid the Compiler warning              **
 ** 1.2.0: 28/07/2020 : Release                                                                                       **
 ** 1.1.0: 19/06/2020 : Release                                                                                       **
 ** 1.0.3: 20/05/2020 : Removed CRITICAL_SECTION in Pwm_HW_SynchronousInit function                                   **
 ** 1.0.2: 19/05/2020 : Changed uiPolarity to ucPolarity                                                              **
 **                     Changed "uiIdleLevel" to "ucIdleLevel"                                                        **
 ** 1.0.1: 18/05/2020 : Changed uint8 to uint32                                                                       **
 ** 1.0.0: 24/03/2020 : Initial Version.                                                                              **
 **                                                                                                                   */
/**********************************************************************************************************************/
/***********************************************************************************************************************
**                                                  Include Section                                                   **
***********************************************************************************************************************/
/* Included for module version information and other types declarations */
#include "Pwm.h"
/* Included for RAM variable declarations */
#include "Pwm_Ram.h"
#if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
/* Included for the declaration of the critical section protection functions */
#include "SchM_Pwm.h"
#endif
#if (PWM_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
/* Included for declaration of the function Dem_SetEventStatus() */
#include "Dem.h"
#endif
#if (PWM_DEV_ERROR_DETECT == STD_ON)
/* Included for the declaration of Det_ReportError() */
#include "Det.h"
#endif

#if (PWM_ATU_UNIT_USED == STD_ON)
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"

STATIC FUNC(Pwm_PeriodType, PWM_PRIVATE_CODE)
  Pwm_HW_CalculateDuty(Pwm_PeriodType LddAbsolutePeriod, Pwm_PeriodType LddRelativeDuty);
STATIC FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_InitUnitRegister(void);
STATIC FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_InitChannelRegister(void);
#if ((PWM_SET_OUTPUT_TO_IDLE_API == STD_ON) || (PWM_SYNC_START_SUPPORT == STD_ON) || (PWM_DE_INIT_API == STD_ON))
STATIC FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_SetDutyToIdleState(
  const Pwm_ChannelConfigType *LpChannelConfig, Pwm_ATUEChDataRegs *LpATUEChDataRegs);
#endif

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif
/***********************************************************************************************************************
**                                                Version Information                                                 **
***********************************************************************************************************************/
#if (PWM_ATU_UNIT_USED == STD_ON)
/* AUTOSAR release version information */
#define PWM_ATU_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION    PWM_AR_RELEASE_MAJOR_VERSION_VALUE
#define PWM_ATU_LLDRIVER_C_AR_RELEASE_MINOR_VERSION    PWM_AR_RELEASE_MINOR_VERSION_VALUE                               /* PRQA S 0791 # JV-01 */
#define PWM_ATU_LLDRIVER_C_AR_RELEASE_REVISION_VERSION PWM_AR_RELEASE_REVISION_VERSION_VALUE
/* Module software version information */
#define PWM_ATU_LLDRIVER_C_SW_MAJOR_VERSION            PWM_SW_MAJOR_VERSION_VALUE
#define PWM_ATU_LLDRIVER_C_SW_MINOR_VERSION            PWM_SW_MINOR_VERSION_VALUE
#define PWM_ATU_LLDRIVER_C_SW_PATCH_VERSION            PWM_SW_PATCH_VERSION_VALUE
/***********************************************************************************************************************
**                                                   Version Check                                                    **
***********************************************************************************************************************/
#if (PWM_ATU_LLDRIVER_AR_RELEASE_MAJOR_VERSION != PWM_ATU_LLDRIVER_C_AR_RELEASE_MAJOR_VERSION)
  #error "Pwm_ATU_LLDriver.c : Mismatch in Release Major Version"
#endif
#if (PWM_ATU_LLDRIVER_AR_RELEASE_MINOR_VERSION != PWM_ATU_LLDRIVER_C_AR_RELEASE_MINOR_VERSION)
  #error "Pwm_ATU_LLDriver.c : Mismatch in Release Minor Version"
#endif
#if (PWM_ATU_LLDRIVER_AR_RELEASE_REVISION_VERSION != PWM_ATU_LLDRIVER_C_AR_RELEASE_REVISION_VERSION)
  #error "Pwm_ATU_LLDriver.c : Mismatch in Release Revision Version"
#endif
#if (PWM_ATU_LLDRIVER_SW_MAJOR_VERSION != PWM_ATU_LLDRIVER_C_SW_MAJOR_VERSION)
  #error "Pwm_ATU_LLDriver.c : Mismatch in Software Major Version"
#endif
#if (PWM_ATU_LLDRIVER_SW_MINOR_VERSION != PWM_ATU_LLDRIVER_C_SW_MINOR_VERSION)
  #error "Pwm_ATU_LLDriver.c : Mismatch in Software Minor Version"
#endif
#if (PWM_ATU_LLDRIVER_SW_PATCH_VERSION != PWM_ATU_LLDRIVER_C_SW_PATCH_VERSION)
  #error "Pwm_ATU_LLDriver.c : Mismatch in Software Patch Version"
#endif
/***********************************************************************************************************************
**                                                    Global Data                                                     **
***********************************************************************************************************************/

/***********************************************************************************************************************
**                                               Coding Rule Violations                                               **
***********************************************************************************************************************/
/* Message (4:0311)    : Dangerous pointer cast results in loss of const qualification.                               */
/* Rule                : CERTCCM EXP05, MISRA C:2012 Rule-11.8                                                        */
/* JV-01 Justification : This is to achieve throughput in the code.                                                   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0316)    : Cast from a pointer to void to a pointer to object type.                                     */
/* Rule                : MISRA C:2012 Rule-11.5                                                                       */
/* JV-01 Justification : A cast should not be performed between a pointer to object type and a different pointer to   */
/*                       object type.                                                                                 */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:0317)    : Implicit conversion from a pointer to void to a pointer to object type.                      */
/* Rule                : MISRA C:2012 Rule-11.5                                                                       */
/* JV-01 Justification : This is done as per implementation requirement.                                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact               */
/**********************************************************************************************************************/
/* Message (2:0499)    : Right operand of shift operator is greater than or equal to the width of the essential type  */
/*                       of the left operand.                                                                         */
/* Rule                : CERTCCM INT34, MISRA C:2012 Rule-12.2                                                        */
/* JV-01 Justification : The result has been ensured by casting to the same Type                                      */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (7:0791)    : Macro identifier does not differ from other macro identifier(s) (e.g. '%s') within the       */
/*                       specified number of significant characters.                                                  */
/* Rule                : CERTCCM DCL23, MISRA C:2012 Rule-5.4                                                         */
/* JV-01 Justification : This macro identifier is following AUTOSAR standard rule (Symbolic Name or Published         */
/*                       Macro's name), so this is accepted.                                                          */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:1532)    : The function '%1s' is only referenced in one translation unit - but not the one in which it  */
/*                       is defined.                                                                                  */
/* Rule                : CERTCCM DCL19, MISRA C:2012 Rule-8.7                                                         */
/* JV-01 Justification : This macro identifier is following AUTOSAR standard rule  (Symbolic Name or Published        */
/*                       Macro's name), so this is accepted                                                           */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:2004)    : No concluding 'else' exists in this 'if'-'else'-'if' statement.                              */
/* Rule                : CERTCCM MSC01, MISRA C:2012 Rule-15.7                                                        */
/* JV-01 Justification : The "else" statement with empty content is removed to improve readability.                   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2814)    : Possible: Dereference of NULL pointer.                                                       */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : This is accepted, due to the implementation is following hardware specification or it was    */
/*                       checked or do not dereference to NULL pointer.                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/* JV-02 Justification : This is accepted since the pointer to the structure that generates the value will depend on  */
/*                       user's settings and the range of that value can't be NULL.                                   */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/* JV-03 Justification : This is accepted since the pointer to the structure that generates the base address of the   */
/*                       related register can't be NULL.                                                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2824)    : Possible: Arithmetic operation on NULL pointer.                                              */
/* Rule                : CERTCCM EXP34                                                                                */
/* JV-01 Justification : This is accepted, due to the implementation is following hardware specification.             */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2844)    : Possible: Dereference of an invalid pointer value.                                           */
/* Rule                : CERTCCM ARR30                                                                                */
/* JV-01 Justification : It is specific for device register accessing and confirmed has no issue in software behavior.*/
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (6:2877)    : This loop will never be executed more than once.                                             */
/* Rule                : CERTCCM MSC07, MISRA C:2012 Dir-4.1                                                          */
/* JV-01 Justification : This loop will only be executed at least once, depends on user configuration.                */
/*       Verification  : This is Hardware Specification, X2x only provides 1 Unit. So it is not having any impact.    */
/**********************************************************************************************************************/
/* Message (5:2916)    : Definite: Storing the address of an object in a pointer that has greater lifetime.           */
/* Rule                : MISRA C:2012 Rule-18.6                                                                       */
/* JV-01 Justification : This message propose that the pointer should not be used many times and should be stored     */
/*                       locally with shorter lifetime                                                                */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2934)    : Possible: Computing an invalid pointer value.                                                */
/* Rule                : CERTCCM ARR30, ARR37, ARR38, EXP08                                                           */
/* JV-01 Justification : This message prevent existing of an out of range pointer                                     */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (6:2962)    : Apparent: Using value of uninitialized automatic object '%s'.                                */
/* Rule                : CERTCCM EXP33, MISRA C:2012 Rule-9.1                                                         */
/* JV-01 Justification : It will be initialized based on scope of 'if' statements where at least an 'if' statement    */
/*                       will be executed that will initialize the variable.                                          */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:2963)    : Suspicious: Using value of uninitialized automatic object '%s'.                              */
/* Rule                : CERTCCM EXP33                                                                                */
/* JV-01 Justification : It will be initialized based on scope of 'if' statements where at least an 'if' statement    */
/*                       will be executed that will initialize the variable in question.                              */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2982)    : This assignment is redundant. The value of this object is never used before being modified.  */
/* Rule                : CERTCCM MSC07, MSC13, MISRA C:2012 Rule-2.2                                                  */
/* JV-01 Justification : The variable needs to be initialized before using it.                                        */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:2983)    : This assignment is redundant. The value of this object is never subsequently used.           */
/* Rule                : CERTCCM MSC07, MSC13, MISRA C:2012 Rule-2.2                                                  */
/* JV-01 Justification : The value is to increment the pointer to the next item.                                      */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (6:2993)    : The value of this 'do - while' loop controlling expression is always 'false'. The loop will  */
/*                       only be executed once.                                                                       */
/* Rule                : MISRA C:2012 Rule-14.3                                                                       */
/* JV-01 Justification : This loop will only be executed atleast once, depends on user configuration.                 */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (6:2996)    : The result of this logical operation is always 'false'.                                      */
/* Rule                : MISRA C:2012 Rule-2.2                                                                        */
/* JV-01 Justification : Depending on device status, there is case where the 'if' will return 'true'.                 */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3383)    : Cannot identify wraparound guard for unsigned arithmetic expression.                         */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : It can still result in values that are out of range for the intended use, as intuitive       */
/*                       "invariants" may not hold                                                                    */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (1:3384)    : Cannot identify wraparound guard for dependent unsigned arithmetic expression.               */
/* Rule                : CERTCCM INT30                                                                                */
/* JV-01 Justification : In order to effectively guard against overflow and wraparound at all stages, the expression  */
/*                       should be split up into individual dynamic operations, with their own guards where applicable*/
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (3:3416)    : Logical operation performed on expression with persistent side effects.                      */
/* Rule                : CERTCCM EXP45                                                                                */
/* JV-01 Justification : Logical operation accesses volatile object which is a register access. All register          */
/*                       addresses are generated with volatile qualifier. There is no impact on the functionality     */
/*                       due to this conditional check for mode change.                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3432)    : Simple macro argument expression is not parenthesized.                                       */
/* Rule                : MISRA C:2012 Rule-20.7                                                                       */
/* JV-01 Justification : Compiler keyword (macro) is defined and used followed AUTOSAR standard rule. It is accepted. */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3469)    : This usage of a function-like macro looks like it could be replaced by an equivalent         */
/*                       function call.                                                                               */
/* Rule                : MISRA C:2012 Dir-4.9                                                                         */
/* JV-01 Justification : This message indicates that a candidate macro may be suitable for replacement by a           */
/*                       function, based on an actual call-site and the arguments passed to it there. (Other uses of  */
/*                       the macro may not necessarily be suitable for replacement.)                                  */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (2:3678)    : The object referenced by '%1s' is not modified through it, so '%1s' could be declared with   */
/*                       type '%2s'.                                                                                  */
/* Rule                : MISRA C:2012 Rule-8.13                                                                       */
/* JV-01 Justification : This is accepted. It just an advise for improve safety by reducing the possibility that the  */
/*                       referenced data is unintentionally modified through an unexpected alias and improves         */
/*                       clarity by indicating that the referenced data is not intended to be modified through this   */
/*                       alias or those depending on it                                                               */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:4399)    : An expression which is the result of a ~ or << operation has been cast to a wider type.      */
/* Rule                : MISRA C:2012 Rule-10.8                                                                       */
/* JV-01 Justification : The result has been ensured by casting to the same Type                                      */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/
/* Message (4:5087)    : Use of #include directive after code fragment.                                               */
/* Rule                : MISRA C:2012 Rule-20.1                                                                       */
/* JV-01 Justification : This is done as per Memory Requirement, (MEMMAP003 - Specification of Memory Mapping).       */
/*       Verification  : However, part of the code is verified manually and it is not having any impact.              */
/**********************************************************************************************************************/

/***********************************************************************************************************************
**                                                Function Definitions                                                **
***********************************************************************************************************************/

/***********************************************************************************************************************
** Function Name         : Pwm_HW_ATU_InitChannel
**
** Service ID            : NA
**
** Description           : This is PWM Driver Component support function. This function sets the clock pre-scaler,
**                         PWM mode, Period, Duty cycle and polarity for all configured channels.
**                         This function also disables the interrupts (Notifications) and resets the interrupt request
**                         pending flags.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Pre-conditions        : NA
**
** Global Variables used : Pwm_GpChannelConfig, Pwm_GaaFpsSupport_Flag, Pwm_GaaNotifStatus
**
** Functions invoked     : Pwm_HW_CalculateDuty, Pwm_HW_InitChannelRegister
**
** Registers Used        : ATUENR, TSTRE, SSTREx, PSCREx, PSCCRExy, TCREx, TSCREx, TIEREx, TOCREx, CYLRExy, DTRExy,
**                         CRLDExy, DRLDExy, RLDCREx, PSCRx, EIC176 to EIC211,IMR5,IMR6,IMR17 EIC548 to 551
**
** Reference ID          : PWM_DUD_ACT_061, PWM_DUD_ACT_061_REG001, PWM_DUD_ACT_061_REG002, PWM_DUD_ACT_061_REG003,
** Reference ID          : PWM_DUD_ACT_061_REG004, PWM_DUD_ACT_061_REG005, PWM_DUD_ACT_061_REG006,
** Reference ID          : PWM_DUD_ACT_061_REG007, PWM_DUD_ACT_061_REG008, PWM_DUD_ACT_061_REG009,
** Reference ID          : PWM_DUD_ACT_061_REG010, PWM_DUD_ACT_061_REG011, PWM_DUD_ACT_061_REG012,
** Reference ID          : PWM_DUD_ACT_061_REG013, PWM_DUD_ACT_061_REG014, PWM_DUD_ACT_061_GBL001,
** Reference ID          : PWM_DUD_ACT_061_GBL002, PWM_DUD_ACT_061_GBL003
***********************************************************************************************************************/
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_ATU_InitChannel(void)                                                               /* PRQA S 1532 # JV-01 */
{
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfig;
  /* ATU config data */
  P2CONST(Pwm_ATUChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpATUChannelConfigData;
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfigNext;
  /* Local Pointer used for each Timer-E channel control registers */
  P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE) LpATUEChCntrlRegs;                                                    /* PRQA S 3432 # JV-01 */
  /* Local Pointer used for each Timer-E channel data registers */
  P2VAR(Pwm_ATUEChDataRegs, AUTOMATIC, REGSPACE) LpATUEChDataRegs;                                                      /* PRQA S 3432 # JV-01 */
  /* Local Pointer used for each Timer-E channel control registers */
  P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelProp;
  /* Local variable for Period and duty */
  VAR(Pwm_PeriodType, AUTOMATIC) LddDefaultPeriod;
  VAR(Pwm_PeriodType, AUTOMATIC) LddDefaultduty;
  /* Local variable for count */
  VAR(uint32, AUTOMATIC) LulCount;
  VAR(uint32, AUTOMATIC) LaaFpsOut_Flag[PWM_TOTAL_CHANNELS_CONFIG + PWM_ONE];
  VAR(volatile uint8, AUTOMATIC) LucTimerUnitType;

  /* Update the configure pointer to point to the current Timer channel*/
  LpChannelConfig = Pwm_GpChannelConfig;
  /* Set ATU channel config data pointer */
  LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                         /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
  /* Set pointer for FIXED_PERIOD_SHIFTED Mode */
  LpChannelConfigNext = &Pwm_GpChannelConfig[PWM_ONE];

  /* Initialize an array LaaFpsOut_Flag to PWM_FALSE */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)(PWM_TOTAL_CHANNELS_CONFIG + PWM_ONE); LulCount++)
  {
    /* Flag for delayed output reference */
    LaaFpsOut_Flag[LulCount] = (uint32)PWM_FALSE;
  }

  /* Initialize Timer E register to clear previous setting */
  Pwm_HW_InitChannelRegister();

  /* Loop to set the attributes of Timer-E channels */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_CHANNELS_CONFIG; LulCount++)
  {
    /* Get the Timer Unit Type*/
    LucTimerUnitType = LpChannelConfig->ucTimerUnitType;                                                                /* PRQA S 2814, 2844 # JV-02, JV-01 */
    if ((uint8)PWM_HWIP_ATU == LucTimerUnitType)                                                                        /* PRQA S 3416 # JV-01 */
    {
      /* Get the pointer to the PWM Channel properties */
      LpChannelProp = (P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST))LpChannelConfig->pChannelProp;      /* PRQA S 0316 # JV-01 */
      /* Get the pointer to the PWM Channel control registers */
      LpATUEChCntrlRegs = (P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE))LpATUChannelConfigData->pATUEChCntrlRegs;    /* PRQA S 2814, 3432 # JV-02, JV-01 */
      /* Get the pointer to the PWM Channel data registers */
      LpATUEChDataRegs = (P2VAR(Pwm_ATUEChDataRegs, AUTOMATIC, REGSPACE))LpATUChannelConfigData->pATUEChDataRegs;       /* PRQA S 3432 # JV-01 */
      /* Get value of the channel period in local variable */
      LddDefaultPeriod = LpChannelProp->ddDefault_Period;                                                               /* PRQA S 2814 # JV-02 */
      /* Get value of the channel duty in local variable */
      LddDefaultduty = LpChannelProp->ddDefault_Duty;

      #if (PWM_SYNC_START_SUPPORT == STD_ON)
      if (PWM_FALSE == LpChannelConfig->blSyncStartReq)
      #endif
      {
        /* Condition to check type of channel PrescalerSelect for PWM channel */
        if (PWM_CH_SB_PRESCL_SEL == (uint8)(LpChannelProp->ucChannelSelMask & PWM_CH_SB_PRESCL_SEL))
        {
          /* Assign configured pre-scalar value to channel Pre-scalar register */
          *(LpATUChannelConfigData->pPSCCRExy_Address) = LpATUChannelConfigData->ucChSelPrescalVal;
        } /* else No action required */

        /* Assign the configured value of the channel period in to register */
        LpATUEChDataRegs->ulATUECYLRExy = (uint32)(LddDefaultPeriod << (uint32)PWM_BYTE_SHIFT_VALUE);                   /* PRQA S 2814 # JV-03 */
        /* Assign the configured value of the channel duty in to register */
        LpATUEChDataRegs->ulATUEDTRExy = Pwm_HW_CalculateDuty(LddDefaultPeriod, LddDefaultduty);
        /* Load the duty value to reload duty register */
        LpATUEChDataRegs->ulATUEDRLDExy = LpATUEChDataRegs->ulATUEDTRExy;
        /* Load the cycle value to reload cycle register */
        LpATUEChDataRegs->ulATUECRLDExy = LpATUEChDataRegs->ulATUECYLRExy;
        /* Enable Reload function for cycle match */
        LpATUEChCntrlRegs->ucATUERLDCREx =
           LpATUEChCntrlRegs->ucATUERLDCREx | (LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK);                /* PRQA S 2814 # JV-03 */

          /* condition to check starting polarity of PWM channel */
        if ((uint8)PWM_LOW == LpChannelConfig->ucPolarity)
        {
          /* PWM channel (TOE) is inverted */
          LpATUEChCntrlRegs->ucATUETOCREx =
            LpATUEChCntrlRegs->ucATUETOCREx | (LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK);
        } /* else No action required */
      } /* else No action required */
      /* End of blSyncStartReq is PWM_FALSE */

      #if (PWM_CLEAR_PENDING_INTERRUPT == STD_ON)
      /* Check if the Interrupt register is present for particular channel */
      if (NULL_PTR != LpATUChannelConfigData->pEICx_Address)
      {
        /* Clear the pending ISR request */
        RH850_SV_MODE_ICR_AND(16, LpATUChannelConfigData->pEICx_Address, PWM_CLEAR_PENDING_INTR_MASK);                  /* PRQA S 0499, 2814, 4399 # JV-01, JV-01, JV-01 */
        /* DummyRead & SYNCP */
        RH850_SV_MODE_REG_READ_ONLY(16, LpATUChannelConfigData->pEICx_Address);
        EXECUTE_SYNCP();
        /* Clear the pending ISR request */
        LpATUEChCntrlRegs->usATUETSCREx = LpChannelProp->usChannelIntSelMask;
      } /* else No action required */
      /* End of LpChannel->pEICx_Address != NULL_PTR */
      #endif /* (PWM_CLEAR_PENDING_INTERRUPT == STD_ON) */

      /* Enable the Interrupt processing of the current channel */
      RH850_SV_CLEAR_ICR_SYNCP(32, (LpATUChannelConfigData->pIMRx_Address), LpATUChannelConfigData->ulImrMaskValue);    /* PRQA S 2814 # JV-01 */
      /* Set the Notification status as PWM_FALSE */
      Pwm_GaaFpsSupport_Flag[LulCount] = PWM_FALSE;

      /* Condition to identify the reference channel of PWM_FIXED_PERIOD_SHIFTED class */
      if ((uint32)PWM_MAX_CHANNEL_ID_CONFIGURED != LulCount)
      {
        if ((uint32)PWM_FALSE == LaaFpsOut_Flag[LulCount])
        {
          /* If Last Channel isn't PWM_FIXED_PERIOD_SHIFTED, not needed this process. If Last two Channels are
             PWM_FIXED_PERIOD_SHIFTED, last Channel is LaaFpsOut_Flag[LulCount] == PWM TRUE */

           /* Increment the pointer to the next ATUE channel */
           /* Condition to check the channel class of Reference channel */
          if ((PWM_FIXED_PERIOD_SHIFTED == LpChannelConfig->enClassType)
              && (PWM_FIXED_PERIOD_SHIFTED == LpChannelConfigNext->enClassType))                                        /* PRQA S 2814, 2844 # JV-02, JV-01 */
          {
            /* Disable duty/cycle match interrupt */
            LpATUEChCntrlRegs->usATUETIEREx &= (uint16)(~(LpChannelProp->usChannelIntSelMask));
            /* Clear interrupt status  */
            LpATUEChCntrlRegs->usATUETSCREx = (LpChannelProp->usChannelIntSelMask & PWM_CYCLE_MATCH_MASK);
            /* Enable duty match interrupt */
            LpATUEChCntrlRegs->usATUETIEREx |= (LpChannelProp->usChannelIntSelMask & PWM_DUTY_MATCH_MASK);
            /* Flag for channel with class PWM_FIXED_PERIOD_SHIFTED to check the corresponding channel flag in ISR */
            Pwm_GaaFpsSupport_Flag[LulCount] = PWM_TRUE;
            /* Flag for delayed output reference */
            LaaFpsOut_Flag[LulCount + (uint32)PWM_ONE] = (uint32)PWM_TRUE;                                              /* PRQA S 3383 # JV-01 */
          } /* else No action required */
        } /* else No action required */
        /* End of blSyncStartReq is PWM_FALSE */
      } /* else No action required */
      /* End of if ((uint32)PWM_MAX_CHANNEL_ID_CONFIGURED != LulCount) */

      #if (PWM_NOTIFICATION_SUPPORTED == STD_ON)
      /* Set the Notification status as PWM_FALSE */
      Pwm_GaaNotifStatus[LulCount] = PWM_FALSE;
      #endif

      /* Increment the pointer to the next ATUE channel */
      LpChannelConfig++;                                                                                                /* PRQA S 2824, 2934 # JV-01, JV-01 */
      /* Set ATU channel config data pointer */
      LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                     /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
      LpChannelConfigNext++;                                                                                            /* PRQA S 2824, 2934 # JV-01, JV-01 */
    } /* End of  if ((uint8)PWM_HWIP_ATU == LucTimerUnitType) */
  } /* End of Timer-E channels 'for' loop */
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
/***********************************************************************************************************************
** Function Name         : Pwm_HW_ATU_InitUnit
**
** Service ID            : NA
**
** Description           : This is PWM Driver Component support function. This function sets the clock pre-scaler,
**                         PWM mode, Period, Duty cycle and polarity for all configured channels.
**                         This function also disables the interrupts (Notifications) and resets the interrupt request
**                         pending flags.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Pre-conditions        : NA
**
** Global Variables used : Pwm_GpATUUnitConfig, Pwm_GaaFpsSupport_Flag, Pwm_GaaHWIP_Used
**
** Functions invoked     : Pwm_HW_InitUnitRegister.
**
** Registers Used        : ATUENR, TSTRE, SSTREx, PSCREx, PSCCRExy, TCREx, TSCREx, TIEREx, TOCREx, CYLRExy, DTRExy,
**                         CRLDExy, DRLDExy, RLDCREx, PSCRx, EIC176 to EIC211,IMR5,IMR6,IMR17 EIC548 to 551
**
** Reference ID          : PWM_DUD_ACT_060, PWM_DUD_ACT_060_REG001, PWM_DUD_ACT_060_REG002, PWM_DUD_ACT_060_REG003,
** Reference ID          : PWM_DUD_ACT_060_REG004, PWM_DUD_ACT_060_REG005, PWM_DUD_ACT_060_REG006,
** Reference ID          : PWM_DUD_ACT_060_REG007, PWM_DUD_ACT_060_REG008, PWM_DUD_ACT_060_REG009,
** Reference ID          : PWM_DUD_ACT_060_GBL001, PWM_DUD_ACT_060_GBL002
***********************************************************************************************************************/
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_ATU_InitUnit(P2CONST (Pwm_ConfigType, AUTOMATIC, PWM_CONFIG_DATA) LddConfigPtr)     /* PRQA S 1532 # JV-01 */
{
  #if (PWM_CLOCK_SETTING == STD_ON)
  /* Array of structures for Common Prescaler Configuration */
  P2CONST(Pwm_ATUConfigType, PWM_VAR_NO_INIT, PWM_CONFIG_DATA) volatile Pwm_LpATUConfiguration;
  /* Local Pointer used for each Common Prescaler registers */
  P2VAR(Pwm_ATUClockConfigType, AUTOMATIC, REGSPACE) LpPrescalerSetting;                                                /* PRQA S 3432, 3678 # JV-01, JV-01 */
  #endif

  /* Index of HW IP Type */
  VAR(volatile uint8, AUTOMATIC) LucHWIPType;
  /* Pointer to Timer-E sub block configuration */
  P2CONST(Pwm_ATUUnitConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpSublkConfig;
  /* Local Pointer used for each Timer-E Sub block control registers */
  P2VAR(Pwm_ATUESubblockRegs, AUTOMATIC, REGSPACE) LpATUESublkRegs;                                                     /* PRQA S 3432 # JV-01 */
  /* Local variable for count */
  VAR(uint32, AUTOMATIC) LulCount;

  #if (PWM_CLOCK_SETTING == STD_ON)
  /* Save the start of CommonPrescaler Configuration in the global pointer */
  Pwm_LpATUConfiguration = (P2CONST(Pwm_ATUConfigType, PWM_VAR_NO_INIT, PWM_CONFIG_DATA))LddConfigPtr->pHWIPConfig;     /* PRQA S 0316, 2814 # JV-01, JV-01 */
  Pwm_GpATUClockConfig =                                                                                                /* PRQA S 2916 # JV-01 */
    (P2CONST(Pwm_ATUClockConfigType, PWM_VAR_NO_INIT, PWM_CONFIG_DATA))(Pwm_LpATUConfiguration->pClockConfig);          /* PRQA S 0316 # JV-01 */
  #endif

  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_HW_IP_CONFIG; LulCount++)                              /* PRQA S 2877 # JV-01 */
  {
    LucHWIPType = Pwm_GaaHWIP_Used[LulCount].ucIndex;
    if (PWM_HWIP_ATU == LucHWIPType)                                                                                    /* PRQA S 3416 # JV-01 */
    {
      /* Save the start of ATU-V Timer E Sub-block Unit Configuration in the global pointer */
      Pwm_GpATUUnitConfig = (P2CONST(Pwm_ATUUnitConfigType, PWM_VAR_NO_INIT, PWM_CONFIG_DATA))                          /* PRQA S 0316 # JV-01 */
        (LddConfigPtr->aaTimerUnitConfig[LucHWIPType]);
    } /* else No action required */
  }

  /* Update the configure pointer to point to the current Timer sub block*/
  LpSublkConfig = Pwm_GpATUUnitConfig;
  /* Initialize Timer E register to clear previous setting */
  Pwm_HW_InitUnitRegister();
  #if (PWM_CLOCK_SETTING == STD_ON)
  /* Get the Prescaler configuration setting. */
  LpPrescalerSetting = (Pwm_ATUClockConfigType *)Pwm_GpATUClockConfig;                                                  /* PRQA S 0311 # JV-01 */

  /* Confirm the total prescaler configured */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_COMMON_PRESCALER_CONFIGURED; LulCount++)
  {
    /* Store the Prescaler divider value to register*/
    *(volatile uint16 *)(LpPrescalerSetting->pPSCRx_Address) = LpPrescalerSetting->usAtuCommonPrescalerDivRatio;        /* PRQA S 2814, 2844 # JV-02, JV-01 */
    LpPrescalerSetting++;                                                                                               /* PRQA S 2824, 2934 # JV-01, JV-01 */
  }

  /* Enable Common Prescaler */
  *(LpSublkConfig->pATUENR_Address) = (uint8)((*(LpSublkConfig->pATUENR_Address) |                                      /* PRQA S 2814, 2844 # JV-03, JV-01 */
                                       ((uint8)LpSublkConfig->usTimerEnableMask & PWM_ENR_COMMON_PRESCALER_EN_MASK)));
  #endif

  /* Enable Timer Unit E */
  *(LpSublkConfig->pATUENR_Address) =
    (uint8)((*(LpSublkConfig->pATUENR_Address) | ((uint8)LpSublkConfig->usTimerEnableMask & PWM_ENR_TIMERE_EN_MASK)));

  /* Loop to Initialize Timer-E Sub block related attributes */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_TIMER_E_SUBBLOCK_CONFIG; LulCount++)                   /* PRQA S 2877 # JV-01 */
  {
    /* Get the pointer to the PWM sub block control registers */
    LpATUESublkRegs = (P2VAR(Pwm_ATUESubblockRegs, AUTOMATIC, REGSPACE))LpSublkConfig->pATUE_SubblockRegs;              /* PRQA S 2814, 2844, 3432 # JV-03, JV-01, JV-01 */
    /* Select the clock source from clock-bus lines 0 to 5 */
    LpATUESublkRegs->ucATUETCREx = (uint8)LpSublkConfig->enSelSubblockClock;                                            /* PRQA S 2814 # JV-03 */
    /* Load the configured Pre-scalar clock source Selection  */
    LpATUESublkRegs->ucATUETCREx = LpATUESublkRegs->ucATUETCREx | LpSublkConfig->ucSelPrescalerMask;
    /* Enable the Configured Sub block of Timer-E */
    *(LpSublkConfig->pTSTRE_Address) =                                                                                  /* PRQA S 2814 # JV-01 */
                                      (uint16)(*(LpSublkConfig->pTSTRE_Address) | LpSublkConfig->usEnableSubblockMask);

    /* Condition to check the block for channel or sub block pre-scalar */
    if (PWM_CH_SB_PRESCL_SEL != LpSublkConfig->ucSelPrescalerMask)
    {
      /* Load the configured pre-scaler value */
      LpATUESublkRegs->ucATUEPSCREx = LpSublkConfig->ucSelSubblockPrescaler;
    } /* else No action required */

    #if (PWM_SYNC_START_SUPPORT == STD_ON)
    /* Bitwise operation from & operator */
    /* Do not enable(start) the timer channel configured as sync */
    LpATUESublkRegs->ucATUESSTREx =
      (LpSublkConfig->ucSubblockChannelMask & (uint8)(~LpSublkConfig->ucSyncChannelMask));
    #else
    /* Enable(start) the timer channel which are not configured as sync */
    LpATUESublkRegs->ucATUESSTREx = LpATUESublkRegs->ucATUESSTREx | LpSublkConfig->ucSubblockChannelMask;
    #endif

    /* Increment to next sub block configuration */
    LpSublkConfig++;                                                                                                    /* PRQA S 2824, 2934, 2983 # JV-01, JV-01, JV-01 */
  } /* End of PWM_TOTAL_TIMER_E_SUBBLOCK_CONFIG 'for' loop */
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
/***********************************************************************************************************************
** Function Name         : Pwm_HW_DeInitUnit
**
** Service ID            : NA
**
** Description           : This is PWM Driver Component support function. This function de-initializes all the PWM
**                         channels by setting to their configured Idle state, disabling the notifications, resetting
**                         all the registers and stopping the PWM mode of operation of corresponding timer.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Pre-conditions        : NA
**
** Global Variables used : Pwm_GpATUUnitConfig
**
** Function(s) invoked   : None
**
** Registers Used        : IMR5,6,17 TIERx, EIC176 to EIC211, TSCREx, PSCREx,EIC548 to 551,RLDCREx,TCREx,PSCCRExy
**
** Reference ID          : PWM_DUD_ACT_062, PWM_DUD_ACT_062_REG001, PWM_DUD_ACT_062_REG002,
***********************************************************************************************************************/
#if (PWM_DE_INIT_API == STD_ON)
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_DeInitUnit(void)                                                                    /* PRQA S 1532 # JV-01 */
{
  /* Pointer to Timer-E sub block configuration */
  P2CONST(Pwm_ATUUnitConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpSublkConfig;
  /* Local pointer to Timer-E sub block control registers */
  P2VAR(Pwm_ATUESubblockRegs, AUTOMATIC, REGSPACE) LpATUESublkRegs;                                                     /* PRQA S 3432 # JV-01 */
  VAR(uint32, AUTOMATIC) LulCount;

  /* Update the con-fig pointer to point to the current Timer sub block*/
  LpSublkConfig = Pwm_GpATUUnitConfig;

  /* Loop to de-initialize Timer-E Sub block related attributes */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_TIMER_E_SUBBLOCK_CONFIG; LulCount++)                   /* PRQA S 2877 # JV-01 */
  {
    /* Get the pointer to the PWM sub block control registers */
    LpATUESublkRegs = (P2VAR(Pwm_ATUESubblockRegs, AUTOMATIC, REGSPACE))LpSublkConfig->pATUE_SubblockRegs;              /* PRQA S 2814, 2844, 3432 # JV-03, JV-01, JV-01 */
    /* Load the default Pre-scalar selection value and clock source Selection */
    LpATUESublkRegs->ucATUETCREx = (uint8)PWM_SUBBLOCK_CLOCK_SEL_RESET_VALUE;                                           /* PRQA S 2814 # JV-03 */
    /* Load the default Pre-scalar value for sub block */
    LpATUESublkRegs->ucATUEPSCREx = (uint8)PWM_SUBBLOCK_PRESCALER_RESET_VALUE;
    /* Increment to next sub block configuration */
    LpSublkConfig++;                                                                                                    /* PRQA S 2824, 2934, 2983 # JV-01, JV-01, JV-01 */
  } /* End of PWM_TOTAL_TIMER_E_SUBBLOCK_CONFIG 'for' loop */
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (PWM_DE_INIT_API == STD_ON) */
/***********************************************************************************************************************
** Function Name         : Pwm_HW_DeInitChannel
**
** Service ID            : NA
**
** Description           : This is PWM Driver Component support function. This function de-initializes all the PWM
**                         channels bysetting to their configured Idle state, disabling the notifications,
**                         resetting all the registers and stopping the PWM mode of operation of corresponding timer.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Pre-conditions        : NA
**
** Global Variables used : Pwm_GpChannelConfig, Pwm_GaaNotifStatus
**
** Function(s) invoked   : Pwm_HW_SetDutyToIdleState
**
** Registers Used        : IMR5,6,17 TIERx, EIC176 to EIC211, TSCREx, PSCREx,EIC548 to 551,RLDCREx,TCREx,PSCCRExy
**
** Reference ID          : PWM_DUD_ACT_063, PWM_DUD_ACT_063_GBL001, PWM_DUD_ACT_063_REG001, PWM_DUD_ACT_063_REG002,
** Reference ID          : PWM_DUD_ACT_063_REG003, PWM_DUD_ACT_063_REG004, PWM_DUD_ACT_063_REG005,
** Reference ID          : PWM_DUD_ACT_063_REG006, PWM_DUD_ACT_063_REG007, PWM_DUD_ACT_063_REG008
***********************************************************************************************************************/
#if (PWM_DE_INIT_API == STD_ON)
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_DeInitChannel(void)                                                                 /* PRQA S 1532 # JV-01 */
{
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfig;
  /* ATU config data */
  P2CONST(Pwm_ATUChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpATUChannelConfigData;
  /* Pointer used for each Timer-E channel control registers */
  P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE) LpATUEChCntrlRegs;                                                    /* PRQA S 3432 # JV-01 */
  /* Local Pointer used for each Timer-E channel data registers */
  P2VAR(Pwm_ATUEChDataRegs, AUTOMATIC, REGSPACE) LpATUEChDataRegs;                                                      /* PRQA S 3432 # JV-01 */
  /* Pointer used for each Timer-E channel control registers */
  P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelProp;
  VAR(uint32, AUTOMATIC) LulCount;
  VAR(volatile uint8, AUTOMATIC) LucTimerUnitType;

  /* Update the con-fig pointer to point to the current Timer channel*/
  LpChannelConfig = Pwm_GpChannelConfig;
  /* Set ATU channel config data pointer */
  LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                         /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */

  /* Loop to de-initialize the attributes of Timer-E channels */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_CHANNELS_CONFIG; LulCount++)
  {
    /* Get the Timer Unit Type*/
    LucTimerUnitType = LpChannelConfig->ucTimerUnitType;                                                                /* PRQA S 2814, 2844 # JV-02, JV-01 */
    if ((uint8)PWM_HWIP_ATU == LucTimerUnitType)                                                                        /* PRQA S 3416 # JV-01 */
    {
      /* Get the pointer to the PWM Channel data registers */
      LpATUEChDataRegs = (P2VAR(Pwm_ATUEChDataRegs, AUTOMATIC, REGSPACE))LpATUChannelConfigData->pATUEChDataRegs;       /* PRQA S 2814, 3432 # JV-03, JV-01 */
      /* Get the pointer to the PWM Channel properties */
      LpChannelProp = (P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST))LpChannelConfig->pChannelProp;      /* PRQA S 0316 # JV-01 */
      /* Get the pointer to the PWM Channel control registers */
      LpATUEChCntrlRegs = (P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE))LpATUChannelConfigData->pATUEChCntrlRegs;    /* PRQA S 3432 # JV-01 */
      /* Disable the Interrupt processing of the current channel */
      RH850_SV_SET_ICR_SYNCP(32, (LpATUChannelConfigData->pIMRx_Address), ~(LpATUChannelConfigData->ulImrMaskValue));   /* PRQA S 2814 # JV-01 */
      /* Disable interrupts */
      LpATUEChCntrlRegs->usATUETIEREx &= (uint16)(~(LpChannelProp->usChannelIntSelMask));                               /* PRQA S 2814 # JV-02 */

      #if (PWM_CLEAR_PENDING_INTERRUPT == STD_ON)
      /* Check if the Interrupt register is present for particular channel */
      if (NULL_PTR != LpATUChannelConfigData->pEICx_Address)
      {
        /* Clear the pending ISR request */
        RH850_SV_MODE_ICR_AND(16, (LpATUChannelConfigData->pEICx_Address), (uint16)PWM_CLEAR_PENDING_INTR_MASK);        /* PRQA S 0499, 2814, 4399 # JV-01, JV-01, JV-01 */
        /* DummyRead & SYNCP */
        RH850_SV_MODE_REG_READ_ONLY(16, LpATUChannelConfigData->pEICx_Address);
        EXECUTE_SYNCP();
        /* Clear the pending ISR request */
        LpATUEChCntrlRegs->usATUETSCREx = LpChannelProp->usChannelIntSelMask;
      } /* else No action required */
        /* End of NULL_PTR != LpChannel->pEICx_Address */
      #endif /* PWM_CLEAR_PENDING_INTERRUPT */

      #if (PWM_NOTIFICATION_SUPPORTED == STD_ON)
      /* Set the Notification status as PWM_FALSE */
      Pwm_GaaNotifStatus[LulCount] = PWM_FALSE;
      #endif /* PWM_NOTIFICATION_SUPPORTED */

      /* Reload Control Disable (RLDCREx) */
      LpATUEChCntrlRegs->ucATUERLDCREx &= (uint8)(~(LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK));
      /* Set Duty to IDLE state */
      Pwm_HW_SetDutyToIdleState(LpChannelConfig, LpATUEChDataRegs);
      /* Enable Reload function on cycle match */
      LpATUEChCntrlRegs->ucATUERLDCREx |= (uint8)((LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK));
      /* Default pre-scalar value to register */
      *(volatile uint8*)(LpATUChannelConfigData->pPSCCRExy_Address) = (uint8)PWM_CH_PRESCALER_RESET_VALUE;
    }  /* End of if ((uint8)PWM_HWIP_ATU == LucTimerUnitType) */
    /* Increment the pointer to the next ATUE channel */
    LpChannelConfig++;                                                                                                  /* PRQA S 2824, 2934 # JV-01, JV-01 */
    /* Set ATU channel config data pointer */
    LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                       /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
  } /* End of Timer-E channels 'for' loop */
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (PWM_DE_INIT_API == STD_ON) */
/***********************************************************************************************************************
** Function Name         : Pwm_HW_EnableNotification
**
** Service ID            : NA
**
** Description           : This routine is used to disable pending interrupts and enable Interrupt processing.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : LddChannelId, LenNotification
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Pre-conditions        : NA
**
** Global Variables used : Pwm_GpChannelConfig, Pwm_GaaNotifStatus
**
** Functions invoked     : Det_ReportError
**
** Registers Used        : TSCREx, TIEREx
**
** Reference ID          : PWM_DUD_ACT_067, PWM_DUD_ACT_067_CRT001, PWM_DUD_ACT_067_CRT002, PWM_DUD_ACT_067_ERR001,
** Reference ID          : PWM_DUD_ACT_067_REG001, PWM_DUD_ACT_067_REG002, PWM_DUD_ACT_067_REG003,
** Reference ID          : PWM_DUD_ACT_067_REG004, PWM_DUD_ACT_067_REG005, PWM_DUD_ACT_067_REG006,
** Reference ID          : PWM_DUD_ACT_067_REG007, PWM_DUD_ACT_067_REG008, PWM_DUD_ACT_067_REG009,
** Reference ID          : PWM_DUD_ACT_067_REG010, PWM_DUD_ACT_067_REG011, PWM_DUD_ACT_067_REG012,
** Reference ID          : PWM_DUD_ACT_067_GBL001
***********************************************************************************************************************/
#if (PWM_NOTIFICATION_SUPPORTED == STD_ON)
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_EnableNotification(                                                                 /* PRQA S 1532 # JV-01 */
  Pwm_ChannelType LddChannelId, Pwm_EdgeNotificationType LenNotification)
{
  /* Pointer pointing to the channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfig;
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ATUChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpATUChannelConfigData;
  /* Pointer used for each Timer-E channel control registers */
  P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelProp;
  /* Pointer used for each Timer-E channel control registers */
  P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE) LpATUEChCntrlRegs;                                                    /* PRQA S 3432 # JV-01 */
  #if (PWM_DEV_ERROR_DETECT == STD_ON)
  VAR(uint8, AUTOMATIC) LucDetErrFlag;
  #endif

  /* Get the pointer to the channel configuration */
  LpChannelConfig = &Pwm_GpChannelConfig[LddChannelId];
  /* Set ATU channel config data pointer */
  LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                         /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
  /* Get the pointer to the PWM Channel control registers */
  LpATUEChCntrlRegs = (P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE))LpATUChannelConfigData->pATUEChCntrlRegs;        /* PRQA S 2814, 3432 # JV-03, JV-01 */
  /* Get the pointer to the PWM Channel properties */
  LpChannelProp = (P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST))LpChannelConfig->pChannelProp;          /* PRQA S 0316 # JV-01 */
  #if (PWM_DEV_ERROR_DETECT == STD_ON)
  LucDetErrFlag = E_OK;

  if (NULL_PTR == LpChannelConfig->pPwmEdgeNotifPtr)
  {
    /* Report to DET module */
    (void)Det_ReportError(PWM_MODULE_ID, PWM_INSTANCE_ID, PWM_ENABLENOTIFICATION_SID, PWM_E_NOTIFY_NOTCONFIG);
    LucDetErrFlag = E_NOT_OK;
  } /* else No action required */

  if (E_OK == LucDetErrFlag)
  #endif
  {
    #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
    /* Enter Protected area */
    PWM_ENTER_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
    #endif

    /* Check whether any notification is configured for this channel */
    if (NULL_PTR != LpChannelConfig->pPwmEdgeNotifPtr)
    {
      /* Set the Notification enable status as PWM_TRUE for this channel */
      Pwm_GaaNotifStatus[LddChannelId] = PWM_TRUE;                                                                      /* PRQA S 2844 # JV-01 */
    } /* else No action required */

    #if (PWM_CLEAR_PENDING_INTERRUPT == STD_ON)
    /* Check if the Interrupt register is present for particular channel */
    if (NULL_PTR != LpATUChannelConfigData->pEICx_Address)
    {
      /* Clear the pending ISR request */
      RH850_SV_MODE_ICR_AND(16, LpATUChannelConfigData->pEICx_Address, PWM_CLEAR_PENDING_INTR_MASK);                    /* PRQA S 0499, 2814, 4399 # JV-01, JV-01, JV-01 */
      /* DummyRead & SYNCP */
      RH850_SV_MODE_REG_READ_ONLY(16, LpATUChannelConfigData->pEICx_Address);
      EXECUTE_SYNCP();
      /* Clear the pending ISR request*/
      LpATUEChCntrlRegs->usATUETSCREx = LpChannelProp->usChannelIntSelMask;                                             /* PRQA S 2814 # JV-02 */
    } /* else No action required */
    #endif /* PWM_CLEAR_PENDING_INTERRUPT == STD_ON */

    /* Condition to enable the interrupt for Rising or Both or falling edge in normal PWM signal OR */
    if ((((uint8)PWM_HIGH == LpChannelConfig->ucPolarity) && (PWM_RISING_EDGE == LenNotification))
      || (((uint8)PWM_LOW == LpChannelConfig->ucPolarity) && (PWM_FALLING_EDGE == LenNotification)))
    {
      /* Disable duty/cycle match interrupt */
      LpATUEChCntrlRegs->usATUETIEREx &= (uint16)(~(LpChannelProp->usChannelIntSelMask));
      /* Clear interrupt status  */
      LpATUEChCntrlRegs->usATUETSCREx = LpChannelProp->usChannelIntSelMask;
      /* Enable cycle match interrupt */
      LpATUEChCntrlRegs->usATUETIEREx |= (LpChannelProp->usChannelIntSelMask & PWM_CYCLE_MATCH_MASK);
    }
    /* Condition to enable the interrupt for Falling or Both or falling edge
       in normal PWM signal */
    else if ((((uint8)PWM_HIGH == LpChannelConfig->ucPolarity) && (PWM_FALLING_EDGE == LenNotification))
           || (((uint8)PWM_LOW == LpChannelConfig->ucPolarity) && (PWM_RISING_EDGE == LenNotification)))
    {
      /* Disable duty/cycle match interrupt */
      LpATUEChCntrlRegs->usATUETIEREx &= (uint16)(~(LpChannelProp->usChannelIntSelMask));
      /* Clear interrupt status  */
      LpATUEChCntrlRegs->usATUETSCREx = LpChannelProp->usChannelIntSelMask;
      /* Enable duty match interrupt */
      LpATUEChCntrlRegs->usATUETIEREx |= (LpChannelProp->usChannelIntSelMask & PWM_DUTY_MATCH_MASK);
    }
    else if (PWM_BOTH_EDGES == LenNotification)                                                                         /* PRQA S 2004 # JV-01 */
    {
      /* Disable interrupt */
      LpATUEChCntrlRegs->usATUETIEREx &= (uint16)(~(LpChannelProp->usChannelIntSelMask));
      /* Clear interrupt status  */
      LpATUEChCntrlRegs->usATUETSCREx = LpChannelProp->usChannelIntSelMask;
      /* Enable duty/cycle match interrupt */
      LpATUEChCntrlRegs->usATUETIEREx |= LpChannelProp->usChannelIntSelMask;
    } /* else No action required */

    #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
    /* Exit Protected area */
    PWM_EXIT_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
    #endif

  } /* else No action required */
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (PWM_NOTIFICATION_SUPPORTED == STD_ON) */

/***********************************************************************************************************************
** Function Name         : Pwm_HW_DisableNotification
**
** Service ID            : NA
**
** Description           : This routine is used to disable Interrupt processing.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : LddChannelId
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Pre-conditions        : NA
**
** Global Variables used : Pwm_GpChannelConfig
**
** Functions invoked     : None
**
** Registers Used        : TIEREx
**
** Reference ID          : PWM_DUD_ACT_068, PWM_DUD_ACT_068_CRT001, PWM_DUD_ACT_068_CRT002, PWM_DUD_ACT_068_REG001
***********************************************************************************************************************/
#if (PWM_NOTIFICATION_SUPPORTED == STD_ON)
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_DisableNotification(Pwm_ChannelType LddChannelId)                                   /* PRQA S 1532 # JV-01 */
{
  /* Pointer pointing to the channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfig;
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ATUChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpATUChannelConfigData;
  /* Pointer used for each Timer-E channel control registers */
  P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelProp;
  /* Pointer used for each Timer-E channel control registers */
  P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE) LpATUEChCntrlRegs;                                                    /* PRQA S 3432 # JV-01 */

  /* Get the pointer to the channel configuration */
  LpChannelConfig = &Pwm_GpChannelConfig[LddChannelId];
  /* Set ATU channel config data pointer */
  LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                         /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
  /* Get the pointer to the PWM Channel control registers */
  LpATUEChCntrlRegs = (P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE))LpATUChannelConfigData->pATUEChCntrlRegs;        /* PRQA S 2814, 3432 # JV-03, JV-01 */
  /* Get the pointer to the PWM Channel properties */
  LpChannelProp = (P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST))LpChannelConfig->pChannelProp;          /* PRQA S 0316 # JV-01 */

  #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Enter Protected area */
  PWM_ENTER_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
  #endif

  /* Bitwise operation from & operator */
  /* Disable Duty Match and Cycle match interrupts */
  LpATUEChCntrlRegs->usATUETIEREx &= (uint16)(~LpChannelProp->usChannelIntSelMask);                                     /* PRQA S 2814 # JV-03 */

  #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Exit Protected area */
  PWM_EXIT_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
  #endif
}
#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (PWM_NOTIFICATION_SUPPORTED == STD_ON) */

/***********************************************************************************************************************
** Function Name         : Pwm_HW_CalculateDuty
**
** Service ID            : NA
**
** Description           : This is PWM Driver Component support function.
**                         This function Calculates Absolute duty for the PWM channel.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : LddAbsolutePeriod, LddRelativeDuty
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : Pwm_PeriodType
**
** Pre-conditions        : NA
**
** Global Variables used : None
**
** Functions invoked     : None
**
** Registers Used        : None
**
** Reference ID          : PWM_DUD_ACT_070
***********************************************************************************************************************/
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

STATIC FUNC(Pwm_PeriodType, PWM_PRIVATE_CODE)
  Pwm_HW_CalculateDuty(Pwm_PeriodType LddAbsolutePeriod, Pwm_PeriodType LddRelativeDuty)
{
  VAR(Pwm_PeriodType, AUTOMATIC) LddAbsoluteDuty;
  VAR(Pwm_PeriodType, AUTOMATIC) LddCorrectionPeriod;

  if ((Pwm_PeriodType)PWM_MAX_DUTY_CYCLE_VALUE == LddRelativeDuty)
  {
    /* If Duty is 100%, Update value with the Cycle-Setting Register */
    LddAbsoluteDuty = LddAbsolutePeriod;
  }
  else if ((Pwm_PeriodType)PWM_MIN_DUTY_CYCLE_VALUE == LddRelativeDuty)
  {
    /* If Duty is 0%, Update 0x0000 value in the CDRm register */
    LddAbsoluteDuty = (Pwm_PeriodType)PWM_MIN_DUTY_CYCLE_VALUE;
  }
  else
  {
    if ((Pwm_PeriodType)PWM_PERIOD_LIMIT < LddAbsolutePeriod)
    {
      /* If Period > 0xFFFF and Duty is between 0x0000 and 0x8000 */
      LddCorrectionPeriod = LddAbsolutePeriod & (Pwm_PeriodType)PWM_CORRECTION_MASK;

      /* Duty Cycle scaling */
      LddAbsoluteDuty = (Pwm_PeriodType)(((LddAbsolutePeriod >> (Pwm_PeriodType)PWM_DUTY_CALC_DIV) * LddRelativeDuty)   /* PRQA S 3384 # JV-01 */
                         + ((LddCorrectionPeriod * LddRelativeDuty) >> (Pwm_PeriodType)PWM_DUTY_CALC_DIV));             /* PRQA S 3383, 3384 # JV-01, JV-01 */
    }
    else
    {
      /* If Duty is between 0x0000 and 0x8000 */
      /* AbsoluteDutyCycle = ((uint32)AbsolutePeriodTime * RelativeDutyCycle) >> 15 */
      LddAbsoluteDuty = ((LddAbsolutePeriod * LddRelativeDuty) >> (Pwm_PeriodType)PWM_DUTY_CALC_DIV);                   /* PRQA S 3383 # JV-01 */
    }
  }

  /* Condition check for shifting the duty value to be loaded to duty register */
  if ((Pwm_PeriodType)PWM_CALC_DUTY_MAX_LIMIT >= LddAbsoluteDuty)
  {
    /* Value to be loaded into register is (31-8) bits */
    LddAbsoluteDuty = (LddAbsoluteDuty << (Pwm_PeriodType)PWM_BYTE_SHIFT_VALUE);
  } /* else No action required */

  return LddAbsoluteDuty;
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name         : Pwm_HW_SetOutputToIdle
**
** Service ID            : NA
**
** Description           : This is PWM Driver Component support function.
**                         This function sets the output of a required channel to its configured Idle state.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : LddChannelId
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Pre-conditions        : NA
**
** Global Variables used : Pwm_GpChannelConfig, Pwm_GpATUUnitConfig
**
** Functions invoked     : Pwm_HW_SetDutyToIdleState
**
** Registers Used        : RLDCREx,FCTRGEx
**
** Reference ID          : PWM_DUD_ACT_064, PWM_DUD_ACT_064_CRT001, PWM_DUD_ACT_064_CRT002
** Reference ID          : PWM_DUD_ACT_064_REG001, PWM_DUD_ACT_064_REG002, PWM_DUD_ACT_064_REG003
***********************************************************************************************************************/
#if (PWM_SET_OUTPUT_TO_IDLE_API == STD_ON)
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_SetOutputToIdle(Pwm_ChannelType LddChannelId)                                       /* PRQA S 1532 # JV-01 */
{
  /* Pointer pointing to the channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfig;
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ATUChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpATUChannelConfigData;
  /* Pointer to Timer-E sub block configuration */
  P2CONST(Pwm_ATUUnitConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpSublkConfig;
  /* Pointer used for each Timer-E channel control registers */
  P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelProp;
  /* Pointer used for each Timer-E channel control registers */
  P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE) LpATUEChCntrlRegs;                                                    /* PRQA S 3432 # JV-01 */
  /* Local Pointer used for each Timer-E channel data registers */
  P2VAR(Pwm_ATUEChDataRegs, AUTOMATIC, REGSPACE) LpATUEChDataRegs;                                                      /* PRQA S 3432 # JV-01 */
  /* Local pointer to Timer-E sub block control registers */
  P2VAR(Pwm_ATUESubblockRegs, AUTOMATIC, REGSPACE) LpATUESublkRegs;                                                     /* PRQA S 3432 # JV-01 */
  VAR(uint32, AUTOMATIC) LulCount;

  /* Get the pointer to the channel configuration */
  LpChannelConfig = &Pwm_GpChannelConfig[LddChannelId];
  /* Set ATU channel config data pointer */
  LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                         /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
  /* Update the con-fig pointer to point to the current Timer sub block*/
  LpSublkConfig = Pwm_GpATUUnitConfig;
  /* Get the pointer to the PWM Channel properties */
  LpChannelProp = (P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST))LpChannelConfig->pChannelProp;          /* PRQA S 0316 # JV-01 */
  /* Get the pointer to the PWM Channel control registers */
  LpATUEChCntrlRegs = LpATUChannelConfigData->pATUEChCntrlRegs;                                                         /* PRQA S 2814 # JV-03 */
  /* Get the pointer to the PWM Channel data registers */
  LpATUEChDataRegs = LpATUChannelConfigData->pATUEChDataRegs;

  /* Loop to Initialize Timer-E Sub block related attributes */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_TIMER_E_SUBBLOCK_CONFIG; LulCount++)                   /* PRQA S 2877 # JV-01 */
  {
    if ((uint16)LpATUChannelConfigData->enSubblockNum == LpSublkConfig->usEnableSubblockMask)                           /* PRQA S 2814, 2844 # JV-03, JV-01 */
    {
      /* Get pointer to the respective sub blocks register structure to local pointer */
      LpATUESublkRegs = (P2VAR(Pwm_ATUESubblockRegs, AUTOMATIC, REGSPACE))LpSublkConfig->pATUE_SubblockRegs;            /* PRQA S 3432 # JV-01 */
    } /* else No action required */

    /* Increment to next sub block configuration */
    LpSublkConfig++;                                                                                                    /* PRQA S 2824, 2934, 2983 # JV-01, JV-01, JV-01 */
  }

  #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Enter Protected area */
  PWM_ENTER_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
  #endif

  /* Stop the timer */
  LpATUESublkRegs->ucATUESSTREx &= (uint8)(~(LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK));                 /* PRQA S 2814, 2962, 2963 # JV-02, JV-01, JV-01 */
  /* Set Duty to Idle state */
  Pwm_HW_SetDutyToIdleState(LpChannelConfig, LpATUEChDataRegs);
  /* forced compare match */
  LpATUEChCntrlRegs->ucATUEFCTRGEx |= (uint8)((LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK));               /* PRQA S 2814 # JV-03 */
  /* Restart the timer */
  LpATUESublkRegs->ucATUESSTREx |= (uint8)((LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK));

  #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Exit Protected area */
  PWM_EXIT_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
  #endif
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

#endif /* End of (PWM_SET_OUTPUT_TO_IDLE_API == STD_ON) */

/***********************************************************************************************************************
** Function Name        : Pwm_HW_SetDutyCycle
**
** Service ID           : NA
**
** Description          : This is PWM Driver Component support function.
**                        This function updates the duty cycle counter value in the hardware registers.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Reentrant
**
** Input Parameters     : LddChannelId and LusDutyCycle
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Pre-conditions       : NA
**
** Global Variable used : Pwm_GpChannelConfig
**
** Functions invoked    : Pwm_HW_CalculateDuty
**
** Registers Used       : DRLDExy, RLDCREx, CYLRExy
**
** Reference ID         : PWM_DUD_ACT_065, PWM_DUD_ACT_065_CRT002, PWM_DUD_ACT_065_REG001,
** Reference ID         : PWM_DUD_ACT_065_REG002, PWM_DUD_ACT_065_REG003, PWM_DUD_ACT_065_REG004
***********************************************************************************************************************/
#if (PWM_SET_DUTY_CYCLE_API == STD_ON)
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_SetDutyCycle(Pwm_ChannelType LddChannelId, uint16 LusDutyCycle)                     /* PRQA S 1532 # JV-01 */
{
  /* Pointer pointing to the channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfig;
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ATUChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpATUChannelConfigData;
  /* Pointer used for each Timer-E channel control registers */
  P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelProp;
  /* Local Pointer used for each Timer-E channel data registers */
  P2VAR(Pwm_ATUEChDataRegs, AUTOMATIC, REGSPACE) LpATUEChDataRegs;                                                      /* PRQA S 3432 # JV-01 */
  /* Update the pointer for base address of the  Channel control register */
  P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE) LpATUEChCntrlRegs;                                                    /* PRQA S 3432 # JV-01 */
  VAR(Pwm_PeriodType, AUTOMATIC) LddDefaultPeriod;

  /* Get the pointer to the channel configuration */
  LpChannelConfig = &Pwm_GpChannelConfig[LddChannelId];
  /* Set ATU channel config data pointer */
  LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                         /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
  /* Update the pointer to point to the current Timer channel properties */
  LpChannelProp = (P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST))LpChannelConfig->pChannelProp;          /* PRQA S 0316 # JV-01 */
  /* Get the pointer to the PWM Channel data registers */
  LpATUEChDataRegs = LpATUChannelConfigData->pATUEChDataRegs;                                                           /* PRQA S 2814 # JV-03 */
  /* Get the pointer to the PWM Channel control registers */
  LpATUEChCntrlRegs = LpATUChannelConfigData->pATUEChCntrlRegs;

  #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Enter Protected area */
  PWM_ENTER_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
  #endif

  /* Disable Reload function for cycle match */
  LpATUEChCntrlRegs->ucATUERLDCREx &= (uint8)(~(LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK));              /* PRQA S 2814 # JV-03 */
  /* Assign the configured value of the channel period in to register */
  LddDefaultPeriod = LpATUEChDataRegs->ulATUECYLRExy >> (Pwm_PeriodType)PWM_BYTE_SHIFT_VALUE;                           /* PRQA S 2814 # JV-03 */
  /* Load the duty value to reload duty register */
  LpATUEChDataRegs->ulATUEDRLDExy = (uint32)Pwm_HW_CalculateDuty(LddDefaultPeriod, (Pwm_PeriodType)LusDutyCycle);
  /* Enable Reload function for cycle match */
  LpATUEChCntrlRegs->ucATUERLDCREx |= (LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK);

  #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Exit Protected area */
  PWM_EXIT_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
  #endif
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (PWM_SET_DUTY_CYCLE_API == STD_ON) */

/***********************************************************************************************************************
** Function Name        : Pwm_HW_SetPeriodAndDuty
**
** Service ID           : NA
**
** Description          : This is PWM Driver Component support function.
**                        This function updates the Period and Duty cycle counter value in the hardware registers.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Reentrant
**
** Input Parameters     : LddChannelId, LddSetPeriod and LusSetDutyCycle
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Pre-conditions       : NA
**
** Global Variable used : Pwm_GpChannelConfig
**
** Function invoked     : Det_ReportError, Pwm_HW_CalculateDuty
**
** Registers Used       : CRLDExy, DRLDExy, RLDCREx
**
** Reference ID         : PWM_DUD_ACT_066, PWM_DUD_ACT_022_CRT001, PWM_DUD_ACT_066_CRT002,
** Reference ID         : PWM_DUD_ACT_066_REG001, PWM_DUD_ACT_066_REG002, PWM_DUD_ACT_066_REG003,
** Reference ID         : PWM_DUD_ACT_066_REG004, PWM_DUD_ACT_066_REG005, PWM_DUD_ACT_066_ERR001
***********************************************************************************************************************/
#if (PWM_SET_PERIOD_AND_DUTY_API == STD_ON)
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_SetPeriodAndDuty(                                                                   /* PRQA S 1532 # JV-01 */
  Pwm_ChannelType LddChannelId, Pwm_PeriodType LddSetPeriod, uint16 LusSetDutyCycle)
{
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfig;
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ATUChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpATUChannelConfigData;
  /* Pointer used for each Timer-E channel control registers */
  P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelProp;
  /* Local Pointer used for each Timer-E channel data registers */
  P2VAR(Pwm_ATUEChDataRegs, AUTOMATIC, REGSPACE) LpATUEChDataRegs;                                                      /* PRQA S 3432 # JV-01 */
  /* Update the pointer for base address of the  Channel control register */
  P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE) LpATUEChCntrlRegs;                                                    /* PRQA S 3432 # JV-01 */
  #if (PWM_DEV_ERROR_DETECT == STD_ON)
  /* Declare the variable to store the DET return value */
  VAR(uint8, AUTOMATIC) LucDetErrFlag;
  #endif

  /* Update the configure pointer to point to the current Timer channel*/
  LpChannelConfig = &Pwm_GpChannelConfig[LddChannelId];
  /* Set ATU channel config data pointer */
  LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                         /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
  #if (PWM_DEV_ERROR_DETECT == STD_ON)
  LucDetErrFlag = E_OK;

  /* Check for valid Duty range Check for valid Period */
  if (PWM_PERIOD_MAX_LIMIT < LddSetPeriod)
  {
    /* Report to DET module */
    (void)Det_ReportError(PWM_MODULE_ID, PWM_INSTANCE_ID, PWM_SET_PERIODANDDUTY_SID, PWM_E_PARAM_VALUE);
    LucDetErrFlag = E_NOT_OK;
  } /* else No action required */

  if (E_OK == LucDetErrFlag)
  #endif /* (PWM_DEV_ERROR_DETECT == STD_ON) */
  {
    /* Update the pointer to point to the current Timer channel properties */
    LpChannelProp = (P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST))LpChannelConfig->pChannelProp;        /* PRQA S 0316 # JV-01 */
    /* Get the pointer to the PWM Channel data registers */
    LpATUEChDataRegs = LpATUChannelConfigData->pATUEChDataRegs;                                                         /* PRQA S 2814 # JV-03 */
    /* Get the pointer to the PWM Channel control registers */
    LpATUEChCntrlRegs = LpATUChannelConfigData->pATUEChCntrlRegs;

    #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
    /* Enter Protected area */
    PWM_ENTER_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
    #endif

    /* Disable Reload function for cycle match */
    LpATUEChCntrlRegs->ucATUERLDCREx &= (uint8)(~(LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK));            /* PRQA S 2814 # JV-03 */

    if ((Pwm_PeriodType)PWM_ZERO != LddSetPeriod)
    {
      /* Load the duty value to reload duty register */
      LpATUEChDataRegs->ulATUEDRLDExy = Pwm_HW_CalculateDuty(LddSetPeriod, (Pwm_PeriodType)LusSetDutyCycle);            /* PRQA S 2814 # JV-03 */
      /* Load the cycle value to reload cycle register */
      LpATUEChDataRegs->ulATUECRLDExy = LddSetPeriod << (uint32)PWM_BYTE_SHIFT_VALUE;
    }
    else
    {
      /* If period is set to zero the setting of the duty-cycle is not relevant */
      /* In this case the output shall be zero (zero percent duty-cycle). */
      LpATUEChDataRegs->ulATUEDRLDExy = (uint32)PWM_DUTY_CYCLE_RESET_VALUE;
      /* Register CRLDExy can only be set from 0x01, set cycle value to one when period is zero */
      LpATUEChDataRegs->ulATUECRLDExy = (uint32)PWM_ONE << (uint32)PWM_BYTE_SHIFT_VALUE;
    }

    /* Enable Reload function for cycle match */
    LpATUEChCntrlRegs->ucATUERLDCREx |= (LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK);

    #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
    /* Exit Protected area */
    PWM_EXIT_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
    #endif
  } /* else No action required */
  /* End of if (E_OK == LucDetErrFlag) */
}
#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (PWM_SET_PERIOD_AND_DUTY_API == STD_ON) */

/***********************************************************************************************************************
** Function Name         : Pwm_HW_SynchronousInit
**
** Service ID            : NA
**
** Description           : This is PWM Driver Component support function. This function sets the Period,
**                         Duty cycle of all PWM channels configured to support synchronous start/stop of timers.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Pre-conditions        : NA
**
** Global Variables used : Pwm_GpChannelConfig
**
** Functions invoked     : None
**
** Registers Used        : TOCREx, PSCCRExy,RLDCREx
**
** Reference ID          : PWM_DUD_ACT_072, PWM_DUD_ACT_072_REG001, PWM_DUD_ACT_072_REG002, PWM_DUD_ACT_072_REG003
***********************************************************************************************************************/
#if (PWM_SYNC_START_SUPPORT == STD_ON)
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_SynchronousInit(void)                                                               /* PRQA S 1532 # JV-01 */
{
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfig;
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ATUChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpATUChannelConfigData;
  /* Pointer used for each Timer-E channel control registers */
  P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelProp;
  /* Update the pointer for base address of the  Channel control register */
  P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE) LpATUEChCntrlRegs;                                                    /* PRQA S 3432 # JV-01 */
  VAR(uint32, AUTOMATIC) LulCount;
  VAR(volatile uint8, AUTOMATIC) LucTimerUnitType;

  /* Update the configure pointer to point to the current Timer channel*/
  LpChannelConfig = Pwm_GpChannelConfig;
  /* Set ATU channel config data pointer */
  LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                         /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */

  /* Loop to set the attributes of Timer-E channels */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_CHANNELS_CONFIG; LulCount++)
  {
    /* Get the Timer Unit Type*/
    LucTimerUnitType = LpChannelConfig->ucTimerUnitType;                                                                /* PRQA S 2814, 2844 # JV-02, JV-01 */
    if ((uint8)PWM_HWIP_ATU == LucTimerUnitType)                                                                        /* PRQA S 3416 # JV-01 */
    {
      if ((boolean)PWM_TRUE == LpChannelConfig->blSyncStartReq)
      {
        /* Get the pointer to the PWM Channel properties */
        LpChannelProp = (P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST))LpChannelConfig->pChannelProp;    /* PRQA S 0316 # JV-01 */
        /* Get the pointer to the PWM Channel control registers */
        LpATUEChCntrlRegs = (P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE))LpATUChannelConfigData->pATUEChCntrlRegs;  /* PRQA S 2814, 3432 # JV-03, JV-01 */
        /* condition to check starting polarity of PWM channel */
        if ((uint8)PWM_LOW == LpChannelConfig->ucPolarity)
        {
          LpATUEChCntrlRegs->ucATUETOCREx |= (LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK);                 /* PRQA S 2814 # JV-03 */
        } /* else No action required */

        /* Condition to check type of channel PrescalerSelect for the PWM channel */
        if ((uint8)PWM_CH_SB_PRESCL_SEL == (uint8)(LpChannelProp->ucChannelSelMask & PWM_CH_SB_PRESCL_SEL))
        {
          /* Assign the configured pre-scalar value to channel Pre-scalar register */
          *(LpATUChannelConfigData->pPSCCRExy_Address) = LpATUChannelConfigData->ucChSelPrescalVal;
        } /* else No action required */

        /* Enable Reload function for cycle match */
        LpATUEChCntrlRegs->ucATUERLDCREx |= (LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK);
      } /* else No action required */
      /* End of blSyncStartReq is PWM_TRUE */
    }   /* End of if ((uint8)PWM_HWIP_ATU == LucTimerUnitType) */
    /* Increment the pointer to the next ATUE channel */
    LpChannelConfig++;                                                                                                  /* PRQA S 2824, 2934 # JV-01, JV-01 */
    /* Set ATU channel config data pointer */
    LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                       /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
  } /* End of Timer-E channels 'for' loop */
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* (PWM_SYNC_START_SUPPORT == STD_ON) */

/***********************************************************************************************************************
** Function Name         : Pwm_HW_SynchronousStart
**
** Service ID            : NA
**
** Description           : This is PWM Driver Component support function.
**                         This routine starts all PWM channels configured to support synchronous start/stop of timers.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Pre-conditions        : NA
**
** Global Variables used : Pwm_GpChannelConfig, Pwm_GpATUUnitConfig
**
** Functions invoked     : Pwm_HW_CalculateDuty
**
** Registers Used        : SSTREx,TCNTExy,CYLRExy,DTRExy,DRLDExy,CRLDExy
**
** Reference ID          : PWM_DUD_ACT_073, PWM_DUD_ACT_073_REG001,
** Reference ID          : PWM_DUD_ACT_073_REG002, PWM_DUD_ACT_073_REG003, PWM_DUD_ACT_073_REG004,
** Reference ID          : PWM_DUD_ACT_073_REG005, PWM_DUD_ACT_073_REG006, PWM_DUD_ACT_073_REG007
***********************************************************************************************************************/
#if (PWM_SYNC_START_SUPPORT == STD_ON)
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_SynchronousStart(void)                                                              /* PRQA S 1532 # JV-01 */
{
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfig;
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ATUChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpATUChannelConfigData;
  /* Local pointer to Timer-E sub block configuration */
  P2CONST(Pwm_ATUUnitConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpSublkConfig;
  /* Pointer used for each Timer-E channel control registers */
  P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelProp;
  /* Local pointer to Timer-E sub block control registers */
  P2VAR(Pwm_ATUESubblockRegs, AUTOMATIC, REGSPACE) LpATUESublkRegs;                                                     /* PRQA S 3432 # JV-01 */
  /* Update the pointer for base address of the  Channel control register */
  P2VAR(Pwm_ATUEChDataRegs, AUTOMATIC, REGSPACE) LpATUEChDataRegs;                                                      /* PRQA S 3432 # JV-01 */
  VAR(Pwm_PeriodType, AUTOMATIC) LddDefaultPeriod;
  VAR(Pwm_PeriodType, AUTOMATIC) LddDefaultduty;
  VAR(uint32, AUTOMATIC) LulCount;
  VAR(volatile uint8, AUTOMATIC) LucTimerUnitType;

  /* Update the configure pointer to point to the current Timer channel*/
  LpChannelConfig = Pwm_GpChannelConfig;
  /* Set ATU channel config data pointer */
  LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                         /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
  /* Update the configure pointer to point to the current Timer sub block*/
  LpSublkConfig = Pwm_GpATUUnitConfig;
  /* Get the Timer Unit Type*/
  LucTimerUnitType = LpChannelConfig->ucTimerUnitType;

  /* Loop to disable(stop) all configured Timer-E channels belonging to respective sub block */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_TIMER_E_SUBBLOCK_CONFIG; LulCount++)                   /* PRQA S 2877 # JV-01 */
  {
    /* Get pointer to the respective sub blocks register structure to local pointer */
    LpATUESublkRegs = (Pwm_ATUESubblockRegs *)LpSublkConfig->pATUE_SubblockRegs;                                        /* PRQA S 2814, 2844 # JV-03, JV-01 */
    if (PWM_ZERO != LpSublkConfig->ucSyncChannelMask)
    {
      /* Set the channel bits that are configured for synchronous stop, to disable the count operation */
      LpATUESublkRegs->ucATUESSTREx = LpATUESublkRegs->ucATUESSTREx & (uint8)(~(LpSublkConfig->ucSyncChannelMask));     /* PRQA S 2814 # JV-03 */
    } /* else No action required */
    /* Increment to next sub block configuration */
    LpSublkConfig++;                                                                                                    /* PRQA S 2824, 2934, 2982 # JV-01, JV-01, JV-01 */
  } /* End of Timer-E channels 'for' loop */
  /* Loop to set the attributes of Timer-E channels */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_CHANNELS_CONFIG; LulCount++)
  {
    if ((uint8)PWM_HWIP_ATU == LucTimerUnitType)                                                                        /* PRQA S 3416 # JV-01 */
    {
      /* Get the pointer to the PWM Channel properties */
      LpChannelProp = (P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST))LpChannelConfig->pChannelProp;      /* PRQA S 0316, 2814, 2844 # JV-01, JV-01, JV-01 */
      if ((boolean)PWM_TRUE == LpChannelConfig->blSyncStartReq)
      {
        /* Get the pointer to the PWM Channel data registers */
        LpATUEChDataRegs = (P2VAR(Pwm_ATUEChDataRegs, AUTOMATIC, REGSPACE))LpATUChannelConfigData->pATUEChDataRegs;     /* PRQA S 2814, 3432 # JV-03, JV-01 */
        /* Clear Timer Counters Exy for restart */
        LpATUEChDataRegs->ulATUETCNTExy = PWM_CH_COUNTER_INIT_VALUE;                                                    /* PRQA S 2814 # JV-03 */
        /* Get value of the channel period in local variable */
        LddDefaultPeriod = LpChannelProp->ddDefault_Period;                                                             /* PRQA S 2814 # JV-02 */
        /* Get value of the channel duty in local variable */
        LddDefaultduty = LpChannelProp->ddDefault_Duty;
        /* Assign the configured value of the channel period in to register */
        LpATUEChDataRegs->ulATUECYLRExy = LddDefaultPeriod << (uint32)PWM_BYTE_SHIFT_VALUE;
        /* Assign the configured value of the channel duty in to register */
        LpATUEChDataRegs->ulATUEDTRExy = Pwm_HW_CalculateDuty(LddDefaultPeriod, LddDefaultduty);
        /* Load the duty value to reload duty register */
        LpATUEChDataRegs->ulATUEDRLDExy = LpATUEChDataRegs->ulATUEDTRExy;
        /* Load the cycle value to reload cycle register */
        LpATUEChDataRegs->ulATUECRLDExy = LpATUEChDataRegs->ulATUECYLRExy;
      } /* else No action required */
    } /* End of if ((uint8)PWM_HWIP_ATU == LucTimerUnitType) */
    /* Increment the pointer to the next ATUE channel */
    LpChannelConfig++;                                                                                                  /* PRQA S 2824, 2934 # JV-01, JV-01 */
    /* Get the Timer Unit Type*/
    LucTimerUnitType = LpChannelConfig->ucTimerUnitType;                                                                /* PRQA S 2814, 2844 # JV-02, JV-01 */
    /* Set ATU channel config data pointer */
    LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                       /* PRQA S 0317 # JV-01 */
  } /* End of Timer-E channels 'for' loop */
  /* Update the configure pointer to point to the current Timer sub block*/
  LpSublkConfig = Pwm_GpATUUnitConfig;
  /* Loop to enable all configured Timer-E channels belonging to respective
     sub block */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_TIMER_E_SUBBLOCK_CONFIG; LulCount++)                   /* PRQA S 2877 # JV-01 */
  {
    /* Get pointer to the respective sub blocks register structure to local pointer */
    LpATUESublkRegs = (Pwm_ATUESubblockRegs *)(LpSublkConfig->pATUE_SubblockRegs);                                      /* PRQA S 2814, 2844 # JV-03, JV-01 */
    if (PWM_ZERO != LpSublkConfig->ucSyncChannelMask)
    {
      /* Set the channel bits that are configured for synchronous start, to enable the count operation */
      LpATUESublkRegs->ucATUESSTREx = LpATUESublkRegs->ucATUESSTREx | LpSublkConfig->ucSyncChannelMask;                 /* PRQA S 2814 # JV-03 */
    } /* else No action required */
    /* Increment to next sub block configuration */
    LpSublkConfig++;                                                                                                    /* PRQA S 2824, 2934, 2983 # JV-01, JV-01, JV-01 */
  } /* End of Timer-E channels 'for' loop */
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* (PWM_SYNC_START_SUPPORT == STD_ON) */

/***********************************************************************************************************************
** Function Name         : Pwm_HW_SynchronousStop
**
** Service ID            : NA
**
** Description           : This is PWM Driver Component support function.
**                         This routine stops all PWM channels configured to support synchronous start/stop of timers.
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : None
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Pre-conditions        : NA
**
** Global Variables used : Pwm_GpChannelConfig
**
** Functions invoked     : Pwm_HW_SetDutyToIdleState
**
** Registers Used        : RLDCREx
**
** Reference ID          : PWM_DUD_ACT_074
** Reference ID          : PWM_DUD_ACT_074_REG001, PWM_DUD_ACT_074_REG002
***********************************************************************************************************************/
#if (PWM_SYNC_START_SUPPORT == STD_ON)
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_SynchronousStop(void)                                                               /* PRQA S 1532 # JV-01 */
{
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfig;
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ATUChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpATUChannelConfigData;
  /* Pointer used for each Timer-E channel control registers */
  P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelProp;
  /* Update the pointer for base address of the  Channel control register */
  P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE) LpATUEChCntrlRegs;                                                    /* PRQA S 3432 # JV-01 */
  /* Update the pointer for base address of the  Channel control register */
  P2VAR(Pwm_ATUEChDataRegs, AUTOMATIC, REGSPACE) LpATUEChDataRegs;                                                      /* PRQA S 3432 # JV-01 */
  VAR(uint32, AUTOMATIC) LulCount;
  VAR(volatile uint8, AUTOMATIC) LucTimerUnitType;

  /* Loop to de-initialize the attributes of Timer-E channels */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_CHANNELS_CONFIG; LulCount++)
  {
    /* Get the pointer to the channel configuration */
    LpChannelConfig = &Pwm_GpChannelConfig[LulCount];
    /* Get the Timer Unit Type*/
    LucTimerUnitType = LpChannelConfig->ucTimerUnitType;                                                                /* PRQA S 2814, 2844 # JV-02, JV-01 */
    if ((uint8)PWM_HWIP_ATU == LucTimerUnitType)                                                                        /* PRQA S 3416 # JV-01 */
    {
      /* Set ATU channel config data pointer */
      LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                     /* PRQA S 0317 # JV-01 */
      if (PWM_TRUE == LpChannelConfig->blSyncStartReq)
      {
        /* Get the pointer to the PWM Channel properties */
        LpChannelProp = (P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST))LpChannelConfig->pChannelProp;    /* PRQA S 0316 # JV-01 */
        /* Get the pointer to the PWM Channel control registers */
        LpATUEChCntrlRegs = LpATUChannelConfigData->pATUEChCntrlRegs;                                                   /* PRQA S 2814 # JV-03 */
        /* Get the pointer to the PWM Channel data registers */
        LpATUEChDataRegs = LpATUChannelConfigData->pATUEChDataRegs;
        /* Reload Control Disable (RLDCREx) */
        LpATUEChCntrlRegs->ucATUERLDCREx &= (uint8)(~(LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK));        /* PRQA S 2814 # JV-03 */
        /* Set Duty to Idle state */
        Pwm_HW_SetDutyToIdleState(LpChannelConfig, LpATUEChDataRegs);
        /* Enable Reload function on cycle match */
        LpATUEChCntrlRegs->ucATUERLDCREx |= (uint8)((LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK));
      } /* else No action required */
    } /* End of if ((uint8)PWM_HWIP_ATU == LucTimerUnitType) */
  } /* End of Timer-E channels 'for' loop */
}
#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* (PWM_SYNC_START_SUPPORT == STD_ON) */

/***********************************************************************************************************************
** Function Name        : Pwm_HW_SelectChannelClk
**
** Service ID           : NA
**
** Description          : This function updates the clock source select value in hardware register.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Reentrant
**
** Input Parameters     : LddChannelId, LenClockSource
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Pre-conditions       : NA
**
** Global Variable used : Pwm_GpChannelConfig, Pwm_GpATUUnitConfig
**
** Functions invoked    : None
**
** Registers Used       : TCREx
**
** Reference ID         : PWM_DUD_ACT_071, PWM_DUD_ACT_071_CRT001,
** Reference ID         : PWM_DUD_ACT_071_CRT002, PWM_DUD_ACT_071_REG001
***********************************************************************************************************************/
#if (PWM_SELECT_CHANNEL_CLK_API == STD_ON)
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_SelectChannelClk(Pwm_ChannelType LddChannelId, Pwm_ClockSourceType LenClockSource)  /* PRQA S 1532 # JV-01 */
{
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfig;
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ATUChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpATUChannelConfigData;
  /* Pointer to Timer-E sub block configuration */
  P2CONST(Pwm_ATUUnitConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpSublkConfig;
  /* Local pointer to Timer-E sub block control registers */
  P2VAR(Pwm_ATUESubblockRegs, AUTOMATIC, REGSPACE) LpATUESublkRegs;                                                     /* PRQA S 3432 # JV-01 */
  VAR(uint32, AUTOMATIC) LulCount;

  /* Update the configure pointer to point to the current Timer channel*/
  LpChannelConfig = &Pwm_GpChannelConfig[LddChannelId];
  /* Set ATU channel config data pointer */
  LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                         /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
  /* Update the configure pointer to point to the current Timer sub block*/
  LpSublkConfig = Pwm_GpATUUnitConfig;

  #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Enter Protected area */
  PWM_ENTER_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
  #endif

  /* Loop to Initialize Timer-E Sub block related attributes */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_TIMER_E_SUBBLOCK_CONFIG; LulCount++)                   /* PRQA S 2877 # JV-01 */
  {
    if ((uint16)LpATUChannelConfigData->enSubblockNum == LpSublkConfig->usEnableSubblockMask)                           /* PRQA S 2814, 2844 # JV-02, JV-01 */
    {
      /* Get pointer to the respective sub blocks register structure to local pointer */
      LpATUESublkRegs = (Pwm_ATUESubblockRegs *)LpSublkConfig->pATUE_SubblockRegs;
      /* Load the clock value */
      LpATUESublkRegs->ucATUETCREx =
                      ((uint8)(LpATUESublkRegs->ucATUETCREx & PWM_SUBBLOCK_CLOCK_SEL_MASK) | (uint8)LenClockSource);    /* PRQA S 2814 # JV-03 */
    } /* else No action required */

    /* Increment to next sub block configuration */
    LpSublkConfig++;                                                                                                    /* PRQA S 2824, 2934, 2983 # JV-01, JV-01, JV-01 */
  }

  #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
  /* Exit Protected area */
  PWM_EXIT_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
  #endif
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif /* End of (PWM_SELECT_CHANNEL_CLK_API == STD_ON) */

/***********************************************************************************************************************
** Function Name         : Pwm_HW_Callback
**
** Service ID            : NA
**
** Description           : This routine is used to invoke the callback notification. This routine also supports
**                         PwmSetChannelOutput and channel class PWM_FIXED_PERIOD_SHIFTED functionality
**
** Sync/Async            : Synchronous
**
** Reentrancy            : Reentrant
**
** Input Parameters      : LddChannelId
**
** InOut Parameters      : None
**
** Output Parameters     : None
**
** Return parameter      : None
**
** Pre-conditions        : NA
**
** Global Variables used : Pwm_GpChannelConfig, Pwm_GpATUUnitConfig, Pwm_GaaNotifStatus, Pwm_GaaFpsSupport_Flag
**
** Functions invoked     : PWM_DEM_REPORT_ERROR
**
** Registers Used        : TIEREx, TSCREx, SSTREx, TSREx
**
** Reference ID          : PWM_DUD_ACT_069, PWM_DUD_ACT_069_CRT001, PWM_DUD_ACT_069_CRT002, PWM_DUD_ACT_069_ERR001,
** Reference ID          : PWM_DUD_ACT_069_GBL001, PWM_DUD_ACT_069_REG001, PWM_DUD_ACT_069_REG002,
** Reference ID          : PWM_DUD_ACT_069_REG003, PWM_DUD_ACT_069_REG004, PWM_DUD_ACT_069_REG005,
***********************************************************************************************************************/

#define PWM_START_SEC_CODE_FAST
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

FUNC(void, PWM_CODE_FAST) Pwm_HW_Callback(Pwm_ChannelType LddChannelId)                                                 /* PRQA S 1532 # JV-01 */
{
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfig;
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ATUChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpATUChannelConfigData;
  /* Pointer used for each Timer-E channel control registers */
  P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelProp;
  /* Pointer to Timer-E sub block configuration */
  P2CONST(Pwm_ATUUnitConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpSublkConfig;
  /* Pointer to Pwm_ATUEChCntrlRegs structure to generate the base address of PWM Channel control register */
  P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE) LpATUEChCntrlRegs;                                                    /* PRQA S 3432 # JV-01 */
  /* Local Pointer used for each Timer-E Sub block control registers */
  P2VAR(Pwm_ATUESubblockRegs, AUTOMATIC, REGSPACE) LpATUESublkRegs;                                                     /* PRQA S 3432 # JV-01 */
  /* Local variable for count */
  VAR(uint32, AUTOMATIC) LulCount;

  /* Initialize count value */
  LulCount = (uint32)PWM_ZERO;
  /* Update the configure pointer to point to the current Timer sub block*/
  LpSublkConfig = Pwm_GpATUUnitConfig;
  /* Update the configure pointer to point to the current Timer channel*/
  LpChannelConfig = &Pwm_GpChannelConfig[LddChannelId];
  /* Set ATU channel config data pointer */
  LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                         /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
  LpATUEChCntrlRegs = LpATUChannelConfigData->pATUEChCntrlRegs;                                                         /* PRQA S 2814 # JV-03 */
  /* Get the pointer to the PWM Channel properties */
  LpChannelProp = (P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_DATA))LpChannelConfig->pChannelProp;           /* PRQA S 0316 # JV-01 */

  #if (PWM_INTERRUPT_CONSISTENCY_CHECK == STD_ON)
  if ((uint16)PWM_ZERO != (uint16)(*LpATUChannelConfigData->pEICx_Address & PWM_EIC_EIMK_MASK))                         /* PRQA S 2814, 4399 # JV-01, JV-01 */
  {
    /* Channel interrupt Masked */
    PWM_DEM_REPORT_ERROR(PWM_E_INT_INCONSISTENT, DEM_EVENT_STATUS_FAILED);                                              /* PRQA S 3469 # JV-01 */
  }
  else
  #endif
  {
    #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
    /* Enter Protected area */
    PWM_ENTER_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
    #endif

    /* Condition to check the duty match interrupt status flag for the channel*/
    if ((uint32)PWM_TRUE == Pwm_GaaFpsSupport_Flag[LddChannelId])                                                       /* PRQA S 2844 # JV-01 */
    {
      /* Disable the duty interrupt of Reference channel */
      LpATUEChCntrlRegs->usATUETIEREx &= (uint16)(~(LpChannelProp->usChannelIntSelMask & PWM_DUTY_MATCH_MASK));         /* PRQA S 2814 # JV-01 */
      /* Clear the ISR request after service */
      LpATUEChCntrlRegs->usATUETSCREx = (LpChannelProp->usChannelIntSelMask & PWM_DUTY_MATCH_MASK);
      /* Point increment to point to channel of class PWM_FIXED_PERIOD_SHIFTED for reference channel */
      LpChannelConfig++;                                                                                                /* PRQA S 2824, 2934 # JV-01, JV-01 */
      /* Set ATU channel config data pointer */
      LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                     /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
      /* Get the pointer to the PWM Channel properties */
      LpChannelProp = (P2CONST(Pwm_ATUChannelPropType, AUTOMATIC, PWM_CONFIG_CONST))LpChannelConfig->pChannelProp;      /* PRQA S 0316 # JV-01 */

      do
      {
        LpATUESublkRegs = (P2VAR(Pwm_ATUESubblockRegs, AUTOMATIC, REGSPACE))LpSublkConfig->pATUE_SubblockRegs;          /* PRQA S 2814, 2844, 3432 # JV-01, JV-01, JV-01 */

        /* To access corresponding enable reg */
        if ((uint16)LpATUChannelConfigData->enSubblockNum == LpSublkConfig->usEnableSubblockMask)                       /* PRQA S 2814 # JV-03 */
        {
          /* Start/En channel of class PWM_FIXED_PERIOD_SHIFTED for reference channel */
          LpATUESublkRegs->ucATUESSTREx |= (LpChannelProp->ucChannelSelMask & PWM_LOWER_NIBBLE_MASK);                   /* PRQA S 2814 # JV-03 */
          /* end Loop */
          LulCount = PWM_TOTAL_TIMER_E_SUBBLOCK_CONFIG;
        } /* else No action required */

        /* Increment to next sub block configuration */
        LpSublkConfig++;                                                                                                /* PRQA S 2824, 2934, 2983 # JV-01, JV-01, JV-01 */
        LulCount++;                                                                                                     /* PRQA S 3383 # JV-01 */
      } while (LulCount < (uint32)PWM_TOTAL_TIMER_E_SUBBLOCK_CONFIG);                                                   /* PRQA S 2993, 2996 # JV-01, JV-01 */

      /* Update the Fixed Period Shift support flag to false after shift */
      Pwm_GaaFpsSupport_Flag[LddChannelId] = PWM_FALSE;                                                                 /* PRQA S 2844 # JV-01 */
    }
    else
    {
      /* Clear the ISR request of the channel serviced depending on enabled type of interrupt */
      if ((uint16)PWM_ZERO !=
        (uint16)(LpATUEChCntrlRegs->usATUETSREx & (LpChannelProp->usChannelIntSelMask & PWM_CYCLE_MATCH_MASK)))
      {
        LpATUEChCntrlRegs->usATUETSCREx = (LpChannelProp->usChannelIntSelMask & PWM_CYCLE_MATCH_MASK);
      } /* else No action required */

      if ((uint16)PWM_ZERO !=
        (uint16)(LpATUEChCntrlRegs->usATUETSREx & (LpChannelProp->usChannelIntSelMask & PWM_DUTY_MATCH_MASK)))
      {
        LpATUEChCntrlRegs->usATUETSCREx = (LpChannelProp->usChannelIntSelMask & PWM_DUTY_MATCH_MASK);
      } /* else No action required */
    }

    #if (PWM_CRITICAL_SECTION_PROTECTION == STD_ON)
    /* Exit Protected area */
    PWM_EXIT_CRITICAL_SECTION(PWM_INTERRUPT_CONTROL_PROTECTION);
    #endif

    #if (PWM_NOTIFICATION_SUPPORTED == STD_ON)
    /* Check if Notification status is enabled for this channel */
    if (PWM_TRUE == Pwm_GaaNotifStatus[LddChannelId])                                                                   /* PRQA S 2844 # JV-01 */
    {
      /* Invoke the Notification function */
      LpChannelConfig->pPwmEdgeNotifPtr();                                                                              /* PRQA S 2814, 2844 # JV-01, JV-01 */
    } /* else No action required */
    #endif
  } /* End of interrupt consistency check */
}

#define PWM_STOP_SEC_CODE_FAST
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name        : Pwm_HW_InitChannelRegister
**
** Service ID           : NA
**
** Description          : This is PWM Driver Component support function.
**                        This function is used to initialize timerE registers
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Non-Rentrant
**
** Input Parameters     : None
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Pre-conditions       : NA
**
** Global Variable used : Pwm_GpChannelConfig
**
** Functions invoked    : NA
**
** Registers Used       : SSTREx,TCREx,RLDCREx,TSCRE,TCNTExy,CYLRExy, DTRExy,CRLDExy,CRLDExy,CRLDExy,DRLDExy
**
** Reference ID          : PWM_DUD_ACT_077, PWM_DUD_ACT_077_REG001, PWM_DUD_ACT_077_REG002, PWM_DUD_ACT_077_REG003,
** Reference ID          : PWM_DUD_ACT_077_REG004, PWM_DUD_ACT_077_REG005, PWM_DUD_ACT_077_REG006,
** Reference ID          : PWM_DUD_ACT_077_REG007
***********************************************************************************************************************/
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

STATIC FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_InitChannelRegister(void)
{
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpChannelConfig;
  /* Local pointer to the each Timer-E channel configuration */
  P2CONST(Pwm_ATUChannelConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpATUChannelConfigData;
  /* Local Pointer used for each Timer-E channel control registers */
  P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE) LpATUEChCntrlRegs;                                                    /* PRQA S 3432 # JV-01 */
  /* Local Pointer used for each Timer-E channel data registers */
  P2VAR(Pwm_ATUEChDataRegs, AUTOMATIC, REGSPACE) LpATUEChDataRegs;                                                      /* PRQA S 3432 # JV-01 */
  /* Local variable for count */
  VAR(uint32, AUTOMATIC) LulCount;
  VAR(volatile uint8, AUTOMATIC) LucTimerUnitType;
  /* Loop to set the attributes of TAU channels */
  LpChannelConfig = Pwm_GpChannelConfig;                                                                                /* PRQA S 2982 # JV-01 */
  /* Update the configure pointer to point to the current Timer channel*/
  LpChannelConfig = Pwm_GpChannelConfig;
  /* Set ATU channel config data pointer */
  LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                         /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
  /* Get the Timer Unit Type*/
  LucTimerUnitType = LpChannelConfig->ucTimerUnitType;

  /* Loop to set the attributes of Timer-E channels */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_CHANNELS_CONFIG; LulCount++)
  {
    if ((uint8)PWM_HWIP_ATU == LucTimerUnitType)                                                                        /* PRQA S 3416 # JV-01 */
    {
      /* Get the pointer to the PWM Channel control registers */
      LpATUEChCntrlRegs = (P2VAR(Pwm_ATUEChCntrlRegs, AUTOMATIC, REGSPACE))LpATUChannelConfigData->pATUEChCntrlRegs;    /* PRQA S 2814, 3432 # JV-03, JV-01 */
      /* Get the pointer to the PWM Channel data registers */
      LpATUEChDataRegs = (P2VAR(Pwm_ATUEChDataRegs, AUTOMATIC, REGSPACE))LpATUChannelConfigData->pATUEChDataRegs;       /* PRQA S 3432 # JV-01 */
      /* Clear RLDCREx register */
      LpATUEChCntrlRegs->ucATUERLDCREx = PWM_RLDCRE_REG_RESET_VALUE;                                                    /* PRQA S 2814 # JV-03 */
      /* Clear TSCREx register */
      LpATUEChCntrlRegs->usATUETSCREx = PWM_TSRE_REG_RESET_VALUE;
      /* Clear TCNTExy register */
      LpATUEChDataRegs->ulATUETCNTExy = PWM_CH_COUNTER_INIT_VALUE;                                                      /* PRQA S 2814 # JV-03 */
      /* Clear CYLRExy register */
      LpATUEChDataRegs->ulATUECYLRExy = PWM_CYLRE_REG_RESET_VALUE;
      /* Clear DTRExy register */
      LpATUEChDataRegs->ulATUEDTRExy = PWM_DTRE_REG_RESET_VALUE;
      /* Clear CRLDExy register */
      LpATUEChDataRegs->ulATUECRLDExy = PWM_CRLDE_REG_RESET_VALUE;
      /* Clear DRLDExy register */
      LpATUEChDataRegs->ulATUEDRLDExy = PWM_DRLDE_REG_RESET_VALUE;
    } /* End of if ((uint8)PWM_HWIP_ATU == LucTimerUnitType) */
    /* Increment the pointer to the next ATUE channel */
    LpChannelConfig++;                                                                                                  /* PRQA S 2824, 2934 # JV-01, JV-01 */
    /* Set ATU channel config data pointer */
    LpATUChannelConfigData = LpChannelConfig->pHWIPChannelConfig;                                                       /* PRQA S 0317, 2814, 2844 # JV-01, JV-01, JV-01 */
    /* Get the Timer Unit Type*/
    LucTimerUnitType = LpChannelConfig->ucTimerUnitType;
  } /* End of Timer-E channels 'for' loop */
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

/***********************************************************************************************************************
** Function Name        : Pwm_HW_InitUnitRegister
**
** Service ID           : NA
**
** Description          : This is PWM Driver Component support function.
**                        This function is used to initialize timerE registers
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Non-Rentrant
**
** Input Parameters     : None
**
** InOut Parameters     : None
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Pre-conditions       : NA
**
** Global Variable used : Pwm_GpATUUnitConfig
**
** Functions invoked    : NA
**
** Registers Used       : SSTREx,TCREx,RLDCREx,TSCRE,TCNTExy,CYLRExy DTRExy,CRLDExy,CRLDExy,CRLDExy,DRLDExy
**
** Reference ID          : PWM_DUD_ACT_075, PWM_DUD_ACT_075_REG001, PWM_DUD_ACT_075_REG002
***********************************************************************************************************************/
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

STATIC FUNC(void, PWM_PRIVATE_CODE) Pwm_HW_InitUnitRegister(void)
{
  /* Pointer to Timer-E sub block configuration */
  P2CONST(Pwm_ATUUnitConfigType, AUTOMATIC, PWM_CONFIG_CONST) LpSublkConfig;
  /* Local Pointer used for each Timer-E Sub block control registers */
  P2VAR(Pwm_ATUESubblockRegs, AUTOMATIC, REGSPACE) LpATUESublkRegs;                                                     /* PRQA S 3432 # JV-01 */
  /* Local variable for count */
  VAR(uint32, AUTOMATIC) LulCount;

  /* Update the configure pointer to point to the current Timer sub block*/
  LpSublkConfig = Pwm_GpATUUnitConfig;
  /* Loop to Initialize Timer-E Sub block related attributes */
  for (LulCount = (uint32)PWM_ZERO; LulCount < (uint32)PWM_TOTAL_TIMER_E_SUBBLOCK_CONFIG; LulCount++)                   /* PRQA S 2877 # JV-01 */
  {
    /* Get the pointer to the PWM sub block control registers */
    LpATUESublkRegs = (P2VAR(Pwm_ATUESubblockRegs, AUTOMATIC, REGSPACE))LpSublkConfig->pATUE_SubblockRegs;              /* PRQA S 2814, 2844, 3432 # JV-03, JV-01, JV-01 */
    /* Clear SSTREx register */
    LpATUESublkRegs->ucATUESSTREx = PWM_SUBBLOCK_STARTING_REG_RESET_VALUE;                                              /* PRQA S 2814 # JV-03 */
    /* Clear TCREx register */
    LpATUESublkRegs->ucATUETCREx = PWM_TCRE_REG_RESET_VALUE;
    /* Increment to next sub block configuration */
    LpSublkConfig++;                                                                                                    /* PRQA S 2824, 2934, 2983 # JV-01, JV-01, JV-01 */
  } /* End of PWM_TOTAL_TIMER_E_SUBBLOCK_CONFIG 'for' loop */
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
/***********************************************************************************************************************
** Function Name        : Pwm_HW_SetDutyToIdleState
**
** Service ID           : NA
**
** Description          : This is PWM Driver Component support function.
**                        This function sets the Duty of a required channel to 0% or 100%.
**
** Sync/Async           : Synchronous
**
** Reentrancy           : Non-Rentrant
**
** Input Parameters     : None
**
** InOut Parameters     : LpChannelConfig, LpATUEChDataRegs
**
** Output Parameters    : None
**
** Return parameter     : None
**
** Pre-conditions       : NA
**
** Global Variable used : None
**
** Functions invoked    : NA
**
** Registers Used       : DRLDExy, CYLRExy
**
** Reference ID         : PWM_DUD_ACT_076, PWM_DUD_ACT_076_REG001, PWM_DUD_ACT_076_REG002
***********************************************************************************************************************/
#if ((PWM_SET_OUTPUT_TO_IDLE_API == STD_ON) || (PWM_SYNC_START_SUPPORT == STD_ON) || (PWM_DE_INIT_API == STD_ON))
#define PWM_START_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */

STATIC FUNC(void, PWM_PRIVATE_CODE)
  Pwm_HW_SetDutyToIdleState(const Pwm_ChannelConfigType *LpChannelConfig, Pwm_ATUEChDataRegs *LpATUEChDataRegs)
{
  /* Condition check for polarity and idle level of the channel */
  if ((((uint8)PWM_HIGH == LpChannelConfig->ucIdleLevel) && ((uint8)PWM_LOW == LpChannelConfig->ucPolarity))            /* PRQA S 2814 # JV-02 */
    || (((uint8)PWM_LOW == LpChannelConfig->ucIdleLevel) && ((uint8)PWM_HIGH == LpChannelConfig->ucPolarity)))
  {
    /* Assign the duty cycle value of the channel to register */
    LpATUEChDataRegs->ulATUEDRLDExy = (uint32)PWM_DUTY_CYCLE_RESET_VALUE;                                               /* PRQA S 2814 # JV-03 */
  }
  else if ((((uint8)PWM_LOW == LpChannelConfig->ucIdleLevel) && ((uint8)PWM_LOW == LpChannelConfig->ucPolarity))        /* PRQA S 2004 # JV-01 */
       || (((uint8)PWM_HIGH == LpChannelConfig->ucIdleLevel) && ((uint8)PWM_HIGH == LpChannelConfig->ucPolarity)))
  {
    /* Assign the duty cycle value of the channel to register */
    LpATUEChDataRegs->ulATUEDRLDExy = LpATUEChDataRegs->ulATUECYLRExy;
  } /* else No action required */
}

#define PWM_STOP_SEC_PRIVATE_CODE
#include "Pwm_Mapping.h"                                                                                                /* PRQA S 5087 # JV-01 */
#endif

#endif /* (PWM_ATU_UNIT_USED == STD_ON) */
/***********************************************************************************************************************
**                                                    End of File                                                     **
***********************************************************************************************************************/
